function [b_nodes,b_elements] = buildMixedRHS(coordinates,dirichlet, ...
                                                   neumann,V,K,W,uDh,phiNh)
%BUILDMIXEDRHS   Right hand-side for a mixed problem.
%   [B_NODES,B_ELEMENTS] = BUILDMIXEDRHS(COORDINATES,DIRICHLET, ...
%                                                  NEUMANN,V,K,W,UDH,PHINH)
%   assembles the right-hand side for an integral formulation of a mixed 
%   boundary value problem with (S1 x P0)-ansatz and test functions.
%
%   Let {E1,...EN} be a given partition of the closed boundary Gamma with 
%   set of nodes {z1,...,zN}. This partition is described in terms of the 
%   (N x 2)-matrix COORDINATES, the (d x 2)-matrix DIRICHLET, and the 
%   (n x 2)-matrix NEUMANN, where N = d + n. COORDINATES gives the 
%   coordinates for all nodes on Gamma. DIRICHLET and NEUMANN describe the 
%   elements on the Dirichlet and Neumann boundary, respectively. V, K, and
%   W are the matrices for the simple-layer and double-layer potential and 
%   for the hypersingular integral operator.
%
%   We stress that the solution of the integral formulation depends on the
%   extension of the given Dirichlet and Neumann data from Gamma_D and 
%   Gamma_N to the entire boundary Gamma. To guarantee that throughout the 
%   same extension is used, UDH and PHINH should be the return values of 
%   the function computeOscMixed. The latter function admits a call that 
%   can be used in an adaptive loop to ensure that the extensions do not 
%   change. The (N x 1)-vector UDH contains the nodal values of the 
%   approximate Dirichlet data uDh, i.e. uDh(zj) = UDH(j). The 
%   (N x 1)-vector PHINH contains the elementwise values of the approximate
%   Neumann data phiNh, i.e. phiNh|_{Ej} = PHINH(j).
%
%   The function returns the (N x 1)-column vectors B_NODES and B_ELEMENTS,
%   where
%
%      B_NODES(j) = rhs for j-th hat-function associated with node zj
%      B_ELEMENTS(j) = rhs for j-th characteristic function for Ej
%
%   Then, The right-hand side for the integral formulation of the mixed 
%   boundary value problem is built of appropriate subvectors of B_NODES 
%   and B_ELEMENTS.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

elements = [dirichlet;neumann];
nE = size(elements,1);

%*** compute mass-type matrix for P0 x S1
M = buildM(coordinates,elements);

%*** compute full right-hand side
b_nodes = (0.5*phiNh'*M - phiNh'*K - uDh'*W)';
b_elements = M*uDh*0.5 + K*uDh - V*phiNh;

