/************************************************************************\
*   Authors:
*       	Luisa D'Amore - University of Naples, Federico II
*		Rosanna Campagna - University of Naples, Federico II
*		Valeria Mele - University of Naples, Federico II
*		Almerico Murli - SPACI - NAPLES
* 		Mariarosaria Rizzardi - University of Naples, Parthenope
\************************************************************************/

#include "ReLaTIve.h"

int ReLaTIve ( double x, double (*fz)(double), int sinf,
    double *sigma0i, int sflag, double *tol, int *nmax,
    int *nopt, int *ncalc, double *absesterr, double *relesterr, int *flag, double *ILf ){


/**************************************************************************************

	1. PURPOSE
	=============================================================================

	THIS ROUTINE COMPUTES AN APPROXIMATE VALUE OF THE INVERSE LAPLACE TRANSFORM,
      BY MEANS OF ITS LAGUERRE POLYNOMIALS EXPANSION.
      THE METHOD - WHICH THIS SOFTWARE IS BASED ON - UTILIZES THE BJORCK-PEREIRA
      ALGORITHM TO COMPUTE THE COEFFICIENTS OF THE SERIES EXPANSION.
      THE SUMMATION OF SUCCESSIVE TERMS IS TRUNCATED WHEN THE SUM OF THE
      ERRORS IS LESS OR EQUAL THAN AN INPUT PROVIDED TOLERANCE.

      THE INVERSE FUNCTION IS COMPUTED AS:

	(1) f(x)=exp((sigma0-b)*x [ c0 L0(2bx)+c1 L1(2bx)+...+ cn Ln(2bx)]

	WHERE SIGMA0 (GREATER OR EQUAL TO ZER) IS THE ABSCISSA OF CONVERGENCE OF

      THE LAPLACE TRANSFORM AND b IS A PARAMETER.

      Lk (K LESS OR EQUAL TO N) IS THE k-TH DEGREE LAGUEREE POLYNOMIAL.

	==================================================================
	INPUT PARAMETERS
	==================================================================

	x  :		DOUBLE PRECISION: ON ENTRY IT CONTAINS VALUE AT WHICH THE INVERSE
				LAPLACE FUNCTION  IS REQUIRED.
				EACH COMPONENT OF X HAS TO HAVE A VALUE  GREATER OR  EQUAL TO ZERO.

	fz :        	DOUBLE PRECISION  FUNCTION POINTER:
				ON ENTRY IT CONTAINS THE NAME OF THE LAPLACE TRANSFORM FUNCTION

	sinf:		INTEGER: PARAMETER TO SAY IF THE TRANSFORM HAS A SINGULARITY AT INFINITY

	sflag :		INTEGER: ON ENTRY  IT MEANS IF USER IS GIVING THE THE ABSCISSA OF CONVERGENCE OF F.

	==================================================================
	INPUT/OUTPUT PARAMETERS
	==================================================================

	sigma0i :	DOUBLE: IF THE PARAMETER sflag>0, ON ENTRY IT IT CONTAINS THE ABSCISSA OF CONVERGENCE OF F,
				      AND IF IT IS LESS THAN ZERO, IT WILL BE POSED TO ZERO.
				IF THE PARAMETER sflag>0, IT WILL BE IGNORED AND POSED TO THE DEFAULT VALUE.

	tol :       	DOUBLE PRECISION: ON ENTRY IT CONTAINS THE REQUIRED ACCURACY ON f.
				IF IT IS LESS THAN 0 IT WILL BE POSED TO THE DEFAULT VALUE.
				IF IT IS EQUAL TO 0 IT WILL BE POSED TO THE DEFAULT VALUE.
				IF IT IS GREATER THAN 1 IT WILL BE POSED TO THE DEFAULT VALUE.
				IF IT IS LESS THAN 1^-7, IT WILL BE POSED TO 1^-7.

	nmax:	    	INTEGER : ON ENTRY IT CONTAINS THE MAXIMUM NUMBER OF LAGUERRE SERIES TERMS
				IF NMAX < 8, IT IS POSED AT A DEFAULT VALUE (NMAX=2000)
				IF NMAX > MaxLength (DEFINED IN RELIADIFF.h), IT IS POSED AT MaxLength=5000.

	==================================================================
	OUTPUT PARAMETERS
	==================================================================

	nopt:       INTEGER: NUMBER OF TERMS USED FOR THE COMPUTATION OF f(x)
                         IN THE LAGUERRE SERIES EXPANSION

	ncalc:		INTEGER: IT CONTAINS THE MAX NUMBER OF TERMS CALCULATED
				TO FIND nopt

	absesterr: 	DOUBLE PRECISION: ABSOLUTE ERROR ESTIMATE. IT CONTAINS AN UPPER BOUND OF
								  ABS(f(x)-ILf)

	relesterr:  DOUBLE PRECISION: RELATIVE ERROR ESTIMATE. IT CONTAINS AN UPPER BOUND OF
								  ABS (f(X)-ILf)/ABS(ILf)

	flag:       INTEGER :  ON OUTPUT IT CONTAINS A DIAGNOSTIC PARAMETER

	ILf :		DOUBLE PRECISION:
				ON OUTPUT IT CONTAINS THE COMPUTED VALUE  OF THE INVERSE  FUNCTION  AT X


	==================================================================
	RETURN VALUE
	==================================================================

	INTEGER: 	ON OUTPUT IT CONTAINS A DIAGNOSTIC ON THE INPUT DATA.
				IF IT IS EQUAL TO 1, IT MEANS THAT THE ROUTINE TERMINED WITHOUT ANY OUTPUT BECAUSE THE INPUT x WAS LESS THAN 0.

	=============================================================================
	PROCEDURE NEEDED
	=============================================================================

	void  ck_calc();  Compute and estimate the error on coefficients

	=============================================================================

	=============================================================================
	AUTHORS

	Luisa D'Amore - University of Naples, Federico II
	Rosanna Campagna - University of Naples, Federico II
	Valeria Mele - University of Naples, Federico II
	Almerico Murli - SPACI - NAPLES
	Mariarosaria Rizzardi - University of Naples Parthenope

	==============================================================================
	REFERENCE

	    A. Murli, S.Cuomo, L. D'Amore, M. Rizzardi - JCAM, Vol 198, N.1, 2007


**************************************************************************************/

	/* ...Local Variables specifications... */
	int 	k,i,n,nstart=8;
	double  b,expsx,esterr,esterr_min,sigma;
	double  fx,Ntol;
	double  esp,L1,L2,LK,y;
	double  temp1,temp2;
	double 	err, err1;
	double	sigma0=1.;
	double	defTol=1.0e-03;
	double	shift=-1;
	double 	*ck=NULL;
	double	MINtol=1.0e-07;


    /*********************************************************************************\
	 INPUT DATA CHECK : THE ABSCISSA OF CONVERGENCE HAS TO BE EQUAL OR GREATER THAN ZERO
                        THE X-VALUES HAVE TO BE EQUAL OR GREATER THAN ZERO
						THE TOLERANCE HAS TO BE SMALLER THAN ONE AND GREATER THAN MACHINE PRECISION
    \********************************************************************************/

	if( x<0.0) return 1;

	if (*tol <= 0.0  ||  *tol > 1.0)
		*tol=defTol;
	else if (*tol<MINtol)
		*tol=MINtol;

	if (sflag>0) {
		sigma0=*sigma0i;
	}
	if(sinf!=0)
		sigma0=sigma0-shift;		/*shift in case F has a singularity at infinity*/

	if(sigma0 < 0.0)
		sigma0=0.0;


	*sigma0i=sigma0;

	if(*nmax<nstart) *nmax=40;

	/****************************************************************************\
	 SPECIFICATION OF THE PARAMETERS EXPSX, SIGMA, B AND SCALING OF THE TOLERANCE TOL.
	 TOL IS SCALED AS FOLLOWS:  Ntol=tol/exp(sigma*x)/exp(sigma0*x)
	\**************************************************************************/

	if(x>0.0){
		expsx=exp(2.5);  /* expsx=exp(sigma*x)*/
		sigma=2.5/x;    /*  optimal value of sigma x = 2.5 */
		b=2.5*sigma;    /*  optimal value of b=2.5 sigma */
		Ntol=(*tol/expsx)/exp(sigma0*x);
	}
	else{	            /* if x=0 sigma =4, b=1, ntol=tol, expsx=1 */
		sigma=4;
		b=1;
		expsx=1;
		Ntol=*tol;
	}
	y=2*b*x;
	esp=exp((sigma-b)*x);

	/**************************************************************************************\
	  THE FOLLOWING PART OF THE CODE COMPUTES:

	  nopt:	      THE VALUE OF n IN (1)
	  esterr:     GLOBAL ERROR ESTIMATE;

        THE GLOBAL ERROR (ESTERR) CONSISTS OF:

	  1) THE TRUNCATION-DISCRETIZATION ERROR(err1).

	  2) THE CONDITION ERROR (err).

	  THE STOPPING CRITERION FOR THE COMPUTATION OF nopt
	  IS BASED ON THE COMPARISON OF THE  GLOBAL ERROR (ESTERR) WITH THE SCALED TOLERANCE NTOL.
	  THE MAXIMUM NUMBER OF TERMS IS NMAX.
	  THE MINIMUM NUMBER OF TERMS IS 8.
	\**************************************************************************************/

	n=nstart;
	*nopt=n;
	do{
		if(ck!=NULL) free(ck);
		                                 /* COMPUTATION OF c0,...,cn AND OF THE COMPONENT-WISE ERROR. */
		ck=(double*)calloc(n,sizeof(double));
        if ( ck == NULL )
        {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
            exit(1);
        }
                                              /* CK_CALC FUNCTION CALL FOR THE COMPUTATION OF COEFFICIENTS CK */
		ck_calc(n,sigma,sigma0,b,fz,ck,&err,sinf);
		                                 /* ERR ESTIMATE */
		for(i=0;i<n;i++) err=(err+fabs((ck)[i]));
		                                /* ERR1 ESTIMATE */
		err1=(fabs((ck)[n-3])+fabs((ck)[n-2])+fabs((ck)[n-1]))/3;
		esterr=DBL_EPSILON*err+err1;  /* GLOBAL ERROR ESTIMATE */
		if (n==8) esterr_min=esterr;
		else if(esterr<esterr_min){
			esterr_min=esterr;
			*nopt=n;
		}
		n += 2;
	}while((esterr_min>Ntol)&&(n<=*nmax));


	if ((n>*nmax)&&(*nopt<n)){
		free(ck);
		ck = (double*)calloc(*nopt,sizeof(double));
        if ( ck == NULL )
        {   fprintf(stderr, "\nERROR: DYNAMIC ALLOCATION FAILED.\n");
            exit(1);
        }
        ck_calc(*nopt,sigma,sigma0,b,fz,ck,&err,sinf);
	}
	*ncalc=n-2;

    /***************************************************************************************
		COMPUTATION OF  THE LAGUERRE SERIES EXPANSION APPROXIMATING THE INVERSE FUNCTION
		[ c0*L0(2bx)+c1*L1(2bx)+...+ cn*Lnopt(2bx)]
	*************************************************************************************/

    L1=1;
	L2=1-y;
	fx=(ck)[0]+ck[1]*L2;
	for(k=2;k<(*nopt);k++){
		temp1=(-y+2*((double) k)-1)/(double)(k);
		temp2=(((double) k)-1)/(double)(k);
		LK=temp1*L2-temp2*L1;
		fx=fx+ck[k]*LK;
		L1=L2;
		L2=LK;
	}
	                                    /* ILf IS THE APPROXIMATED VALUE OF THE INVERSE LAPLACE FUNCTION*/
    *ILf=fx*esp;
	*ILf=*ILf*exp(sigma0*x);

	if(sinf!=0)
		*ILf=*ILf*exp(shift*x);       /*damping in case F has a singularity at infinity*/

	                                    /* ABSOLUTE ERROR ESTIMATE ON ILF*/
	*absesterr=(esterr_min*expsx)*exp(sigma0*x);

	                                   /* RELATIVE ERROR ESTIMATE ON ILF*/
	if (fx<=MINtol) *relesterr=(esterr_min)/fabs((fx+1)*exp(-b*x));
	else *relesterr=(esterr_min)/fabs(fx*exp(-b*x));


	/************************************************************************************
	 * 			           WARNING
	--------Calculated errors are just errors estimate, they are not errors bounds-------	
	
	*************************************************************************************/

                       /*IF relesterr<tol  AND  absesterr<tol THEN IFLAG = 0*/
	*flag=0;
	if(*relesterr>*tol){
		                /*IF relesterr>tol absesterr>tol, n>=nmax THEN IFLAG= 1*/
		if(*absesterr>*tol) *flag=1;
		                /*IF relesterr>tol AND  absesterr<tol THEN IFLAG = 2*/
		else *flag=2;
	}
	else{
		                /*IF relesterr<tol AND absesterr>tol THEN IFLAG = 3 */
		if(*absesterr>*tol) *flag=3;
	}

	return 0;
}
/*END OF RELATIVE()*/

