      MODULE LA_BLAS3
!
!  -- Generic interfaces to the Fortran 77 Level 3 BLAS --
!     E. Anderson
!     April 10, 2002
!
!  Purpose
!  =======
!
!  This module defines generic interfaces for the FORTRAN 77 Level 3
!  BLAS using the prefixes S, D, C, and Z.  Two interfaces are defined
!  for each type-specific BLAS routine: the "point" interface, in which
!  the array arguments are defined as scalars, and the "natural"
!  interface, in which the array arguments are defined as in the
!  subroutine or function declaration.  The point interface in this
!  implementation references the standard BLAS with no additional
!  overhead.  It is accessed by indexing, as in
!
!     CALL LA_GEMM( 'N', 'N', M-JB, N-JB, JB, -ONE, A( 1+JB, 1 ), LDA,
!    $              A( 1, 1+JB ), LDA, ONE, A( 1+JB, 1+JB ), LDA )
!
!  The natural interface is specified as a wrapper to the point
!  interface and carries the overhead of an extra subroutine call.
!  The natural interface is used for a call without indexing, such as
!
!     CALL LA_GEMM( 'N', 'N', M, N, K, ALPHA, A, LDA, B, LDB, BETA,
!    $              C, LDC )
!
!  If only a small number of BLAS are used, it may be desired to include
!  only the necessary interfaces in the calling subprogram, for example,
!
!     USE LA_BLAS3, ONLY: LA_GEMM, LA_TRSM
!
!-----------------------------------------------------------------------
!
!!!!!!!!!!!! ============ !!!!!!!!!!!!
!!!!!!!!!!!! Level 3 BLAS !!!!!!!!!!!!
!!!!!!!!!!!! ============ !!!!!!!!!!!!
!
!-----Interface for xGEMM-----
!
      INTERFACE LA_GEMM
 
      SUBROUTINE SGEMM( TRANSA, TRANSB, M, N, K, ALPHA, A, LDA, B, LDB, &
     &                   BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANSA, TRANSB
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, B
         REAL(WP), INTENT(INOUT) :: C
      END SUBROUTINE SGEMM
 
      SUBROUTINE CGEMM( TRANSA, TRANSB, M, N, K, ALPHA, A, LDA, B, LDB, &
     &                   BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANSA, TRANSB
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, B
         COMPLEX(WP), INTENT(INOUT) :: C
      END SUBROUTINE CGEMM
 
      MODULE PROCEDURE SGEMM_A2B2C2
      MODULE PROCEDURE CGEMM_A2B2C2
 
      END INTERFACE ! LA_GEMM
      PRIVATE SGEMM_A2B2C2, CGEMM_A2B2C2
!
!-----Interface for xSYMM and xHEMM-----
!
      INTERFACE LA_SYMM
 
      SUBROUTINE SSYMM( SIDE, UPLO, M, N, ALPHA, A, LDA, B, LDB,        &
     &                  BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: SIDE, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, LDC, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, B
         REAL(WP), INTENT(INOUT) :: C
      END SUBROUTINE SSYMM
 
      SUBROUTINE CSYMM( SIDE, UPLO, M, N, ALPHA, A, LDA, B, LDB,        &
     &                  BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: SIDE, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, LDC, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, B
         COMPLEX(WP), INTENT(INOUT) :: C
      END SUBROUTINE CSYMM
 
      MODULE PROCEDURE SSYMM_A2B2C2
      MODULE PROCEDURE CSYMM_A2B2C2
 
      END INTERFACE ! LA_SYMM
      PRIVATE SSYMM_A2B2C2, CSYMM_A2B2C2
 
      INTERFACE LA_HEMM
 
      SUBROUTINE CHEMM( SIDE, UPLO, M, N, ALPHA, A, LDA, B, LDB,        &
     &                  BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: SIDE, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, LDC, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, B
         COMPLEX(WP), INTENT(INOUT) :: C
      END SUBROUTINE CHEMM
 
      MODULE PROCEDURE CHEMM_A2B2C2
 
      END INTERFACE ! LA_HEMM
      PRIVATE CHEMM_A2B2C2
!
!-----Interface for xSYRK and xHERK-----
!
      INTERFACE LA_SYRK
 
      SUBROUTINE SSYRK( UPLO, TRANS, N, K, ALPHA, A, LDA, BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDC, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: C
      END SUBROUTINE SSYRK
 
      SUBROUTINE CSYRK( UPLO, TRANS, N, K, ALPHA, A, LDA, BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDC, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: C
      END SUBROUTINE CSYRK
 
      MODULE PROCEDURE SSYRK_A2C2
      MODULE PROCEDURE CSYRK_A2C2
 
      END INTERFACE ! LA_SYRK
      PRIVATE SSYRK_A2C2, CSYRK_A2C2
 
      INTERFACE LA_HERK
 
      SUBROUTINE CHERK( UPLO, TRANS, N, K, ALPHA, A, LDA, BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDC, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: C
      END SUBROUTINE CHERK
 
      MODULE PROCEDURE CHERK_A2C2
 
      END INTERFACE ! LA_HERK
      PRIVATE CHERK_A2C2
!
!-----Interface for xSYR2K and xHER2K-----
!
      INTERFACE LA_SYR2K
 
      SUBROUTINE SSYR2K( UPLO, TRANS, N, K, ALPHA, A, LDA, B, LDB,      &
     &                   BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, B
         REAL(WP), INTENT(INOUT) :: C
      END SUBROUTINE SSYR2K
 
      SUBROUTINE CSYR2K( UPLO, TRANS, N, K, ALPHA, A, LDA, B, LDB,      &
     &                   BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, B
         COMPLEX(WP), INTENT(INOUT) :: C
      END SUBROUTINE CSYR2K
 
      MODULE PROCEDURE SSYR2K_A2B2C2
      MODULE PROCEDURE CSYR2K_A2B2C2
 
      END INTERFACE ! LA_SYR2K
      PRIVATE SSYR2K_A2B2C2, CSYR2K_A2B2C2
 
      INTERFACE LA_HER2K
 
      SUBROUTINE CHER2K( UPLO, TRANS, N, K, ALPHA, A, LDA, B, LDB,      &
     &                   BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: BETA
         COMPLEX(WP), INTENT(IN) :: A, B
         COMPLEX(WP), INTENT(INOUT) :: C
      END SUBROUTINE CHER2K
 
      MODULE PROCEDURE CHER2K_A2B2C2
 
      END INTERFACE ! LA_HER2K
      PRIVATE CHER2K_A2B2C2
!
!-----Interface for xTRMM-----
!
      INTERFACE LA_TRMM
 
      SUBROUTINE STRMM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A, LDA,  &
     &                  B, LDB )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: DIAG, SIDE, TRANSA, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, M, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: B
      END SUBROUTINE STRMM
 
      SUBROUTINE CTRMM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A, LDA,  &
     &                  B, LDB )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: DIAG, SIDE, TRANSA, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: B
      END SUBROUTINE CTRMM
 
      MODULE PROCEDURE STRMM_A2B2
      MODULE PROCEDURE CTRMM_A2B2
 
      END INTERFACE ! LA_TRMM
      PRIVATE STRMM_A2B2, CTRMM_A2B2
!
!-----Interface for xTRSM-----
!
      INTERFACE LA_TRSM
 
      SUBROUTINE STRSM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A, LDA,  &
     &                  B, LDB )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: DIAG, SIDE, TRANSA, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, M, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: B
      END SUBROUTINE STRSM
 
      SUBROUTINE CTRSM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A, LDA,  &
     &                  B, LDB )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: DIAG, SIDE, TRANSA, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: B
      END SUBROUTINE CTRSM
 
      MODULE PROCEDURE STRSM_A2B2
      MODULE PROCEDURE CTRSM_A2B2
 
      END INTERFACE ! LA_TRSM
      PRIVATE STRSM_A2B2, CTRSM_A2B2
 
      CONTAINS
 
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!! Natural interfaces !!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!-----Natural interface for xGEMM-----
!
      SUBROUTINE SGEMM_A2B2C2( TRANSA, TRANSB, M, N, K, ALPHA, A, LDA,  &
     &                         B, LDB, BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANSA, TRANSB
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), B(LDB,*)
         REAL(WP), INTENT(INOUT) :: C(LDC,*)
         CALL SGEMM( TRANSA, TRANSB, M, N, K, ALPHA, A(1,1), LDA,       &
     &               B(1,1), LDB, BETA, C(1,1), LDC )
      END SUBROUTINE SGEMM_A2B2C2
 
      SUBROUTINE CGEMM_A2B2C2( TRANSA, TRANSB, M, N, K, ALPHA, A, LDA,  &
     &                         B, LDB, BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANSA, TRANSB
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), B(LDB,*)
         COMPLEX(WP), INTENT(INOUT) :: C(LDC,*)
         CALL CGEMM( TRANSA, TRANSB, M, N, K, ALPHA, A(1,1), LDA,       &
     &               B(1,1), LDB, BETA, C(1,1), LDC )
      END SUBROUTINE CGEMM_A2B2C2
!
!-----Natural interface for xSYMM and xHEMM-----
!
      SUBROUTINE SSYMM_A2B2C2( SIDE, UPLO, M, N, ALPHA, A, LDA, B, LDB, &
     &                  BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: SIDE, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, LDC, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), B(LDB,*)
         REAL(WP), INTENT(INOUT) :: C(LDC,*)
         CALL SSYMM( SIDE, UPLO, M, N, ALPHA, A(1,1), LDA, B(1,1), LDB, &
     &               BETA, C(1,1), LDC )
      END SUBROUTINE SSYMM_A2B2C2
 
      SUBROUTINE CSYMM_A2B2C2( SIDE, UPLO, M, N, ALPHA, A, LDA, B, LDB, &
     &                  BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: SIDE, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, LDC, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), B(LDB,*)
         COMPLEX(WP), INTENT(INOUT) :: C(LDC,*)
         CALL CSYMM( SIDE, UPLO, M, N, ALPHA, A(1,1), LDA, B(1,1), LDB, &
     &               BETA, C(1,1), LDC )
      END SUBROUTINE CSYMM_A2B2C2
 
      SUBROUTINE CHEMM_A2B2C2( SIDE, UPLO, M, N, ALPHA, A, LDA, B, LDB, &
     &                  BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: SIDE, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, LDC, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), B(LDB,*)
         COMPLEX(WP), INTENT(INOUT) :: C(LDC,*)
         CALL CHEMM( SIDE, UPLO, M, N, ALPHA, A(1,1), LDA, B(1,1), LDB, &
     &               BETA, C(1,1), LDC )
      END SUBROUTINE CHEMM_A2B2C2
!
!-----Natural interface for xSYRK and xHERK-----
!
      SUBROUTINE SSYRK_A2C2( UPLO, TRANS, N, K, ALPHA, A, LDA, BETA,    &
     &                       C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDC, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: C(LDC,*)
         CALL SSYRK( UPLO, TRANS, N, K, ALPHA, A(1,1), LDA, BETA,       &
     &               C(1,1), LDC )
      END SUBROUTINE SSYRK_A2C2
 
      SUBROUTINE CSYRK_A2C2( UPLO, TRANS, N, K, ALPHA, A, LDA, BETA,    &
     &                       C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDC, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: C(LDC,*)
         CALL CSYRK( UPLO, TRANS, N, K, ALPHA, A(1,1), LDA, BETA,       &
     &               C(1,1), LDC )
      END SUBROUTINE CSYRK_A2C2
 
      SUBROUTINE CHERK_A2C2( UPLO, TRANS, N, K, ALPHA, A, LDA, BETA,    &
     &                       C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDC, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: C(LDC,*)
         CALL CHERK( UPLO, TRANS, N, K, ALPHA, A(1,1), LDA, BETA,       &
     &               C(1,1), LDC )
      END SUBROUTINE CHERK_A2C2
!
!-----Natural interface for xSYR2K and xHER2K-----
!
      SUBROUTINE SSYR2K_A2B2C2( UPLO, TRANS, N, K, ALPHA, A, LDA,       &
     &                          B, LDB, BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), B(LDB,*)
         REAL(WP), INTENT(INOUT) :: C(LDC,*)
         CALL SSYR2K( UPLO, TRANS, N, K, ALPHA, A(1,1), LDA,            &
     &                B(1,1), LDB, BETA, C(1,1), LDC )
      END SUBROUTINE SSYR2K_A2B2C2
 
      SUBROUTINE CSYR2K_A2B2C2( UPLO, TRANS, N, K, ALPHA, A, LDA,       &
     &                          B, LDB, BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), B(LDB,*)
         COMPLEX(WP), INTENT(INOUT) :: C(LDC,*)
         CALL CSYR2K( UPLO, TRANS, N, K, ALPHA, A(1,1), LDA,            &
     &                B(1,1), LDB, BETA, C(1,1), LDC )
      END SUBROUTINE CSYR2K_A2B2C2
 
      SUBROUTINE CHER2K_A2B2C2( UPLO, TRANS, N, K, ALPHA, A, LDA,       &
     &                          B, LDB, BETA, C, LDC )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS, UPLO
         INTEGER, INTENT(IN) :: K, LDA, LDB, LDC, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), B(LDB,*)
         COMPLEX(WP), INTENT(INOUT) :: C(LDC,*)
         CALL CHER2K( UPLO, TRANS, N, K, ALPHA, A(1,1), LDA,            &
     &                B(1,1), LDB, BETA, C(1,1), LDC )
      END SUBROUTINE CHER2K_A2B2C2
!
!-----Natural interface for xTRMM-----
!
      SUBROUTINE STRMM_A2B2( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA,     &
     &                       A, LDA, B, LDB )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: DIAG, SIDE, TRANSA, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, M, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: B(LDB,*)
         CALL STRMM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A(1,1), LDA,&
     &               B(1,1), LDB )
      END SUBROUTINE STRMM_A2B2
 
      SUBROUTINE CTRMM_A2B2( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA,     &
     &                       A, LDA, B, LDB )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: DIAG, SIDE, TRANSA, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: B(LDB,*)
         CALL CTRMM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A(1,1), LDA,&
     &               B(1,1), LDB )
      END SUBROUTINE CTRMM_A2B2
!
!-----Natural interface for xTRSM-----
!
      SUBROUTINE STRSM_A2B2( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA,     &
     &                       A, LDA, B, LDB )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: DIAG, SIDE, TRANSA, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, M, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: B(LDB,*)
         CALL STRSM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A(1,1), LDA,&
     &               B(1,1), LDB )
      END SUBROUTINE STRSM_A2B2
 
      SUBROUTINE CTRSM_A2B2( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA,     &
     &                       A, LDA, B, LDB )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: DIAG, SIDE, TRANSA, UPLO
         INTEGER, INTENT(IN) :: LDA, LDB, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: B(LDB,*)
         CALL CTRSM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A(1,1), LDA,&
     &               B(1,1), LDB )
      END SUBROUTINE CTRSM_A2B2
 
      END MODULE LA_BLAS3
