      MODULE LA_BLAS2
!
!  -- Generic interfaces to the Fortran 77 Level 2 BLAS --
!     E. Anderson
!     April 10, 2002
!
!  Purpose
!  =======
!
!  This module defines generic interfaces for the FORTRAN 77 Level 2
!  BLAS using the prefixes S, D, C, and Z.  Two interfaces are defined
!  for each type-specific BLAS routine: the "point" interface, in which
!  the array arguments are defined as scalars, and the "natural"
!  interface, in which the array arguments are defined as in the
!  subroutine or function declaration.  The point interface in this
!  implementation references the standard BLAS with no additional
!  overhead.  It is accessed by indexing, as in
!
!     CALL LA_GEMV( 'N', N, N-J, -ONE, A( 1, J+1 ), LDA, WORK( J+1 ), 1,
!    $              ONE, A( 1, J ), 1 )
!
!  The natural interface is specified as a wrapper to the point
!  interface and carries the overhead of an extra subroutine call.
!  The natural interface is used for a call without indexing, such as
!
!     CALL LA_GEMV( 'N', M, N, ALPHA, A, LDA, X, INCX, BETA, Y, INCY )
!
!  If only a small number of BLAS are used, it may be desired to include
!  only the necessary interfaces in the calling subprogram, for example,
!
!     USE LA_BLAS2, ONLY: LA_GER, LA_TRSV
!
!-----------------------------------------------------------------------
!
!!!!!!!!!!!! ============ !!!!!!!!!!!!
!!!!!!!!!!!! Level 2 BLAS !!!!!!!!!!!!
!!!!!!!!!!!! ============ !!!!!!!!!!!!
!
!-----Interface for xGBMV-----
!
      INTERFACE LA_GBMV

      SUBROUTINE SGBMV( TRANS, M, N, KL, KU, ALPHA, A, LDA, X, INCX,    &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, KL, KU, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE SGBMV

      SUBROUTINE DGBMV( TRANS, M, N, KL, KU, ALPHA, A, LDA, X, INCX,    &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, KL, KU, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE DGBMV

      SUBROUTINE CGBMV( TRANS, M, N, KL, KU, ALPHA, A, LDA, X, INCX,    &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, KL, KU, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CGBMV

      SUBROUTINE ZGBMV( TRANS, M, N, KL, KU, ALPHA, A, LDA, X, INCX,    &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, KL, KU, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZGBMV

      MODULE PROCEDURE SGBMV_A2X1Y1
      MODULE PROCEDURE DGBMV_A2X1Y1
      MODULE PROCEDURE CGBMV_A2X1Y1
      MODULE PROCEDURE ZGBMV_A2X1Y1

      END INTERFACE ! LA_GBMV
      PRIVATE SGBMV_A2X1Y1, DGBMV_A2X1Y1, CGBMV_A2X1Y1, ZGBMV_A2X1Y1
!
!-----Interface for xGEMV-----
!
      INTERFACE LA_GEMV

      SUBROUTINE SGEMV( TRANS, M, N, ALPHA, A, LDA, X, INCX,            &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE SGEMV

      SUBROUTINE DGEMV( TRANS, M, N, ALPHA, A, LDA, X, INCX,            &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE DGEMV

      SUBROUTINE CGEMV( TRANS, M, N, ALPHA, A, LDA, X, INCX,            &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CGEMV

      SUBROUTINE ZGEMV( TRANS, M, N, ALPHA, A, LDA, X, INCX,            &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZGEMV

      MODULE PROCEDURE SGEMV_A2X1Y1
      MODULE PROCEDURE DGEMV_A2X1Y1
      MODULE PROCEDURE CGEMV_A2X1Y1
      MODULE PROCEDURE ZGEMV_A2X1Y1

      END INTERFACE ! LA_GEMV
      PRIVATE SGEMV_A2X1Y1, DGEMV_A2X1Y1, CGEMV_A2X1Y1, ZGEMV_A2X1Y1
!
!-----Interface for xGER, xGERC, and xGERU-----
!
      INTERFACE LA_GER

      SUBROUTINE SGER( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X, Y
         REAL(WP), INTENT(INOUT) :: A
      END SUBROUTINE SGER

      SUBROUTINE DGER( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X, Y
         REAL(WP), INTENT(INOUT) :: A
      END SUBROUTINE DGER

      MODULE PROCEDURE SGER_X1Y1A2
      MODULE PROCEDURE DGER_X1Y1A2

      END INTERFACE ! LA_GER
      PRIVATE SGER_X1Y1A2, DGER_X1Y1A2

      INTERFACE LA_GERC

      SUBROUTINE CGERC( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X, Y
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE CGERC

      SUBROUTINE ZGERC( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X, Y
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE ZGERC

      MODULE PROCEDURE CGERC_X1Y1A2
      MODULE PROCEDURE ZGERC_X1Y1A2

      END INTERFACE ! LA_GERC
      PRIVATE CGERC_X1Y1A2, ZGERC_X1Y1A2

      INTERFACE LA_GERU

      SUBROUTINE CGERU( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X, Y
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE CGERU

      SUBROUTINE ZGERU( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X, Y
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE ZGERU

      MODULE PROCEDURE CGERU_X1Y1A2
      MODULE PROCEDURE ZGERU_X1Y1A2

      END INTERFACE ! LA_GERU
      PRIVATE CGERU_X1Y1A2, ZGERU_X1Y1A2
!
!-----Interface for xSBMV and xHBMV-----
!     (CSBMV is an LAPACK extension)
!
      INTERFACE LA_SBMV

      SUBROUTINE SSBMV( UPLO, N, K, ALPHA, A, LDA, X, INCX,             &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE SSBMV

      SUBROUTINE DSBMV( UPLO, N, K, ALPHA, A, LDA, X, INCX,             &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE DSBMV

      SUBROUTINE CSBMV( UPLO, N, K, ALPHA, A, LDA, X, INCX,             &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CSBMV

      SUBROUTINE ZSBMV( UPLO, N, K, ALPHA, A, LDA, X, INCX,             &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZSBMV

      MODULE PROCEDURE SSBMV_A2X1Y1
      MODULE PROCEDURE DSBMV_A2X1Y1
      MODULE PROCEDURE CSBMV_A2X1Y1
      MODULE PROCEDURE ZSBMV_A2X1Y1

      END INTERFACE ! LA_SBMV
      PRIVATE SSBMV_A2X1Y1, DSBMV_A2X1Y1, CSBMV_A2X1Y1, ZSBMV_A2X1Y1

      INTERFACE LA_HBMV

      SUBROUTINE CHBMV( UPLO, N, K, ALPHA, A, LDA, X, INCX,             &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CHBMV

      SUBROUTINE ZHBMV( UPLO, N, K, ALPHA, A, LDA, X, INCX,             &
     &                  BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZHBMV

      MODULE PROCEDURE CHBMV_A2X1Y1
      MODULE PROCEDURE ZHBMV_A2X1Y1

      END INTERFACE ! LA_HBMV
      PRIVATE CHBMV_A2X1Y1, ZHBMV_A2X1Y1
!
!-----Interface for xSPMV and xHPMV-----
!     (CSPMV is an LAPACK extension)
!
      INTERFACE LA_SPMV

      SUBROUTINE SSPMV( UPLO, N, ALPHA, AP, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: AP, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE SSPMV

      SUBROUTINE DSPMV( UPLO, N, ALPHA, AP, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: AP, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE DSPMV

      SUBROUTINE CSPMV( UPLO, N, ALPHA, AP, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: AP, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CSPMV

      SUBROUTINE ZSPMV( UPLO, N, ALPHA, AP, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: AP, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZSPMV

      MODULE PROCEDURE SSPMV_A1X1Y1
      MODULE PROCEDURE DSPMV_A1X1Y1
      MODULE PROCEDURE CSPMV_A1X1Y1
      MODULE PROCEDURE ZSPMV_A1X1Y1

      END INTERFACE ! LA_SPMV
      PRIVATE SSPMV_A1X1Y1, DSPMV_A1X1Y1, CSPMV_A1X1Y1, ZSPMV_A1X1Y1

      INTERFACE LA_HPMV

      SUBROUTINE CHPMV( UPLO, N, ALPHA, AP, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: AP, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CHPMV

      SUBROUTINE ZHPMV( UPLO, N, ALPHA, AP, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: AP, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZHPMV

      MODULE PROCEDURE CHPMV_A1X1Y1
      MODULE PROCEDURE ZHPMV_A1X1Y1

      END INTERFACE ! LA_HPMV
      PRIVATE CHPMV_A1X1Y1, ZHPMV_A1X1Y1
!
!-----Interface for xSPR and xHPR-----
!     (CSPR is an LAPACK extension)
!
      INTERFACE LA_SPR

      SUBROUTINE SSPR( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X
         REAL(WP), INTENT(INOUT) :: AP
      END SUBROUTINE SSPR

      SUBROUTINE DSPR( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X
         REAL(WP), INTENT(INOUT) :: AP
      END SUBROUTINE DSPR

      SUBROUTINE CSPR( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(INOUT) :: AP
      END SUBROUTINE CSPR

      SUBROUTINE ZSPR( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(INOUT) :: AP
      END SUBROUTINE ZSPR

      MODULE PROCEDURE SSPR_X1A1
      MODULE PROCEDURE DSPR_X1A1
      MODULE PROCEDURE CSPR_X1A1
      MODULE PROCEDURE ZSPR_X1A1

      END INTERFACE ! LA_SPR
      PRIVATE SSPR_X1A1, DSPR_X1A1, CSPR_X1A1, ZSPR_X1A1

      INTERFACE LA_HPR

      SUBROUTINE CHPR( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(INOUT) :: AP
      END SUBROUTINE CHPR

      SUBROUTINE ZHPR( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(INOUT) :: AP
      END SUBROUTINE ZHPR

      MODULE PROCEDURE CHPR_X1A1
      MODULE PROCEDURE ZHPR_X1A1

      END INTERFACE ! LA_HPR
      PRIVATE CHPR_X1A1, ZHPR_X1A1
!
!-----Interface for xSPR2 and xHPR2-----
!
      INTERFACE LA_SPR2

      SUBROUTINE SSPR2( UPLO, N, ALPHA, X, INCX, Y, INCY, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X, Y
         REAL(WP), INTENT(INOUT) :: AP
      END SUBROUTINE SSPR2

      SUBROUTINE DSPR2( UPLO, N, ALPHA, X, INCX, Y, INCY, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X, Y
         REAL(WP), INTENT(INOUT) :: AP
      END SUBROUTINE DSPR2

      MODULE PROCEDURE SSPR2_X1Y1A1
      MODULE PROCEDURE DSPR2_X1Y1A1

      END INTERFACE ! LA_SPR2
      PRIVATE SSPR2_X1Y1A1, DSPR2_X1Y1A1

      INTERFACE LA_HPR2

      SUBROUTINE CHPR2( UPLO, N, ALPHA, X, INCX, Y, INCY, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X, Y
         COMPLEX(WP), INTENT(INOUT) :: AP
      END SUBROUTINE CHPR2

      SUBROUTINE ZHPR2( UPLO, N, ALPHA, X, INCX, Y, INCY, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X, Y
         COMPLEX(WP), INTENT(INOUT) :: AP
      END SUBROUTINE ZHPR2

      MODULE PROCEDURE CHPR2_X1Y1A1
      MODULE PROCEDURE ZHPR2_X1Y1A1

      END INTERFACE ! LA_HPR2
      PRIVATE CHPR2_X1Y1A1, ZHPR2_X1Y1A1
!
!-----Interface for xSYMV and xHEMV-----
!     (CSYMV is an LAPACK extension)
!
      INTERFACE LA_SYMV

      SUBROUTINE SSYMV( UPLO, N, ALPHA, A, LDA, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE SSYMV

      SUBROUTINE DSYMV( UPLO, N, ALPHA, A, LDA, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE DSYMV

      SUBROUTINE CSYMV( UPLO, N, ALPHA, A, LDA, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CSYMV

      SUBROUTINE ZSYMV( UPLO, N, ALPHA, A, LDA, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZSYMV

      MODULE PROCEDURE SSYMV_A2X1Y1
      MODULE PROCEDURE DSYMV_A2X1Y1
      MODULE PROCEDURE CSYMV_A2X1Y1
      MODULE PROCEDURE ZSYMV_A2X1Y1

      END INTERFACE ! LA_SYMV
      PRIVATE SSYMV_A2X1Y1, DSYMV_A2X1Y1, CSYMV_A2X1Y1, ZSYMV_A2X1Y1

      INTERFACE LA_HEMV

      SUBROUTINE CHEMV( UPLO, N, ALPHA, A, LDA, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CHEMV

      SUBROUTINE ZHEMV( UPLO, N, ALPHA, A, LDA, X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZHEMV

      MODULE PROCEDURE CHEMV_A2X1Y1
      MODULE PROCEDURE ZHEMV_A2X1Y1

      END INTERFACE ! LA_HEMV
      PRIVATE CHEMV_A2X1Y1, ZHEMV_A2X1Y1
!
!-----Interface for xSYR and xHER-----
!     (CSYR is an LAPACK extension)
!
      INTERFACE LA_SYR

      SUBROUTINE SSYR( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X
         REAL(WP), INTENT(INOUT) :: A
      END SUBROUTINE SSYR

      SUBROUTINE DSYR( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X
         REAL(WP), INTENT(INOUT) :: A
      END SUBROUTINE DSYR

      SUBROUTINE CSYR( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE CSYR

      SUBROUTINE ZSYR( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE ZSYR

      MODULE PROCEDURE SSYR_X1A2
      MODULE PROCEDURE DSYR_X1A2
      MODULE PROCEDURE CSYR_X1A2
      MODULE PROCEDURE ZSYR_X1A2

      END INTERFACE ! LA_SYR
      PRIVATE SSYR_X1A2, DSYR_X1A2, CSYR_X1A2, ZSYR_X1A2

      INTERFACE LA_HER

      SUBROUTINE CHER( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE CHER

      SUBROUTINE ZHER( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE ZHER

      MODULE PROCEDURE CHER_X1A2
      MODULE PROCEDURE ZHER_X1A2

      END INTERFACE ! LA_HER
      PRIVATE CHER_X1A2, ZHER_X1A2
!
!-----Interface for xSYR2 and xHER2-----
!
      INTERFACE LA_SYR2

      SUBROUTINE SSYR2( UPLO, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X, Y
         REAL(WP), INTENT(INOUT) :: A
      END SUBROUTINE SSYR2

      SUBROUTINE DSYR2( UPLO, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X, Y
         REAL(WP), INTENT(INOUT) :: A
      END SUBROUTINE DSYR2

      MODULE PROCEDURE SSYR2_X1Y1A2
      MODULE PROCEDURE DSYR2_X1Y1A2

      END INTERFACE ! LA_SYR2
      PRIVATE SSYR2_X1Y1A2, DSYR2_X1Y1A2

      INTERFACE LA_HER2

      SUBROUTINE CHER2( UPLO, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X, Y
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE CHER2

      SUBROUTINE ZHER2( UPLO, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X, Y
         COMPLEX(WP), INTENT(INOUT) :: A
      END SUBROUTINE ZHER2

      MODULE PROCEDURE CHER2_X1Y1A2
      MODULE PROCEDURE ZHER2_X1Y1A2

      END INTERFACE ! LA_HER2
      PRIVATE CHER2_X1Y1A2, ZHER2_X1Y1A2
!
!-----Interface for xTBMV-----
!
      INTERFACE LA_TBMV

      SUBROUTINE STBMV( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE STBMV

      SUBROUTINE DTBMV( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE DTBMV

      SUBROUTINE CTBMV( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE CTBMV

      SUBROUTINE ZTBMV( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE ZTBMV

      MODULE PROCEDURE STBMV_A2X1
      MODULE PROCEDURE DTBMV_A2X1
      MODULE PROCEDURE CTBMV_A2X1
      MODULE PROCEDURE ZTBMV_A2X1

      END INTERFACE ! LA_TBMV
      PRIVATE STBMV_A2X1, DTBMV_A2X1, CTBMV_A2X1, ZTBMV_A2X1
!
!-----Interface for xTPMV-----
!
      INTERFACE LA_TPMV

      SUBROUTINE STPMV( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: AP
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE STPMV

      SUBROUTINE DTPMV( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: AP
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE DTPMV

      SUBROUTINE CTPMV( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: AP
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE CTPMV

      SUBROUTINE ZTPMV( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: AP
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE ZTPMV

      MODULE PROCEDURE STPMV_A1X1
      MODULE PROCEDURE DTPMV_A1X1
      MODULE PROCEDURE CTPMV_A1X1
      MODULE PROCEDURE ZTPMV_A1X1

      END INTERFACE ! LA_TPMV
      PRIVATE STPMV_A1X1, DTPMV_A1X1, CTPMV_A1X1, ZTPMV_A1X1
!
!-----Interface for xTRMV-----
!
      INTERFACE LA_TRMV

      SUBROUTINE STRMV( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE STRMV

      SUBROUTINE DTRMV( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE DTRMV

      SUBROUTINE CTRMV( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE CTRMV

      SUBROUTINE ZTRMV( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE ZTRMV

      MODULE PROCEDURE STRMV_A2X1
      MODULE PROCEDURE DTRMV_A2X1
      MODULE PROCEDURE CTRMV_A2X1
      MODULE PROCEDURE ZTRMV_A2X1

      END INTERFACE ! LA_TRMV
      PRIVATE STRMV_A2X1, DTRMV_A2X1, CTRMV_A2X1, ZTRMV_A2X1
!
!-----Interface for xTBSV-----
!
      INTERFACE LA_TBSV

      SUBROUTINE STBSV( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE STBSV

      SUBROUTINE DTBSV( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE DTBSV

      SUBROUTINE CTBSV( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE CTBSV

      SUBROUTINE ZTBSV( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE ZTBSV

      MODULE PROCEDURE STBSV_A2X1
      MODULE PROCEDURE DTBSV_A2X1
      MODULE PROCEDURE CTBSV_A2X1
      MODULE PROCEDURE ZTBSV_A2X1

      END INTERFACE ! LA_TBSV
      PRIVATE STBSV_A2X1, DTBSV_A2X1, CTBSV_A2X1, ZTBSV_A2X1
!
!-----Interface for xTPSV-----
!
      INTERFACE LA_TPSV

      SUBROUTINE STPSV( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: AP
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE STPSV

      SUBROUTINE DTPSV( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: AP
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE DTPSV

      SUBROUTINE CTPSV( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: AP
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE CTPSV

      SUBROUTINE ZTPSV( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: AP
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE ZTPSV

      MODULE PROCEDURE STPSV_A1X1
      MODULE PROCEDURE DTPSV_A1X1
      MODULE PROCEDURE CTPSV_A1X1
      MODULE PROCEDURE ZTPSV_A1X1

      END INTERFACE ! LA_TPSV
      PRIVATE STPSV_A1X1, DTPSV_A1X1, CTPSV_A1X1, ZTPSV_A1X1
!
!-----Interface for xTRSV-----
!
      INTERFACE LA_TRSV

      SUBROUTINE STRSV( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE STRSV

      SUBROUTINE DTRSV( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: A
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE DTRSV

      SUBROUTINE CTRSV( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE CTRSV

      SUBROUTINE ZTRSV( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: A
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE ZTRSV

      MODULE PROCEDURE STRSV_A2X1
      MODULE PROCEDURE DTRSV_A2X1
      MODULE PROCEDURE CTRSV_A2X1
      MODULE PROCEDURE ZTRSV_A2X1

      END INTERFACE ! LA_TRSV
      PRIVATE STRSV_A2X1, DTRSV_A2X1, CTRSV_A2X1, ZTRSV_A2X1

      CONTAINS

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!! Natural interfaces !!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!-----Natural interface for xGBMV-----
!
      SUBROUTINE SGBMV_A2X1Y1( TRANS, M, N, KL, KU, ALPHA, A, LDA,      &
     &                         X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, KL, KU, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL SGBMV( TRANS, M, N, KL, KU, ALPHA, A(1,1), LDA,           &
     &               X(1), INCX, BETA, Y(1), INCY )
      END SUBROUTINE SGBMV_A2X1Y1

      SUBROUTINE DGBMV_A2X1Y1( TRANS, M, N, KL, KU, ALPHA, A, LDA,      &
     &                         X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, KL, KU, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL DGBMV( TRANS, M, N, KL, KU, ALPHA, A(1,1), LDA,           &
     &               X(1), INCX, BETA, Y(1), INCY )
      END SUBROUTINE DGBMV_A2X1Y1

      SUBROUTINE CGBMV_A2X1Y1( TRANS, M, N, KL, KU, ALPHA, A, LDA,      &
     &                         X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, KL, KU, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CGBMV( TRANS, M, N, KL, KU, ALPHA, A(1,1), LDA,           &
     &               X(1), INCX, BETA, Y(1), INCY )
      END SUBROUTINE CGBMV_A2X1Y1

      SUBROUTINE ZGBMV_A2X1Y1( TRANS, M, N, KL, KU, ALPHA, A, LDA,      &
     &                         X, INCX, BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, KL, KU, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZGBMV( TRANS, M, N, KL, KU, ALPHA, A(1,1), LDA,           &
     &               X(1), INCX, BETA, Y(1), INCY )
      END SUBROUTINE ZGBMV_A2X1Y1
!
!-----Natural interface for xGEMV-----
!
      SUBROUTINE SGEMV_A2X1Y1( TRANS, M, N, ALPHA, A, LDA, X, INCX,     &
     &                         BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL SGEMV( TRANS, M, N, ALPHA, A(1,1), LDA, X(1), INCX,       &
     &               BETA, Y(1), INCY )
      END SUBROUTINE SGEMV_A2X1Y1

      SUBROUTINE DGEMV_A2X1Y1( TRANS, M, N, ALPHA, A, LDA, X, INCX,     &
     &                         BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL DGEMV( TRANS, M, N, ALPHA, A(1,1), LDA, X(1), INCX,       &
     &               BETA, Y(1), INCY )
      END SUBROUTINE DGEMV_A2X1Y1

      SUBROUTINE CGEMV_A2X1Y1( TRANS, M, N, ALPHA, A, LDA, X, INCX,     &
     &                         BETA, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CGEMV( TRANS, M, N, ALPHA, A(1,1), LDA, X(1), INCX,       &
     &               BETA, Y(1), INCY )
      END SUBROUTINE CGEMV_A2X1Y1

      SUBROUTINE ZGEMV_A2X1Y1( TRANS, M, N, ALPHA, A, LDA, X, INCX,     &
     &                         BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: TRANS
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZGEMV( TRANS, M, N, ALPHA, A(1,1), LDA, X(1), INCX,       &
     &               BETA, Y(1), INCY )
      END SUBROUTINE ZGEMV_A2X1Y1
!
!-----Natural interface for xGER, xGERC, and xGERU-----
!
      SUBROUTINE SGER_X1Y1A2( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*), Y(*)
         REAL(WP), INTENT(INOUT) :: A(LDA,*)
         CALL SGER( M, N, ALPHA, X(1), INCX, Y(1), INCY, A(1,1), LDA )
      END SUBROUTINE SGER_X1Y1A2

      SUBROUTINE DGER_X1Y1A2( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*), Y(*)
         REAL(WP), INTENT(INOUT) :: A(LDA,*)
         CALL DGER( M, N, ALPHA, X(1), INCX, Y(1), INCY, A(1,1), LDA )
      END SUBROUTINE DGER_X1Y1A2

      SUBROUTINE CGERC_X1Y1A2( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL CGERC( M, N, ALPHA, X(1), INCX, Y(1), INCY, A(1,1), LDA )
      END SUBROUTINE CGERC_X1Y1A2

      SUBROUTINE ZGERC_X1Y1A2( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL ZGERC( M, N, ALPHA, X(1), INCX, Y(1), INCY, A(1,1), LDA )
      END SUBROUTINE ZGERC_X1Y1A2

      SUBROUTINE CGERU_X1Y1A2( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL CGERU( M, N, ALPHA, X(1), INCX, Y(1), INCY, A(1,1), LDA )
      END SUBROUTINE CGERU_X1Y1A2

      SUBROUTINE ZGERU_X1Y1A2( M, N, ALPHA, X, INCX, Y, INCY, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, M, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL ZGERU( M, N, ALPHA, X(1), INCX, Y(1), INCY, A(1,1), LDA )
      END SUBROUTINE ZGERU_X1Y1A2
!
!-----Natural interface for xSBMV and xHBMV-----
!     (CSBMV is an LAPACK extension)
!
      SUBROUTINE SSBMV_A2X1Y1( UPLO, N, K, ALPHA, A, LDA, X, INCX, BETA,&
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL SSBMV( UPLO, N, K, ALPHA, A(1,1), LDA, X(1), INCX, BETA,  &
     &               Y(1), INCY )
      END SUBROUTINE SSBMV_A2X1Y1

      SUBROUTINE DSBMV_A2X1Y1( UPLO, N, K, ALPHA, A, LDA, X, INCX, BETA,&
     &                         Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL DSBMV( UPLO, N, K, ALPHA, A(1,1), LDA, X(1), INCX, BETA,  &
     &               Y(1), INCY )
      END SUBROUTINE DSBMV_A2X1Y1

      SUBROUTINE CSBMV_A2X1Y1( UPLO, N, K, ALPHA, A, LDA, X, INCX, BETA,&
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CSBMV( UPLO, N, K, ALPHA, A(1,1), LDA, X(1), INCX, BETA,  &
     &               Y(1), INCY )
      END SUBROUTINE CSBMV_A2X1Y1

      SUBROUTINE ZSBMV_A2X1Y1( UPLO, N, K, ALPHA, A, LDA, X, INCX, BETA,&
     &                         Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZSBMV( UPLO, N, K, ALPHA, A(1,1), LDA, X(1), INCX, BETA,  &
     &               Y(1), INCY )
      END SUBROUTINE ZSBMV_A2X1Y1

      SUBROUTINE CHBMV_A2X1Y1( UPLO, N, K, ALPHA, A, LDA, X, INCX, BETA,&
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CHBMV( UPLO, N, K, ALPHA, A(1,1), LDA, X(1), INCX, BETA,  &
     &               Y(1), INCY )
      END SUBROUTINE CHBMV_A2X1Y1

      SUBROUTINE ZHBMV_A2X1Y1( UPLO, N, K, ALPHA, A, LDA, X, INCX,      &
     &                         BETA, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZHBMV( UPLO, N, K, ALPHA, A(1,1), LDA, X(1), INCX, BETA,  &
     &               Y(1), INCY )
      END SUBROUTINE ZHBMV_A2X1Y1
!
!-----Natural interface for xSPMV and xHPMV-----
!     (CSPMV is an LAPACK extension)
!
      SUBROUTINE SSPMV_A1X1Y1( UPLO, N, ALPHA, AP, X, INCX, BETA,       &
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: AP(*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL SSPMV( UPLO, N, ALPHA, AP(1), X(1), INCX, BETA,           &
     &               Y(1), INCY )
      END SUBROUTINE SSPMV_A1X1Y1

      SUBROUTINE DSPMV_A1X1Y1( UPLO, N, ALPHA, AP, X, INCX, BETA,       &
     &                         Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: AP(*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL DSPMV( UPLO, N, ALPHA, AP(1), X(1), INCX, BETA,           &
     &               Y(1), INCY )
      END SUBROUTINE DSPMV_A1X1Y1

      SUBROUTINE CSPMV_A1X1Y1( UPLO, N, ALPHA, AP, X, INCX, BETA,       &
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: AP(*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CSPMV( UPLO, N, ALPHA, AP(1), X(1), INCX, BETA,           &
     &               Y(1), INCY )
      END SUBROUTINE CSPMV_A1X1Y1

      SUBROUTINE ZSPMV_A1X1Y1( UPLO, N, ALPHA, AP, X, INCX, BETA,       &
     &                         Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: AP(*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZSPMV( UPLO, N, ALPHA, AP(1), X(1), INCX, BETA,           &
     &               Y(1), INCY )
      END SUBROUTINE ZSPMV_A1X1Y1

      SUBROUTINE CHPMV_A1X1Y1( UPLO, N, ALPHA, AP, X, INCX, BETA,       &
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: AP(*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CHPMV( UPLO, N, ALPHA, AP(1), X(1), INCX, BETA,           &
     &               Y(1), INCY )
      END SUBROUTINE CHPMV_A1X1Y1

      SUBROUTINE ZHPMV_A1X1Y1( UPLO, N, ALPHA, AP, X, INCX, BETA,       &
     &                         Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: AP(*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZHPMV( UPLO, N, ALPHA, AP(1), X(1), INCX, BETA,           &
     &               Y(1), INCY )
      END SUBROUTINE ZHPMV_A1X1Y1
!
!-----Natural interface for xSPR and xHPR-----
!     (CSPR is an LAPACK extension)
!
      SUBROUTINE SSPR_X1A1( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*)
         REAL(WP), INTENT(INOUT) :: AP(*)
         CALL SSPR( UPLO, N, ALPHA, X(1), INCX, AP(1) )
      END SUBROUTINE SSPR_X1A1

      SUBROUTINE DSPR_X1A1( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*)
         REAL(WP), INTENT(INOUT) :: AP(*)
         CALL DSPR( UPLO, N, ALPHA, X(1), INCX, AP(1) )
      END SUBROUTINE DSPR_X1A1

      SUBROUTINE CSPR_X1A1( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(INOUT) :: AP(*)
         CALL CSPR( UPLO, N, ALPHA, X(1), INCX, AP(1) )
      END SUBROUTINE CSPR_X1A1

      SUBROUTINE ZSPR_X1A1( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(INOUT) :: AP(*)
         CALL ZSPR( UPLO, N, ALPHA, X(1), INCX, AP(1) )
      END SUBROUTINE ZSPR_X1A1

      SUBROUTINE CHPR_X1A1( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(INOUT) :: AP(*)
         CALL CHPR( UPLO, N, ALPHA, X(1), INCX, AP(1) )
      END SUBROUTINE CHPR_X1A1

      SUBROUTINE ZHPR_X1A1( UPLO, N, ALPHA, X, INCX, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(INOUT) :: AP(*)
         CALL ZHPR( UPLO, N, ALPHA, X(1), INCX, AP(1) )
      END SUBROUTINE ZHPR_X1A1
!
!-----Natural interface for xSPR2 and xHPR2-----
!
      SUBROUTINE SSPR2_X1Y1A1( UPLO, N, ALPHA, X, INCX, Y, INCY, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*), Y(*)
         REAL(WP), INTENT(INOUT) :: AP(*)
         CALL SSPR2( UPLO, N, ALPHA, X(1), INCX, Y(1), INCY, AP(1) )
      END SUBROUTINE SSPR2_X1Y1A1

      SUBROUTINE DSPR2_X1Y1A1( UPLO, N, ALPHA, X, INCX, Y, INCY, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*), Y(*)
         REAL(WP), INTENT(INOUT) :: AP(*)
         CALL DSPR2( UPLO, N, ALPHA, X(1), INCX, Y(1), INCY, AP(1) )
      END SUBROUTINE DSPR2_X1Y1A1

      SUBROUTINE CHPR2_X1Y1A1( UPLO, N, ALPHA, X, INCX, Y, INCY, AP )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         COMPLEX(WP), INTENT(INOUT) :: AP(*)
         CALL CHPR2( UPLO, N, ALPHA, X(1), INCX, Y(1), INCY, AP(1) )
      END SUBROUTINE CHPR2_X1Y1A1

      SUBROUTINE ZHPR2_X1Y1A1( UPLO, N, ALPHA, X, INCX, Y, INCY, AP )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         COMPLEX(WP), INTENT(INOUT) :: AP(*)
         CALL ZHPR2( UPLO, N, ALPHA, X(1), INCX, Y(1), INCY, AP(1) )
      END SUBROUTINE ZHPR2_X1Y1A1
!
!-----Natural interface for xSYMV and xHEMV-----
!     (CSYMV is an LAPACK extension)
!
      SUBROUTINE SSYMV_A2X1Y1( UPLO, N, ALPHA, A, LDA, X, INCX, BETA,   &
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL SSYMV( UPLO, N, ALPHA, A(1,1), LDA, X(1), INCX, BETA,     &
     &               Y(1), INCY )
      END SUBROUTINE SSYMV_A2X1Y1

      SUBROUTINE DSYMV_A2X1Y1( UPLO, N, ALPHA, A, LDA, X, INCX, BETA,   &
     &                         Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA, BETA
         REAL(WP), INTENT(IN) :: A(LDA,*), X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL DSYMV( UPLO, N, ALPHA, A(1,1), LDA, X(1), INCX, BETA,     &
     &               Y(1), INCY )
      END SUBROUTINE DSYMV_A2X1Y1

      SUBROUTINE CSYMV_A2X1Y1( UPLO, N, ALPHA, A, LDA, X, INCX, BETA,   &
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CSYMV( UPLO, N, ALPHA, A(1,1), LDA, X(1), INCX, BETA,     &
     &               Y(1), INCY )
      END SUBROUTINE CSYMV_A2X1Y1

      SUBROUTINE ZSYMV_A2X1Y1( UPLO, N, ALPHA, A, LDA, X, INCX, BETA,   &
     &                         Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZSYMV( UPLO, N, ALPHA, A(1,1), LDA, X(1), INCX, BETA,     &
     &               Y(1), INCY )
      END SUBROUTINE ZSYMV_A2X1Y1

      SUBROUTINE CHEMV_A2X1Y1( UPLO, N, ALPHA, A, LDA, X, INCX, BETA,   &
     &                         Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CHEMV( UPLO, N, ALPHA, A(1,1), LDA, X(1), INCX, BETA,     &
     &               Y(1), INCY )
      END SUBROUTINE CHEMV_A2X1Y1

      SUBROUTINE ZHEMV_A2X1Y1( UPLO, N, ALPHA, A, LDA, X, INCX, BETA,   &
     &                         Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, BETA
         COMPLEX(WP), INTENT(IN) :: A(LDA,*), X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZHEMV( UPLO, N, ALPHA, A(1,1), LDA, X(1), INCX, BETA,     &
     &               Y(1), INCY )
      END SUBROUTINE ZHEMV_A2X1Y1
!
!-----Natural interface for xSYR and xHER-----
!     (CSYR is an LAPACK extension)
!
      SUBROUTINE SSYR_X1A2( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*)
         REAL(WP), INTENT(INOUT) :: A(LDA,*)
         CALL SSYR( UPLO, N, ALPHA, X(1), INCX, A(1,1), LDA )
      END SUBROUTINE SSYR_X1A2

      SUBROUTINE DSYR_X1A2( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*)
         REAL(WP), INTENT(INOUT) :: A(LDA,*)
         CALL DSYR( UPLO, N, ALPHA, X(1), INCX, A(1,1), LDA )
      END SUBROUTINE DSYR_X1A2

      SUBROUTINE CSYR_X1A2( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL CSYR( UPLO, N, ALPHA, X(1), INCX, A(1,1), LDA )
      END SUBROUTINE CSYR_X1A2

      SUBROUTINE ZSYR_X1A2( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL ZSYR( UPLO, N, ALPHA, X(1), INCX, A(1,1), LDA )
      END SUBROUTINE ZSYR_X1A2

      SUBROUTINE CHER_X1A2( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL CHER( UPLO, N, ALPHA, X(1), INCX, A(1,1), LDA )
      END SUBROUTINE CHER_X1A2

      SUBROUTINE ZHER_X1A2( UPLO, N, ALPHA, X, INCX, A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL ZHER( UPLO, N, ALPHA, X(1), INCX, A(1,1), LDA )
      END SUBROUTINE ZHER_X1A2
!
!-----Natural interface for xSYR2 and xHER2-----
!
      SUBROUTINE SSYR2_X1Y1A2( UPLO, N, ALPHA, X, INCX, Y, INCY,        &
     &                         A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*), Y(*)
         REAL(WP), INTENT(INOUT) :: A(LDA,*)
         CALL SSYR2( UPLO, N, ALPHA, X(1), INCX, Y(1), INCY,            &
     &               A(1,1), LDA )
      END SUBROUTINE SSYR2_X1Y1A2

      SUBROUTINE DSYR2_X1Y1A2( UPLO, N, ALPHA, X, INCX, Y, INCY,        &
     &                         A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(IN) :: X(*), Y(*)
         REAL(WP), INTENT(INOUT) :: A(LDA,*)
         CALL DSYR2( UPLO, N, ALPHA, X(1), INCX, Y(1), INCY,            &
     &               A(1,1), LDA )
      END SUBROUTINE DSYR2_X1Y1A2

      SUBROUTINE CHER2_X1Y1A2( UPLO, N, ALPHA, X, INCX, Y, INCY,        &
     &                         A, LDA )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL CHER2( UPLO, N, ALPHA, X(1), INCX, Y(1), INCY,            &
     &               A(1,1), LDA )
      END SUBROUTINE CHER2_X1Y1A2

      SUBROUTINE ZHER2_X1Y1A2( UPLO, N, ALPHA, X, INCX, Y, INCY,        &
     &                         A, LDA )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO
         INTEGER, INTENT(IN) :: INCX, INCY, LDA, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         COMPLEX(WP), INTENT(INOUT) :: A(LDA,*)
         CALL ZHER2( UPLO, N, ALPHA, X(1), INCX, Y(1), INCY,            &
     &               A(1,1), LDA )
      END SUBROUTINE ZHER2_X1Y1A2
!
!-----Natural interface for xTBMV-----
!
      SUBROUTINE STBMV_A2X1( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL STBMV( UPLO, TRANS, DIAG, N, K, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE STBMV_A2X1

      SUBROUTINE DTBMV_A2X1( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL DTBMV( UPLO, TRANS, DIAG, N, K, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE DTBMV_A2X1

      SUBROUTINE CTBMV_A2X1( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL CTBMV( UPLO, TRANS, DIAG, N, K, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE CTBMV_A2X1

      SUBROUTINE ZTBMV_A2X1( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL ZTBMV( UPLO, TRANS, DIAG, N, K, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE ZTBMV_A2X1
!
!-----Natural interface for xTPMV-----
!
      SUBROUTINE STPMV_A1X1( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: AP(*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL STPMV( UPLO, TRANS, DIAG, N, AP(1), X(1), INCX )
      END SUBROUTINE STPMV_A1X1

      SUBROUTINE DTPMV_A1X1( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: AP(*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL DTPMV( UPLO, TRANS, DIAG, N, AP(1), X(1), INCX )
      END SUBROUTINE DTPMV_A1X1

      SUBROUTINE CTPMV_A1X1( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: AP(*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL CTPMV( UPLO, TRANS, DIAG, N, AP(1), X(1), INCX )
      END SUBROUTINE CTPMV_A1X1

      SUBROUTINE ZTPMV_A1X1( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: AP(*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL ZTPMV( UPLO, TRANS, DIAG, N, AP(1), X(1), INCX )
      END SUBROUTINE ZTPMV_A1X1
!
!-----Natural interface for xTRMV-----
!
      SUBROUTINE STRMV_A2X1( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL STRMV( UPLO, TRANS, DIAG, N, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE STRMV_A2X1

      SUBROUTINE DTRMV_A2X1( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL DTRMV( UPLO, TRANS, DIAG, N, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE DTRMV_A2X1

      SUBROUTINE CTRMV_A2X1( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL CTRMV( UPLO, TRANS, DIAG, N, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE CTRMV_A2X1

      SUBROUTINE ZTRMV_A2X1( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL ZTRMV( UPLO, TRANS, DIAG, N, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE ZTRMV_A2X1
!
!-----Natural interface for xTBSV-----
!
      SUBROUTINE STBSV_A2X1( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL STBSV( UPLO, TRANS, DIAG, N, K, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE STBSV_A2X1

      SUBROUTINE DTBSV_A2X1( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL DTBSV( UPLO, TRANS, DIAG, N, K, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE DTBSV_A2X1

      SUBROUTINE CTBSV_A2X1( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL CTBSV( UPLO, TRANS, DIAG, N, K, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE CTBSV_A2X1

      SUBROUTINE ZTBSV_A2X1( UPLO, TRANS, DIAG, N, K, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, K, LDA, N
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL ZTBSV( UPLO, TRANS, DIAG, N, K, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE ZTBSV_A2X1
!
!-----Natural interface for xTPSV-----
!
      SUBROUTINE STPSV_A1X1( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: AP(*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL STPSV( UPLO, TRANS, DIAG, N, AP(1), X(1), INCX )
      END SUBROUTINE STPSV_A1X1

      SUBROUTINE DTPSV_A1X1( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: AP(*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL DTPSV( UPLO, TRANS, DIAG, N, AP(1), X(1), INCX )
      END SUBROUTINE DTPSV_A1X1

      SUBROUTINE CTPSV_A1X1( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: AP(*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL CTPSV( UPLO, TRANS, DIAG, N, AP(1), X(1), INCX )
      END SUBROUTINE CTPSV_A1X1

      SUBROUTINE ZTPSV_A1X1( UPLO, TRANS, DIAG, N, AP, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: AP(*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL ZTPSV( UPLO, TRANS, DIAG, N, AP(1), X(1), INCX )
      END SUBROUTINE ZTPSV_A1X1
!
!-----Natural interface for xTRSV-----
!
      SUBROUTINE STRSV_A2X1( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL STRSV( UPLO, TRANS, DIAG, N, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE STRSV_A2X1

      SUBROUTINE DTRSV_A2X1( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         REAL(WP), INTENT(IN) :: A(LDA,*)
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL DTRSV( UPLO, TRANS, DIAG, N, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE DTRSV_A2X1

      SUBROUTINE CTRSV_A2X1( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL CTRSV( UPLO, TRANS, DIAG, N, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE CTRSV_A2X1

      SUBROUTINE ZTRSV_A2X1( UPLO, TRANS, DIAG, N, A, LDA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         CHARACTER(LEN=1), INTENT(IN) :: UPLO, TRANS, DIAG
         INTEGER, INTENT(IN) :: INCX, LDA, N
         COMPLEX(WP), INTENT(IN) :: A(LDA,*)
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL ZTRSV( UPLO, TRANS, DIAG, N, A(1,1), LDA, X(1), INCX )
      END SUBROUTINE ZTRSV_A2X1

      END MODULE LA_BLAS2
