      MODULE LA_BLAS1
!
!  -- Generic interfaces to the Fortran 77 Level 1 BLAS --
!     E. Anderson
!     April 10, 2002
!
!  Purpose
!  =======
!
!  This module defines generic interfaces for the FORTRAN 77 Level 1
!  BLAS using the prefixes S, D, C, and Z.  Two interfaces are defined
!  for each type-specific BLAS routine: the "point" interface, in which
!  the array arguments are defined as scalars, and the "natural"
!  interface, in which the array arguments are defined as in the
!  subroutine or function declaration.  The point interface in this
!  implementation references the standard BLAS with no additional
!  overhead.  It is accessed by indexing, as in
!
!     CALL LA_COPY( N, X(1), 1, X(N+1), 1 )
!
!  The natural interface is specified as a wrapper to the point
!  interface and carries the overhead of an extra subroutine call.
!  The natural interface is used for a call without indexing, such as
!
!     CALL LA_COPY( N, X, 1, Y, 1 )
!
!  If only a small number of BLAS are used, it may be desired to include
!  only the necessary interfaces in the calling subprogram, for example,
!
!     USE LA_BLAS1, ONLY: LA_COPY, LA_SCAL
!
!-----------------------------------------------------------------------
!
!!!!!!!!!!!! ============ !!!!!!!!!!!!
!!!!!!!!!!!! Level 1 BLAS !!!!!!!!!!!!
!!!!!!!!!!!! ============ !!!!!!!!!!!!
!
!-----Interface for IxAMAX-----
!
      INTERFACE LA_IAMAX

      FUNCTION ISAMAX( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER :: ISAMAX
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X
      END FUNCTION ISAMAX

      FUNCTION IDAMAX( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER :: IDAMAX
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X
      END FUNCTION IDAMAX

      FUNCTION ICAMAX( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER :: ICAMAX
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X
      END FUNCTION ICAMAX

      FUNCTION IZAMAX( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER :: IZAMAX
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X
      END FUNCTION IZAMAX

      MODULE PROCEDURE ISAMAX_X1
      MODULE PROCEDURE IDAMAX_X1
      MODULE PROCEDURE ICAMAX_X1
      MODULE PROCEDURE IZAMAX_X1

      END INTERFACE ! LA_IAMAX
      PRIVATE ISAMAX_X1, IDAMAX_X1, ICAMAX_X1, IZAMAX_X1
!
!-----Interface for xASUM-----
!
      INTERFACE LA_ASUM

      FUNCTION SASUM( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SASUM
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X
      END FUNCTION SASUM

      FUNCTION DASUM( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DASUM
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X
      END FUNCTION DASUM

      FUNCTION SCASUM( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SCASUM
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X
      END FUNCTION SCASUM

      FUNCTION DZASUM( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DZASUM
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X
      END FUNCTION DZASUM

      MODULE PROCEDURE SASUM_X1
      MODULE PROCEDURE DASUM_X1
      MODULE PROCEDURE SCASUM_X1
      MODULE PROCEDURE DZASUM_X1

      END INTERFACE ! LA_ASUM
      PRIVATE SASUM_X1, DASUM_X1, SCASUM_X1, DZASUM_X1
!
!-----Interface for xAXPY-----
!
      INTERFACE LA_AXPY

      SUBROUTINE SAXPY( N, ALPHA, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE SAXPY

      SUBROUTINE DAXPY( N, ALPHA, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA, X
         REAL(WP), INTENT(INOUT) :: Y
      END SUBROUTINE DAXPY

      SUBROUTINE CAXPY( N, ALPHA, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE CAXPY

      SUBROUTINE ZAXPY( N, ALPHA, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, X
         COMPLEX(WP), INTENT(INOUT) :: Y
      END SUBROUTINE ZAXPY

      MODULE PROCEDURE SAXPY_X1Y1
      MODULE PROCEDURE DAXPY_X1Y1
      MODULE PROCEDURE CAXPY_X1Y1
      MODULE PROCEDURE ZAXPY_X1Y1

      END INTERFACE ! LA_AXPY
      PRIVATE SAXPY_X1Y1, DAXPY_X1Y1, CAXPY_X1Y1, ZAXPY_X1Y1
!
!-----Interface for xCOPY-----
!
      INTERFACE LA_COPY

      SUBROUTINE SCOPY( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: X
         REAL(WP), INTENT(OUT) :: Y
      END SUBROUTINE SCOPY

      SUBROUTINE DCOPY( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: X
         REAL(WP), INTENT(OUT) :: Y
      END SUBROUTINE DCOPY

      SUBROUTINE CCOPY( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(OUT) :: Y
      END SUBROUTINE CCOPY

      SUBROUTINE ZCOPY( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X
         COMPLEX(WP), INTENT(OUT) :: Y
      END SUBROUTINE ZCOPY

      MODULE PROCEDURE SCOPY_X1Y1
      MODULE PROCEDURE DCOPY_X1Y1
      MODULE PROCEDURE CCOPY_X1Y1
      MODULE PROCEDURE ZCOPY_X1Y1

      END INTERFACE ! LA_COPY
      PRIVATE SCOPY_X1Y1, DCOPY_X1Y1, CCOPY_X1Y1, ZCOPY_X1Y1
!
!-----Interface for xDOT, xDOTC, and xDOTU-----
!
      INTERFACE LA_DOT

      FUNCTION SDOT( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SDOT
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: X, Y
      END FUNCTION SDOT

      FUNCTION DDOT( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DDOT
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: X, Y
      END FUNCTION DDOT

      MODULE PROCEDURE SDOT_X1Y1
      MODULE PROCEDURE DDOT_X1Y1

      END INTERFACE ! LA_DOT
      PRIVATE SDOT_X1Y1, DDOT_X1Y1

      INTERFACE LA_DOTC

      FUNCTION CDOTC( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         COMPLEX(WP) :: CDOTC
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X, Y
      END FUNCTION CDOTC

      FUNCTION ZDOTC( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         COMPLEX(WP) :: ZDOTC
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X, Y
      END FUNCTION ZDOTC

      MODULE PROCEDURE CDOTC_X1Y1
      MODULE PROCEDURE ZDOTC_X1Y1

      END INTERFACE ! LA_DOTC
      PRIVATE CDOTC_X1Y1, ZDOTC_X1Y1

      INTERFACE LA_DOTU

      FUNCTION CDOTU( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         COMPLEX(WP) :: CDOTU
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X, Y
      END FUNCTION CDOTU

      FUNCTION ZDOTU( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         COMPLEX(WP) :: ZDOTU
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X, Y
      END FUNCTION ZDOTU

      MODULE PROCEDURE CDOTU_X1Y1
      MODULE PROCEDURE ZDOTU_X1Y1

      END INTERFACE ! LA_DOTU
      PRIVATE CDOTU_X1Y1, ZDOTU_X1Y1
!
!-----Interface for xNRM2-----
!
      INTERFACE LA_NRM2

      FUNCTION SNRM2( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SNRM2
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X
      END FUNCTION SNRM2

      FUNCTION DNRM2( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DNRM2
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X
      END FUNCTION DNRM2

      FUNCTION SCNRM2( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SCNRM2
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X
      END FUNCTION SCNRM2

      FUNCTION DZNRM2( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DZNRM2
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X
      END FUNCTION DZNRM2

      MODULE PROCEDURE SNRM2_X1
      MODULE PROCEDURE DNRM2_X1
      MODULE PROCEDURE SCNRM2_X1
      MODULE PROCEDURE DZNRM2_X1

      END INTERFACE ! LA_NRM2
      PRIVATE SNRM2_X1, DNRM2_X1, SCNRM2_X1, DZNRM2_X1
!
!-----Interface for xROT-----
!     (CROT and CSROT are LINPACK/LAPACK extensions)
!
      INTERFACE LA_ROT

      SUBROUTINE SROT( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X, Y
         REAL(WP), INTENT(IN) :: C, S
      END SUBROUTINE SROT

      SUBROUTINE DROT( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X, Y
         REAL(WP), INTENT(IN) :: C, S
      END SUBROUTINE DROT

      SUBROUTINE CROT( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X, Y
         REAL(WP), INTENT(IN) :: C
         COMPLEX(WP), INTENT(IN) :: S
      END SUBROUTINE CROT

      SUBROUTINE ZROT( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X, Y
         REAL(WP), INTENT(IN) :: C
         COMPLEX(WP), INTENT(IN) :: S
      END SUBROUTINE ZROT

      SUBROUTINE CSROT( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X, Y
         REAL(WP), INTENT(IN) :: C, S
      END SUBROUTINE CSROT

      SUBROUTINE ZDROT( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X, Y
         REAL(WP), INTENT(IN) :: C, S
      END SUBROUTINE ZDROT

      MODULE PROCEDURE SROT_X1Y1
      MODULE PROCEDURE DROT_X1Y1
      MODULE PROCEDURE CROT_X1Y1
      MODULE PROCEDURE ZROT_X1Y1
      MODULE PROCEDURE CSROT_X1Y1
      MODULE PROCEDURE ZDROT_X1Y1

      END INTERFACE ! LA_ROT
      PRIVATE SROT_X1Y1, DROT_X1Y1, CROT_X1Y1, ZROT_X1Y1
      PRIVATE CSROT_X1Y1, ZDROT_X1Y1
!
!-----Interface for xROTG-----
!     (CROTG is an LAPACK extension)
!
      INTERFACE LA_ROTG

      SUBROUTINE SROTG( A, B, C, S )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP), INTENT(INOUT) :: A, B
         REAL(WP), INTENT(OUT) :: C, S
      END SUBROUTINE SROTG

      SUBROUTINE DROTG( A, B, C, S )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP), INTENT(INOUT) :: A, B
         REAL(WP), INTENT(OUT) :: C, S
      END SUBROUTINE DROTG

      SUBROUTINE CROTG( A, B, C, S )
         USE LA_CONSTANTS32, ONLY : WP
         COMPLEX(WP), INTENT(INOUT) :: A
         COMPLEX(WP), INTENT(IN) :: B
         REAL(WP), INTENT(OUT) :: C
         COMPLEX(WP), INTENT(OUT) :: S
      END SUBROUTINE CROTG

      SUBROUTINE ZROTG( A, B, C, S )
         USE LA_CONSTANTS, ONLY : WP
         COMPLEX(WP), INTENT(INOUT) :: A
         COMPLEX(WP), INTENT(IN) :: B
         REAL(WP), INTENT(OUT) :: C
         COMPLEX(WP), INTENT(OUT) :: S
      END SUBROUTINE ZROTG

      END INTERFACE ! LA_ROTG
!
!-----Interface for xROTM-----
!
      INTERFACE LA_ROTM

      SUBROUTINE SROTM( N, X, INCX, Y, INCY, PARAM )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X, Y
         REAL(WP), INTENT(IN) :: PARAM(5)
      END SUBROUTINE SROTM

      SUBROUTINE DROTM( N, X, INCX, Y, INCY, PARAM )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X, Y
         REAL(WP), INTENT(IN) :: PARAM(5)
      END SUBROUTINE DROTM

      MODULE PROCEDURE SROTM_X1Y1
      MODULE PROCEDURE DROTM_X1Y1

      END INTERFACE ! LA_ROTM
      PRIVATE SROTM_X1Y1, DROTM_X1Y1
!
!-----Interface for xROTMG-----
!
      INTERFACE LA_ROTMG

      SUBROUTINE SROTMG( D1, D2, A, B, PARAM )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP), INTENT(INOUT) :: A, D1, D2
         REAL(WP), INTENT(IN) :: B
         REAL(WP), INTENT(OUT) :: PARAM
      END SUBROUTINE SROTMG

      SUBROUTINE DROTMG( D1, D2, A, B, PARAM )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP), INTENT(INOUT) :: A, D1, D2
         REAL(WP), INTENT(IN) :: B
         REAL(WP), INTENT(OUT) :: PARAM
      END SUBROUTINE DROTMG

      END INTERFACE ! LA_ROTMG
!
!-----Interface for xSCAL-----
!
      INTERFACE LA_SCAL

      SUBROUTINE SSCAL( N, ALPHA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE SSCAL

      SUBROUTINE DSCAL( N, ALPHA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(INOUT) :: X
      END SUBROUTINE DSCAL

      SUBROUTINE CSCAL( N, ALPHA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE CSCAL

      SUBROUTINE ZSCAL( N, ALPHA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE ZSCAL

      SUBROUTINE CSSCAL( N, ALPHA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE CSSCAL

      SUBROUTINE ZDSCAL( N, ALPHA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(INOUT) :: X
      END SUBROUTINE ZDSCAL

      MODULE PROCEDURE SSCAL_X1
      MODULE PROCEDURE DSCAL_X1
      MODULE PROCEDURE CSCAL_X1
      MODULE PROCEDURE ZSCAL_X1
      MODULE PROCEDURE CSSCAL_X1
      MODULE PROCEDURE ZDSCAL_X1

      END INTERFACE ! LA_SCAL
      PRIVATE SSCAL_X1, DSCAL_X1, CSCAL_X1, ZSCAL_X1
      PRIVATE CSSCAL_X1, ZDSCAL_X1
!
!-----Interface for xSWAP-----
!
      INTERFACE LA_SWAP

      SUBROUTINE SSWAP( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X, Y
      END SUBROUTINE SSWAP

      SUBROUTINE DSWAP( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X, Y
      END SUBROUTINE DSWAP

      SUBROUTINE CSWAP( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X, Y
      END SUBROUTINE CSWAP

      SUBROUTINE ZSWAP( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X, Y
      END SUBROUTINE ZSWAP

      MODULE PROCEDURE SSWAP_X1Y1
      MODULE PROCEDURE DSWAP_X1Y1
      MODULE PROCEDURE CSWAP_X1Y1
      MODULE PROCEDURE ZSWAP_X1Y1

      END INTERFACE ! LA_SWAP
      PRIVATE SSWAP_X1Y1, DSWAP_X1Y1, CSWAP_X1Y1, ZSWAP_X1Y1

      CONTAINS
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!! Natural interfaces !!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!-----Natural interface for IxAMAX-----
!
      FUNCTION ISAMAX_X1( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER :: ISAMAX_X1
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X(*)
         ISAMAX_X1 = ISAMAX( N, X(1), INCX )
      END FUNCTION ISAMAX_X1

      FUNCTION IDAMAX_X1( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER :: IDAMAX_X1
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X(*)
         IDAMAX_X1 = IDAMAX( N, X(1), INCX )
      END FUNCTION IDAMAX_X1

      FUNCTION ICAMAX_X1( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER :: ICAMAX_X1
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X(*)
         ICAMAX_X1 = ICAMAX( N, X(1), INCX )
      END FUNCTION ICAMAX_X1

      FUNCTION IZAMAX_X1( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER :: IZAMAX_X1
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X(*)
         IZAMAX_X1 = IZAMAX( N, X(1), INCX )
      END FUNCTION IZAMAX_X1
!
!-----Natural interface for xASUM-----
!
      FUNCTION SASUM_X1( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SASUM_X1
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X(*)
         SASUM_X1 = SASUM( N, X(1), INCX )
      END FUNCTION SASUM_X1

      FUNCTION DASUM_X1( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DASUM_X1
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X(*)
         DASUM_X1 = DASUM( N, X(1), INCX )
      END FUNCTION DASUM_X1

      FUNCTION SCASUM_X1( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SCASUM_X1
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X(*)
         SCASUM_X1 = SCASUM( N, X(1), INCX )
      END FUNCTION SCASUM_X1

      FUNCTION DZASUM_X1( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DZASUM_X1
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X(*)
         DZASUM_X1 = DZASUM( N, X(1), INCX )
      END FUNCTION DZASUM_X1
!
!-----Natural interface for xAXPY-----
!
      SUBROUTINE SAXPY_X1Y1( N, ALPHA, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA, X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL SAXPY( N, ALPHA, X(1), INCX, Y(1), INCY )
      END SUBROUTINE SAXPY_X1Y1

      SUBROUTINE DAXPY_X1Y1( N, ALPHA, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: ALPHA, X(*)
         REAL(WP), INTENT(INOUT) :: Y(*)
         CALL DAXPY( N, ALPHA, X(1), INCX, Y(1), INCY )
      END SUBROUTINE DAXPY_X1Y1

      SUBROUTINE CAXPY_X1Y1( N, ALPHA, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL CAXPY( N, ALPHA, X(1), INCX, Y(1), INCY )
      END SUBROUTINE CAXPY_X1Y1

      SUBROUTINE ZAXPY_X1Y1( N, ALPHA, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: ALPHA, X(*)
         COMPLEX(WP), INTENT(INOUT) :: Y(*)
         CALL ZAXPY( N, ALPHA, X(1), INCX, Y(1), INCY )
      END SUBROUTINE ZAXPY_X1Y1
!
!-----Natural interface for xCOPY-----
!
      SUBROUTINE SCOPY_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: X(*)
         REAL(WP), INTENT(OUT) :: Y(*)
         CALL SCOPY( N, X(1), INCX, Y(1), INCY )
      END SUBROUTINE SCOPY_X1Y1

      SUBROUTINE DCOPY_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: X(*)
         REAL(WP), INTENT(OUT) :: Y(*)
         CALL DCOPY( N, X(1), INCX, Y(1), INCY )
      END SUBROUTINE DCOPY_X1Y1

      SUBROUTINE CCOPY_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(OUT) :: Y(*)
         CALL CCOPY( N, X(1), INCX, Y(1), INCY )
      END SUBROUTINE CCOPY_X1Y1

      SUBROUTINE ZCOPY_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X(*)
         COMPLEX(WP), INTENT(OUT) :: Y(*)
         CALL ZCOPY( N, X(1), INCX, Y(1), INCY )
      END SUBROUTINE ZCOPY_X1Y1
!
!-----Natural interface for xDOT, xDOTC, and xDOTU-----
!
      FUNCTION SDOT_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SDOT_X1Y1
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: X(*), Y(*)
         SDOT_X1Y1 = SDOT( N, X(1), INCX, Y(1), INCY )
      END FUNCTION SDOT_X1Y1

      FUNCTION DDOT_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DDOT_X1Y1
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(IN) :: X(*), Y(*)
         DDOT_X1Y1 = DDOT( N, X(1), INCX, Y(1), INCY )
      END FUNCTION DDOT_X1Y1

      FUNCTION CDOTC_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         COMPLEX(WP) :: CDOTC_X1Y1
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         CDOTC_X1Y1 = CDOTC( N, X(1), INCX, Y(1), INCY )
      END FUNCTION CDOTC_X1Y1

      FUNCTION ZDOTC_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         COMPLEX(WP) :: ZDOTC_X1Y1
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         ZDOTC_X1Y1 = ZDOTC( N, X(1), INCX, Y(1), INCY )
      END FUNCTION ZDOTC_X1Y1

      FUNCTION CDOTU_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         COMPLEX(WP) :: CDOTU_X1Y1
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         CDOTU_X1Y1 = CDOTU( N, X(1), INCX, Y(1), INCY )
      END FUNCTION CDOTU_X1Y1

      FUNCTION ZDOTU_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         COMPLEX(WP) :: ZDOTU_X1Y1
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(IN) :: X(*), Y(*)
         ZDOTU_X1Y1 = ZDOTU( N, X(1), INCX, Y(1), INCY )
      END FUNCTION ZDOTU_X1Y1
!
!-----Natural interface for xNRM2-----
!
      FUNCTION SNRM2_X1( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SNRM2_X1
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X(*)
         SNRM2_X1 = SNRM2( N, X(1), INCX )
      END FUNCTION SNRM2_X1

      FUNCTION DNRM2_X1( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DNRM2_X1
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: X(*)
         DNRM2_X1 = DNRM2( N, X(1), INCX )
      END FUNCTION DNRM2_X1

      FUNCTION SCNRM2_X1( N, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         REAL(WP) :: SCNRM2_X1
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X(*)
         SCNRM2_X1 = SCNRM2( N, X(1), INCX )
      END FUNCTION SCNRM2_X1

      FUNCTION DZNRM2_X1( N, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         REAL(WP) :: DZNRM2_X1
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: X(*)
         DZNRM2_X1 = DZNRM2( N, X(1), INCX )
      END FUNCTION DZNRM2_X1
!
!-----Natural interface for xROT-----
!     (CROT is an LAPACK extension)
!
      SUBROUTINE SROT_X1Y1( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X(*), Y(*)
         REAL(WP), INTENT(IN) :: C, S
         CALL SROT( N, X(1), INCX, Y(1), INCY, C, S )
      END SUBROUTINE SROT_X1Y1

      SUBROUTINE DROT_X1Y1( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X(*), Y(*)
         REAL(WP), INTENT(IN) :: C, S
         CALL DROT( N, X(1), INCX, Y(1), INCY, C, S )
      END SUBROUTINE DROT_X1Y1

      SUBROUTINE CROT_X1Y1( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X(*), Y(*)
         REAL(WP), INTENT(IN) :: C
         COMPLEX(WP), INTENT(IN) :: S
         CALL CROT( N, X(1), INCX, Y(1), INCY, C, S )
      END SUBROUTINE CROT_X1Y1

      SUBROUTINE ZROT_X1Y1( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X(*), Y(*)
         REAL(WP), INTENT(IN) :: C
         COMPLEX(WP), INTENT(IN) :: S
         CALL ZROT( N, X(1), INCX, Y(1), INCY, C, S )
      END SUBROUTINE ZROT_X1Y1

      SUBROUTINE CSROT_X1Y1( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X(*), Y(*)
         REAL(WP), INTENT(IN) :: C, S
         CALL CSROT( N, X(1), INCX, Y(1), INCY, C, S )
      END SUBROUTINE CSROT_X1Y1

      SUBROUTINE ZDROT_X1Y1( N, X, INCX, Y, INCY, C, S )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X(*), Y(*)
         REAL(WP), INTENT(IN) :: C, S
         CALL ZDROT( N, X(1), INCX, Y(1), INCY, C, S )
      END SUBROUTINE ZDROT_X1Y1
!
!-----Natural interface for xROTM-----
!
      SUBROUTINE SROTM_X1Y1( N, X, INCX, Y, INCY, PARAM )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X(*), Y(*)
         REAL(WP), INTENT(IN) :: PARAM(5)
         CALL SROTM( N, X(1), INCX, Y(1), INCY, PARAM )
      END SUBROUTINE SROTM_X1Y1

      SUBROUTINE DROTM_X1Y1( N, X, INCX, Y, INCY, PARAM )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X(*), Y(*)
         REAL(WP), INTENT(IN) :: PARAM(5)
         CALL DROTM( N, X(1), INCX, Y(1), INCY, PARAM )
      END SUBROUTINE DROTM_X1Y1
!
!-----Natural interface for xSCAL-----
!
      SUBROUTINE SSCAL_X1( N, ALPHA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL SSCAL( N, ALPHA, X(1), INCX )
      END SUBROUTINE SSCAL_X1

      SUBROUTINE DSCAL_X1( N, ALPHA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         REAL(WP), INTENT(INOUT) :: X(*)
         CALL DSCAL( N, ALPHA, X(1), INCX )
      END SUBROUTINE DSCAL_X1

      SUBROUTINE CSCAL_X1( N, ALPHA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL CSCAL( N, ALPHA, X(1), INCX )
      END SUBROUTINE CSCAL_X1

      SUBROUTINE ZSCAL_X1( N, ALPHA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         COMPLEX(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL ZSCAL( N, ALPHA, X(1), INCX )
      END SUBROUTINE ZSCAL_X1

      SUBROUTINE CSSCAL_X1( N, ALPHA, X, INCX )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL CSSCAL( N, ALPHA, X(1), INCX )
      END SUBROUTINE CSSCAL_X1

      SUBROUTINE ZDSCAL_X1( N, ALPHA, X, INCX )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, N
         REAL(WP), INTENT(IN) :: ALPHA
         COMPLEX(WP), INTENT(INOUT) :: X(*)
         CALL ZDSCAL( N, ALPHA, X(1), INCX )
      END SUBROUTINE ZDSCAL_X1
!
!-----Natural interface for xSWAP-----
!
      SUBROUTINE SSWAP_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X(*), Y(*)
         CALL SSWAP( N, X(1), INCX, Y(1), INCY )
      END SUBROUTINE SSWAP_X1Y1

      SUBROUTINE DSWAP_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         REAL(WP), INTENT(INOUT) :: X(*), Y(*)
         CALL DSWAP( N, X(1), INCX, Y(1), INCY )
      END SUBROUTINE DSWAP_X1Y1

      SUBROUTINE CSWAP_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS32, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X(*), Y(*)
         CALL CSWAP( N, X(1), INCX, Y(1), INCY )
      END SUBROUTINE CSWAP_X1Y1

      SUBROUTINE ZSWAP_X1Y1( N, X, INCX, Y, INCY )
         USE LA_CONSTANTS, ONLY : WP
         INTEGER, INTENT(IN) :: INCX, INCY, N
         COMPLEX(WP), INTENT(INOUT) :: X(*), Y(*)
         CALL ZSWAP( N, X(1), INCX, Y(1), INCY )
      END SUBROUTINE ZSWAP_X1Y1

      END MODULE LA_BLAS1
