/*
 * Algebraic manipulator unfactorizing (expanding) routines.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "am.h"
#include "externs.h"

int	euclid_count;

/*
 * This is the subroutine used by the "unfactor" command.
 */
unfa_sub(n)
int	n;
{
	uf_simp(&lhs[n][0], &n_lhs[n]);
	uf_simp(&rhs[n][0], &n_rhs[n]);
}

/*
 * Totally unfactor equation side and simplify.
 */
uf_simp(equation, np)
token_type	*equation;
int		*np;
{
	do {
		organize(equation, np);
	} while (sub_ufactor(equation, np, 0));
	uf_repeat(equation, np);
	uf_tsimp(equation, np);
}

uf_simp_no_repeat(equation, np)
token_type	*equation;
int		*np;
{
	do {
		organize(equation, np);
	} while (sub_ufactor(equation, np, 0));
	uf_tsimp(equation, np);
}

/*
 * Unfactor times and divide and simplify.
 */
uf_tsimp(equation, np)
token_type	*equation;
int		*np;
{
	uf_times(equation, np);
	simp_loop(equation, np);
	while (uf_times(equation, np))
		simp_loop(equation, np);
}

/*
 * Totally unfactor equation side with no simplification.
 */
ufactor(equation, np)
token_type	*equation;
int		*np;
{
	int	i;

	do {
		organize(equation, np);
	} while (sub_ufactor(equation, np, 0));
	uf_repeat(equation, np);
	uf_times(equation, np);
}

/*
 * Unfactor power only.
 * (a * b)^c -> a^c * b^c
 * Return true if equation side is modified.
 */
int
uf_power(equation, np)
token_type	*equation;
int		*np;
{
	int	rv;
	int	i;

	rv = false;
	if (debug_level >= 5)
		printf("uf_power()\n");
	do {
		organize(equation, np);
		i = sub_ufactor(equation, np, 2);
		rv |= i;
	} while (i);
	return rv;
}

/*
 * Unfactor power only.
 * a^(b + c) -> a^b * a^c
 * Return true if equation side is modified.
 */
int
uf_pplus(equation, np)
token_type	*equation;
int		*np;
{
	int	rv;
	int	i;

	if (debug_level >= 5)
		printf("uf_pplus()\n");
	organize(equation, np);
	rv = sub_ufactor(equation, np, 4);
	if (rv) {
		organize(equation, np);
	}
	return rv;
}

/*
 * Unfactor power only.
 * a^2 -> a * a
 * Return true if equation side is modified.
 */
int
uf_repeat(equation, np)
token_type	*equation;
int		*np;
{
	int	rv;
	int	i;

	if (debug_level >= 5)
		printf("uf_repeat()\n");
	organize(equation, np);
	rv = sub_ufactor(equation, np, 6);
	if (rv) {
		organize(equation, np);
	}
	return rv;
}

/*
 * Unfactor times and divide only.
 * (a + b)*c -> a*c + b*c
 * If "partial_flag", don't unfactor (a+b)/(c+d).
 * Return true if equation side is modified.
 */
int
uf_times(equation, np)
token_type	*equation;
int		*np;
{
	int	i;
	int	rv;

	rv = false;
	if (debug_level >= 5)
		printf("uf_times()\n");
	do {
		organize(equation, np);
		if (partial_flag)
			reorder(equation, np);
		group_proc(equation, np, false);
		if (partial_flag)
			no_divide(equation, np);
		i = sub_ufactor(equation, np, 1);
		rv |= i;
	} while (i);
	organize(equation, np);
	return rv;
}

no_divide(equation, np)
token_type	*equation;
int		*np;
{
	int	i, j;
	int	level;
	int	plus_flag;

	for (i = 1; i < *np; i += 2) {
		if (equation[i].token.operatr == POWER) {
			level = equation[i].level;
			for (i += 2; i < *np && equation[i].level > level; i += 2)
				;
			i -= 2;
			continue;
		}
		if (equation[i].token.operatr == DIVIDE) {
			level = equation[i].level;
#if	true
			plus_flag = false;
			for (j = i + 2; j < *np; j += 2) {
				if (equation[j].level <= level)
					break;
				if (equation[j].level == (level + 1)) {
					switch (equation[j].token.operatr) {
					case PLUS:
					case MINUS:
						plus_flag = true;
						break;
					case TIMES:
						for (j = i + 2; j < *np; j += 2) {
							if (equation[j].level <= level)
								break;
							if (equation[j].level == (level + 2)) {
								if (equation[j].token.operatr == PLUS
								    || equation[j].token.operatr == MINUS) {
									plus_flag = true;
									break;
								}
							}
						}
						break;
					}
					break;
				}
			}
			if (!plus_flag)
				continue;
#endif
			for (j = i - 1; j >= 0; j--) {
				if (equation[j].level < level)
					break;
				equation[j].level += 2;
			}
		}
	}
}

int
sub_ufactor(equation, np, ii)
token_type	*equation;
int		*np;
int		ii;
{
	int	modified;
	int	i;
	int	b1, e1;
	int	level;
	int	op;

	modified = false;
	for (i = 1; i < *np; i += 2) {
		op = equation[i].token.operatr;
		switch (op) {
		case TIMES:
		case DIVIDE:
			if (ii == 1)
				break;
			else
				continue;
		case POWER:
			if (ii == 0 || ii == 2 || ii == 4 || ii == 6)
				break;
		default:
			continue;
		}
		level = equation[i].level;
		for (b1 = i - 2; b1 >= 0; b1 -= 2)
			if (equation[b1].level < level)
				break;
		b1++;
		for (e1 = i + 2; e1 < *np; e1 += 2) {
			if (equation[e1].level < level)
				break;
		}
		if (unf_sub(equation, np, b1, i, e1, level, ii)) {
			modified = true;
			i = b1 - 1;
			continue;
		}
	}
	return modified;
}

int
unf_sub(equation, np, b1, loc, e1, level, ii)
token_type	*equation;
int		*np;
int		b1, loc, e1, level;
int		ii;
{
	int		i, j, k;
	int		b2, eb1, be1;
	int		len;
	double		d1, d2;

	switch (equation[loc].token.operatr) {
	case TIMES:
	case DIVIDE:
		for (i = b1 + 1; i < e1; i += 2) {
			if (equation[i].level == level + 1) {
				switch (equation[i].token.operatr) {
				case PLUS:
				case MINUS:
					break;
				default:
					continue;
				}
				for (b2 = i - 2; b2 >= b1; b2 -= 2)
					if (equation[b2].level <= level)
						break;
				b2++;
				eb1 = b2;
				for (be1 = i + 2; be1 < e1; be1 += 2)
					if (equation[be1].level <= level)
						break;
				if (eb1 > b1 && equation[eb1-1].token.operatr == DIVIDE) {
					i = be1 - 2;
					continue;
				}
				len = 0;
u_again:
				if (len + (eb1 - b1) + (i - b2) + (e1 - be1) + 1 > N_TOKENS) {
					error_huge();
				}
				blt(&scratch[len], &equation[b1], (eb1 - b1) * sizeof(*equation));
				j = len;
				len += (eb1 - b1);
				for (; j < len; j++)
					scratch[j].level++;
				blt(&scratch[len], &equation[b2], (i - b2) * sizeof(*equation));
				len += (i - b2);
				blt(&scratch[len], &equation[be1], (e1 - be1) * sizeof(*equation));
				j = len;
				len += (e1 - be1);
				for (; j < len; j++)
					scratch[j].level++;
				if (i < be1) {
					blt(&scratch[len], &equation[i], sizeof(*equation));
					scratch[len].level--;
					len++;
					i++;
					b2 = i;
					for (; i < be1; i++) {
						if (equation[i].kind == OPERATOR && equation[i].level == (level + 1))
							break;
					}
					goto u_again;
				} else {
					if (*np - (e1 - b1) + len > N_TOKENS) {
						error_huge();
					}
					blt(&equation[b1+len], &equation[e1], (*np - e1) * sizeof(*equation));
					*np += len - (e1 - b1);
					blt(&equation[b1], &scratch[0], len * sizeof(*equation));
					return true;
				}
			}
		}
		break;
	case POWER:
		if (ii == 2 || ii == 0) {
			for (i = b1 + 1; i < loc; i += 2) {
				if (equation[i].level != level + 1)
					continue;
				switch (equation[i].token.operatr) {
				case TIMES:
				case DIVIDE:
					break;
				default:
					goto do_pplus;
				}
				b2 = b1;
				len = 0;
u1_again:
				if (len + (i - b2) + (e1 - loc) + 1 > N_TOKENS) {
					error_huge();
				}
				blt(&scratch[len], &equation[b2], (i - b2) * sizeof(*equation));
				len += (i - b2);
				blt(&scratch[len], &equation[loc], (e1 - loc) * sizeof(*equation));
				j = len;
				len += (e1 - loc);
				for (; j < len; j++)
					scratch[j].level++;
				if (i < loc) {
					blt(&scratch[len], &equation[i], sizeof(*equation));
					scratch[len].level--;
					len++;
					i++;
					b2 = i;
					for (; i < loc; i++) {
						if (equation[i].kind == OPERATOR && equation[i].level == level + 1)
							break;
					}
					goto u1_again;
				} else {
					if (*np - (e1 - b1) + len > N_TOKENS) {
						error_huge();
					}
					blt(&equation[b1+len], &equation[e1], (*np - e1) * sizeof(*equation));
					*np += len - (e1 - b1);
					blt(&equation[b1], &scratch[0], len * sizeof(*equation));
					return true;
				}
			}
		}
do_pplus:
		if (ii == 4 || ii == 0) {
			for (i = loc + 2; i < e1; i += 2) {
				if (equation[i].level != level + 1)
					continue;
				switch (equation[i].token.operatr) {
				case PLUS:
				case MINUS:
					break;
				default:
					goto do_repeat;
				}
				b2 = loc + 1;
				len = 0;
u2_again:
				if (len + (loc - b1) + (i - b2) + 2 > N_TOKENS) {
					error_huge();
				}
				j = len;
				blt(&scratch[len], &equation[b1], (loc + 1 - b1) * sizeof(*equation));
				len += (loc + 1 - b1);
				for (; j < len; j++)
					scratch[j].level++;
				blt(&scratch[len], &equation[b2], (i - b2) * sizeof(*equation));
				len += (i - b2);
				if (i < e1) {
					scratch[len].level = level;
					scratch[len].kind = OPERATOR;
					if (equation[i].token.operatr == PLUS)
						scratch[len].token.operatr = TIMES;
					else
						scratch[len].token.operatr = DIVIDE;
					len++;
					i++;
					b2 = i;
					for (; i < e1; i++) {
						if (equation[i].kind == OPERATOR && equation[i].level == level + 1)
							break;
					}
					goto u2_again;
				} else {
					if (*np - (e1 - b1) + len > N_TOKENS) {
						error_huge();
					}
					blt(&equation[b1+len], &equation[e1], (*np - e1) * sizeof(*equation));
					*np += len - (e1 - b1);
					blt(&equation[b1], &scratch[0], len * sizeof(*equation));
					return true;
				}
			}
		}
do_repeat:
		if (ii != 6)
			break;
		i = loc;
		if (equation[i+1].level != level
		    || equation[i+1].kind != CONSTANT
		    || equation[i+1].token.constant <= 1.0)
			break;
		for (b2 = i - 2; b2 >= b1; b2 -= 2) {
			switch (equation[b2].token.operatr) {
			case PLUS:
			case MINUS:
				goto has_plus;
			}
		}
		d2 = modf(equation[i+1].token.constant, &d1);
		if (d2 == 0.0) {
			return false;
		}
		k = (e1 - b1) + 1;
		if (*np + k > N_TOKENS) {
			error_huge();
		}
		for (j = b1; j < e1; j++)
			equation[j].level++;
		blt(&equation[e1+1], &equation[b1], (*np - b1) * sizeof(token_type));
		*np += k;
		equation[e1].level = level;
		equation[e1].kind = OPERATOR;
		equation[e1].token.operatr = TIMES;
		equation[i+1].token.constant = d1;
		equation[i+1+k].token.constant = d2;
		return true;
has_plus:
		d1 = ceil(equation[i+1].token.constant) - 1.0;
		if (d1 < 1.0) {
			printf("Error in ceil() function!\n");
			longjmp(jmp_save, 13);
		}
		d2 = d1 * ((i - b1) + 1.0);
		if (*np + d2 > N_TOKENS) {
			error_huge();
		}
		j = d1;
		k = d2;
		blt(&equation[e1+k], &equation[e1], (*np - e1) * sizeof(token_type));
		*np += k;
		equation[i+1].token.constant -= d1;
		k = e1;
		while (j-- > 0) {
			equation[k].level = level;
			equation[k].kind = OPERATOR;
			equation[k].token.operatr = TIMES;
			blt(&equation[k+1], &equation[b1], (i - b1) * sizeof(token_type));
			k += (i - b1) + 1;
		}
		if (equation[i+1].token.constant == 1.0) {
			blt(&equation[i], &equation[e1], (*np - e1) * sizeof(token_type));
			*np -= (e1 - i);
		} else {
			for (j = b1; j < e1; j++)
				equation[j].level++;
		}
		return true;
	}
	return false;
}

/*
 * Convert 1/(x^y) to x^(-1*y) if y is not a constant.
 * Return true if equation side is modified.
 */
int
unsimp_power(equation, np)
token_type	*equation;
int		*np;
{
	int	modified;
	int	i;

	modified = false;
	for (i = 1; i < *np; i += 2) {
		if (equation[i].token.operatr == POWER) {
			modified |= usp_sub(equation, np, i);
		}
	}
	return modified;
}

int
usp_sub(equation, np, i)
token_type	*equation;
int		*np, i;
{
	int	level;
	int	j;

	level = equation[i].level;
	if ((i + 2 >= *np || equation[i+2].level <= level)
	    && equation[i+1].kind == CONSTANT)
		return false;
	for (j = i - 2;; j -= 2) {
		if (j < 0)
			return false;
		if (equation[j].level < level) {
			if (equation[j].level == (level - 1)
			    && equation[j].token.operatr == DIVIDE)
				break;
			else
				return false;
		}
	}
	if ((*np + 2) > N_TOKENS) {
		error_huge();
	}
	equation[j].token.operatr = TIMES;
	for (j = i + 1;; j++) {
		if (j >= *np || equation[j].level < level)
			break;
		equation[j].level++;
	}
	i++;
	blt(&equation[i+2], &equation[i], (*np - i) * sizeof(*equation));
	*np += 2;
	equation[i].level = level + 1;
	equation[i].kind = CONSTANT;
	equation[i].token.constant = -1.0;
	i++;
	equation[i].level = level + 1;
	equation[i].kind = OPERATOR;
	equation[i].token.operatr = TIMES;
	return true;
}

#if	true
/*
 * Convert 1/(x^y) to (1/x)^y.
 * Return true if equation side is modified.
 */
int
unsimp2_power(equation, np)
token_type	*equation;
int		*np;
{
	int	modified;
	int	i;

	modified = false;
	for (i = 1; i < *np; i += 2) {
		if (equation[i].token.operatr == POWER) {
			modified |= usp2_sub(equation, np, i);
		}
	}
	return modified;
}

int
usp2_sub(equation, np, i)
token_type	*equation;
int		*np, i;
{
	int	level;
	int	j, k;

	level = equation[i].level;
	for (j = i - 2;; j -= 2) {
		if (j < 0)
			return false;
		if (equation[j].level < level) {
			if (equation[j].level == (level - 1)
			    && equation[j].token.operatr == DIVIDE)
				break;
			else
				return false;
		}
	}
	if ((*np + 2) > N_TOKENS) {
		error_huge();
	}
	equation[j].token.operatr = TIMES;
	for (k = j + 1; k < i; k++) {
		equation[k].level++;
	}
	blt(&equation[j+3], &equation[j+1], (*np - (j + 1)) * sizeof(*equation));
	*np += 2;
	j++;
	equation[j].level = level + 1;
	equation[j].kind = CONSTANT;
	equation[j].token.constant = 1.0;
	j++;
	equation[j].level = level + 1;
	equation[j].kind = OPERATOR;
	equation[j].token.operatr = DIVIDE;
	return true;
}
#endif

uf_neg_help(equation, np)
token_type	*equation;
int		*np;
{
	int	i, j;
	int	level;

	for (i = 0; i < *np - 1; i += 2) {
		if (equation[i].kind == CONSTANT && equation[i].token.constant < 0.0) {
			level = equation[i].level;
			if (equation[i+1].level == level) {
				switch (equation[i+1].token.operatr) {
				case TIMES:
				case DIVIDE:
#if	false
					for (j = i + 3; j < *np; j += 2) {
						if (equation[j].level < level) {
							if (equation[j].level == level - 1) {
								switch (equation[j].token.operatr) {
								case PLUS:
								case MINUS:
									goto skip;
								}
							}
							break;
						}
					}
#endif
					neg_sub(equation, np, i);
					i += 2;
				}
			}
		}
skip:
		;
	}
}

neg_sub(equation, np, i)
token_type	*equation;
int		*np;
int		i;
{
	int	level;

	level = equation[i].level;
	if ((*np + 2) > N_TOKENS) {
		error_huge();
	}
	blt(&equation[i+3], &equation[i+1], (*np - (i + 1)) * sizeof(*equation));
	*np += 2;
	equation[i].token.constant = -equation[i].token.constant;
	i++;
	equation[i].level = level;
	equation[i].kind = OPERATOR;
	equation[i].token.operatr = DIVIDE;
	i++;
	equation[i].level = level;
	equation[i].kind = CONSTANT;
	equation[i].token.constant = -1.0;
}

/*
 * Return the Greatest Common Divisor of doubles "d1" and d2".
 * Will work with non-integers, but there may be some floating point error.
 * Always works correctly with integers.
 * Returns 0 if it failed.
 */
double
gcd(d1, d2)
double	d1, d2;
{
	int	count;
	double	larger;
	double	divisor;
	double	r1;
	double	d;

	d1 = fabs(d1);
	d2 = fabs(d2);
	if (d1 > d2) {
		larger = d1;
		divisor = d2;
	} else {
		larger = d2;
		divisor = d1;
	}
	d = larger * epsilon;
	if ((2 * d) >= divisor)
		return 0.0;
	for (count = 1; count < 50; count++) {
		euclid_count = count;
		r1 = fmod(larger, divisor);
		if (r1 <= d)
			return divisor;
		if ((divisor - r1) <= d)
			return divisor;
		larger = divisor;
		divisor = r1;
	}
	return 0.0;
}

/*
 * Convert the passed double "d" to a fully reduced fraction.
 * "d1p" points to the numerator and "d2p" points to the denominator.
 * Return true if conversion was successful.
 */
int
f_to_fraction(d, d1p, d2p)
double	d;
double	*d1p;
double	*d2p;
{
	double	divisor;
	double	temp;
	double	d1, d2;
	k_type	k3, k4;

	*d1p = d;
	*d2p = 1.0;
	if (modf(d, &temp) == 0.0)
		return false;
	if ((divisor = gcd(1.0, d)) > epsilon) {
		d1 = d / divisor;
		d2 = 1.0 / divisor;
		if (d1 >= 0.0) {
			modf(d1 + 0.5, &d1);
		} else {
			modf(d1 - 0.5, &d1);
		}
		modf(d2 + 0.5, &d2);
		if (fabs(d1) >= 1.0e10)
			return false;
		if (d2 >= 1.0e10)
			return false;
		if (d2 < 1.5)
			return false;
		divisor = gcd(d1, d2);
		if (modf(divisor, &temp) != 0.0) {
			printf("Error in gcd() function!\n");
			return false;
		}
		if (divisor > 1.0) {
			d1 = d1 / divisor;
			d2 = d2 / divisor;
		}
		k3 = (d1 / d2);
		k4 = d;
		if (k3 != k4)
			return false;
		*d1p = d1;
		*d2p = d2;
		return true;
	}
	return false;
}

make_fractions(equation, np)
token_type	*equation;
int		*np;
{
	int	i, j, k;
	int	level;
	double	d1, d2;
	int	inc_level;

	for (i = 0; i < *np; i += 2) {
		if (equation[i].kind == CONSTANT) {
			level = equation[i].level;
			if (i > 0 && equation[i-1].level == level
			    && equation[i-1].token.operatr == DIVIDE)
				continue;
			if (!f_to_fraction((double) equation[i].token.constant, &d1, &d2))
				continue;
			if ((*np + 2) > N_TOKENS) {
				error_huge();
			}
			inc_level = (*np > 1);
			if ((i + 1) < *np && equation[i+1].level == level) {
				switch (equation[i+1].token.operatr) {
				case TIMES:
					if (d1 == 1.0) {
						blt(&equation[i], &equation[i+2], (*np - (i + 2)) * sizeof(*equation));
						*np -= 2;
					} else {
						equation[i].token.constant = d1;
					}
					for (j = i + 1; j < *np && equation[j].level >= level; j += 2) {
						if (equation[j].level == level && equation[j].token.operatr == DIVIDE) {
							break;
						}
					}
					blt(&equation[j+2], &equation[j], (*np - j) * sizeof(*equation));
					*np += 2;
					equation[j].level = level;
					equation[j].kind = OPERATOR;
					equation[j].token.operatr = DIVIDE;
					j++;
					equation[j].level = level;
					equation[j].kind = CONSTANT;
					equation[j].token.constant = d2;
					if (d1 == 1.0) {
						i -= 2;
					}
					continue;
				case DIVIDE:
					inc_level = false;
				default:
					break;
				}
			}
			j = i;
			blt(&equation[i+3], &equation[i+1], (*np - (i + 1)) * sizeof(*equation));
			*np += 2;
			equation[j].token.constant = d1;
			j++;
			equation[j].level = level;
			equation[j].kind = OPERATOR;
			equation[j].token.operatr = DIVIDE;
			j++;
			equation[j].level = level;
			equation[j].kind = CONSTANT;
			equation[j].token.constant = d2;
			if (inc_level) {
				for (k = i; k <= j; k++)
					equation[k].level++;
			}
		}
	}
}
