/*
 * Algebraic manipulator simplifying routines.
 * Includes polynomial and smart division,
 * polynomial factoring, etc.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "am.h"
#include "externs.h"

/*
 * The following defines the stack storage size, in tokens, for divisors
 * and quotients during polynomial and smart division.
 */
#if	UNIX
#define	DIVISOR_SIZE	4000
#else
#define	DIVISOR_SIZE	400
#endif

typedef struct {
	long	v;
	int	count;
} sort_type;

static int
vcmp(p1, p2)
sort_type	*p1, *p2;
{
	if (p2->count == p1->count) {
		if (p1->v < p2->v)
			return -1;
		if (p1->v == p2->v)
			return 0;
		return 1;
	}
	return(p2->count - p1->count);
}

/*
 * Factor polynomials.
 * Return true if equation side was modified.
 */
int
poly_factor(equation, np)
token_type	*equation;
int		*np;
{
	return pf_recurse(equation, np, 0, 1);
}

int
pf_recurse(equation, np, loc, level)
token_type	*equation;
int		*np, loc, level;
{
	int	modified;
	int	i, j, k;
	int	op;
	int	len;

	modified = false;
	op = 0;
	for (i = loc + 1; i < *np && equation[i].level >= level; i += 2) {
		if (equation[i].level == level) {
			op = equation[i].token.operatr;
		}
	}
	len = i - loc;
	switch (op) {
	case PLUS:
	case MINUS:
		modified = pf_sub(equation, np, loc, len, level);
		break;
	}
	if (modified)
		return true;
	for (i = loc;;) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].level > level) {
			modified |= pf_recurse(equation, np, i, level + 1);
			i++;
			for (; i < *np && equation[i].level > level; i += 2)
				;
			continue;
		}
		i++;
	}
	return modified;
}

/*
 * Remove trivial factors from "tlhs".
 */
int
remove_factors(v)
long	v;
{
	int	i, j, k;
	int	plus_flag;
	int	divide_flag;
	int	vfound_flag;
	int	op;

	if (debug_level >= 3) {
		printf("Entering remove_factors() with:\n");
		side_debug(3, &tlhs[0], n_tlhs);
	}
	do {
		simp_ssub(&tlhs[0], &n_tlhs, 0L, 0.0, false, false, 2);
	} while (uf_power(&tlhs[0], &n_tlhs));
	if (n_tlhs > 1 && tlhs[0].level == 1 && tlhs[0].kind == CONSTANT
	    && (tlhs[1].token.operatr == TIMES || tlhs[1].token.operatr == DIVIDE)) {
		tlhs[0].token.constant = 1.0;
	}
	plus_flag = false;
	divide_flag = false;
	vfound_flag = false;
	k = 0;
	for (i = 0, j = 0;; i++) {
		if (i >= n_tlhs) {
			if ((v == 0L || vfound_flag) && plus_flag && !divide_flag) {
				if (k > 0)
					j--;
				blt(&scratch[k], &tlhs[j], (i - j) * sizeof(token_type));
				k += i - j;
			}
			if (k <= 0)
				return false;
			blt(&tlhs[0], &scratch[0], k * sizeof(token_type));
			n_tlhs = k;
			side_debug(3, &tlhs[0], n_tlhs);
			return true;
		}
		if (tlhs[i].kind == OPERATOR && tlhs[i].level == 1) {
			op = tlhs[i].token.operatr;
			if (op == PLUS || op == MINUS) {
				plus_flag = true;
				continue;
			}
			if (op != TIMES && op != DIVIDE)
				return false;
			if ((v == 0L || vfound_flag) && plus_flag && !divide_flag) {
				if (k > 0)
					j--;
				blt(&scratch[k], &tlhs[j], (i - j) * sizeof(token_type));
				k += i - j;
			}
			vfound_flag = false;
			plus_flag = false;
			divide_flag = (op == DIVIDE);
			j = i + 1;
			continue;
		}
		if (tlhs[i].kind == OPERATOR && tlhs[i].level == 2) {
			op = tlhs[i].token.operatr;
			if (op == PLUS || op == MINUS)
				plus_flag = true;
		} else if (tlhs[i].kind == VARIABLE && tlhs[i].token.variable == v) {
			vfound_flag = true;
		}
	}
}

/*
 * Return the number of level 1 additive type operators.
 */
int
level1_plus(p1, n1)
token_type	*p1;
int		n1;
{
	int	i;
	int	op;
	int	count;

	count = 0;
	for (i = 1; i < n1; i += 2) {
		if (p1[i].level == 1) {
			op = p1[i].token.operatr;
			if (op == PLUS || op == MINUS) {
				count++;
			}
		}
	}
	return count;
}

/*
 * Polynomial factoring subroutine.
 */
int
pf_sub(equation, np, loc, len, level)
token_type	*equation;
int		*np, loc, len, level;
{
	token_type	*p1;
	int		modified;
	int		single_modified;
	int		i, j, k;
	long		v;
	long		big_v;
	long		last_v;
	long		last_try_v;
	int		len_first;
	int		loc1;
	int		loc2, len2;
	int		loct, lent;
	int		count;
	int		gcd_count;
	int		flag;
	jmp_buf		save_save;
	int		found1, found2;
	int		div_flag;
	int		vc, cnt;
	long		v1;
	sort_type	va[MAX_VARS];
	double		d;
	int		three;

	modified = false;
	single_modified = false;
	blt(save_save, jmp_save, sizeof(jmp_save));
	if ((i = setjmp(jmp_save)) != 0) {
		domain_check = false;
		blt(jmp_save, save_save, sizeof(jmp_save));
		if (i == 13) {
			longjmp(jmp_save, i);
		}
		printf("Processing continuing without error...\n");
		return(modified || single_modified);
	}
/* First factor polynomials with repeated factors. */
	loc1 = loc;
	len2 = 0;
	v = 0;
	div_flag = 2;
	find_greatest_power(&equation[loc1], len, &v, &d, &j, &k, &div_flag);
	for (count = 1;; count++) {
		blt(&trhs[0], &equation[loc1], len * sizeof(token_type));
		n_trhs = len;
		uf_simp(&trhs[0], &n_trhs);
		if (level1_plus(&trhs[0], n_trhs) < 2)
			goto skip_factor;
		vc = 0;
		last_v = -1;
		for (;;) {
			v1 = -1;
			for (i = 0; i < n_trhs; i += 2) {
				if (trhs[i].kind == VARIABLE
				    && trhs[i].token.variable > last_v) {
					if (v1 == -1 || trhs[i].token.variable < v1) {
						v1 = trhs[i].token.variable;
						cnt = 1;
					} else if (trhs[i].token.variable == v1) {
						cnt++;
					}
				}
			}
			if (v1 == -1)
				break;
			last_v = v1;
			if (vc >= ARR_CNT(va)) {
				break;
			}
			va[vc].v = v1;
			va[vc++].count = cnt;
		}
		big_v = 0L;
		cnt = -1;
		for (i = 0; i < vc; i++) {
			if ((va[i].v & VAR_MASK) <= SIGN) {
				continue;
			}
			if (cnt < 0 || va[i].count < cnt) {
				big_v = va[i].v;
				cnt = va[i].count;
			}
		}
		if (cnt == 1)
			goto skip_factor;
dagain:
		if (v <= 0L)
			break;
		if (debug_level >= 3) {
			side_debug(3, &equation[loc1], len);
			printf("Taking derivative with respect to (");
			list_var(v, true);
			printf(").\n");
		}
		if (!differentiate(&trhs[0], n_trhs, &tlhs[0], &n_tlhs, v, true)) {
			if (v != big_v) {
				v = big_v;
				goto dagain;
			}
			break;
		}
		simp_loop(&tlhs[0], &n_tlhs);
		if ((n_tlhs + 2) > (N_TOKENS / 2))
			break;
		for (i = 0; i < n_tlhs; i++)
			tlhs[i].level++;
		tlhs[n_tlhs].kind = OPERATOR;
		tlhs[n_tlhs].level = 1;
		tlhs[n_tlhs].token.operatr = TIMES;
		n_tlhs++;
		tlhs[n_tlhs].kind = VARIABLE;
		tlhs[n_tlhs].level = 1;
		tlhs[n_tlhs].token.variable = v;
		n_tlhs++;
		ufactor(&tlhs[0], &n_tlhs);
		if ((n_tlhs / 5) > len)
			break;
		if ((gcd_count = poly_gcd(&equation[loc1], len, &tlhs[0], n_tlhs, v, true, true)) == 0)
			break;
		if (!level1_plus(&tlhs[0], n_tlhs))
			break;
		save_factors(equation, np, loc1, len, level);
		loc1 += n_tlhs + 1;
		len = n_trhs;
		switch (count) {
		case 1:
			printf("Polynomial factored.  Difficulty = %d...\n", gcd_count);
			len_first = n_tlhs;
			loc2 = loc1;
			break;
		case 2:
			len2 = n_tlhs;
			break;
		}
		modified = true;
	}
/* Now try and factor polynomials with symbolic factors. */
	if (!modified) {
		last_v = -1;
		last_try_v = 0L;
		flag = false;
next_v:
		if (!flag) {
			p1 = &equation[loc];
			blt(&trhs[0], p1, len * sizeof(token_type));
			n_trhs = len;
			uf_simp(&trhs[0], &n_trhs);
		}
		for (;;) {
break_cont:
			if (flag)
				break;
			v = -1;
			for (i = 0; i < len; i += 2) {
				if (p1[i].kind == VARIABLE
				    && p1[i].token.variable > last_v) {
					if (v == -1 || p1[i].token.variable < v) {
						v = p1[i].token.variable;
					}
				}
			}
			if (v == -1) {
				if (last_try_v) {
					v = last_try_v;
					flag = true;
				} else {
					break;
				}
			}
			last_v = v;
			if (!flag && v != big_v && big_v > 0L) {
				found1 = false;
				found2 = false;
				div_flag = false;
				for (i = 0;; i++) {
					if (i >= n_trhs
					    || (trhs[i].kind == OPERATOR
					    && trhs[i].level == 1)) {
						if (found1 && found2) {
							break;
						}
						if (i >= n_trhs) {
							goto break_cont;
						} else {
							found1 = false;
							found2 = false;
							div_flag = false;
							continue;
						}
					}
					if (trhs[i].kind == OPERATOR
					    && trhs[i].level == 2) {
						div_flag = (trhs[i].token.operatr == DIVIDE);
						count = 0;
					}
					if (trhs[i].kind == VARIABLE) {
						if (trhs[i].token.variable == big_v)
							found1 = true;
						if (found2 != 2) {
							if (div_flag) {
								count++;
								if (trhs[i].token.variable == v) {
									if (count == 1)
										found2 = 2;
									else
										found2 = false;
								}
							} else {
								if (trhs[i].token.variable == v)
									found2 = true;
							}
						}
					}
				}
			}
			three = 3;
			if ((count = find_greatest_power(&trhs[0], n_trhs, &v, &d, &j, &k, &three)) > 1) {
				blt(&tlhs[0], &trhs[0], n_trhs * sizeof(token_type));
				n_tlhs = n_trhs;
				factorv(&tlhs[0], &n_tlhs, v);
				if ((i = find_greatest_power(&tlhs[0], n_tlhs, &v, &d, &j, &k, &three)) != 1) {
					if (!flag) {
						if (i < count)
							last_try_v = v;
						continue;
					}
				}
				blt(&tlhs[0], &tlhs[j], k * sizeof(token_type));
				n_tlhs = k;
				if (debug_level >= 3) {
					printf("Trying factor: ");
					list_proc(&tlhs[0], n_tlhs);
					printf("\n");
				}
				if (!remove_factors(0L))
					continue;
				if ((gcd_count = poly_gcd(&equation[loc1], len, &tlhs[0], n_tlhs, 0L, true, true)) == 0)
					goto next_v;
				if (!level1_plus(&tlhs[0], n_tlhs))
					goto next_v;
				save_factors(equation, np, loc1, len, level);
				printf("Found polynomial factor.  Difficulty = %d...\n", gcd_count);
				loc1 += n_tlhs + 1;
				len = n_trhs;
				len_first = n_tlhs;
				loc2 = loc1;
				single_modified = true;
				break;
			}
		}
	}
skip_factor:
	blt(jmp_save, save_save, sizeof(jmp_save));
	if (modified || single_modified) {
		if (debug_level >= 3) {
			printf("Derived factors are:\n");
			side_debug(3, &equation[0], *np);
		}
	}
	if (modified) {
/* Repeated factor was factored out. */
/* See if we can factor out more of the repeated factor. */
		if (len2) {
			loct = loc2;
			lent = len2;
			flag = false;
		} else {
			loct = loc;
			lent = len_first;
			flag = true;
		}
		if (poly_gcd(&equation[loc1], len, &equation[loct], lent, v, flag, false)) {
			if (level1_plus(&tlhs[0], n_tlhs)) {
				save_factors(equation, np, loc1, len, level);
				loc1 += n_tlhs + 1;
				len = n_trhs;
			}
		}
		if (len2) {
			loc1 = loc2;
			len = len2;
		}
		if (poly_gcd(&equation[loc], len_first, &equation[loc1], len, 0L, false, false)) {
			save_factors(equation, np, loc, len_first, level);
		}
	}
	return(modified || single_modified);
}

save_factors(equation, np, loc1, len, level)
token_type	*equation;
int		*np, loc1, len, level;
{
	int	i, j;

	i = n_tlhs + 1 + n_trhs;
	if ((*np + (i - len)) > N_TOKENS)
		error_huge();
	blt(&equation[loc1+i], &equation[loc1+len], (*np - (loc1 + len)) * sizeof(token_type));
	*np += i - len;
	blt(&equation[loc1], &tlhs[0], n_tlhs * sizeof(token_type));
	i = loc1 + n_tlhs;
	equation[i].level = 0;
	equation[i].kind = OPERATOR;
	equation[i].token.operatr = TIMES;
	i++;
	blt(&equation[i], &trhs[0], n_trhs * sizeof(token_type));
	i += n_trhs;
	for (j = loc1; j < i; j++)
		equation[j].level += level;
}

/*
 * Compute polynomial Greatest Common Divisor of "larger" and "smaller".
 * Return true if successful.
 * Return the GCD in "trhs".
 * Return "larger"/GCD in "tlhs".
 */
int
poly_gcd(larger, llen, smaller, slen, v, rf_flag, dont_copy)
token_type	*larger;	/* larger polynomial */
token_type	*smaller;	/* smaller polynomial */
int		llen, slen;
long		v;
int		rf_flag;	/* remove factors flag */
int		dont_copy;
{
	token_type	divisor[DIVISOR_SIZE];
	int		len_d;
	int		i, j, k;
	int		count;
	int		div_flag;

	div_flag = 2;
	for (count = 1;; count++) {
		if (count == 1) {
			if (!dont_copy) {
				blt(&trhs[0], &larger[0], llen * sizeof(token_type));
				n_trhs = llen;
			}
			if (&tlhs[0] != &smaller[0]) {
				if (slen > N_TOKENS)
					return 0;
				blt(&tlhs[0], smaller, slen * sizeof(token_type));
				n_tlhs = slen;
			}
			if (!remove_factors(0L) && rf_flag)
				return 0;
			if (n_tlhs > ARR_CNT(divisor))
				return 0;
			blt(&divisor[0], &tlhs[0], n_tlhs * sizeof(token_type));
			len_d = n_tlhs;
		} else {
			if (len_d > N_TOKENS || n_trhs > ARR_CNT(divisor))
				return 0;
			blt(&scratch[0], &trhs[0], n_trhs * sizeof(token_type));
			blt(&trhs[0], &divisor[0], len_d * sizeof(token_type));
			blt(&divisor[0], &scratch[0], n_trhs * sizeof(token_type));
			i = n_trhs;
			n_trhs = len_d;
			len_d = i;
		}
		i = poly_div(&trhs[0], n_trhs, &divisor[0], len_d, &v, &div_flag);
		if (!i) {
			return 0;
		}
		if (n_trhs == 1 && trhs[0].kind == CONSTANT && trhs[0].token.constant == 0.0) {
			if (count > 1) {
				if (len_d > N_TOKENS)
					return 0;
				blt(&tlhs[0], &divisor[0], len_d * sizeof(token_type));
				n_tlhs = len_d;
				if (!remove_factors(0L) && rf_flag)
					return 0;
				if (n_tlhs > ARR_CNT(divisor))
					return 0;
				blt(&divisor[0], &tlhs[0], n_tlhs * sizeof(token_type));
				len_d = n_tlhs;
				div_flag = 2;
				if (!poly_div(&larger[0], llen, &divisor[0], len_d, &v, &div_flag)
				    || n_trhs != 1 || trhs[0].kind != CONSTANT || trhs[0].token.constant != 0.0)
					return 0;
			}
			if (len_d > N_TOKENS)
				return 0;
			blt(&trhs[0], &divisor[0], len_d * sizeof(token_type));
			n_trhs = len_d;
			uf_simp(&tlhs[0], &n_tlhs);
			uf_simp(&trhs[0], &n_trhs);
			return(count);
		}
	}
}

/*
 * Compute polynomial Greatest Common Divisor of "larger" and "smaller".
 * Return true if successful.
 * Return "larger"/GCD in "tlhs".
 * Return "smaller"/GCD in "trhs".
 */
int
poly2_gcd(larger, llen, smaller, slen, v)
token_type	*larger;	/* larger polynomial */
token_type	*smaller;	/* smaller polynomial */
int		llen, slen;
long		v;
{
	token_type	divisor[DIVISOR_SIZE];
	int		len_d;
	int		i, j, k;
	int		count;
	int		div_flag;

	div_flag = 2;
	for (count = 1;; count++) {
		if (count == 1) {
			blt(&trhs[0], &larger[0], llen * sizeof(token_type));
			n_trhs = llen;
			if (slen > ARR_CNT(divisor))
				return 0;
			blt(&divisor[0], &smaller[0], slen * sizeof(token_type));
			len_d = slen;
		} else {
			if (len_d > N_TOKENS || n_trhs > ARR_CNT(divisor))
				return 0;
			blt(&scratch[0], &trhs[0], n_trhs * sizeof(token_type));
			blt(&trhs[0], &divisor[0], len_d * sizeof(token_type));
			blt(&divisor[0], &scratch[0], n_trhs * sizeof(token_type));
			i = n_trhs;
			n_trhs = len_d;
			len_d = i;
		}
		i = poly_div(&trhs[0], n_trhs, &divisor[0], len_d, &v, &div_flag);
		if (!i) {
			return 0;
		}
		if (n_trhs == 1 && trhs[0].kind == CONSTANT && trhs[0].token.constant == 0.0) {
			if (count > 1) {
				if (len_d > N_TOKENS)
					return 0;
				blt(&tlhs[0], &divisor[0], len_d * sizeof(token_type));
				n_tlhs = len_d;
				if (!remove_factors(0L))
					return 0;
				if (n_tlhs > ARR_CNT(divisor))
					return 0;
				blt(&divisor[0], &tlhs[0], n_tlhs * sizeof(token_type));
				len_d = n_tlhs;
				div_flag = 2;
				if (!poly_div(&smaller[0], slen, &divisor[0], len_d, &v, &div_flag)
				    || n_trhs != 1 || trhs[0].kind != CONSTANT || trhs[0].token.constant != 0.0) {
					printf("WARNING: Polynomial GCD found, but smaller divide failed!\n");
					return 0;
				}
				blt(&trhs[0], &divisor[0], len_d * sizeof(token_type));
				n_trhs = len_d;
				if (n_tlhs > ARR_CNT(divisor))
					return 0;
				blt(&divisor[0], &tlhs[0], n_tlhs * sizeof(token_type));
				len_d = n_tlhs;
				blt(&tlhs[0], &trhs[0], n_trhs * sizeof(token_type));
				n_tlhs = n_trhs;
				div_flag = 2;
				if (!poly_div(&larger[0], llen, &tlhs[0], n_tlhs, &v, &div_flag)
				    || n_trhs != 1 || trhs[0].kind != CONSTANT || trhs[0].token.constant != 0.0) {
					printf("WARNING: Polynomial GCD found, but larger divide failed!\n");
					return 0;
				}
				blt(&trhs[0], &divisor[0], len_d * sizeof(token_type));
				n_trhs = len_d;
			} else {
				n_trhs = 1;
				trhs[0].level = 1;
				trhs[0].kind = CONSTANT;
				trhs[0].token.constant = 1.0;
			}
			uf_simp(&tlhs[0], &n_tlhs);
			uf_simp(&trhs[0], &n_trhs);
			return count;
		}
	}
}

/*
 * Check for divides and do polynomial division.
 * Return true if a Greatest Common Divisor was found and
 * the expression was simplified.
 */
int
polydiv_simp(equation, np)
token_type	*equation;
int		*np;
{
	return polydiv_recurse(equation, np, 0, 1);
}

int
polydiv_recurse(equation, np, loc, level)
token_type	*equation;
int		*np, loc, level;
{
	int	modified;
	int	i, j, k;
	int	op;
	int	last_op2;
	int	len1, len2;
	int	iflag;

	modified = false;
	for (i = loc;;) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].level > level) {
			modified |= polydiv_recurse(equation, np, i, level + 1);
			i++;
			for (; i < *np && equation[i].level > level; i += 2)
				;
			continue;
		}
		i++;
	}
	if (modified)
		return true;
	for (i = loc + 1;; i += 2) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].level == level) {
			if (equation[i].token.operatr == DIVIDE) {
				op = 0;
				for (k = i + 2;; k += 2) {
					if (k >= *np || equation[k].level <= level)
						break;
					if (equation[k].level == level + 1)
						op = equation[k].token.operatr;
				}
				if (op != PLUS && op != MINUS)
					continue;
				len1 = k - (i + 1);
				last_op2 = 0;
				for (j = loc;; j++) {
					if (j >= *np || equation[j].level < level)
						break;
					if (equation[j].level == level && equation[j].kind == OPERATOR) {
						last_op2 = equation[j].token.operatr;
						continue;
					}
					if (equation[j].level >= level && last_op2 != DIVIDE) {
						op = 0;
						iflag = false;
						for (k = j + 1;; k += 2) {
#if	false
							if (equation[k-1].kind == VARIABLE
							    && equation[k-1].token.variable == IMAGINARY) {
								iflag = true;
							}
#endif
							if (k >= *np || equation[k].level <= level)
								break;
							if (equation[k].level == level + 1)
								op = equation[k].token.operatr;
						}
						if (iflag || (op != PLUS && op != MINUS)) {
							j = k - 1;
							continue;
						}
						len2 = k - j;
						if (poly2_gcd(&equation[i+1], len1, &equation[j], len2, 0L)) {
store_code:
							for (k = 0; k < n_tlhs; k++)
								tlhs[k].level += level;
							for (k = 0; k < n_trhs; k++)
								trhs[k].level += level;
							if (((*np + (n_trhs - len2)) > N_TOKENS)
							    || ((*np + (n_trhs - len2) + (n_tlhs - len1)) > N_TOKENS))
								error_huge();
							blt(&equation[j+n_trhs], &equation[j+len2], (*np - (j + len2)) * sizeof(*equation));
							*np += n_trhs - len2;
							if (i > j)
								i += n_trhs - len2;
							blt(&equation[j], &trhs[0], n_trhs * sizeof(token_type));
							blt(&equation[i+n_tlhs+1], &equation[i+1+len1], (*np - (i + 1 + len1)) * sizeof(*equation));
							*np += n_tlhs - len1;
							blt(&equation[i+1], &tlhs[0], n_tlhs * sizeof(*equation));
							printf("Found polynomial Greatest Common Divisor.  Division simplified.\n");
							return true;
						}
						if (poly2_gcd(&equation[j], len2, &equation[i+1], len1, 0L)) {
							k = j - 1;
							j = i + 1;
							i = k;
							k = len1;
							len1 = len2;
							len2 = k;
							goto store_code;
						}
						last_op2 = DIVIDE;
					}
				}
			} else if (equation[i].token.operatr != TIMES)
				break;
		}
	}
	return modified;
}

/*
 * Check for divides and do polynomial and smart division.
 * Return true if expression was simplified.
 */
int
div_remainder(equation, np, poly_flag)
token_type	*equation;
int		*np;
int		poly_flag;
{
	return pdiv_recurse(equation, np, 0, 1, poly_flag);
}

int
pdiv_recurse(equation, np, loc, level, code)
token_type	*equation;
int		*np, loc, level, code;
{
	int	modified;
	int	i, j, k;
	int	op;
	int	last_op2;
	int	len1, len2;
	int	real_len1;
	long	v;
	int	rv;
	int	flag;
	int	div_flag;
	int	power_flag;
	int	do_again;
	int	rem_zero;
	double	d;

	modified = false;
	for (i = loc + 1;; i += 2) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].level == level) {
			if (equation[i].token.operatr == DIVIDE) {
				for (k = i + 2;; k += 2) {
					if (k >= *np || equation[k].level <= level)
						break;
				}
				real_len1 = k - (i + 1);
				last_op2 = 0;
				for (j = loc;; j++) {
					if (j >= *np || equation[j].level < level)
						break;
					if (equation[j].level == level && equation[j].kind == OPERATOR) {
						last_op2 = equation[j].token.operatr;
						continue;
					}
					if (equation[j].level >= level && last_op2 != DIVIDE) {
						op = 0;
						for (k = j + 1;; k += 2) {
							if (k >= *np || equation[k].level <= level)
								break;
							if (equation[k].level == level + 1)
								op = equation[k].token.operatr;
						}
						if (op != PLUS && op != MINUS) {
							j = k - 1;
							continue;
						}
						len2 = k - j;
						flag = code;
						len1 = real_len1;
						do_again = false;
#if	true
						if (real_len1 > 2 && equation[i+real_len1-1].kind == OPERATOR
						    && equation[i+real_len1-1].level == (level + 1)
						    && equation[i+real_len1-1].token.operatr == POWER
						    && equation[i+real_len1].kind == CONSTANT
						    && equation[i+real_len1].token.constant > 1.0) {
							do_again = true;
						}
#endif
next_thingy:
						power_flag = do_again;
						if (power_flag) {
							d = equation[i+real_len1].token.constant;
							len1 = real_len1 - 2;
						} else {
							len1 = real_len1;
						}
						v = 0;
						div_flag = 2;
						if (flag || power_flag) {
							rv = poly_div(&equation[j], len2, &equation[i+1], len1, &v, &div_flag);
						} else {
							rv = smart_div(&equation[j], len2, &equation[i+1], len1);
						}
						rem_zero = (rv > 0 && n_trhs == 1 && trhs[0].kind == CONSTANT
						    && trhs[0].token.constant == 0.0);
						if (power_flag && !rem_zero) {
							rv = 0;
						}
						if (rv > 0 && (n_tlhs + 2 + n_trhs + len1) <= N_TOKENS) {
							for (k = 0; k < n_tlhs; k++)
								tlhs[k].level++;
							tlhs[n_tlhs].level = 1;
							tlhs[n_tlhs].kind = OPERATOR;
							tlhs[n_tlhs].token.operatr = PLUS;
							n_tlhs++;
							for (k = 0; k < n_trhs; k++)
								trhs[k].level += 2;
							blt(&tlhs[n_tlhs], &trhs[0], n_trhs * sizeof(token_type));
							n_tlhs += n_trhs;
							tlhs[n_tlhs].level = 2;
							tlhs[n_tlhs].kind = OPERATOR;
							tlhs[n_tlhs].token.operatr = DIVIDE;
							n_tlhs++;
							k = n_tlhs;
							blt(&tlhs[n_tlhs], &equation[i+1], len1 * sizeof(token_type));
							n_tlhs += len1;
							for (; k < n_tlhs; k++)
								tlhs[k].level += 2;
							side_debug(3, &equation[j], len2);
							side_debug(3, &equation[i+1], len1);
							simpb_side(&tlhs[0], &n_tlhs);
							side_debug(3, &tlhs[0], n_tlhs);
							if (power_flag) {
								if (d == 2.0)
									k = (n_tlhs < (len2 + 2));
								else
									k = (n_tlhs < len2);
							} else {
								k = (n_tlhs < (len1 + 1 + len2));
							}
							if (k) {
								if (power_flag) {
									if ((*np - len2 + n_tlhs) > N_TOKENS)
										error_huge();
								} else {
									if ((*np - (len1 + 1 + len2) + n_tlhs) > N_TOKENS)
										error_huge();
								}
								for (k = 0; k < n_tlhs; k++)
									tlhs[k].level += level;
								if (power_flag) {
									equation[i+len1+2].token.constant -= 1.0;
								} else {
									blt(&equation[i], &equation[i+1+len1], (*np - (i + 1 + len1)) * sizeof(token_type));
									*np -= len1 + 1;
									if (i < j) {
										j -= len1 + 1;
									}
								}
								blt(&equation[j+n_tlhs], &equation[j+len2], (*np - (j + len2)) * sizeof(token_type));
								*np -= len2 - n_tlhs;
								blt(&equation[j], &tlhs[0], n_tlhs * sizeof(token_type));
								modified = true;
								if (flag || power_flag) {
									printf("Polynomial");
								} else {
									printf("Smart");
								}
								printf(" division successful (quotient ");
								if (!rem_zero) {
									printf("and remainder ");
								}
								printf("substituted).\n");
								return true;
							}
						}
						if (do_again) {
							do_again = false;
							goto next_thingy;
						}
						if (flag == code) {
							flag = !flag;
							goto next_thingy;
						}
						last_op2 = DIVIDE;
					}
				}
			} else if (equation[i].token.operatr != TIMES)
				break;
		}
	}
	for (i = loc;;) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].level > level) {
			modified |= pdiv_recurse(equation, np, i, level + 1, code);
			i++;
			for (; i < *np && equation[i].level > level; i += 2)
				;
			continue;
		}
		i++;
	}
	return modified;
}

/*
 * Do polynomial division.
 * Returns true if successful.
 * Quotient is returned in "tlhs" and remainder in "trhs".
 */
int
poly_div(d1, len1, d2, len2, vp, div_flagp)
token_type	*d1;		/* pointer to dividend */
int		len1;		/* length of dividend */
token_type	*d2;		/* pointer to divisor */
int		len2;		/* length of divisor */
long		*vp;
int		*div_flagp;
{
	int		i, j, k;
	jmp_buf		save_save;

	blt(save_save, jmp_save, sizeof(jmp_save));
	if ((i = setjmp(jmp_save)) != 0) {
		domain_check = false;
		blt(jmp_save, save_save, sizeof(jmp_save));
		if (i == 13) {
			longjmp(jmp_save, i);
		}
		printf("Polynomial division recovering...\n");
		return false;
	}
	j = poly_div_sub(d1, len1, d2, len2, vp, div_flagp);
	blt(jmp_save, save_save, sizeof(jmp_save));
	return j;
}

int
poly_div_sub(d1, len1, d2, len2, vp, div_flagp)
token_type	*d1;		/* pointer to dividend */
int		len1;		/* length of dividend */
token_type	*d2;		/* pointer to divisor */
int		len2;		/* length of divisor */
long		*vp;
int		*div_flagp;
{
	int		i, j, k;
	int		t1, len_t1;
	int		t2, len_t2;
	int		sign;
	int		divide_flag;
	token_type	divisor[DIVISOR_SIZE];
	int		n_divisor;
	token_type	quotient[DIVISOR_SIZE];
	int		n_quotient;
	double		last_power;
	double		divisor_power;
	int		sum_size;
	double		d;
	int		last_count, count;

	if (len1 <= 0 || len2 <= 0) {
		printf("Programming error in call to poly_div(): bad length.\n");
		return false;
	}
	if (len1 > N_TOKENS || len2 > N_TOKENS)
		return false;
	if (&trhs[0] != &d1[0]) {
		blt(&trhs[0], d1, len1 * sizeof(token_type));
		n_trhs = len1;
	}
	if (&tlhs[0] != &d2[0]) {
		blt(&tlhs[0], d2, len2 * sizeof(token_type));
		n_tlhs = len2;
	}
	uf_simp(&trhs[0], &n_trhs);
	uf_simp(&tlhs[0], &n_tlhs);
	if (*vp == 0) {
		if (find_highest_count(&trhs[0], n_trhs, &tlhs[0], n_tlhs, vp) <= 0)
			return false;
	}
	if (debug_level >= 3) {
		printf("poly_div starts using variable (");
		list_var(*vp, true);
		printf("):\n");
		side_debug(3, &trhs[0], n_trhs);
		side_debug(3, &tlhs[0], n_tlhs);
	}
	divide_flag = 2;
	last_count = find_greatest_power(&trhs[0], n_trhs, vp, &last_power, &t1, &len_t1, &divide_flag);
	*div_flagp = divide_flag;
	find_greatest_power(&tlhs[0], n_tlhs, vp, &divisor_power, &t2, &len_t2, &divide_flag);
	if (divisor_power <= 0 || last_power < divisor_power) {
		divide_flag = !divide_flag;
		*div_flagp = divide_flag;
		last_count = find_greatest_power(&trhs[0], n_trhs, vp, &last_power, &t1, &len_t1, &divide_flag);
		find_greatest_power(&tlhs[0], n_tlhs, vp, &divisor_power, &t2, &len_t2, &divide_flag);
		if (divisor_power <= 0 || last_power < divisor_power) {
			return false;
		}
	}
	n_quotient = 1;
	quotient[0].level = 1;
	quotient[0].kind = CONSTANT;
	quotient[0].token.constant = 0.0;
	if (n_tlhs > ARR_CNT(divisor))
		return false;
	blt(&divisor[0], &tlhs[0], n_tlhs * sizeof(token_type));
	n_divisor = n_tlhs;
	sum_size = n_trhs + n_quotient;
	for (;;) {
		sign = PLUS;
		if (t1 > 0 && trhs[t1-1].token.operatr == MINUS)
			sign = MINUS;
		if (t2 > 0 && divisor[t2-1].token.operatr == MINUS) {
			if (sign == MINUS)
				sign = PLUS;
			else
				sign = MINUS;
		}
		if ((len_t1 + len_t2 + 1) > N_TOKENS)
			return false;
		blt(&tlhs[0], &trhs[t1], len_t1 * sizeof(token_type));
		n_tlhs = len_t1;
		for (i = 0; i < n_tlhs; i++)
			tlhs[i].level++;
		tlhs[n_tlhs].level = 1;
		tlhs[n_tlhs].kind = OPERATOR;
		tlhs[n_tlhs].token.operatr = DIVIDE;
		n_tlhs++;
		blt(&tlhs[n_tlhs], &divisor[t2], len_t2 * sizeof(token_type));
		i = n_tlhs;
		n_tlhs += len_t2;
		for (; i < n_tlhs; i++)
			tlhs[i].level++;
		if (!simp_loop(&tlhs[0], &n_tlhs))
			return false;
		if ((n_quotient + 1 + n_tlhs) > min(ARR_CNT(quotient), N_TOKENS))
			return false;
		for (i = 0; i < n_tlhs; i++)
			tlhs[i].level++;
		quotient[n_quotient].level = 1;
		quotient[n_quotient].kind = OPERATOR;
		quotient[n_quotient].token.operatr = sign;
		n_quotient++;
		blt(&quotient[n_quotient], &tlhs[0], n_tlhs * sizeof(token_type));
		n_quotient += n_tlhs;
		if ((n_trhs + n_tlhs + n_divisor + 2) > N_TOKENS)
			return false;
		blt(&trhs[t1+1], &trhs[t1+len_t1], (n_trhs - (t1 + len_t1)) * sizeof(token_type));
		n_trhs -= len_t1 - 1;
		trhs[t1].level = 1;
		trhs[t1].kind = CONSTANT;
		trhs[t1].token.constant = 0.0;
		for (i = 0; i < n_trhs; i++)
			trhs[i].level++;
		trhs[n_trhs].level = 1;
		trhs[n_trhs].kind = OPERATOR;
		if (sign == PLUS)
			trhs[n_trhs].token.operatr = MINUS;
		else
			trhs[n_trhs].token.operatr = PLUS;
		n_trhs++;
		blt(&trhs[n_trhs], &tlhs[0], n_tlhs * sizeof(token_type));
		i = n_trhs;
		n_trhs += n_tlhs;
		for (; i < n_trhs; i++)
			trhs[i].level++;
		trhs[n_trhs].level = 2;
		trhs[n_trhs].kind = OPERATOR;
		trhs[n_trhs].token.operatr = TIMES;
		n_trhs++;
		i = n_trhs;
		blt(&trhs[n_trhs], &divisor[0], t2 * sizeof(token_type));
		n_trhs += t2;
		trhs[n_trhs].level = 1;
		trhs[n_trhs].kind = CONSTANT;
		trhs[n_trhs].token.constant = 0.0;
		n_trhs++;
		blt(&trhs[n_trhs], &divisor[t2+len_t2], (n_divisor - (t2 + len_t2)) * sizeof(token_type));
		n_trhs += (n_divisor - (t2 + len_t2));
		for (; i < n_trhs; i++)
			trhs[i].level += 2;
		side_debug(3, &trhs[0], n_trhs);
		uf_tsimp(&trhs[0], &n_trhs);
		side_debug(4, &trhs[0], n_trhs);
		count = find_greatest_power(&trhs[0], n_trhs, vp, &d, &t1, &len_t1, &divide_flag);
		if (d < divisor_power) {
			blt(&tlhs[0], &quotient[0], n_quotient * sizeof(token_type));
			n_tlhs = n_quotient;
			side_debug(3, &tlhs[0], n_tlhs);
			side_debug(3, &trhs[0], n_trhs);
			if (n_trhs == 1 && trhs[0].kind == CONSTANT && trhs[0].token.constant == 0.0)
				return true;
			if ((n_trhs + n_quotient) >= (sum_size - (sum_size / 10))) {
				if ((n_trhs + 1) > sum_size
				    && n_trhs > n_divisor)
					return -2;
				else
					return -1;
			}
			return true;
		} else if (d < last_power) {
			last_power = d;
			last_count = count;
		} else if (d > last_power) {
			return false;
		} else {
			if (count >= last_count) {
				return false;
			}
			last_count = count;
		}
	}
}

#define	MAX_TRIES	40

/*
 * Do smart division.
 * Returns true if successful.
 * Quotient is returned in "tlhs" and remainder in "trhs".
 */
int
smart_div(d1, len1, d2, len2)
token_type	*d1;		/* pointer to dividend */
int		len1;		/* length of dividend */
token_type	*d2;		/* pointer to divisor */
int		len2;		/* length of divisor */
{
	int		i, j, k;
	int		t1, len_t1;
	int		t2, len_t2;
	int		sign;
	token_type	divisor[DIVISOR_SIZE];
	int		n_divisor;
	token_type	quotient[DIVISOR_SIZE];
	int		n_quotient, old_n_quotient;
	int		trhs_size;
	int		term_size, term_count;
	int		term_pos, skip_terms[MAX_TRIES];
	int		skip_count;
	token_type	*qp;
	int		q_size;
	int		sum_size;
	int		count;
	int		dcount;		/* divisor term count */
	int		one_count;
	int		vflag;

	blt(&trhs[0], d1, len1 * sizeof(token_type));
	n_trhs = len1;
	blt(&tlhs[0], d2, len2 * sizeof(token_type));
	n_tlhs = len2;
	uf_simp_no_repeat(&trhs[0], &n_trhs);
	uf_simp_no_repeat(&tlhs[0], &n_tlhs);
	if (debug_level >= 3) {
		printf("smart_div starts:\n");
		side_debug(3, &trhs[0], n_trhs);
		side_debug(3, &tlhs[0], n_tlhs);
	}
	dcount = 0;
	for (i = 0, j = 0, len_t2 = 0, vflag = false;; i++) {
		if (i >= n_tlhs || (tlhs[i].kind == OPERATOR && tlhs[i].level == 1
		    && (tlhs[i].token.operatr == PLUS || tlhs[i].token.operatr == MINUS))) {
			dcount++;
			if (vflag) {
				if ((i - j) > len_t2) {
					len_t2 = i - j;
					t2 = j;
				}
			}
			vflag = false;
			j = i + 1;
		} else if (tlhs[i].kind == VARIABLE
		    && (tlhs[i].token.variable & VAR_MASK) > SIGN) {
			vflag = true;
		}
		if (i >= n_tlhs)
			break;
	}
	if (len_t2 <= 0)
		return false;
#if	false
	if (dcount <= 1)
		return false;
#endif
	n_quotient = 1;
	quotient[0].level = 1;
	quotient[0].kind = CONSTANT;
	quotient[0].token.constant = 0.0;
	if (n_tlhs > ARR_CNT(divisor))
		return false;
	blt(&divisor[0], &tlhs[0], n_tlhs * sizeof(token_type));
	n_divisor = n_tlhs;
	one_count = false;
	trhs_size = n_trhs;
	skip_count = 0;
	for (count = 0;;) {
		sum_size = n_trhs + n_quotient;
		for (term_count = 1, q_size = 0;; term_count++) {
			if (!get_term(&trhs[0], n_trhs, term_count, &t1, &len_t1))
				break;
			j = false;
			for (i = 0; i < skip_count; i++) {
				if (skip_terms[i] == t1) {
					j = true;
					break;
				}
			}
			if (j)
				continue;
			if ((len_t1 + len_t2 + 1) > N_TOKENS)
				return false;
			blt(&tlhs[0], &trhs[t1], len_t1 * sizeof(token_type));
			n_tlhs = len_t1;
			for (i = 0; i < n_tlhs; i++)
				tlhs[i].level++;
			tlhs[n_tlhs].level = 1;
			tlhs[n_tlhs].kind = OPERATOR;
			tlhs[n_tlhs].token.operatr = DIVIDE;
			n_tlhs++;
			blt(&tlhs[n_tlhs], &divisor[t2], len_t2 * sizeof(token_type));
			i = n_tlhs;
			n_tlhs += len_t2;
			for (; i < n_tlhs; i++)
				tlhs[i].level++;
			if (!simp_loop(&tlhs[0], &n_tlhs))
				continue;
			if (basic_size(&tlhs[0], n_tlhs) <= basic_size(&trhs[t1], len_t1)) {
				one_count = true;
				q_size = n_tlhs;
				term_pos = t1;
				term_size = len_t1;
				break;
			} else if (one_count) {
				continue;
			}
			if (q_size == 0 || n_tlhs < q_size) {
				q_size = n_tlhs;
				term_pos = t1;
				term_size = len_t1;
			}
		}
		if (q_size <= 0) {
			if (count <= 0)
				return false;
end_div:
			if (dcount > 1) {
				i = n_quotient + n_trhs;
				if (i + (i / 5) >= trhs_size + 1) {
					return false;
				}
			}
end_div2:
			blt(&tlhs[0], &quotient[0], n_quotient * sizeof(token_type));
			n_tlhs = n_quotient;
			side_debug(3, &tlhs[0], n_tlhs);
			side_debug(3, &trhs[0], n_trhs);
			return true;
		}
		t1 = term_pos;
		len_t1 = term_size;
		sign = PLUS;
		if (t1 > 0 && trhs[t1-1].token.operatr == MINUS)
			sign = MINUS;
		if (t2 > 0 && divisor[t2-1].token.operatr == MINUS) {
			if (sign == MINUS)
				sign = PLUS;
			else
				sign = MINUS;
		}
		if ((len_t1 + len_t2 + 1) > N_TOKENS)
			return false;
		blt(&tlhs[0], &trhs[t1], len_t1 * sizeof(token_type));
		n_tlhs = len_t1;
		for (i = 0; i < n_tlhs; i++)
			tlhs[i].level++;
		tlhs[n_tlhs].level = 1;
		tlhs[n_tlhs].kind = OPERATOR;
		tlhs[n_tlhs].token.operatr = DIVIDE;
		n_tlhs++;
		blt(&tlhs[n_tlhs], &divisor[t2], len_t2 * sizeof(token_type));
		i = n_tlhs;
		n_tlhs += len_t2;
		for (; i < n_tlhs; i++)
			tlhs[i].level++;
		simp_loop(&tlhs[0], &n_tlhs);
		if ((n_quotient + 1 + n_tlhs) > min(ARR_CNT(quotient), N_TOKENS))
			return false;
		for (i = 0; i < n_tlhs; i++)
			tlhs[i].level++;
		old_n_quotient = n_quotient;
		quotient[n_quotient].level = 1;
		quotient[n_quotient].kind = OPERATOR;
		quotient[n_quotient].token.operatr = sign;
		n_quotient++;
		qp = &quotient[n_quotient];
		q_size = n_tlhs;
		blt(&quotient[n_quotient], &tlhs[0], n_tlhs * sizeof(token_type));
		n_quotient += n_tlhs;
		if ((n_trhs + q_size + n_divisor + 2) > N_TOKENS)
			return false;
		blt(&tlhs[0], &trhs[0], n_trhs * sizeof(token_type));
		n_tlhs = n_trhs;
		blt(&trhs[t1+1], &trhs[t1+len_t1], (n_trhs - (t1 + len_t1)) * sizeof(token_type));
		n_trhs -= len_t1 - 1;
		trhs[t1].level = 1;
		trhs[t1].kind = CONSTANT;
		trhs[t1].token.constant = 0.0;
		for (i = 0; i < n_trhs; i++)
			trhs[i].level++;
		trhs[n_trhs].level = 1;
		trhs[n_trhs].kind = OPERATOR;
		if (sign == PLUS)
			trhs[n_trhs].token.operatr = MINUS;
		else
			trhs[n_trhs].token.operatr = PLUS;
		n_trhs++;
		blt(&trhs[n_trhs], qp, q_size * sizeof(token_type));
		i = n_trhs;
		n_trhs += q_size;
		for (; i < n_trhs; i++)
			trhs[i].level++;
		trhs[n_trhs].level = 2;
		trhs[n_trhs].kind = OPERATOR;
		trhs[n_trhs].token.operatr = TIMES;
		n_trhs++;
		i = n_trhs;
		blt(&trhs[n_trhs], &divisor[0], t2 * sizeof(token_type));
		n_trhs += t2;
		trhs[n_trhs].level = 1;
		trhs[n_trhs].kind = CONSTANT;
		trhs[n_trhs].token.constant = 0.0;
		n_trhs++;
		blt(&trhs[n_trhs], &divisor[t2+len_t2], (n_divisor - (t2 + len_t2)) * sizeof(token_type));
		n_trhs += (n_divisor - (t2 + len_t2));
		for (; i < n_trhs; i++)
			trhs[i].level += 2;
		side_debug(3, &trhs[0], n_trhs);
		uf_tsimp(&trhs[0], &n_trhs);
		side_debug(4, &trhs[0], n_trhs);
		if (n_trhs == 1 && trhs[0].kind == CONSTANT && trhs[0].token.constant == 0.0)
			goto end_div2;
		if (dcount > 1) {
			one_count = false;
			if ((n_trhs + n_quotient) >= sum_size) {
				if (skip_count >= ARR_CNT(skip_terms)) {
					if (count == 0) {
						return false;
					} else {
						n_quotient = old_n_quotient;
						blt(&trhs[0], &tlhs[0], n_tlhs * sizeof(token_type));
						n_trhs = n_tlhs;
						goto end_div;
					}
				}
				skip_terms[skip_count] = term_pos;
				skip_count++;
				n_quotient = old_n_quotient;
				blt(&trhs[0], &tlhs[0], n_tlhs * sizeof(token_type));
				n_trhs = n_tlhs;
				continue;
			}
		}
		if (n_trhs == 1 && trhs[0].kind == CONSTANT)
			goto end_div;
		skip_count = 0;
		count++;
	}
}

int
basic_size(p1, len)
token_type	*p1;
int		len;
{
	int	i;
	int	level;

	if (len <= 1) {
		if (len == 1 && p1[0].kind == CONSTANT)
			return -1;
		else
			return len;
	}
	level = min_level(p1, len);
	if (p1[1].level == level
	    && (p1[1].token.operatr == TIMES || p1[1].token.operatr == DIVIDE)
	    && (p1[0].kind == CONSTANT)) {
		return(len - 2);
	} else {
		return len;
	}
}

int
get_term(p1, n1, count, tp1, lentp1)
token_type	*p1;
int		n1;
int		count;
int		*tp1, *lentp1;
{
	int	i, j;
	int	no;

	for (no = 0, i = 1, j = 0;; i += 2) {
		if (i >= n1 || (p1[i].level == 1
		    && (p1[i].token.operatr == PLUS
		    || p1[i].token.operatr == MINUS))) {
			no++;
			if (no >= count) {
				*tp1 = j;
				*lentp1 = i - j;
				return true;
			}
			j = i + 1;
		}
		if (i >= n1)
			return false;
	}
}

int
find_highest_count(p1, n1, p2, n2, vp1)
token_type	*p1;
int		n1;
token_type	*p2;
int		n2;
long		*vp1;
{
	int		i, j;
	int		vc, cnt;
	long		v1, last_v;
	sort_type	va[MAX_VARS];
	int		divide_flag;
	int		t1, len_t1;
	int		t2, len_t2;
	double		d1, d2;
	int		try_sign;
	int		count1, count2;

	vc = 0;
	last_v = -1;
	for (;;) {
		v1 = -1;
		for (i = 0; i < n1; i += 2) {
			if (p1[i].kind == VARIABLE
			    && p1[i].token.variable > last_v) {
				if (v1 == -1 || p1[i].token.variable < v1) {
					v1 = p1[i].token.variable;
					cnt = 1;
				} else if (p1[i].token.variable == v1) {
					cnt++;
				}
			}
		}
		if (v1 == -1)
			break;
		last_v = v1;
		if (vc >= ARR_CNT(va)) {
			printf("Warning: Too many variables!\n");
			break;
		}
		va[vc].v = v1;
		va[vc++].count = cnt;
	}
	if (vc <= 0)
		return 0;
	qsort((char *) &va[0], vc, sizeof(*va), vcmp);
	try_sign = false;
do_again:
	for (i = 0; i < vc; i++) {
		if ((va[i].v & VAR_MASK) <= SIGN) {
			if (!try_sign)
				continue;
		} else {
			if (try_sign)
				continue;
		}
		*vp1 = va[i].v;
		divide_flag = 2;
		count1 = find_greatest_power(p1, n1, vp1, &d1, &t1, &len_t1, &divide_flag);
		count2 = find_greatest_power(p2, n2, vp1, &d2, &t2, &len_t2, &divide_flag);
		if (d2 <= 0 || d1 < d2 || count2 > count1) {
			divide_flag = !divide_flag;
			count1 = find_greatest_power(p1, n1, vp1, &d1, &t1, &len_t1, &divide_flag);
			count2 = find_greatest_power(p2, n2, vp1, &d2, &t2, &len_t2, &divide_flag);
			if (d2 <= 0 || d1 < d2 || count2 > count1) {
				continue;
			}
		}
		return va[i].count;
	}
	if (!try_sign) {
		try_sign = true;
		goto do_again;
	}
	return 0;
}

#define	VALUE_CNT	3

term_value(dp, p1, n1, loc)
double		*dp;
token_type	*p1;
int		n1, loc;
{
	int	i, j, k;
	int	divide_flag;
	int	level, div_level;
	double	d, sub_count, sub_sum;

	for (i = 0; i < VALUE_CNT; i++)
		dp[i] = 0.0;
	divide_flag = false;
	for (i = loc; i < n1; i++) {
		level = p1[i].level;
		if (p1[i].kind == VARIABLE) {
			if (divide_flag) {
				dp[0] -= 1.0;
				dp[1] -= p1[i].token.variable;
				dp[2] -= p1[i].token.variable;
			} else {
				dp[0] += 1.0;
				dp[1] += p1[i].token.variable;
				dp[2] += p1[i].token.variable;
			}
		} else if (p1[i].kind == OPERATOR) {
			if (level == 1 && (p1[i].token.operatr == PLUS || p1[i].token.operatr == MINUS))
				break;
			if (p1[i].token.operatr == DIVIDE) {
				if (divide_flag && level >= div_level)
					continue;
				div_level = level;
				divide_flag = true;
			} else if (divide_flag && level <= div_level) {
				divide_flag = false;
			}
		}
	}
	divide_flag = false;
	for (j = loc + 1; j < i; j += 2) {
		level = p1[j].level;
		if (p1[j].token.operatr == DIVIDE) {
			if (divide_flag && level >= div_level)
				continue;
			div_level = level;
			divide_flag = true;
		} else if (divide_flag && level <= div_level) {
			divide_flag = false;
		}
		if (p1[j].token.operatr == POWER && level == p1[j+1].level
		    && p1[j+1].kind == CONSTANT) {
			d = p1[j+1].token.constant - 1.0;
			sub_count = 0.0;
			sub_sum = 0.0;
			for (k = j - 1; k >= loc && p1[k].level >= level; k--) {
				if (p1[k].kind == VARIABLE) {
					sub_count += 1;
					sub_sum += p1[k].token.variable;
				}
			}
			if (divide_flag) {
				dp[0] -= d * sub_count;
				dp[2] -= d * sub_sum;
			} else {
				dp[0] += d * sub_count;
				dp[2] += d * sub_sum;
			}
		}
	}
}

int
find_greatest_power(p1, n1, vp1, pp1, tp1, lentp1, dcodep)
token_type	*p1;
int		n1;
long		*vp1;
double		*pp1;
int		*tp1, *lentp1;
int		*dcodep;
{
	int		i, j, k;
	int		ii;
	int		flag;
	double		d;
	int		divide_flag;
	int		div_level;
	int		level;
	long		v;
	int		was_power;
	double		last_va[VALUE_CNT];
	double		va[VALUE_CNT];
	int		rv;
	int		count;

	*pp1 = 0.0;
	*tp1 = -1;
	v = 0;
	was_power = false;
	rv = *dcodep;
	count = 0;
	if (n1 < 1)
		return count;
	divide_flag = false;
	for (j = 0, i = 1;; i += 2) {
		if (i >= n1 || ((p1[i].token.operatr == PLUS || p1[i].token.operatr == MINUS) && p1[i].level == 1)) {
			divide_flag = false;
			if (!was_power && *pp1 <= 1.0) {
				for (k = j; k < i; k++) {
					if (p1[k].kind == VARIABLE) {
						if (*dcodep <= 1 && *dcodep != divide_flag)
							continue;
						if (*vp1) {
							if (p1[k].token.variable == *vp1) {
								term_value(&va[0], p1, n1, j);
								flag = (*pp1 == 1.0 && (rv > divide_flag));
								if (*pp1 == 1.0 && rv == divide_flag) {
									if (*tp1 != j)
										count++;
									for (ii = 0; ii < ARR_CNT(va); ii++) {
										if (va[ii] == last_va[ii])
											continue;
										if (va[ii] < last_va[ii])
											flag = true;
										break;
									}
								} else if (*pp1 < 1.0 || flag) {
									count = 1;
								}
								if (*pp1 < 1.0 || flag) {
									blt(&last_va[0], &va[0], sizeof(last_va));
									*pp1 = 1.0;
									*tp1 = j;
									rv = divide_flag;
								}
								break;
							}
						} else {
							v = p1[k].token.variable;
							*pp1 = 1.0;
							*tp1 = j;
							rv = divide_flag;
							break;
						}
					} else if (p1[k].kind == OPERATOR) {
						if (p1[k].token.operatr == DIVIDE) {
							if (divide_flag && p1[k].level >= div_level)
								continue;
							div_level = p1[k].level;
							divide_flag = true;
						} else if (divide_flag && p1[k].level <= div_level) {
							divide_flag = false;
						}
						if (p1[k].token.operatr == POWER) {
							level = p1[k].level;
							do {
								k += 2;
							} while (k < i && p1[k].level > level);
							k--;
						}
					}
				}
			}
			if (i >= n1)
				break;
			j = i + 1;
			was_power = false;
			divide_flag = false;
			continue;
		}
		level = p1[i].level;
		if (p1[i].token.operatr == DIVIDE) {
			if (divide_flag && level >= div_level)
				continue;
			div_level = level;
			divide_flag = true;
		} else if (divide_flag && level <= div_level) {
			divide_flag = false;
		}
		if (p1[i].token.operatr == POWER && p1[i+1].kind == CONSTANT
		    && (*vp1 || level == p1[i+1].level)) {
			if (*dcodep <= 1 && *dcodep != divide_flag)
				continue;
			d = p1[i+1].token.constant;
			for (k = i;;) {
				if (p1[k-1].kind == VARIABLE) {
					if (*vp1) {
						if (p1[k-1].token.variable == *vp1) {
							was_power = true;
							term_value(&va[0], p1, n1, j);
							flag = (d == *pp1 && (rv > divide_flag));
							if (d == *pp1 && rv == divide_flag) {
								if (*tp1 != j)
									count++;
								for (ii = 0; ii < ARR_CNT(va); ii++) {
									if (va[ii] == last_va[ii])
										continue;
									if (va[ii] < last_va[ii])
										flag = true;
									break;
								}
							} else if (d > *pp1 || flag) {
								count = 1;
							}
							if (d > *pp1 || flag) {
								blt(&last_va[0], &va[0], sizeof(last_va));
								*pp1 = d;
								*tp1 = j;
								rv = divide_flag;
							}
							break;
						}
					} else {
						was_power = true;
						if (d > *pp1) {
							v = p1[k-1].token.variable;
							*pp1 = d;
							*tp1 = j;
							rv = divide_flag;
						}
						break;
					}
				}
				k -= 2;
				if (k <= j)
					break;
				if (p1[k].level <= level)
					break;
			}
		}
	}
	if (*vp1 == 0)
		*vp1 = v;
	if (*tp1 >= 0) {
		for (i = *tp1 + 1; i < n1; i += 2) {
			if ((p1[i].token.operatr == PLUS || p1[i].token.operatr == MINUS)
			    && p1[i].level == 1) {
				break;
			}
		}
		*lentp1 = i - *tp1;
	}
	if (*dcodep == 2)
		*dcodep = rv;
	return count;
}

/*
 * Combine denominators in equation side.
 * This means converting "a/b+c/d" to "(a*d+c*b)/b/d".
 * Return true if equation side was modified.
 */
int
super_factor(equation, np, start_flag)
token_type	*equation;
int		*np;
int		start_flag;
{
	int	rv;

	group_proc(equation, np, true);
	rv = sf_recurse(equation, np, 0, 1, start_flag);
	if (!rv) {
		do {
			organize(equation, np);
		} while (elim_k(equation, np));
	}
	return rv;
}

int
sf_recurse(equation, np, loc, level, start_flag)
token_type	*equation;
int		*np, loc, level;
int		start_flag;
{
	int	modified;
	int	i, j, k;
	int	op;
	int	len1, len2;

	if (!start_flag) {
		for (i = loc + 1;; i += 2) {
			if (i >= *np || equation[i].level < level)
				break;
			if (equation[i].level == level
			    && (equation[i].token.operatr == DIVIDE
			    || equation[i].token.operatr == POWER)) {
				if (equation[i+1].level == level
				    && equation[i+1].kind == CONSTANT
				    && equation[i+1].token.constant == 1.0)
					continue;
				start_flag = true;
				break;
			}
		}
	}
	modified = false;
	op = 0;
	for (i = loc;;) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].level > level) {
			modified |= sf_recurse(equation, np, i, level + 1, start_flag);
			i++;
			for (; i < *np && equation[i].level > level; i += 2)
				;
			continue;
		} else if (equation[i].kind == OPERATOR) {
			op = equation[i].token.operatr;
		}
		i++;
	}
	if (modified)
		return true;
	switch (op) {
	case PLUS:
	case MINUS:
		break;
	default:
		return modified;
	}
	if (!start_flag)
		return modified;
sf_again:
	i = loc;
	for (k = i + 1;; k += 2) {
		if (k >= *np || equation[k].level <= level)
			break;
	}
	len1 = k - i;
	for (j = i + len1 + 1;; j += len2 + 1) {
		if (j >= *np || equation[j-1].level < level)
			break;
		for (k = j + 1;; k += 2) {
			if (k >= *np || equation[k].level <= level)
				break;
		}
		len2 = k - j;
		if (sf_sub(equation, np, loc, i, len1, j, len2, level + 1)) {
			modified = true;
			goto sf_again;
		}
	}
	return modified;
}

int
sf_sub(equation, np, loc, i1, n1, i2, n2, level)
token_type	*equation;
int		*np, loc, i1, n1, i2, n2, level;
{
	int	i, j, k;
	int	b1, b2;
	int	len;
	int	e1, e2;
	int	op1, op2;

	e1 = i1 + n1;
	e2 = i2 + n2;
	op2 = equation[i2-1].token.operatr;
	if (i1 <= loc) {
		op1 = PLUS;
	} else
		op1 = equation[i1-1].token.operatr;
	for (i = i1 + 1;; i += 2) {
		if (i >= e1)
			return false;
		if (equation[i].level == level && equation[i].token.operatr == DIVIDE)
			break;
	}
	b1 = i + 1;
	for (i += 2; i < e1; i += 2) {
		if (equation[i].level == level)
			break;
	}
	for (j = i2 + 1;; j += 2) {
		if (j >= e2)
			return false;
		if (equation[j].level == level && equation[j].token.operatr == DIVIDE)
			break;
	}
	b2 = j + 1;
	for (j += 2; j < e2; j += 2) {
		if (equation[j].level == level)
			break;
	}
	if (j - b2 == 1 && equation[b2].kind == CONSTANT
	    && equation[b2].token.constant == 1.0
	    && i - b1 == 1 && equation[b1].kind == CONSTANT
	    && equation[b1].token.constant == 1.0)
		return false;
	if (n1 + n2 + (i - b1) + (j - b2) + 8 > N_TOKENS) {
		error_huge();
	}
	len = (b1 - 1) - i1;
	blt(&scratch[0], &equation[i1], len * sizeof(*equation));
	if (op1 == MINUS) {
		equation[i1-1].token.operatr = PLUS;
		scratch[len].level = level;
		scratch[len].kind = OPERATOR;
		scratch[len].token.operatr = TIMES;
		len++;
		scratch[len].level = level;
		scratch[len].kind = CONSTANT;
		scratch[len].token.constant = -1.0;
		len++;
	}
	blt(&scratch[len], &equation[i], (e1 - i) * sizeof(*equation));
	len += e1 - i;
	scratch[len].level = level;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = TIMES;
	len++;
	blt(&scratch[len], &equation[b2], (j - b2) * sizeof(*equation));
	len += j - b2;
	for (k = 0; k < len; k++)
		scratch[k].level += 2;
	scratch[len].level = level + 1;
	scratch[len].kind = OPERATOR;
	if (op2 == MINUS)
		scratch[len].token.operatr = MINUS;
	else
		scratch[len].token.operatr = PLUS;
	len++;
	k = len;
	blt(&scratch[len], &equation[i2], (b2 - i2 - 1) * sizeof(*equation));
	len += b2 - i2 - 1;
	blt(&scratch[len], &equation[j], (e2 - j) * sizeof(*equation));
	len += e2 - j;
	scratch[len].level = level;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = TIMES;
	len++;
	blt(&scratch[len], &equation[b1], (i - b1) * sizeof(*equation));
	len += i - b1;
	for (; k < len; k++)
		scratch[k].level += 2;
	scratch[len].level = level;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = DIVIDE;
	len++;
	k = len;
	blt(&scratch[len], &equation[b1], (i - b1) * sizeof(*equation));
	len += i - b1;
	scratch[len].level = level;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = TIMES;
	len++;
	blt(&scratch[len], &equation[b2], (j - b2) * sizeof(*equation));
	len += j - b2;
	for (; k < len; k++)
		scratch[k].level++;
	if (*np + len - n1 - (n2 + 1) > N_TOKENS) {
		error_huge();
	}
	blt(&equation[i2-1], &equation[e2], (*np - e2) * sizeof(*equation));
	*np -= n2 + 1;
	blt(&equation[i1+len], &equation[e1], (*np - e1) * sizeof(*equation));
	*np += len - n1;
	blt(&equation[i1], &scratch[0], len * sizeof(*equation));
	return true;
}

/*
 * This function is the guts of the "group" command.
 */
group_sub(n)
int	n;
{
	elim_loop(&lhs[n][0], &n_lhs[n]);
	make_fractions(&lhs[n][0], &n_lhs[n]);
	group_proc(&lhs[n][0], &n_lhs[n], false);
	elim_loop(&rhs[n][0], &n_rhs[n]);
	make_fractions(&rhs[n][0], &n_rhs[n]);
	group_proc(&rhs[n][0], &n_rhs[n], false);
}

/*
 * Group denominators.
 * Make denominators when there are none if "dflag" is true.
 * Grouping here means converting "a/b/c" to "a/(b*c)".
 */
group_proc(equation, np, dflag)
token_type	*equation;
int		*np;
int		dflag;
{
	group_recurse(equation, np, 0, 1, dflag, 0);
}

group_recurse(equation, np, loc, level, dflag, high_op)
token_type	*equation;
int		*np, loc, level, dflag, high_op;
{
	int		i, j;
	int		len;
	int		di, edi;
	int		op;
	int		grouper;
	int		e1;

	op = 0;
	for (i = loc + 1;; i += 2) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].level == level) {
			op = equation[i].token.operatr;
			break;
		}
	}
	for (i = loc;;) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].level > level) {
g_single:
			group_recurse(equation, np, i, level + 1, dflag, op);
			i++;
			for (; i < *np && equation[i].level > level; i += 2)
				;
			continue;
		} else if (dflag && equation[i].kind != OPERATOR) {
			switch (op) {
			case PLUS:
			case MINUS:
				equation[i].level++;
				goto g_single;
			}
		}
		i++;
	}
	e1 = i;
	di = -1;
	edi = *np;
	grouper = false;
	for (i = loc + 1; i < e1; i += 2) {
		if (equation[i].level == level) {
			if (equation[i].token.operatr == DIVIDE) {
				if (di < 0)
					di = i;
				else {
					grouper = true;
					for (len = i + 2; len < e1; len += 2) {
						if (equation[len].level == level
						    && equation[len].kind == OPERATOR
						    && equation[len].token.operatr != DIVIDE)
							break;
					}
					len -= i;
					if (edi == *np) {
						i += len;
						edi = i;
						continue;
					}
					blt(&scratch[0], &equation[i], len * sizeof(*equation));
					blt(&equation[di+len], &equation[di], (i - di) * sizeof(*equation));
					blt(&equation[di], &scratch[0], len * sizeof(*equation));
					edi += len;
					i += len - 2;
				}
			} else if (equation[i].token.operatr == TIMES) {
				if (di >= 0 && edi == *np)
					edi = i;
			} else
				break;
		}
	}
	if (di < 0) {
		if (!dflag)
			return;
		switch (high_op) {
		case PLUS:
		case MINUS:
			break;
		default:
			return;
		}
		if (*np + 2 > N_TOKENS) {
			error_huge();
		}
		switch (op) {
		case 0:
		case TIMES:
		case DIVIDE:
			break;
		default:
			for (j = loc; j < e1; j++)
				equation[j].level++;
			break;
		}
		blt(&equation[e1+2], &equation[e1], (*np - e1) * sizeof(*equation));
		*np += 2;
		equation[e1].level = level;
		equation[e1].kind = OPERATOR;
		equation[e1].token.operatr = DIVIDE;
		e1++;
		equation[e1].level = level;
		equation[e1].kind = CONSTANT;
		equation[e1].token.constant = 1.0;
		return;
	}
	if (!grouper)
		return;
	di++;
	for (i = di; i < edi; i++) {
		if (equation[i].level == level && equation[i].kind == OPERATOR)
			equation[i].token.operatr = TIMES;
		equation[i].level++;
	}
}
