/*
 * Algebraic manipulator differentiation routines.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "am.h"
#include "externs.h"

int	special_diff_flag;

/*
 * Compute the derivative of "source", with respect to the variable "v",
 * using the fast transform method,
 * and place the result in "equation".
 * The result must be simplified by the caller.
 *
 * Return "false" if differentiation failed.
 */
int
differentiate(source, n, equation, np, v, sflag)
token_type	*source;	/* source equation */
int		n;		/* length of source equation in tokens */
token_type	*equation;	/* destination equation */
int		*np;		/* pointer to length of destination equation */
long		v;
int		sflag;		/* do special differentiation for poly_factor */
{
	int	i;
	int	rv;

	special_diff_flag = sflag;
	blt(equation, source, n * sizeof(token_type));
	*np = n;
	organize(equation, np);
	for (i = 1; i < *np; i += 2) {
		if (equation[i].token.operatr == TIMES || equation[i].token.operatr == DIVIDE) {
			binary_parenthesize(equation, *np, i);
		}
	}
	rv = d_recurse(equation, np, 0, min_level(equation, *np), v);
	return rv;
}

/*
 * Recursive differentiation routine.
 *
 * Symbolically differentiate expression in "equation"
 * (which is a standard equation side) starting at "loc".
 * The current level of parentheses is "level" and
 * do the differentation with respect to variable "v".
 *
 * Return "true" if successful, "false" if it is beyond this program's
 * capabilities or an error was encountered.
 */
int
d_recurse(equation, np, loc, level, v)
token_type	*equation;
int		*np, loc, level;
long		v;
{
	int	i, j;
	int	n;
	int	op;
	int	oploc;
	int	endloc;
	double	d;

	if (equation[loc].level < level) {
/* First differentiate if it is a single variable or constant. */
/* If it is the specified variable, change it to the constant "1.0", */
/* otherwise change it to "0.0". */
		if (equation[loc].kind == VARIABLE
		    && ((v == MATCH_ANY && (equation[loc].token.variable & VAR_MASK) > SIGN)
		    || equation[loc].token.variable == v)) {
			equation[loc].kind = CONSTANT;
			equation[loc].token.constant = 1.0;
		} else {
			equation[loc].kind = CONSTANT;
			equation[loc].token.constant = 0.0;
		}
		return true;
	}
	op = 0;
	for (endloc = loc + 1;; endloc += 2) {
		if (endloc >= *np || equation[endloc].level < level)
			break;
		if (equation[endloc].level == level) {
			switch (op) {
			case 0:
			case PLUS:
			case MINUS:
				break;
			default:
/* Oops.  More than one operator on the same level in this expression. */
				printf("Error in d_recurse().\n");
				return false;
			}
			op = equation[endloc].token.operatr;
			oploc = endloc;
		}
	}
	switch (op) {
	case 0:
	case PLUS:
	case MINUS:
		break;
	case TIMES:
		goto d_times;
	case DIVIDE:
		goto d_divide;
	case POWER:
		goto d_power;
	default:
/* Differentiate an unsupported operator. */
/* This is possible if the expression doesn't contain the specified variable. */
/* In that case, the expression is replaced with "0", otherwise return false. */
		for (i = loc; i < endloc; i += 2) {
			if (equation[i].kind == VARIABLE
			    && ((v == MATCH_ANY && (equation[i].token.variable & VAR_MASK) > SIGN)
			    || equation[i].token.variable == v)) {
				return special_diff_flag;
			}
		}
		blt(&equation[loc+1], &equation[endloc], (*np - endloc) * sizeof(token_type));
		*np -= (endloc - (loc + 1));
		equation[loc].level = level;
		equation[loc].kind = CONSTANT;
		equation[loc].token.constant = 0.0;
		return true;
	}
/* Differentiate PLUS and MINUS operators. */
/* Use rule: d(u+v) = d(u) + d(v), */
/* where "d()" is the derivative function, */
/* "u" and "v" are expressions. */
	for (i = loc;;) {
		if (i >= *np || equation[i].level < level)
			break;
		if (equation[i].kind != OPERATOR) {
			if (!d_recurse(equation, np, i, level + 1, v))
				return false;
			i++;
			for (; i < *np && equation[i].level > level; i += 2)
				;
			continue;
		}
		i++;
	}
	return true;
d_times:
/* Differentiate TIMES operator. */
/* Use rule: d(u*v) = u*d(v) + v*d(u). */
	if (*np + 1 + (endloc - loc) > N_TOKENS) {
		error_huge();
	}
	for (i = loc; i < endloc; i++)
		equation[i].level++;
	blt(&equation[endloc+1], &equation[loc], (*np - loc) * sizeof(token_type));
	*np += 1 + (endloc - loc);
	equation[endloc].level = level;
	equation[endloc].kind = OPERATOR;
	equation[endloc].token.operatr = PLUS;
	if (!d_recurse(equation, np, endloc + (oploc - loc) + 2, level + 2, v))
		return false;
	if (!d_recurse(equation, np, loc, level + 2, v))
		return false;
	return true;
d_divide:
/* Differentiate DIVIDE operator. */
/* Use rule: d(u/v) = (v*d(u) - u*d(v))/v^2. */
	if (*np + 3 + (endloc - loc) + (endloc - oploc) > N_TOKENS) {
		error_huge();
	}
	for (i = loc; i < endloc; i++)
		equation[i].level += 2;
	equation[oploc].token.operatr = TIMES;
	j = 1 + (endloc - loc);
	blt(&equation[endloc+1], &equation[loc], (*np - loc) * sizeof(token_type));
	*np += j;
	equation[endloc].level = level + 1;
	equation[endloc].kind = OPERATOR;
	equation[endloc].token.operatr = MINUS;
	j += endloc;
	blt(&equation[j+2+(endloc-oploc)], &equation[j], (*np - j) * sizeof(token_type));
	*np += 2 + (endloc - oploc);
	equation[j].level = level;
	equation[j].kind = OPERATOR;
	equation[j].token.operatr = DIVIDE;
	blt(&equation[j+1], &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
	j += endloc - oploc;
	equation[j].level = level + 1;
	equation[j].kind = OPERATOR;
	equation[j].token.operatr = POWER;
	j++;
	equation[j].level = level + 1;
	equation[j].kind = CONSTANT;
	equation[j].token.constant = 2.0;
	if (!d_recurse(equation, np, endloc + (oploc - loc) + 2, level + 3, v))
		return false;
	if (!d_recurse(equation, np, loc, level + 3, v))
		return false;
	return true;
d_power:
/* Differentiate POWER operator. */
/* Since we don't have logarithms, just do what we can without them. */
	if (!special_diff_flag) {
		for (i = oploc; i < endloc; i++) {
			if (equation[i].kind == VARIABLE
			    && ((v == MATCH_ANY && (equation[i].token.variable & VAR_MASK) > SIGN)
			    || equation[i].token.variable == v)) {
				if ((oploc - loc) == 1) {
					switch (equation[loc].kind) {
					case CONSTANT:
						d = equation[loc].token.constant;
						break;
					case VARIABLE:
						if (!var_is_const(equation[loc].token.variable, &d))
							return false;
						break;
					default:
						return false;
					}
					n = (endloc - oploc) + 2;
					if (*np + n > N_TOKENS) {
						error_huge();
					}
					blt(&equation[endloc+n], &equation[endloc], (*np - endloc) * sizeof(token_type));
					*np += n;
					n = endloc;
					equation[n].level = level;
					equation[n].kind = OPERATOR;
					equation[n].token.operatr = TIMES;
					n++;
					equation[n].level = level;
					equation[n].kind = CONSTANT;
					equation[n].token.constant = log(d);
					n++;
					equation[n].level = level;
					equation[n].kind = OPERATOR;
					equation[n].token.operatr = TIMES;
					n++;
					blt(&equation[n], &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
					for (i = loc; i < endloc; i++) {
						equation[i].level++;
					}
					if (!d_recurse(equation, np, n, level + 1, v))
						return false;
					return true;
				} else {
					return false;
				}
			}
		}
	}
	blt(&scratch[0], &equation[oploc+1], (endloc - (oploc + 1)) * sizeof(token_type));
	n = endloc - (oploc + 1);
	scratch[n].level = level;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = TIMES;
	n++;
	if (n + (endloc - loc) + 2 > N_TOKENS) {
		error_huge();
	}
	blt(&scratch[n], &equation[loc], (endloc - loc) * sizeof(token_type));
	i = n;
	n += oploc + 1 - loc;
	for (; i < n; i++)
		scratch[i].level++;
	n += endloc - (oploc + 1);
	for (; i < n; i++)
		scratch[i].level += 2;
	scratch[n].level = level + 2;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = MINUS;
	n++;
	scratch[n].level = level + 2;
	scratch[n].kind = CONSTANT;
	scratch[n].token.constant = 1.0;
	n++;
	if (n + (oploc - loc) + 1 > N_TOKENS) {
		error_huge();
	}
	scratch[n].level = level;
	scratch[n].kind = OPERATOR;
	scratch[n].token.operatr = TIMES;
	n++;
	j = n;
	blt(&scratch[n], &equation[loc], (oploc - loc) * sizeof(token_type));
	n += oploc - loc;
	if (*np - (endloc - loc) + n > N_TOKENS) {
		error_huge();
	}
	blt(&equation[loc+n], &equation[endloc], (*np - endloc) * sizeof(token_type));
	*np += loc + n - endloc;
	blt(&equation[loc], &scratch[0], n * sizeof(token_type));
	if (!d_recurse(equation, np, loc + j, level + 1, v))
		return false;
	return true;
}
