; Programs the PLL and Shift registers of a Kenwood TM-741/742 2 meter
: module to control the frequency and TX/RX control.
; This program runs on an L.S.Electronics ELC-31 8031 microcontroller.
; Clock xtal = 11.0592 MHz.
; Assembled using the PseudoSam(TM) 51 Assembler V1.6.02
;
; By Hugh Duff  VA3TO  April 1995  Brampton, Ontario, Canada
;
;***************************************************************
; Port pin definitions

        .equ    PTT,p1.0        ; Push-To-Talk input...Lo=TX, Hi=RX
        .equ    EP,p1.1         ; PLL enable output
        .equ    CK,p1.2         ; Clock output
        .equ    DT,p1.3         ; Data output
        .equ    ES,p1.4         ; Shift Register enable output

;**************************************************************
; Start of program..micro initialization
; PSW (Program Status Word) BANK 0 contains the RX data, PSW BANK 1
; contains the TX data.
; R7 to R5 contain the Reference register word
; R4 to R2 contain the Comparsion register word
; Both words are 21 bits in length. The top 3 bits of R7 and R4 are
; unused. These get purged at the "bit_5" subroutine.

INIT:   mov     SP,#0x2F        ; setup stack
        clr     PSW.3           ; set PSW to register to point at bank 0
        clr     PSW.4           ;                 "

; Load PSW BANK0 with RX step,frequency and keying data
        mov     R7,#0x06        ; REF data for 5kHz step,      1st byte
        mov     R6,#0x01        ;        "                     2nd byte
        mov     R5,#0x40        ;        "                     3rd byte
        mov     R4,#0x04        ; COMP data for 146.88 MHz RX, 1st byte
        mov     R3,#0x6A        ;        "                     2nd byte
        mov     R2,#0x64        ;        "                     3rd byte
        mov     R1,#0xB4        ; Shift Reg. data for RX

; Load PSW BANK1 with TX step,frequency and keying data
        setb    PSW.3           ; select TX bank in PSW
        mov     R7,#0x06        ; REF data for 5 kHz step,     1st byte
        mov     R6,#0x01        ;        "                     2nd byte
        mov     R5,#0x40        ;        "                     3rd byte
        mov     R4,#0x08        ; COMP data for 146.28 MHz TX, 1st byte
        mov     R3,#0x72        ;        "                     2nd byte
        mov     R2,#0x48        ;        "                     3rd byte
        mov     R1,#0xB5        ; Shift Reg. data for TX

; Set module control lines to appropriate resting levels
        setb    EP              ; PLL enable line normally HI
        setb    CK              ; PLL clock line normally HI
        clr     DT              ; PLL data line normally LO
        setb    ES              ; Shift Register enable normally HI


;**************************************************************
; Main routine loads receive data,waits for a PTT then loads
; transmit data and waits for PTT to drop...

MAIN:   clr     PSW.3           ; select RX bank in PSW
        acall   LD_SHFT         ; load RX shift register data
        acall   LD_REF          ; load RX Reference data into PLL
        acall   LD_COMP         ; load RX Comparison data into PLL
RX_IDLE:jb      PTT,*           ; wait for PTT to ground
        acall   TX              ; go load TX data and return
TX_IDLE:jnb     PTT,*           ; wait for PTT release
        acall   RX              ; go load RX data and return
        ajmp    RX_IDLE         ; repeat this loop

;**************************************************************
; Selects receive databank and calls load routines

RX:     clr     PSW.3           ; select receive data bank
        acall   LD_COMP         ; load com.freq. data to PLL
        acall   LD_SHFT         ; disable TX in shift reg.
        ret                     ; return

;**************************************************************
; Selects transmit databank and calls load routines

TX:     setb    PSW.3           ; select transmit data bank
        acall   LD_COMP         ; load comp.freq. data to PLL
        acall   LD_SHFT         ; enable TX in shift reg.
        ret                     ; return

;**************************************************************
; Loads PLL Reference register

LD_REF: clr     EP              ; enable PLL
        acall   DELAY           ; wait
        mov     A,R7            ; get 1st PLL REF data byte
        acall   bit_5           ; send it out
        mov     A,R6            ; get 2nd PLL REF data byte
        acall   bit_8           ; send it out
        mov     A,R5            ; get 3rd PLL REF data byte
        acall   bit_8           ; send it out
        acall   DELAY           ; wait
        setb    EP              ; disable PLL
        ret                     ;

;**************************************************************
; Loads PLL Comparison register

LD_COMP:clr     EP              ; enable PLL
        acall   DELAY           ; wait
        mov     A,R4            ; get 1st PLL COMP data byte
        acall   bit_5           ; send it out
        mov     A,R3            ; get 2nd PLL COMP data byte
        acall   bit_8           ; send it out
        mov     A,R2            ; get 3rd PLL COMP data byte
        acall   bit_8           ; send it out
        acall   DELAY           ; wait
        setb    EP              ; disable PLL
        ret                     ;

;**************************************************************
; Loads Shift Register

LD_SHFT:mov     A,R1            ; get data
        acall   bit_8           ; send it out
        clr     ES              ; strobe data into shift register
        acall   DELAY           ; wait
        setb    ES              ; return ES to rest level
        ret                     ;

;**************************************************************
; Rotates 8 bits of data in reg.A out to DATA line

BIT_8:  mov     B,#0X08         ; setup reg.B as an 8 bit counter
LOOP8:  rlc     A               ; rotate left 1 bit thru carry
        mov     DT,C            ; move carry status to DATA port line
CLK8:   acall   CLOCK           ; clock the data
        djnz    B,LOOP8         ; keep looping until 8 bits sent
        ret                     ;

;**************************************************************
; Rotates 5 bits of data in reg.A out to DATA line
; The top 3 bits of R7 and R4 are not sent since the registers
; are only 21 bits in length.

BIT_5:  mov     B,#0X05         ; setup reg.B as a 5 bit counter
        rlc     A               ; waste unused MSB
        rlc     A               ;         "
        rlc     A               ;         "
LOOP5:  rlc     A               ; rotate left 1 bit thru carry
        mov     DT,C            ; move carry status to DATA port line
CLK5:   acall   CLOCK           ; clock the data
        djnz    B,LOOP5         ; keep looping until 5 bits sent
        ret                     ;

;**************************************************************
; Toggles the clock line

CLOCK:  clr     CK              ; Clock line LO
        acall   DELAY           ; wait
        setb    CK              ; Clock line HI
        acall   DELAY           ; wait
        ret                     ;

;**************************************************************
; Delay 200 us based on 11.0592 MHz xtal

DELAY:  mov     R0,#0X5C        ; load count
DLY1:   djnz    R0,DLY1         ; kill time
        ret

        .end



