//
//    Copyright 2004, Thomas C. McDermott, N5EG
//    This file is part of VNAR - the Vector Network Analyzer program.
//
//    VNAR is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    VNAR is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with VNAR, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

#pragma once
//
//
//	Routines to manipulate display points in rectangular
//  and polar format for the Vector Network Analyzer
//
//	7-4-03  TCMcDermott
//  7-10-03 Add Frequency_Grid Class.
//  7-11-03 Add CalDataSet.
//  7-12-03 Add routines to compute S-parameter error terms
//			  and convert measured S11 to actual S11.
//  7-13-03 Add routines to save CalDat to file, and load
//			  CalData from file.
//  7-27-03 Add routines to compute rectangular display
//			  coordinates.
//  8-17-03 Add Detector Class to support separate detector
//			  calibration.
//  9-16-03 Phase correction algorithm implemented for
//			  detector calibration.  ResolveTranPolar and
//			  ResolveReflPolar changed to use detector cal values.
//  3-28-04 Add TxLevel adjust to IDAC on the VNA board
//  4-15-04 Revise amplitude interpretation algorithm

#include "stdafx.h"
#using <mscorlib.dll>
#using <System.dll>
using namespace System;
using namespace System::ComponentModel;
using namespace System::IO;
using namespace System::Windows::Forms;

#include <math.h>
#include <complex>
#include "DataDisplay.h"


#define	VNA_XTAL_CONST	0xEE9BF				// translate VNA Xtal Osc to DDS divisor
	// the VNA crystal is ~12.000 MHz. It gets multiplied by 24 going to the DDS accumulator 
	// The clock rate is thus approximately 24 * ~12.000 MHz. = ~288.000 MHz.
	// The synthesizer accumulator value needed is:   N = Fdesired * 2^48 / Fclk
	// The constant 2^48/Fclk is 0xEE9BF, but can vary slightly for each oscillator module.


#define RAD2DEGR 180.0/3.14159265358	// Converts radians to degrees

// Unused (old) constants before detector cal routines added...
//#define MAXREADP 3800					// Full-scale phase reading observed on S/N 1
//#define MINREADP   90					// Minimum-scale phase reading observed on S/N 1
//const int MIDREADP = MAXREADP/2;		// Mid-scale phase reading
//#define MODMIDREADP (MAXREADP+MINREADP)/2  // modified mid-scale phase reading

//const int UPLIMITP = MAXREADP - MAXREADP/4;	// Phase readings upper quartile
//const int DNLIMITP = MAXREADP/4	;		// Phase readings lower quartile



//	Constants related to the actual numeric VNA instrument readings
#define MAXREFLMAGNITUDE 2100			// temporary until CALIBRATION is designed,
										// observed on S/N 1
#define MAXTRANMAGNITUDE 3800			// temporary until CALIBRATION is designed,
										// observed on S/N 1

#define MAXCALFREQ 120000000.0			// Calibration parameters
#define MINCALFREQ 200000.0
#define NUMCALPTS 1024.0


	// resolve measured data sets to Magnitude and Phase

	// Draw reflection measurements on Polar Chart
	// Requires resolving the phase quandrant from I/Q measurements
	//   and magnitude from the reflection or tranmsission magnitude
	//	 (which is a logrithmic value).
	//
	// when ReflPI is low,  the phase is near   0 degrees
	// when ReflPI is high, the phase is near 180 degrees
	// when ReflPQ is low,  the phase is near  90 degrees
	// when ReflPQ is high, the phase is near -90 degrees
	//
	// The I and Q phase readings saturates near the bottom and top of the reading,
	//   but the other reading is then in the middle of it's useful range.
	//   This it's always possible to accurately resolve the phase and quadrant.
	//
	// Quadrant 1 is 0 to +90 degrees, quadrant 2 is +90 to +180,
	//   quadrant 3 is -180 to -90, quadrant 4 is 0 to -90 degrees.
	//

	// Convert Mag and Phase to X,Y screen display coordinates in polar display mode
void ToDisplayPolar(double magnitude, double phase, int polarRad, int xoffset, int yoffset, int& X, int& Y)
{

	double fx, fy, phase_radians;
	int	x, y;

	phase_radians = phase / 180.0 * 3.1415926;

	fx = magnitude * cos(phase_radians);
	fy = magnitude * sin(phase_radians);

	// Scale to the size of the chart (polar radius)

	fx = fx * static_cast<double>(polarRad);
	fy = fy * static_cast<double>(polarRad);

	// Convert to integer

	x = (int)fx;
	y = (int)fy;

	// Center on the chart

	x = x + polarRad + xoffset;
	y = polarRad + yoffset - y;

	X = x;		// return X and Y display points
	Y = y;

}



	// Convert Magnitude to rectangular (X, Y) display coordinates
int ToDisplayRectMag(double magnitude, int height, int dbScaleFactor, int refLevel)
{
	// magnitude has a normalized value.
	// Zero dB is the top of the screen - magnitude value = 1.000
	// -100 dB is the bottom of the screen (@ 10 db/div) - magnitude value 1e-5
	// Convert magnitude to vertical display accounting for the scale factor in dB/division

	double dbmag = refLevel -20.0 * log10(magnitude);	// 0 to 100 for zero dB to -100 dB.
	dbmag /= ((double)dbScaleFactor * 10.0);			// 0 to +1 for 10db/div

	// scale to the vertical screen display area
	int vertical = (int)(dbmag * (double)height);	// 0 to scopeDisp.height for zero dB to height for max dB

	if (vertical > height)					// crop display - probably a better way to do this
		vertical = height;
	if (vertical < 0)
		vertical = 0;

	return(vertical);						// return Y as positive number

}
	// Convert Phase to rectangular (X, Y) display coordinates
int ToDisplayRectPhs(double phase, int height)
{
	// phase in degrees ranges from -180 to +180

	phase += 180.0;								// offset the phase display so zero is centerline
	double range = phase * (double)height / 360.0; // top to bottom of display is 360 degrees

	return((int)range);
}
int ToDisplayRectGD(double groupdelay, int height, int scaleFactor)
{

	// scale factor = 1  implies 100 picoseconds
	// convert to nanoseconds
	groupdelay *= 1000000000.0;

	// set full screen display to 10 units of the selected scale
	// 100 picoseconds * 10 = 1 nanosecond full-scale at max resolution
	groupdelay /= scaleFactor;

	// scale it to display region
	groupdelay *= (double)height;

	return((int)groupdelay);
}

__gc public class FrequencyGrid
{
private:
	int FrequencyIndex __gc[];
//	bool Valid;
	int startFreq, stopFreq;
	int indexer;
	int points;
	float delta;

public:

	FrequencyGrid(int numPoints)
		{
			if (numPoints > 1024)
				throw new System::IndexOutOfRangeException;

			FrequencyIndex = new int __gc[numPoints];
			startFreq = 200000;
			stopFreq = 120000000;
			points = numPoints;
			Build();
//			Valid = true;
		};
	void SetStartF(int start)
	{
		startFreq = start;
		Build();
		//Valid = false;		// Frequency Grid is now invalid, needs to be recomputed
	}
	void SetStopF(int stop)
	{
		stopFreq = stop;
		Build();
		//Valid = false;		// Frequency Grid is now invalid, needs to be recomputed
	}

	int Frequency(int gridpoint)
	{
//		if (!Valid)
//			Build();		//rebuild the grid if something has changed
		return(FrequencyIndex[gridpoint]);
	}

	long long int DDS(int Frequency)	// Derive DDS divisor value (as 64-bit integer) from Frequency
	{
	//	if (!Valid)
	//		Build();		//rebuild the grid if something has changed

	// Calculate the 48-bit synthesizer accumulator value from Frequency
	// the VNA crystal is ~12.000 MHz. It gets multiplied by 24 going to the DDS accumulator 
	// The clock rate is thus approximately 24 * ~12.000 MHz. = ~288.000 MHz.
	// The synthesizer accumulator value needed is:   N = Fdesired * 2^48 / Fclk
	// The constant 2^48/Fclk is 0xEE9BF - it is defined as VNA_XTAL_CONST

		long long int N, Fdesired;
		Fdesired = Convert::ToInt64(Frequency);
		N = Fdesired * VNA_XTAL_CONST;
		return(N);
	}

	int GridPoint(int Frequency)	// find nearest gridpoint to Frequency
	{
	//	if (!Valid)
	//		Build();
	
		if (((float)Frequency > stopFreq) || ((float)Frequency < startFreq))
			return(-1);				// Frequency is outside the grid range

	    float result = ((float)Frequency - startFreq) / delta;
		int iresult = (int)(result + 0.5);

		return(iresult);
	}

	__property int get_Points() { return points; }
	__property int get_StartF() { return startFreq; }
	__property int get_StopF() { return stopFreq; }
private:
	void Build(void)
	{
		delta = ((float)stopFreq-(float)startFreq)/points;

		// build the frequency grid
		for (int i = 0; i<points; i++)
			FrequencyIndex[i] = startFreq + (i * (int)delta);

//		Valid = true;		// FrequencyGrid is now valid
	}

};

__gc public class Detector	// Analog Devices AD8302 device-dependent constants

{
public:
	int lowerPhase, upperPhase;			// lower & upper I/Q phase crossover points
	int centerPhase;					// phase midpoint
	int maxMagnitude;					// maximum magnitude reading
	float unitsPerdB;					// magnitude ADC counts per dB. (~57.0)
	int cxFreqIdx;						// frequency index of 360 degrees
	int centerQindex, centerQ2index;
	int centerIindex, centerI2index, centerI3index;
	float quadratureError;				// I-Q phase quadrature error
	float phaseSlope;
	double pherror __gc[];				// phase error for each degree
	int MagTable __gc[ , ];				// magnitude raw data table
	bool calibrated;					// true if values have been calibrated

	Detector(void)							// Constructor, allocate default values
	{
		unitsPerdB = 57.0;				// default for uncalibrated data

		centerPhase = 1815;				// starting guess at phase center
		lowerPhase = 900;				// starting guess at lower and upper phase crossovers
		upperPhase = 2750;	
		centerQindex = 0, centerQ2index = 0;
		centerIindex = 0, centerI2index = 0, centerI3index = 0;
		quadratureError = 0.0;
		phaseSlope = 0;
		pherror = new double __gc[360];	// hold phaserror array
		MagTable = new int __gc[21, 4];	// hold raw Mag count data
		calibrated = false;				// values are not yet calibrated
	}					
	void PhaseCal(int Iphase __gc[], int Qphase __gc[])	// construct phase points from raw data
	{

		// Find 3 consecutive center-crossing values for I.
		// This determines the value for 360 degrees of phase, and the centerPhase value.
		// Then offset 90 degrees, and see how far off centerPhase the Q value is.
		// Use this as the value of Quadrature Error.
		// CenterPhase takes several iterations to zero in on.

		for (int k=0; k<5; k++)
		{
			// find 1st frequency index where I(index) crosses centerPhase
			for (int i=50; i<900; i++)
			{
				if (Iphase[i] >= centerPhase)
				{
					centerIindex = i;
					break;
				}
			}
			// find 2nd frequency index where I(index) crosses centerPhase
			for (int i=centerIindex+5; i<900; i++)
			{
				if (Iphase[i] <= centerPhase)
				{
					centerI2index = i;
					break;
				}
			}
			
			// find 3rd frequency index where I(index) crosses centerPhase
			for (int i=centerI2index+5; i<1024; i++)
			{
				if (Iphase[i] >= centerPhase)
				{
					centerI3index = i;
					break;
				}
			}

			// Compute level error based on I-I2 and I2-I3 mismatch.
			// Converge on value for centerPhase that equalizes these two distances.

			int error = (centerI3index - centerI2index - (centerI2index - centerIindex)) / 4;
			centerPhase = Iphase[centerI2index+error];
		}

		cxFreqIdx = centerI3index - centerIindex;		// 360 degrees of phase difference as index count
		upperPhase = Iphase[centerI2index-cxFreqIdx/8];	// revise uppercrossover = centerpoint - 45 degrees
		lowerPhase = Iphase[centerI2index+cxFreqIdx/8];	// revise lowercorssover = centerpoint + 45 degrees
		phaseSlope = (float)90.0 / (float)(upperPhase - lowerPhase);	// compute degrees per unit count

		int qerrampl = Qphase[centerIindex + cxFreqIdx/4] - centerPhase;  // Q error as amplitude
		quadratureError = -(float)qerrampl * phaseSlope;						// Q error as degrees

		// Now determine error of measured phase derived from the atan2() function.
		// It's not completely periodic, so generate the error every closest degree.
		// pherror[degrees] = difference.

		for (int k= centerIindex; k<centerI3index; k++)		// sweep over 360 degrees - from +90 to 0 to -180..+180 to -90.
		{
			// derive phase using atan approximation
			double phase = RAD2DEGR * atan2((double)(centerPhase - Qphase[k]),(double)(Iphase[k] - centerPhase));
			// derive phase using cable linearity approximation
			double cphase = 90.0 - 360.0 * (double)(k - centerIindex) / (double)cxFreqIdx;
			if (cphase < -180.0)
				cphase += 360.0;
			double error = cphase - phase;
			if (error < -180.0)
				error += 360.0;
			if (error > 180.0)
				error -= 360.0;
			pherror[(int)cphase+180] = error;	// offset index -->  [-180 .. +180] is the index range
		}

		calibrated = true;
	}

	void AmpCal(int Mag __gc[ , ])		// construct amplitude calibration table from raw data
	{
		//  store the data into our object's local table

		for (int i=0; i<21; i++)
			for (int level = 0; level<4; level++)
				MagTable[ i, level] = Mag[ i, level];
	}	
	double IQtoDegrees(int I, int Q)		// convert I/Q ADC reading to degrees
	{
		double degrees;

		degrees = RAD2DEGR * atan2((double)(centerPhase - Q),(double)(I - centerPhase));
		if (degrees > 180.0)
			degrees = degrees - 360.0;
		if (calibrated)
			degrees += pherror[(int)degrees+180];	// correct for phase error

		// pherror could change degress to be outside the range -180 to +180 - wrap the phase

		if (degrees > 180.0)
			degrees -= 360.0;
		if (degrees < -180.0)
			degrees += 360.0;

		return (degrees);

		//// find whether I or Q is closer to centerline
		//if (abs(I - centerPhase) < abs(Q - centerPhase))	// I is closer
		//{
		//	if (Q < centerPhase)
		//	    degrees = 90.0 - ((float)(I - centerPhase) / (float)(upperPhase - lowerPhase)) * 90.0;
		//	else
		//		degrees = -90.0 + ((float)(I - centerPhase) / (float)(upperPhase - lowerPhase)) * 90.0;
		//}
		//else		// Q is closer
		//{
		//	if (I < centerPhase)
		//		degrees = -quadratureError + 180.0 - ((float)(centerPhase - Q) / (float)(upperPhase - lowerPhase)) * 90.0;
		//	else
		//	    degrees = -quadratureError + ((float)(centerPhase - Q) / (float)(upperPhase - lowerPhase)) * 90.0;
		//}

	}
	double MagTodB(int Freq, int Mag)
	{
		double result;
		double countPerdB;				// Detector ADC counts per dB of magnitude
		int FreqBase, FreqBaseNext;		// Two adjacent cal frequency points
		double FreqRemainder;			// remainder of freq between cal points
		double MagInterp[4];			// 0, -1, -2, -20 interpolated cal data


		if (calibrated)
		{
			// Piecewise linear approximation. Three slope ranges: 0 to -1 dbm, -1 to -2 dbm, below -2 dbm.
			// Result is frequnecy-dependent. Interpolate the frequency between two datasets to construct
			// a linear-frequency approximation of the cal data for the desired frequency.
			// Every 200 KHz from 200K to 1MHz, then every 10 MHz up to 120 MHz.
			// Index:  0    1    2    3    4    5    6    7    8   9   10   11   12   13   14   15   16   17    18    19 
			// Freq:  200k 300k 400k 500k 600k 700k 800k 900k 1M  10M  20M  30M  40M  50M  60M  70M  80M  90M  100M  110M 


			// Build interpolated dataset between two frequencies

			if (Freq <= 1000000)	// 1 MHZ and below
			{
				FreqBase = (Freq - 200000)/100000;
				FreqRemainder = (double)(Freq - (FreqBase + 2) * 100000)/100000;  // Remainder of frequency, value between  0..1
			}
			else if (Freq < 10000000)	// Between 1 and 10 MHz
			{
				FreqBase = 8;
				FreqRemainder = (double)(Freq - 1000000)/9000000;
			}
			else		// 10 MHz to 120 MHz
			{
				FreqBase = (Freq / 10000000) + 8;						// Base is the integer part of the frequency index
				FreqRemainder = (double)(Freq - (FreqBase - 8) * 10000000)/10000000;  // Remainder of frequency, value between  0..1
			}


			if (FreqRemainder < 0.001)
				FreqBaseNext = FreqBase;					// Prevent index overflow
			else
				FreqBaseNext = FreqBase + 1;

			for (int i=0; i<4; i++)							// build interpolated amplitude table
                MagInterp[i] = MagTable[FreqBase, i] - (MagTable[FreqBase, i] - MagTable[FreqBaseNext, i]) * FreqRemainder;

			if (Mag > MagInterp[1])					// greater than -1 dbm
			{
				countPerdB = (MagInterp[0] - MagInterp[1]) / 1.0; // 1 db between 0 and -1 cal values
				result = (Mag - MagInterp[1]) / countPerdB;
				result = result - 1.0;
				return(result);
			}
			if (Mag > MagInterp[2])				// greater than -2 dbm
			{
				countPerdB = (MagInterp[1] - MagInterp[2]) / 1.0; // 1 dB between -1 and -2 cal values
				result = (Mag - MagInterp[2]) / countPerdB;
				result = result - 2.0;
				return(result);
			}
														// less than -2 dbm
			countPerdB = (MagInterp[2] - MagInterp[3])/ 18.0; // 18 dB between the -2 and -20 cal values
			result = - (MagInterp[2] - Mag) / countPerdB;
			result = result - 2.0;
			return(result);
		}
		else									// uncalibrated measurement
			return(((Mag-MAXTRANMAGNITUDE)/57.0));			// roughly 57 counts per dB
	}
};


__gc public class CalDataSet
{
public:
	// Holds raw and computed calibration data	

	Detector* RxDet, * TxDet;		// Holds detector calibration constants

	double EdReal __gc[], EdImag __gc[], EsReal __gc[], EsImag __gc[];
	double EtReal __gc[], EtImag __gc[], ThReal __gc[], ThImag __gc[];
	double S11shortReal __gc[], S11shortImag __gc[];
	double S11openReal __gc[], S11openImag __gc[];
	double S11termReal __gc[], S11termImag __gc[];

CalDataSet(String* StartUpDir)	// Constructor - allocate memory for Calibration Data
{
	RxDet = new Detector();		// construct AD8702 objects
	TxDet = new Detector();

	EdReal = new double __gc[1024]; EdImag = new double __gc[1024];
	EsReal = new double __gc[1024]; EsImag = new double __gc[1024];
	EtReal = new double __gc[1024]; EtImag = new double __gc[1024];
	ThReal = new double __gc[1024]; ThImag = new double __gc[1024];
	S11shortReal = new double __gc[1024]; S11shortImag = new double __gc[1024];
	S11openReal = new double __gc[1024]; S11openImag = new double __gc[1024];
	S11termReal = new double __gc[1024], S11termImag = new double __gc[1024];

	// Try to read in the detector.ica file to pre-load the Detector calibration constants
	// If the file does not exist, warn user to run detector cal first

			FileStream* fs;
			BinaryReader* br;

			String * filename = String::Concat(StartUpDir, S"\\detector.ica");
			try
			{
				// Create a filestream & binary reader

				fs = new FileStream(filename, FileMode::Open, FileAccess::Read);
				br = new BinaryReader(fs);
			}
			catch(System::Exception* pe)
			{
				MessageBox::Show(S"Detector Calibration File Not Found.\n\r"
					S"Be sure to run   Calibration->Detector Calibration...   first", pe->Message,
					MessageBoxButtons::OK, MessageBoxIcon::Exclamation);
				return;
			}

			// Define header to match file identifying type and version

			String* recognized = new String(S"VNA Detector Calibration Data Set Version 1.0.0");
			String* header;

			header = br->ReadString();		// get string header on infile
			if (String::Compare(recognized, header) != 0)
			{
				MessageBox::Show(
					S"detector.ica file is incompatible type or version.\n\rExpecting Detector Version 1.0.0",
					S"Error", MessageBoxButtons::OK, MessageBoxIcon::Error);
				return;
			}

		
			// read in the AD8302 Phase Detector constants and error table

			try
			{
				RxDet->centerPhase = br->ReadInt32();
				RxDet->lowerPhase = br->ReadInt32();
				RxDet->upperPhase = br-> ReadInt32();
				for (int degree = 0; degree<360; degree++)
					RxDet->pherror[degree] = br->ReadDouble();

				TxDet->centerPhase = br->ReadInt32();
				TxDet->lowerPhase = br->ReadInt32();
				TxDet->upperPhase = br->ReadInt32();
				for (int degree = 0; degree<360; degree++)
					TxDet->pherror[degree] = br->ReadDouble();

				// Read in the AD8302 Amplitude Detector raw cal table

				for (int FreqIdx=0; FreqIdx<21; FreqIdx++)
					for (int level = 0; level<4; level++)
						RxDet->MagTable[FreqIdx, level] = br->ReadInt32();

				for (int FreqIdx=0; FreqIdx<21; FreqIdx++)
					for (int level = 0; level<4; level++)
						TxDet->MagTable[FreqIdx, level] = br->ReadInt32();
			}
			catch(System::Exception* pe)
			{
				MessageBox::Show(S"Error trying to load Detector Calibration File.\n\r"
					S"Delete the file:   detector.ica   and re-run Detector Calibration", pe->Message,
					MessageBoxButtons::OK, MessageBoxIcon::Error);
				return;
			}


			fs->Close();

			RxDet->calibrated = true;		// loaded a valid detector calibration
			TxDet->calibrated = true;
};

void ResolveReflPolar(int ReflPI, int ReflPQ, int ReflMQ, int Frequency, double& rmag, double& rphs)
{
	double magnitude, phase;

	phase = RxDet->IQtoDegrees(ReflPI, ReflPQ);
	magnitude = RxDet->MagTodB(Frequency, ReflMQ);

	magnitude = pow(10.0, (magnitude/20.0));		// translate db_return_loss to voltage ratio
	
	rmag = magnitude;		// return linear magnitude (0..1)
	rphs = phase;			// return phase in degrees (-180..+180)
}

void ResolveTranPolar(int TranPI, int TranPQ, int TranMQ, int Frequency, double& rmag, double& rphs)
{
	double magnitude, phase;

	phase = TxDet->IQtoDegrees(TranPI, TranPQ);
	magnitude = TxDet->MagTodB(Frequency, TranMQ);

	magnitude = pow(10.0, (magnitude/20.0));		// translate db to voltage ratio

	rmag = magnitude;		// return linear magnitude (0..1)
	rphs = phase;			// return phase in degrees (-180..+180)
}

};

void CalToErrorTerms(CalDataSet* Cal)
{
	// Construct Cal error terms at each frequency point
    for (int i=0; i<1024; i++)
	{
		std::complex<double> two(2.0, 0.0);
		std::complex<double> rslt(0.0, 0.0);
		
		double realpart, imagpart;

		realpart = Cal->S11shortReal[i];
		imagpart = Cal->S11shortImag[i];
        std::complex<double>	Sshort(realpart, imagpart);

		realpart = Cal->S11openReal[i];
		imagpart = Cal->S11openImag[i];
		std::complex<double> Sopen(realpart, imagpart);

		realpart = Cal->S11termReal[i];
		imagpart = Cal->S11termImag[i];
		std::complex<double> Sterm(realpart, imagpart);

		// Directivity error term
		Cal->EdReal[i] = Cal->S11termReal[i];
		Cal->EdImag[i] = Cal->S11termImag[i];

		// Source mismatch error term
		rslt = ((two * Sterm) - (Sshort + Sopen)) / (Sshort - Sopen);
		Cal->EsReal[i] = real(rslt);
		Cal->EsImag[i] = imag(rslt);

		// Tracking error term
		rslt = ((two * (Sopen - Sterm) * (Sshort - Sterm)) / (Sshort - Sopen));
		Cal->EtReal[i] = real(rslt);
		Cal->EtImag[i] = imag(rslt);
	}

};


void CorrectS11(CalDataSet* Cal, int Frequency, double measmag, double measphs, double& rsltmag, double& rsltphs)
{
	// There are 1024 calibration frequencies, find the one nearest to the measured frequency.

	double delta = (MAXCALFREQ - MINCALFREQ)/ (NUMCALPTS - 1);		// frequency separation of cal points
	int i = (int)((((float)Frequency - MINCALFREQ) + delta/2.0) / delta); // nearest one


	// transform measured S11 into actual S11

	double realpart, imagpart;
	std::complex<double> rslt;

	realpart = Cal->EdReal[i];
	imagpart = Cal->EdImag[i];
    std::complex<double>	Ed(realpart, imagpart);

	realpart = Cal->EsReal[i];
	imagpart = Cal->EsImag[i];
    std::complex<double>	Es(realpart, imagpart);

	realpart = Cal->EtReal[i];
	imagpart = Cal->EtImag[i];
    std::complex<double>	Et(realpart, imagpart);

	// Convert measured data to rectangular coordinates
    double measreal, measimag;
	double phase_radians = measphs / 180 * 3.1415926;
	measreal = measmag * cos(phase_radians);
	measimag = measmag * sin(phase_radians);
    std::complex<double>	S11meas(measreal, measimag);

	rslt = ((S11meas - Ed) / (Es * (S11meas - Ed) + Et));

	// Convert results to polar coordinates
	double x = real(rslt);
	double y = imag(rslt);

	double fphase = 180/3.1415926 * atan2(y, x);
	double fmagnitude = sqrt(x*x + y*y);

	rsltmag = fmagnitude;
	rsltphs = fphase;
}

void CorrectS21(CalDataSet* Cal, int Frequency, double measmag, double measphs, double& rsltmag, double& rsltphs)
{
	// There are 1024 calibration frequencies, find the one nearest to the measured frequency.

	double delta = (MAXCALFREQ - MINCALFREQ)/ (NUMCALPTS - 1);		// frequency separation of cal points
	int i = (int)((((float)Frequency - MINCALFREQ) + delta/2.0) / delta); // nearest one


	// transform measured S21 into actual S21

	double realpart, imagpart;
	std::complex<double> rslt;

	realpart = Cal->ThReal[i];
	imagpart = Cal->ThImag[i];
    std::complex<double>	Th(realpart, imagpart);

	// Convert measured data to rectangular coordinates
    double measreal, measimag;
	double phase_radians = measphs / 180 * 3.1415926;
	measreal = measmag * cos(phase_radians);
	measimag = measmag * sin(phase_radians);
    std::complex<double>	S21meas(measreal, measimag);

	rslt = S21meas / Th;

	// Convert results to polar coordinates
	double x = real(rslt);
	double y = imag(rslt);

	double fphase = 180/3.1415926 * atan2(y, x);
	double fmagnitude = sqrt(x*x + y*y);

	rsltmag = fmagnitude;
	rsltphs = fphase;
}
void LoadCalDataSet(OpenFileDialog* infile, CalDataSet* Cal)
{
	FileStream* fs;
	BinaryReader* br;

	try
	{
		// Create a filestream & binary reader

		fs = new FileStream(infile->FileName, FileMode::Open, FileAccess::Read);
		br = new BinaryReader(fs);
    }
	catch(System::Exception* pe)
	{
		MessageBox::Show(S"Error", pe->Message);
		return;
	}

	// Define header to match file identifying type and version

	String* recognized = new String(S"VNA Calibration Data Set Version 1.0.0");
	String* header;

	header = br->ReadString();		// get string header on infile
	if (String::Compare(recognized, header) != 0)
	{
		MessageBox::Show(
			S"CalDataSet file is incompatible type or version.\n\rExpecting Version 1.0.0",
			S"Error", MessageBoxButtons::OK, MessageBoxIcon::Error);
		return;
	}

	try
	{
		for (int i=0; i<1024; i++)		// read in the data set
		{
			Cal->EdReal[i] = br->ReadDouble();
			Cal->EdImag[i] = br->ReadDouble();
			Cal->EsReal[i] = br->ReadDouble();
			Cal->EsImag[i] = br->ReadDouble();
			Cal->EtReal[i] = br->ReadDouble();
			Cal->EtImag[i] = br->ReadDouble();
			Cal->ThReal[i] = br->ReadDouble();
			Cal->ThImag[i] = br->ReadDouble();
			// debug - load in the generating dataset
			Cal->S11openReal[i] = br->ReadDouble();
			Cal->S11openImag[i] = br->ReadDouble();
			Cal->S11shortReal[i] = br->ReadDouble();
			Cal->S11shortImag[i] = br->ReadDouble();
			Cal->S11termReal[i] = br->ReadDouble();
			Cal->S11termImag[i] = br->ReadDouble();
		}
	}
	catch(System::Exception* pe)
	{
		MessageBox::Show(
			S"Error reading Cal Data Set. Try re-creating the Dataset", pe->Message,
			MessageBoxButtons::OK, MessageBoxIcon::Error);
	}
	fs->Close();
}
void SaveCalDataSet(SaveFileDialog* outfile, CalDataSet* Cal)
{
	FileStream* fs;
	BinaryWriter* bw;

	try
	{
		// Create a filestream & binary writer

		fs = new FileStream(outfile->FileName, FileMode::Create, FileAccess::Write);
		bw = new BinaryWriter(fs);
    }
	catch(System::Exception* pe)
	{
		MessageBox::Show(S"Error", pe->Message);
		return;
	}

	// Define header to match file identifying type and version

	String* recognized = new String(S"VNA Calibration Data Set Version 1.0.0");

	bw->Write(recognized);		// put string header on outfile

	for (int i=0; i<1024; i++)		// write out the data set
	{
		bw->Write(Cal->EdReal[i]);
		bw->Write(Cal->EdImag[i]);
		bw->Write(Cal->EsReal[i]);
		bw->Write(Cal->EsImag[i]);
		bw->Write(Cal->EtReal[i]);
		bw->Write(Cal->EtImag[i]);
		bw->Write(Cal->ThReal[i]);
		bw->Write(Cal->ThImag[i]);
		// debug - also write out the measurements
		bw->Write(Cal->S11openReal[i]);
		bw->Write(Cal->S11openImag[i]);
		bw->Write(Cal->S11shortReal[i]);
		bw->Write(Cal->S11shortImag[i]);
		bw->Write(Cal->S11termReal[i]);
		bw->Write(Cal->S11termImag[i]);
	}

	fs->Flush();
	fs->Close();
}




// Convert Log Tx Level to linear value needed by DAC register
unsigned short TxLevLinear(int tranLevel)
{
	// 0 dbm is the max level. Corresponds to 4095 decimal for IDAC control register.
	// 0 dbm -> 4095     -20 dbm -> 410    -40 dbm -> 41

	unsigned short linlevel = (unsigned short)(4095.0 * pow(10.0,(((double)tranLevel)/20.0)) + 0.5);
	return (linlevel);
};

void public ExportSParams(int mode,  FrequencyGrid * FG, 
				double S11Mag __gc[],  double S11Phs __gc[],
				double S21Mag __gc[],  double S21Phs __gc[],
				double S22Mag __gc[],  double S22Phs __gc[],
				double S12Mag __gc[],  double S12Phs __gc[])
{
	// Export S-parameters to file. Uses Mag/Phase Forward and Reverse held in internal
	// storage variables.
	// mode: 0 -> rectangular mode, 1 -> polar mode

	double FwdTranMagdB, FwdTranPhsDeg, FwdReflMagdB, FwdReflPhsDeg;
	double RevTranMagdB, RevTranPhsDeg, RevReflMagdB, RevReflPhsDeg;
	double FwdTranRe, FwdTranIm, FwdReflRe, FwdReflIm;
	double RevTranRe, RevTranIm, RevReflRe, RevReflIm;

	FileStream* fs;
	StreamWriter* sw;

	SaveFileDialog* outfile = new SaveFileDialog();
	outfile->Filter = "text files (*.txt)|*.txt";
	outfile->AddExtension = true;
	if (outfile->ShowDialog() != DialogResult::OK)
		return;

	try
	{
		// Create a filestream & stream writer

		fs = new FileStream(outfile->FileName, FileMode::Create, FileAccess::Write);
		sw = new StreamWriter(fs);
    }
	catch(System::Exception* pe)
	{
		MessageBox::Show(S"Error", pe->Message);
		return;
	}

	// Write Header for S-parameter file

	sw->WriteLine(S"! Vector Network Analyzer VNA R2");		// put string header on outfile
	sw->WriteLine(S"! Tucson Amateur Packet Radio");		// put string header on outfile
	DateTime today = DateTime::Now;
	sw->Write(S"! ");
	sw->Write(today.ToLongDateString());
	sw->Write(S"   ");
	sw->WriteLine(today.ToLongTimeString());

	// Header line defines type of file
	if (mode)
	{
		sw->WriteLine(S"! Frequency         S11                 S21                  S12                  S22");
		sw->WriteLine(S"! HZ  S  DB R 50");	// polar is in DB amplitude and degrees WRT 50 ohms
	}
	else
	{
		sw->WriteLine(S"! Frequency              S11                           S21                            S12                            S22");
		sw->WriteLine(S"# HZ  S  RI R 50");	// rectangular is in Real and Imaginary WRT 50 ohms
	}

	// write out the parameters

	if (mode)											// polar format
        for(int i=0; i<FG->Points; i++)
		{
			FwdReflMagdB = 20.0 * log10(S11Mag[i]);
			FwdReflPhsDeg = S11Phs[i];
			FwdTranMagdB = 20.0 * log10(S21Mag[i]);
			FwdTranPhsDeg = S21Phs[i];
			RevTranMagdB = 20.0 * log10(S12Mag[i]);
			RevTranPhsDeg = S12Phs[i];
			RevReflMagdB = 20.0 * log10(S22Mag[i]);
			RevReflPhsDeg = S22Phs[i];

			sw->Write(FG->Frequency(i).ToString("D9"));			sw->Write(S"    ");
			sw->Write(FwdReflMagdB.ToString("F3"));			sw->Write(S"  ");
			sw->Write(FwdReflPhsDeg.ToString("F3"));			sw->Write(S"    ");
			sw->Write(FwdTranMagdB.ToString("F3"));			sw->Write(S"  ");
			sw->Write(FwdTranPhsDeg.ToString("F3"));			sw->Write(S"    ");
			sw->Write(RevTranMagdB.ToString("F3"));			sw->Write(S"  ");
			sw->Write(RevTranPhsDeg.ToString("F3"));			sw->Write(S"    ");
			sw->Write(RevReflMagdB.ToString("F3"));			sw->Write(S"  ");
			sw->WriteLine(RevReflPhsDeg.ToString("F3"));
		}
	else												// rectangular format
        for(int i=0; i<FG->Points; i++)
		{
			FwdReflRe = S11Mag[i] * cos(S11Phs[i] / RAD2DEGR);
			FwdReflIm = S11Mag[i] * sin(S11Phs[i] / RAD2DEGR);
			FwdTranRe = S21Mag[i] * cos(S21Phs[i] / RAD2DEGR);
			FwdTranIm = S21Mag[i] * sin(S21Phs[i] / RAD2DEGR);
			RevTranRe = S12Mag[i] * cos(S12Phs[i] / RAD2DEGR);
			RevTranIm = S12Mag[i] * sin(S12Phs[i] / RAD2DEGR);
			RevReflRe = S22Mag[i] * cos(S22Phs[i] / RAD2DEGR);
			RevReflIm = S22Mag[i] * sin(S22Phs[i] / RAD2DEGR);

			sw->Write(FG->Frequency(i).ToString("D9"));		sw->Write(S"    ");
			sw->Write(FwdReflRe.ToString("E5"));				sw->Write(S"  ");
			sw->Write(FwdReflIm.ToString("E5"));				sw->Write(S"    ");
			sw->Write(FwdTranRe.ToString("E5"));				sw->Write(S"  ");
			sw->Write(FwdTranIm.ToString("E5"));				sw->Write(S"    ");
			sw->Write(RevTranRe.ToString("E5"));				sw->Write(S"  ");
			sw->Write(RevTranIm.ToString("E5"));				sw->Write(S"    ");
			sw->Write(RevReflRe.ToString("E5"));				sw->Write(S"  ");
			sw->WriteLine(RevReflIm.ToString("E5"));
		}

	sw->WriteLine();

	sw->Flush();
	sw->Close();

}

void public StoreSParams(bool calmode, FrequencyGrid * FG, CalDataSet * CalData,
				unsigned short trace2 __gc[], unsigned short trace3 __gc[], unsigned short trace4 __gc[],
				unsigned short trace7 __gc[], unsigned short trace8 __gc[], unsigned short trace9 __gc[],
				double reflMag __gc[],  double reflPhs __gc[],  double tranMag __gc[], double tranPhs __gc[])
{
	// Store S-parameters to internal temporary storage.
	// Store Forward caller gives us pointers to Forward storage,
	//   store Reverse caller gives us pointers to Reverse storage
	// calmode: 1 -> use fixture calibration

	double fmagnitude;			// 0 to 1
	double fphase;				// -180 to +180	
	double& rmag = fmagnitude;
	double& rphs = fphase;

	for (int i=0; i<FG->Points; i++)	// Display measurements on the frequency grid
	{
		// Compute S11 or S22
		CalData->ResolveReflPolar(trace3[i], trace4[i], trace2[i], FG->Frequency(i), rmag, rphs);
		if (calmode)
			CorrectS11(CalData, FG->Frequency(i), fmagnitude, fphase, rmag, rphs);

		reflMag[i] = fmagnitude;	// store reflected component in polar form
		reflPhs[i] = fphase;

		
		// Compute S21 or S12
		CalData->ResolveTranPolar(trace8[i], trace9[i], trace7[i], FG->Frequency(i), rmag, rphs);
		if (calmode)
			CorrectS21(CalData, FG->Frequency(i), fmagnitude, fphase, rmag, rphs);

		tranMag[i] = fmagnitude;	// store transmitted component in polar form
		tranPhs[i] = fphase;

	}

}