'
'
'
' Program to run the az-el rotator system
'
'  First draft written 30 November, 1997
'
' J.A. Koehler, VE5FP
' 3290 N. Koehler Rd.,
' Florence,AZ, 85232
'
'--------------------------------------------------
'
' The basic idea is that the controller keeps its own time
' which includes the year, month, day, hour, min, sec 
' by means of internal interrupts.
'
' A sequence of desired azimuths and elevations for 
' particular times is written into internal RAM
' memory and the controller just keeps 
' comparing the actual time to the desired times.  When
' they match, the controller moves the antennas to the
' desired position
'
'  first written as a 68HC11 program using  Karl Lunt's
' SBASIC compiler for a Techarts 68HC11 "ADAPT-
' 11C75 card with 8K of EEPROM
'
'
'
' modified starting 1 Feb., 1998 to add an interrupt driven
' serial input routine because polled input is not fast enough
' to work at 9600 baud
'-------------------------------------------------
'
' Main features of the program are:
'
'
' 1. Timer program runs in the background and, via interrupts,
' updates the variables; NOW_DAY, NOW_TIME, YEEARS, MONTHS, DAYS,
' HOURS, MINUTES, SECONDS
'
' 2. The current antenna position is kept in an EEPROM location
' named AZ_EL
'
' 3. The table of desired time/positions is stored in three arrays:
' DAY(60), TIME(60), and POS(60) - a total RAM space of 360 bytes.
'
' The 16 byte values used in SBASIC may be used to store
' a complete 'day', 'time' or 'azimuth/elevation' in the
' following way:
'
' DAY - most significant 7 bits for year 00 -> 99
'   NOTE the number of (year -1990) is saved
'       next most significant 4 bits for month 01 -> 12
'          least 5 bits for day 00 -> 31
'
' TIME - most significant 5 bits for hour 00 -> 23
'           next most significant 6 bits for minute 00 -> 59
'           least most significant 4 bits for seconds/4 00 -> 14
'
'               NOTE: time can only be specified accurate to 4
'               seconds, not one!
'
' POS  (AZ/EL) - both stored as 1/2 the real value
'                most significant 10 bits for azimuth 00 -> 180
'                 least 6 bits for elevation 00 -> 45
'
' 4. One pointer is used for these arrays:
' END_OF_LIST.  It is initialized to 0 on reset of the
' system.  The array contents are also all zeroes on reset
' of the system.  When data is entered into the arrays, END_OF_LIST
' points to the next available empty (blank) position available.
'  The program checks to make sure that the array is never 'overfilled'.
' The antenna is pointed to the position of the zero_th array
' element.  When the time of the second position is equal to the
' clock time, the arrays are all shuffled down and the antemma
' is pointed to the new position of the zero_th element.
'
' 5.  The present azimuth and elevation angle of the rotors is stored
' in EEPROM as is the number of cegrees per contact closure and the
' rate at which the rotator turns in degrees/sec.  These variables are called:
'
' AZ_EL - the present azimuth and elevation in terms of the number of contacts from zero
' CLICKS - the number of degrees between contact closures
' RATE - the number of degrees per sec that the rotors go
'
'  In all these three variable, the azimuth portion is the most significant byte and
' elevation is the least significant byte of the number.  These numbers are written into EEPROM
' during the calibration portion of the program
'
'       There is one other thing about AZ_EL.  The most significant bit of each byte is set if
' the rotor is on the high side of the contact closure and cleared if the rotor is on the
' low side of the closure.
'
' 6.  Pseudo code listing of the program is:
'
'       main:
'
'         initialize;  {set up SCI port, timers, zero pointers, etc}
'
'         repeat
'         if the antenna position is not equal to the
'           zero_th position, move it there
'
'         if END_OF_LIST <> 0 then
'           if the time pointed to by array element number 1 matches
'                the actual time then move the antenna and shuffle
'                down the array;
'         if a character has been received at serial port then
'           CASE char of:
'                 'c', 'C': calibrate;
'                 't', 'T': set the time;
'                 'u', 'U': move elevation up;
'                 'd', 'D': move elevation down;
'                 'r', 'R': move azimuth CW;
'                 'l', 'L': move azimuth CCW;
'                 's', 'S': stop and disable all motion;
'                 'z', 'Z': print out controller status;
'                 'g', 'G': enable motion;
'                 'x', 'X': clear memory of stored values
'                 'p', 'P': point the rotors at some azimuth and elevation
'               else {must be incoming data}
'                 parse line and store data;
'               end {CASE}
'         until hell freezes over;
'
'       END
'
'
'__________________________________________________________________
'
' last amended 17 June, 1998
'
' Version 1 of working code - no known bugs
'
'__________________________________________________________________
'
'
' First, declare all the variables, etc.'
'
'
include "rot_r11.lib"   ' gives proper addresses for ADAPT 11C75 card

const array_size = 50
'
' put EEPROM storage for AZ_EL, etc on upper page
'
  org $ffb0
  asm

az_el  fcb 0     ; EEPROM storage for AZ_EL - default value for U110
       fcb 0
clicks fcb 10    ; ditto for degrees per click - default value for U110
       fcb 10
calibrated
       fcb 0    ; a word which, if not zero, means that the system HAS been calibrated
       fcb 0
  endasm
  org code

asmfunc get_az_el
asmfunc get_clicks
asmfunc get_cal
asmfunc write_az_el
asmfunc write_clicks
asmfunc write_cal

declare half_click      ' half the click size in degrees
declare number_needed   ' used in MOVE_ANTENNA routine
declare desired         ' ditto
declare click_counter   ' ditto
declare new_value       ' in MOVE_AZ and MOVE_EL routine
declare go_enable       ' variable which, when zero, inhibits motion
declare chr             ' input character from SCI
declare i               ' general purpose loop counter
declare j               ' another of the same
declare k               ' ditto
declare scratch         ' general purpose variable use in getting numbers from input
declare scr2            ' ditto
declare click_return       ' variable which returns result of GET_A_CLICK routine
declare ticks           ' two bytes used for tick counting
declare end_of_list
declare now_day         ' compressed year, month, day
declare now_time        ' compressed hour, minute and second
declare seconds         '  correct time in seconds
declare minutes         '                  minutes
declare hours           '                  hours
declare days            '                  days
declare months          '                  months
declare years           '                  years
declare day(array_size)
declare time(array_size)
declare pos(array_size)
declare buffer          ' this declaration MUST be the last one in the list
'
'
' Then, define the various procedures used
'

  asm

*
* AZ_EL is stored as described in the prologue to the program
*
* CLICKS is the number of degrees per rotator click
* the high order byte applies to the Azimuth rotator, the low
* order byte is for the Elevation rotator
*
* CALIBRATED is a word which is zero if calibration routine has
* not been done - the upper byte has a value of 1 after azimuth
* calibration, the lower byte becomes 1 after elevation calibration
*

get_az_el
	ldd     az_el
	rts

get_clicks
	ldd     clicks
	rts

get_cal
	ldd     calibrated
	rts

delay_1
	pshx            ; generate > 5 mS delay
	ldx     #$1600
dly
	dex
	bne     dly
	pulx
	rts

prelim
	psha            ; save upper byte temporarily
	ldaa    #$aa
	staa    $f555
	ldaa    #$55
	staa    $faaa
	ldaa    #$a0
	staa    $f555
	pula            ; get upper byte back again
	rts

write_az_el
	sei
	jsr     prelim
	std     az_el
	jsr     delay_1
	cli
	rts

write_clicks
	sei
	jsr     prelim
	std     clicks
	jsr     delay_1
	cli
	rts

write_cal
	sei
	jsr     prelim
	std     calibrated
	jsr     delay_1
	cli
	rts

  endasm
'
'___________________________________________
'
'___________________________________________
'
' Then,the interrrupt service routines
'
' first, the timer interrupt routine
'
'
  interrupt $ffe6       ' address of OC2 interrupt service routine
  asm
one_sec       equ     50     ; number of ticks per second
  ldaa  #clear
  staa  _tflg1
  ldd   _toc2
  addd   #dly20ms
  std   _toc2
  ldy   _ticks
  dey
  sty   _ticks
  beq   ready
  rti
ready
*
* if here, the second is up and we need to increment time
*
  ldd   #one_sec
  std   _ticks          ; reset tick timer
  inc   _seconds+1      ; the least signif. byte of _SECONDS
  ldd   _seconds
  cpd   #60
  bne   donetime        ; if it isn't 60, skip through
  ldd   #0
  std   _seconds
  inc   _minutes+1
  ldd   _minutes
  cpd   #60             ; if it isn't 60 minutes, just skip through
  bne   donetime
  ldd   #0
  std   _minutes
  inc   _hours+1
  ldd   _hours
  cpd   #24             ; if it isn't 24 hours, just skip through
  bne   donetime
  ldd   #0
  std   _hours          ; hours, mins and secs are now correct
  inc   _days+1
  bra   fix_days        ; if here, need to check the day, month and year

donetime
  jmp make_now          ; need to jump because of limited range of BR instructions

fix_days
  endasm
'
' it is now easier to correct days, months, years in SBASIC
'
select months

  case 2        ' February
    select years

    case 2000      ' the millenium
    if days = 29 then
      days = 1
      months = 3
    endif
			'
			' then the default if not the millenium
			'
    i = years mod 4
    if i = 0 then
      if days = 30 then
	days = 1
	months = 3
      endif
    else
      if days = 29 then
	days = 1
	months = 3
      endif
    endif
    endcase
    endselect
  endcase


  case 4        'April
  case 6        'June
  case 9        'September
  case 11       'November
  if days = 31 then
    days = 1
    months = months + 1
  endif
  endcase

  if days = 32 then
    days = 1
    months = months + 1
  endif
  endselect

  if months = 13 then
    months = 1
    years = years + 1
  endif

'
'  now to make the words NOW_TIME and NOW_DAY
'
'  NOTE need to define label for previous assembly code
'  to jump to here

asm

make_now

endasm

  now_time = (hours * 1024) + (minutes * 16) + (seconds / 4)
  now_day = years - 1990
  now_day = (now_day * 512) + (months * 32) + days
'
'
  end
'
'
' then, the serial input interrupt routine
'
'
  interrupt $ffd6       ' SCI interrupt vector

  asm
  ldd   #_buffer        ; get buffer address
  addb  ptr             ; add pointer to it
  adca  #0              ; just in case there's a carry
  xgdx                  ; put result into x register
  ldaa  _scsr           ; clear receive flag
  ldaa  _scdr           ; get received character into A
  staa  0,x             ; save it into buffer

  staa  _scdr           ; transmit it also

  ldaa  ptr
  inca
  cmpa  #_array_size
  bne   done_sci
  ldaa  #0              ; make it zero if pointer = array_size
done_sci
  staa  ptr
  endasm

  end

'
'__________________________________________
'
'
initialize:
'
' here is where we set up the ports, timers, and initialize the arrays
'
  asm
*
*       PTR and LBP are stored just after the BUFFER space
*
*  these are the pointer to the buffer space for the circular serial
* input buffer.  PTR points to the next available location to store a byte
* LBP points to the location of the last character read from the buffer
*
* if they are equal to one another, it means a new character has NOT been
* received
*
ptr     equ     _buffer+(_array_size)
lbp     equ     ptr+1

  sei                   ; disable the interrupts
  ldaa    #0
  staa    lbp           ; clear last byte pointer, LBP
  staa    ptr           ; clear pointer PTR

  ldaa  #$30            ; set up the SCI for 9600 baud
  staa  _baud
  ldaa  #$2c            ; enable RCVR interrupt
  staa  _sccr2

  ldaa  _scdr           ; flush receive register

  ldd   #50             ; set up ticks register
  std   _ticks

  ldaa  #%01001000      ; set PORT B (Xicor ports A)  to outputs, C to input
  staa  _cr

  ldaa  #0              ; set all outputs to low 
  staa  _pdrb

*
* then, set up the real-time clock
*
* NOTE!! the actual interrupt routine
* which changes the clock values, etc is
* written just in front of the MAIN program
*
*
clear         equ     $40     ; value to clear OC2F flag
dly20ms       equ     40007   ; number of E clock cycles per 20 mS
*
* note - this should be 40000 for a 8 MHz xtal but for the one
* that I'm using, 40007 is closer to the real rate
*
* using 40000, the clock gained 81 seconds in 126 hours
*

  ldaa  #clear
  staa  _tflg1   ; clear the OC2F flag
  staa  _tmsk1   ; enable the interrupt
  ldd   _tcnt    ; start OC2 input-compare operation
  addd  dly20ms
  std   _toc2
  cli           ; enable interrupt
  endasm  
'
'
' then initialize all the variables to zero
'
'
'

    go_enable = 0
    now_day = 0
    now_time = 0
    seconds = 0
    minutes = 0
    hours = 0
    days = 0
    months = 0
    years = 0

    gosub clear_mem     ' clear the stored positions
'
' finally, set HALF_CLICK
'
    half_click = get_clicks(0) / 512             ' half of most sig. byte
    half_click = (half_click * 256) + ((get_clicks(0) mod 256) / 2)

  return

clear_mem:
'
' routine to set all stored values to zero
'

  for i = 0 to array_size - 1
    day(i) = 0
    time(i) = 0
    pos(i) = 0
    next
  end_of_list = 0


  return

one_msec:
'
' routine which just takes one millisec to run
'
  asm
  pshx
  ldx   #332
one_msec_loop
  dex
  bne   one_msec_loop
  pulx
  endasm

  return


wait_millisecs:
'
' routine to generate time delays in integral numbers of milliseconds
'
' for example, the instruction
'       GOSUB WAIT_MILLISECS, 100
' will generate a delay of 100 milliseconds
'
  j = pull()
  for i = 1 to j
    gosub one_msec
  next
  return
'
'
'
' GET_A_CLICK
'
' routine to monitor the motion of either rotor
'
' returns the variable CLICK_RETURN with a value of -1 if the routine timed out
' returns a -10 if a character was typed during the motion
' or, if normal and a click was detected, it returns with
' a value of 3
'
' the routine is invoked as follows:
' gosub get_a_click
'
' NOTE - uses all four throwaway variables: I, J, K and scr2
'
'
get_a_click:
  
  click_return = 3              ' default value

  do
    scr2 = seconds
  loop until scr2 = seconds  ' just want to be sure seconds isn't changing

  do
    i = peekb(pprc) and 3       ' just detect the lower two bits

    do
      j = seconds - scr2
      k = seconds - scr2
    loop until j = k

    if j >= 4                   ' if it has been here for more than 4 sec
      print "Timed out"
      pokeb pdrb, 0             ' stop the rotor(s)
      click_return = -1
      i = 1                     ' to get out of loop
    endif
    gosub get_character
    
    if chr <> 0
      pokeb pdrb, 0             ' stop the rotor(s)
      click_return = -10
      i = 1                     ' to get out of loop
    endif
    
  loop until i <> 3             ' skip out if one contact has been made

  if i > 0                      ' do this part only if a contact has been made
    gosub wait_millisecs, 80
    
    do
      scr2 = seconds
    loop until scr2 = seconds  ' just want to be sure seconds isn't changing

    do
      i = peekb(pprc) and 3       ' just detect the lower two bits

      do
	j = seconds - scr2
	k = seconds - scr2
      loop until j = k

      if j >= 4                   ' if it has been here for more than 4 sec
	print "Timed out"
	pokeb pdrb, 0             ' stop the rotor(s)
	click_return = -1
	i = 3                     ' to get out of loop
      endif
      
      gosub get_character
      if chr <> 0
	pokeb pdrb, 0             ' stop the rotor(s)
	click_return = -10
	i = 3                     ' to get out of loop
      endif
    
    loop until i = 3              ' skip out if a contact has released
  endif

  gosub wait_millisecs, 80        ' make sure you're past the contact
  return

'
make_a_digit:
'
' MAKE_A_DIGIT
'
' subroutine which enters with CHR as an ASCII character'
' and returns chr as a number between 0 and 9 if character was
' a number between 0 and 9 and with a negative number otherwise
'
  chr = chr - '0'
  if chr > 9
    chr = -1
  endif
  if chr < -1
    chr = -1
  endif
  return
'
make_a_number:
'
' MAKE _A_NUMBER returns a value in the variable SCRATCH
' which consists of the number input
'
  do
    gosub make_a_digit
    if chr < 0
      do
	gosub get_character
      loop until chr <> 0
      gosub make_a_digit
    endif
  loop while chr = -1

  scratch = chr
'  print "In MAKE_A_NUMBER routine, scratch is"; scratch
  do
    do
      gosub get_character
    loop until chr <> 0
    gosub make_a_digit
    if chr <> -1
      scratch = (10 * scratch) + chr
'      print "New scratch ="; scratch; "Chr ="; chr
    endif
  loop while chr <> -1
  return
'
get_character:
'
' subroutine to get a character and return it in
' the global variable, CHR - it returns a zero
' if no character was available
'
  asm
  ldd   #0
  std   _chr            ; make CHR = 0 to start
  ldaa  ptr
  cmpa  lbp
  bne   is_available    ; if NE, it means a character is available to be read
  rts
is_available
  ldd   #_buffer
  addb  lbp
  adca  #0
  xgdx                  ; put pointer to character into X
  ldaa  0,x
  staa  _chr+1
  ldaa  lbp
  inca
  cmpa  #_array_size
  bne   done_getchr
  ldaa  #0
done_getchr
  staa  lbp
  endasm

  return
'
'
move_az:
'
' routine to move azimuth rotor
'
' NUMBER_NEEDED contains the number of clicks needed
' if positive, it is clicks in increasing az., otherwise decreasing
'
'
  new_value = get_az_el(0)
  
  j = get_az_el(0) and $8000    ' if zero, was below click
  if number_needed > 0          ' i.e., want to go upward

    new_value = (get_az_el(0) / 256) and $7f      ' present az in clicks
    new_value = ((new_value + number_needed) or $80) * 256  ' will be above click
    new_value = (get_az_el(0) and $ff) + new_value

    if j = 0
      number_needed = number_needed + 1
    endif

    pokeb pdrb, %00000010       ' start rotating CW - increasing az.
    for click_counter = 1 to number_needed
      do
	gosub get_a_click
	if click_return = -1
	  print "Stalled azimuth rotor"
	  pokeb pdrb, 0                 ' stop the rotor
	endif
      loop until click_return > 0
    next
    pokeb pdrb, 0               ' stop the rotor
  endif

  if number_needed < 0
    new_value = (get_az_el(0) / 256) and $7f      ' present az in clicks
    new_value = (new_value + number_needed) * 256  ' will be below click
    new_value = (get_az_el(0) and $ff) + new_value

    if j <> 0
      number_needed = number_needed - 1 ' it is already negative
    endif

    pokeb pdrb, %00000001       ' start rotating CCW - decreasing az.
    number_needed = -number_needed
    for click_counter = 1 to number_needed
      do
	 gosub get_a_click
	 if click_return = -1
	   print "Stalled azimuth rotor"
	   pokeb pdrb, 0                 ' stop the rotor
	 endif
      loop until click_return > 0
    next
    pokeb pdrb, 0               ' stop the rotor
  endif

  new_value = write_az_el(new_value)
  return
'
'
move_el:
'
' routine to move elevation rotor
'
' NUMBER_NEEDED contains the number of clicks needed
' if positive, it is clicks upward, otherwise downward
'
  new_value = get_az_el(0)
  
  j = get_az_el(0) and $80      ' if zero, was below click
  if number_needed > 0          ' i.e., want to go upward

    new_value = (get_az_el(0) mod 256) and $7f      ' present el in clicks
    new_value = (new_value + number_needed) or $80  ' will be above click
    new_value = (get_az_el(0) and $ff00) + new_value

    if j = 0
      number_needed = number_needed + 1
    endif
    pokeb pdrb, %00001000       ' start rotating upward - increasing el.
    for click_counter = 1 to number_needed
      do
	gosub get_a_click
	if click_return = -1
	  print "Stalled elevation rotor"
	  pokeb pdrb, 0                 ' stop the rotor
	endif
      loop until click_return > 0
    next
    pokeb pdrb, 0               ' stop the rotor
  endif

  if number_needed < 0

    new_value = (get_az_el(0) mod 256) and $7f      ' present el in clicks
    new_value = new_value + number_needed           ' will be below click
    new_value = (get_az_el(0) and $ff00) + new_value

    if j <> 0
      number_needed = number_needed - 1 ' it is already negative
    endif

    pokeb pdrb, %00000100       ' start rotating downward - decreasing el.
    number_needed = -number_needed
    for click_counter = 1 to number_needed
      do
	gosub get_a_click
	if click_return = -1
	  print "Stalled elevation rotor"
	  pokeb pdrb, 0                 ' stop the rotor
	endif
      loop until click_return > 0
    next
    pokeb pdrb, 0               ' stop the rotor
  endif
  new_value = write_az_el(new_value)
  return
'
'
move_antenna:
'
' here is where we move the antenna to point to POS(0)
'

  desired = ((pos(0) / 64) * 2) + (half_click / 256)
  if desired > 359
    desired = 0
  endif
  desired = desired / (get_clicks(0) / 256)
  desired = (256 * desired) + ((((pos(0) mod 64) * 2) + (half_click mod 256))/(get_clicks(0) mod 256))

'
' DESIRED now contains the number of clicks needed for the desired position
' with the high order byte being azimuth and the lower order byte being
' elevation
'
go_to_clicks:
'
' this provides an entry point for pointing the antenna if and when
' the global variable DESIRED contains the desired position in clicks
'
'  print "In GO_TO_CLICKS routine"  
  
  i = get_cal(0) / 256
    if i > 0                    ' i.e., if CALIBRATED
      number_needed = (desired / 256) - ((get_az_el(0) / 256) and $7f)
'
' number_needed is now the difference between the present position and the desired in
' terms of clicks
'
'     print "Az NUMBER_NEEDED is "; number_needed
      gosub move_az
    else
      print "Azimuth is NOT calibrated"
    endif
'
' now do elevation motion
'
  i = get_cal(0) mod 256
    if i > 0                    ' i.e., if calibrated
      number_needed = (desired mod 256) - ((get_az_el(0) mod 256) and $7f)
'     print "El NUMBER_NEEDED is "; number_needed
      gosub move_el
    else
      print "Elevation is NOT calibrated"
    endif
'  print "Leaving GO_TO_CLICKS routine"
  return
'
'
point_antenna:
'
' here is where az. and el. are accepted and the
' antenna moved to that position
'
  gosub make_a_number           ' get desired azimuth
  desired = (scratch + (half_click / 256))
  
  if desired > 359
    desired = 0
  endif
  
  desired = desired / (get_clicks(0) / 256)
'  print "Az DESIRED value is "; desired  
  
  gosub make_a_number           ' get desired elevation
  desired = (256 * desired) + ((scratch + (half_click mod 256)) / (get_clicks(0) mod 256))
'  print "El DESIRED value is "; desired mod 256
'
' DESIRED now contains the desired position in terms of clicks
'
  gosub go_to_clicks
  return
'
'
shuffle_down:
'
' here is where we shuffle down the array
'
' END_OF_LIST is decremented inside this routine
'
  if end_of_list > 0

    for i = 0 to (end_of_list -1)
      j = i + 1
      pos(i) = pos(j)
      time(i) = time(j)
      day(i) = day(j)
      pos(j) = 0
      time(j) = 0
      day(j) = 0
      next

    end_of_list = end_of_list - 1

  endif

'  print "END_OF_LIST ="; end_of_list

  return
'
'
check_az:
'
' routine to set the number of degrees / click for azimuth rotor
'
  print "Is the number of degrees per click correct (y/n)?: ";
  do
    gosub get_character
  loop until chr <> 0
  select chr
    case 'n'
    case 'N'
    '
    ' do the calibration here
    '
    print "\n\rType any character to start azimuth"
    print "rotating CCW towards the stop"
    do
      gosub get_character
    loop until chr <> 0

    print "Then, when it hits the stop, type any character"
    pokeb pdrb, %00000001       ' start rotating CCW - decreasing az.
    do
      gosub get_character
    loop until chr <> 0
    pokeb pdrb, 0
'
' az. rotor is now at zero so need to count clicks to 360 degree position
'
    click_counter = 0
    print "\n\rType any character to start towards other stop"
    print "Then type any character when it has hit the stop"

    do
      gosub get_character
    loop until chr <> 0

    pokeb pdrb, %00000010       ' start rotating CW - increasing az.
    do
      gosub get_a_click
      if click_return > 0
	click_counter = click_counter + 1
	print "Click number "; click_counter; " Click_return "; click_return
      endif
    loop until click_return = -10

    pokeb pdrb, 0
    print "Degrees per click is "; 370 / click_counter
    print "\n\rType the number you think should apply";
    gosub make_a_number
    print "\n\rYou typed "; scratch
    i = scratch
    scratch = get_clicks(0)
    scratch = (scratch mod 256) + (i * 256)
    scratch = write_clicks(scratch)

    endcase

    print "\n\rNo azimuth calibration is needed - there"
    print "are ten degrees between contact closures"

  endselect

  print "\n\rNow, we need to set the azimuth rotor to some reference"
  print "If the rotor is stalled at maximum CW direction"
  print "Type 's', otherwise, type any character"

  do
    gosub get_character
  loop until chr <> 0

  if chr <> 's'
    print "\n\rType any character to start the rotor CW towards the stop"
    print "Then, type any character to stop it as soon as it stalls"
    do
      gosub get_character
    loop until chr <> 0

    pokeb pdrb, %00000010       ' start rotating CW - increasing az.

    do
      gosub get_character
    loop until chr <> 0

    pokeb pdrb, 0
  endif

  gosub wait_millisecs, 1000            ' wait a second

  pokeb pdrb, %00000001                 ' start back
  do
    gosub get_a_click
  loop until click_return > 0                ' go back one click

  pokeb pdrb, 0                         ' stop it

  scratch = get_az_el(0)
  i = get_clicks(0) / 256
  scratch = (scratch mod 256) + (((360 - i) / i) * 256)
  scratch = write_az_el(scratch)        ' write the result

  print "Azimuth is now located at "; 360 - i
  scratch = get_cal(0)
  scratch = scratch or $0100
  scratch = write_cal(scratch)          ' set high byte to be $01

  return

check_el:
'
' routine to set the number of degrees / click for elevation rotor
'
  print "Is the number of degrees per click correct (y/n)?: ";
  do
    gosub get_character
  loop until chr <> 0
  select chr
    case 'n'
    case 'N'
    '
    ' do the calibration here
    '
    print "\n\rType any character to start elevation"
    print "rotating upwards towards the stop"
    do
      gosub get_character
    loop until chr <> 0

    print "Then, when it hits the stop, type any other character"
    pokeb pdrb, %00001000       ' start rotating upward
    do
      gosub get_character
    loop until chr <> 0
    pokeb pdrb, 0
'
' el. rotor is now at max so need to count clicks to 180 degrees less
'
    click_counter = 0
    print "\n\rType any character to start downward to 180 degrees"
    print "from its original position"
    print "Then type again to stop it there"

    do
      gosub get_character
    loop until chr <> 0

    pokeb pdrb, %00000100       ' start rotating downward
    do
      gosub get_a_click
      if click_return > 0
	click_counter = click_counter + 1
	print "Click number "; click_counter; " Click_return "; click_return
      endif
    loop until click_return = -10

    pokeb pdrb, 0

    print "Degrees per click is "; 190 / click_counter
    print "\n\r"
    print "Type number you think should apply ";
    gosub make_a_number
    print "\n\r"
    print "You typed "; scratch
    i = scratch
    scratch = get_clicks(0)
    scratch = (scratch and $ff00) + i
    scratch = write_clicks(scratch)

    endcase

    print "\n\rNo elevation calibration is needed"
  endselect

  print "\n\rNow, we need to set the elevation rotor to some reference"
  print "If the rotor is in the middle of its range,"
  print "type 's', otherwise, type any character."
  do
    gosub get_character
  loop until chr <> 0
  if chr <> 's'
    print "Type any character to start the rotor upward towards the stop"
    print "Type again to stop rotor when it is stalled"

    do
      gosub get_character
    loop until chr <> 0

    pokeb pdrb, %00001000               ' start upward

    do
      gosub get_character
    loop until chr <> 0

    pokeb pdrb, 0                       ' then stop it

    gosub wait_millisecs, 1000          ' wait a second

    number_needed = 180 / (get_clicks(0) mod 256)   '  number of clicks in 180 degrees
    print "We need to go down "; number_needed; " clicks"

    pokeb pdrb, %00000100               ' start downward

    for click_counter = 1 to number_needed
      do
	gosub get_a_click
	if click_return = -1
	  gosub wait_millisecs, 200
	  pokeb pdrb, %00000100 ' start rotating downward
	endif
      loop until click_return > 0
      print click_counter
    next

    pokeb pdrb, 0

  endif

  gosub wait_millisecs, 1000            ' wait a second
  pokeb pdrb, %00000100                 ' start downward
  do
    gosub get_a_click
  loop until click_return > 0                ' go one click

  pokeb pdrb, 0                         ' stop it

  scratch = get_az_el(0) and $ff00      ' since elevation is now zero
  scratch = write_az_el(scratch)        ' write the result

  print "Elevation is now located at 0"
  scratch = get_cal(0)
  scratch = scratch or $0001
  scratch = write_cal(scratch)          ' set low byte to be $01
  return


calibrate:
'
' calibration subroutine
'
  print "\n\rType 'e' to calibrate elevation, 'a' for azimuth:";
  do
    gosub get_character
  loop until chr <> 0
  print "\n\r"

  select chr
    case 'e'
    case 'E'
    gosub check_el
    endcase


    case 'a'
    case 'A'
    gosub check_az
    endcase

  endselect

  half_click = ((get_clicks(0) / 512) * 256) + ((get_clicks(0) mod 256) / 2)

  return
'
set_time:
'
' time setting subroutine
'

  do
    gosub get_character
  loop until chr <> 0
  if chr <> 13          ' "ENTER" key has been struck?
    ticks = 1000        ' enough time to input the whole line
    gosub make_a_number
    years = scratch
    gosub make_a_number
    months = scratch
    gosub make_a_number
    days = scratch
    gosub make_a_number
    hours = scratch
    gosub make_a_number
    minutes = scratch
    gosub make_a_number
    seconds = scratch
    ticks = 50         ' start clock countdown again
  endif
  print "\nDate/time is "; years; months; days; hours; minutes; seconds
'  print "Now_time, Now_day"; now_time, now_day

  return
'
'
report_status:
'
' routine to report on controller status
'
  print "\n\rAz-El system status\n\n\r"

  if go_enable = 0
    print "Rotator drive is DISABLED"
  else
    print "Rotator drive is ENABLED"
  endif

  scratch = get_cal(0) / 256
  if scratch = 0
    print "Azimuth: NOT calibrated; ";
  else
    print "Azimuth: calibrated; ";
  endif

  scratch = get_cal(0) mod 256
  if scratch = 0
    print "Elevation: NOT calibrated "
  else
    print "Elevation: calibrated "
  endif



  scratch = get_az_el(0)
  i = get_clicks(0)
  print "Azimuth: "; ((scratch / 256) and $7f) * (i / 256);
  print "Elevation: "; ((scratch mod 256) and $7F) * (i mod 256)
  print "Az. rotor degrees/click is: "; i / 256
  print "El. rotor degrees/click is: "; i mod 256
  print "Number of stored positions: "; end_of_list
  print "Date/time: "; years; months; days; hours; minutes; seconds

  if end_of_list <> 0
  for scratch = 0 to (end_of_list - 1)
    i = (day(scratch) / 512)
    j = (day(scratch) - (i * 512)) / 32
    i = i + 1990
    print "Next day is:"; i; j; day(scratch) mod 32;
    i = time(scratch) / 1024
    j = (time(scratch) - (i * 1024)) / 16
    print " time: "; i; j; (time(scratch) mod 16) * 4;
    print " position: "; (pos(scratch) / 64) * 2; (pos(scratch) mod 64) * 2
  next
  endif
  return
'
parse_line:
'
' routine to get and parse a line and put the
' positions and time into the arrays
'
' it assumes that the next 8 numbers are (respectively)
'
' year, month, day, hour, minute, second, asimuth, elevation
'
'
 
  gosub make_a_number
  day(end_of_list) = (scratch - 1990) * 512
  gosub make_a_number
  day(end_of_list) =day(end_of_list) + (scratch *32)
  gosub make_a_number
  day(end_of_list) = day(end_of_list) + scratch
'  print "DAY = "; day(end_of_list)

  gosub make_a_number
  time(end_of_list) = scratch * 1024
  gosub make_a_number
  time(end_of_list)  = time(end_of_list) + (scratch * 16)
  gosub make_a_number
  time(end_of_list) = time(end_of_list) + (scratch / 4)
'  print "TIME = "; time(end_of_list)

  gosub make_a_number
  pos(end_of_list) = (scratch / 2) * 64
  gosub make_a_number
  pos(end_of_list) = pos(end_of_list) + (scratch / 2)
'  print "Position = "; pos(end_of_list)

  end_of_list = end_of_list + 1

  do while chr <> 13  ' get to END_OF_LINE character
    gosub get_character
  loop

  outch 10              ' send a LF to acknowledge END_OF_LINE

  return
'
'_____________________________________________________
'_____________________________________________________
'_____________________________________________________
'
'
' Finally, get to the main program itself
'
'
'_____________________________________________________
'




main:

  gosub initialize

  do

    if end_of_list <> 0        ' on RESET everything is zero
      if time(0) = now_time
	if day(0) = now_day
	  if go_enable <> 0
	    gosub move_antenna
	  endif
	  gosub shuffle_down
	endif
      endif
    endif

    gosub get_character
    if chr <> 0 then
      chr = chr and $ff
      select chr
	case 'C'
	case 'c'
	gosub calibrate
	endcase

	case 'T'
	case 't'
	gosub set_time
	endcase

	case 'U'
	case 'u'
	pokeb pdrb, %00001000   ' start upward
	print "\n\rMove UP - type any character to stop"
	do
	  gosub get_character
	loop until chr <> 0
	pokeb pdrb, 0
	scratch = get_cal(0) and $ff00  ' el. is UNcalibrated by moving it
	scratch = write_cal(scratch)
	endcase

	case 'D'
	case 'd'
	pokeb pdrb, %00000100   ' start downward
	print "\n\rMove down - type any character to stop"
	do
	  gosub get_character
	loop until chr <> 0
	pokeb pdrb, 0
	scratch = get_cal(0) and $ff00  ' el. is UNcalibrated by moving it
	scratch = write_cal(scratch)
	endcase

	case 'R'
	case 'r'
	pokeb pdrb, %00000010   ' increasing azimuth
	print "\n\rMove CW - type any character to stop"
	do
	  gosub get_character
	loop until chr <> 0
	pokeb pdrb, 0
	scratch = get_cal(0) and $00ff  ' az. is UNcalibrated by moving it
	scratch = write_cal(scratch)
	endcase

	case 'L'
	case 'l'
	pokeb pdrb, %00000001           ' decreasing azimuth
	print "\n\rMove CCW - type any character to stop"
	do
	  gosub get_character
	loop until chr <> 0
	pokeb pdrb, 0
	scratch = get_cal(0) and $00ff  ' az. is UNcalibrated by moving it
	scratch = write_cal(scratch)
	endcase

	case 'S'
	case 's'
	pokeb pdrb, 0
	go_enable = 0
	print "\n\rSTOP"
	endcase

	case 'Z'
	case 'z'
	gosub report_status
	endcase

	case 'G'
	case 'g'
	go_enable = 1
	print "\n\rGO"
	endcase

	case 'X'
	case 'x'
	gosub clear_mem
	print "\n\rMemory cleared"
	endcase

	case 'P'
	case 'p'
	gosub point_antenna
	print "\n\r"
	endcase

	case '1'        ' the year will start with either a 1 or a 2
	case '2'
	gosub parse_line
	endcase

      endselect
    endif
  loop

end

