UNIT COMMS;
(****************************************************************)
(*								*)
(*	Interrupt driven, Serial Data, reception program. This	*)
(*	set of routines of routines can be used to receive data	*)
(*	through COM1 or COM2.					*)
(*	Upon program initialisation program is configured	*)
(*	to COM1, 9600 baud, 8 bits, no parity , 1 stop bit.	*)
(*								*)
(*	Functions you need to call:-				*)
(*								*)
(*	SETUP_SLU( PORTNUM: BYTE;				*)
(*		   BITRATE: WORD;				*)
(*		   PARITY : CHAR;				*)
(*		   NSTOP  : BYTE);				*)
(*								*)
(*	Initialises the appropriate COM port.			*)
(*								*)
(*	Parameters						*)
(*	----------						*)
(*	PORTNUM = 1 or 2,  Selects COM1 or COM2			*)
(*	BITRATE = 150,300,600,1200,2400,4800,9600,19200		*)
(*	PARITY  = 'E','O' or 'N' for Even, Odd or None		*)
(*	NSTOP   = 1 or 2, Number of stop bits.			*)
(*								*)
(*	RECVD_CHAR						*)
(*	----------						*)
(*	TRUE if characters in input buffer, FALSE if input	*)
(*	buffer is empty.					*)
(*								*)
(*	RINGPOP							*)
(*	-------							*)
(*	Pops next character from input buffer. Waits until	*)
(*	buffer is not empty.					*)
(*								*)
(*	STOPSLU							*)
(*	-------							*)
(*	Stops data input from the COM port. Call SETUP_SLU to	*)
(*	start it again.						*)
(*								*)
(*	XMITCHAR( C: CHAR );					*)
(*	--------------------					*)
(*	Transmits the character C out of the selected COM port	*)
(*								*)
(*	Global variables					*)
(*	----------------					*)
(*	OVERERR	Number of over run errors since setup.		*)
(*	PARYERR	Number of parity errors since setup.		*)
(*	FRAMERR	Number of framing errors since setup.		*)
(*	RECVERR	Total number of errors since setup		*)
(*								*)
(****************************************************************)

INTERFACE

{ ------ global procedures and functions ---------}

PROCEDURE SETUP_SLU( PORTNUM: BYTE;
		     BITRATE: WORD;
		     PARITY : CHAR;
		     NSTOP  : BYTE);
FUNCTION    RECVD_CHAR: BOOLEAN;
FUNCTION    RINGPOP: CHAR;
PROCEDURE   STOPSLU;
PROCEDURE   XMITCHAR( C: CHAR );

{ ------ global variables -----------}

VAR
OVERERR     : LONGINT;		{Number of overrun errors}
PARYERR     : LONGINT;		{Number of parity errors}
FRAMERR     : LONGINT;		{Number of framing errors}
RECVERR     : LONGINT;		{Total number of errors}

(****************************************************************)
(*								*)
(*	Start of Implementation					*)
(*								*)
(****************************************************************)

IMPLEMENTATION
USES DOS;

CONST
DLAB    = $80;		{Data Latch Access Bit }
I8259   = $20;		{Interrupt Mask Register }
EOI     = $20;		{End Of Interrupt }
STI     = $FB;		{STI machine code }
CLI     = $FA;		{CLI machine code }
RINGLEN = $7FFF;	{Length of char buffer must be power of 2**N - 1}

TYPE
BIGCHARBUFF = ARRAY [0..RINGLEN] OF CHAR;
WORDREP     = RECORD
		 L0 : BYTE;
		 L1 : BYTE;
	      END;

VAR
RINGCHARS   : ^BIGCHARBUFF;	{the buffer for characters 32K}
IMR_MASK    : BYTE;		{Mask of master IMR before interrupts enabled}
ISR_EXITSAVE: POINTER;		{pointer to end of program power down}
OLDCOM1     : POINTER;  	{pointer to save BIOS COM1 ISR}
OLDCOM2     : POINTER;		{pointer to save BIOS COM2 ISR}
THR         : WORD;		{SLU Transmitter holding register }
IIR         : WORD;		{SLU Interrupt Ident Register }
LCR         : WORD;		{SLU Line Control Register }
LSR         : WORD;		{SLU Line Status Register }
IER         : WORD;		{SLU Interrupt enable register }
MSR         : WORD;		{SLU Modem status register }
MCR         : WORD;		{SLU Modem Control Register }
RINGSPAN    : WORD;		{Number of chars in ringbuffer }
RINGTOP     : WORD;		{Points to next vacant space in buffer }
RINGBOT     : WORD;		{Points to next char to be taken from buffer }

(*>*)
(*********************************************************************)
(*                                                                   *)
(*   Sluset                                                          *)
(*   ------                                                          *)
(*   Procedure SLUSET sets up either slu1 or Slu2 for interrupt      *)
(*   driven I/O.                                                     *)
(*                                                                   *)
(*********************************************************************)

PROCEDURE SLUSET( PORTNUM  : BYTE;	{ Set up software controllable}
		  BAUDIV   : WORD;	{Baud rate generator}
		  BITSCODE : BYTE;	{Code fo No of data bits}
		  STOPCODE : BYTE);	{Code for no of stop bits}

VAR
PORTADD : BYTE;

BEGIN
   IF PORTNUM = 2 THEN PORTADD := 2 ELSE PORTADD := 3;
   WORDREP(THR).L1 := PORTADD;				{get proper address}
   WORDREP(IIR).L1 := PORTADD;				{of the port}
   WORDREP(LCR).L1 := PORTADD;				{COM1 at $300}
   WORDREP(LSR).L1 := PORTADD;				{COM2 at $200}
   WORDREP(IER).L1 := PORTADD;				{set the high byte}
   WORDREP(MSR).L1 := PORTADD;				{of the address}
   WORDREP(MCR).L1 := PORTADD;				{accordingly}

   PORT[LCR]   := DLAB;                  { Set Data Latch Access Bit for next part }
   PORT[THR]   := LO (BAUDIV);           { Load baud rate }
   PORT[THR+1] := HI (BAUDIV);           { Load baud rate (contd) }
   PORT[LCR]   := BITSCODE OR STOPCODE;  { Load in bits, parity and stop bits}
   PORT[MCR]   := $0F;                   { sout2 CONTROLS INTERRUPT SIGNALS}
   PORT[IER]   := 00;                    { Disable interrupt on char received }
END;

(****************************************************************)
(*								*)
(*	RINGINIT						*)
(*	--------						*)
(*	Initialises the Current input buffers			*)
(*								*)
(****************************************************************)

PROCEDURE RINGINIT;

VAR
PORTADD : BYTE;

BEGIN
   INLINE(CLI);
   PORTADD     := PORT[LSR];		{Clear any pending error bits}
   PORTADD     := PORT[THR];		{clear any characters in buffer}
   RINGSPAN    := 0;			{reset No of characters in buffer}
   RINGTOP     := 0;			{Reset next place to insert incoming characters}
   RINGBOT     := 0;			{Points to next char to be taken from buffer }
   OVERERR     := 0;            	{Number of over run errors}
   PARYERR     := 0;			{Number of parity errors}
   FRAMERR     := 0;			{Number of framing errors}
   RECVERR     := 0;			{Total number of receiver errors}
   INLINE(STI);				{renable interrupts}
END;

(************************************************************************)
(*									*)
(*	RECVD_CHAR							*)
(*	----------							*)
(*	True if there are characters in the RECEIVE buffer.		*)
(*									*)
(************************************************************************)

FUNCTION RECVD_CHAR: BOOLEAN;

BEGIN
   RECVD_CHAR := RINGSPAN > 0;
END;

(**********************************************************************)
(*                                                                    *)
(*    Ringpop                                                         *)
(*    -------                                                         *)
(*    Function ringpop fetches a character from the interrupt driven  *)
(*    character collection routine.                                   *)
(*                                                                    *)
(**********************************************************************)

FUNCTION RINGPOP: CHAR;

BEGIN
   REPEAT UNTIL RINGSPAN > 0;			{wait until Char in buffer}
   RINGPOP := RINGCHARS^[RINGBOT];		{Return next char in buffer }
   RINGBOT := (RINGBOT + 1) AND RINGLEN;	{Bump bottom pointer }
   INLINE( CLI );				{disable interrupts}
   RINGSPAN:= RINGSPAN - 1;			{One less char in buffer }
   INLINE (STI);				{Enable CPU interrupts }
END;

(*********************************************************************)
(*                                                                   *)
(*   Sluisr                                                          *)
(*   ------                                                          *)
(*   Turbo Pascal Interrupt service routine to receive and buffer    *)
(*   characters received into the SLU. Note that the interrupt       *)
(*   can access typed constants but not global constants. The routine*)
(*   simply places the character into a buffer and increments buffer *)
(*   pointers.                                                       *)
(*                                                                   *)
(*********************************************************************)

{$R-}
{$S-}
VAR ERR : BYTE;

PROCEDURE SLUISR; INTERRUPT;		{SLU Interrupt Service Routine }

BEGIN
   RINGCHARS^[RINGTOP] := CHR(PORT[THR]);		{Plonk in buffer }
   ERR := PORT[LSR];					{get the error status}
   IF (ERR AND $02) <> 0 THEN OVERERR := OVERERR+1;	{check Over run error}
   IF (ERR AND $04) <> 0 THEN PARYERR := PARYERR+1;	{check parity error}
   IF (ERR AND $08) <> 0 THEN FRAMERR := FRAMERR+1;	{check framing error}
   IF (ERR AND $0E) <> 0 THEN RECVERR := RECVERR+1;	{Increment total error counter}
   RINGTOP     := (RINGTOP +  1) AND RINGLEN;		{Increment next free space in buffer}
   RINGSPAN    := (RINGSPAN + 1);			{Increment buffer counter}
   PORT[I8259] := EOI;					{Acknowlege interrupt to 8259}
END;
{$R+}
{$S+}

(**********************************************************************)
(*                                                                    *)
(*   StopSlu                                                          *)
(*   -------                                                          *)
(*   Stop the SLU from generating any more interrupts.                *)
(*                                                                    *)
(**********************************************************************)

PROCEDURE STOPSLU;

BEGIN
   PORT[IER] := 0;               {disable interrupts on the SLU}
END;

(***********************+*************************************************)
(*									*)
(*	XMITCHAR							*)
(*	--------							*)
(*	Sends a char to the COM port. Simply polls the THR status bit	*)
(*	and sends character when buffer empty.				*)
(*									*)
(************************************************************************)

PROCEDURE XMITCHAR( C: CHAR );

BEGIN
   REPEAT UNTIL (PORT[LSR] AND $40) <> 0;
   PORT[THR] := ORD(C);
END;

(************************************************************************)
(*									*)
(*  SETUP_SLU								*)
(*  ---------								*)
(*  Initialises all the interrupt software necessary to drive the	*)
(*  serial line unit.							*)
(*									*)
(************************************************************************)

PROCEDURE SETUP_SLU( PORTNUM: BYTE;
		     BITRATE: WORD;
		     PARITY : CHAR;
		     NSTOP  : BYTE);

VAR
BAUDIV   : WORD; 		{Baud rate 9600}
BITSCODE : BYTE;		{Bits per byte and parity = 8N}
STOPCODE : BYTE;		{Stop bits = 2}

BEGIN
   STOPSLU;				{disable current COM interrupts}
   IF PORTNUM <> 2 THEN PORTNUM := 1;	{get port number in valid range}
   BAUDIV := 115200 DIV BITRATE;	{get the baudrate in appropriate units}
   CASE UPCASE(PARITY) OF		{get the parity bits}
      'E' : BITSCODE := $1B;		{check for even parity, set 8 bits even}
      'O' : BITSCODE := $0B;		{check for Odd parity, set 8 bits odd}
      ELSE  BITSCODE := $03;		{else use no parity, set 8 bits no parity}
   END;					{get number of stop code bits}
   IF NSTOP < 2 THEN STOPCODE := 0 ELSE STOPCODE := $04;
   IF PORTNUM = 1 THEN				{now set up interrupts}
   BEGIN					{first do COM1}
      SETINTVEC( $0C,@SLUISR );			{set up COM1 ISR}
      SETINTVEC( $0B,OLDCOM2 );			{restore COM2 ISR to BIOS}
      PORT[I8259+1] := IMR_MASK AND $EF;	{enable COM1 interrupts at 8259}
   END ELSE					{else its not COM1}
   BEGIN					{therfore}
      SETINTVEC( $0B,@SLUISR );			{setup COM2 ISR}
      SETINTVEC( $0C, OLDCOM1 );		{restore COM1 to BIOS}
      PORT[I8259+1] := IMR_MASK AND $F7;	{setup 8259 interrupt bits}
   END;
   SLUSET(PORTNUM,BAUDIV,BITSCODE,STOPCODE);	{Set up SLU }
   RINGINIT;					{Initialise the ringbuffer input system }
   PORT[IER] := $01;				{Enable Receiver interrupts}
END;

(****************************************************************)
(*								*)
(*	ISREXIT							*)
(*	-------							*)
(*      Disables interrupts upon program termination.		*)
(*								*)
(****************************************************************)

{$F+}
PROCEDURE ISREXIT;

BEGIN
   STOPSLU;				{ Disable SLU interrupts}
   PORT[I8259+1] := IMR_MASK;		{restore the Interrupt mask regsister}
   EXITPROC := ISR_EXITSAVE;		{Point to next exit procedure}
   SETINTVEC( $0C,OLDCOM1 );		{restore the old timer tick}
   SETINTVEC( $0B,OLDCOM2  );		{restore old IRQ 7 interrupt routine}
END;
{$F-}

(****************************************************************)
(*								*)
(*	Once only initialisation code.				*)
(*								*)
(****************************************************************)

BEGIN
   ISR_EXITSAVE := EXITPROC;		{save the old exit pointer}
   EXITPROC     := @ISREXIT;		{setup new exit procedure into list}
   GETINTVEC( $0C, OLDCOM1 );		{get address of old RTC ISR routine}
   GETINTVEC( $0B, OLDCOM2  );		{get address of old IRQ 7}

   THR := $3F8;			{COM1 SLU Transmitter holding register }
   IIR := $3FA;			{COM1 SLU Interrupt Ident Register }
   LCR := $3FB;			{COM1 SLU Line Control Register }
   LSR := $3FD;			{COM1 SLU Line Status Register }
   IER := $3F9;			{COM1 SLU Interrupt enable register }
   MSR := $3FE;			{COM1 SLU Modem status register }
   MCR := $3FC;			{COM1 SLU Modem Control Register }
   NEW(RINGCHARS);		{allocate space on heap for buffer}
   IMR_MASK := PORT[I8259+1];	{Get the old interrupt mask}
   SETUP_SLU( 1,9600,'N',1 );	{start I/O on SLU 1}
END.
