{

ObjectName: TList navigator
Author: Robert M. Czerwinski - based on TNavigator
Target: I created this using Delphi 4.0 but it should work in other
        32 bit versions of Delphi
Purpose: This allows me manage visually a TList object containing
        authors of particular upload to DSP.
Usage:
 1) drop on the form
 2) assign ListNavigator.ListRef to your TList object on Form create event
          ( yup, you have to type it )
 3) assign events:
       OnClicking: you HAVE TO add to your TList new object on LNQB_ADD button
                   event
        OnClicked: mainly for on-screen update
 OnCurrentChanged: notification about change of the currently selected item in TList

}
unit ListNav;

interface
uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Graphics,
  Controls,
  StdCtrls,
  ExtCtrls,
  Buttons;


{$R *.DCR}

type
  TListNavButton = class;

  TListNavigateBtn = (
          LNQB_FIRST,
          LNQB_PRIOR,
          LNQB_NEXT,
          LNQB_LAST,
          LNQB_ADD,
          LNQB_DELETE,
          LNQB_CANCEL,
          LNQB_APPLY,
          LNQB_MOVE_UP,
          LNQB_MOVE_DOWN
         );

  TListNavigatorStatus = ( LNS_BROWSE, LNS_EDIT, LNS_INSERT );

  TListNavClickEvent = procedure (Sender: TObject; Button: TListNavigateBtn ) of object;
  TListNavCurrentChangedEvent = procedure (Sender: TObject; Current: integer ) of object;

  { TListNavigator }
  TListNavigator = class (TCustomPanel)
  private
    FListRef: TList;
    FCurrent: Integer;
//    FObjectDataChanged: Boolean;
    FListItemChanging: Boolean;
    FNavStatus: TListNavigatorStatus;
    FButtons: array[TListNavigateBtn] of TListNavButton;
    MinBtnSize: TPoint;
    FOnNavClicking: TListNavClickEvent;
    FOnNavClicked: TListNavClickEvent;
    FOnCurrentChanged: TListNavCurrentChangedEvent;
    FFlat: Boolean;
    procedure ClickHandler(Sender: TObject);
    procedure InitButtons;
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure SetFlat(Value: Boolean);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure SetListRef(Value: TList);
    procedure SetCurrent(Value: Integer);
//    procedure SetObjectDataChanged(Value: Boolean);
    procedure FromListToButtons;
  protected
    procedure Loaded; override;
    function GetNavButton(Index: TListNavigateBtn): TListNavButton;
  public
    constructor Create(AOwner: TComponent ); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;

    procedure ObjectDataChanged;

    procedure BtnClick(Index: TListNavigateBtn);
    property  NavButton[Index: TListNavigateBtn]: TListNavButton read GetNavButton;
    property  ListRef: TList read FListRef write SetListRef;
    property  Current: Integer read FCurrent write SetCurrent;
    property  NavStatus: TListNavigatorStatus read FNavStatus;
//    property  ObjectDataChanged: boolean read FObjectDataChanged
//                                        write SetObjectDataChanged;
  published
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Flat: Boolean read FFlat write SetFlat default False;
    property Ctl3D;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClicking: TListNavClickEvent read FOnNavClicking write FOnNavClicking;
    property OnClicked: TListNavClickEvent read FOnNavClicked write FOnNavClicked;
    property OnCurrentChanged: TListNavCurrentChangedEvent read FOnCurrentChanged write FOnCurrentChanged;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
    property OnStartDrag;
  end;

{ TListNavButton }
  TListNavButton = class(TSpeedButton)
  private
    FIndex: TListNavigateBtn;
  protected
  public
    property Index : TListNavigateBtn read FIndex write FIndex;
  end;

  procedure Register;

implementation
{$R *.RES}

{ TListNavigator }

var
  BtnTypeName: array[TListNavigateBtn] of PChar =
                 ('FIRST',
                  'PRIOR',
                  'NEXT',
                  'LAST',
                  'ADD',
                  'DELETE',
                  'CANCEL',
                  'APPLY',
                  'MOVE_UP',
                  'MOVE_DOWN');

constructor TListNavigator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  MinBtnSize := Point(20, 18);
  FListRef:= nil;
  FCurrent:= 1;
  FNavStatus:= LNS_BROWSE;
  FListItemChanging:= false;
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Width := 250;
  Height := 25;
  FromListToButtons;
end;

destructor TListNavigator.Destroy;
begin
  inherited Destroy;
end;

procedure TListNavigator.InitButtons;
var
  I: TListNavigateBtn;
  Btn: TListNavButton;
  X: Integer;
  ResName: string;
begin
  X := 0;
  for I := Low(FButtons) to High(FButtons) do
  begin
    Btn := TListNavButton.Create (Self);
    Btn.Flat := Flat;
    Btn.Index := I;
    Btn.Visible := true;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    FmtStr(ResName, 'RMC_LNQB_%s', [BtnTypeName[I]]);
    Btn.Glyph.LoadFromResourceName(HInstance, ResName);
    Btn.NumGlyphs := 2;
    Btn.Enabled := True;
    Btn.OnClick := ClickHandler;
    Btn.Parent := Self;
    FButtons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
end;

procedure TListNavigator.SetListRef(Value: TList);
Begin
  if Value = nil then Exit;
  FListRef:= Value;
  if FListRef.Count > 0 then FCurrent:= 0
                        else FCurrent:= -1;
  FromListToButtons;
End;

procedure TListNavigator.SetCurrent(Value: Integer);
Begin
  FCurrent:= Value;
  FromListToButtons;
  if Assigned( FOnCurrentChanged ) then FOnCurrentChanged(Self, FCurrent);
End;

procedure TListNavigator.SetFlat(Value: Boolean);
var
  I: TListNavigateBtn;
begin
  if FFlat <> Value then
  begin
    FFlat := Value;
    for I := Low(FButtons) to High(FButtons) do
      FButtons[I].Flat := Value;
  end;
end;

procedure TListNavigator.AdjustSize (var W: Integer; var H: Integer);
var
  Count, ButtonWidth, MinW: Integer;
  I: TListNavigateBtn;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if FButtons[LNQB_First] = nil then Exit;
  Count:= Ord(High(FButtons))+1;
  MinW := MinBtnSize.X * Count;
  if W < MinW then W := MinW;
  if H < MinBtnSize.Y then H := MinBtnSize.Y;

  ButtonWidth := W div Count;
  Temp := ButtonWidth * Count;
  if Align = alNone then W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(FButtons) to High(FButtons) do
  begin
    Space := 0;
    if Remain <> 0 then
    begin
      Dec(Temp, Remain);
      if Temp < 0 then
      begin
        Inc(Temp, Count);
        Space := 1;
      end;
    end;
    FButtons[I].SetBounds(X, 0, ButtonWidth + Space, Height);
    Inc(X, ButtonWidth + Space);
  end;
end;

procedure TListNavigator.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  if not HandleAllocated then AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TListNavigator.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;
  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TListNavigator.FromListToButtons;
var
  I: TListNavigateBtn;
Begin
  if FListRef = nil then
  Begin
    for I := Low(FButtons) to High(FButtons) do
      FButtons[I].Enabled := False;
  End else
  Begin
    FButtons[LNQB_FIRST].Enabled := ( FCurrent > 0 ) and ( FNavStatus = LNS_BROWSE );
    FButtons[LNQB_PRIOR].Enabled := ( FCurrent > 0 ) and ( FNavStatus = LNS_BROWSE );
    FButtons[LNQB_NEXT].Enabled := ( FCurrent < FListRef.Count - 1 ) and ( FCurrent >= 0 ) and ( FNavStatus = LNS_BROWSE );
    FButtons[LNQB_LAST].Enabled := ( FCurrent < FListRef.Count - 1 ) and ( FCurrent >= 0 ) and ( FNavStatus = LNS_BROWSE );
    FButtons[LNQB_ADD].Enabled := FNavStatus = LNS_BROWSE;
    FButtons[LNQB_CANCEL].Enabled := not ( FNavStatus = LNS_BROWSE );
    FButtons[LNQB_DELETE].Enabled := ( FListRef.Count > 0 ) and ( FNavStatus = LNS_BROWSE );
    FButtons[LNQB_APPLY].Enabled := not ( FNavStatus = LNS_BROWSE );
    FButtons[LNQB_MOVE_UP].Enabled := ( FCurrent > 0 ) and ( FNavStatus = LNS_BROWSE );
    FButtons[LNQB_MOVE_DOWN].Enabled := ( FCurrent < FListRef.Count - 1 ) and ( FCurrent >= 0 ) and ( FNavStatus = LNS_BROWSE );
  End;
end;


procedure TListNavigator.ClickHandler(Sender: TObject);
begin
  BtnClick(TListNavButton(Sender).Index);
end;

procedure TListNavigator.BtnClick(Index: TListNavigateBtn);
begin
  if (csDesigning in ComponentState) then Exit;
  if Assigned(FOnNavClicking) then FOnNavClicking(Self, Index);
  case Index of

      LNQB_FIRST:
         Begin
           FListItemChanging:= true;
           Current:= 0;
           FListItemChanging:= false;
         End;

      LNQB_PRIOR:
         Begin
           FListItemChanging:= true;
           Current:= Current - 1;
           FListItemChanging:= false;
         End;

      LNQB_NEXT:
         Begin
           FListItemChanging:= true;
           Current:= Current + 1;
           FListItemChanging:= false;
         End;

      LNQB_LAST:
         Begin
           FListItemChanging:= true;
           Current:= FListRef.Count - 1;
           FListItemChanging:= false;
         End;

      LNQB_ADD:
         Begin
           // make sure that object was created
           if FListRef.Count = 0 then
             raise Exception.Create('Create object on OnClicking event');
           FNavStatus:= LNS_INSERT;
           Current:= Current + 1;
         End;

      LNQB_DELETE:
         Begin
           // free current object from the list
           TObject(FListRef.Items[FCurrent]).Free;
           FListRef.Delete(FCurrent);
           FListItemChanging:= true;
           // last item removed ?
           if FCurrent >= FListRef.Count then Current:= Current - 1
                                         else Current:= Current; // "fake" to call method
         End;

      LNQB_CANCEL:
         Begin
           // free current object from the list if it was just added
           if FNavStatus = LNS_INSERT then
           Begin
             TObject(FListRef.Items[FCurrent]).Free;
             FListRef.Delete(FCurrent);
             // last item removed ?
             if FCurrent >= FListRef.Count then
             Begin
               FListItemChanging:= true;
               Current:= Current - 1;
               FListItemChanging:= false;
             End;
           End else Current:= Current;
           FNavStatus:= LNS_BROWSE;
         End;

      LNQB_APPLY:
         Begin
           FNavStatus:= LNS_BROWSE;
         End;

      LNQB_MOVE_UP:
         Begin
           FListItemChanging:= true;
           FListRef.Exchange( FCurrent, FCurrent - 1);
           Current:= Current - 1;
           FListItemChanging:= false;
         End;

      LNQB_MOVE_DOWN:
         Begin
           FListItemChanging:= true;
           FListRef.Exchange( FCurrent, FCurrent + 1);
           Current:= Current + 1;
           FListItemChanging:= false;
         End;
   end; { case }
   FromListToButtons;
  if Assigned(FOnNavClicked) then FOnNavClicked(Self, Index);
end;

procedure TListNavigator.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
end;


function TListNavigator.GetNavButton(Index: TListNavigateBtn): TListNavButton;
begin
  result:= FButtons[Index];
end;

//procedure TListNavigator.SetObjectDataChanged(Value: Boolean);
procedure TListNavigator.ObjectDataChanged;
begin
  if FListItemChanging then Exit;
  if FNavStatus = LNS_BROWSE then
  Begin
    FNavStatus:= LNS_EDIT;
    FromListToButtons;
  End;
end;

procedure Register;
Begin
  RegisterComponents('RMC', [TListNavigator]);
End;

Begin
end.
