unit UplClass;
{$I '..\DSP.inc'}

interface


uses
  SysUtils,
  Windows,
  Messages,
  Classes,
  IniFiles,

{$IFDEF DSP_DATABASE}
  DB,
  DBTables,
{$ENDIF}

  forms;

type

(******************************************************************************)
  TDSPFilePlatform = class(TObject)
  private
    FPlatformName: String;
{$IFDEF DSP_DATABASE}
    FPlatformID: integer;
{$ENDIF}
  public
    constructor Create; virtual;
    property PlatformName: string read FPlatformName write FPlatformName;
{$IFDEF DSP_DATABASE}
    property PlatformID: integer read FPlatformID write FPlatformID;
{$ENDIF}
  End;

(******************************************************************************)
  TDSPFilePlatformList = class(TObject)
  private
    FList: TList;
    function  GetPlatform(Index: Integer): TDSPFilePlatform;
  public
    constructor Create; virtual;
    destructor Destroy; override;
    procedure BuildFromIni(Ini: TIniFile);
    procedure BuildFromIniFile;
{$IFDEF DSP_DATABASE}
    procedure BuildFromTable(aTblFilesPlt: TTable);
    procedure BuildFromDM(DM: TDataModule);
{$ENDIF}
    property  Platforms[Index: Integer]: TDSPFilePlatform read GetPlatform;
    property  Items: TList read FList;
  End;


(******************************************************************************)
  TDSPFileGroup = class(TObject)
  private
    FGroupName: String;
    FGroupDir: String;
{$IFDEF DSP_DATABASE}
    FGroupID: integer;
{$ENDIF}
  public
    constructor Create; virtual;
    property GroupName: string read FGroupName write FGroupName;
    property GroupDir: string read FGroupDir write FGroupDir;
{$IFDEF DSP_DATABASE}
    property GroupID: integer read FGroupID write FGroupID;
{$ENDIF}
  End;

(******************************************************************************)
  TDSPFileGroupList = class(TObject)
  private
    FList: TList;
    function  GetGroup(Index: Integer): TDSPFileGroup;
  public
    constructor Create; virtual;
    destructor Destroy; override;
    procedure BuildFromIni(Ini: TIniFile);
    procedure BuildFromIniFile;
{$IFDEF DSP_DATABASE}
    procedure BuildFromTable(aTblFilesGrp: TTable);
    procedure BuildFromDM(DM: TDataModule);
{$ENDIF}
    function  GetGroupDir(aGroupName: string): string;
    property  Groups[Index: Integer]: TDSPFileGroup read GetGroup;
    property  Items: TList read FList;
  End;


(******************************************************************************)
  TDSPAuthor = class(TObject)
  private
    FAuthorName: string;
    FContact: string;
    FAuthorEmail: string;
    FAuthorHome: string;
    FAuthorInfo: TStringList;
    FAuthorGroup: string;
    FUpdateDSP: boolean;
{$IFDEF DSP_DATABASE}
    FAuthorID: integer;
{$ENDIF}
  public
    constructor Create; virtual;
    destructor Destroy; override;
    procedure ReadFromIniFile(S: String);
    procedure ReadFromIni(Ini: TIniFile);
    procedure ReadFromIniByPos(Ini: TIniFile; Pos: integer);
    procedure ReadFromIniBySection(Ini: TIniFile; Section: string); dynamic;
    procedure WriteToIniFile(S: String);
    procedure WriteToIni(Ini: TIniFile);
    procedure WriteToIniByPos(Ini: TIniFile; Pos: integer);
    procedure WriteToIniBySection(Ini: TIniFile; Section: string); dynamic;
    function  IsTheSameAs(anAuthor: TDSPAuthor): boolean; dynamic;
    procedure CopyFrom(anAuthor: TDSPAuthor);
{$IFDEF DSP_DATABASE}
    procedure ReadFromTable(aTblAuthors, aTblSitesCat: TTable);
{$ENDIF}
    procedure Reset; virtual;
    property AuthorName: String read FAuthorName write FAuthorName;
    property Contact: String read FContact write FContact;
    property AuthorEmail: String read FAuthorEmail write FAuthorEmail;
    property AuthorHome: String read FAuthorHome write FAuthorHome;
    property AuthorInfo: TStringList read FAuthorInfo write FAuthorInfo;
    property AuthorGroup: string read FAuthorGroup write FAuthorGroup;
    property UpdateDSP: boolean read FUpdateDSP write FUpdateDSP;
{$IFDEF DSP_DATABASE}
    property AuthorID: integer read FAuthorID write FAuthorID;
{$ENDIF}
  End;

(******************************************************************************)
  TDSPAuthorList = class(TObject)
  private
    FList: TList;
    FExcludeAuthor: String;
    function  GetDSPAuthor(Index: Integer): TDSPAuthor;
    procedure PutDSPAuthor(Index: Integer; aDSPAuthor: TDSPAuthor);
  public
    constructor Create; virtual;
    destructor Destroy; override;
    function  AddAuthor: integer;
    function  InsertAuthor(AtPos: Integer): integer;
    function  CreateAuthor: TDSPAuthor; virtual;
    procedure ReadFromIni(Ini: TIniFile);
    procedure ReadFromIniFile(S: String);
    procedure WriteToIni(Ini: TIniFile);
    procedure WriteToIniFile(S: String);
    procedure Reset;
    function  FindAuthorByName(aName: string): Integer;
    property  DSPAuthor[Index: Integer]: TDSPAuthor read GetDSPAuthor write PutDSPAuthor;
    property  ExcludeAuthor: String read FExcludeAuthor write FExcludeAuthor;
    property  Items: TList read FList;
  end;

(******************************************************************************)
  TDSPCompilerBase = class
  private
     FCompilerName: String;
     FCompilerID: String;

//   there were two internal fields to store compiler data
//     FSupported: boolean;
//     FAvailable: String;
//   these will be declared in descendent TDSPCompiler
//   other descendent TDSPCompilerEx will define different objects but
//   will support the same functionality as TDSPCompiler object

  public
    constructor Create(aCompilerName, aCompilerID: string); virtual;
    destructor Destroy; Override;
    function GetAvailable: String; virtual; abstract;
    procedure SetAvailable(const Value: String); virtual; abstract;
    function GetSupported: boolean; virtual; abstract;
    procedure SetSupported(const Value: boolean); virtual; abstract;
    procedure ReadFromIni(Ini: TIniFile); virtual;
    procedure WriteToIni(Ini: TIniFile);  virtual;
{$IFDEF DSP_DATABASE}
    procedure ReadFromTable(aTblFiles: TTable);
{$ENDIF}
    procedure Reset; virtual;
    property  Supported: boolean read GetSupported write SetSupported;
    property  Available: String read GetAvailable write SetAvailable;
    property  CompilerName: String read FCompilerName write FCompilerName;
    property  CompilerID: String  read FCompilerID write FCompilerID;
  end;


(******************************************************************************)
  TDSPCompiler = class(TDSPCompilerBase)
  private
     FSupported: boolean;
     FAvailable: String;
  public
    constructor Create(aCompilerName, aCompilerID: string); override;
    function GetAvailable: String; override;
    procedure SetAvailable(const Value: String); override;
    function GetSupported: boolean; override;
    procedure SetSupported(const Value: boolean); override;
  end;


(******************************************************************************)
  TDSPCompilerBaseList = class
  private
     FList: TList;
  public
     constructor Create; virtual;
     function CreateCompiler( aCompilerName, aCompilerID: String): TDSPCompilerBase; virtual;
     destructor Destroy; override;
     procedure Reset;
     function  GetCompiler(Index: Integer): TDSPCompilerBase;
     procedure ReadFromIni(Ini: TIniFile);
     procedure WriteToIni(Ini: TIniFile);
     procedure ReadFromInf(Ini: TIniFile);
     procedure BuildFromIniFile;
{$IFDEF DSP_DATABASE}
     procedure BuildFromTable(Table: TTable);
     procedure BuildFromDM(DM: TDataModule);
     procedure ReadFromTable(aTblFiles: TTable);
{$ENDIF}
     function  GetBaseCompilerID: string;
     function  GetNameFromID(aCompilerID: String): string;
     function  GetCompilerFromID(aCompilerID: String): TDSPCompilerBase;
     property  Items: TList read FList;
     property  Compiler[Index: integer] : TDSPCompilerBase read GetCompiler;
  End;


(******************************************************************************)
  TDSPCompilerList = class(TDSPCompilerBaseList)
  public
     function CreateCompiler( aCompilerName, aCompilerID: String): TDSPCompilerBase; override;
  End;


(******************************************************************************)
  TDSPInfoBaseClass = class
  private
{$IFDEF DSP_DATABASE}
    FFileID: Integer;
{$ENDIF}
    FAuthors   : TDSPAuthorList;
    FCompilers : TDSPCompilerBaseList;
    FGroups    : TDSPFileGroupList;
    FPlatforms  : TDSPFilePlatformList;
    FFileName: String;
    FReplaceFile: String;
    FUploadDesc: TStringList;
    FNote : String;
    FExtDescFile: String;
    FFreeware: boolean;
    FWithSource: boolean;
    FSupport: boolean;
    FCategory: string;
    FFileVersion: String;
    FFileGroup: string;
    FFilePlatform: string;
    FUpdateDSP: boolean;
    FUPLFile: String;
    FFileDir: String;
    procedure DeduceTargetDir;
   public
     Constructor Create; virtual;
     procedure   CreateAuthors; virtual;
     procedure   CreateCompilers; virtual;
     Destructor  Destroy; override;
     procedure   Reset;
     procedure   ReadFromIniFile(IniName: String);
     procedure   WriteToIniFile(IniName: String);
     procedure   ReadFromInfFile(InfName: String);
{$IFDEF DSP_DATABASE}
     procedure BuildFromDM(DM: TDataModule);
     property FileID: Integer read FFileID write FFileID;
{$ENDIF}
     property FileName: string read FFileName write FFileName;
     property ReplaceFile: string read FReplaceFile write FReplaceFile;
     property UploadDesc: TStringList read FUploadDesc write FUploadDesc;
     property Note: string read FNote write FNote;
     property FileVersion: string read FFileVersion write FFileVersion;
     property ExtDescFile: string read FExtDescFile write FExtDescFile;
     property Freeware: boolean  read FFreeware write FFreeware;
     property WithSource: boolean  read FWithSource write FWithSource;
     property Support: boolean  read FSupport write FSupport;
     property UpdateDSP: boolean  read FUpdateDSP write FUpdateDSP;
     property Category: string read FCategory write FCategory;
     property FileGroup: string read FFileGroup write FFileGroup;
     property FilePlatform: string read FFilePlatform write FFilePlatform;
     property Authors: TDSPAuthorList read FAuthors write FAuthors;
     property Compilers: TDSPCompilerBaseList read FCompilers write FCompilers;
     property Groups: TDSPFileGroupList read FGroups write FGroups;
     property Platforms: TDSPFilePlatformList read FPlatforms write FPlatforms;
     property UPLFile: string read FUPLFile write FUPLFile;
     property FileDir: string read FFileDir write FFileDir;
  End;

(******************************************************************************)
  TDSPInfoClass = class(TDSPInfoBaseClass)
   public
     procedure   CreateCompilers; override;
  End;



(******************************************************************************)
(******************************************************************************)
(******************************************************************************)
implementation
(******************************************************************************)
(******************************************************************************)
(******************************************************************************)

uses
  UPL_Cons;

function  GetWorkingDir: string;
var
  FN: array[0..255] of Char;
Begin
  GetModuleFileName(hInstance, @FN, SizeOf(FN));
  PChar(StrRScan(FN, '\')+ 1)^:=  chr(0);
  result:= StrPas(FN);
End;


procedure ReadStringListFromIni(Ini: TIniFile;
                                c_Section: String;
                                SL: TSTringList);
var
  i, Count: integer;
Begin
  SL.Clear;
  Count:= Ini.ReadInteger(c_Section, cUPL_Count, 0);
  if Count = 0 then Exit;
  for i:= 1 to Count do
    SL.Add(Ini.ReadString(c_Section, IntToStr(i), ''));
End;


procedure WriteStringListToIni(Ini: TIniFile;
                               c_Section: String;
                               SL: TSTringList);
var
  i: integer;
Begin
  Ini.WriteInteger(c_Section, cUPL_Count, SL.Count);
  if SL.Count = 0 then Exit;
  for i:= 0 to SL.Count - 1 do
    Ini.WriteString(c_Section, IntToStr(i + 1), SL.Strings[i]);
End;


{$IFDEF DSP_DATABASE}
function LocateTable(TableName: String; DM: TDataModule): TTable;
var
  i: integer;
begin
  For i:= 0 to DM.ComponentCount - 1 do
    if ( DM.Components[i] is TTable ) and
       ( CompareText(TTable(DM.Components[i]).TableName, TableName) = 0 ) then
    Begin
      result:= TTable(DM.Components[i]);
      Exit;
    End;
  raise Exception.Create('No table ' + TableName + ' in DataModule!!');
End;
{$ENDIF}

(******************************************************************************)
{ TDSPFilePlatform }
constructor TDSPFilePlatform.Create;
begin
  FPlatformName:= '';
{$IFDEF DSP_DATABASE}
  FPlatformID:= -1;
{$ENDIF}
end;

(******************************************************************************)
{ TDSPFilePlatformList }
constructor TDSPFilePlatformList.Create;
begin
  FList:= TList.Create;
end;

destructor TDSPFilePlatformList.Destroy;
var
  i: integer;
Begin
  for i:= FList.Count - 1 downto 0 do
  Begin
    TDSPFilePlatform(FList.Items[i]).Free;
    FList.Items[i]:= nil;
  End;
  FList.Count:= 0;
  FList.Free;
  Inherited Destroy;
End;

function  TDSPFilePlatformList.GetPlatform(Index: Integer): TDSPFilePlatform;
begin
  result:= TDSPFilePlatform(FList.Items[Index]);
End;

{$IFDEF DSP_DATABASE}
procedure TDSPFilePlatformList.BuildFromTable(aTblFilesPlt: TTable);
var
  FilePlatform: TDSPFilePlatform;
begin
  aTblFilesPlt.First;
  while not aTblFilesPlt.EOF do
  Begin
    FilePlatform:= TDSPFilePlatform.Create;
    with FilePlatform do
    Begin
      PlatformName:= aTblFilesPlt.FieldByName('PlatformName').AsString;
      PlatformID:= aTblFilesPlt.FieldByName('PlatformID').AsInteger;
    End;
    FList.Add(FilePlatform);
    aTblFilesPlt.Next;
  End;
end;
{$ENDIF}

{$IFDEF DSP_DATABASE}
procedure TDSPFilePlatformList.BuildFromDM(DM: TDataModule);
Begin
  BuildFromTable(LocateTable('FilesPlt.db', DM));
end;
{$ENDIF}


procedure TDSPFilePlatformList.BuildFromIni(Ini: TIniFile);
var
  i, count: integer;
  FilePlatform: TDSPFilePlatform;
begin
  // first read lines count
  count:= Ini.ReadInteger(cUPL_FilePlatform, cUPL_count, 0);
  // if 0 then silent abort
  if count = 0 then Abort;
  for i:= 1 to count do
  Begin
    FilePlatform:= TDSPFilePlatform.Create;
    try
      FilePlatform.PlatformName:= Ini.ReadString(cUPL_FilePlatform, IntToStr(i), '!');
      if ( Length(FilePlatform.PlatformName) <= 1 ) or ( FilePlatform.PlatformName[1] = '!' )
        then raise Exception.Create('Invalid entry in section [' + cUPL_FilePlatform + ']' +
                                 #13#10 + 'Line : ' + IntToStr(i) + #13#10 +
                                 'File: ' + Ini.FileName);
      FList.Add(FilePlatform);
    except
      // sth went wrong - remove created group
      FilePlatform.Free;
      // reraise exception
      raise
    End;
  End;
end;

procedure TDSPFilePlatformList.BuildFromIniFile;
var
  Ini: TIniFile;
begin
  Ini := TIniFile.Create(GetWorkingDir +  cUPL_UplBuilderDataFile);
  try
    BuildFromIni(Ini);
  finally
    Ini.Free;
  End;
End;


(******************************************************************************)
{ TDSPFileGroup }
constructor TDSPFileGroup.Create;
begin
  FGroupName:= '';
  FGroupDir:= '';
{$IFDEF DSP_DATABASE}
  FGroupID:= -1;
{$ENDIF}
end;


(******************************************************************************)
{ TDSPFileGroupList }
constructor TDSPFileGroupList.Create;
begin
  FList:= TList.Create;
end;

destructor TDSPFileGroupList.Destroy;
var
  i: integer;
Begin
  for i:= FList.Count - 1 downto 0 do
  Begin
    TDSPFileGroup(FList.Items[i]).Free;
    FList.Items[i]:= nil;
  End;
  FList.Count:= 0;
  FList.Free;
  Inherited Destroy;
End;

function TDSPFileGroupList.GetGroup(Index: Integer): TDSPFileGroup;
Begin
  result:= TDSPFileGroup(FList.Items[Index]);
End;  

function TDSPFileGroupList.GetGroupDir(aGroupName: string): string;
var
  i: integer;
Begin
  for i:= 0 to FList.Count - 1 do
   if CompareStr(aGroupName, TDSPFileGroup(FList.Items[i]).GroupName ) = 0 then
   Begin
     result:= TDSPFileGroup(FList.Items[i]).GroupDir;
     Exit;
   End;
  raise Exception.Create('Couldn''t locate directory for group: "' + aGroupName + '"');
End;

{$IFDEF DSP_DATABASE}
procedure TDSPFileGroupList.BuildFromTable(aTblFilesGrp: TTable);
var
  FileGroup: TDSPFileGroup;
begin
  aTblFilesGrp.First;
  while not aTblFilesGrp.EOF do
  Begin
    FileGroup:= TDSPFileGroup.Create;
    with FileGroup do
    Begin
      GroupName:= aTblFilesGrp.FieldByName('FileGrpName').AsString;
      GroupDir:= aTblFilesGrp.FieldByName('DSPDir').AsString;
      GroupID:= aTblFilesGrp.FieldByName('FileGrpID').AsInteger;
    End;
    FList.Add(FileGroup);
    aTblFilesGrp.Next;
  End;
end;
{$ENDIF}

{$IFDEF DSP_DATABASE}
procedure TDSPFileGroupList.BuildFromDM(DM: TDataModule);
Begin
  BuildFromTable(LocateTable('FilesGrp.db', DM));
end;
{$ENDIF}

procedure TDSPFileGroupList.BuildFromIni(Ini: TIniFile);
var
  i, count: integer;
  FileGroup: TDSPFileGroup;
begin
  // first read lines count
  count:= Ini.ReadInteger(cUPL_DSPGroups, cUPL_count, 0);
  // if 0 then exit
  if count = 0 then Exit;
  for i:= 1 to count do
  Begin
    FileGroup:= TDSPFileGroup.Create;
    try
      FileGroup.GroupName:= Ini.ReadString(cUPL_DSPGroups, IntToStr(i), '!');
      if ( Length(FileGroup.GroupName) <= 1 ) or ( FileGroup.GroupName[1] = '!' )
        then raise Exception.Create('Invalid entry in section [' + cUPL_DSPGroups + ']' + #13#10 +
                                    'Line : ' + IntToStr(i) + #13#10 +
                                    'File: ' + Ini.FileName);
      // group dir is listed in section [GroupName] ( only !Component/Code )
      if CompareText( FileGroup.GroupName, cUPL_ComponentCode ) <> 0 then
      Begin
        FileGroup.GroupDir:= Ini.ReadString(FileGroup.GroupName, cUPL_DSPDir, '!');
        if ( Length(FileGroup.GroupDir) <= 1 ) or ( FileGroup.GroupDir[1] = '!' )
          then raise Exception.Create('Couldn''t locate directory of ' + FileGroup.GroupName + #13#10 +
                                      'File: ' + Ini.FileName);
      end;
      FList.Add(FileGroup);
    except
      // sth went wrong - remove created group
      FileGroup.Free;
      raise
    End;
  End;
end;

procedure TDSPFileGroupList.BuildFromIniFile;
var
  Ini: TIniFile;
begin
  Ini := TIniFile.Create(GetWorkingDir +  cUPL_UplBuilderDataFile);
  try
    BuildFromIni(Ini);
  finally
    Ini.Free;
  End;
End;


(*******************************************************************************)
{ TDSPAuthor }
constructor TDSPAuthor.Create;
begin
  FAuthorInfo:= TStringList.Create;
  Reset;
end;

destructor TDSPAuthor.Destroy;
begin
  FAuthorInfo.Free;
  inherited Destroy;
end;

procedure TDSPAuthor.ReadFromIniFile(S: String);
var
  Ini: TIniFile;
Begin
  Ini:= TIniFile.Create(S);
  try
    ReadFromIni(Ini);
  finally
    Ini.Free;
  End;
End;

procedure TDSPAuthor.ReadFromIni(Ini: TIniFile);
begin
// use this for reading author info from uploader.dat file ( ONE author )
  ReadFromIniBySection(Ini, cUPL_UploaderSection);
end;

procedure TDSPAuthor.ReadFromIniByPos(Ini: TIniFile; Pos: integer);
begin
// use this for reading author info from uploader.dat file ( ONE author )
  ReadFromIniBySection(Ini, cUPL_UploaderSection + '_' + IntToStr(Pos));
end;

procedure TDSPAuthor.ReadFromIniBySection(Ini: TIniFile; Section: String);
begin
  with Ini do
  begin
    FAuthorName := ReadString(Section, cUPL_AuthorName, '');
    FContact := ReadString(Section, cUPL_Contact, '');
    FAuthorEMail := ReadString(Section, cUPL_AuthorEmail, '');
    FAuthorHome := ReadString(Section, cUPL_AuthorHome, '');
    FAuthorInfo.Clear;
    ReadStringListFromIni(Ini, Section, FAuthorInfo);
    FAuthorGroup:= ReadString(Section, cUPL_AuthorGroup, '');
    // force to update
    FUpdateDSP := CompareStr(ReadString(Section, cUPL_UpdateDSP, cUPL_YES), cUPL_YES) = 0;
  End;
{$IFDEF DSP_DATABASE}
  FAuthorID:= -1;
{$ENDIF}
end;

procedure TDSPAuthor.WriteToIniFile(S: String);
var
  Ini: TIniFile;
Begin
  Ini:= TIniFile.Create(S);
  try
    WriteToIni(Ini);
  finally
    Ini.Free;
  End;
End;

procedure TDSPAuthor.WriteToIni(Ini: TIniFile);
begin
  WriteToIniBySection(Ini, cUPL_UploaderSection);
end;

procedure TDSPAuthor.WriteToIniByPos(Ini: TIniFile; Pos: integer);
begin
  WriteToIniBySection(Ini, cUPL_UploaderSection + '_' + IntToStr(Pos));
end;

procedure TDSPAuthor.WriteToIniBySection(Ini: TIniFile; Section: String);
begin
  with Ini do
  begin
    { I assume that only following items exist in this section thus I can
      erase the entire section }
    WritePrivateProfileString(PChar(Section), nil, nil, PChar(Ini.FileName));
    WriteString(Section, cUPL_AuthorName, FAuthorName);
    WriteString(Section, cUPL_Contact, FContact);
    WriteString(Section, cUPL_AuthorEmail, FAuthorEMail );
    WriteString(Section, cUPL_AuthorHome, FAuthorHome);
    WriteStringListToIni(Ini, Section, FAuthorInfo);
    WriteString(Section, cUPL_AuthorGroup, FAuthorGroup);
    if FUpdateDSP then WriteString(Section, cUPL_UpdateDSP, cUPL_YES )
                  else WriteString(Section, cUPL_UpdateDSP, cUPL_NO );
  End;
end;

function  TDSPAuthor.IsTheSameAs(anAuthor: TDSPAuthor): boolean;
Begin
  result:= false;
  if CompareStr(AuthorName, anAuthor.AuthorName) <> 0 then Exit;
  if CompareStr(Contact, anAuthor.Contact) <> 0 then Exit;
  if CompareStr(AuthorEmail, anAuthor.AuthorEmail) <> 0 then Exit;
  if CompareStr(AuthorHome, anAuthor.AuthorHome) <> 0 then Exit;
  if CompareStr(AuthorGroup, anAuthor.AuthorGroup) <> 0 then Exit;
  if CompareStr(AuthorInfo.Text, anAuthor.AuthorInfo.Text) <> 0 then Exit;
  result:= true;
End;

procedure TDSPAuthor.CopyFrom(anAuthor: TDSPAuthor);
Begin
  FAuthorName:= anAuthor.AuthorName;
  FContact:= anAuthor.Contact;
  FAuthorEmail:= anAuthor.AuthorEmail;
  FAuthorHome:= anAuthor.AuthorHome;
  FAuthorGroup:= anAuthor.AuthorGroup;
  FUpdateDSP:= anAuthor.UpdateDSP;
  FAuthorInfo.Assign(anAuthor.AuthorInfo);
{$IFDEF DSP_DATABASE}
  FAuthorID:= anAuthor.AuthorID;
{$ENDIF}
End;

{$IFDEF DSP_DATABASE}
procedure TDSPAuthor.ReadFromTable(aTblAuthors, aTblSitesCat: TTable);
Begin
  FAuthorID:= aTblAuthors.FieldByName('AuthorID').AsInteger;
  FAuthorName:= aTblAuthors.FieldByName('AuthorName').AsString;
  if aTblAuthors.FieldByName('Contact').IsNull
    then FContact:= ''
    else FContact:= aTblAuthors.FieldByName('Contact').AsString;
  if aTblAuthors.FieldByName('Email').IsNull
    then FAuthorEmail:= ''
    else FAuthorEmail:= aTblAuthors.FieldByName('Email').AsString;
  if aTblAuthors.FieldByName('URL').IsNull
    then FAuthorHome:= ''
    else FAuthorHome:= aTblAuthors.FieldByName('URL').AsString;
  FAuthorGroup:= '';
  if ( not aTblAuthors.FieldByName('SiteCatID').IsNull ) and
     aTblSitesCat.FindKey( [ aTblAuthors.FieldByName('SiteCatID').AsInteger ] )
    then FAuthorGroup:= aTblSitesCat.FieldByName('Description').AsString;
  if aTblAuthors.FieldByName('Property').IsNull
    then FAuthorInfo.Text:= ''
    else FAuthorInfo.Text:= aTblAuthors.FieldByName('Property').AsString;
  FUpdateDSP:= false;
End;
{$ENDIF}

procedure TDSPAuthor.Reset;
begin
  FAuthorName:= '';
  FContact:= '';
  FAuthorEmail:= '';
  FAuthorHome:= '';;
  FAuthorInfo.Clear;
  FAuthorGroup:= '';
  FUpdateDSP:= true;
{$IFDEF DSP_DATABASE}
  FAuthorID:= -1;
{$ENDIF}
end;

{********************************************************************************}
{ TDSPAuthorList }
constructor TDSPAuthorList.Create;
begin
  FList:= TList.Create;
end;

destructor TDSPAuthorList.Destroy;
begin
  Reset;
  FList.Free;
  inherited destroy;
end;

function TDSPAuthorList.CreateAuthor: TDSPAuthor;
begin
  result:= TDSPAuthor.Create;
end;

function TDSPAuthorList.AddAuthor: integer;
begin
  result:= FList.Add(CreateAuthor);
end;

function TDSPAuthorList.InsertAuthor(AtPos: Integer): integer;
begin
  FList.Insert(AtPos, CreateAuthor);
  result:= AtPos;
end;

function TDSPAuthorList.GetDSPAuthor(Index: Integer): TDSPAuthor;
begin
  result:= TDSPAuthor(FList.Items[Index]);
end;

procedure TDSPAuthorList.PutDSPAuthor(Index: Integer; aDSPAuthor: TDSPAuthor);
begin
  FList.Items[Index]:= aDSPAuthor;
end;

procedure TDSPAuthorList.ReadFromIniFile(S: String);
var
  Ini: TIniFile;
Begin
  Ini:= TIniFile.Create(S);
  try
    ReadFromIni(Ini);
  finally
    Ini.Free;
  End;
End;

procedure TDSPAuthorList.ReadFromIni(Ini: TIniFile);
var
  i: Integer;
  Section, S: String;
begin
  if CompareStr(Ini.ReadString(cUPL_UploaderSection, cUPL_AuthorName, cUPL_Error ), cUPL_Error ) <> 0 then
  Begin
    // old UPL - one author only
    DSPAuthor[AddAuthor].ReadFromIniBySection(Ini, cUPL_UploaderSection);
  End else
  Begin
    // more than ONE author, read in a loop until first error occur
    i:= 1;
    Section:= cUPL_UploaderSection + '_' + IntToStr(i);
    S:= Ini.ReadString(Section, cUPL_AuthorName, cUPL_Error );
    while CompareStr(Ini.ReadString(Section, cUPL_AuthorName, cUPL_Error ), cUPL_Error ) <> 0 do
    Begin
      DSPAuthor[AddAuthor].ReadFromIniBySection(Ini, Section);
      Inc(i);
      Section:= cUPL_UploaderSection + '_' + IntToStr(i);
    End;
  End;
end;

procedure TDSPAuthorList.WriteToIniFile(S: String);
var
  Ini: TIniFile;
Begin
  Ini:= TIniFile.Create(S);
  try
    WriteToIni(Ini);
  finally
    Ini.Free;
  End;
End;

procedure TDSPAuthorList.WriteToIni(Ini: TIniFile);
var
  i, Pos: Integer;
begin
  Pos:= 1;
  if FList.Count = 0 then Exit;
  for i:= 0 to FList.Count - 1 do
  Begin
    if CompareStr(TDSPAuthor(FList.Items[i]).AuthorName, ExcludeAuthor) <> 0 then
    Begin
      TDSPAuthor(FList.Items[i]).WriteToIniBySection(Ini, cUPL_UploaderSection + '_' + IntToStr(Pos));
      Inc(Pos);
    End;
  End;
end;

function  TDSPAuthorList.FindAuthorByName(aName: string): Integer;
var
  i: integer;
Begin
  result:= -1;  // not found
  if FList.Count > 0 then
  for i:= 0 to FList.Count - 1 do
  Begin
    if CompareStr(aName, DSPAuthor[i].AuthorName) = 0 then
    Begin
      result:= i;
      Exit;
    End;
  End;
End;

procedure TDSPAuthorList.Reset;
// remove all authors from FList
var
  i: integer;
begin
  if FList.Count > 0 then
  Begin
    for i:= FList.Count - 1 downto 0 do
    Begin
      TDSPAuthor(FList.Items[i]).Free;
      FList.Items[i]:= nil;
    End;
    FList.Count:= 0;
  End;
end;

(************************************************)
constructor TDSPCompilerBase.Create(aCompilerName, aCompilerID: string);
Begin
  Inherited Create;
  FCompilerName:= aCompilerName;
  FCompilerID:= aCompilerID;
End;

destructor TDSPCompilerBase.Destroy;
Begin
  Inherited Destroy;
End;

procedure TDSPCompilerBase.ReadFromIni(Ini: TIniFile);
Begin
  // write through properties
  Supported:= Ini.ReadString( cUPL_FileSection, FCompilerName, cUPL_No)[1] = cUPL_YES ;
  if Supported then Available:= ''
               else Available:= Ini.ReadString( cUPL_FileAvailableSection, FCompilerName, '');
End;

procedure TDSPCompilerBase.WriteToIni(Ini: TIniFile);
Begin
  if Supported then
  Begin
    Ini.WriteString( cUPL_FileSection, FCompilerName, cUPL_YES);
    Ini.WriteString( cUPL_FileAvailableSection, FCompilerName, '');
  end else
  Begin
    Ini.WriteString( cUPL_FileSection, FCompilerName, cUPL_NO);
    if Length(Available) > 0
     then Ini.WriteString( cUPL_FileAvailableSection, FCompilerName, Available)
     else Ini.WriteString( cUPL_FileAvailableSection, FCompilerName, '');
  End;
End;

{$IFDEF DSP_DATABASE}
procedure TDSPCompilerBase.ReadFromTable(aTblFiles: TTable);
Begin
  Supported:= aTblFiles.FieldByName(FCompilerID).AsBoolean;
  // try to load available
  if not Supported
    then Available:= aTblFiles.FieldByName(FCompilerID + 'a').AsString;
  // please do note that we have loaded FileID - not FileName itself
End;
{$ENDIF}


procedure TDSPCompilerBase.Reset;
Begin
  Supported:= false;
  Available:= '';
End;

(***********************************************************************************)
{ TDSPCompiler }
constructor TDSPCompiler.Create(aCompilerName, aCompilerID: string);
begin
  inherited Create(aCompilerName, aCompilerID);
  Reset;
end;

function TDSPCompiler.GetAvailable: String;
begin
  result:= FAvailable;
end;

function TDSPCompiler.GetSupported: boolean;
begin
  result:= FSupported;
end;

procedure TDSPCompiler.SetAvailable(const Value: String);
begin
  if CompareStr(Value, FAvailable) <> 0 then
  FAvailable:= value;
end;

procedure TDSPCompiler.SetSupported(const Value: boolean);
begin
  if FSupported <> Value
   then FSupported:= Value;
end;

(***********************************************************************************)
constructor TDSPCompilerBaseList.Create;
begin
  FList:= TList.Create;
end;

destructor TDSPCompilerBaseList.Destroy;
var
 i: integer;
Begin
  for i:= FList.Count - 1 downto 0 do
  Begin
    TDSPCompilerBase(FList.Items[i]).Free;
    FList.Items[i]:= nil;
  End;
  FList.Free;
  inherited Destroy;
End;

function TDSPCompilerBaseList.CreateCompiler( aCompilerName, aCompilerID: String): TDSPCompilerBase;
begin
  result:= TDSPCompilerBase.Create(aCompilerName, aCompilerID);
end;

procedure TDSPCompilerBaseList.Reset;
var
  i: integer;
Begin
  for i:= 0 to FList.Count - 1 do
   TDSPCompilerBase(FList.Items[i]).Reset;
End;

procedure TDSPCompilerBaseList.BuildFromIniFile;
var
  IniDat: TIniFile;
  i, CompCount: integer;
  aCompilerName,
  aCompilerID: String;
Begin
  IniDat := TIniFile.Create(GetWorkingDir +  cUPL_UplBuilderDataFile);
  try
    CompCount:= IniDat.ReadInteger(cUPL_ComponentCode, cUPL_Count, 0);
    if CompCount = 0 then SysUtils.Abort;
    for i:= 0 to CompCount - 1 do
    Begin
      aCompilerName:= IniDat.ReadString(cUPL_ComponentCode, IntToStr(i+1), '');
      aCompilerID:= System.Copy(aCompilerName, 1, System.Pos(',', aCompilerName)-1 );
      System.Delete(aCompilerName, 1, System.Pos(',', aCompilerName));
      FList.Add(CreateCompiler( aCompilerName, aCompilerID));
    End;
  finally
    IniDat.Destroy;
  End;
End;

{$IFDEF DSP_DATABASE}
procedure TDSPCompilerBaseList.BuildFromTable(Table: TTable);
Begin
  if not Table.Active then Table.Open;
  Table.IndexName:= 'byOrder';
  Table.First;
  while not Table.EOF do
  Begin
    FList.Add(CreateCompiler(Table.FieldByName('Compilername').AsString,
                             Table.FieldByName('CompilerID').AsString ));
    Table.Next;
  End;
End;
{$ENDIF}

{$IFDEF DSP_DATABASE}
procedure TDSPCompilerBaseList.BuildFromDM(DM: TDataModule);

  function LocateTable: TTable;
  var
    i: integer;
  const
    TableName = 'DSPComp.db';
  begin
    For i:= 0 to DM.ComponentCount - 1 do
      if ( DM.Components[i] is TTable ) and
         ( CompareText(TTable(DM.Components[i]).TableName, TableName) = 0 ) then
      Begin
        result:= TTable(DM.Components[i]);
        Exit;
      End;
    raise Exception.Create('No table ' + TableName + ' in DataModule!!');
  End;
Begin
  BuildFromTable(LocateTable);
end;
{$ENDIF}

{$IFDEF DSP_DATABASE}
procedure TDSPCompilerBaseList.ReadFromTable(aTblFiles: TTable);
var
  i: integer;
Begin
  for i:= 0 to FList.Count - 1 do
     TDSPCompilerBase(FList.Items[i]).ReadFromTable(aTblFiles);
End;
{$ENDIF}

function TDSPCompilerBaseList.GetNameFromID(aCompilerID: String): string;
var
  i: integer;
Begin
  for i:= 0 to FList.Count - 1 do
  Begin
    if CompareStr(TDSPCompilerBase(FList.Items[i]).CompilerID, aCompilerID) = 0 then
    Begin
      result:= TDSPCompilerBase(FList.Items[i]).CompilerName;
      exit;
    End;
  End;
  raise Exception.Create('Undefined copiler ID : ' + aCompilerID);
End;

procedure TDSPCompilerBaseList.ReadFromIni(Ini: TIniFile);
var
  i: integer;
Begin
  for i:= 0 to FList.Count - 1 do
    TDSPCompilerBase(FList.Items[i]).ReadFromIni(Ini);
End;

procedure TDSPCompilerBaseList.WriteToIni(Ini: TIniFile);
var
  i: integer;
Begin
  for i:= 0 to FList.Count - 1 do
    TDSPCompilerBase(FList.Items[i]).WriteToIni(Ini);
End;

procedure TDSPCompilerBaseList.ReadFromInf(Ini: TIniFile);

   procedure ProcessOne(Compiler: TDSPCompilerBase; Key: string);
   Begin
      Compiler.Supported:= Ini.ReadString( cUPL_FileSection, Key, cUPL_No)[1] = cUPL_YES ;
      if Compiler.Supported then Compiler.Available:= ''
               else Compiler.Available:= Ini.ReadString( cUPL_FileAvailableSection, Key, '');
   End;


Begin
  // silly me - I have no idea why I've changed compilers keys :(
  ProcessOne(GetCompilerFromID('d10'), cINF_d10);
  ProcessOne(GetCompilerFromID('d20'), cINF_d20);
  ProcessOne(GetCompilerFromID('d30'), cINF_d30);
  ProcessOne(GetCompilerFromID('c10'), cINF_c10);
  // these two were abasent in InfBuilder but you add them manually
  ProcessOne(GetCompilerFromID('d40'), cINF_d40);
  ProcessOne(GetCompilerFromID('c30'), cINF_c30);
  // just in case
  ProcessOne(GetCompilerFromID('j10'), cINF_j10);
End;



function TDSPCompilerBaseList.GetBaseCompilerID: string;
var
  i: integer;
Begin
  // to run this correctly, items have to be in the rigth order d10->d40->c10->c30
  for i:= 0 to FList.Count - 1 do
  Begin
    if TDSPCompilerBase(FList.Items[i]).Supported then
    Begin
      result:= TDSPCompilerBase(FList.Items[i]).CompilerID;
      exit;
    End;
  End;
  result:= '';
End;

function TDSPCompilerBaseList.GetCompiler(Index: Integer): TDSPCompilerBase;
begin
  if (Index > FList.Count - 1) or ( Index < 0 )  then
    raise Exception.Create('DSP compilers: Index out of range');
  result:= TDSPCompilerBase(FList.Items[Index]);
end;

function TDSPCompilerBaseList.GetCompilerFromID( aCompilerID: String): TDSPCompilerBase;
var
  i: integer;
Begin
  for i:= 0 to FList.Count - 1 do
  Begin
    if CompareStr(TDSPCompilerBase(FList.Items[i]).CompilerID, aCompilerID) = 0 then
    Begin
      result:= TDSPCompilerBase(FList.Items[i]);
      exit;
    End;
  End;
  raise Exception.Create('Undefined copiler ID : ' + aCompilerID);
End;


(******************************************************************************)
function TDSPCompilerList.CreateCompiler( aCompilerName, aCompilerID: String): TDSPCompilerBase;
begin
  result:= TDSPCompiler.Create(aCompilerName, aCompilerID);
end;


(******************************************************************************)
Constructor TDSPInfoBaseClass.Create;
Begin
  Inherited Create;
  CreateAuthors;
  CreateCompilers;
  FUploadDesc:= TStringList.Create;
  // groups we can create directly
  FGroups:= TDSPFileGroupList.Create;
  FPlatforms:= TDSPFilePlatformList.Create;
  Reset;
End;

{$IFDEF DSP_DATABASE}
procedure TDSPInfoBaseClass.BuildFromDM(DM: TDataModule);
Begin
  Compilers.BuildFromDM(DM);
  Groups.BuildFromDM(DM);
  Platforms.BuildFromDM(DM);
end;
{$ENDIF}


procedure  TDSPInfoBaseClass.CreateAuthors;
Begin
  FAuthors:= TDSPAuthorList.Create;
End;

procedure  TDSPInfoBaseClass.CreateCompilers;
Begin
  FCompilers:= TDSPCompilerBaseList.Create;
End;

Destructor TDSPInfoBaseClass.Destroy;
Begin
  FAuthors.Free;
  FCompilers.Free;
  FUploadDesc.Free;
  FGroups.Free;
  FPlatforms.Free;
  inherited Destroy;
End;

procedure  TDSPInfoBaseClass.Reset;
Begin
{$IFDEF DSP_DATABASE}
  FFileID:= -1;
{$ENDIF}
  FAuthors.Reset;
  FCompilers.Reset;
  FFileName:= '';
  FReplaceFile:= '';
  FUploadDesc.Clear;
  FNote:= '';
  FExtDescFile:= '';
  FFileVersion:= '';
  FFreeware:= true;
  FWithSource:= False;
  FSupport:= False;
  FCategory:= '';
  FFileGroup:= '';
  FFilePlatform:='';
  FFileDir:= '';
  FUpdateDSP:= true;
  FFreeware:= true;
  FFileGroup:= '';
End;

procedure TDSPInfoBaseClass.ReadFromIniFile(IniName: String);
var
  IniFile: TIniFile;
  s: string;
Begin
{$IFDEF DSP_DATABASE}
  FFileID:= -1;
{$ENDIF}
  IniFile := TInifile.Create(IniName);
  try
    FAuthors.ReadFromIni(IniFile);
    FCompilers.ReadFromIni(IniFile);
    with IniFile do
    begin
      FFileName:= ReadString(cUPL_FileSection, cUPL_FileName, '');
      FReplaceFile := ReadString(cUPL_FileSection, cUPL_ReplaceFile,'');
      FFreeware:= CompareText(ReadString(cUPL_FileSection, cUPL_FileStatus, cUPL_Freeware), cUPL_Freeware) = 0;
      FWithSource:= CompareStr(ReadString(cUPL_FileSection, cUPL_FullSource, cUPL_NO), cUPL_YES) = 0;
      FFileVersion := ReadString(cUPL_FileSection, cUPL_FileVersion, '');

      S:= ReadString(cUPL_FileSection, cUPL_Description, '');
      if Length(S) > 0 then FUploadDesc.Add(S)
                       else ReadStringListFromIni(IniFile, cUPL_FileSection, FUploadDesc );
      FNote := ReadString(cUPL_FileSection, cUPL_Note, '');
      FExtDescFile := readString(cUPL_FileSection, cUPL_AdditionalInfo, '');
      FCategory:= ReadString(cUPL_FileSection, cUPL_Category, '');
      FFilePlatform:= ReadString(cUPL_FileSection, cUPL_FilePlatform, '');
      FFileGroup:= ReadString(cUPL_FileSection, cUPL_FileGroup, '');
      FSupport:= CompareStr(ReadString(cUPL_DSPSupport, cUPL_DSPSupport, cUPL_NO), cUPL_YES) = 0;
    End;
  finally
    IniFile.Free;
  end;
  DeduceTargetDir;
End;

procedure TDSPInfoBaseClass.ReadFromInfFile(InfName: String);
var
  IniFile: TIniFile;
Begin
{$IFDEF DSP_DATABASE}
  FFileID:= -1;
{$ENDIF}
  IniFile := TInifile.Create(InfName);
  try
    // it will read only one
    FAuthors.ReadFromIni(IniFile);
    // buuuu, I have changed compilers keywords
    FCompilers.ReadFromInf(IniFile);
    with IniFile do
    begin
      // there was only one group: Component/Code
      FFileGroup:= cUPL_ComponentCode;
      // file name was splitted in two parts: filename itself
      //                                      URL is link was requested
      //   most probbably it will not work as users have been used these fields in a
      //   slightly different way than I expacted :(
      FFileName:= ReadString(cUPL_FileSection, cINF_FileURL, '') +
                  ReadString(cUPL_FileSection, cINF_FileName, '');
      FReplaceFile := ReadString(cUPL_FileSection, cINF_ReplaceFile,'');
      // file status is managed the same way
      FFreeware:= CompareText(ReadString(cUPL_FileSection, cUPL_FileStatus, cUPL_Freeware), cUPL_Freeware) = 0;
      // source: now is simple [Y] or [N]
      FWithSource:= CompareStr(ReadString(cUPL_FileSection, cINF_FullSource, cINF_Included), cINF_Included) = 0;
      // file vesrion is managed the same way
      FFileVersion := ReadString(cUPL_FileSection, cUPL_FileVersion, '');
      FUploadDesc.Add(ReadString(cUPL_FileSection, cUPL_Description, ''));
      FNote := ReadString(cUPL_FileSection, cUPL_Note, '');
      FExtDescFile := readString(cUPL_FileSection, cUPL_AdditionalInfo, '');
      FCategory:= ReadString(cUPL_FileSection, cUPL_Category, '');
      FFilePlatform:= ReadString(cUPL_FileSection, cUPL_FilePlatform, '');
      FSupport:= CompareStr(ReadString(cUPL_DSPSupport, cUPL_DSPSupport, cUPL_NO), cUPL_YES) = 0;
    End;
  finally
    IniFile.Free;
  end;
  DeduceTargetDir;
End;

procedure TDSPInfoBaseClass.WriteToIniFile(IniName: String);
var
  IniFile: TIniFile;
Begin
  IniFile := TInifile.Create(IniName);
  try
    with IniFile do
    begin
      WriteString(cUPL_FileSection, cUPL_FileName, FFileName);
      WriteString(cUPL_FileSection, cUPL_ReplaceFile, FReplaceFile);
      if FFreeware then WriteString(cUPL_FileSection, cUPL_FileStatus, cUPL_Freeware)
                   else WriteString(cUPL_FileSection, cUPL_FileStatus, cUPL_Shareware);
      if FWithSource then WriteString(cUPL_FileSection, cUPL_FullSource, cUPL_YES)
                     else WriteString(cUPL_FileSection, cUPL_FullSource, cUPL_NO);
      WriteString(cUPL_FileSection, cUPL_FileVersion, FFileVersion);
      WriteStringListToIni(IniFile, cUPL_FileSection, FUploadDesc );
      WriteString(cUPL_FileSection, cUPL_Note, FNote);
      WriteString(cUPL_FileSection, cUPL_AdditionalInfo, FExtDescFile);
      WriteString(cUPL_FileSection, cUPL_Category, FCategory);
      WriteString(cUPL_FileSection, cUPL_FilePlatform, FFilePlatform);
      WriteString(cUPL_FileSection, cUPL_FileGroup, FFileGroup);
      if FSupport then WriteString(cUPL_DSPSupport, cUPL_DSPSupport, cUPL_YES)
                  else WriteString(cUPL_DSPSupport, cUPL_DSPSupport, cUPL_NO);
      FAuthors.WriteToIni(IniFile);
      FCompilers.WriteToIni(IniFile);
    End;
  finally
    IniFile.Free;
  end;
End;

procedure TDSPInfoBaseClass.DeduceTargetDir;
Begin
  // check that base compiler is OK
  if CompareStr( FFileGroup, cUPL_ComponentCode ) = 0 then
  Begin
    if FFreeware
      then FFileDir:= Compilers.GetBaseCompilerID + 'free'
      else FFileDir:= Compilers.GetBaseCompilerID + 'share';
  End else FFileDir:= Groups.GetGroupDir( FFileGroup );
End;


(******************************************************************************)
procedure TDSPInfoClass.CreateCompilers;
Begin
  FCompilers:= TDSPCompilerList.Create;
End;



end.

