unit KA.Utilities.DriveLetter;

interface

uses
  Windows, SysUtils, Classes, System.Text;

type
  ActionType=(DoAssign,DoRemove,DoChange);
  TKADriveLetter = class(TComponent)
  private
    FActivate  : Boolean;
    FOldLetter : String;
    FPartition : String;
    FNewLetter : String;
    FDrives    : TStringList;
    FVolumes   : TStringList;
    FAction    : ActionType;
    procedure SetAction(const Value: ActionType);
    procedure SetActivate(const Value: Boolean);
    procedure SetNewLetter(const Value: String);
    procedure SetOldLetter(const Value: String);
    procedure SetPartition(const Value: String);
    procedure SetDrives(const Value: TStringList);
    procedure SetVolumes(const Value: TStringList);
    { Private declarations }
  protected
    { Protected declarations }
    Function  DriveExists(Drive : String):Boolean;
    Function  CorrectDriveLetter(Drive : String):String;
    Procedure EnumDrives;
    Procedure EnumVolumes;
    Function  GetDriveVolumeName(Drive : String):String;
    Function  RemoveDriveLetter(Drive : String):Boolean;
    Function  AssignDriveLetter(Drive : String; NTDevice : String):Boolean;
    Function  ChangeDriveLetter:Boolean;
  public
    { Public declarations }
    Constructor Create(AOwner:TComponent); Override;
    Destructor  Destroy; Override;
  published
    function GetVolumes: TStringList;
    function GetDrives: TStringList;
    { Published declarations }
    Property Action    : ActionType  read FAction    write SetAction;
    Property OldDrive  : String      read FOldLetter write SetOldLetter;
    Property NewDrive  : String      read FNewLetter write SetNewLetter;
    Property Partition : String      read FPartition write SetPartition;
    Property Drives    : TStringList read GetDrives  write SetDrives;
    Property Volumes   : TStringList read GetVolumes write SetVolumes;
    Property Activate  : Boolean     read FActivate  write SetActivate;
  end;

procedure Register;

implementation
{$R 'KA.Utilities.DriveLetter.TKADriveLetter.bmp'}
Uses
 System.Runtime.InteropServices;

const
  kernel32  = 'kernel32.dll';

[DllImport(kernel32, CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, SetLastError = True, EntryPoint = 'FindVolumeClose')]
Procedure FindVolumeClose(FH:THandle); external;

[DllImport(kernel32, CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, SetLastError = True, EntryPoint = 'FindFirstVolumeW')]
Function FindFirstVolume(lpszVolumeName: StringBuilder; cchBufferLength : DWORD):THANDLE;external;

[DllImport(kernel32, CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, SetLastError = True, EntryPoint = 'FindNextVolumeW')]
Function FindNextVolume(hFindVolume:THANDLE; lpszVolumeName:StringBuilder; cchBufferLength:DWORD):BOOL;external;

[DllImport(kernel32, CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, SetLastError = True, EntryPoint = 'DeleteVolumeMountPointW')]
Function DeleteVolumeMountPoint(lpszVolumeMountPoint:String):BOOL;external;

[DllImport(kernel32, CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, SetLastError = True, EntryPoint = 'GetVolumeNameForVolumeMountPointW')]
Function GetVolumeNameForVolumeMountPoint(lpszVolumeMountPoint:String; [OUT]lpszVolumeName : StringBuilder; cchBufferLength : DWORD):BOOL;external;

[DllImport(kernel32, CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, SetLastError = True, EntryPoint = 'SetVolumeMountPointW')]
Function SetVolumeMountPoint(lpszVolumeMountPoint:String; lpszVolumeName:String):BOOL;external;




procedure Register;
begin
  RegisterComponents('KA', [TKADriveLetter]);
end;

{ TKADriveLetter }
Function TKADriveLetter.DriveExists(Drive : String):Boolean;
Begin
  Result := GetDriveType(Drive+'\') <> 1;
End;

Function TKADriveLetter.CorrectDriveLetter(Drive : String):String;
Var
  L : Integer;
Begin
  Result := '';
  L      := Length(Drive);
  if L<2 Then Exit;
  Result :=Drive ;
  if L > 3 Then Delete(Result,1,2);
  L := 2;
  if Result[L] <> ':' Then
     Begin
       Result := '';
       Exit;
     End;
  if (NOT (Result[1] in ['A'..'Z'])) AND (NOT (Result[1] in ['a'..'z'])) Then
     Begin
       Result := '';
       Exit;
     End;
End;


Procedure TKADriveLetter.EnumVolumes;
Var
 FH                : THandle;
 PVolumeName       : StringBuilder;
Begin
 FVolumes.Clear;
 PVolumeName := StringBuilder.Create(MAX_PATH);
 FH := FindFirstVolume(PVolumeName, MAX_PATH);
 if FH <> INVALID_HANDLE_VALUE Then
    Begin
       FVolumes.Add(PVolumeName.ToString);
       While FindNextVolume(FH, PVolumeName, MAX_PATH) Do
          Begin
            FVolumes.Add(PVolumeName.ToString);
          End;
       FindVolumeClose(FH);
    End;
End;

Procedure TKADriveLetter.EnumDrives;
Var
  X                                : Integer;
  Y                                : Integer;
  Drive                            : String;
  Bytes                            : Array[0..MAX_PATH] of Byte;
  UniqueVolumeName                 : String;
Begin
  FDrives.Clear;
  For X := Ord('A') To Ord('Z') do
      Begin
        Drive := CHR(X)+':';
        UniqueVolumeName := '';
        if GetDriveType(Drive+'\') <> 1 Then
           Begin
              For Y := 0 To MAX_PATH Do Bytes[Y] := 0;
              QueryDosDevice(Drive,Bytes,MAX_PATH);
              For Y := 0 To MAX_PATH Do
                  Begin
                    if Bytes[Y] <> 0 Then
                       Begin
                         UniqueVolumeName := UniqueVolumeName+CHR(Bytes[Y]);
                       End;
                  End;
              FDrives.Add(Drive+'='+UniqueVolumeName);
           End;
      End;
End;

Function TKADriveLetter.GetDriveVolumeName(Drive : String):String;
Var
  I : Integer;
Begin
  Result := '';
  I := Drives.IndexOfName(Drive);
  if I <> -1 Then
     Begin
       Result := Drives.Values[Drives.Names[I]];
     End;
End;

Function TKADriveLetter.RemoveDriveLetter(Drive : String):Boolean;
Begin
  Result    := False;
  if CorrectDriveLetter(Drive) = '' Then Exit;
  if Drives.IndexOfName(Drive)=-1 Then Exit;
  Result := DeleteVolumeMountPoint(Drive+'\');
End;

Function TKADriveLetter.AssignDriveLetter(Drive : String; NTDevice : String):Boolean;
Var
  UniqueVolumeName : StringBuilder;
Begin
  Result := False;
  if CorrectDriveLetter(Drive) = '' Then Exit;
  Result := DefineDosDevice(DDD_RAW_TARGET_PATH, Drive, NTDevice);
  UniqueVolumeName := StringBuilder.Create(MAX_PATH);
  If Result Then
     Begin
       if (NOT GetVolumeNameForVolumeMountPoint(Drive+'\', UniqueVolumeName, MAX_PATH)) Then
          Begin
            UniqueVolumeName[0] := #0;
          End;
       Result := DefineDosDevice(DDD_RAW_TARGET_PATH OR DDD_REMOVE_DEFINITION OR DDD_EXACT_MATCH_ON_REMOVE, Drive, NTDevice);
       if Result Then
          Begin
            Result := SetVolumeMountPoint(Drive+'\',UniqueVolumeName.ToString);
          End;
     End;
End;

function TKADriveLetter.ChangeDriveLetter: Boolean;
Var
  TempPartition : String;
begin
 Result := False;
 if (FOldLetter <> '') AND (FNewLetter <> '') Then
    Begin
      TempPartition := GetDriveVolumeName(FOldLetter);
      Result        := RemoveDriveLetter(FOldLetter);
      EnumDrives;
      EnumVolumes;
      if Result Then Result  := AssignDriveLetter(FNewLetter,TempPartition);
    End;
end;

constructor TKADriveLetter.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAction := DoAssign;
  FDrives  := TStringList.Create;
  FVolumes := TStringList.Create;
  EnumDrives;
  EnumVolumes;
end;

destructor TKADriveLetter.Destroy;
begin
  FDrives.Free;
  FVolumes.Free;
  inherited;
end;


procedure TKADriveLetter.SetAction(const Value: ActionType);
begin
  FAction := Value;
end;

procedure TKADriveLetter.SetActivate(const Value: Boolean);
begin
  if csLoading in ComponentState Then Exit;
  FActivate := Value;
  if FActivate Then
     Begin
      if FAction=DoAssign Then
         Begin
           FActivate := AssignDriveLetter(FNewLetter,FPartition);
         End
      Else
      if FAction=DoRemove Then
         Begin
           FActivate := RemoveDriveLetter(FOldLetter);
         End
      Else
      if FAction=DoChange Then
         Begin
           FActivate := ChangeDriveLetter;
         End;
      EnumDrives;
      EnumVolumes;
     End;
end;

procedure TKADriveLetter.SetNewLetter(const Value: String);
begin
  if (CorrectDriveLetter(Value) <> '') And (NOT DriveExists(Value)) Then
     FNewLetter := Value
  Else
     FNewLetter := '';
end;

procedure TKADriveLetter.SetOldLetter(const Value: String);
begin
  if (CorrectDriveLetter(Value) <> '') And (DriveExists(Value)) Then
     FOldLetter := Value
  Else
     FOldLetter := '';
end;

procedure TKADriveLetter.SetPartition(const Value: String);
begin
  FPartition := Value;
end;

function TKADriveLetter.GetDrives: TStringList;
begin
 EnumDrives;
 EnumVolumes;
 Result := FDrives;
end;

function TKADriveLetter.GetVolumes: TStringList;
begin
 EnumDrives;
 EnumVolumes;
 Result := FVolumes;
end;


procedure TKADriveLetter.SetDrives(const Value: TStringList);
begin
  //****************************************************************** READ ONLY
end;

procedure TKADriveLetter.SetVolumes(const Value: TStringList);
begin
  //****************************************************************** READ ONLY
end;






end.
