unit KA.Data.KADao.DataBase platform;
{$B-}
{$DEFINE USEDB}
interface
Uses
  Borland.VCL.Windows,
  Borland.VCL.SysUtils,
  Borland.VCL.Classes,
  Borland.VCL.FileCtrl,
  Borland.VCL.Registry,
  Borland.VCL.TypInfo,
  Borland.VCL.Db,
  Borland.VCL.Variants,
  KA.Data.Dao360,
  KA.Data.KADao.KAOle,
  KA.Data.KADao.DaoApi,
  KA.Data.KADao.DaoUtils,
  KA.Data.KADao.DummyDataset,
  KA.Data.KADao.DbLoginUnit;

//******************************************************* DatabaseError Messages
{$I 'KA.Data.KADao.ErrLangDB.pas'}
//******************************************************************************

Const
  szUSERNAME   = 'USER NAME';
  szPASSWORD   = 'PASSWORD';
  szDBPASSWORD = 'DBPASSWORD';
Type
 TKADaoDatabase=Class;

 TDaoErrRec=Record
            ErrNo       : Integer;
            Source      : String;
            Description : String;
            HelpFile    : String;
            HelpContext : Integer;
          End;
 PDaoErrRec=^TDaoErrRec;

 TLoginEvent    = procedure(Database: TKADaoDatabase; LoginParams: TStrings) of object;
 TConnectEvent  = procedure(Database: TKADaoDatabase) of object;

 TKADaoDatabase = Class(TComponent)
       Private
         FDatabase           : String;
         FDatabaseParameters : String;
         FSmartOpen          : Boolean;
         FEngineType         : Integer;
         FPrivateEngine      : Boolean;
         FDatabaseType       : String;
         FWorkspace          : String;
         FCollatingOrder     : String;
         FDaoVersion         : String;
         FActualDaoVersion   : String;
         FDatabaseVersion    : String;
         FSystemDB           : String;
         FActive             : Boolean;
         FReadOnly           : Boolean;
         FExclusive          : Boolean;
         FLoginPrompt        : Boolean;
         FUsername           : String;
         FPassword           : String;
         FDatabasePassword   : String;
         FSaveUserName       : Boolean;
         FMachineName        : String;
         FQueryTimeout       : Integer;
         FLoginDialog        : TDbLogin;
         FTableNames         : TStringList;
         FActiveTableNames   : TStringList;
         FQueryDefNames      : TStringList;
         FDriverList         : TStringList;
         FSystemDSNs         : TStringList;
         FUserDSNs           : TStringList;
         FDSNFileNames       : TStringList;
         FParams             : TStringList;
         FOnLogin            : TLoginEvent;
         FBeforeConnect      : TConnectEvent;
         FAfterConnect       : TConnectEvent;
         FBeforeDisconnect   : TConnectEvent;
         FAfterDisconnect    : TConnectEvent;
         FOffline            : Boolean;
         FShowSysObjects     : Boolean;

         FTransInfo          : TStringList;
         FTrackTransactions  : Boolean;
         FKAOle              : TKAOle;

         FComponentVersion   : String;
         FDefaultCursorDriver: Integer;
         FUseODBCDialog      : Boolean;

         procedure GetDBTypesList(List: TStrings);
         Function  GetDBTypeFileExtension(DBType:String):String;
         Function  GetDBTypeTableType(DBType:String):String;
         Function  GetODBCFileName(DSN:String;SystemWideDSN:Boolean):String;
         Function  GetTableRN(Tables:String;TableName:String):String;
         procedure FFillDSNFileNames(List: TStrings);
         procedure GetOdbcDriverList(List: TStrings);
         procedure GetSystemDSNs(DSNs: TStrings);
         procedure GetUserDSNs(DSNs: TStrings);
         Procedure SetDaoVersion(Value : String);
         Procedure SetActualDaoVersion(Value : String);
         Procedure SetDatabaseVersion(Value : String);
         Procedure SetDatabase(Value : String);
         Procedure SetDatabaseParameters(Value : String);
         Function  GetSystemDatabaseFromRegistry:String;
         Procedure SetSystemDatabase(Value : String);
         Procedure SetWorkspace(Value : String);
         Function  GetDatabaseType:String;
         Procedure SetDatabaseType(Value : String);
         Function  GetCollatingOrder:String;
         Procedure SetEngineType(Value : Integer);
         Procedure SetPrivateEngine(Value : Boolean);
         Procedure SetShowSysObjects(Value : Boolean);
         Procedure SetUserName(Value : String);
         Procedure SetPassword(Value : String);
         Procedure SetDatabasePassword(Value : String);
         Procedure SetExclusive(Value : Boolean);
         Procedure SetLoginPrompt(Value : Boolean);
         Procedure SetReadOnly(Value : Boolean);
         Procedure SetComponentVersion(Value: String);
         Procedure SetParams(Value : TStringList);
         Procedure SetDefaultCursorDriver(Value : Integer);
         Procedure SetActive(Value : Boolean);
         Procedure SetTrackTransactions(Value : Boolean);

       Protected
         Procedure                     CreateDBEngine;
         Procedure                     Loaded; override;
       Public
         //********************************* Public for Property Editors request
         FDBTypesList               : TStringList;
         //*********************************************************************
         DatabaseLanguageInt         : Integer;
         CoreDBEngine                : DBEngine;
         CoreDatabase                : Database;
         CoreWorkspace               : Workspace;
         Property    Params          : TStringList Read FParams Write SetParams;
         Property    DSNFileNames    : TStringList Read FDSNFileNames;
         Property    QueryDefNames   : TStringList Read FQueryDefNames;
         Property    TableNames      : TStringList Read FTableNames;
         Property    ActiveTableNames: TStringList Read FActiveTableNames;
         Property    DatabaseTypes   : TStringList Read FDBTypesList;
         Property    ComWrapper      : TKAOle      Read FKAOle;

         //********************************* Public for Property Editors request
         Function  FChooseDatabase  : String;
         //*********************************************************************
         Function                      CreateOleDBEngine: DBEngine;
         Function                      GetLastDaoError:TDaoErrRec;
         Constructor                   Create(AOwner : TComponent); override;
         Destructor                    Destroy; override;

         //****************************************************** Online/Offline
         Procedure   GoOffline;
         Procedure   GoOnline;
         //****************************************************** Transactions
         Procedure                   StartTransaction;
         Procedure                   Commit;
         Procedure                   Rollback;
         Procedure                   RollbackRefresh;
         Procedure                   AddRNToTransaction(TableName : String;RN:Integer);

         Procedure                   DBEngineLevel_StartTransaction;
         Procedure                   DBEngineLevel_Commit;
         Procedure                   DBEngineLevel_Rollback;

         Procedure                   WorkspaceLevel_StartTransaction;
         Procedure                   WorkspaceLevel_Commit;
         Procedure                   WorkspaceLevel_Rollback;

         Function                    GetTransactionCount:Integer;

         //****************************************************** Utils
         Procedure                   RepairAccessDatabase  (DatabaseName,Password:String);
         Procedure                   RepairAccessDatabaseEx(DatabaseName : String;
                                                            NewLocale    : String;
                                                            Encrypt      : Boolean;
                                                            Decrypt      : Boolean;
                                                            NewVersion   : Integer;
                                                            Password     : String);
         Procedure                   CompactAccessDatabase  (DatabaseName,Password:String);
         Procedure                   CompactAccessDatabaseEx(DatabaseName: String;
                                                             NewLocale   : String;
                                                             Encrypt     : Boolean;
                                                             Decrypt     : Boolean;
                                                             NewVersion  : Integer;
                                                             Password    : String);

         Procedure                   CreateAccessDatabase    (DatabaseName:String);
         Procedure                   CreateAccessDatabaseEx  (DatabaseName,LANGID,CP,COUNTRY,Password,Version:String;Encrypt:Boolean);
         Procedure                   CreateAccessDatabaseEx2 (DatabaseName,Language,Password,Version:String;Encrypt:Boolean);
         //****************************************************** Utils II
         Function                    ChangeDatabasePassword(OldPassword,NewPassword:String):Boolean;
         Function                    RegisterDatabase(DatabaseName, DriverName:String; Silent:Boolean; Attributes:String):Boolean;
         Function                    CreateEmptyTable(TableName:String):Boolean;
         Function                    CreateTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant; FieldsRequired:Variant):Boolean;
         Function                    AddFieldsToTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant; FieldsRequired:Variant):Boolean;
         Procedure                   LinkExternalTable(Database,TableName,TableType:String;TableAttributes:Integer);
         Procedure                   LinkExternalTableEx(Database,TableName,TableFileName,TableType:String;TableAttributes:Integer);
         Procedure                   RefreshLink(Database,TableName,TableType:String);

         Procedure                   RenameTable(OldTableName,NewTableName:String);
         Function                    EmptyTable(TableName:String):Boolean;
         Procedure                   DeleteTable(TableName:String);


         Function                    HasPrimaryKey(NewTable:TableDef):Boolean;
         Function                    TablePrimaryKeyName(NewTable:TableDef):String;
         Procedure                   DeletePrimaryKey(NewTable:TableDef);

         Function                    CreateIndex(TableName,FieldName:String;IndexType:Integer):Boolean;
         Procedure                   RenameIndex(TableName,OldIndexName,NewIndexName:String);
         Procedure                   DeleteIndexByName(TableName,IndexName:String);
         Procedure                   DeleteIndexByFieldName(TableName,FieldName:String);

         Procedure                   RenameField(TableName,OldFieldName,NewFieldName:String);
         Procedure                   DeleteField(TableName,FieldName:String);



         Function                    CreateQueryDef(Name:String;SQL:String):Boolean;
         Procedure                   ModifyQueryDef(Name:String;SQL:String);
         Function                    GetQueryDefSQLText(Name:String):String;
         Procedure                   RenameQueryDef(OldQueryName,NewQueryName:String);
         Procedure                   DeleteQueryDef(QueryName:String);

         Function                    FindWorkspace(WS:String):Boolean;
         Procedure                   RefreshDefinitions;
         Procedure                   Idle;

         Procedure                   Open;
         Procedure                   Close;
         Procedure                   CloseDatasets;
         Procedure                   RefreshDatasets;
         Function                    ChooseDatabase: Boolean;

         Procedure                   RecreateCore;
      Published
         Property ComponentVersion     : String  Read FComponentVersion Write SetComponentVersion;
         Property Exclusive            : Boolean Read FExclusive Write SetExclusive;
         Property DatabaseLanguage     : String  Read GetCollatingOrder Write FCollatingOrder;
         Property DatabaseType         : String  Read GetDatabaseType Write SetDatabaseType;
         Property Database             : String  Read FDatabase Write SetDatabase;
         Property DatabaseParameters   : String  Read FDatabaseParameters Write SetDatabaseParameters;
         Property DatabaseVersionInfo  : String  Read FDatabaseVersion   Write SetDatabaseVersion;
         Property ReadOnly             : Boolean Read FReadOnly Write SetReadOnly;
         Property LoginPrompt          : Boolean Read FLoginPrompt Write SetLoginPrompt;
         Property UserName             : String  Read FUsername  Write SetUserName;
         Property UseODBCDialog        : Boolean Read FUseODBCDialog  Write FUseODBCDialog;
         Property Password             : String  Read FPassword  Write SetPassword;
         Property DatabasePassword     : String  Read FDatabasePassword Write SetDatabasePassword;
         Property SystemDatabase       : String  Read FSystemDB Write SetSystemDatabase;
         Property SaveUserName         : Boolean Read FSaveUserName Write FSaveUserName;
         Property ShowSystemObjects    : Boolean Read FShowSysObjects Write SetShowSysObjects;
         Property SmartOpen            : Boolean Read FSmartOpen Write FSmartOpen;
         Property EngineType           : Integer Read FEngineType Write SetEngineType;
         Property PrivateEngine        : Boolean Read FPrivateEngine Write SetPrivateEngine;
         Property TrackTransactions    : Boolean Read FTrackTransactions Write SetTrackTransactions;
         Property Version              : String  Read FDaoVersion Write SetDaoVersion;
         Property VersionDetails       : String  Read FActualDaoVersion Write SetActualDaoVersion;
         Property Workspace            : String  Read FWorkspace Write SetWorkspace;
         Property DefaultCursorDriver  : Integer Read FDefaultCursorDriver Write SetDefaultCursorDriver;
         Property QueryTimeout         : Integer Read FQueryTimeout Write FQueryTimeout;
         Property OnLogin              : TLoginEvent   Read FOnLogin Write FOnLogin;
         Property OnBeforeConnect      : TConnectEvent Read FBeforeConnect Write FBeforeConnect;
         Property OnAfterConnect       : TConnectEvent Read FAfterConnect Write FAfterConnect;
         Property OnBeforeDisconnect   : TConnectEvent Read FBeforeDisconnect Write FBeforeDisconnect;
         Property OnAfterDisconnect    : TConnectEvent Read FAfterDisconnect Write FAfterDisconnect;
         Property Connected            : Boolean       Read FActive Write SetActive Default False;
      End;

TKADaoTableManager = Class(TObject)
      Private
         FDatabase      : TKADaoDatabase;
         FDummyDataset  : TDummyDataset;
         Function          CheckStatus:Boolean;
         Procedure         StringToList(Items: String; List: TStringList);
      Public
         FieldDefs   : TFieldDefs;
         IndexDefs   : TIndexDefs;
         TableName   : String;
         Procedure   CreateTable;
         Procedure   AppendTable;
         Procedure   CreateIndex(PreservePrimaryKeys:Boolean);
         Constructor Create(Database : TKADaoDatabase);
         Destructor  Destroy;override;
      End;

Procedure Register;


//*************************************************************************************************
implementation
Uses
  System.Text,
  Borland.Vcl.Dialogs,
  Borland.Vcl.Forms,
  Borland.Vcl.ActiveX,
  KA.Data.KADao.ODBCDialogUnit
  {$IFDEF USEDB},
  KA.Data.KADao.Table
  {$ENDIF};

Const
  dbLangGeneral = ';LANGID=%s;CP=%s;COUNTRY=%s';

//******************************************************************************

function GetExeDir: String;
Var
  SB : StringBuilder;
begin
  SB := StringBuilder.Create(1000);
  Try
    GetModuleFileName(0,SB,1000);
    Result := ExtractFilePath(SB.ToString);
  Finally
    SB.Free;
  End;
end;

function GetWorkDir: String;
begin
     GetDir(0, Result);
     if Result[Length(Result)] <> '\' Then Result:=Result+'\';
end;

Function  TKADaoDatabase.GetLastDaoError:TDaoErrRec;
Var
  ItemNo : TObject;
Begin
  Result.ErrNo         := 0;
  Result.Source        := '';
  Result.Description   := '';
  Result.HelpFile      := '';
  Result.HelpContext   := 0;
  ItemNo               := TObject(0);
  if CoreDBEngine=NIL Then Exit;
  if CoreDBEngine.Errors.Count=0 Then Exit;
  Result.ErrNo       := CoreDBEngine.Errors.Item[ItemNo].Number;
  Result.Source      := CoreDBEngine.Errors.Item[ItemNo].Source;
  Result.Description := CoreDBEngine.Errors.Item[ItemNo].Description;
  Result.HelpFile    := CoreDBEngine.Errors.Item[ItemNo].HelpFile;
  Result.HelpContext := CoreDBEngine.Errors.Item[ItemNo].HelpContext;
End;

Function TKADaoDatabase.CreateOleDBEngine: DBEngine;
Begin
  if (FPrivateEngine) or (csDesigning in ComponentState) Then
     Begin
       Result           := DBEngine(PrivDBEngineClass.Create);
       FKAOle           := CreateOleObject('DAO.PrivateDBEngine.36');
       FKAOle.ComObject := TObject(Result);
     End
  Else
     Begin
       Result           := DBEngine(DBEngineClass.Create);
       FKAOle           := CreateOleObject('DAO.DBEngine.36');
       FKAOle.ComObject := TObject(Result);
     End;
End;

//*************************************************************************************************
Procedure TKADaoDatabase.CreateDBEngine;
Begin
  CoreDBEngine := CreateOleDBEngine;
  FDaoVersion  := CoreDBEngine.Version;
End;

Constructor TKADaoDatabase.Create(AOwner : TComponent);
Begin
  Inherited Create(AOwner);
  CoInitialize(Nil);
  //*******************************************
  FComponentVersion             := '9.00.NET';
  //********************************************************************* Events
  FOnLogin                      := Nil;
  FBeforeConnect                := Nil;
  FAfterConnect                 := Nil;
  FBeforeDisconnect             := Nil;
  FAfterDisconnect              := Nil;
  //********************************************************************* Events
  CoreWorkspace                 := Nil;
  CoreDatabase                  := Nil;
  CoreDBEngine                  := Nil;
  FOffline                      := False;
  FPrivateEngine                := False;
  //*******************************************
  FKAOle                        := Nil;
  CreateDBEngine;
  //*******************************************
  FSystemDB                     := GetSystemDatabaseFromRegistry;
  if FSystemDB <> '' Then
  CoreDBEngine.SystemDB         := FSystemDB;
  FUsername                     := 'Admin';
  FPassword                     := '';
  FDatabasePassword             := '';
  FDatabaseParameters           := '';
  FSaveUserName                 := True;
  FSmartOpen                    := True;
  CoreDBEngine.DefaultUser      := 'Admin';
  CoreDBEngine.DefaultPassword  := '';
  FEngineType                   := dbUseJet;
  CoreDBEngine.DefaultType      := FEngineType;
  FDefaultCursorDriver          := dbUseDefaultCursor;

  //****************************************************************************
   FWorkspace                   := 'DaoWorkspace';
  //****************************************************************************

  FActualDaoVersion             := CoreDBEngine.Version;
  FDatabaseVersion              := '';
  FMachineName                  := '';
  FDatabaseType                 :='Access';
  FActive                       := False;
  FDatabase                     := '';
  FReadOnly                     := False;
  FExclusive                    := False;
  FLoginPrompt                  := False;
  FShowSysObjects               := False;


  FTableNames                   := TStringList.Create;
  FActiveTableNames             := TStringList.Create;
  FQueryDefNames                := TStringList.Create;
  FDBTypesList                  := TStringList.Create;
  FDriverList                   := TStringList.Create;
  FSystemDSNs                   := TStringList.Create;
  FUserDSNs                     := TStringList.Create;
  FDSNFileNames                 := TStringList.Create;
  FParams                       := TStringList.Create;

  FQueryTimeout                 := 60;

  FTransInfo                    := TStringList.Create;
  FTrackTransactions            := True;
  FUseODBCDialog                := True;

  GetDBTypesList(FDBTypesList);
  GetOdbcDriverList(FDriverList);
  GetSystemDSNs(FSystemDSNs);
  GetUserDSNs(FUserDSNs);
  FFillDSNFileNames(FDSNFileNames);
End;

Destructor  TKADaoDatabase.Destroy;
Begin
 If FActive Then Connected := False;
 FTableNames.Free;
 FActiveTableNames.Free;
 FQueryDefNames.Free;
 FDBTypesList.Free;
 FDriverList.Free;
 FSystemDSNs.Free;
 FUserDSNs.Free;
 FDSNFileNames.Free;
 FParams.Free;
 FTransInfo.Free;
 if CoreWorkspace <> Nil Then CoreWorkspace.Close;
 CoreDatabase  := Nil;
 CoreWorkspace := Nil;
 CoreDBEngine  := Nil;
 if Assigned(FKAOle) Then FKAOle.Free;
 FKAOle        := Nil;
 CoUnInitialize;
 Inherited Destroy;
End;

Procedure TKADaoDatabase.RecreateCore;
Var
  TempPrivate  : Boolean;
Begin
  if FOffline Then Exit;
   If (Assigned(CoreWorkspace)) Then CoreWorkspace.Close;
   CoreWorkspace := Nil;
   CoreDBEngine  := Nil;
   //*************************************************** Borland, Microsoft ...
   TempPrivate:=True;
   if (csDesigning in ComponentState) And (FEngineType=dbUseJet) Then
      Begin
        TempPrivate      := FPrivateEngine;
        FPrivateEngine  := True;
      End;
   CreateDBEngine;
   if (csDesigning in ComponentState) And (FEngineType=dbUseJet) Then FPrivateEngine  := TempPrivate;
   //***************************************************************************
   CoreDBEngine.SystemDB         := FSystemDB;
   FActualDaoVersion             := CoreDBEngine.Version;
   CoreDBEngine.DefaultUser      := FUsername;
   CoreDBEngine.DefaultPassword  := FPassword;
   CoreWorkspace                 := CoreDBEngine.CreateWorkspace(FWorkspace,FUsername,FPassword,TObject(FEngineType));
   CoreDBEngine.Workspaces.Append(CoreWorkspace);
   if FEngineType=dbUseODBC Then
      Begin
       CoreWorkspace.DefaultCursorDriver:=FDefaultCursorDriver;
      End;
   FWorkspace                   := CoreWorkspace.Name;
End;

Procedure TKADaoDatabase.Loaded;
Begin
  Try
    inherited Loaded;
    if Not FActive Then
       Begin
         RecreateCore;
       End;
  Except
  End;
End;

Procedure TKADaoDatabase.SetComponentVersion(Value: String);
Begin
 //******************************************************************** ReadOnly
End;

Procedure TKADaoDatabase.SetParams(Value : TStringList);
Begin
  FParams.Assign(Value);
End;

Procedure TKADaoDatabase.SetDefaultCursorDriver(Value : Integer);
Begin
 FDefaultCursorDriver:=Value;
 if csLoading in ComponentState Then Exit;
 if FOffline Then Exit;
 if FEngineType=dbUseODBC Then
    Begin
      CoreWorkspace.DefaultCursorDriver:=FDefaultCursorDriver;
    End;
End;

Function TKADaoDatabase.GetODBCFileName(DSN:String;SystemWideDSN:Boolean):String;
Var
  Reg : TRegistry;
Begin
  Result:='';
  Reg := TRegistry.Create;
  Reg.Access:=KEY_READ;
  if SystemWideDSN Then
     Reg.RootKey := HKEY_LOCAL_MACHINE
  Else
     Reg.RootKey := HKEY_CURRENT_USER;
  if Reg.OpenKeyReadOnly('SOFTWARE\ODBC\ODBC.INI\'+DSN) then
     Begin
       Result:=Reg.ReadString('DBQ');
       Reg.CloseKey;
     End;
  Reg.Free;
End;

procedure TKADaoDatabase.FFillDSNFileNames(List: TStrings);
Var
  X : Integer;
  S : String;
Begin
  List.Clear;
  For X:=0 to FUserDSNs.Count-1 do
      Begin
       S:=GetODBCFileName(FUserDSNs.Strings[X],False);
       if Length(S) > 0 Then List.Add(FUserDSNs.Strings[X]+'='+S);
      End;
  For X:=0 to FSystemDSNs.Count-1 do
      Begin
       S:=GetODBCFileName(FSystemDSNs.Strings[X],True);
       if Length(S) > 0 Then List.Add(FSystemDSNs.Strings[X]+'='+S);
      End;
End;

procedure TKADaoDatabase.GetOdbcDriverList(List: TStrings);
var
   Reg : TRegistry;
Begin
     Reg := TRegistry.Create;
     Reg.Access:=KEY_READ;
     try
     Begin
          Reg.RootKey := HKEY_LOCAL_MACHINE;
          if Reg.OpenKeyReadOnly('SOFTWARE\ODBC\ODBCINST.INI\ODBC Drivers') then
          Begin
               List.Clear;
               Reg.GetValueNames(List);
               Reg.CloseKey;
          End;
     End;
     finally
          Reg.Free;
     End;
End;

procedure TKADaoDatabase.GetSystemDSNs(DSNs: TStrings);
var
  Reg: TRegistry;
begin
  DSNs.Clear;
  Reg:= TRegistry.Create;
  Reg.Access:=KEY_READ;
  Reg.RootKey:= HKEY_LOCAL_MACHINE;
  if Reg.OpenKeyReadOnly('\SOFTWARE\ODBC\odbc.ini\ODBC Data Sources')  Then
     Begin
       Reg.GetValueNames(DSNs);
       Reg.CloseKey;
     End;
  Reg.Free;
end;

procedure TKADaoDatabase.GetUserDSNs(DSNs: TStrings);
var
  Reg: TRegistry;
begin
  DSNs.Clear;
  Reg:= TRegistry.Create;
  Reg.Access:=KEY_READ;
  Reg.RootKey:= HKEY_CURRENT_USER;
  if Reg.OpenKeyReadOnly('\SOFTWARE\ODBC\odbc.ini\ODBC Data Sources')  Then
     Begin
       Reg.GetValueNames(DSNs);
       Reg.CloseKey;
     End;
  Reg.Free;
end;

procedure TKADaoDatabase.GetDBTypesList(List: TStrings);
var
   Reg : TRegistry;
Begin
     Reg := TRegistry.Create;
     Reg.Access:=KEY_READ;
     try
     Begin
          Reg.RootKey := HKEY_LOCAL_MACHINE;
          if Reg.OpenKeyReadOnly('SOFTWARE\Microsoft\JET\4.0\ISAM Formats') then
             Begin
               List.Clear;
               Reg.GetKeyNames(List);
               Reg.CloseKey;
             End;
     End;
     finally
          Reg.Free;
     End;
    List.Insert(0,'ODBC');
    List.Insert(0,'Access');
End;


Function TKADaoDatabase.GetDBTypeFileExtension(DBType:String):String;
var
   Reg : TRegistry;
Begin
     Reg := TRegistry.Create;
     Reg.Access:=KEY_READ;
     try
     Begin
          Reg.RootKey := HKEY_LOCAL_MACHINE;
          if Reg.OpenKeyReadOnly('SOFTWARE\Microsoft\JET\4.0\ISAM Formats\'+DBType) then
             Begin
               Result:=Reg.ReadString('ExportFilter');
               if Result='' Then Result:=Reg.ReadString('ImportFilter');
               Reg.CloseKey;
             End;
     End;
     finally
          Reg.Free;
     End;
End;

Function TKADaoDatabase.GetDBTypeTableType(DBType:String):String;
var
   Reg : TRegistry;
   BUF  : Array[1..1000] of Byte;
Begin
     Reg := TRegistry.Create;
     Reg.Access:=KEY_READ;
     try
     Begin
          Reg.RootKey := HKEY_LOCAL_MACHINE;
          if Reg.OpenKeyReadOnly('SOFTWARE\Microsoft\JET\4.0\ISAM Formats\'+DBType) then
             Begin
               Reg.ReadBinaryData('OneTablePerFile',BUF,1000);
               Result:=IntToStr(BUF[1]);
               Reg.CloseKey;
             End;
     End;
     finally
          Reg.Free;
     End;
End;

Procedure TKADaoDatabase.SetTrackTransactions(Value : Boolean);
Begin
  if FActive  Then DatabaseError('Cannot set TrackTransactions property when Database is connected!');
  FTrackTransactions := Value;
End;

Procedure TKADaoDatabase.SetActive(Value : Boolean);
Label START;
Var
  Pwd          : String;
  LoginParams  : TStringList;
  BadPassword  : Boolean;
  ExitDB       : Boolean;
Begin
  if (FActive) And (Value) Then Exit;
  if (FDatabase='') And (Value) Then
      Begin
       DatabaseError(E1005);
       Exit;
     End;
  if (FDatabaseType='') And (Value) Then
      Begin
       DatabaseError(E1006);
       Exit;
     End;
  if (FActive) And (NOT Value) Then
     Begin
       if Assigned(FBeforeDisconnect) Then FBeforeDisconnect(Self);
       FTableNames.Clear;
       FQueryDefNames.Clear;
       {$IFDEF USEDB}
       CloseDatasets;
       {$ENDIF}
       FActiveTableNames.Clear;
       CoreDatabase.Close;
       CoreDatabase := Nil;
       FTransInfo.Clear;
       FActive:=False;
       if Assigned(FAfterDisconnect) Then FAfterDisconnect(Self);
     End;
  if (NOT FActive) And (Value) Then
     Begin
        if Assigned(FBeforeConnect) Then FBeforeConnect(Self);
START:
        ExitDB      := False;
        BadPassword := False;
        if FLoginPrompt Then
           Begin
             FLoginDialog   := TDbLogin(TDbLogin.CreateParented(Application.Handle));
             if FSaveUserName Then
                FLoginDialog.UserName.Text    := FUserName
             Else
                FLoginDialog.UserName.Text    := '';
             FLoginDialog.Password.Text    := '';
             FLoginDialog.DbPassword.Text  := '';
             FLoginDialog.DatabaseName.Caption:=FDatabase;
             FLoginDialog.ActiveControl:=FLoginDialog.UserName;
             if Assigned(FOnLogin) Then
                Begin
                  LoginParams  := TStringList.Create;
                  LoginParams.Add(szUSERNAME+'='+FUsername);
                  LoginParams.Add(szPASSWORD+'='+FPassword);
                  LoginParams.Add(szDBPASSWORD+'='+FDatabasePassword);
                  FOnLogin(Self, LoginParams);
                  FUsername:=LoginParams.Values[szUSERNAME];
                  FPassword:=LoginParams.Values[szPASSWORD];
                  FDatabasePassword:=LoginParams.Values[szDBPASSWORD];
                  LoginParams.Free;
                End
             Else
                Begin
                  if (FParams.Count > 0) Then
                     Begin
                      if FParams.IndexOfName(szUSERNAME) <> -1 Then
                         FUsername         := FParams.Values[szUSERNAME];
                      if FParams.IndexOfName(szPASSWORD) <> -1 Then
                         FPassword         := FParams.Values[szPASSWORD];
                      if FParams.IndexOfName(szDBPASSWORD) <> -1 Then
                         FDatabasePassword := FParams.Values[szDBPASSWORD];
                     End
                  Else
                  if FLoginDialog.ShowModal=ID_OK Then
                     Begin
                      FUsername          := FLoginDialog.UserName.Text;
                      FPassword          := FLoginDialog.Password.Text;
                      FDatabasePassword  := FLoginDialog.DbPassword.Text;
                     End
                  Else
                     Begin
                       ShowMessage('If You not enter Username and Password You may not gain access to your data!');
                       FUsername         := '';
                       FPassword         := '';
                       FDatabasePassword := '';
                       ExitDB:=True;
                     End;
                End;
             FLoginDialog.Free;
           End
        Else
           Begin
             if Assigned(FOnLogin) Then
                Begin
                  LoginParams  := TStringList.Create;
                  LoginParams.Add(szUSERNAME+'='+FUsername);
                  LoginParams.Add(szPASSWORD+'='+FPassword);
                  LoginParams.Add(szDBPASSWORD+'='+FDatabasePassword);
                  FOnLogin(Self, LoginParams);
                  FUsername:=LoginParams.Values[szUSERNAME];
                  FPassword:=LoginParams.Values[szPASSWORD];
                  FDatabasePassword:=LoginParams.Values[szDBPASSWORD];
                  LoginParams.Free;
                End
             Else
                Begin
                  if (FParams.Count > 0) Then
                     Begin
                      if FParams.IndexOfName(szUSERNAME) <> -1 Then
                         FUsername         := FParams.Values[szUSERNAME];
                      if FParams.IndexOfName(szPASSWORD) <> -1 Then
                         FPassword         := FParams.Values[szPASSWORD];
                      if FParams.IndexOfName(szDBPASSWORD) <> -1 Then
                         FDatabasePassword := FParams.Values[szDBPASSWORD];
                     End
                End;
           End;
        Try                                                               
          FOffline := False;
          RecreateCore;
        Except
          On E:Exception do
             Begin
              if FLoginPrompt Then
                 Begin
                   if ExitDB Then Exit;
                   ShowMessage(E.Message);
                   BadPassword :=True;
                 End
              Else
                 Begin
                   Raise;
                 End;
             End;
        End;
        if BadPassword Then Goto Start;
        if (WideCompareText(FDatabaseType,'Access')=0) Then
           Begin
             Pwd:=FDatabasePassword;
             if FSmartOpen Then
                Begin
                 if NOT FileExists(FDatabase) Then
                    Begin
                      if csDesigning in ComponentState Then
                         FDatabase := GetWorkDir+ExtractFileName(FDatabase)
                      Else
                         FDatabase := GetExeDir+ExtractFileName(FDatabase);
                    End;
                End;
             if NOT FileExists(FDatabase) Then DatabaseError(E1038+#13#10+FDatabase);
             if FEngineType=dbUseJet Then
                CoreDatabase := CoreWorkspace.OpenDatabase(FDatabase,TObject(FExclusive),TObject(FReadOnly),TObject(Format(';UID=%s;PWD=%s;%s',[FUsername,Pwd,FDatabaseParameters])))
             Else
                DatabaseError(E1007);
           End
        Else
           Begin
             Pwd:=FPassword;
             if WideCompareText(FDatabaseType,'ODBC')=0 Then
                Begin
                  if FEngineType=dbUseJet Then
                    Begin
                      if Pos('odbc;',WideLowerCase(FDatabase))=1 Then
                         CoreDatabase := CoreWorkspace.OpenDatabase(FDatabase,TObject(dbDriverNoPrompt),TObject(FReadOnly),TObject(Format('%s;%s',[FDatabase,FDatabaseParameters])))
                      Else
                         CoreDatabase := CoreWorkspace.OpenDatabase(FDatabase,TObject(dbDriverNoPrompt),TObject(FReadOnly),TObject(Format('%s;UID=%s;PWD=%s;DSN=%s;%s',[FDatabaseType,FUsername,Pwd,FDatabase,FDatabaseParameters])));
                    End
                 Else
                    Begin
                      if Pos('odbc;',WideLowerCase(FDatabase))=1 Then
                           CoreDatabase := KA.Data.Dao360.Database(CoreWorkspace.OpenConnection(FDatabase,TObject(dbDriverNoPrompt),TObject(FReadOnly),TObject(Format('%s;%s',[FDatabase,FDatabaseParameters]))))
                        Else
                           CoreDatabase := KA.Data.Dao360.Database(CoreWorkspace.OpenConnection(FDatabase,TObject(dbDriverNoPrompt),TObject(FReadOnly),TObject(Format('%s;UID=%s;PWD=%s;DSN=%s;%s',[FDatabaseType,FUsername,Pwd,FDatabase,FDatabaseParameters]))));
                        //******************************* Temporaty disabled 03.02.2005
                        // DatabaseError(E1008);
                        //******************************* Temporaty disabled 03.02.2005
                    End;
                End
             Else
                Begin
                 if FEngineType=dbUseJet Then
                    Begin
                      if (Pwd='') or (FUsername='')  Then
                         CoreDatabase := CoreWorkspace.OpenDatabase(FDatabase,TObject(FExclusive),TObject(FReadOnly),TObject(Format('%s;%s',[FDatabaseType,FDatabaseParameters])))
                      Else
                         CoreDatabase := CoreWorkspace.OpenDatabase(FDatabase,TObject(FExclusive),TObject(FReadOnly),TObject(Format('%s;UID=%s;PWD=%s;%s',[FDatabaseType,FUsername,Pwd,FDatabaseParameters])));
                    End
                 Else
                    DatabaseError(E1009);
                End;
           End;
        if FQueryTimeout <> 60 Then
           Begin
             CoreDatabase.QueryTimeout := FQueryTimeout;
           End;
        if FDatabaseType<>'ODBC' Then
           FDatabaseVersion := CoreDatabase.Version;
        RefreshDefinitions;
        FCollatingOrder:=GetCollatingOrder;
        FActive:=True;
        Idle;
        if Assigned(FAfterConnect) Then FAfterConnect(Self);
    End;
End;

Procedure TKADaoDatabase.Open;
Begin
 Connected := True;
End;

Procedure TKADaoDatabase.Close;
Begin
  Connected := False;
End;

Procedure TKADaoDatabase.CloseDatasets;
{$IFDEF USEDB}
Var
  X            : Integer;
  ATable       : TKADaoTable;
{$ENDIF}
Begin
{$IFDEF USEDB}
For X:=0 to FActiveTableNames.Count-1 do
    Begin
     ATable:=TKADaoTable(FActiveTableNames.Objects[X]);
     Try
      ATable.MainDatabaseShutdown := True;
      ATable.Active:=False;
     Except
     End;
    End;
{$ENDIF}
FActiveTableNames.Clear;
End;

Procedure TKADaoDatabase.RefreshDatasets;
{$IFDEF USEDB}
Var
  X            : Integer;
  ATable       : TKADaoTable;
{$ENDIF}
Begin
Idle;
{$IFDEF USEDB}
For X:=0 to FActiveTableNames.Count-1 do
    Begin
     ATable:=TKADaoTable(FActiveTableNames.Objects[X]);
     Try
      ATable.RefreshData;
     Except
     End;
    End;
{$ENDIF}
End;

Function TKADaoDatabase.ChooseDatabase: Boolean;
Var
   NewDB    : String;
begin
  NewDB  := FChooseDatabase;
  Result := NewDB <> '';
  if Result Then Database := NewDB
end;


Procedure TKADaoDatabase.RefreshDefinitions;
Var
  X: Integer;
Begin
 FTableNames.Clear;
 FQueryDefNames.Clear;
 //*****************************************************************************
 Try
   if FEngineType = dbUseJet Then ComWrapper.Call(CoreDatabase,'TableDefs.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   ComWrapper.Call(CoreDatabase,'QueryDefs.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   if FDatabaseType='Access' Then ComWrapper.Call(CoreDatabase,'Containers.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   if FDatabaseType='Access' Then ComWrapper.Call(CoreDatabase,'Relations.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   ComWrapper.Call(CoreDatabase,'Recordsets.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   if FEngineType = dbUseJet Then ComWrapper.Call(CoreDatabase,'Properties.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   ComWrapper.Call(CoreDBEngine,'Errors.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   ComWrapper.Call(CoreDBEngine,'Workspaces.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   ComWrapper.Call(CoreDBEngine,'Properties.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   GoOnline;
   if FEngineType = dbUseJet Then ComWrapper.Call(CoreWorkspace,'Users.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   GoOnline;
   if FEngineType = dbUseJet Then ComWrapper.Call(CoreWorkspace,'Groups.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   GoOnline;
   ComWrapper.Call(CoreWorkspace,'Databases.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
   GoOnline;
   ComWrapper.Call(CoreWorkspace,'Properties.Refresh');
 Except
 End;
 //*****************************************************************************
 Try
 if FEngineType = dbUseJet Then
    Begin
      For X:=0 To CoreDatabase.TableDefs.Count-1 do
          Begin
            if FShowSysObjects Then
               Begin
                 FTableNames.Add(CoreDatabase.TableDefs.Item[TObject(X)].Name);
               End
            Else
               Begin
                 if (CoreDatabase.TableDefs.Item[TObject(X)].Attributes And dbSystemObject) = 0 Then
                    Begin
                      FTableNames.Add(CoreDatabase.TableDefs.Item[TObject(X)].Name);
                    End;
               End;
          End;
    End;
 Except
 End;
 //*****************************************************************************
 Try
 For X:=0 To CoreDatabase.QueryDefs.Count-1 do
     Begin
      FQueryDefNames.Add(CoreDatabase.QueryDefs.Item[TObject(X)].Name);
     End;
 Except
 End;
End;

Procedure TKADaoDatabase.Idle;
Begin
 CoreDBEngine.Idle(TObject(KA.Data.KADao.DaoApi.dbFreeLocks));
 CoreDBEngine.Idle(TObject(dbRefreshCache));
End;

Procedure TKADaoDatabase.SetDatabase(Value : String);
Begin
  if (FActive) Then
     Begin
       DatabaseError(E1010);
       Exit;
     End;
  FDatabase:=Value;
End;

Procedure TKADaoDatabase.SetDatabaseParameters(Value : String);
Begin
  if (FActive) Then
     Begin
       DatabaseError(E1037);                                                                                 
       Exit;
     End;
  FDatabaseParameters:=Value;
End;

Procedure TKADaoDatabase.SetSystemDatabase(Value : String);
Var
  Tmp : String;
Begin
  if (FActive) Then
     Begin
       DatabaseError(E1011);
       Exit;
     End;
  Tmp:=FSystemDB;
  FSystemDB:=Value;
  if FSystemDB = '' Then FSystemDB := GetSystemDatabaseFromRegistry;
  if csLoading In ComponentState then Exit;
  //*********************** RECREATE???
  Try
   RecreateCore;
  Except
   FSystemDB:=Tmp;
   RecreateCore;
   Raise;
  End;
End;

Procedure TKADaoDatabase.SetDaoVersion(Value : String);
Begin
 FActualDaoVersion := CoreDBEngine.Version;
End;

Procedure TKADaoDatabase.SetActualDaoVersion(Value : String);
Begin
  //This property is read only
End;

Procedure TKADaoDatabase.SetDatabaseVersion(Value : String);
Begin
  //This property is read only
End;


Function TKADaoDatabase.GetSystemDatabaseFromRegistry:String;
Var
  RS   : String;
  Reg : TRegistry;
Begin
  Result:='';
  RS:='4.0';
  Reg := TRegistry.Create;
  Reg.Access:=KEY_READ;
  Try
    Reg.RootKey := HKEY_LOCAL_MACHINE;
    if Reg.OpenKeyReadOnly(Format('SOFTWARE\Microsoft\JET\%s\Engines',[RS])) then
       Begin
         Result:=Reg.ReadString('SystemDB');
         Reg.CloseKey;
       End;
  Finally
    Reg.Free;
  End;
End;

Function TKADaoDatabase.FindWorkspace(WS:String):Boolean;
Var
  X : Integer;
Begin
  Result := False;
  For X :=0 to CoreDBEngine.Workspaces.Count-1 do
      Begin
       if CoreDBEngine.Workspaces.Item[TObject(X)].Name=WS Then
          Begin
            Result := True;
            Exit;
          End;
      End;
End;

Procedure TKADaoDatabase.SetWorkspace(Value : String);
Var
  Tmp : String;
Begin
  if (FActive) Then
     Begin
       DatabaseError(E1013);
       Exit;
     End;
  Tmp:=FWorkspace;
  FWorkspace:=Value;
  if csLoading In ComponentState then Exit;
  //*********************** RECREATE???
  Try
   RecreateCore;
  Except
   FWorkspace:=Tmp;
   RecreateCore;
   Raise;
  End;
End;


Procedure TKADaoDatabase.SetDatabaseType(Value : String);
Begin
  if (FActive) Then
     Begin
       DatabaseError(E1014);
       Exit;
     End;
  FDatabase:='';
  FDatabaseType:=Value;
End;

Function TKADaoDatabase.GetCollatingOrder:String;
Var
  CO : Integer;
Begin
  Result := '';
  DatabaseLanguageInt:=0;
  if Not FActive Then Exit;
  CO:=dbSortUndefined;
  Try
   CO := CoreDatabase.CollatingOrder;
  Except
  End;
  DatabaseLanguageInt := CO;
  Case CO of
     dbSortGeneral	          : Result := 'General (English, French, German, Portuguese, Italian, and Modern Spanish)';
     dbSortArabic	            : Result := 'Arabic';
     dbSortChineseSimplified	: Result := 'Simplified Chinese';
     dbSortChineseTraditional	: Result := 'Traditional Chinese';
     dbSortCyrillic	          : Result := 'Bulgarian or Russian';
     dbSortCzech	            : Result := 'Czech';
     dbSortDutch	            : Result := 'Dutch';
     dbSortGreek	            : Result := 'Greek';
     dbSortHebrew	            : Result := 'Hebrew';
     dbSortHungarian	        : Result := 'Hungarian';
     dbSortIcelandic	        : Result := 'Icelandic';
     dbSortJapanese	          : Result := 'Japanese';
     dbSortKorean	            : Result := 'Korean';
     dbSortNeutral	          : Result := 'Neutral';
     dbSortNorwDan	          : Result := 'Norwegian or Danish';
     dbSortPolish	            : Result := 'Polish';
     dbSortSlovenian	        : Result := 'Slovenian';
     dbSortSpanish	          : Result := 'Spanish';
     dbSortSwedFin	          : Result := 'Swedish or Finnish';
     dbSortThai	              : Result := 'Thai';
     dbSortTurkish	          : Result := 'Turkish';
     dbSortUndefined	        : Result := 'Undefined or unknown';
  Else
     Result := 'Unknown ('+IntToStr(CO)+')';
  End;
  FCollatingOrder:=Result;
End;


Procedure TKADaoDatabase.SetEngineType(Value : Integer);
Var
  Tmp : Integer;
Begin
  if (FActive) Then
     Begin
       DatabaseError(E1015);
       Exit;
     End;
  Tmp:=FEngineType;
  FEngineType:=Value;
  if csLoading In ComponentState then Exit;
  //*********************** RECREATE???
  Try
   RecreateCore;
  Except
   FEngineType:=Tmp;
   RecreateCore;
   Raise;
  End;
End;

Procedure TKADaoDatabase.SetPrivateEngine(Value : Boolean);
Var
  Tmp : Boolean;
Begin
  if (FActive) Then
     Begin
       DatabaseError(E1016);
       Exit;
     End;
  Tmp:=FPrivateEngine;
  FPrivateEngine:=Value;
  if csLoading In ComponentState then Exit;
  //*********************** RECREATE???
  Try
   RecreateCore;
  Except
   FPrivateEngine:=Tmp;
   RecreateCore;
   Raise;
  End;
End;

Procedure TKADaoDatabase.SetShowSysObjects(Value : Boolean);
Begin
 FShowSysObjects := Value;
 if FActive Then RefreshDefinitions;
End;

Function  TKADaoDatabase.GetDatabaseType:String;
Begin
  Result:=FDatabaseType;
End;

Procedure TKADaoDatabase.SetReadOnly(Value : Boolean);
{$IFDEF USEDB}
Var
  X      : Integer;
  ATable : TKADaoTable;
 {$ENDIF}
Begin
 if (FActive) Then
     Begin
       DatabaseError(E1017);
       Exit;
     End;
 FReadOnly:=Value;
 {$IFDEF USEDB}
 if FReadOnly Then
    Begin
     For X :=0 To FActiveTableNames.Count-1 do
      Begin
      ATable:=TKADaoTable(FActiveTableNames.Objects[X]);
      ATable.ReadOnly:=True;
     End;
    End;
 {$ENDIF}
End;

Procedure TKADaoDatabase.SetExclusive(Value : Boolean);
Begin
 if (FActive) Then
     Begin
       DatabaseError(E1018);
       Exit;
     End;
 FExclusive:=Value;
End;

Procedure TKADaoDatabase.SetLoginPrompt(Value : Boolean);
Begin
 if (FActive) Then
     Begin
       DatabaseError(E1019);
       Exit;
     End;
 FLoginPrompt:=Value;
End;

Procedure TKADaoDatabase.SetUserName(Value : String);
Begin
 if (FActive) Then
     Begin
       DatabaseError(E1020);
       Exit;
     End;
 FUserName:=Value;
 if csLoading in ComponentState Then Exit;
 Try
  RecreateCore;
 Except
 End;
End;

Procedure TKADaoDatabase.SetPassword(Value : String);
Begin
 if (FActive) Then
     Begin
       DatabaseError(E1021);
       Exit;
     End;
 FPassword:=Value;
 if csLoading in ComponentState Then Exit;
 Try
  RecreateCore;
 Except
 End;
End;

Procedure TKADaoDatabase.SetDatabasePassword(Value : String);
Begin
 if (FActive) Then
     Begin
       DatabaseError(E1022);
       Exit;
     End;
 FDatabasePassword:=Value;
End;

Procedure TKADaoDatabase.GoOffline;
Begin
 FOffline := True;
End;

Procedure TKADaoDatabase.GoOnline;
Begin
 FOffline := False;
End;

Procedure TKADaoDatabase.AddRNToTransaction(TableName : String;RN:Integer);
Var
 SL : TStringList;
 I  : Integer;
Begin
 if FTransInfo.Count = 0 Then Exit;
 SL := TStringList.Create;
 Try
  SL.CommaText := FTransInfo.Strings[FTransInfo.Count-1];
  I := SL.IndexOfName(TableName);
  if I <> -1 Then
     Begin
       SL.Values[TableName] := IntToStr(RN);
     End
 Else
     Begin
      if FTransInfo.Strings[FTransInfo.Count-1] <> '' Then
         SL.Add(','+TableName+'='+IntToStr(RN))
      Else
         SL.Add(TableName+'='+IntToStr(RN))
     End;
 FTransInfo.Strings[FTransInfo.Count-1]:=SL.CommaText;
 Except
 End;
 SL.Free;
End;

Function TKADaoDatabase.GetTableRN(Tables:String;TableName:String):String;
Var
 SL : TStringList;
 I  : Integer;
Begin
 Result := '-1';
 SL := TStringList.Create;
 Try
  SL.CommaText := Tables;
  I := SL.IndexOfName(TableName);
  if I <> -1 Then Result := SL.Values[TableName];
 Except
 End;
 SL.Free;
End;

Procedure TKADaoDatabase.StartTransaction;
{$IFDEF USEDB}
Var
  X       : Integer;
  S       : String;
  ATable  : TKADaoTable;
{$ENDIF}
Begin
  if (NOT FActive) Then
     Begin
       DatabaseError(E1023);
       Exit;
     End;
  CoreWorkspace.BeginTrans;
  {$IFDEF USEDB}
  if FTrackTransactions Then
     Begin
       S:= '';
       For X := 0 To FActiveTableNames.Count-1 do
           Begin
             ATable:=TKADaoTable(FActiveTableNames.Objects[X]);
             if ATable.IsEmpty Then
                Begin
                  S := S+ATable.Name+'=-1';
                End
             Else
                Begin
                  if (ATable.Bookmarkable) And (ATable.MasterSource = Nil) Then
                     S := S+ATable.Name+'='+IntToStr(ATable.BookmarkToInteger(ATable.GetBookmark))
                  Else
                     S := S+ATable.Name+'='+IntToStr(ATable.RecNo);
                End;
             if X < FActiveTableNames.Count-1 Then S := S + ',';
           End;
       FTransInfo.Add(S);
     End;
  {$ENDIF}
End;
                       
Procedure TKADaoDatabase.Commit;
Begin
 if (NOT FActive) Then
     Begin
       DatabaseError(E1024);
       Exit;
     End;
 CoreWorkspace.CommitTrans(dbForceOSFlush);
 if FTrackTransactions Then
    Begin
      if FTransInfo.Count > 0 Then FTransInfo.Delete(FTransInfo.Count-1);
    End
End;

Procedure TKADaoDatabase.Rollback;
{$IFDEF USEDB}
Var
  X       : Integer;
  RN      : String;
  BKS     : Integer;
  ATable  : TKADaoTable;
{$ENDIF}
Begin
 CoreWorkspace.Rollback;
 {$IFDEF USEDB}
 For X := 0 to FActiveTableNames.Count-1 do
     Begin
      ATable:=TKADaoTable(FActiveTableNames.Objects[X]);
      ATable.RollbackRefresh;
      if FTrackTransactions Then
         Begin
          if FTransInfo.Count > 0 Then
             Begin
              RN := GetTableRN(FTransInfo.Strings[FTransInfo.Count-1],ATable.Name);
              if RN <> '-1' Then
                 Begin
                   Try
                     if NOT ATable.IsEmpty Then
                        Begin
                          if (ATable.Bookmarkable) And (ATable.MasterSource = Nil) Then
                             Begin
                               BKS := StrToInt(RN);
                               ATable.Bookmark := ATable.IntegerToString(BKS);
                             End
                          Else
                             Begin
                               ATable.RecNo := StrToInt(RN);
                             End;  
                        End;
                   Except
                   End;
                 End;
             End;
         End;
     End;
  if FTrackTransactions Then
     Begin
      if FTransInfo.Count > 0 Then FTransInfo.Delete(FTransInfo.Count-1);
     End;
 {$ENDIF}
End;

Function TKADaoDatabase.GetTransactionCount:Integer;
Begin
 Result := FTransInfo.Count;
End;


Procedure TKADaoDatabase.RollbackRefresh;
{$IFDEF USEDB}
Var
  X       : Integer;
  ATable  : TKADaoTable;
{$ENDIF}
Begin
 {$IFDEF USEDB}
 For X :=0 To FActiveTableNames.Count-1 do
     Begin
      ATable:=TKADaoTable(FActiveTableNames.Objects[X]);
      ATable.RollbackRefresh;
     End;
 {$ENDIF}
End;



Procedure TKADaoDatabase.DBEngineLevel_StartTransaction;
Begin
 CoreDBEngine.BeginTrans;
End;

Procedure TKADaoDatabase.DBEngineLevel_Commit;
Begin
 CoreDBEngine.CommitTrans(dbForceOSFlush);
End;

Procedure TKADaoDatabase.DBEngineLevel_Rollback;
{$IFDEF USEDB}
Var
  X       : Integer;
  ATable  : TKADaoTable;
{$ENDIF}
Begin
 CoreDBEngine.Rollback;
 {$IFDEF USEDB}
 For X :=0 To FActiveTableNames.Count-1 do
     Begin
      ATable:=TKADaoTable(FActiveTableNames.Objects[X]);
      ATable.RollbackRefresh;
     End;
 {$ENDIF}
End;

Procedure TKADaoDatabase.WorkspaceLevel_StartTransaction;
Begin
 GoOnline;
 CoreWorkspace.BeginTrans;
End;

Procedure TKADaoDatabase.WorkspaceLevel_Commit;
Begin
 GoOnline;
 CoreWorkspace.CommitTrans(dbForceOSFlush);
End;

Procedure TKADaoDatabase.WorkspaceLevel_Rollback;
{$IFDEF USEDB}
Var
  X       : Integer;
  ATable  : TKADaoTable;
{$ENDIF}
Begin
 GoOnline;
 CoreWorkspace.Rollback;
 {$IFDEF USEDB}
 For X :=0 To FActiveTableNames.Count-1 do
     Begin
      ATable:=TKADaoTable(FActiveTableNames.Objects[X]);
      ATable.RollbackRefresh;
     End;
 {$ENDIF}
End;

//********************************************** WORKS ONLY ON DAO 3.5X
//                                              ON DAO 3.6 USE COMPACT DATABASE
//                                              WICH ALSO DOES REPAIR
//******************************************************************************
Procedure TKADaoDatabase.RepairAccessDatabase(DatabaseName,Password:String);
Begin
  CompactAccessDatabase(DatabaseName,Password);
End;

Procedure TKADaoDatabase.RepairAccessDatabaseEx(DatabaseName : String;
                                               NewLocale    : String;
                                               Encrypt      : Boolean;
                                               Decrypt      : Boolean;
                                               NewVersion   : Integer;
                                               Password     : String);
Begin
  CompactAccessDatabaseEx(DatabaseName,NewLocale,Encrypt,Decrypt,NewVersion,Password);
End;

Procedure  TKADaoDatabase.CompactAccessDatabase(DatabaseName,Password:String);
Var
  TempName : StringBuilder;
  TempPath : String;
  Name     : String;
Begin
  TempPath := ExtractFilePath(DatabaseName);
  if TempPath='' Then TempPath := GetCurrentDir;
  TempName := StringBuilder.Create(1000);
  Try
     GetTempFileName(TempPath,'mdb',0,TempName);
     Name     := TempName.ToString;
     DeleteFile(Name);
     if Password <> '' Then Password:=';pwd='+Password;
     CoreDBEngine.CompactDatabase(DatabaseName,Name,Nil,Nil,TObject(Password));
     DeleteFile(DatabaseName);
     RenameFile(Name,DatabaseName);
  Finally
    TempName.Free;
  End;
End;

Procedure  TKADaoDatabase.CompactAccessDatabaseEx(DatabaseName: String;
                                                  NewLocale   : String;
                                                  Encrypt     : Boolean;
                                                  Decrypt     : Boolean;
                                                  NewVersion  : Integer;
                                                  Password    : String);
Var
  TempName : StringBuilder;
  TempPath : String;
  Name     : String;
  Options  : Integer;
Begin
  TempPath  := ExtractFilePath(DatabaseName);
  if TempPath='' Then TempPath := GetCurrentDir;
  TempName  := StringBuilder.Create(1000);
  Try
    GetTempFileName(TempPath,'mdb',0,TempName);
    Name      := TempName.ToString;
    DeleteFile(Name);
    Options   := 0;
    if Encrypt Then Options := dbEncrypt;
    if Decrypt Then Options := dbDecrypt;
    if NewVersion <> 0 Then Options:=Options+NewVersion;
    if Password <> '' Then Password:=' ;pwd='+Password;
    CoreDBEngine.CompactDatabase(DatabaseName,Name,TObject(NewLocale),TObject(Options),TObject(Password));
    DeleteFile(DatabaseName);
    RenameFile(Name,DatabaseName);
  Finally
    TempName.Free;
  End;
End;

Procedure TKADaoDatabase.CreateAccessDatabase(DatabaseName:String);
Var
 CreateOptions : String;
 DB            : KA.Data.Dao360.Database;
Begin
 CreateOptions:=Format(dbLangGeneral,['0x0409','1252','0']);
 GoOnline;
 DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion40));
 DB.Close;
End;

Procedure TKADaoDatabase.CreateAccessDatabaseEx(DatabaseName,LANGID,CP,COUNTRY,Password,Version:String;Encrypt:Boolean);
Var
 CreateOptions : String;
 DB            : KA.Data.Dao360.Database;
Begin
 CreateOptions:=Format(dbLangGeneral,[LANGID,CP,COUNTRY]);
 if Password <> '' Then CreateOptions:=CreateOptions+';PWD='+Password;
 GoOnline;
 if Version='30' Then
     if Encrypt Then
        DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion30 OR dbEncrypt))
     Else
        DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion30))
  Else
     if Encrypt Then
        DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion40 OR dbEncrypt))
     Else
        DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion40));
 DB.Close;
End;

Procedure TKADaoDatabase.CreateAccessDatabaseEx2(DatabaseName,Language,Password,Version:String;Encrypt:Boolean);
Var
 CreateOptions : String;
 DB            : KA.Data.Dao360.Database;
Begin
 CreateOptions := Language;
 if Password <> '' Then CreateOptions:=CreateOptions+';PWD='+Password;
 GoOnline;
  if Version='30' Then
     if Encrypt Then
        DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion30 OR dbEncrypt))
     Else
        DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion30))
  Else
     if Encrypt Then
        DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion40 OR dbEncrypt))
     Else
        DB := CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, TObject(dbVersion40));
  DB.Close;
End;

Function TKADaoDatabase.ChangeDatabasePassword(OldPassword,NewPassword:String):Boolean;
Begin
  Result := False;
  if NOT FActive Then DatabaseError(E1025);
  if NOt FExclusive Then DatabaseError(E1026);
  Try
    CoreDatabase.NewPassword(OldPassword,NewPassword);
  Except
   Exit;
  End;
  Result := True;
End;

Function TKADaoDatabase.RegisterDatabase(DatabaseName, DriverName:String; Silent:Boolean; Attributes:String):Boolean;
Begin
  Result := False;
  Try
    CoreDBEngine.RegisterDatabase(DatabaseName,DriverName,Silent,Attributes);
  Except
   Exit;
  End;
  Result := True;
End;

Procedure TKADaoDatabase.RenameTable(OldTableName,NewTableName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.TableDefs.Item[OldTableName].Name:=NewTableName;
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteTable(TableName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.TableDefs.Delete(TableName);
 RefreshDefinitions;
End;

//******************************************************************************
//  1 = Primary index
//  2 = Unique
//  4 = NormalIndex
//******************************************************************************
Function TKADaoDatabase.HasPrimaryKey(NewTable:TableDef):Boolean;
Var
  X:Integer;
Begin
 Result:=False;
 For X :=0 to NewTable.Indexes.Count-1 do
     Begin
       if NewTable.Indexes.Item[TObject(X)].Primary Then
          Begin
            Result:=True;
            Exit;
          End;
     End;
End;

Function TKADaoDatabase.TablePrimaryKeyName(NewTable:TableDef):String;
Var
  X:Integer;
Begin
 Result:='';
 For X :=0 to NewTable.Indexes.Count-1 do
     Begin
       if NewTable.Indexes.Item[TObject(X)].Primary Then
          Begin
            Result:=NewTable.Indexes.Item[TObject(X)].name;
            Exit;
          End;
     End;
End;

Procedure TKADaoDatabase.DeletePrimaryKey(NewTable:TableDef);
Var
  X:Integer;
Begin
 For X :=0 to NewTable.Indexes.Count-1 do
     Begin
       if NewTable.Indexes.Item[TObject(X)].Primary Then
          Begin
            NewTable.Indexes.Delete(NewTable.Indexes.Item[TObject(X)].Name);
            Exit;
          End;
     End;
End;


Function TKADaoDatabase.CreateIndex(TableName,FieldName:String;IndexType:Integer):Boolean;
Var
  NewTable         : TableDef;
  NewField         : Field;
  NewIndex         : Index;
  PrimIndex        : Index;
  PrimaryKeyName   : String;
Begin
  Result:=False;
  RefreshDefinitions;
  Try
   NewTable  := CoreDatabase.TableDefs.Item[TableName];
   if Pos('paradox',WideLowerCase(FDatabaseType)) > 0 Then PrimaryKeyName := TableName Else PrimaryKeyName:='PrimaryKey';
   if ((IndexType And 1) > 0) Then
      Begin
        if HasPrimaryKey(NewTable) Then DeletePrimaryKey(NewTable);
        PrimIndex          := NewTable.CreateIndex(PrimaryKeyName);
        PrimIndex.Primary  := True;
        PrimIndex.Unique   := True;
        NewField           := FieldClass.Create;
        NewField.Name      := FieldName;
        IndexFields(PrimIndex.Fields).Append(NewField);
        NewTable.Indexes.Append(PrimIndex);
        if NOT ((IndexType And 2) > 0) Then IndexType:=IndexType+2;
      End;
   if ((IndexType And 2) > 0) or ((IndexType And 4) > 0) Then
      Begin
        NewIndex  := NewTable.CreateIndex(FieldName);
        if ((IndexType And 2) = 0) Then NewIndex.Unique  := False  Else  NewIndex.Unique  := True;
        NewField      := FieldClass.Create;
        NewField.Name := FieldName;
        IndexFields(NewIndex.Fields).Append(NewField);
        NewTable.Indexes.AppEnd(NewIndex);
      End;
  Except
   Exit;
  End;
  RefreshDefinitions;
  Result:=True;
End;

Procedure TKADaoDatabase.RenameIndex(TableName,OldIndexName,NewIndexName:String);
Begin
  RefreshDefinitions;
  CoreDatabase.TableDefs.Item[TableName].Indexes.Item[OldIndexName].Name:=NewIndexName;
  RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteIndexByName(TableName,IndexName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.TableDefs.Item[TableName].Indexes.Delete(IndexName);
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteIndexByFieldName(TableName,FieldName:String);
Var
 X         : Integer;
 TmpName   : String;
 IndexName : String;
 NotFound  : Boolean;
 Fld       : Field;
Begin
 RefreshDefinitions;
 Try
  Repeat
   NotFound:=True;
   ComWrapper.Call(CoreDatabase,'TableDefs.Refresh');
   For X:=0 To CoreDatabase.TableDefs.Item[TableName].Indexes.Count-1 do
       Begin
         Fld     := Field(IndexFields(CoreDatabase.TableDefs.Item[TableName].Indexes.Item[TObject(X)].Fields).Item[TObject(0)]);
         TmpName := Fld.Name;
         if TmpName=FieldName Then
            Begin
              IndexName := CoreDatabase.TableDefs.Item[TableName].Indexes.Item[TObject(X)].Name;
              DeleteIndexByName(TableName,IndexName);
              NotFound:=False;
              Break;
            End;
       End;
  Until NotFound;
 Except
 End;
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteField(TableName,FieldName:String);
Var
 X,Y       : Integer;
 TmpName   : String;
 IndexName : String;
 Found     : Boolean;
 Fld       : Field;
Begin
 RefreshDefinitions;
 Try
  Repeat
   Found:=False;
   ComWrapper.Call(CoreDatabase,'TableDefs.Refresh');
   For X:=0 To CoreDatabase.TableDefs.Item[TableName].Indexes.Count-1 do
       Begin
         For Y := 0 To IndexFields(CoreDatabase.TableDefs.Item[TableName].Indexes.Item[TObject(X)].Fields).Count-1 do
             Begin
               Fld     := Field(IndexFields(CoreDatabase.TableDefs.Item[TableName].Indexes.Item[TObject(X)].Fields).Item[TObject(Y)]);
               TmpName := Fld.Name;
               if WideCompareText(TmpName,FieldName)=0 Then
                  Begin
                    IndexName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[TObject(X)].Name;
                    DeleteIndexByName(TableName,IndexName);
                    Found:=True;
                    Break;
                  End;
             End;
         if Found Then Break;
       End;
  Until NOT Found;
 Except
 End;
 CoreDatabase.TableDefs.Item[TableName].Fields.Delete(FieldName);
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.RenameField(TableName,OldFieldName,NewFieldName:String);
Begin
  RefreshDefinitions;
  CoreDatabase.TableDefs.Item[TableName].Fields.Item[OldFieldName].Name:=NewFieldName;
  RefreshDefinitions;
End;

Function TKADaoDatabase.EmptyTable(TableName:String):Boolean;
Begin
  Result:=False;
  Try
    CoreDatabase.Execute('DELETE * FROM ['+TableName+'];',TObject(dbFailOnError));
  Except
    on E:Exception do
       Begin
         Exit;
       End;
  End;
  Result:=True;
End;

Function TKADaoDatabase.CreateEmptyTable(TableName:String):Boolean;
Var
  NewTable : TableDef;
  NewField : Field;
Begin
 Result:=False;
 Try
   NewTable  := CoreDatabase.CreateTableDef(TableName,TObject(0),TObject(''),TObject(''));
   NewField  := NewTable.CreateField('Temp',TObject(KA.Data.KADao.DaoApi.dbLong),TObject(0));
   NewTable.Fields.AppEnd(NewField);
 Except
   On E : Exception do
      Begin
        Exit;
      End;
 End;
 CoreDatabase.TableDefs.AppEnd(NewTable);
 RefreshDefinitions;
 DeleteField(TableName,'Temp');
 Result:=True;
End;

//******************************************************************************
//  1 = Primary index
//  2 = Unique
//  4 = NormalIndex
//******************************************************************************
Function TKADaoDatabase.CreateTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant; FieldsRequired:Variant):Boolean;
Var
  NewTable       : TableDef;
  NewField       : Field;
  NewIndex       : Index;
  PrimIndex      : Index;
  Primary        : Boolean;
  X              : Integer;
  Count          : Integer;
  AutoInc        : Boolean;
  IdxName        : String;
  PrimaryKeyName : String;
Begin
 if (NOT FActive) Then
     Begin
       DatabaseError(E1027);
       CreateTable:=False;
       Exit;
     End;
 if TableName='' Then
    Begin
       DatabaseError(E1028);
       CreateTable:=False;
       Exit;
     End;
 Primary := False;
 NewTable      := TableDefClass.Create;
 NewTable.Name := TableName;
 Count:=VarArrayHighBound(FieldTypes,VarArrayDimCount(FieldTypes));
 if Pos('paradox',WideLowerCase(FDatabaseType)) > 0 Then PrimaryKeyName := TableName Else PrimaryKeyName:='PrimaryKey';
 For X:=0 to Count do
     Begin
      AutoInc:=False;
      if FieldTypes[X]=dbAutoIncInteger Then
         Begin
           FieldTypes[X]:=dbLong;
           AutoInc:=True;
         End;
      NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
      NewTable.Fields.AppEnd(NewField);
      if AutoInc Then NewTable.Fields[FieldNames[X]].Attributes:=dbAutoIncrField;
      //************************************************************************
      // First Create Primary Key Indexes
      //************************************************************************
      if FieldIndexes[X] > 0 Then
         Begin
           if ((FieldIndexes[X] And 1) > 0) Then
               Begin
                 if Not Primary Then
                    Begin
                       PrimIndex          := NewTable.CreateIndex(PrimaryKeyName);
                       PrimIndex.Primary  := True;
                       PrimIndex.Unique   := True;
                       Primary:=True;
                    End;
                 NewField         := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 IndexFields(PrimIndex.Fields).Append(NewField);
               End
         End;
     End;
 if Primary Then NewTable.Indexes.AppEnd(PrimIndex);
 //*****************************************************************************
 // Then create Unique and NonUnique indexes
 //*****************************************************************************
 For X:=0 to Count do
     Begin
        if (FieldIndexes[X] And 2 > 0) Or (FieldIndexes[X] And 4 > 0) Then
         Begin
           IdxName:=FieldNames[X];
           NewIndex:=NewTable.CreateIndex(IdxName);
           if ((FieldIndexes[X] And 2) > 0) Then NewIndex.Unique  := True;
           NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
           IndexFields(NewIndex.Fields).Append(NewField);
           NewTable.Indexes.AppEnd(NewIndex);
         End;
     End;
 CoreDatabase.TableDefs.AppEnd(NewTable);
 //*****************************************************************************
 // Then mark required fields
 //*****************************************************************************
 RefreshDefinitions;
 For X:=0 to Count do
     Begin
       if FieldsRequired[X]=1 Then
          CoreDatabase.TableDefs.Item[TableName].Fields.Item[FieldNames[X]].Required := True;
     End;
 RefreshDefinitions;
 CreateTable:=True;
End;

//******************************************************************************
//  1 = Primary index
//  2 = Unique
//  4 = NormalIndex
//******************************************************************************
Function TKADaoDatabase.AddFieldsToTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant;  FieldsRequired:Variant):Boolean;
Var
  NewTable          : TableDef;
  NewField          : Field;
  PrimIndex         : Index;
  NewIndex          : Index;
  X                 : Integer;
  Count             : Integer;
  Primary           : Boolean;
  PrimaryKeyName    : String;
  IdxName           : String;
Begin
if (NOT FActive) Then
     Begin
       DatabaseError(E1029);
       AddFieldsToTable:=False;
       Exit;
     End;
 if TableName='' Then
    Begin
       DatabaseError(E1030);
       AddFieldsToTable:=False;
       Exit;
     End;
 NewTable:=CoreDatabase.TableDefs.Item[TableName];
 //*****************************************************************************
 // Delete PrimaryKey if new Primary key is required
 //*****************************************************************************
 Primary := False;
 Count:=VarArrayHighBound(FieldTypes,VarArrayDimCount(FieldTypes));
 For X:=0 to Count do
     Begin
       if ((FieldIndexes[X] And 1) = 1) Then
          Begin
           Primary:=True;
          End;
     End;
 if Pos('paradox',WideLowerCase(FDatabaseType)) > 0 Then PrimaryKeyName := TableName Else PrimaryKeyName:='PrimaryKey';
 if Primary then DeletePrimaryKey(NewTable);
 //*****************************************************************************
 Primary := False;
 For X:=0 to Count do
     Begin
      if FieldTypes[X] = dbAutoIncInteger then
         Begin
           FieldTypes[X]:= dbLong;
           NewField            := NewTable.CreateField(FieldNames[X], TObject(dbLong),TObject(FieldSizes[X]));
           NewField.Attributes := NewField.Attributes or dbAutoIncrField;
         End
      else
         Begin
           NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
         End;
      NewTable.Fields.AppEnd(NewField);
      //************************************************************************
      // First Create Primary Key Indexes
      //************************************************************************
      if FieldIndexes[X] > 0 Then
         Begin
           if ((FieldIndexes[X] And 1) = 1) Then
               Begin
                 if Not Primary Then
                    Begin
                       PrimIndex          := NewTable.CreateIndex(PrimaryKeyName);
                       PrimIndex.Primary  := True;
                       PrimIndex.Unique   := True;
                       Primary:=True;
                    End;
                 NewField         := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 IndexFields(PrimIndex.Fields).Append(NewField);
               End
         End;
     End;
 if Primary Then NewTable.Indexes.AppEnd(PrimIndex);
 //*****************************************************************************
 // Then create Unique and NonUnique indexes
 //*****************************************************************************
 For X:=0 to Count do
     Begin
        if (FieldIndexes[X] And 2 > 0) Or (FieldIndexes[X] And 4 > 0) Then
         Begin
           IdxName:=FieldNames[X];
           NewIndex:=NewTable.CreateIndex(IdxName);
           if ((FieldIndexes[X] And 2) > 0) Then NewIndex.Unique  := True;
           NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
           IndexFields(NewIndex.Fields).Append(NewField);
           NewTable.Indexes.Append(NewIndex);
         End;
     End;
 RefreshDefinitions;
 //*****************************************************************************
 // Then mark required fields
 //*****************************************************************************
 For X:=0 to Count do
     Begin
       if FieldsRequired[X]=1 Then
          CoreDatabase.TableDefs.Item[TableName].Fields.Item[FieldNames[X]].Required := True;
     End;
 RefreshDefinitions;
 AddFieldsToTable:=True;
End;

//******************************************************************************
// See _PredefinedTableTypes in KA.Data.KADao.DaoApi for information about TableType
//******************************************************************************
Procedure TKADaoDatabase.LinkExternalTable(Database,TableName,TableType:String;TableAttributes:Integer);
Var
 NewTable : TableDef;
 TDEFName : String;
 X, L     : Integer;
Begin
 TDEFName:=TableName;
 L := Length(TDEFName);
 For X := 1 to L do If TDEFName[X]='.' Then TDEFName[X]:='#';
 NewTable      := TableDefClass.Create;
 NewTable.Name := TableName;
 if Pos('%s',TableType) > 0 Then
    NewTable.Connect         := Format(TableType,[Database])
 Else
    NewTable.Connect         := TableType;
 if TableAttributes <> 0 Then NewTable.Attributes := TableAttributes;
 NewTable.SourceTableName := TableName;
 CoreDatabase.TableDefs.Append(NewTable);
End;

Procedure TKADaoDatabase.LinkExternalTableEx(Database,TableName,TableFileName,TableType:String;TableAttributes:Integer);
Var
 NewTable : TableDef;
Begin
 NewTable      := TableDefClass.Create;
 NewTable.Name := TableName;
 if Pos('%s',TableType) > 0 Then
    NewTable.Connect         := Format(TableType,[Database])
 Else
    NewTable.Connect         := TableType;
 if TableAttributes <> 0 Then NewTable.Attributes := TableAttributes;
 NewTable.SourceTableName := TableFileName;
 CoreDatabase.TableDefs.Append(NewTable);
End;

Procedure TKADaoDatabase.RefreshLink(Database,TableName,TableType:String);
Var
 LinkedTable : TableDef;
Begin
 LinkedTable := CoreDatabase.TableDefs.Item[TableName];
 if Pos('%s',TableType) > 0 Then
    LinkedTable.Connect         := Format(TableType,[Database])
 Else
    LinkedTable.Connect         := TableType;
 LinkedTable.RefreshLink;
End;

Function TKADaoDatabase.CreateQueryDef(Name:String;SQL:String):Boolean;
Var
 Query : QueryDef;
Begin
 Query:=CoreDatabase.CreateQueryDef(Name,SQL);
 RefreshDefinitions;
 CreateQueryDef:=True;
End;

Procedure TKADaoDatabase.ModifyQueryDef(Name:String;SQL:String);
Begin
 RefreshDefinitions;
 CoreDatabase.QueryDefs.Item[Name].SQL:=SQL;
 RefreshDefinitions;
End;

Function TKADaoDatabase.GetQueryDefSQLText(Name:String):String;
Begin
 Try
   Result:=CoreDatabase.QueryDefs.Item[Name].SQL;
 Except
   Result:='';
 End;
End;

Procedure TKADaoDatabase.RenameQueryDef(OldQueryName,NewQueryName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.QueryDefs.Item[OldQueryName].Name:=NewQueryName;
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteQueryDef(QueryName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.QueryDefs.Delete(QueryName);
 RefreshDefinitions;
End;

Function  TKADaoDatabase.FChooseDatabase: String;
var
   FileName              : String;
   Filter                : String;
   Temp                  : String;
   P                     : Integer;
   TableType             : String;
   DSN                   : String;
   DlgChooseOdbcDatabase : TODBCDialog;
   DlgChooseDatabase     : TOpenDialog;
Begin
  Result := '';
  If FDatabaseType='' Then DatabaseError(E1031);
  If FDatabaseType='ODBC' Then
    Begin
      DSN:=FDatabase;
      GetSystemDSNs(FSystemDSNs);
      GetUserDSNs(FUserDSNs);
      Application.CreateForm(TODBCDialog,DlgChooseOdbcDatabase);
      if DlgChooseOdbcDatabase.Execute(FSystemDSNs,FUserDSNs,Dsn,FUseODBCDialog) Then Result := DSN;
      DlgChooseOdbcDatabase.Free;
    End
  Else
    Begin
     DlgChooseDatabase := TOpenDialog.Create(Nil);
     FileName := Database;
     if FileName = '' then
        Begin
           DlgChooseDatabase.FileName   := '';
           if csDesigning in ComponentState Then
              DlgChooseDatabase.InitialDir := GetExeDir
           Else
              DlgChooseDatabase.InitialDir := GetExeDir;
        End
     Else
        Begin
           DlgChooseDatabase.FileName   := ExtractFileName(FileName);
           DlgChooseDatabase.InitialDir := ExtractFileDir(FileName);
        End;
     if FDatabaseType='Access' Then
        Begin
         Filter:='Microsoft Access (*.mdb)|*.mdb';
         Filter:=Filter+'|All files (*.*)|*.*';
         DlgChooseDatabase.Title:='Choose '+FDatabaseType+' Database:';
         DlgChooseDatabase.Options:=[ofPathMustExist,ofFileMustExist,ofHideReadOnly];
         DlgChooseDatabase.Filter :=Filter;
         DlgChooseDatabase.DefaultExt:='mdb';
         if DlgChooseDatabase.Execute then Result := DlgChooseDatabase.FileName;
        End
     Else
        Begin
         Filter:=GetDBTypeFileExtension(FDatabaseType);
         TableType:=GetDBTypeTableType(FDatabaseType);
         if TableType='1' Then
            Begin
              if SelectDirectory(FileName,[],0) Then Result := FileName;
            End
         Else
            Begin
             Temp:=Filter;
             P:=Pos('(',Temp);
             if P > 0 Then
                Begin
                  Delete(Temp,1,P);
                  P:=Pos(')',Temp);
                  if P > 0 Then Temp:=Copy(Temp,1,P-1);
                  Filter:=Filter+'|'+Temp;
                End;
             Filter:=Filter+'|All files (*.*)|*.*';
             DlgChooseDatabase.Title:='Choose '+FDatabaseType+' Database:';
             DlgChooseDatabase.Options:=[ofFileMustExist,ofPathMustExist,ofHideReadOnly];
             DlgChooseDatabase.Filter :=Filter;
             if DlgChooseDatabase.Execute then Result :=DlgChooseDatabase.FileName;
            End;
        End;
      DlgChooseDatabase.Free;
    End;
end;

//******************************************************************************
// EASY WRAPPER TO CREATE TABLES USING METHODS SIMILAR TO BORLAND'S TTABLE
//******************************************************************************

Constructor TKADaoTableManager.Create(Database : TKADaoDatabase);
Begin
  Inherited Create;
  FDatabase       := Database;
  FDummyDataset   := TDummyDataset.Create(Nil);
  IndexDefs       := TIndexDefs.Create(FDummyDataset);
  FieldDefs       := TFieldDefs.Create(FDummyDataset);
  TableName       := '';
End;

Destructor  TKADaoTableManager.Destroy;
Begin
  FieldDefs.Free;
  IndexDefs.Free;
  FDummyDataset.Free;
  Inherited Destroy;
End;

Function TKADaoTableManager.CheckStatus:Boolean;
Begin
 Result := False;
 if Not Assigned(FDatabase) Then DatabaseError(E1032);
 if Not (FDatabase.Connected) Then DatabaseError(E1025);
 if TableName='' Then
    Begin
      DatabaseError('Missing TableName!');
      Exit;
    End;
 Result := True;
End;

Procedure TKADaoTableManager.StringToList(Items: String; List: TStringList);
var
  X: Integer;
begin
  Items := StringReplace(Items,';',#13#10,[rfReplaceAll]);
  List.Clear;
  List.Text:=Items;
  For X:= 0 To List.Count - 1 Do List[X]:= Trim(List[X]);
end;

Procedure   TKADaoTableManager.AppendTable;
Var
  FN,FT,FS,FI,FR  : Variant;
  Count           : Integer;
  X               : Integer;
  Idx             : Integer;
Begin
  if Not CheckStatus Then Exit;
  Count:=FieldDefs.Count-1;
  FN:=VarArrayCreate([0, Count], varString);
  FT:=VarArrayCreate([0, Count], varInteger);
  FS:=VarArrayCreate([0, Count], varInteger);
  FI:=VarArrayCreate([0, Count], varInteger);
  FR:=VarArrayCreate([0, Count], varInteger);
  For X :=0 To Count Do
      Begin
        FN[X]:=FieldDefs.Items[X].Name;
        FT[X]:=BDEToDao(FieldDefs.Items[X].DataType);
        FS[X]:=DaoSizeToBDESize(FT[X],FieldDefs.Items[X].Size);
        if FieldDefs.Items[X].Required Then FR[X]:=1 Else FR[X]:=0;
        Idx:=0;
        FI[X]:=Idx;
      End;
  FDatabase.AddFieldsToTable(TableName,FN,FT,FS,FI,FR);
  VarClear(FN); FN:=NULL;
  VarClear(FT); FT:=NULL;
  VarClear(FS); FS:=NULL;
  VarClear(FI); FI:=NULL;
  VarClear(FR); FR:=NULL;
  CreateIndex(False);
End;


Procedure  TKADaoTableManager.CreateIndex(PreservePrimaryKeys:Boolean);
Var
  Count           : Integer;
  NT              : TableDef;
  NF              : Field;
  FI              : Index;
  X,Y             : Integer;
  PrimaryKeyName  : String;
  Primary         : Boolean;
  FieldNames      : TStringList;
  INam            : String;
Begin
  Count:=IndexDefs.Count;
  if Count=0 Then Exit;
  if Not CheckStatus Then Exit;
  FieldNames := TStringList.Create;
  Try
   NT      := FDatabase.CoreDatabase.TableDefs.Item[TableName];
   Primary := False;
   For X :=0 To Count-1 Do
      Begin
       if ixPrimary in IndexDefs[X].Options Then
          Begin
            Primary := True;
            PrimaryKeyName:=IndexDefs[X].Name;
          End;
      End;
   if Pos('paradox',WideLowerCase(FDatabase.FDatabaseType)) > 0 Then PrimaryKeyName := TableName;
   if Primary Then
     Begin
       if FDatabase.HasPrimaryKey(NT) Then FDatabase.DeletePrimaryKey(NT);
       FI:=NT.CreateIndex(PrimaryKeyName);
       FI.Primary := True;
       For X :=0 To Count-1 Do
          Begin
           if ixPrimary in IndexDefs[X].Options Then
              Begin
                 StringToList(IndexDefs[X].Fields,FieldNames);
                 For Y := 0 To FieldNames.Count-1 do
                     Begin
                       NF      := FieldClass.Create;
                       NF.Name := FieldNames.Strings[Y];
                       IndexFields(FI.Fields).Append(NF);
                     End;
              End;
          End;
       NT.Indexes.Append(FI);
     End;
   For X :=0 To Count-1 Do
      Begin
       if (IndexDefs[X].Options=[])
       or (IndexDefs[X].Options=[ixPrimary,ixUnique])
       or (IndexDefs[X].Options=[ixUnique])Then
          Begin
           StringToList(IndexDefs[X].Fields,FieldNames);
           if IndexDefs[X].Name='' Then
              INam:= FieldNames.Strings[0]
           Else
              INam:=IndexDefs[X].Name;
           if (WideCompareText(INam,PrimaryKeyName)=0) And (Primary) Then
               Begin
                 //******************* Don't Create again PRIMARY KEY
               End
           Else
               Begin
                FI:=NT.CreateIndex(INam);
                if ixUnique in IndexDefs[X].Options Then FI.Unique := True;
                For Y := 0 To FieldNames.Count-1 do
                    Begin
                      NF      := FieldClass.Create;
                      NF.Name := FieldNames.Strings[Y];
                      IndexFields(FI.Fields).Append(NF);
                    End;
                NT.Indexes.AppEnd(FI);
               End;
          End;
      End;
   FDatabase.RefreshDefinitions;
  Finally
    FieldNames.Free;
  End;
End;

Procedure   TKADaoTableManager.CreateTable;
Var
  FN,FT,FS,FI,FR  : Variant;
  Count           : Integer;
  X               : Integer;
  Idx             : Integer;
Begin
  if Not CheckStatus Then Exit;
  Count:=FieldDefs.Count-1;
  FN:=VarArrayCreate([0, Count], varString);
  FT:=VarArrayCreate([0, Count], varInteger);
  FS:=VarArrayCreate([0, Count], varInteger);
  FI:=VarArrayCreate([0, Count], varInteger);
  FR:=VarArrayCreate([0, Count], varInteger);
  For X :=0 To Count Do
      Begin
        FN[X]:=FieldDefs.Items[X].Name;
        FT[X]:=BDEToDao(FieldDefs.Items[X].DataType);
        FS[X]:=DaoSizeToBDESize(FT[X],FieldDefs.Items[X].Size);
        if FieldDefs.Items[X].Required Then FR[X]:=1 Else FR[X]:=0;
        Idx:=0;
        FI[X]:=Idx;
      End;
  FDatabase.CreateTable(TableName,FN,FT,FS,FI,FR);
  VarClear(FN); FN:=NULL;
  VarClear(FT); FT:=NULL;
  VarClear(FS); FS:=NULL;
  VarClear(FI); FI:=NULL;
  VarClear(FR); FR:=NULL;
  CreateIndex(False);
End;

//******************************************************************************
procedure Register;
Begin
  RegisterComponents('KA Dao', [TKADaoDatabase]);
End;

End.


