unit KA.Data.KADao.ConnectionCheck;

interface

uses
  Borland.Vcl.SysUtils,
  Borland.Vcl.Classes,
  Borland.Vcl.ExtCtrls,
  Borland.Vcl.DB,
  KA.Data.KADao.Database;


type
  TErrAction=(CloseDatabase,RaiseException);
  TErrActionSet=Set of TErrAction;
  TKADaoConnectionCheck = class(TComponent)
  private
    { Private declarations }
    Timer              : TTimer;
    InTimer            : Boolean;
    Procedure            TimerProcedure(Sender: TObject);
  protected
    { Protected declarations }
    FActive           : Boolean;
    FCheckInterval    : Integer;
    FDatabase         : TKADaoDatabase;
    FErrorCode        : Integer;
    FExceptionText    : String;
    FOnErrorAction    : TErrActionSet;
    FOnNoConnection   : TNotifyEvent;
    FOnConnectionAgain: TNotifyEvent;

    Procedure            SetActive(Value:Boolean);
    Procedure            SetCheckInterval(Value:Integer);
    Procedure            SetDatabase(Value:TKADaoDatabase);
    Procedure            SetErrorCode(Value:Integer);
    Procedure            SetExceptionText(Value: String);
    Procedure            SetOnErrorAction(Value:TErrActionSet);
    Procedure            Activate(Value:Boolean);
    Procedure            Notification(AComponent: TComponent; Operation: TOperation); override;
    Procedure            Loaded; override;
  public
    { Public declarations }

     Constructor         Create(AOwner : TComponent); override;
     Destructor          Destroy; override;
  published
    { Published declarations }
    Property Active                : Boolean         Read FActive            Write SetActive;
    Property Interval              : Integer         Read FCheckInterval     Write SetCheckInterval;
    Property Database              : TKADaoDatabase  Read FDatabase          Write SetDatabase;
    Property ErrorCode             : Integer         Read FErrorCode         Write SetErrorCode;
    Property ExceptionText         : String          Read FExceptionText     Write SetExceptionText;
    Property ErrorAction           : TErrActionSet   Read FOnErrorAction     Write SetOnErrorAction;
    Property OnNoConnection        : TNotifyEvent    Read FOnNoConnection    Write FOnNoConnection;
    Property OnConnectionRestored  : TNotifyEvent    Read FOnConnectionAgain Write FOnConnectionAgain;
  end;

procedure Register;

implementation


Constructor TKADaoConnectionCheck.Create(AOwner : TComponent);
Begin
 Inherited Create(AOwner);
 FActive           := False;
 FCheckInterval    := 1000;
 FDatabase         := Nil;
 FExceptionText    := 'Connection to Database %s is broken!';
 FOnErrorAction    := [];
 FErrorCode        := 0;
 InTimer            := False;
 Timer              := TTimer.Create(Self);
 Timer.Enabled      := False;
 Timer.Interval     := FCheckInterval;
 Timer.OnTimer      := TimerProcedure;
End;

Destructor TKADaoConnectionCheck.Destroy;
Begin
 if FActive Then Active := False;
 Timer.OnTimer:=Nil;
 Timer.Free;
 Inherited Destroy;
End;

Procedure TKADaoConnectionCheck.TimerProcedure(Sender: TObject);
Var
 Res : Integer;
 Dir : TSearchRec;
 S   : String;
Begin
 if InTimer Then Exit;
 if Not Assigned(FDatabase) Then Exit;
 InTimer := True;
 Try
  Res:=FindFirst(FDatabase.Database,faAnyFile,Dir);
  FindClose(Dir);
  if (Res <> 0) And (FErrorCode=0) Then
     Begin
       FErrorCode := Res;
       if FDatabase.Connected Then
          Begin
           if Assigned(FOnNoConnection) Then FOnNoConnection(Self);
           if CloseDatabase  in FOnErrorAction Then FDatabase.Close;
           if RaiseException in FOnErrorAction Then
              Begin
               if Pos('%s',WideLowerCase(FExceptionText)) <> 0  Then
                  S := Format(FExceptionText,[FDatabase.Database])
               Else
                 S := FExceptionText;
              DatabaseError(S);
             End;
          End;
     End
  Else
     Begin
       FErrorCode:=Res;
       if (FErrorCode=0) And Assigned(FOnConnectionAgain) Then FOnConnectionAgain(Self);
     End;
 Finally
  InTimer := False;
 End;
 if Not FDatabase.Connected Then Exit;
End;

Procedure TKADaoConnectionCheck.Activate(Value:Boolean);
Begin
 If Value Then
    Begin
      Timer.Enabled := True;
    End
 Else
    Begin
     Timer.Enabled := False;
    End;
End;

Procedure TKADaoConnectionCheck.Loaded;
begin
  Try
    inherited Loaded;
    if FActive Then Activate(FActive);
  Except
  End;
end;

Procedure TKADaoConnectionCheck.SetActive(Value:Boolean);
Begin
 if FActive=Value Then Exit;
 FActive := Value;
 if csLoading in ComponentState Then Exit;
 if (FActive) And (Not Assigned(FDatabase)) Then
     Begin
       FActive:=False;
       DatabaseError('Database property is not set!');
     End;
 if (FActive) And (Not FDatabase.Connected) Then
    Begin
       FActive:=False;
       FActive:=False;DatabaseError('Database is not connected!');
    End;
 Activate(FActive);
End;

Procedure TKADaoConnectionCheck.SetCheckInterval(Value:Integer);
Begin
 if csLoading in ComponentState Then
    Begin
      FCheckInterval := Value;
      Exit;
    End
 Else
    Begin
     if FActive Then DatabaseError('Cannot set CheckInterval while Active property is true!');
     FCheckInterval := Value;
    End;
End;

Procedure TKADaoConnectionCheck.SetDatabase(Value:TKADaoDatabase);
Begin
 if csLoading in ComponentState Then
    Begin
      FDatabase := Value;
      Exit;
    End
 Else
    Begin
     if FActive Then DatabaseError('Cannot set Database while Active property is true!');
     FDatabase := Value;
    End;
End;

Procedure TKADaoConnectionCheck.SetErrorCode(Value:Integer);
Begin
 //*************************************************** ReadOnly
End;

Procedure TKADaoConnectionCheck.SetExceptionText(Value: String);
Begin
  if csLoading in ComponentState Then
    Begin
      FExceptionText := Value;
      Exit;
    End
 Else
    Begin
     if FActive Then DatabaseError('Cannot set ExceptionText while Active property is true!');
     FExceptionText := Value;
    End;
End;

Procedure TKADaoConnectionCheck.SetOnErrorAction(Value:TErrActionSet);
Begin
   if csLoading in ComponentState Then
    Begin
      FOnErrorAction := Value;
      Exit;
    End
 Else
    Begin
     if FActive Then DatabaseError('Cannot set OnErrorAction while Active property is true!');
     FOnErrorAction:=Value;
    End;
End;

Procedure TKADaoConnectionCheck.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDatabase <> nil) and (AComponent = FDatabase) then FDatabase := nil;
end;

procedure Register;
begin
  RegisterComponents('KA Dao', [TKADaoConnectionCheck]);
end;

end.
