﻿unit Commons.Settings;

interface

{$region 'Settings unit usage'}
{
class TSettingsObject
  Is an abstract class to be derived to obtain a specific custom setting class. Any public field (both
  value and reference types) of this class can be saved to an XML file or Stream using the StoreTo methods.
  LoadFrom class methods create a new instance of the class and fill its public fields from a XML file,
  a Stream or a XMLNode. If the class name is TMySettings and NameSpace is set to 'MyNameSpace' the XML
  format is:

    <?xml version="1.0"?>
    <TMySettings xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="MyNameSpace">
    :
    </TMySettings>

class TSettingsSectionHandler
  Is an abstract class to be derived to obtain an handler to load settings class using &ConfigurationSettings.
  Referring the previous TMySettings class TSettingsSectionHandler is to be derived this way:

    TMySettingsSectionHandler = class (TSettingsSectionHandler)
    public
      function &Create(Parent: TObject; ConfigContext: TObject; Section: &XmlNode): TObject; override;
    end;
    [assembly: RuntimeRequiredAttribute(TypeOf(TMySettingsSectionHandler))]

    function TMySettingsSectionHandler.&Create(Parent, ConfigContext: TObject; Section: XmlNode): TObject;
    begin
      Result := inherited &CreateForClass(TMySettings, Parent, ConfigContext,Section);
    end;

  The setting class can then be instanciated this way:

    I := &ConfigurationSettings.GetConfig('TMySettings') as TMySettings;

  provided the configuration file contains:

    <?xml version="1.0" encoding="utf-8" ?>
    <configuration>
      <configSections>
        :
        <section name="TMySettings" type="SettingsNamespace.TMySettingsSectionHandler,MyAssembly" />
        :
      </configSections>
      :
      <TMySettings xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="MyNameSpace">
        :
      </TMySettings>
      :
      <appSettings>
        :
      </appSettings>
    </configuration>

  where SettingsNamespace is the namespace where TMySettingsSectionHandler resides (tipically the unit it is
  defined inside) and MyAssembly is the name of the assembly that contains SettingsNamespace.

  N.B. Derived classes are to be public, that is declared inside the interface part of units.

class TUserSettings
  Is a reimplementation of System.Configuration.ConfigurationSettings with more features.
}
{$endregion}

uses
	System.Collections,
  System.Configuration,
  System.Collections.Specialized,
  System.IO,
  System.Xml,
  System.Xml.Serialization;

type
  TSettingsObject = class abstract (TObject,ICloneable)
  private
    class var
      FNameSpace: string;
  public
    class property NameSpace: string read FNameSpace write FNameSpace;
    procedure StoreTo(Stream: System.IO.Stream); overload;
    procedure StoreTo(const FileName: string); overload;
    class function LoadFrom(Stream: System.IO.Stream): TObject; overload;
    class function LoadFrom(const FileName: string): TObject; overload;
    class function LoadFrom(Node: XmlNode): TObject; overload;
    function Clone: TObject;
    class constructor Create;
  end;

  TSettingsObjectClass = class of TSettingsObject;
  TSettingsSectionHandler = class abstract (TObject,IConfigurationSectionHandler)
  strict protected
    function &CreateForClass(SettingsClass: TSettingsObjectClass; Parent: TObject; ConfigContext: TObject; Section: &XmlNode): TObject; overload;
  public
    function &Create(Parent: TObject; ConfigContext: TObject; Section: &XmlNode): TObject; overload; virtual; abstract;
  end;

  TUserSettings = class (TObject)
  private
    FFileName: string;
    FFileValid: Boolean;
    FAppSettings: NameValueCollection;
    FAllSections: ArrayList;
    FConfigSections: ArrayList;
    // read the info (Section Type, Section Handler and its assembly name)
    // for a given Config Section
    procedure GetSectionInfo(const aSectionName: string; out aSectionPresent: Boolean; out aSectionType, aSectionHandlerAssembly: string);
    // handle the standard "NameValueCollection" sections, since the standard
    // .NET handler will "destroy" those sections....
    function HandleNameValueSection(aNode: XmlNode): TObject;
  protected
    FXmlDoc: XmlDocument;  // XML document representing the config file
    procedure CreateEmptyXmlDoc; 
  public
    // AppSettings - defined as a NameValueCollection
    property AppSettings: NameValueCollection read FAppSettings;
    // all the sections defined in the config file
    property AllSections: ArrayList read FAllSections;
    // list of the user-defined sections (that show up in <configSections>)
    property ConfigSections: ArrayList read FConfigSections;
    // Filename of the currently loaded config file - read-only
    // can only be set by CreateEx or LoadConfigFile calls
    property FileName: string read FFileName;
    // Indicator whether the file was loaded ok, and ready to be used
    property Valid: Boolean read FFileValid;
    // Get a specific value, specified by an XPath expression and
    // an attribute name - versions for string, int (dword), Boolean
    // If the node is found, the "aAttrName" attribute is returned.
    // If the node is not found, the default value is returned
    function GetValue(const aKeyName, aAttrName, aDefaultValue: string): string; overload;
    function GetValue(const aKeyName, aAttrName: string; aDefaultValue: Integer): Integer; overload;
    // returns "True", if the entry is "yes", "true", "on", or "1" - otherwise false
    function GetValue(const aKeyName, aAttrName: string; aDefaultValue: Boolean): Boolean; overload;
    // Method to load a specific config section
    function GetConfig(const aSectionName: string): TObject;
    // Method to load a specific config file
    procedure LoadConfigFile(const aFileName: string);
    // Method to reload the currently loaded config file, to pick up
    // any changes that have been made to the file
    procedure ReloadCurrentFile;
    // Method to save currently loaded config file
    // will throw an exception if no file name has been set,
    // or if user can't write back to the specified location due to
    // lack of permissions
    procedure SaveConfigFile;
    // Method to save currently loaded config file
    // under a new name (including new location, e.g. directory)
    procedure SaveConfigFileAs(const aFileName: string);
    // standard constructor with no parameters
    constructor Create; overload; virtual;
    // extended constructor, passing in a file name
    constructor Create(const aFileName: string); overload; virtual;
  end;
  EConfigurationFileName = class(ConfigurationException);
  EConfigurationSection = class(ConfigurationException);
  EConfigurationSectionHandler = class(ConfigurationException);
  EConfigurationSectionHandlerAssembly = class(ConfigurationException);
  EConfigurationSectionHandlerInterface = class(ConfigurationException);
  EConfigurationNotLoaded = class(ConfigurationException);

implementation

uses
  System.Reflection;

{ TSettingsObject }

procedure TSettingsObject.StoreTo(Stream: System.IO.Stream);
begin
  with XmlSerializer.Create(ClassInfo,FNameSpace) do begin
    Serialize(Stream,Self);
    Free;
  end;
end;

procedure TSettingsObject.StoreTo(const FileName: string);
var
  Stream: System.IO.Stream;
begin
  Stream := FileStream.Create(FileName,FileMode.Create);
  try
    StoreTo(Stream);
  finally
    Stream.Close;
    Stream.Free;
  end;
end;

class function TSettingsObject.LoadFrom(Stream: System.IO.Stream): TObject;
begin
  with XmlSerializer.Create(ClassInfo,FNameSpace) do begin
    Result := Deserialize(Stream) as TObject;
    Free;
  end;
end;

class function TSettingsObject.LoadFrom(const FileName: string): TObject;
var
  Stream: System.IO.Stream;
begin
  Stream := FileStream.Create(FileName,FileMode.Open);
  try
    Result := LoadFrom(Stream);
  finally
    Stream.Close;
    Stream.Free;
  end;
end;

class function TSettingsObject.LoadFrom(Node: XmlNode): TObject;
begin
  with XmlSerializer.Create(ClassInfo,FNameSpace) do begin
    Result := Deserialize(XmlNodeReader.Create(Node)) as TObject;
    Free;
  end;
end;

function TSettingsObject.Clone: TObject;
var
  Stream: System.IO.MemoryStream;
begin
  Stream := System.IO.MemoryStream.Create;
  StoreTo(Stream);
  Stream.Position := 0;
  Result := LoadFrom(Stream);
  Stream.Free;
end;

class constructor TSettingsObject.Create;
begin
  FNameSpace := 'http://www.ocem.com/settings';
end;

{ TSettingsSectionHandler }

function TSettingsSectionHandler.&CreateForClass(SettingsClass: TSettingsObjectClass; Parent, ConfigContext: TObject; Section: XmlNode): TObject;
begin
  Result := SettingsClass.LoadFrom(Section);
end;

{ TUserSettings }

procedure TUserSettings.CreateEmptyXmlDoc;
begin
  FXmlDoc := XmlDocument.Create;
  FXmlDoc.LoadXml('<?xml version="1.0" encoding="utf-8"?><configuration><configSections></configSections><appSettings></appSettings></configuration>');
end;

// get the internal info about a section (by consulting the <configSections> section)
// aSectionName [IN]      Name of the section you're interested in
// aSectionPresent [OUT]  Indicating, whether section was sound in <configSections> or not
// aSectionType [OUT]     Name of the section type (e.g. SingleTagSectionHandler)
// aSectionHandlerAssembly [OUT]  Name of the assembly containing the section handler
procedure TUserSettings.GetSectionInfo(const aSectionName: string;
  out aSectionPresent: Boolean;
  out aSectionType, aSectionHandlerAssembly: string);
var
  sTemp: string;
  iPos, iPos2: integer;
  oSectionNode: XmlNode;
begin
  oSectionNode := FXmlDoc.SelectSingleNode('/configuration/configSections/section[@name=''' + aSectionName + ''']');
  aSectionPresent := (oSectionNode <> nil);
  if aSectionPresent then begin
    sTemp := oSectionNode.Attributes['type'].Value;
    // find the first and (optionally) second comma in that string
    iPos := sTemp.IndexOf(',');
    if iPos >= 0 then begin
      iPos2 := sTemp.IndexOf(',', iPos + 1);
      // the first part (up to the first comma) is the type of the section
      // (or more precisely - the section handler for that section)
      aSectionType := sTemp.Substring(0, iPos).Trim;
      // The part between the first and second comma (if there is one)
      // is the assembly in which the section handler resides
      if iPos2 > 0 then
        aSectionHandlerAssembly := sTemp.Substring(iPos + 1, iPos2 - iPos - 1).Trim
      else
        // if no second comma is found, the rest of the string is the assembly name
        aSectionHandlerAssembly := sTemp.Substring(iPos + 1).Trim;
    end
    else begin
      // if we don't have a comma - assume we're dealing with the "system" assembly
      aSectionType := sTemp.Trim;
      aSectionHandlerAssembly := 'system';
    end;
  end
  else begin
    aSectionType := '';
    aSectionHandlerAssembly := '';
  end;
end;

function TUserSettings.HandleNameValueSection(aNode: XmlNode): TObject;
var
  ix: integer;
  oColl: NameValueCollection;
  oWrkNode: XmlNode;
  sKey, sValue: string;
begin
  // create the name value collection
  oColl := NameValueCollection.Create;
  for ix := 0 to aNode.ChildNodes.Count-1 do begin
    oWrkNode := aNode.ChildNodes[ix];
    if oWrkNode.Name = 'add' then begin
      // <add.....> node - get key and value
      sKey := oWrkNode.Attributes['key'].Value;
      sValue := oWrkNode.Attributes['value'].Value;
      // add key and value to the name value collection
      oColl.Add(sKey, sValue);
    end
    else if oWrkNode.Name = 'remove' then begin
      // get the key for the node
      sKey := oWrkNode.Attributes['key'].Value;
      // remove key from the name value collection
      oColl.Remove(sKey);
    end
    else if oWrkNode.Name = 'clear' then
      // clear the currently existing entries
      oColl.Clear;
  end;
  // return the resulting collection
  Result := oColl;
end;

function TUserSettings.GetValue(const aKeyName, aAttrName, aDefaultValue: string): string;
var
  oTmpNode: XmlNode;
begin
  oTmpNode := FXmlDoc.SelectSingleNode(aKeyName);
  // if we found the node - get its "aAttrName" attribute
  if Assigned(oTmpNode) and Assigned(oTmpNode.Attributes[aAttrName]) then
    Result := oTmpNode.Attributes[aAttrName].Value
  else
    Result := aDefaultValue;
end;

function TUserSettings.GetValue(const aKeyName, aAttrName: string; aDefaultValue: Integer): Integer;
var
  sTemp: string;
  iResult: integer;
  oTmpNode: XmlNode;
begin
  oTmpNode := FXmlDoc.SelectSingleNode(aKeyName);
  iResult := aDefaultValue;
  // if we found the node - get its "aAttrName" attribute
  if Assigned(oTmpNode) and Assigned(oTmpNode.Attributes[aAttrName]) then begin
    sTemp := oTmpNode.Attributes[aAttrName].Value;
    try
      iResult := Convert.ToInt32(sTemp);
    except
      iResult := aDefaultValue;
    end;
  end;
  Result := iResult;
end;

function TUserSettings.GetValue(const aKeyName, aAttrName: string; aDefaultValue: Boolean): Boolean;
var
  sTemp: string;
  bResult: Boolean;
  oTmpNode: XmlNode;
begin
  oTmpNode := FXmlDoc.SelectSingleNode(aKeyName);
  bResult := aDefaultValue;
  // if we found the node - get its "aAttrName" attribute
  if Assigned(oTmpNode) and Assigned(oTmpNode.Attributes[aAttrName]) then begin
    sTemp := oTmpNode.Attributes[aAttrName].Value.ToLower;
    bResult := (sTemp = 'yes') or (sTemp = 'true') or (sTemp = 'on') or (sTemp = '1');
  end;
  Result := bResult;
end;

function TUserSettings.GetConfig(const aSectionName: string): TObject;
var
  oConfigNode: XmlNode;
  bSectionFound: Boolean;
  oSecHandler, oSection: TObject;
  oSecHandlerIntf: IConfigurationSectionHandler;
  oSecHndlrAssembly: Assembly;
  sFullSectionName, sSectionHandler, sSectionHandlerAssembly: string;
begin
  oSection := nil;
  // handle special cases - any system-defined section such as <appSettings> etc.
  // will not be mentioned in the <configSections> section - so we don't get any
  // info from there how to handle it....
  GetSectionInfo(aSectionName, bSectionFound, sSectionHandler, sSectionHandlerAssembly);
  if bSectionFound then begin
    // get the XML node representing the config section
    if(aSectionName.ToLower.StartsWith('configuration') or
       aSectionName.ToLower.StartsWith('/configuration') or
       aSectionName.ToLower.StartsWith('//')) then
      sFullSectionName := aSectionName
    else
      sFullSectionName := '/configuration/' + aSectionName;
    oConfigNode := FXmlDoc.SelectSingleNode(sFullSectionName);
    if not Assigned(oConfigNode) then
      raise EConfigurationSection.Create('Cannot find the config section: ' + aSectionName);
    if sSectionHandlerAssembly.ToLower <> 'system' then
      // load the assembly containing the config section handler
      oSecHndlrAssembly := Assembly.Load(sSectionHandlerAssembly)
    else
      // get a handle to the "System" assembly
      oSecHndlrAssembly := Assembly.GetAssembly(typeof(ConfigurationSettings));
    if not Assigned(oSecHndlrAssembly) then
      raise EConfigurationSectionHandlerAssembly.Create('Cannot load the section handler assembly: ' + sSectionHandlerAssembly);
    // special case - the standard "NameValueSection" in the "System" assembly
    // is being treated in a destructive manner by the .NET framework code
    // so invoke our own, non-destructive handler for this case!
    if sSectionHandler.ToLower.IndexOf('namevaluesectionhandler') >= 0 then
      oSection := HandleNameValueSection(oConfigNode)
    else begin
      // instantiate a section handler object
      oSecHandler := oSecHndlrAssembly.CreateInstance(sSectionHandler);
      if not Assigned(oSecHandler) then
        raise EConfigurationSectionHandler.Create('Cannot instantiate the section handler: ' + sSectionHandler);
      oSecHandlerIntf := oSecHandler as IConfigurationSectionHandler;
      if not Assigned(oSecHandlerIntf) then
        raise EConfigurationSectionHandlerInterface.Create('Cannot get the IConfigurationSectionHandler interface: ' + sSectionHandler);
      // convert XML node to config section object as needed
      oSection := oSecHandlerIntf.&Create(nil, nil, oConfigNode.Clone);
    end;
  end;
  Result := oSection;
end;

procedure TUserSettings.LoadConfigFile(const aFileName: string);
var
  ix: Integer;
  oSection: TObject;
  oSectionList: XmlNodeList;
  oAppSettingsNode, oConfigRootNode: XmlNode;
  oIConfigHandler: IConfigurationSectionHandler;
begin
  // create the XML document
  FXmlDoc := XmlDocument.Create;
  // try to read in the file contents
  try
    FXmlDoc.Load(aFileName);
    FFileName := aFileName;
    FFileValid := true;
    // try to load the AppSettings section, if we can find one
    oAppSettingsNode := FXmlDoc.SelectSingleNode('/configuration/appSettings');
    if Assigned(oAppSettingsNode) then begin
      oIConfigHandler := NameValueSectionHandler.Create as IConfigurationSectionHandler;
      oSection := oIConfigHandler.&Create(nil, nil, oAppSettingsNode.Clone);
      FAppSettings := oSection as NameValueCollection;
    end;
    // load *all* sections
    FAllSections := ArrayList.Create;
    oConfigRootNode := FXmlDoc.SelectSingleNode('/configuration');
    for ix := 0 to oConfigRootNode.ChildNodes.Count - 1 do
      FAllSections.Add(oConfigRootNode.ChildNodes[ix].Name);
    // load all <configSections> sections
    FConfigSections := ArrayList.Create;
    oSectionList := FXmlDoc.SelectNodes('/configuration/configSections/section');
    for ix := 0 to oSectionList.Count - 1 do
      FConfigSections.Add(oSectionList[ix].Attributes['name'].Value);
  except
    on E: Exception do begin
      FXmlDoc := nil;
      FFileValid := false;
      FFileName := '';
      raise E;
    end;
  end;
end;

procedure TUserSettings.ReloadCurrentFile;
begin
  FXmlDoc := nil;
  FFileValid := System.IO.File.Exists(FFileName);
  if FFileValid then
    LoadConfigFile(FFileName)
  else
    raise System.IO.FileNotFoundException.Create('Could not find specified config file: ', FFileName);
end;

procedure TUserSettings.SaveConfigFile;
begin
  SaveConfigFileAs(FFileName);
end;

procedure TUserSettings.SaveConfigFileAs(const aFileName: string);
begin
  if not Assigned(FXmlDoc) then
    raise EConfigurationNotLoaded.Create('No config data loaded');
  if aFileName <> '' then
    FXmlDoc.Save(aFileName)
  else
    raise EConfigurationFileName.Create('No name for the config file specified');
end;

constructor TUserSettings.Create;
begin
  inherited Create;
  FFileName := '';
  FFileValid := false;
  FAppSettings := nil;
end;

constructor TUserSettings.Create(const aFileName: string);
begin
  inherited Create;
  FFileName := aFileName;
  LoadConfigFile(FFileName);
end;


end.
