unit EditNum;

{
  Edit Numeric
  ============
  Date: Sep 2002
  Author: Rosi (http://www.rosinsky.cz/delphi.html)

  Description:
  EditNum is enhanced TEdit component which provides functionality
  for editing of integer or float number.

  Note:
  Full functional demo
{}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, StdCtrls;

type
  enStyle=(enInteger, enFloat);

  TEditNum = class(TEdit)
  private
    FStyle: enStyle;
    FMinValue: extended;
    FMaxValue: extended;
    FStepValue: extended;
    FStepOnly: boolean;
    FFormat: string;
    procedure SetValue(Value: extended);
    function GetValue: extended;
    procedure SetAsInteger(Value: integer);
    function GetAsInteger: integer;
    procedure SetFormat(Value: string);
    function CheckValue: boolean;
  protected
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
  public
    constructor Create(AOwner: TComponent); override;
    procedure StepUp;
    procedure StepDown;
  published
    property Style: enStyle read FStyle write FStyle;
      {Defines type of edit value - Integer or Float}
    property MinValue: extended read FMinValue write FMinValue;
    property MaxValue: extended read FMaxValue write FMaxValue;
    property StepValue: extended read FStepValue write FStepValue;
    property StepOnly: boolean read FStepOnly write FStepOnly;
      {Defines whether only values correspond with step value can be entered}
    property Format:string read FFormat write SetFormat;
      {Defines format of number}
    property Value: extended read GetValue write SetValue;
      {Get or Set number value}
    property AsInteger: integer read GetAsInteger write SetAsInteger stored false;
    property Text stored false;
  end;

procedure Register;

implementation

{$R *.res}

procedure Register;
begin
  RegisterComponents('Rosi', [TEditNum]);
end;

constructor TEditNum.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width:=40;
  Value:=0;
end;

procedure TEditNum.KeyDown(var Key: Word; Shift: TShiftState);
begin
  case Key of
    VK_UP: begin StepUp; Key:=0; end;
    VK_DOWN: begin StepDown; Key:=0; end;
  end;
  if Key<>0 then inherited;
end;

procedure TEditNum.KeyPress(var Key: Char);
var Valid: Set of Char;
begin
  if FStyle=enInteger then Valid:=['0'..'9','-','+']
  else Valid:=['0'..'9','-','+','e','E',DecimalSeparator];
  if (not (Key in Valid)) and (Key>=#32) then Key:=#0;
  if ord(Key)=VK_RETURN then
  begin
    CheckValue;
    SelectAll;
    Key:=#0;
  end;
  if Key<>#0 then inherited KeyPress(Key);
end;

procedure TEditNum.CMExit(var Message: TCMExit);
begin
  inherited;
  if not CheckValue then SetFocus;
end;

procedure TEditNum.SetValue(Value: extended);
begin
  Text:=FormatFloat(FFormat,Value);
  CheckValue;
  if not (csDesigning in ComponentState) then SelectAll;
end;

function TEditNum.GetValue: extended;
begin
  Result:=0;
  if CheckValue then Result:=StrToFloat(Text);
end;

procedure TEditNum.SetAsInteger(Value: integer);
begin
  Text:=FormatFloat(FFormat,Value);
  CheckValue;
  if not (csDesigning in ComponentState) then SelectAll;
end;

function TEditNum.GetAsInteger: integer;
begin
  Result:=0;
  if CheckValue then Result:=Round(StrToFloat(Text));
end;

procedure TEditNum.SetFormat(Value: string);
begin
  FFormat:=Value;
  CheckValue;
end;

function TEditNum.CheckValue: boolean;
var f: single;
begin
  Result:=true;
  try
    if Text='' then f:=0
               else f:=StrToFloat(Text);
    if (FMinValue<>0) or (FMaxValue<>0) then
    begin
      if f>FMaxValue then f:=FMaxValue;
      if f<FMinValue then f:=FMinValue;
    end;
    if StepOnly and (StepValue<>0) then f:=StepValue*Round(f/StepValue);
    Text:=FormatFloat(FFormat,f);
  except
    Result:=false;
  end;
end;

procedure TEditNum.StepUp;
begin
  SetValue(GetValue+StepValue);
end;

procedure TEditNum.StepDown;
begin
  SetValue(GetValue-StepValue);
end;


end.
