unit EditDate;

{
  Edit Date
  =========
  Date: Sep 2002
  Author: Rosi (http://www.rosinsky.cz/delphi.html)

  Description:
  EditDate is enhanced TEdit component which provides functionality
  for entering of date value.

  Note:
  Full functional demo
{}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, StdCtrls;

type
  TEditDate = class(TEdit)
  private
    FMinDate: TDateTime;
    FMaxDate: TDateTime;
    FNowIfEmpty: boolean;
    FNowOnStart: boolean;
    procedure SetDate(DateValue: TDateTime);
    function GetDate: TDateTime;
    function CheckDate: boolean;
  protected
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property MinDate: TDateTime read FMinDate write FMinDate;
    property MaxDate: TDateTime read FMaxDate write FMaxDate;
    property NowIfEmpty: boolean read FNowIfEmpty write FNowIfEmpty;
      {Defines whether Now is used if Text is empty}
    property NowOnStart: boolean read FNowOnStart write FNowOnStart;
      {Defines whether Now is entered on startup}
    property DateValue: TDateTime read GetDate write SetDate;
      {Get or Set actual Date}
    property Text stored false;
  end;

procedure Register;

implementation

{$R *.res}

uses rstring, rdlg;

procedure Register;
begin
  RegisterComponents('Rosi', [TEditDate]);
end;

constructor TEditDate.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width:=65;
  DateValue:=Date;
end;

procedure TEditDate.Loaded;
begin
  inherited Loaded;
  if FNowOnStart then DateValue:=Date;
  Text:=DateToStr(DateValue);
end;

procedure TEditDate.KeyDown(var Key: Word; Shift: TShiftState);
begin
  case Key of
    VK_UP: begin
             if Shift=[] then SetDate(GetDate+1);
             if Shift=[ssCtrl] then SetDate(IncMonth(GetDate,1));
             Key:=0;
           end;
    VK_DOWN: begin
               if Shift=[] then SetDate(GetDate-1);
               if Shift=[ssCtrl] then SetDate(IncMonth(GetDate,-1));
               Key:=0;
             end;
  end;
  if Key<>0 then inherited;
end;

procedure TEditDate.KeyPress(var Key: Char);
var Valid, Sep: Set of Char;
begin
  Sep:=['.',',','/','-'];
  Valid:=['0'..'9']+Sep;
  if (not (Key in Valid)) and (Key>=#32) then Key:=#0;
  if ord(Key)=VK_RETURN then
  begin
    CheckDate;
    SelectAll;
    Key:=#0;
  end;
  if (Key in Sep) then Key:=DateSeparator;
  if Key<>#0 then inherited KeyPress(Key);
end;

procedure TEditDate.CMExit(var Message: TCMExit);
begin
  inherited;
  if not CheckDate then SetFocus;
end;

procedure TEditDate.SetDate(DateValue: TDateTime);
begin
  Text:=DateToStr(DateValue);
  CheckDate;
  if not (csDesigning in ComponentState) then SelectAll;
end;

function TEditDate.GetDate: TDateTime;
begin
  Result:=0;
  if CheckDate and (Text<>'') then Result:=StrToDate(Text);
end;

function TEditDate.CheckDate: boolean;
var d: TDateTime;
    y,m,day: word;
begin
  Result:=true;
  if (csLoading in ComponentState) then Exit;
  try
    if Text='' then
    begin
      if FNowIfEmpty then d:=Date
                     else Exit;
    end
    else
    begin
      DecodeDate(Date,y,m,day);
      if Text[Length(Text)]=DateSeparator then Text:=Copy(Text,1,Length(Text)-1);
      if Pos(DateSeparator,Text)>0 then d:=StrToDate(Text)
      else d:=EncodeDate(y,m,StrToInt(Text));
    end;
    if FMaxDate>1 then
      if d>FMaxDate then d:=FMaxDate;
    if FMinDate>1 then
      if d<FMinDate then d:=FMinDate;
    Text:=DateToStr(d);
  except
    Result:=false;
    DlgE2(sInputErr,sFormatDateErr);
  end;
end;


end.
