unit AppDirCfg;

{
  Application Directory Configuration
  ===================================
  Date: Sep 2003
  Author: Rosi (http://www.rosinsky.cz/delphi.html)

  Description:
  AppDirCfg is non visual component which provides functionality
  for loading and processing of configuration file.

  CfgFile can contain this symbols:
    %APPDIR%, %SYSTEMDIR%, %WINDIR%

  Default name of cfg file: ApplicationName.dir
  Example of cfg file:
    DataDir=%APPDIR%\Data
    ReportDir=%APPDIR%\Reports
    DLLDir=%SYSTEMDIR%
    INIDir=%WINDIR%

  Note:
  Full functional demo
{}

interface

uses
  Windows, Messages, SysUtils, Classes, Forms;

type
  TAppDirCfg = class(TComponent)
  private
    Dirs: TStringList;
    FCfgFileName: string;
    FCreateAllDir: boolean;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;

    function SystemDir: string;
    function WinDir: string;

    function GetCfg(Name: string): string;
     {Function returns "Name" value from CFG File}
    function GetCfgDef(Name, DefaultCfg: string): string;
     {Function returns "Name" value from CFG File
      or "DefaultCfg" if "Name" not found}
    function GetDir(DirName: string): string;
      {Function returns "DirName" directory from CFG File;
       last character is backslash}
    function GetDirDef(DirName, DefaultDir: string): string;
      {Function returns "DirName" directory from CFG file or
       "DefaultDir" if "DirName" not found;
       last character is backslash}
  published
    property CfgFileName: string
      read FCfgFileName write FCfgFileName;
      {Cfg File Name, 'ApplicationName.dir' is used if empty}
    property CreateAllDir: boolean
      read FCreateAllDir write FCreateAllDir;
      {if true, creates all directories on startup}
  end;

procedure Register;

implementation

{$R *.res}

uses rtool;

procedure Register;
begin
  RegisterComponents('Rosi', [TAppDirCfg]);
end;

constructor TAppDirCfg.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Dirs:=TStringList.Create;
end;

destructor TAppDirCfg.Destroy;
begin
  Dirs.Free;
  inherited Destroy;
end;

procedure TAppDirCfg.Loaded;
var a,i: integer;
    s,d: string;
begin
  inherited Loaded;
  if (csDesigning in ComponentState) then Exit;
  try
    if FCfgFileName='' then FCfgFileName:=ChangeFileExt(Application.ExeName,'.dir')
    else FCfgFileName:=GetAppPath+FCfgFileName;
    Dirs.LoadFromFile(FCfgFileName);
    Dirs.Text:=ReplaceStr('%APPDIR%\',GetAppPath,Dirs.Text,true,false);
    Dirs.Text:=ReplaceStr('%APPDIR%',GetAppPath,Dirs.Text,true,false);
    Dirs.Text:=ReplaceStr('%SYSTEMDIR%\',SystemDir,Dirs.Text,true,false);
    Dirs.Text:=ReplaceStr('%SYSTEMDIR%',SystemDir,Dirs.Text,true,false);
    Dirs.Text:=ReplaceStr('%WINDIR%\',WinDir,Dirs.Text,true,false);
    Dirs.Text:=ReplaceStr('%WINDIR%',WinDir,Dirs.Text,true,false);
    if FCreateAllDir then
      for a:=0 to Dirs.Count-1 do
      begin
        s:=Dirs[a];
        i:=Pos('=',s);
        if i>1 then
          if Pos('DIR',AnsiUpperCase(copy(s,1,i)))>0 then
          begin
            d:=copy(s,i+1,Length(s));
            if d<>'' then ForceDirectories(d);
          end;
      end;
  except
  end;
end;

function TAppDirCfg.SystemDir: string;
var p: PChar;
begin
  GetMem(p,200);
  try
    GetSystemDirectory(p,200);
    Result:=SetEndOfPath(p);
  finally
    FreeMem(p);
  end;
end;

function TAppDirCfg.WinDir: string;
var p: PChar;
begin
  GetMem(p,200);
  try
    GetWindowsDirectory(p,200);
    Result:=SetEndOfPath(p);
  finally
    FreeMem(p);
  end;
end;

function TAppDirCfg.GetCfg(Name: string): string;
begin
  Result:=Dirs.Values[Name];
end;

function TAppDirCfg.GetCfgDef(Name, DefaultCfg: string): string;
begin
  Result:=GetCfg(Name);
  if Result='' then Result:=DefaultCfg;
end;

function TAppDirCfg.GetDir(DirName: string): string;
begin
  Result:=SetEndOfPath(GetCfg(DirName));
end;

function TAppDirCfg.GetDirDef(DirName, DefaultDir: string): string;
begin
  Result:=GetDir(DirName);
  if Result='' then Result:=DefaultDir;
end;

end.
