unit RSGen;

{ ********************( RandomSignalGen ) ****************************
  Made by : Gran Pettersson, date : 9-10 Oct 2003
  =================================================
  The code hereunder is for the RSG.
 ********************************************************************* }
interface

uses
  SysUtils, Classes, ExtCtrls;



{ ************** Type for RSG ***************}
type TRSG_Type =(RSG_Disabled, RSG_Sin, RSG_Cos, RSG_Triangle, RSG_Sawtooth, RSG_Square, RSG_Random);

type TRSG_REC = record
    rsgType   : TRSG_Type;
    TimeBase,               // Set between 1.0 to 10.0 Sec's
    StepVal,                // Calculated value from TimeBase setting
    ActVal,                 // Actual value ...
    Max,                    // Max outvalue
    Min,                    // Max outvalue
    Offset,                 // Timebase offset
    Out0,                   // Output value ..
    Out180    : Single	;   // Output value, inverted value (180 phaseshift ..)..
    Enabled   : Boolean;
end;
{ *********** End Type for RSG **************}

type
  TRSGen = class(TComponent)
  private
    { Private declarations }
    FTimer  : TTimer;
    FRSGEnabled : Boolean;
//    RSGenA : Array [1..8] of TRSG_REC;
    RSGenA : Array of TRSG_REC;
    FNoOfChan : Integer;
    FVersion : String;

    Procedure TimerProc(Sender : TObject);
    procedure SetRSGEnabled(Val: Boolean) ;
    procedure SetRSNoOfChan(Val : Integer);
    procedure SetVersion( Val : String);

    function GetRSGEnabled(): Boolean;
    function GetRSNoOfChan(): Integer;

  protected
    { Protected declarations }
  public
    { Public declarations }

    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    function GetOut0( Chan : Integer): Single;
    function GetOut180( Chan : Integer): Single;

    procedure SetTimeBase( Chan : Integer ; tBase : Single);
    procedure SetType( Chan : Integer ; rsgType : TRSG_Type);
    procedure SetMax( Chan : Integer ; Val : Single);
    procedure SetMin( Chan : Integer ; Val : Single);
    procedure SetEnable( Chan : Integer ; Val : Boolean);

    function GetTimeBase( Chan : Integer) : Single;
    function GetType( Chan : Integer) : TRSG_Type;
    function GetMax( Chan : Integer) : Single;
    function GetMin( Chan : Integer) : Single;
    function GetEnable( Chan : Integer) : Boolean;
  published
    { Published declarations }
    property RSGenEnabled : Boolean read GetRSGEnabled write SetRSGEnabled;
    property RSNoOfChan : Integer read GetRSNoOfChan write SetRSNoOfChan;
    property Version : String read FVersion write SetVersion;

  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('GP', [TRSGen]);
end;

{ This procedure inititate the structure for the RandomSignalGen ....  }
constructor TRSGen.Create(AOwner : TComponent);
var
  i :Integer;
begin
  Inherited Create(AOwner);

  FTimer := TTimer.Create(self);
  FTimer.Interval := 50;
  FTimer.OnTimer := TimerProc;
  if (csdesigning in ComponentState) then
    FTimer.Enabled := False
  else
    FTimer.Enabled := FRSGEnabled;
  FNoOfChan := -1;
  RSNoOfChan := 2;

  Randomize;
end;

destructor TRSGen.Destroy;
begin
  FTimer.free;
  inherited Destroy;
end;

procedure TRSGen.SetRSGEnabled(Val: Boolean) ;
begin
  FTimer.Enabled := Val;
end;

Function TRSGen.GetRSGEnabled() : Boolean ;
begin
  Result := FTimer.Enabled ;
end;

{ This procedure is done on every timer event. It calculate the new values for each Channel ... }
procedure TRSGen.TimerProc(Sender: TObject);
var
  i :Integer;
  Val : Single;
begin
  for I:= Low(RSGenA) to High(RSGenA) do begin
    with RSGenA[i] do if Enabled then begin
      ActVal := ActVal + StepVal;
      Val := ActVal - (Trunc(ActVal/(2*pi)) * 2 * pi);
      case rsgType of
                      // -----------------------------------------------------
         RSG_Disabled : Out0 := Min ;
                      // -----------------------------------------------------
         RSG_Sin      : Out0 := ((Sin(ActVal) +1) *(Max-Min)*0.5) + Min;
                      // -----------------------------------------------------
         RSG_Cos      : Out0 := ((Cos(ActVal) +1) *(Max-Min)*0.5) + Min;
                      // -----------------------------------------------------
         RSG_Triangle : if (Val/pi) <= 1 then
                          Out0 := ((Val/pi)* (Max -Min)) + Min
                         else
                          Out0 := Max -(
                                        ((Val/pi) -1)
                                         * (Max -Min)) ;
                      // -----------------------------------------------------
         RSG_Sawtooth : Out0 := ((Val/(2*pi)) *(Max-Min)) + Min;
                      // -----------------------------------------------------
         RSG_Square  : if (Val/(pi)) < 1 then
                          Out0 := Min
                        else
                          Out0 := Max;
                      // -----------------------------------------------------
         RSG_Random   : Out0 := (Random * (Max -Min)) + Min ;
                      // -----------------------------------------------------
       end;
       Out180 := Max - Out0;
    end;
  end;
end;

function TRSGen.GetOut0( Chan : Integer): Single;
begin
  Result := RSGenA[Chan].Out0 ;
end;

function TRSGen.GetOut180( Chan : Integer): Single;
begin
  Result := RSGenA[Chan].Out180 ;
end;

procedure TRSGen.SetTimeBase( Chan : Integer ; tBase : Single);
begin
  if Chan >= 0 then
    with RSGenA[Chan] do begin
        if tBase <> 0 then begin
          TimeBase := tBase;
          StepVal  := (2 *pi)/((TimeBase * 1000) / FTimer.Interval);
        end;
    end;
end;

procedure TRSGen.SetType( Chan : Integer ; rsgType : TRSG_Type);
begin
  if Chan >= 0 then
     RSGenA[Chan].rsgType := rsgType;
end;

procedure TRSGen.SetMax( Chan : Integer ; Val : Single);
begin
  if Chan >= 0 then
     RSGenA[Chan].Max := Val;
end;

procedure TRSGen.SetMin( Chan : Integer ; Val : Single);
begin
  if Chan >= 0 then
     RSGenA[Chan].Min := Val;
end;

procedure TRSGen.SetEnable( Chan : Integer ; Val : Boolean);
begin
  if Chan >= 0 then
     RSGenA[Chan].Enabled := Val;
end;

function TRSGen.GetTimeBase( Chan : Integer) : Single;
begin
  if Chan >= 0 then
    Result := RSGenA[Chan].TimeBase ;
end;

function TRSGen.GetType( Chan : Integer) : TRSG_Type;
begin
  if Chan >= 0 then
    Result := RSGenA[Chan].rsgType ;
end;

function TRSGen.GetMax( Chan : Integer) : Single;
begin
  if Chan >= 0 then
    Result := RSGenA[Chan].Max ;
end;

function TRSGen.GetMin( Chan : Integer) : Single;
begin
  if Chan >= 0 then
    Result := RSGenA[Chan].Min ;
end;

function TRSGen.GetEnable( Chan : Integer) : Boolean;
begin
  if Chan >= 0 then
    Result := RSGenA[Chan].Enabled ;
end;

procedure TRSGen.SetRSNoOfChan( Val : Integer);
var
  PrevNoOfChan, I : Integer ;
begin
  if (val >= 0) AND (Val <> FNoOfChan) then begin
    PrevNoOfChan := FNoOfChan ;
    SetLength( RSGenA, Val +1);
    FNoOfChan := Val;

    if fNoOfChan > PrevNoOfChan then begin
      for I:= PrevNoOfChan +1 to fNoOfChan do begin
        with RSGenA[i] do begin
          rsgType := TRSG_Type((i MOD Integer(RSG_Random)) +1);
          Enabled  := True;
          SetTimeBase(i, 20.0);
          ActVal   := 0;
          Max      := 100;
          Min      := 0;
          Offset   := 0;
          Out0     := 0;
          Out180   := Max - Out0 ;
        end;
      end;
    end;
  end;
end;

procedure TRSGen.SetVersion( Val : String);
begin
  //
end;

function TRSGen.GetRSNoOfChan(): Integer;
begin
  Result := FNoOfChan;
end;

end.
