{******************************************************************************

ParamSynchronize Version 1.2

Copyright (C) 2003 by OCEM SpA - Mauro Venturini - e-mail: mpv@acm.org

You  are  free  to  use  ParamSynchronize  in compiled  form  for any  purpose.
However, use in commercial or shareware applications requires registration. The
ParamSynchronize source code, in  whole or in part, modified or unmodified, may
not be redistributed  for profit or as part of another  commercial or shareware
software package without express written permission from me or  my company.  It
can be redistributed as freeware provided this banner is not removed or altered
in any way.

This software is distributed AS IS without any warranties, express or  implied,
of  its  correctness  or  of  its  suitability  to  any  particular purpose. No
responsability is accepted for any damage, direct or indirect, due to its use.

******************************************************************************}

{******************************************************************************
The TThread.Synchronize causes a method to be executed using the main thread,
thereby avoiding multi-thread conflicts (expecially on the VCL). Unfortunately
it only supports parameterless methods. This unit provides techniques to extend
this to methods with parameters.
- Any method of a thread derived from TParamThread can be synchronized to the
  main thread using ParamSynchronize.
- Any method executed inside any thread can be synchronized to the main thread
  using the Execute method of TParamSynchronize. It can be accessed directly
  (it is a class method) or creating a TParamSynchronize object (a singleton).
- With some limitation about parameter types a thread derived from TParamThread
  can launch one of its method using ParamAsynchronize and go on without
  waiting for its completion.
- With some limitation about parameter types a any thread can launch any method
  using the Execute method of TParamAsynchronize and go on without waiting for
  its completion.
  In the last two cases the method execution is delegated to a server thread
  and the execution can be synchronized or not to the main thread.
******************************************************************************}

unit ParamSync;

interface

uses
  Classes, Windows, Messages, SysUtils;

type
  TParamBeginLosingPendingMethod = TNotifyEvent;
  TParamEndLosingPendingMethod = procedure (Sender: TObject; LostCnt: Integer) of object;
  TParamMethodException = procedure (Sender: TObject; MethodOwner: TObject; MethodEntry: Pointer; E: Exception) of object;
  TParamAsynchronizeThread = class(TThread)
  private
    FID: Integer;
    FPendingMethodLost: Boolean;
    FOnBeginLosingPendingMethod: TNotifyEvent;
    FOnEndLosingPendingMethod: TParamEndLosingPendingMethod;
    FOnMethodException: TParamMethodException;
    UserCnt: Integer;
    CreationEvent: THandle;
    QueueMaxLength: LongInt;
    SendCnt: Integer;
    ReceiveCnt: Integer;
    LostCnt: LongWord;
    ExceptionValue: Exception;
    FCaptureBuffer: Pointer;
    function IncUserCnt: Integer;
    function DecUserCnt: Integer;
    procedure PostCarrier(Msg: Cardinal;
                          CaptureBuffer: Pointer);
    procedure BeginLosingPendingMethod;
    procedure EndLosingPendingMethod;
    procedure MethodException;
  protected
    property CaptureBuffer: Pointer read FCaptureBuffer;
  	procedure Execute; override;
  public
    property ID: Integer read FID;
    property PendingMethodLost: Boolean read FPendingMethodLost write FPendingMethodLost;
    property OnBeginLosingPendingMethod: TParamBeginLosingPendingMethod read FOnBeginLosingPendingMethod write FOnBeginLosingPendingMethod;
    property OnEndLosingPendingMethod: TParamEndLosingPendingMethod read FOnEndLosingPendingMethod write FOnEndLosingPendingMethod;
    property OnMethodException: TParamMethodException read FOnMethodException write FOnMethodException;
    procedure FlushMsgQueue;
    constructor Create(const Priority: TThreadPriority;
                       const QueueMaxLength: LongWord;
                       const ID: Integer); reintroduce;
    destructor Destroy; override;
  end;
  TParamThread = class(TThread)
  private
    FServerThread: TParamAsynchronizeThread;
  public
    property ServerThread: TParamAsynchronizeThread read FServerThread;
    procedure StartServerThread(const ID: Integer = 0;
                                const Priority: TThreadPriority = tpNormal;
                                const QueueMaxLength: LongWord = 0);
    class function ParamSynchronize(MethodEntry: Pointer): Boolean;
    function ParamAsynchronize(MethodEntry: Pointer;
                               SynchronizeWithMainThread: Boolean = true): Boolean;
    function ParamAsynchronizeNewCarrier(var Carrier;
                                             Size: LongWord;
                                             MethodEntry: Pointer;
                                             SynchronizeWithMainThread: Boolean = true): Boolean;
    procedure ParamAsynchronizeSendCarrier(Carrier: Pointer);
    destructor Destroy; override;
  end;
  TParamSynchronize = class(TObject)
  public
    class function NewInstance: TObject; override;
    procedure FreeInstance; override;
  public
    class function Execute(MethodEntry: Pointer): Boolean;
  end;
  TParamAsynchronize = class(TObject)
  private
    FServerThread: TParamAsynchronizeThread;
  public
    property ServerThread: TParamAsynchronizeThread read FServerThread;
    procedure StartServerThread(const ID: Integer = 0;
                                const Priority: TThreadPriority = tpNormal;
                                const QueueMaxLength: LongWord = 0);
    function Execute(MethodEntry: Pointer;
                     SynchronizeWithMainThread: Boolean = true): Boolean;
    function NewCarrier(var Carrier;
                            Size: LongWord;
                            MethodEntry: Pointer;
                            SynchronizeWithMainThread: Boolean = true): Boolean;
    procedure SendCarrier(Carrier: Pointer);
    destructor Destroy; override;
  end;

implementation

uses
  Math;

{*****************************************************************************}

type
  TCaptureParamsBuffer = packed record
    SynchronizeWithMainThread: LongBool;
    Size: LongWord;
    MethodOwner: TObject;
    MethodEntry: Pointer;
    Params: array [0..0] of LongWord;
  end;
  PCaptureParamsBuffer = ^TCaptureParamsBuffer;
  TThreadNameInfo = record
    FType: LongWord;
    FName: PChar;
    FThreadID: LongWord;
    FFlags: LongWord;    
  end;

var
  ThreadList: TThreadList;
  Singleton: record
    Instance: TParamSynchronize;
    Finalize: Boolean;
  end;

const
  CaptureBufferHeadSize = SizeOf(TCaptureParamsBuffer) - SizeOf(LongWord);
	ParCaptureMsg = WM_User + 1024;
	ParCarrierMsg = WM_User + 1025;
	FlushMsgQueueMsg = WM_User + 1026;

resourcestring
  ThreadNameFormat = 'MethodAsyncCaller - %d';

{*****************************************************************************}

procedure SetThreadName(const Name: string);
var
  ThreadNameInfo: TThreadNameInfo;
begin
  ThreadNameInfo.FType := $1000;
  ThreadNameInfo.FName := PChar(Name);
  ThreadNameInfo.FThreadID := $FFFFFFFF;
  ThreadNameInfo.FFlags := 0;
  try
    RaiseException( $406D1388, 0, sizeof(ThreadNameInfo) div sizeof(LongWord), @ThreadNameInfo );
  except
  end;
end;

procedure CallFromCapture(Capture: PCaptureParamsBuffer);
var
  CaptureSize: LongWord;
  Entry: Pointer;
  Params: Pointer;
begin
  CaptureSize := Capture^.Size;
  Entry := Capture^.MethodEntry;
  asm
        mov     EAX,CaptureSize;
        sub     ESP,EAX
        mov     Params,ESP
  end;
  move(Capture^.Params,Params^,CaptureSize);
  asm
        mov     EAX,Entry
        call    EAX
  end;
end;

procedure CallFromCarrier(Carrier: PCaptureParamsBuffer);
var
  CaptureSize: LongWord;
  Entry: Pointer;
  Params: Pointer;
  MethodOwner: TObject;
begin
  CaptureSize := Carrier^.Size;
  Entry := Carrier^.MethodEntry;
  MethodOwner := Carrier^.MethodOwner;
  asm
        mov     EAX,CaptureSize;
        sub     ESP,EAX
        mov     Params,ESP
  end;
  FillChar(Params^,CaptureSize,0); {Null any dangling pointer}
  asm
        mov     EAX,MethodOwner
        mov     [ESP],EAX
        mov     EAX,Entry
        call    EAX
  end;
end;

function PackProcAndBuffer(Proc: TProcedure; Buffer: PCaptureParamsBuffer): TThreadMethod;
asm
        mov     [ECX],EAX
        mov     [ECX + 4],EDX
end;

{* TThreadEx *****************************************************************}

destructor TParamThread.Destroy;
begin
  if Assigned(FServerThread) then begin
    with ThreadList.LockList do try
      if FServerThread.DecUserCnt = 0 then begin
        Items[FServerThread.ID] := nil;
        FServerThread.Free;
      end;
    finally
      ThreadList.UnlockList;
    end;
  end;
  inherited Destroy;
end;

{******************************************************************************
 -- method StartServerThread [method]
 Start/give access to the thread that will execute the method specified with
 the Execute method.
 To use the ParamAsynchronize method a TParamThread thread must be connected to
 a server thread calling StartServerThread. If a server thread with the same ID
 does not exist a new one is created, otherwise the existing one is used.
 More calls to StartServerThread are from then ignored.
 --	Parameters:
    ID:             in Integer
                    An ID for the thread.
    Priority:       in TThreadPriority
                    The thread priority.
    QueueMaxLength: in LongWord
                    Max number of pending method execution requests (0 means
                    no limit).
******************************************************************************}

procedure TParamThread.StartServerThread(const ID: Integer;
                                         const Priority: TThreadPriority;
                                         const QueueMaxLength: LongWord);
begin
  if not Assigned(FServerThread) then
    with ThreadList.LockList do try
      while Count <= ID do
        Add(nil);
      if not Assigned(Items[ID]) then begin
        FServerThread := TParamAsynchronizeThread.Create(Priority,QueueMaxLength,ID);
        Items[ID] := FServerThread;
      end
      else
        FServerThread := Items[ID];
      FServerThread.IncUserCnt;
    finally
      ThreadList.UnlockList;
    end;
end;

{******************************************************************************
 -- method ParamSynchronize [method]
 ParamSynchronize causes a method called by the thread to be executed using the
 main thread.
 --	Parameters:
    MethodEntry: 		in Pointer
                    The address of the calling method (obtained using the @
                    operator), method that it is to be executed using the main
                    thread.
    Result:         out Boolean
                    True when the method runs inside a thread that is not
                    the main one. In that case the caller must immediately
                    call the Exit procedure.
 -- Example:
 procedure TASubclassOfTParamThread.AMethod(<parameters>);
 begin
   if ParamSynchronize(@TASublassOfTThreadEx.AMethod) then Exit;
   <The method instructions>;
 end;
******************************************************************************}

class function TParamThread.ParamSynchronize(MethodEntry: Pointer): Boolean;
var
  CaptureSize: LongWord;
  Params: Pointer;
  CaptureBuffer: PCaptureParamsBuffer;
begin
  asm
        mov     ECX,[EBP]
        mov     CaptureSize,ECX
        mov     ECX,[ECX]
        mov     Params,ECX
  end;
  if GetCurrentThreadId <> MainThreadID then begin
    CaptureSize := LongWord(Params) - CaptureSize - SizeOf(Pointer);
    GetMem(CaptureBuffer,CaptureBufferHeadSize + CaptureSize);
    CaptureBuffer^.SynchronizeWithMainThread := true;
    CaptureBuffer^.MethodOwner := nil;
    CaptureBuffer^.MethodEntry := MethodEntry;
    CaptureBuffer^.Size := CaptureSize;
    LongWord(Params) := LongWord(Params) - CaptureSize + SizeOf(Pointer);
    move(Params^,CaptureBuffer^.Params,CaptureSize);
    StaticSynchronize(nil,PackProcAndBuffer(@CallFromCapture,CaptureBuffer));
    FreeMem(CaptureBuffer);
    Result := true;
  end
  else
    Result := false;
end;

{******************************************************************************
 -- method ParamAsynchronize [method]
 ParamAsynchronize causes a method called by the thread to be executed
 asynchronously from it.
 --	Parameters:
    MethodEntry: 		in Pointer
                    The address of the calling method (obtained using the @
                    operator), method that it is to be executed asynchronously.
    SynchronizeWithMainThread: in Boolean
                    If true the method is executed using the main thread,
                    otherwise it is executed inside the server thread.
    Result:         out Boolean
                    True when the method runs inside a thread that is nor
                    the main nor the server thread. In that case the caller
                    must immediately call the Exit procedure.
 -- Example:
 procedure TASubclassOfTParamThread.AMethod(<parameters>);
 begin
   if ParamAsynchronize(@TASublassOfTThreadEx.AMethod,true) then Exit;
   <The method instructions>;
 end;
 -- N.B.
 The parameters of the called method can be passed by value or passed by
 reference to value statically allocated. This excludes variables inside a
 method or procedure, that are on the stack, and strings and dynamic arrays
 that are passed by reference and dynamically allocated using reference
 counting.
******************************************************************************}

function TParamThread.ParamAsynchronize(MethodEntry: Pointer;
                                        SynchronizeWithMainThread: Boolean): Boolean;
var
  CaptureSize: LongWord;
  Params: Pointer;
  MethodOwner: TObject;
  CaptureBuffer: PCaptureParamsBuffer;
begin
  asm
        mov     ECX,[EBP]
        mov     CaptureSize,ECX
        mov     ECX,[ECX]
        mov     Params,ECX
        mov     ECX,[EBP]
        mov     ECX,[ECX + 8]
        mov     MethodOwner,ECX
  end;
  if Assigned(FServerThread) then begin
    if (GetCurrentThreadId <> MainThreadID) and (GetCurrentThreadId <> FServerThread.ThreadId) then begin
      CaptureSize := LongWord(Params) - CaptureSize - SizeOf(Pointer);
      GetMem(CaptureBuffer,CaptureBufferHeadSize + CaptureSize);
      CaptureBuffer^.SynchronizeWithMainThread := SynchronizeWithMainThread;
      CaptureBuffer^.MethodOwner := MethodOwner;
      CaptureBuffer^.MethodEntry := MethodEntry;
      CaptureBuffer^.Size := CaptureSize;
      LongWord(Params) := LongWord(Params) - CaptureSize + SizeOf(Pointer);
      move(Params^,CaptureBuffer^.Params,CaptureSize);
      FServerThread.PostCarrier(ParCaptureMsg,CaptureBuffer);
      Result := true;
    end
    else
      Result := false;
  end
  else
    Result := false;
end;

{******************************************************************************
 -- method ParamAsynchronizeNewCarrier [method]
 -- method ParamAsynchronizeSendCarrier [method]
 ParamAsynchronizeNewCarrier plus ParamAsynchronizeSendCarrier are similar to
 ParamAsynchronize but in this case an explicit carrier buffer is provided to
 copy the parameter values and overcome limitations on the parameter types.
 --	Parameters:
    Carrier:        out Pointer
                    Reference to the carrier to be used for parameter values.
    Size:           in LongWord
                    The size required for the carrier buffer.
    MethodEntry: 		in Pointer
                    The address of the calling method (obtained using the @
                    operator), method that it is to be executed asynchronously.
    SynchronizeWithMainThread: in Boolean
                    If true the method is executed using the main thread,
                    otherwise it is executed inside the server thread.
    Result:         out Boolean
                    True when the method runs inside a thread that is nor
                    the main nor the server thread. In that case the caller
                    must copy the parameter values to the carrier,
                    call ParamAsynchronizeSendCarrier method and then call
                    the Exit procedure. When false the caller can get the
                    parameter values from the carrier and proceed with
                    execution.
 -- Example:
 procedure TASubclassOfTParamThread.AMethod(<parameters>);
 type
   TCarrier = record
     <Field for the parameter values>;
   end;
 var
   Carrier: ^TCarrier;
 begin
   if ParamAsynchronizeNewCarrier(Carrier,SizeOf(Carrier^),@TASublassOfTThreadEx.AMethod,true) then begin
     <Copy parameters to Carrier^>;
     ParamAsynchronizeSendCarrier(Carrier);
     Exit;
   end;
   <The method instructions (do not use the original parameters but the values from Carrier^)>;
 end;
******************************************************************************}

function TParamThread.ParamAsynchronizeNewCarrier(var Carrier;
                                                      Size: LongWord;
                                                      MethodEntry: Pointer;
                                                      SynchronizeWithMainThread: Boolean): Boolean;
var
  CaptureSize: LongWord;
  Params: Pointer;
  MethodOwner: TObject;
  CaptureBuffer: PCaptureParamsBuffer;
begin
  asm
        mov     ECX,[EBP]
        mov     CaptureSize,ECX
        mov     ECX,[ECX]
        mov     Params,ECX
        mov     ECX,[EBP]
        mov     ECX,[ECX + 8]
        mov     MethodOwner,ECX
  end;
  if Assigned(FServerThread) then begin
    if (GetCurrentThreadId <> MainThreadID) and (GetCurrentThreadId <> FServerThread.ThreadId) then begin
      CaptureSize := LongWord(Params) - CaptureSize - SizeOf(Pointer);
      GetMem(CaptureBuffer,CaptureBufferHeadSize + Size);
      CaptureBuffer^.SynchronizeWithMainThread := SynchronizeWithMainThread;
      CaptureBuffer^.MethodOwner := MethodOwner;
      CaptureBuffer^.MethodEntry := MethodEntry;
      CaptureBuffer^.Size := CaptureSize;
      Pointer(Carrier) := Ptr(LongWord(CaptureBuffer) + CaptureBufferHeadSize);
      Result := true;
    end
    else begin
      Pointer(Carrier) := Ptr(LongWord(FServerThread.CaptureBuffer) + CaptureBufferHeadSize);
      Result := false;
    end;
  end
  else begin
    Pointer(Carrier) := nil;
    Result := false;
  end;
end;

procedure TParamThread.ParamAsynchronizeSendCarrier(Carrier: Pointer);
begin
  if Assigned(FServerThread) then
    FServerThread.PostCarrier(ParCarrierMsg,Ptr(LongWord(Carrier) - CaptureBufferHeadSize));
end;

{* TParamSynchronize *********************************************************}

class function TParamSynchronize.NewInstance: TObject;
begin
  if not Assigned(Singleton.Instance) then
    Singleton.Instance := TParamSynchronize(inherited NewInstance);
  Result := Singleton.Instance;
end;

procedure TParamSynchronize.FreeInstance;
begin
  if Singleton.Finalize then
    inherited FreeInstance;
end;

{******************************************************************************
 -- method Execute [method]
 ParamSynchronize causes a method called by a thread to be executed using the
 main thread.
 --	Parameters:
    MethodEntry:    in Pointer
                    The address of the calling method (obtained using the @
                    operator), method that it is to be executed using the main
                    thread.
    Result:         out Boolean
                    True when the method runs inside a thread that is not
                    the main one. In that case the caller must immediately
                    call the Exit procedure.
 -- Example:
 procedure TAClass.AMethod(<parameters>);
 begin
   if TParamSynchronize.Execute(@TAClass.AMethod) then Exit;
   <The method instructions>;
 end;
******************************************************************************}

class function TParamSynchronize.Execute(MethodEntry: Pointer): Boolean;
var
  CaptureSize: LongWord;
  Params: Pointer;
  CaptureBuffer: PCaptureParamsBuffer;
begin
  asm
        mov     ECX,[EBP]
        mov     CaptureSize,ECX
        mov     ECX,[ECX]
        mov     Params,ECX
  end;
  if GetCurrentThreadId <> MainThreadID then begin
    CaptureSize := LongWord(Params) - CaptureSize - SizeOf(Pointer);
    GetMem(CaptureBuffer,CaptureBufferHeadSize + CaptureSize);
    CaptureBuffer^.SynchronizeWithMainThread := true;
    CaptureBuffer^.MethodOwner := nil;
    CaptureBuffer^.MethodEntry := MethodEntry;
    CaptureBuffer^.Size := CaptureSize;
    LongWord(Params) := LongWord(Params) - CaptureSize + SizeOf(Pointer);
    move(Params^,CaptureBuffer^.Params,CaptureSize);
    TThread.StaticSynchronize(nil,PackProcAndBuffer(@CallFromCapture,CaptureBuffer));
    FreeMem(CaptureBuffer);
    Result := true;
  end
  else
    Result := false;
end;

{* TParamAsynchronize *********************************************************}

destructor TParamAsynchronize.Destroy;
begin
  if Assigned(FServerThread) then begin
    with ThreadList.LockList do try
      if FServerThread.DecUserCnt = 0 then begin
        Items[FServerThread.ID] := nil;
        FServerThread.Free;
      end;
    finally
      ThreadList.UnlockList;
    end;
  end;
  inherited Destroy;
end;

{******************************************************************************
 -- method StartServerThread [method]
 Start/give access to the thread that will execute the method specified with
 the Execute method.
 After creation a TParamAsynchronize object must be connected to a server
 thread calling StartServerThread. If a server thread with the same ID does not
 exist a new one is created, otherwise the existing one is used. More calls to
 StartServerThread are from then ignored.
 --	Parameters:
    ID:             in Integer
                    An ID for the thread.
    Priority:       in TThreadPriority
                    The thread priority.
    QueueMaxLength: in LongWord
                    Max number of pending method execution requests (0 means
                    no limit).
******************************************************************************}

procedure TParamAsynchronize.StartServerThread(const ID: Integer;
                                               const Priority: TThreadPriority;
                                               const QueueMaxLength: LongWord);
begin
  if not Assigned(FServerThread) then
    with ThreadList.LockList do try
      while Count <= ID do
        Add(nil);
      if not Assigned(Items[ID]) then begin
        FServerThread := TParamAsynchronizeThread.Create(Priority,QueueMaxLength,ID);
        Items[ID] := FServerThread;
      end
      else
        FServerThread := Items[ID];
      FServerThread.IncUserCnt;
    finally
      ThreadList.UnlockList;
    end;
end;

{******************************************************************************
 -- method Execute [method]
 ParamSynchronize causes a method called by a thread to be executed
 asynchronously from it.
 --	Parameters:
    MethodEntry: 		in Pointer
                    The address of the calling method (obtained using the @
                    operator), method that it is to be executed asynchronously.
    SynchronizeWithMainThread: in Boolean
                    If true the method is executed using the main thread,
                    otherwise it is executed inside the server thread.
    Result:         out Boolean
                    True when the method runs inside a thread that is nor
                    the main nor the server thread. In that case the caller
                    must immediately call the Exit procedure.
 -- Example:
 procedure TAClass.AMethod(<parameters>);
 begin
   if ParamAsynchronizeObject.Execute(@TAClass.AMethod,true) then Exit;
   <The method instructions>;
 end;
 -- N.B.
 The parameters of the called method can be passed by value or passed by
 reference to value statically allocated. This excludes variables inside a
 method or procedure, that are on the stack, and strings and dynamic arrays
 that are passed by reference and dynamically allocated using reference
 counting.
******************************************************************************}

function TParamAsynchronize.Execute(MethodEntry: Pointer;
                                    SynchronizeWithMainThread: Boolean): Boolean;
var
  CaptureSize: LongWord;
  Params: Pointer;
  MethodOwner: TObject;
  CaptureBuffer: PCaptureParamsBuffer;
begin
  asm
        mov     ECX,[EBP]
        mov     CaptureSize,ECX
        mov     ECX,[ECX]
        mov     Params,ECX
        mov     ECX,[EBP]
        mov     ECX,[ECX + 8]
        mov     MethodOwner,ECX
  end;
  if Assigned(FServerThread) then begin
    if (GetCurrentThreadId <> MainThreadID) and (GetCurrentThreadId <> FServerThread.ThreadId) then begin
      CaptureSize := LongWord(Params) - CaptureSize - SizeOf(Pointer);
      GetMem(CaptureBuffer,CaptureBufferHeadSize + CaptureSize);
      CaptureBuffer^.SynchronizeWithMainThread := SynchronizeWithMainThread;
      CaptureBuffer^.MethodOwner := MethodOwner;
      CaptureBuffer^.MethodEntry := MethodEntry;
      CaptureBuffer^.Size := CaptureSize;
      LongWord(Params) := LongWord(Params) - CaptureSize + SizeOf(Pointer);
      move(Params^,CaptureBuffer^.Params,CaptureSize);
      FServerThread.PostCarrier(ParCaptureMsg,CaptureBuffer);
      Result := true;
    end
    else
      Result := false;
  end
  else
    Result := false;
end;

{******************************************************************************
 -- method NewCarrier [method]
 -- method SendCarrier [method]
 NewCarrier plus SendCarrier are similar to Execute but in this case an
 explicit carrier buffer is provided to copy the parameter values and overcome
 limitations on the parameter types.
 --	Parameters:
    Carrier:        out Pointer
                    Reference to the carrier to be used for parameter values.
    Size:           in LongWord
                    The size required for the carrier buffer.
    MethodEntry: 		in Pointer
                    The address of the calling method (obtained using the @
                    operator), method that it is to be executed asynchronously.
    SynchronizeWithMainThread: in Boolean
                    If true the method is executed using the main thread,
                    otherwise it is executed inside the server thread.
    Result:         out Boolean
                    True when the method runs inside a thread that is nor
                    the main nor the server thread. In that case the caller
                    must copy the parameter values to the carrier,
                    call SendCarrier method and then call the Exit procedure.
                    When false the caller can get the parameter values from
                    the carrier and proceed with execution.
 -- Example:
 procedure TAClass.AMethod(<parameters>);
 type
   TCarrier = record
     <Field for the parameter values>;
   end;
 var
   Carrier: ^TCarrier;
 begin
   if ParamAsynchronizeObject.NewCarrier(Carrier,SizeOf(Carrier^),@TAClass.AMethod,true) then begin
     <Copy parameters to Carrier^>;
     SendCarrier(Carrier);
     Exit;
   end;
   <The method instructions (do not use the original parameters but the values from Carrier^)>;
 end;
******************************************************************************}

function TParamAsynchronize.NewCarrier(var Carrier;
                                           Size: LongWord;
                                           MethodEntry: Pointer;
                                           SynchronizeWithMainThread: Boolean): Boolean;
var
  CaptureSize: LongWord;
  Params: Pointer;
  MethodOwner: TObject;
  CaptureBuffer: PCaptureParamsBuffer;
begin
  asm
        mov     ECX,[EBP]
        mov     CaptureSize,ECX
        mov     ECX,[ECX]
        mov     Params,ECX
        mov     ECX,[EBP]
        mov     ECX,[ECX + 8]
        mov     MethodOwner,ECX
  end;
  if Assigned(FServerThread) then begin
    if (GetCurrentThreadId <> MainThreadID) and (GetCurrentThreadId <> FServerThread.ThreadId) then begin
      CaptureSize := LongWord(Params) - CaptureSize - SizeOf(Pointer);
      GetMem(CaptureBuffer,CaptureBufferHeadSize + Size);
      CaptureBuffer^.SynchronizeWithMainThread := SynchronizeWithMainThread;
      CaptureBuffer^.MethodOwner := MethodOwner;
      CaptureBuffer^.MethodEntry := MethodEntry;
      CaptureBuffer^.Size := CaptureSize;
      Pointer(Carrier) := Ptr(LongWord(CaptureBuffer) + CaptureBufferHeadSize);
      Result := true;
    end
    else begin
      Pointer(Carrier) := Ptr(LongWord(FServerThread.CaptureBuffer) + CaptureBufferHeadSize);
      Result := false;
    end;
  end
  else begin
    Pointer(Carrier) := nil;
    Result := false;
  end;
end;

procedure TParamAsynchronize.SendCarrier(Carrier: Pointer);
begin
  if Assigned(FServerThread) then
    FServerThread.PostCarrier(ParCarrierMsg,Ptr(LongWord(Carrier) - CaptureBufferHeadSize));
end;

{* TParamAsynchronizeThread **************************************************}

function TParamAsynchronizeThread.IncUserCnt: Integer;
begin
  Inc(UserCnt);
  Result := UserCnt;
end;

function TParamAsynchronizeThread.DecUserCnt: Integer;
begin
  Dec(UserCnt);
  Result := UserCnt;
end;

procedure TParamAsynchronizeThread.PostCarrier(Msg: Cardinal;
                                               CaptureBuffer: Pointer);
begin
  InterlockedIncrement(SendCnt);
  PostThreadMessage(ThreadID,Msg,0,LParam(CaptureBuffer));
end;

procedure TParamAsynchronizeThread.BeginLosingPendingMethod;
begin
  FOnBeginLosingPendingMethod(Self);
end;

procedure TParamAsynchronizeThread.EndLosingPendingMethod;
begin
  FOnEndLosingPendingMethod(Self,LostCnt);
end;

procedure TParamAsynchronizeThread.MethodException;
begin
  with PCaptureParamsBuffer(FCaptureBuffer)^ do
    FOnMethodException(Self,MethodOwner,MethodEntry,ExceptionValue);
end;

procedure TParamAsynchronizeThread.Execute;
var
  Msg: TMsg;
  Quit,Flush: Boolean;
begin
  SetThreadName(Format(ThreadNameFormat,[FID]));
  SetEvent(CreationEvent);
  LostCnt := 0;
  Quit := false;
  try
    while not Quit do begin
      Flush := false;
      WaitMessage;
      while PeekMessage(Msg,0,0,0,PM_REMOVE) and not Quit do with Msg do begin
        case Message of
          ParCaptureMsg,ParCarrierMsg: begin
            Inc(ReceiveCnt);
            if (QueueMaxLength > 0) and ((SendCnt - ReceiveCnt) > QueueMaxLength) then begin
              FreeMem(Pointer(LParam));
              Inc(LostCnt);
              if not FPendingMethodLost then begin
                FPendingMethodLost := true;
                if Assigned(FOnBeginLosingPendingMethod) then
                  Synchronize(BeginLosingPendingMethod);
              end;
            end
            else begin
              if LostCnt > 0 then begin
                if Assigned(FOnEndLosingPendingMethod) then
                  Synchronize(EndLosingPendingMethod);
                LostCnt := 0;
              end;
              try
                FCaptureBuffer := Pointer(LParam);
                case Message of
                  ParCaptureMsg: begin
                    if not Flush then begin
                      if PCaptureParamsBuffer(LParam)^.SynchronizeWithMainThread then
                        StaticSynchronize(nil,PackProcAndBuffer(@CallFromCapture,FCaptureBuffer))
                      else
                        CallFromCapture(FCaptureBuffer);
                    end;
                  end;
                  ParCarrierMsg: begin
                    if not Flush then begin
                      if PCaptureParamsBuffer(LParam)^.SynchronizeWithMainThread then
                        StaticSynchronize(nil,PackProcAndBuffer(@CallFromCarrier,FCaptureBuffer))
                      else
                        CallFromCarrier(FCaptureBuffer);
                    end;
                  end;
                end;
                FreeMem(FCaptureBuffer);
                FCaptureBuffer := nil;
              except
                on E: Exception do begin
                  if Assigned(FOnMethodException) then begin
                    ExceptionValue := E;
                    Synchronize(MethodException);
                  end;
                  FreeMem(Pointer(LParam));
                end;
              end;
            end
          end;
          FlushMsgQueueMsg: Flush := true;
          WM_Quit: Quit := true;
        else
          TranslateMessage(Msg);
          DispatchMessage(Msg);
        end;
      end;
    end;
	finally
 		;
  end;
end;

constructor TParamAsynchronizeThread.Create(const Priority: TThreadPriority;
                                            const QueueMaxLength: LongWord;
                                            const ID: Integer);
begin
  inherited Create(true);
  FID := ID;
  FPendingMethodLost := False;
  Self.Priority := Priority;
  Self.QueueMaxLength := Min(QueueMaxLength,MaxLongInt);
  UserCnt := 0;
  SendCnt := 0;
  ReceiveCnt := 0;
  ExceptionValue := nil;
  FCaptureBuffer := nil;
 	CreationEvent := CreateEvent(nil,Longbool(false),Longbool(false),nil);
  Resume;
  WaitForSingleObject(CreationEvent,Infinite);
  CloseHandle(CreationEvent);
end;

destructor TParamAsynchronizeThread.Destroy;
begin
  PostThreadMessage(ThreadID,WM_Quit,0,0);
	Self.WaitFor;
  inherited Destroy;
end;

procedure TParamAsynchronizeThread.FlushMsgQueue;
begin
  PostThreadMessage(ThreadID,FlushMsgQueueMsg,0,0);
end;

{*****************************************************************************}

initialization

  ThreadList := TThreadList.Create;
  Singleton.Instance := nil;
  Singleton.Finalize := false;

finalization

  Singleton.Finalize := true;
  Singleton.Instance.Free;
  with ThreadList.LockList do begin
    while Count > 0 do begin
      TParamAsynchronizeThread(Items[Count - 1]).Free;
      Delete(Count - 1);
    end;
  end;
  ThreadList.UnlockList;
  ThreadList.Free;

end.
