unit LightTimer;

{$IFDEF VER140} {$DEFINE DELPHI_6} {$ENDIF}
{$IFDEF VER150} {$DEFINE DELPHI_7} {$ENDIF}
{$IFDEF DELPHI_6} {$DEFINE DELPHI_6_UP} {$ENDIF}
{$IFDEF DELPHI_7} {$DEFINE DELPHI_6_UP} {$ENDIF}

interface

uses
  Windows, Classes;

type
  TLightTimerOptionsEnum = (ltoCreateEnabled,ltoAutoDisable,ltoAutoFree);
  TLightTimerOptionsSet = set of TLightTimerOptionsEnum;
  TLightTimer = class(TObject)
  private
    FId: UINT;
    FEnabled: Boolean;
    FInterval: Cardinal;
    FAutoDisable: Boolean;
    FAutoFree: Boolean;
    FTag: Integer;
    FReferee: TObject;
    FOnTimer: TNotifyEvent;
    procedure SetEnabled(Value: Boolean);
    procedure SetInterval(Value: Cardinal);
    procedure SetOnTimer(Value: TNotifyEvent);
  protected
    function Start: Boolean;
    function Stop(Disable: Boolean): Boolean;
  public
    property Enabled: Boolean read FEnabled write SetEnabled;
    property Interval: Cardinal read FInterval write SetInterval default 1000;
    property AutoDisable: Boolean read FAutoDisable write FAutoDisable;
    property AutoFree: Boolean read FAutoFree write FAutoFree;
    property Tag: Integer read FTag write FTag;
    property Referee: TObject read FReferee write FReferee;
    property OnTimer: TNotifyEvent read FOnTimer write SetOnTimer;
    constructor Create(Interval: Cardinal = 1000;
                       OnTimer: TNotifyEvent = nil;
                       Options: TLightTimerOptionsSet = [ltoCreateEnabled,ltoAutoDisable,ltoAutoFree]); reintroduce; overload;
    constructor Create(Interval: Cardinal = 1000;
                       OnTimer: TNotifyEvent = nil;
                       Tag: Integer = 0;
                       Referee: TObject = nil;
                       Options: TLightTimerOptionsSet = [ltoCreateEnabled,ltoAutoDisable,ltoAutoFree]); reintroduce; overload;
    destructor Destroy; override;
  end;

function GetLightTimerCount: Cardinal;
function GetLightTimerActiveCount: Cardinal;

implementation

uses
  Messages{$IFNDEF DELPHI_6_UP}, Forms {$ENDIF};

type
  TLightTimerHandler = class(TObject)
  private
    RefCount: Cardinal;
    ActiveCount: Cardinal;
    FWindowHandle: HWND;
    procedure WndProc(var Msg: TMessage);
  public
    constructor Create;
    destructor Destroy; override;
    procedure AddTimer;
    procedure RemoveTimer;
  end;

var
  LightTimerHandler: TLightTimerHandler;

function GetLightTimerCount: Cardinal;
begin
  if Assigned(LightTimerHandler) then
    Result := LightTimerHandler.RefCount
  else
    Result := 0;
end;

function GetLightTimerActiveCount: Cardinal;
begin
  if Assigned(LightTimerHandler) then
    Result := LightTimerHandler.ActiveCount
  else
    Result := 0;
end;

{--------------- TLightTimerHandler ------------------}

constructor TLightTimerHandler.Create;
begin
  inherited Create;
{$IFDEF DELPHI_6_UP}
  FWindowHandle := Classes.AllocateHWnd(WndProc);
{$ELSE}
  FWindowHandle := AllocateHWnd(WndProc);
{$ENDIF}
end;

destructor TLightTimerHandler.Destroy;
begin
{$IFDEF DELPHI_6_UP}
  Classes.DeallocateHWnd(FWindowHandle);
{$ELSE}
  DeallocateHWnd(FWindowHandle);
{$ENDIF}
  inherited Destroy;
end;

procedure TLightTimerHandler.AddTimer;
begin
  Inc(RefCount);
end;

procedure TLightTimerHandler.RemoveTimer;
begin
  if RefCount > 0 then
    Dec(RefCount);
end;

procedure TLightTimerHandler.WndProc(var Msg: TMessage);
var
  Timer: TLightTimer;
begin
  if Msg.Msg = WM_TIMER then begin
{$WARNINGS OFF}
    Timer := TLightTimer(Msg.wParam);
{$WARNINGS ON}
    if Timer.FAutoDisable then
      Timer.Stop(True);
    // Call OnTimer event method if assigned
    if Assigned(Timer.FOnTimer) then
      Timer.FOnTimer(Timer);
    if Timer.FAutoFree then
      Timer.Free;
  end
  else
    Msg.Result := DefWindowProc(FWindowHandle, Msg.Msg, Msg.wParam, Msg.lParam);
end;

{---------------- Container management ----------------}

procedure AddTimer;
begin
  if not Assigned(LightTimerHandler) then
    // Create new handler
    LightTimerHandler := TLightTimerHandler.Create;
  LightTimerHandler.AddTimer;
end;

procedure RemoveTimer;
begin
  if Assigned(LightTimerHandler) then begin
    LightTimerHandler.RemoveTimer;
    if LightTimerHandler.RefCount = 0 then begin
      // Destroy handler
      LightTimerHandler.Free;
      LightTimerHandler := nil;
    end;
  end;
end;

{------------------- TLightTimer ---------------------}

constructor TLightTimer.Create(Interval: Cardinal;
                               OnTimer: TNotifyEvent;
                               Options: TLightTimerOptionsSet);
begin
  Create(Interval,OnTimer,0,nil,Options);
end;

constructor TLightTimer.Create(Interval: Cardinal;
                               OnTimer: TNotifyEvent;
                               Tag: Integer;
                               Referee: TObject;
                               Options: TLightTimerOptionsSet);
begin
  inherited Create;
{$WARNINGS OFF}
  FId := UINT(Self);         // Use Self as id in call to SetTimer and callback method
{$WARNINGS ON}
  FEnabled := ltoCreateEnabled in Options;
  FInterval := Interval;
  FAutoDisable := ltoAutoDisable in Options;
  FAutoFree := ltoAutoFree in Options;
  FTag := Tag;
  FReferee := Referee;
  SetOnTimer(OnTimer);
  AddTimer;                  // Container management
  if FEnabled then
    Start;
end;

destructor TLightTimer.Destroy;
begin
  if FEnabled then
    Stop(True);
  RemoveTimer;               // Container management
  inherited Destroy;
end;

procedure TLightTimer.SetEnabled(Value: Boolean);
begin
  if Value then
    Start
  else
    Stop(True);
end;

procedure TLightTimer.SetInterval(Value: Cardinal);
begin
  if Value <> FInterval then begin
    FInterval := Value;
    if FEnabled then
      if FInterval <> 0 then
        Start
      else
        Stop(False);
  end;
end;

procedure TLightTimer.SetOnTimer(Value: TNotifyEvent);
begin
  FOnTimer := Value;
  if (not Assigned(Value)) and (FEnabled) then
    Stop(False);
end;

function TLightTimer.Start: Boolean;
begin
  if FInterval = 0 then begin
    Result := False;
    Exit;
  end;
  if FEnabled then
    Stop(True);
  Result := (SetTimer(LightTimerHandler.FWindowHandle, FId, FInterval, nil) <> 0);
  if Result then begin
    FEnabled := True;
    Inc(LightTimerHandler.ActiveCount);
  end
end;

function TLightTimer.Stop(Disable: Boolean): Boolean;
begin
  if Disable then
    FEnabled := False;
  Result := KillTimer(LightTimerHandler.FWindowHandle, FId);
  if Result and (LightTimerHandler.ActiveCount > 0) then
    Dec(LightTimerHandler.ActiveCount);
end;

initialization

  LightTimerHandler:= nil;

finalization

  if Assigned(LightTimerHandler) then begin
    LightTimerHandler.Free;
    LightTimerHandler := nil;
  end;

end.

