unit ExFile;
{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
Author:       Bill BEAM

Description:  The TExFile component is used to launch one or more processes.
              A single instance of the component can keep track of multiple
              processes and provides a variety of useful properties and methods.
              You can launch one or more processes in the background or cause
              the main application to wait for a process to terminate or time-
              out.  You can use the event handlers to make something happen when
              a process terminates or fails to launch.

Creation:     September 22, 1998

Version:      1.06  November 4, 1999

EMail:        billb@catoctinsoftware.com
              wilbeam@erols.com            http://www.catoctinsoftware.com

Support:      Use comments/suggestions at website or email above. I am
              in any comments or questions you may have. Please report any
              problems via the above email or website.

Legal issues: Copyright (C) 1998, 1999 by Bill Beam
              6714 Fish Hatchery Road
              Frederick, MD 21702 (USA)

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restriction:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

Updates:

Jul 07, 1999 V1.04
     1. Added ProcCurrentDir Property. Suggested by Lani.
     2. Expanded Priority Property. Suggested by Lani.

Jul 10, 1999 V1.05
     3. Removed hThread from GetProcInfo method. IMO better off
        being closed before starting new thread.
     4. Added WaitForInputIdle for ThreadedWait should someone
        want to find a slow starting window just after launch.

Nov 4, 1999 V1.06
     5. Reinstated hThread into GetProcInfo method.
     6. Fixed 'External Exception 000000C' occuring occasionally.
     7. Added KillProcess function.
     8. Added several functions that operate without setting properties.
     9. Simplified Destructor.
     10. Modified process list so that it is updated before it is used.
     11. Updated Help.
     12. Leak tested.

{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}
interface

uses
  Windows, SysUtils, Classes, Messages;

type
  TProcCompleted = procedure(sender: Tobject;
    evFileName: string;
    evIdentifier: string;
    evRetValue: Integer;
    evTimedOut: Boolean) of object;
  TLaunchFailed = procedure(sender: Tobject;
    evFileName: string;
    evIdentifier: string;
    evErrorCode: Integer;
    evErrorMessage: string) of object;
  TWindowType = (wtNorm, wtMinimize, wtMaximize, wtHide, wtMinNoActivate,
    wtShowNoActivate);
  TWindowTypes = array[TWindowType] of Word;
  TErrMsg = (emZero, emDuplicateProc, emOnlyOneMeth, emTimedOut,
    emInValidDir, emUnknown);
  TErrMsgs = array[TErrMsg] of string[55];
  TUseEvent = (ueAll, ueOnLaunchFailed, ueOnProcCompleted, ueNone);
  TPriorityClass = (prNormal, prIdle, prHigh, prRealTime);
  TPriorityClasses = array[TPriorityClass] of Word;
  TStartType = (NonThreadedWait, ThreadedWait, Independent);
  TVersion = string;
  TProcInfo = (PrhProcess, PrDwProcessId, PrHWND, PrdwThreadId);
  PProcList = ^AThreadRecord;
  AThreadRecord = record
    PrName: string;
    PrProcIdentifier: string;
    PrhProcess: THandle;
    PrDwProcessId: DWORD;
    PrDwThreadId: DWORD;
    PrHWND: HWND;
    PrStartType: TStartType;
  end;

  TExFile = class(TComponent)
  private
    FOnLaunchFailed: TLaunchFailed;
    FOnProcCompleted: TProcCompleted;
    FProcFileName: string;
    FProcFileNamelc: string;
    FFParams: string;
    FProcIdentifier: string;
    FProcCurrentDir: string;
    FWindowType: TWindowType;
    FWaitUntilDone: Boolean;
    FPriorityClass: TPriorityClass;
    StartUpInfo: TStartUpInfo;
    ProcessInfo: TProcessInformation;
    hEventHandle: THandle;
    hMutex: THandle;
    FErrorCode: Integer;
    FExitCode: Integer;
    FUseEvent: TUseEvent;
    FTimeOutSec: Integer;
    FTimedOut: Boolean;
    FMilliSeconds: DWORD;
    FProcEnvironment: Pointer;
  protected
    HandLst: TList;
    AProcList: PProcList;
    FVersion: TVersion;
    PCurDir: PChar;
    procedure SetVersion(Value: TVersion);
    procedure SetWindowType(Value: TWindowType);
    procedure SetPriorityClass(Value: TPriorityClass);
    procedure SetTimeOutSec(Value: Integer);
    procedure SetProcFileName(Value: string);
    procedure ListMaint;
    procedure AddToList(StartType: TStartType);
    function GethProcess(StatProcName, StatProcIdentifier: string; var Hidx:
      Integer): Boolean;
    function GetExitCode(hProcess: THandle): Boolean;
    function DuplicateProc: Boolean;
    function StartProcess(StartType: TStartType): Boolean;
    procedure ErrorEvent(efError: integer; efMessage: string);
    function AlreadyRunning(TypeWait: TStartType): Boolean;
    function AssignCurrentDir: Boolean;
  public
    function Execute: Boolean;
    function CloseProcess: Boolean;
    function KillProcess(RtnCode: Integer): Boolean;
    function GetProcStatus: Boolean;
    function CloseThreads: Boolean;
    function GetErrorCode: Integer;
    function GetReturnCode: Integer;
    function ExErrorMessage(ExErrorCode: Integer): string;
    procedure ResetProps;
    function GetProcInfo(GPIType: TProcInfo; var GPIReturn: Integer):
      Boolean;
    function CloseFunction(pFileName, PIdentifier: string): Boolean;
    function KillFunction(pFileName, PIdentifier: string; RtnCode: Integer):
      Boolean;
    function ProcStatusFunction(pFileName, PIdentifier: string): Boolean;
    property ProcEnvironment: Pointer read FProcEnvironment write
      FProcEnvironment; //unpublished - for dos environment settings
    constructor Create(Aowner: TComponent); override;
    destructor Destroy; override;
  published
    property Version: TVersion read FVersion write SetVersion stored False;
    property ProcFileName: string read FProcFileName write SetProcFileName;
    property ProcParameters: string read FFParams write FFParams;
    property ProcIdentifier: string read FProcIdentifier write FProcIdentifier;
    property ProcCurrentDir: string read FProcCurrentDir write FProcCurrentDir;
    property OnProcCompleted: TProcCompleted read FOnProcCompleted write
      FonProcCompleted;
    property OnLaunchFailed: TLaunchFailed read FOnLaunchFailed write
      FOnLaunchFailed;
    property WindowType: TWindowType read FWindowType write SetWindowType;
    property WaitUntilDone: Boolean read FWaitUntilDone write FWaitUntilDone;
    property UseEvent: TUseEvent read FUseEvent write FUseEvent;
    property Priority: TPriorityClass read FPriorityClass write
      SetPriorityClass;
    property TimeOutSec: Integer read FTimeOutSec write SetTimeOutSec;
  end;

type
  TProcThread = class(TThread)
  private
    thArray: array[0..1] of THandle;
    thFileName: string;
    thIdentifier: string;
    thRetVal: DWord;
    FOnThreadDone: TProcCompleted;
    thMutex: THandle;
    thUseEvent: TUseEvent;
    thMilliseconds: DWORD;
    thRetType: Boolean;
  protected
    procedure Execute; override;
    procedure CallOnTerminate;
    constructor Create(vProcHandle: THandle;
      vProcEventHandle: THandle;
      vFileName: string;
      vProcIdentifier: string;
      vDoneMethod: TProcCompleted;
      vMutex: THandle;
      vUseEvent: TUseEvent;
      vMilliseconds: DWORD);
  end;

procedure Register;

implementation

const
  cWindowType: TWindowTypes = (SW_SHOWNORMAL, SW_SHOWMINIMIZED,
    SW_SHOWMAXIMIZED, SW_HIDE, SW_SHOWMINNOACTIVE, SW_SHOWNA);
  cPriorityClass: TPriorityClasses = (NORMAL_PRIORITY_CLASS,
    IDLE_PRIORITY_CLASS, HIGH_PRIORITY_CLASS, REALTIME_PRIORITY_CLASS);
  cErrMsg: TErrMsgs = ('Zero',
    'Another Process with the same Name and ID is executing.',
    'Cannot mix ''WaitUntilDone'' types.',
    'Process timed out',
    'Current Directory Invalid',
    'Unknown Error Code');
  {Thread array locations}
  ChPROCESS = 0;
  ChEvent = 1;
  {Timeout Constants}
  MAX_ALLOWED: DWORD = 3600000;
  MAX_IDLE: DWORD = 3000;
  cVersion: TVersion = ('1.06'); { Current version number }

constructor TExFile.Create(Aowner: TComponent);
begin
  inherited Create(AOwner);
  HandLst := TList.Create;
  Fversion := Cversion;
  FProcEnvironment := nil;
  hEventHandle := CreateEvent(nil, True, False, nil);
  hMutex := CreateMutex(nil, false, nil);
end;

destructor TExFile.Destroy;
var i: Integer;
begin
{signal and wait for waiting threads to release.}
  PulseEvent(hEventHandle);
  for i := 0 to Pred(HandLst.Count) do
    Dispose(PProcList(HandLst.Items[i]));
  HandLst.Free;
  CloseHandle(hEventHandle);
  CloseHandle(hMutex);
  inherited Destroy;
end;

constructor TProcThread.Create(vProcHandle: THandle;
  vProcEventHandle: THandle;
  vFileName: string;
  vProcIdentifier: string;
  vDoneMethod: TProcCompleted;
  vMutex: THandle;
  vUseEvent: TUseEvent;
  vMilliSeconds: DWORD);
begin
  inherited Create(True);
  FreeOnTerminate := True;
  thRetType := False;
  thArray[ChPROCESS] := vProcHandle;
  thArray[ChEvent] := vProcEventHandle;
  thFileName := vFileName;
  thIdentifier := vProcIdentifier;
  FonThreadDone := vDoneMethod;
  thMutex := vMutex;
  thUseEvent := vUseEvent;
  thMilliseconds := vMilliSeconds;
  Resume;                               {UnSuspend}
end;

procedure TProcThread.Execute;
var Signaled: Integer;
begin
  Signaled := WaitForMultipleObjects(2, @thArray, False, thMilliseconds);
  if Signaled <> WAIT_OBJECT_0 + ChEvent then //Event not signaled
    begin
      if Signaled = WAIT_OBJECT_0 + ChPROCESS then //hProcess signaled
        GetExitCodeProcess(thArray[ChPROCESS], thRetVal)
      else
        thRetVal := thMilliseconds div 1000; //WAIT_TIMEOUT or WAIT_ABANDONED
      thRetType := (Signaled <> WAIT_OBJECT_0 + ChPROCESS);
      if assigned(FOnThreadDone) and
        ((thUseEvent = ueAll) or (thUseEvent = ueOnProcCompleted)) then
        begin
          WaitForSingleObject(thMutex, INFINITE);
          Synchronize(CallOnTerminate);
          ReleaseMutex(thMutex);
        end;
    end;
  CloseHandle(thArray[ChPROCESS]);
  Terminate;
end;

procedure TProcThread.CallOnTerminate;
begin
  FOnThreadDone(Self, thFileName, thIdentifier, thRetVal, thRetType);
end;

procedure TExFile.SetWindowType(Value: TWindowType);
begin
  FWindowType := Value;
end;

procedure TexFile.ResetProps;
begin
  ProcFileName := '';
  ProcParameters := '';
  ProcIdentifier := '';
  WindowType := wtNorm;
  Priority := prNormal;
  UseEvent := ueAll;
  WaitUntilDone := false;
  TimeOutSec := 0;
  ProcCurrentDir := '';
  ProcEnvironment := nil;
end;

procedure TExFile.SetPriorityClass(Value: TPriorityClass);
begin
  FPriorityClass := Value;
end;

procedure TExFile.SetVersion(Value: TVersion);
begin
  {Do Nothing.  Set by create}
end;

procedure TExFile.SetProcFileName(Value: string);
begin
  FProcFileNamelc := LowerCase(Value);
  FProcFileName := Value;
end;

procedure TExFile.SetTimeOutSec(Value: Integer);
begin
  FTimeOutSec := Value;
  if Value = 0 then
    FMilliSeconds := INFINITE
  else
    if Value > 3600 then
      FMilliSeconds := MAX_ALLOWED
    else
      FMilliSeconds := Value * 1000;
end;

function TExFile.Execute: Boolean;
var
  WaitStatus: Integer;
begin
  Result := False;
  FErrorCode := 0;

  if FWaitUntilDone then
    begin
      Result := StartProcess(NonThreadedWait);
      if Result then
        begin
          WaitStatus := WaitforSingleObject(ProcessInfo.hProcess,
            FMilliSeconds);
          if WaitStatus = WAIT_OBJECT_0 + ChProcess then
            GetExitCodeProcess(ProcessInfo.hProcess, DWord(FExitCode))
          else
            begin
              FExitCode := FTimeOutSec;
              FErrorCode := ord(emTimedOut) * -1;
            end;
          FTimedOut := (WaitStatus <> WAIT_OBJECT_0 + ChProcess);
          CloseHandle(ProcessInfo.hProcess);
          if assigned(FonProcCompleted) and
            ((FUseEvent = ueAll) or (FUseEvent = ueOnProcCompleted)) then
            FonProcCompleted(Self, FProcFileName, FProcIdentifier,
              FExitCode, FTimedOut);
          ListMaint;
        end;
    end
  else
    begin
      Result := StartProcess(ThreadedWait);
      if Result then
        begin
          TProcThread.create(ProcessInfo.hProcess,
            hEventHandle,
            FProcFileName,
            FProcIdentifier,
            FonProcCompleted,
            hMutex,
            FUseEvent,
            FMilliseconds);
        end;
    end;
end;

procedure TExFile.AddToList(StartType: TStartType);
begin
  new(AProcList);
  with AProcList^ do begin
      PrName := FProcFileNamelc;
      PrProcIdentifier := FProcIdentifier;
      PrhProcess := ProcessInfo.hProcess;
      PrDwProcessId := ProcessInfo.dwProcessId;
      PrDwThreadId := ProcessInfo.dwThreadId;
      //PrhThread := ProcessInfo.hThread;
      PrStartType := StartType;
      HandLst.add(AProcList);
    end;
end;

function TExFile.StartProcess(StartType: TStartType): boolean;
var
  vProcFileName: array[0..512] of char;
  vIdle: DWORD;
begin
  Result := false;
  ListMaint;
{sets pointer to lpCurrentDirectory}
  if not AssignCurrentDir then
    exit;
{see if trying to start a NonThreadedWait while another process is running}
  if StartType = NonThreadedWait then
    if AlreadyRunning(ThreadedWait) then
      exit;
{see if trying to start a duplicate process}
  if DuplicateProc then
    exit;
{Start the process}
  StrPCopy(vProcFileName, FProcFileNamelc + ' ' + FFParams);
  FillChar(Startupinfo, SizeOf(TstartupInfo), #0);
  with StartupInfo do
    begin
      cb := SizeOf(TStartupInfo);
      dwFlags := STARTF_USESHOWWINDOW;
      wShowWindow := cWindowType[FWindowType];
    end;
  Result := CreateProcess(nil,
    VProcFileName,
    nil,
    nil,
    False,
    CREATE_NEW_CONSOLE or cPriorityClass[FPriorityClass],
    FProcEnvironment,
    pCurDir,
    StartupInfo,
    ProcessInfo);
{Wait no longer than MAX_IDLE for initialization. For ThreadedWait types only}
  if Result then
    CloseHandle(ProcessInfo.hThread);
  if (Result) and (StartType = ThreadedWait) then
    begin
      vIdle := WaitForInputIdle(ProcessInfo.hProcess, MAX_IDLE);
      Result := ((vIdle = 0) or (vIdle = WAIT_TIMEOUT));
    end;
  if Result then
    AddToList(StartType)
  else
    ErrorEvent(GetLastError, SysErrorMessage(GetLastError));
end;

procedure TExFile.ErrorEvent(efError: integer; efMessage: string);
begin
  FErrorCode := efError;
  if Assigned(FOnLaunchFailed) and
    ((FUseEvent = ueAll) or (FUseEvent = ueOnLaunchFailed)) then
    FOnLaunchFailed(Self, FProcFileName, FProcIdentifier, efError, efMessage);
end;

function TExFile.DuplicateProc: Boolean; //returns true if duplicate found
var x: integer;
begin
  Result := GethProcess(FProcFileNamelc, FProcIdentifier, x);
  if Result then
    ErrorEvent(ord(emDuplicateProc) * -1, cErrMsg[emDuplicateProc]);
end;

function TExFile.AssignCurrentDir: Boolean;
var Code: Integer;
begin
  Result := true;
  PCurDir := nil;
  if FProcCurrentDir <> '' then
    begin {avoid bringing in the FileCtrl unit for the DirectoryExists function}
      Code := GetFileAttributes(PChar(FProcCurrentDir));
      Result := (Code <> -1) and (FILE_ATTRIBUTE_DIRECTORY and Code <> 0);
      if Result then
        PCurDir := PChar(FprocCurrentDir)
      else
        ErrorEvent(ord(emInvalidDir) * -1, cErrMsg[emInValidDir]);
    end;
end;

function TExFile.GetErrorCode: integer;
begin
  Result := FErrorCode;
end;

procedure TExFile.ListMaint;
var
  i: Integer;
begin
  for i := Pred(HandLst.count) downto 0 do
    if not GetExitCode(PProcList(HandLst.Items[i]).PrhProcess) then
      begin
        Dispose(PProcList(HandLst.Items[i]));
        HandLst.delete(i);
      end;
end;

function TExFile.GetProcStatus: Boolean;
var i: integer;
begin
  Result := GethProcess(FProcFileNamelc, FProcIdentifier, i);
  if Result then
    Result := GetExitCode(PProcList(HandLst.Items[i]).PrhProcess);
end;

function TExFile.ProcStatusFunction(pFileName, PIdentifier: string): Boolean;
var i: integer;
begin
  Result := GethProcess(LowerCase(pFileName), PIdentifier, i);
  if Result then
    Result := GetExitCode(PProcList(HandLst.Items[i]).PrhProcess);
end;

function TExFile.GethProcess(StatProcName, StatProcIdentifier: string;
  var Hidx: Integer): Boolean;
var i: integer;
begin
  Result := false;
  ListMaint;
  Hidx := -1;
  for i := 0 to Pred(HandLst.count) do
    if (PProcList(HandLst.Items[i]).PrName = StatProcName) and
      (PProcList(HandLst.Items[i]).PrProcIdentifier = StatProcIdentifier) then
      begin
        Result := true;
        Hidx := i;
        break;
      end;
end;

function TExFile.AlreadyRunning(TypeWait: TStartType): Boolean;
var i: integer;
begin
  Result := false;
  ListMaint;
  for i := 0 to Pred(HandLst.count) do
    if (PProcList(HandLst.Items[i]).PrStartType = TypeWait) then
      begin
        Result := true;
        break;
      end;
  if Result then
    ErrorEvent(ord(emOnlyOneMeth) * -1, cErrMsg[emOnlyOneMeth]);
end;

function TExFile.GetExitCode(hProcess: THandle): Boolean;
var vExitCode: Dword;
begin
  Result := false;
  if GetExitCodeProcess(hProcess, vExitCode) then
    Result := (vExitCode = STILL_ACTIVE);
end;

function TExFile.GetReturnCode: Integer;
begin
  Result := FExitCode;
end;

function EnumThreadProc(Wnd: HWND; lp: Integer): Bool stdcall;
begin
  Result := true;
  if ((GetParent(Wnd) = 0) and (GetWindowLong(Wnd, GWL_HWNDPARENT) = 0))
    or (HWND(GetWindowLong(Wnd, GWL_HWNDPARENT)) = GetDesktopWindow) then
    begin
      PProcList(lp).PrHWND := Wnd;
      Result := false;
    end;
end;

function TExFile.GetProcInfo(GPIType: TProcInfo; var GPIReturn: Integer):
  Boolean;
var i: Integer;
begin
  Result := GethProcess(FProcFileNamelc, FProcIdentifier, i);
  if Result then
    case GPIType of
      PrhProcess:
        GPIReturn := PProcList(HandLst.Items[i]).PrhProcess;
      PrdwProcessId:
        GPIReturn := PProcList(HandLst.Items[i]).PrdwProcessid;
      PrHWND:
        begin
          Result := not
            EnumThreadWindows(PProcList(HandLst.Items[i]).PrdwThreadId,
            @EnumThreadProc, Integer(HandLst[i]));
          if Result then
            GPIReturn := PProcList(HandLst.Items[i]).PrHWND;
        end;
      PrdwThreadId:
        GPIReturn := PProcList(HandLst.Items[i]).PrdwThreadId;
    end;
end;

function TExFile.CloseProcess: Boolean;
var i: Integer;
begin
  Result := false;
  if GethProcess(FProcFileNamelc, FProcIdentifier, i) then
    Result := not EnumThreadWindows(PProcList(HandLst.Items[i]).PrdwThreadId,
      @EnumThreadProc, Integer(HandLst[i]));
  if Result then
    SendMessage(PProcList(HandLst.Items[i]).PrHWND, WM_CLOSE, 0, 0);
end;

function TExFile.CloseFunction(pFileName, PIdentifier: string): Boolean;
var i: Integer;
begin
  Result := false;
  if GethProcess(LowerCase(pFileName), PIdentifier, i) then
    Result := not EnumThreadWindows(PProcList(HandLst.Items[i]).PrdwThreadId,
      @EnumThreadProc, Integer(HandLst[i]));
  if Result then
    SendMessage(PProcList(HandLst.Items[i]).PrHWND, WM_CLOSE, 0, 0);
end;

function TExFile.KillProcess(RtnCode: Integer): Boolean;
var i: Integer;
begin
  Result := false;
  if GethProcess(FProcFileNamelc, FProcIdentifier, i) then
    Result := TerminateProcess(PProcList(HandLst.Items[i]).PrhProcess,
      RtnCode);
end;

function TExFile.KillFunction(pFileName, PIdentifier: string; RtnCode: Integer):
  Boolean;
var i: Integer;
begin
  Result := false;
  if GethProcess(LowerCase(pFileName), PIdentifier, i) then
    Result := TerminateProcess(PProcList(HandLst.Items[i]).PrhProcess,
      RtnCode);
end;

function TExFile.ExErrorMessage(ExErrorCode: Integer): string;
begin
  if ExErrorCode < 0 then
    if abs(ExErrorCode) < abs(ord(emUnknown)) then
      Result := cErrMsg[TErrMsg(abs(ExErrorCode))]
    else
      Result := cErrMsg[emUnknown]
  else
    Result := SysErrorMessage(ExErrorCode);
end;

function TExFile.CloseThreads: Boolean;
begin
  Result := PulseEvent(hEventHandle);
end;

procedure Register;
begin
  RegisterComponents('Not Visible', [TExFile]);
end;
end.

