unit Main;

interface

uses
  Classes, Controls,
  ExFile,
  Forms,
  ShellLink, StdCtrls, SysUtils,
  Windows;

type
  TFormMain = class(TForm)
    List: TListBox;
    ExFile: TExFile;
    ShellLink: TShellLink;
    procedure FormCreate(Sender: TObject);
  private
    Param: record
      Verbose: Boolean;
      Wait: Boolean;
      WaitInterval: DWord;
      Hide: Boolean;
      ProcessName: string;
      Mask: DWord;
    end;
    procedure SetViewSize;
  end;

var
  FormMain: TFormMain;

implementation

{$R *.dfm}

uses
  Math, Messages,
  shellApi,
  tlhelp32;

procedure Delay(const Interval: DWord);
var
  FirstTickCount: DWord;
begin
  FirstTickCount:= GetTickCount;
  repeat
    Application.ProcessMessages;
  until ((GetTickCount - FirstTickCount) >= Interval);
end;

procedure HideFromTaskBar;
begin
  with Application do begin
    ShowWindow(Handle,SW_HIDE);
    SetWindowLong(Handle,GWL_EXSTYLE,GetWindowLong(Handle,GWL_EXSTYLE) or
                                     WS_EX_TOOLWINDOW and not WS_EX_APPWINDOW or WS_EX_TOPMOST);
    ShowWindow(Handle,SW_SHOW);
  end;
end;

function ParamOption(const Option: string): Boolean;
begin
	Result := FindCmdLineSwitch(Copy(Option,2,Length(Option)),['/','-'],false);
end;

function ParamOptionValue(const Option: string;
													var		Value: string): Boolean;
var
	I: Integer;
  S: string;
begin
	Value := '';
  for I := 1 to ParamCount do begin
    S := ParamStr(I);
		if (S[1] = '/') or (S[1] = '-') then
    	if AnsiCompareStr(Copy(S, 2, Length(Option) - 1),Copy(Option, 2, Length(Option) - 1)) = 0 then begin
        Result := True;
				if S[Length(Option)] = Option[Length(Option)] then
	        Value := Copy(S, Length(Option) + 1,MaxInt);
        Exit;
      end;
  end;
  Result := False;
end;

function FindProcess(Name: string; var ID: DWord): Boolean;
var
 Snap: THandle;
 PE32: TPROCESSENTRY32;
begin
  Result := false;
  Snap := 0;
  try
    try
      Snap := CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
      if Snap <> 0 then begin
        PE32.dwSize := SizeOf(TPROCESSENTRY32);
        if Process32First(Snap,PE32) then begin
          if SameText(ExtractFileName(string(PE32.szExeFile)),Name) then begin
            ID := PE32.th32ProcessID;
            Result := true;
          end
          else while Process32Next(Snap,PE32) and not Result do begin
            if SameText(ExtractFileName(string(PE32.szExeFile)),Name) then begin
              ID := PE32.th32ProcessID;
              Result := true;
            end;
          end;
        end;
      end;
    finally
      CloseHandle(Snap);
    end;
  except
    Result := false;
  end;
end;

procedure TFormMain.SetViewSize;
var
  M,I: Integer;
begin
  List.Canvas.Font := List.Font;
  M := 0;
  for I := 0 to Pred(List.Count) do
    M := Max(M,List.Canvas.TextWidth(List.Items[I]));
  ClientWidth := Max(M + 15,250);
  ClientHeight := (List.ItemHeight + 1) * List.Items.Count;
end;

procedure TFormMain.FormCreate(Sender: TObject);
var
  ID: DWord;
  Handle: THandle;
  FirstTickCount: DWord;
  Found,Error: Boolean;
  ProcessMask,SystemMask: Dword;
  S: string;
begin
  HideFromTaskBar;
  Handle := 0;
  if ParamOption('/?') or (ParamCount = 0) then begin
    List.Items.Add('');
    List.Items.Add('To launch a Process and change its Affinity Mask:');
    List.Items.Add('');
    List.Items.Add('    pam /l:<exe file> /m:<mask hex value> [/p:{I|N|H|R}] [/v]');
    List.Items.Add('    pam /l:<link file> /m:<mask hex value> [/p:{I|N|H|R}] [/v]');
    List.Items.Add('');
    List.Items.Add(' /p option to set the Process Priority to Idle, Normal, High or Realtime.');
    List.Items.Add(' /v option to run in verbose mode.');
    List.Items.Add('');
    List.Items.Add(' Example:');
    List.Items.Add('');
    List.Items.Add('    pam /l:"C:\winnt\shortcut to notepad.exe.lnk" /p:I /m:1 /v');
    List.Items.Add('');
    List.Items.Add('');
    List.Items.Add('To change the Affinity Mask of an existing Process:');
    List.Items.Add('');
    List.Items.Add('    pam /p:<process name> /m:<mask hex value> [/w[:<wait time>]] [/v]');
    List.Items.Add('');
    List.Items.Add(' /w option to wait indefinitely for the Process.');
    List.Items.Add(' /w:<wait time> option to wait for the Process for the specified time (seconds).');
    List.Items.Add(' /v option to run in verbose mode.');
    List.Items.Add('');
    List.Items.Add(' Example:');
    List.Items.Add('');
    List.Items.Add('    pam /p:notepad.exe /m:2 /w:10 /v');
    SetViewSize;
    Visible := true;
    Exit;
  end;
  with Param do begin
    Error := false;
    ParamOptionValue('/l:',S);
    S := LowerCase(S);
    if S <> '' then begin
      ProcessName := '';
      if FileExists(S) then begin
        if SameText(ExtractFileExt(S),'.lnk') then begin
          List.Items.Add(Format('Target link file: %s',[S]));
          ShellLink.LinkFile := S;
          ShellLink.Read;
          ExFile.ProcFileName := ShellLink.ProgramFile;
          ExFile.ProcParameters := ShellLink.Arguments;
          ExFile.ProcCurrentDir := ShellLink.WorkingDirectory;
          case ShellLink.WindowState of
            wsNormal: ExFile.WindowType := wtNorm;
            wsMaximized: ExFile.WindowType := wtMaximize;
            wsMinimized: ExFile.WindowType := wtMinimize;
          end;
        end
        else begin
          List.Items.Add(Format('Target exe file: %s',[S]));
          ExFile.ProcFileName := S;
          ExFile.ProcParameters := '';
          ExFile.ProcCurrentDir := GetCurrentDir;
          ExFile.WindowType := wtNorm;
        end;
        ParamOptionValue('/p:',S);
        if S <> '' then begin
          S := UpperCase(S);
          case S[1] of
            'I': ExFile.Priority := prIdle;
            'N': ExFile.Priority := prNormal;
            'H': ExFile.Priority := prHigh;
            'R': ExFile.Priority := prRealTime;
            else
              ExFile.Priority := prNormal;
          end;
        end
        else
          ExFile.Priority := prNormal;
      end
      else begin
        Error := true;
        List.Items.Add('Error: Target Link does not exists');
      end;
    end
    else begin
      ParamOptionValue('/p:',ProcessName);
      if ProcessName = '' then begin
        Error := true;
        List.Items.Add('Error: no link, exe or Process');
      end
      else begin
        ProcessName := LowerCase(ProcessName);
        List.Items.Add(Format('Target Process: %s',[ProcessName]));
      end;
    end;
    if not Error then begin
      ParamOptionValue('/m:',S);
      try
        Mask := StrToInt('$' + S);
        List.Items.Add(Format('Affinity Mask: %8.8xH',[Mask]));
      except
        Error := true;
        List.Items.Add('Error: invalid value for Affinity Mask');
      end;
      if ProcessName <> '' then begin
        Wait := ParamOption('/w');
        if Wait then begin
          WaitInterval := 0;
          List.Items.Add('Wait Time: no limit');
        end
        else begin
          ParamOptionValue('/w:',S);
          if S <> '' then begin
            Wait := true;
            try
              WaitInterval := StrToInt(S);
              if WaitInterval = 0 then
                List.Items.Add('Wait Time: no limit')
              else
                List.Items.Add(Format('Wait Time: %d s',[WaitInterval]));
            except
              Error := true;
              List.Items.Add('Error: invalid value for Wait Time');
            end;
          end;
        end;
      end
      else begin
        Wait := false;
        WaitInterval := 0;
        case ExFile.Priority of
          prIdle: List.Items.Add('Priority: Idle');
          prNormal: List.Items.Add('Priority: Normal');
          prHigh: List.Items.Add('Priority: High');
          prRealTime: List.Items.Add('Priority: Realtime');
        end;
      end;
    end;
    Verbose := ParamOption('/v');
  end;
  List.Items.Add('');
  if not Error and (Param.ProcessName <> '') then begin
    List.Items.Add('Looking for Process ' + Param.ProcessName);
    FirstTickCount:= GetTickCount;
    repeat
      Found := FindProcess(Param.ProcessName,ID);
      Delay(100);
    until Found or not Param.Wait or ((GetTickCount - FirstTickCount) >= Param.WaitInterval);
    if Found then
      List.Items.Add('Process found')
    else begin
      Error := true;
      List.Items.Add('Error: Process not found');
    end;
  end
  else if not Error then begin
    List.Items.Add('Starting Process');
    if ExFile.Execute then
      ExFile.GetProcInfo(PrhProcess,Integer(Handle))
    else begin
      Error := true;
      List.Items.Add('Error: Process start fail');
    end;
  end;
  if not Error then begin
    if Param.ProcessName = '' then
      List.Items.Add('Process started')
    else begin
      Handle := OpenProcess(PROCESS_QUERY_INFORMATION or PROCESS_SET_INFORMATION,false,ID);
      if Handle <> 0 then
        List.Items.Add('Process opened')
      else begin
        Error := true;
        List.Items.Add('Error: Process open failed');
      end;
    end;
  end;
  if not Error then begin
    Error := not GetProcessAffinityMask(Handle,ProcessMask,SystemMask);
    Param.Mask := Param.Mask and SystemMask;
    if not Error then begin
      List.Items.Add(Format('System Affinity Mask: %8.8xH',[SystemMask]));
      List.Items.Add(Format('Old Process Affinity Mask: %8.8xH',[ProcessMask]));
    end
    else
      List.Items.Add('Error: Process Affinity Mask lookup failed');
    if not Error then begin
      if SystemMask = 1 then begin
        Error := true;
        List.Items.Add('Warning: uniprocessor system, Affinity Mask can not be changed');
      end
      else if Param.Mask <> (Param.Mask and SystemMask) then begin
        Param.Mask := (Param.Mask and SystemMask);
        if Param.Mask <> 0 then
          List.Items.Add(Format('Warning: Affinity Mask trimmed to %8.8xH',[Param.Mask]))
        else begin
          Error := true;
          List.Items.Add(Format('Error: Affinity Mask trimmed to %8.8xH',[0]));
        end;
      end;
    end;
  end;
  if not Error then begin
    Error := not SetProcessAffinityMask(Handle,Param.Mask);
    if not Error then
      List.Items.Add(Format('Process Affinity Mask successfully set to %8.8xH',[Param.Mask]))
    else
      List.Items.Add('Error: Process Affinity Mask set failed');
  end;
  if (Handle <> 0) and (Param.ProcessName <> '') then begin
    CloseHandle(Handle);
    List.Items.Add('Process closed');
  end;
  Param.Verbose := Param.Verbose or Error;
  SetViewSize;
  Visible := Param.Verbose;
  if not Param.Verbose then
    Application.Terminate;
end;


end.
