(*

  TMPPrintDialogEx v 02-09-2003<br>

  @author((C)2003 markus stephany, merkes@mirkes.de, all rights reserved.)
  @abstract(TMPPrintDialogEx is an enhanced TPrintDialog which supportes the new Print Dialog style of Windows 2000)
  @lastmod(02-09-2003)

  	 TMPPrintDialogEx is an enhanced version of TPrintDialog that shows the new "Printer Property Page"
     on systems which support it. on older systems the standard TPrintDialog is displayed.<br><br>
     if the new dialog is used, PrintRange can also be set to prCurrentPage to only print one specific
     page.<br><br>


  <h3>history:</h3>
  <p><ul>
  <li>v 02-09-2003: february 09, 2003<br><br>
         - initial release</li>
  </ul></p>

*)

unit MPPrintDialogEx;

interface

uses
  Windows, Messages, SysUtils, Classes, Dialogs;

type
  (* dialog options:<br>
     - poDisableCurrentPage: disables (grays) the current page check box.<br>
     - poDisablePrintToFile: disables (grays) the print to file check box. (applicable only if the poprinttofile flag is set.)<br>
     - poHelp: displays a help button in the dialog. seems not to work, so it's not used.<br>
     - poPageNums: enables the pages radio button, allowing the user to specify a page range.<br>
     - poPrintToFile: displays a print to file check box in the dialog.<br>
     - poSelection: enables the selection radio button, allowing the user to print selected (highlighted) text.<br>
     - poWarning: generates a warning message if the user tries to send a job to an uninstalled printer.

  *)
  TMPPrintDialogExOption = (poPrintToFile, poPageNums, poSelection, poWarning,
    poHelp, poDisablePrintToFile, poDisableCurrentPage);

  // see @link(TMPPrintDialogExOption)
  TMPPrintDialogExOptions = set of TMPPrintDialogExOption;

  (* print range:<br>
     - prAllPages: the all radio button is selected.<br>
     - prCurrentPage: the current page radio button is selected.<br>
     - prSelection: the selection radio button is selected.<br>
     - prPageNums: the pages radio button is selected.


  *)
  TMPPrintDialogExRange = (prAllPages, prSelection, prPageNums, prCurrentPage);

  (*
  	 TMPPrintDialogEx is an enhanced version of TPrintDialog that shows the new "Printer Property Page"
     on systems which support it. on older systems the standard TPrintDialog is displayed.<br><br>
     if the new dialog is used, PrintRange can also be set to prCurrentPage to only print one specific
     page.
  *)
  TMPPrintDialogEx = class(TPrintDialog)
  private
    FHandle: HWND;
    FOptions: TMPPrintDialogExOptions;
    FPrintRange: TMPPrintDialogExRange;
    procedure SetInheritedOptions;
    procedure GetInheritedOptions;
    function GetVersion: string;
    procedure SetVersion(const Value: string);
  public
    // @exclude()
    constructor Create(AOwner: TComponent); override;
    // @exclude()
    function Execute: Boolean; override;
    // @exclude()
    property Handle: HWND read FHandle;
  published
    // see @link(TMPPrintDialogExOption)
    property Options: TMPPrintDialogExOptions read FOptions write FOptions
      default [poDisableCurrentPage];
    // see @link(TMPPrintDialogExRange)
    property PrintRange: TMPPrintDialogExRange read FPrintRange write FPrintRange
      default prAllPages;
    // current component version
    property Version: string read GetVersion write SetVersion stored False;
  end;

procedure Register;

implementation
uses
  Forms, Printers, CommDlg;

procedure Register;
begin
  RegisterComponents('mirkes.de', [TMPPrintDialogEx]);
end;

const
  MPH_PDEVERSION = 'February 09, 2003';

type
  // com object used to catch window messages sent to the Printer Property Sheet dialog
  IPrintDialogCallback = interface(IUnknown)
    ['{5852A2C3-6530-11D1-B6A3-0000F8757BF9}']
    function InitDone: HResult; stdcall;
    function SelectionChange: HResult; stdcall;
    function HandleMessage(hDlg: HWND; uMsg: UINT; wParam: WPARAM; lParam:
      LPARAM; var pResult: LRESULT): HResult; stdcall;
  end;

  TPrintDialogCallback = class(TInterfacedObject, IPrintDialogCallback)
  private
    FDlg: TMPPrintDialogEx;
  public
    constructor Create(Dlg: TMPPrintDialogEx);
    function InitDone: HResult; stdcall;
    function SelectionChange: HResult; stdcall;
    function HandleMessage(hDlg: HWND; uMsg: UINT; wParam: WPARAM; lParam:
      LPARAM; var pResult: LRESULT): HResult; stdcall;
  end;

  { Printer dialog routines }

procedure GetPrinter(var DeviceMode, DeviceNames: THandle);
var
  Device, Driver, Port: array[0..1023] of char;
  DevNames: PDevNames;
  Offset: PChar;
begin
  Printer.GetPrinter(Device, Driver, Port, DeviceMode);
  if DeviceMode <> 0 then
  begin
    DeviceNames := GlobalAlloc(GHND, SizeOf(TDevNames) +
      StrLen(Device) + StrLen(Driver) + StrLen(Port) + 3);
    DevNames := PDevNames(GlobalLock(DeviceNames));
    try
      Offset := PChar(DevNames) + SizeOf(TDevnames);
      with DevNames^ do
      begin
        wDriverOffset := Longint(Offset) - Longint(DevNames);
        Offset := StrECopy(Offset, Driver) + 1;
        wDeviceOffset := Longint(Offset) - Longint(DevNames);
        Offset := StrECopy(Offset, Device) + 1;
        wOutputOffset := Longint(Offset) - Longint(DevNames);
        StrCopy(Offset, Port);
      end;
    finally
      GlobalUnlock(DeviceNames);
    end;
  end;
end;

procedure SetPrinter(DeviceMode, DeviceNames: THandle);
var
  DevNames: PDevNames;
begin
  DevNames := PDevNames(GlobalLock(DeviceNames));
  try
    with DevNames^ do
      Printer.SetPrinter(PChar(DevNames) + wDeviceOffset,
        PChar(DevNames) + wDriverOffset,
        PChar(DevNames) + wOutputOffset, DeviceMode);
  finally
    GlobalUnlock(DeviceNames);
    GlobalFree(DeviceNames);
  end;
end;

function CopyData(Handle: THandle): THandle;
var
  Src, Dest: PChar;
  Size: Integer;
begin
  if Handle <> 0 then
  begin
    Size := GlobalSize(Handle);
    Result := GlobalAlloc(GHND, Size);
    if Result <> 0 then
    try
      Src := GlobalLock(Handle);
      Dest := GlobalLock(Result);
      if (Src <> nil) and (Dest <> nil) then
        Move(Src^, Dest^, Size);
    finally
      GlobalUnlock(Handle);
      GlobalUnlock(Result);
    end
  end
  else
    Result := 0;
end;

procedure CenterWindow(Wnd: HWnd);
var
  Rect: TRect;
  Monitor: TMonitor;
begin
  GetWindowRect(Wnd, Rect);
  if Application.MainForm <> nil then
  begin
    if Assigned(Screen.ActiveForm) then
      Monitor := Screen.ActiveForm.Monitor
    else
      Monitor := Application.MainForm.Monitor;
  end
  else
    Monitor := Screen.Monitors[0];
  SetWindowPos(Wnd, 0,
    Monitor.Left + ((Monitor.Width - Rect.Right + Rect.Left) div 2),
    Monitor.Top + ((Monitor.Height - Rect.Bottom + Rect.Top) div 3),
    0, 0, SWP_NOACTIVATE or SWP_NOSIZE or SWP_NOZORDER);
end;

{ TNewPrintDialog }

type
  //
  //  Page Range structure for PrintDlgEx.
  //
  tagPRINTPAGERANGE = packed record
    nFromPage: DWORD;
    nToPage: DWORD;
  end;
  PPrintPageRange = ^tagPRINTPAGERANGE;
  TPrintPageRange = tagPRINTPAGERANGE;

  //
  //  PrintDlgEx structure.
  //
  tagPDEX = packed record
    lStructSize: DWORD;
    hwndOwner: HWND;
    hDevMode: HGLOBAL;
    hDevNames: HGLOBAL;
    hDC: HDC;
    Flags: DWORD;
    Flags2: DWORD;
    ExclusionFlags: DWORD;
    nPageRanges: DWORD;
    nMaxPageRanges: DWORD;
    lpPageRanges: PPRINTPAGERANGE;
    nMinPage: DWORD;
    nMaxPage: DWORD;
    nCopies: DWORD;
    hInstance: HINST;
    lpPrintTemplateName: LPCSTR;
    lpCallback: IPrintDialogCallback;
    nPropertyPages: DWORD;
    lphPropertyPages: THandle;
    nStartPage: DWORD;
    dwResultAction: DWORD;
  end;
  PPrintDlgEx = ^tagPDEX;
  TPrintDlgEx = tagPDEX;

  TPrintDialogExProc = function(var PrintDlgEx: TPrintDlgEx): HRESULT; stdcall;

const
  PD_CURRENTPAGE = $00400000;
  PD_NOCURRENTPAGE = $00800000;
  PD_EXCLUSIONFLAGS = $01000000;
  PD_USELARGETEMPLATE = $10000000;

  //
  //  Exclusion flags for PrintDlgEx.
  //
  PD_EXCL_COPIESANDCOLLATE = (DM_COPIES or DM_COLLATE);

  //
  //  Define the start page for the print dialog when using PrintDlgEx.
  //
  START_PAGE_GENERAL = $FFFFFFFF;

  //
  //  Result action ids for PrintDlgEx.
  //
  PD_RESULT_CANCEL = 0;
  PD_RESULT_PRINT = 1;
  PD_RESULT_APPLY = 2;

constructor TMPPrintDialogEx.Create(AOwner: TComponent);
begin
  inherited;
  FOptions := [poDisableCurrentPage];
  FPrintRange := prAllPages;
end;

function TMPPrintDialogEx.Execute: Boolean;
const
  PrintRanges: array[TMPPrintDialogExRange] of Integer =
  (PD_ALLPAGES, PD_SELECTION, PD_PAGENUMS, PD_CURRENTPAGE);
var
  PrintDlgRec: TPrintDlgEx;
  DevHandle: THandle;
  hLib: HMODULE;
  DlgProc: TPrintDialogExProc;
  recRanges: TPrintPageRange;
begin
  hLib := LoadLibrary('comdlg32.dll');
  DlgProc := nil;
  if hLib <> 0 then
    DlgProc := GetProcAddress(hLib, 'PrintDlgExA');
  try
    if (hLib <> 0) and Assigned(DlgProc) then
    begin
      Result := False;
      FillChar(PrintDlgRec, SizeOf(PrintDlgRec), 0);
      with PrintDlgRec do
      begin
        lStructSize := SizeOf(PrintDlgRec);
        GetPrinter(DevHandle, hDevNames);
        hDevMode := CopyData(DevHandle);
        Flags := PrintRanges[PrintRange];
        if Collate then
          Inc(Flags, PD_COLLATE);
        if not (poPrintToFile in Options) then
          Inc(Flags, PD_HIDEPRINTTOFILE);
        if not (poPageNums in Options) then
          Inc(Flags, PD_NOPAGENUMS);
        if not (poSelection in Options) then
          Inc(Flags, PD_NOSELECTION);
        if poDisableCurrentPage in Options then
          Inc(Flags, PD_NOCURRENTPAGE);
        if poDisablePrintToFile in Options then
          Inc(Flags, PD_DISABLEPRINTTOFILE);
        if PrintToFile then
          Inc(Flags, PD_PRINTTOFILE);
        if poHelp in Options then
          Inc(Flags, PD_SHOWHELP);
        if not (poWarning in Options) then
          Inc(Flags, PD_NOWARNING);
        lpPageRanges := @recRanges;
        nPageRanges := 1;
        nMaxPageRanges := 1;
        recRanges.nFromPage := FromPage;
        recRanges.nToPage := ToPage;
        nMinPage := MinPage;
        nMaxPage := MaxPage;
        lpCallback := TPrintDialogCallback.Create(self);
        hWndOwner := Application.Handle;
        nStartPage := START_PAGE_GENERAL;

        if HRESULT(TaskModalDialog(@DlgProc, PrintDlgRec)) = S_OK then
        begin
          case PrintDlgRec.dwResultAction of
            PD_RESULT_PRINT, PD_RESULT_APPLY:
              begin
                SetPrinter(hDevMode, hDevNames);
                Collate := Flags and PD_COLLATE <> 0;
                PrintToFile := Flags and PD_PRINTTOFILE <> 0;
                if Flags and PD_SELECTION <> 0 then
                  PrintRange := prSelection
                else if Flags and PD_PAGENUMS <> 0 then
                  PrintRange := prPageNums
                else if Flags and PD_CURRENTPAGE <> 0 then
                  PrintRange := prCurrentPage
                else
                  PrintRange := prAllPages;
                FromPage := recRanges.nFromPage;
                ToPage := recRanges.nToPage;
                if nCopies = 1 then
                  Copies := Printer.Copies
                else
                  Copies := nCopies;
              end;
          else
            if hDevMode <> 0 then
              GlobalFree(hDevMode);
            if hDevNames <> 0 then
              GlobalFree(hDevNames);
          end;
          Result := PrintDlgRec.dwResultAction = PD_RESULT_PRINT;
        end
      end;
    end
    else
    begin
      SetInheritedOptions;
      Result := inherited Execute;
      GetInheritedOptions;
    end;
  finally
    if hLib <> 0 then
      FreeLibrary(hLib);
  end;
end;

{ TPrintDialogCallback }

constructor TPrintDialogCallback.Create(Dlg: TMPPrintDialogEx);
begin
  inherited Create;
  FDlg := Dlg;
end;

function TPrintDialogCallback.HandleMessage(hDlg: HWND; uMsg: UINT;
  wParam: WPARAM; lParam: LPARAM; var pResult: LRESULT): HResult;
begin
  Result := S_FALSE;
  case uMsg of
    WM_INITDIALOG:
      begin
        // huh ???
        CenterWindow(GetParent(GetParent(hDlg)));

        FDlg.FHandle := hDlg;

        if Assigned(FDlg.OnShow) then
          FDlg.OnShow(FDlg);
      end;
    WM_DESTROY:
      begin
        if Assigned(FDlg.OnClose) then
          FDlg.OnClose(FDlg);
      end;
  end;
end;

function TPrintDialogCallback.InitDone: HResult;
begin
  Result := S_FALSE;
end;

function TPrintDialogCallback.SelectionChange: HResult;
begin
  Result := S_FALSE;
end;

procedure TMPPrintDialogEx.GetInheritedOptions;
var
  Opts: TMPPrintDialogExOptions;
begin
  case inherited PrintRange of
    Dialogs.prSelection: PrintRange := prSelection;
    Dialogs.prPageNums: PrintRange := prPageNums;
  else
    PrintRange := prAllPages;
  end;
  Opts := [];
  if Dialogs.poPrintToFile in inherited Options
  then
    Include(Opts, poPrintToFile);
  if Dialogs.poPageNums in inherited Options
  then
    Include(Opts, poPageNums);
  if Dialogs.poSelection in inherited Options
  then
    Include(Opts, poSelection);
  if Dialogs.poWarning in inherited Options
  then
    Include(Opts, poWarning);
  if Dialogs.poHelp in inherited Options
  then
    Include(Opts, poHelp);
  if Dialogs.poDisablePrintToFile in inherited Options
  then
    Include(Opts, poDisablePrintToFile);
  Options := Opts;
end;

function TMPPrintDialogEx.GetVersion: string;
begin
  Result := MPH_PDEVERSION
end;

procedure TMPPrintDialogEx.SetInheritedOptions;
var
  Opts: TPrintDialogOptions;
begin
  case PrintRange of
    prSelection: inherited PrintRange := Dialogs.prSelection;
    prPageNums: inherited PrintRange := Dialogs.prPageNums;
  else
    inherited PrintRange := Dialogs.prAllPages;
  end;
  Opts := [];
  if poPrintToFile in Options
  then
    Include(Opts, Dialogs.poPrintToFile);
  if poPageNums in Options
  then
    Include(Opts, Dialogs.poPageNums);
  if poSelection in Options
  then
    Include(Opts, Dialogs.poSelection);
  if poWarning in Options
  then
    Include(Opts, Dialogs.poWarning);
  if poHelp in Options
  then
    Include(Opts, Dialogs.poHelp);
  if poDisablePrintToFile in Options
  then
    Include(Opts, Dialogs.poDisablePrintToFile);
  inherited Options := Opts;
end;

procedure TMPPrintDialogEx.SetVersion(const Value: string);
begin
  //
end;

end.