{ FAT File System Module, Copyright (C) 2001, 2002 by Alexander Grau }

{ FAT file system structure:

   |----|-------|-------|-------|------------|----------------------------|
   | BS | FAT 1 |  ...  | FAT n | root 12/16 |  data area                 |
   |----|-------|-------|-------|------------|----------------------------|

   example for FAT12:

   1. boot sector                       sector 0 (in partition)
   2. first FAT   (4085 Cluster)
   3. second FAT
   4. root directory (32 Byte/entry)
   5. data area (data cluster)

  Remark: Long filename support has nothing to do with FAT32! FAT32 is only used for big volumes. }

unit diskfs;

interface

uses classes, devices, comctrls, statusdlg;

const
  { user defined FAT Type constants }
  FATnone = 0  ; { not detected }
  FAT12   = 1  ;
  FAT16   = 2  ;
  FAT32   = 3  ;

  { user defined cluster types }
  clusUsed  = 1;
  clusFree  = 2;
  clusBad   = 4;
  clusEOF   = 8;

  attrReadOnly   = 1;               { File Attribute }
  attrHidden     = 2;
  attrSystem     = 4;
  attrVolume     = 8;
  attrSubDir     = 16;
  attrArchive    = 32;
  attrLongName   = attrReadOnly + attrHidden + attrSystem + attrVolume;


type

  PBootSec = ^TBootSec;
  {: Boot Sector and BPB Structure }
  TBootSec = packed record
    BS_jmpBoot     : array[0..2] of byte;
    BS_OEMName     : array[0..7] of char;
   { ;-----now the BPB (BIOS Parameter Block) begins ---------------------------------- }
    BPB_BytesPSec  : word;
    BPB_SecPerClus : byte;
    BPB_RsvdSecCnt : word;
    BPB_NumFATs    : byte;
    BPB_RootEntCnt : word;
    BPB_TotSec16   : word;
    BPB_Media      : byte;
    BPB_FATsz16    : word;
    BPB_SecPerTrk  : word;
    BPB_NumHeads   : word;
    BPB_HiddSec    : longword;
    BPB_TotSec32   : longword;
    {;------now the BPB/boot sector for FAT12/16 differs from the BPB/boot sector for FAT32...}
    BS_DrvNum      : byte;
    BS_Reserved1   : byte;
    BS_BootSig     : byte;
    BS_VolID       : longword;
    BS_VolLab      : array[0..10] of char;
    BS_FilSysType  : array[0..7] of char;
    {; ... Loader Routine follows ...}
  end;


  PBootSec32 = ^TBootSec32;
  {: boot sector structure for FAT32 }
  TBootSec32 = packed record
    BS_jmpBoot     : array[0..2] of byte;
    BS_OEMName     : array[0..7] of char;
   { ;-----now the BPB (BIOS Parameter Block) begins ---------------------------------- }
    BPB_BytesPSec  : word;
    BPB_SecPerClus : byte;
    BPB_RsvdSecCnt : word;
    BPB_NumFATs    : byte;
    BPB_RootEntCnt : word;
    BPB_TotSec16   : word;
    BPB_Media      : byte;
    BPB_FATsz16    : word;
    BPB_SecPerTrk  : word;
    BPB_NumHeads   : word;
    BPB_HiddSec    : longword;
    BPB_TotSec32   : longword;
    { ;------now the BPB/boot sector for FAT32 differs from the BPB/boot sector for FAT12/16... }
    BPB_FATSz32    : longword;
    BPB_ExtFlags   : word;
    BPB_FSVer      : word;
    BPB_RootClus   : longword;
    BPB_FSInfo     : word;
    BPB_BkBootSec  : word;
    BPB_Reserved   : array[0..11] of byte;
    BS32_DrvNum    : byte;
    BS32_Reserved1_: byte;
    BS32_BootSig   : byte;
    BS32_VolID     : array[0..3] of byte;
    BS32_VolLab    : array[0..10] of char;
    BS32_FilSysType: array[0..7] of char;
  end;


  {: FAT32 FSInfo Sector Structure }
  TFSInfo = packed record
     FSI_LeadSig   : longword;
     FSI_Reserved1 : array[0..479] of byte;
     FSI_StrucSig  : longword;
     FSI_FreeCount : longword;
     FSI_NxtFree   : longword;
     FSI_Reserved2 : array[0..11] of byte;
     FSI_TrailSig  : longword;
  end;


  {: 32 Byte Directory Entry Structure }
  PDirEntry = ^TDirEntry;
  TDirEntry = packed record
    DIR_Name         : array[0..10] of byte;
    DIR_Attr         : byte;
    DIR_NTRes        : byte;
    DIR_CrtTimeTenth : byte;
    DIR_CrtTime      : word;
    DIR_CrtDate      : word;
    DIR_LstAccDate   : word;
    DIR_FstClusHI    : word;
    DIR_WrtTime      : word;
    DIR_WrtDate      : word;
    DIR_FstClusLO    : word;
    DIR_FileSize     : longword;
  end;

  PLnDirEntry = ^TLnDirEntry;
  {: Long Name Directory Entry Structure }
  TLnDirEntry = packed record
    DIR_Sig          : byte;
    DIR_LName1       : array[0..4] of word;
    DIR_Attr         : byte;
    DIR_Flags        : byte;
    DIR_ChkSum       : byte;
    DIR_LName2       : array[0..5] of word;
    DIR_First        : word;
    DIR_LName3       : array[0..1] of word;
  end;

  {: Information structure for Reset, Blockread }
  TFATFileInfo = record
    FI_FirstClus: longword; { erster Cluster }
    FI_Cluster  : longword; { aktueller Cluster }
    FI_recpos   : longword; { current record position (0 to filesize-1) }
    FI_size     : longword; { file size }
    FI_LastClus : longword; { last used cluster (for blockwrite) }
  end;


  { File }
  TFATFile = class (TCustomFile)
  public
    attr: byte;
    cluster: longword;
    time, date: word;
    procedure duplicate(dest: TCustomFile); override;
    function Rename(aname: string): boolean; override;
    function GetPath(RelativeToDir: TCustomDirectory): string; override;
    procedure SaveTo(dest: string; FATno: byte; dlg: TStatusDialog); virtual;
    procedure ChangeListViewItem(listitem: TListItem); override;
  end;

  { Directory }
  TFATDirectory = class (TCustomDirectory)
  public
    cluster: longword;
    sector: longword;    // if FAT12/16 root directory, this is the sector (and cluster = 0)
    time, date: word;
    procedure duplicate(dest: TCustomDirectory); override;
    function Rename(aname: string): boolean; override;
    function GetPath(RelativeToDir: TCustomDirectory): string; override;
    procedure AddDirToTree(TreeView: TTreeView; node: TTreeNode; deleted: boolean); override;
    procedure ChangeListViewItem(listitem: TListItem); override;
    procedure AddChildrenToListView(listview: TListView; deleted: boolean); override;
    function CompareChildren(item1, item2: TListItem; useIdx: integer): integer; override;
    procedure ForEachChild(proc: TProcessDirProc; deleted, exclusive, recursive: boolean;
      UserParams: integer); virtual;
    function ChildIsSubDir: boolean; override;
  private
    function SubDirFound(finddeleted: boolean): boolean;
    //function IsValid: boolean;
    procedure AddChildren(deleted, exclusive: boolean);
  end;

  TFATAnalyser = class;  

  {: The FAT drive object }
  TFATdrive = class(TCustomDrive)
  public
    { current options }
    useFAT             : byte;        // use: 0=no FAT, 1=first, 2=second FAT
    skipBadMarkedClus  : boolean;     // skip bad marked cluster?

    BootSec            : TBootSec32; { Boot sector of device (accessed via BootSec/BootSec32 structure) }
    FSInfo            : TFSInfo;     { FSInfo Structure }

    { optional parameters calculated from the boot sector... }
    FATtype           : byte;        { determined FAT type (FAT12, FAT16 or FAT32) }
    TotSec            : longword;    { Total count of sectors }
    BytePerClus       : word;        { Bytes per Cluster }
    RootClus          : longword;
    RootDirSectors    : word;        { Sectors occupied by the root directory }
    FirstRootDirSecNum: longword;
    FATsz             : longword;    { Count of sectors occupied by ONE FAT }
    FATSectors        : longword;    { Sectors occupied by all FATs }
    FirstDataSector   : longword;    { Start of the data region, the first sector of cluster 2 }
    TotDataSec        : longword;    { Data sector count }
    CountOfClusters   : longword;    { Total count of Clusters }

    constructor Create; override;
    destructor Destroy; override;
    function ReadSec(LBA: longint; blocks: word; buf: pointer; ErrorDlg: boolean): boolean; override;
    function MountDrive(quiet: boolean): boolean; override;
    procedure FindLostData(dlg: TStatusDialog); override;
    procedure AddDriveToTree(TreeView: TTreeView); override;
    procedure AddListViewColumns(ListView: TListView); override;
    function FindFiles: boolean; override;
    procedure SaveListViewItems(ListView: TListView); override;
    function SaveFile(afile: TfatFile; dest: string; FATno: byte;
      dlg: TStatusDialog; var overwrite: boolean): integer;
    function SaveDirectory(dir: TfatDirectory; dest: string;
      deleted, exclusive, recursive: boolean; FATno: byte;
      dlg: TStatusDialog; var overwrite: boolean; var fileoverwrite: boolean): integer;

    function CalcDriveInfo(pbs: PBootSec; quiet: boolean): boolean;
    function GetFAT(N: longword; FATno: byte): longword;
    function GetClusterMask(clustype: byte): longword;
    function IsEOF(value: longword): boolean;

    function Cluster2Sec(cluster: longword): longword;
    function Sec2Cluster(sec: longword): longword;

    function ReadDataSec(var cluster: longword; var sector: longword; var clussec: word; xferbuf: pointer): boolean;    
    function GetNextRecord(var cluster: longword; var sector: longword; var recno: word; xferbuf: pointer): boolean;
    function GetLongName(var cluster: longword; var recsec: longword; var recno: word;
      entry: PDirEntry; var longname: string): boolean;
    function FindNextEntry(var cluster: longword; var recsec: longword; var recno: word;
      var entry: TDirentry; var name: string; findDeleted: boolean): boolean;

    function ResetByCluster(cluster, size: longword; var handle: TFATFileInfo): boolean;
    function BlockRead(var handle: TFATFileInfo; count: longword; buf: pointer;  FATno: byte;
      var bytesread: longword): boolean;
    function GetRecPosCluster(var N: longword; recpos: longword; FATno: byte): boolean;
    function Seek(var handle: TFATFileInfo; N: longword; FATno: byte): boolean;
    function filepos(handle: TFATFileInfo): longword;
    function filesize(handle: TFATFileInfo): longword;
    procedure Close(var handle: TFATFileInfo);
  private
    { sector buffers }
    FFATSecBuf         : array[0..512*2 -1] of byte; { 2 Sectors FAT Buffer (2 because FAT12 can bound a sector!) }
    FFATSec            : longword;                  { current FAT sector in buffer }
    FFATchg            : byte;  { 0=FAT not changed / 1=first sector changed, 2=both sectors changed (FAT12 only) }

    FWorkSecBuf        : array[0..511] of byte; { Working Sector Buffer for getnextrecord }
    FWorkSec           : longword;              { current working sector in buffer }

    FDataSecBuf        : array[0..511] of byte; { Data Sector Buffer for Blockread }
    FDataSec           : longword;              { current data sector in buffer }
    FFATAnalyser       : TFATAnalyser;
    procedure CreateStartItems;
  end;


  {: The FAT drive analyser }
  TFATAnalyser = class
  public
    AnalyseDev: TDevice;     // device to analyse
    LastFATsec: longword;    // last FAT sector found
    MaxFATclus: longword;    // last maximum FAT cluster number

    FirstFATsecbuf: pointer; // first FAT sector buffer
    FATstartsec: longword;   // FAT start sector (of all FATs)
    FATendsec: longword;     // FAT end sector   (of all FATs)
    NumberOfFATsFound: byte; // number of FATs found
    OneFATsecsize: longword; // sector size of one FAT
    fattype: byte;           // FAT type found

    FATAreafound: boolean;   // FAT area found?
    DataAreaFound: boolean;  // data area found?
    RootAreaFound: boolean;  // Root area found?

    SecPerClus   : word;     // sectors per cluster
    ClusterCount: longword;  // count of clusters
    FirstDirClus: longword;  // cluster of first directory found
    FirstDirSec: longword;   // sector of first directory found
    FirstDataSec: longword;  // sector of first data cluster (cluster 2)

    succSecValidEntries: byte; // number of succesive sectors with valid entries

    RootSec: longword;       // root directory sector
    RootClus: longword;      // root directory cluster (=0 if not found)
    RootEntCnt: word;        // number of root directory entries
    succSecValidEntriesRoot: byte; // number of successive sectors with valid entries in the root area

    function CnvName(buf: pointer): shortstring;
    function ClusterType(FATtype: byte; value: longword): byte;

    function IsBlankEntry(entry: TDirEntry): boolean;

    function IsBootSecB(p: pointer): boolean;
    function AreBootSecEqualB(p1, p2: pointer): boolean;
    function DirEntriesValidB(buf: pointer; bufsize: word;
      ValidClusterCount, ValidTotalDataSec: longword): boolean;
    function GetNextRecordB(buf: pointer; bufsize: longword; var recno: word; xferbuf: pointer): boolean;
    function IsFATbeginB(buf: pointer; bufsize: longword; FATtype: byte): boolean;
    function IsFATB(buf: pointer; bufsize: longword; var IsFATbegin: boolean; var FATtype: byte;
      var maxclus: longword): boolean;
    function IsDirB(buf: pointer; bufsize: longword; var cluster: longword): boolean;
    function IsRootB(buf: pointer; bufsize: longword): boolean;
    function GetFATB(buf: pointer; bufsize: longword; FATtype: byte; N: longword): longword;

    function IsRootC(drv: TfatDrive; clus, sec: longword): boolean;
    function IsDirC(drv: TfatDrive; clus, sec: longword): boolean;
    function DirEntriesValidC(drv: TfatDrive; clus, sec: longword): boolean;
    function IsDirEmptyC(drv: TfatDrive; clus, sec: longword): boolean;
    function IsFileC(drv: TfatDrive; scanclus: longword; var fileext: string; var datatype: byte): boolean;

    procedure AnalyseSecStart(dev: TDevice);
    procedure AnalyseSecStop;
    function AnalyseSec(dev: TDevice; physsec: longword; buf: pointer; bufsize: longword): boolean;
    procedure AnalyseRestart;

    procedure RebuildBootSec(dev: TDevice; pbs: pbootsec32; FATtype, SecPerclus: word;
      RootEntCnt: word; RootClus: longword;
      OneFATsecSize: longword; NumFATs: byte;
      ClusterCount: longword);
  end;


  {: FAT file stream object, for easy use with text/hex viewer etc. }
  TFATStream = class(TStream)
  private
    FPosition: longint;
    FSize: longint;
    Fcluster: longint;
    Fdrv: TfatDrive;
    FFATno: byte;
    Fhandle: TfatFileInfo;
    procedure SetPosition(offset: longint);
  public
    property position: longint read FPosition write SetPosition;
    property size: longint read Fsize;
    constructor create(drv: TfatDrive; clus, size: longint; FATno: byte);
    destructor destroy; override;
    function read(var buffer; count: longint): longint; override;
    function seek(offset: longint; origin: word): longint; override;
  end;


  function ReplaceDeletedChar(const s: string): string;



implementation

  uses common, windows, sysutils, main, clusdlg, helpers, dialogs, finddlg, dirseldlg, controls, filedet;


function ReplaceDeletedChar(const s: string): string;
begin
  if MainForm.options.DeletedUseMultiByte then
    result:=StringReplace(s, char($e5), MainForm.options.DeletedCharMultibyte, [])
  else
    result:=StringReplace(s, char($e5), MainForm.options.DeletedCharAnsi, []);
end;

// ---------------------------------------------------------------------------
//  TfatAnalyser
// ---------------------------------------------------------------------------

function TFATAnalyser.CnvName(buf: pointer): shortstring;
var
  i: byte;
  ext, name, res: shortstring;
  c: ^char;
begin
  name:=''; ext:='';
  i:=0; c:=buf;
  while (i < 11) do
  begin
    if (c^ <> ' ') then
    begin
      if (i < 8) then name:=name+c^
        else ext:=ext+c^;
    end;
    inc(i);
    inc(c);
  end;
  res:=name;
  if ext <> '' then res:=res+'.'+ext;
  CnvName:=res;
end;


function TFATAnalyser.ClusterType(FATtype: byte; value: longword): byte;
var
  res: byte;
begin
  if value = 0 then res:=clusFree
    else res:=clusUsed;
  begin
    case FATtype of
      FAT12: begin
               if value >= $0ff8 then res:=clusEOF
                 else if value >= $0ff1 then res:=clusBad;
             end;
      FAT16: begin
               if value >= $fff8 then res:=clusEOF
                 else if value >=$fff1 then res:=clusBad;
             end;
      FAT32: begin
               if value >= $0ffffff8 then res:=clusEOF
                 else if value >= $0ffffff1 then res:=clusBad;
             end;
    end;
  end;
  ClusterType:=res;
end;


function TFATAnalyser.IsBootSecB(p: pointer): boolean;
type
  tbuf= array[0..511] of byte;
var
  pbs: pbootsec;
  pbuf: ^tbuf;
  valid: boolean;
  i: integer;
begin
  valid:=false;
  pbs:=p; pbuf:=p;
  if (pbuf^[510]=$55) AND (pbuf^[511]=$aa) AND (pbuf^[0]=$eb) then
  begin
    for i:=8 to 12 do
    begin
      if (pbs^.BPB_BytesPSec) = word(2 SHL i)  then valid:=true;
    end;
  end;
  IsBootSecB:=valid;
end;

function TFATAnalyser.AreBootSecEqualB(p1, p2: pointer): boolean;
type
  pbuf= array[0..511] of byte;
var
  pbs1: pbootsec;
  pbs2: pbootsec;
  buf: ^pbuf;
begin
  result:=FALSE;
  pbs1:=p1;
  pbs2:=p2;

  if pbs1^.BPB_BytesPSec <> pbs2^.BPB_BytesPSec then exit;
  if pbs1^.BPB_SecPerClus <> pbs2^.BPB_SecPerClus then exit;
  if pbs1^.BPB_RsvdSecCnt <> pbs2^.BPB_RsvdSecCnt then exit;
  if pbs1^.BPB_NumFATs <> pbs2^.BPB_NumFATs then exit;
  if pbs1^.BPB_RootEntCnt <> pbs2^.BPB_RootEntCnt then exit;
  if pbs1^.BPB_TotSec16 <> pbs2^.BPB_TotSec16 then exit;
  if pbs1^.BPB_FATsz16 <> pbs2^.BPB_FATsz16 then exit;
  if pbs1^.BPB_SecPerTrk <> pbs2^.BPB_SecPerTrk then exit;
  if pbs1^.BPB_NumHeads <> pbs2^.BPB_NumHeads then exit;
  if pbs1^.BPB_HiddSec <> pbs2^.BPB_HiddSec then exit;
  if pbs1^.BPB_TotSec32 <> pbs2^.BPB_TotSec32 then exit;

  result:=TRUE;
end;

{ checks if entry is blank }
function TFATAnalyser.IsBlankEntry(entry: TDirEntry): boolean;
var
  noblanks: word;
  i: integer;
begin
  result:=FALSE;
  noblanks:=0;
  for i:=0 to sizeof(entry)-1 do
    if byte(pointer(longint(@entry)+i)^) <> 0  then inc(noblanks);
  if noblanks = 0 then result:=TRUE;
end;

function TFATAnalyser.DirEntriesValidC(drv: TfatDrive; clus, sec: longword): boolean;
var
  currclus: longword;
  recsec: longword;
  clussec: word;
  secbuf: array[0..511] of byte;
begin
  if clus = 11393 then
  begin
    //debug('root!', debugHigh);
  end;
  if drv.ReadDataSec(clus, sec, clussec, @secbuf[0]) then
  begin
    currclus:=clus; recsec:=sec;
    result:=DirEntriesValidB(@secbuf[0], 512, drv.CountOfClusters, drv.TotDataSec);
  end else result:=FALSE;
end;


{ sec paramter is only used if clus=0 ! (for FAT12/16) }
function TFATAnalyser.DirEntriesValidB(buf: pointer; bufsize: word;
  ValidClusterCount, ValidTotalDataSec: longword): boolean;
var
  no: integer;
  recno: word;
  entry: TDirEntry;
  shortname: shortstring;
  i: integer;
  entclus: longword;
  longfound: boolean;
  res: boolean;
begin
  recno:=0;
  result:=FALSE; longfound:=FALSE;
  for no:=1 to 16 do
  begin
    res:=GetNextRecordB(buf, bufsize, recno, @entry);
    if (NOT res) then exit;

    // check if first entry used => if not, this is NOT a directory...
    if no = 1 then if entry.DIR_Name[0]=0 then {IsBlankEntry(entry) then} exit;

    with TLnDirEntry(entry) do
      if ( ((DIR_Attr = $0f) AND (DIR_First = 0) AND (DIR_sig AND $40=$40) )
        OR ((DIR_Attr = $0f) AND (DIR_First = 0) AND longfound) ) then
    begin
      { Long file name entry found... }
      longfound:=TRUE;


    end else
    begin
      longfound:=FALSE;
      //debug(cnvname(@entry.dir_name), debugHigh);
      { short file name entry... }
      { check valid short filename (not valid are e.g. : '\', '/', ':', '*', '?', '"', '<', '>', '\') }
      for i:=0 to 10 do
      begin
        if (entry.DIR_Name[i] < $20) then
        begin
          if (i > 0) then exit;  { values less 20hex not legal }
          if entry.DIR_Name[0] <> $05 then exit; { only one case: KANJI lead byte => then deleted is 05hex, not 0E5hex! }
        end else if byte(entry.DIR_Name[i]) IN [$22, $2A, $2B, $2C, $2E, $2F, $3A, $3B, $3C, $3D, $3E, $3F, $5B, $5C, $5D, $7C] then exit;
      end;
      { check cluster value }
      entclus:=entry.DIR_FstClusLO OR (longint(entry.DIR_FstClusHI) SHL 16);
      if (entry.DIR_attr AND attrVolume <> 0) then
      begin
        if entclus <> 0 then exit;        // volume label must be cluster 0
      end else if (entclus < 2) OR (entclus > ValidClusterCount+1) then exit;
      { check attribute }
      if (entry.DIR_Attr AND 192) <> 0 then exit;
      { check file size }
      if longword(entry.DIR_FileSize div 512) > longword(ValidTotalDataSec) then exit;
      { check date / time }
      { write time/date (must be supported) }
      if entry.DIR_WrtTime <> 0 then
      begin
        if (entry.DIR_WrtTime AND 31) > 29 then exit;
        if ((entry.DIR_WrtTime SHR 5) AND 63) > 59 then exit;
        if (entry.DIR_WrtTime SHR 11) > 23 then exit;
      end;
      if entry.DIR_WrtDate <> 0 then
      begin
        if (entry.DIR_WrtDate AND 31 > 31) OR (entry.DIR_WrtDate AND 31 = 0) then exit;
        if (((entry.DIR_WrtDate SHR 5) AND 15) > 12) OR ((entry.DIR_WrtDate SHR 5) AND 15 = 0) then exit;
      end;
      if entry.DIR_CrtTime <> 0 then
      begin
        if (entry.DIR_CrtTime AND 31) > 29 then exit;
        if ((entry.DIR_CrtTime SHR 5) AND 63) > 59 then exit;
        if (entry.DIR_CrtTime SHR 11) > 23 then exit;
      end;
      if entry.DIR_CrtDate <> 0 then
      begin
        if (entry.DIR_CrtDate AND 31 > 31) OR (entry.DIR_CrtDate AND 31 = 0) then exit;
        if (((entry.DIR_CrtDate SHR 5) AND 15) > 12) OR ((entry.DIR_CrtDate SHR 5) AND 15 = 0) then exit;
      end;
      if entry.DIR_LstAccDate <> 0 then
      begin
        if (entry.DIR_LstAccDate AND 31 > 31) OR (entry.DIR_LstAccDate AND 31 = 0) then exit;
        if (((entry.DIR_LstAccDate SHR 5) AND 15) > 12) OR ((entry.DIR_LstAccDate SHR 5) AND 15 = 0) then exit;
      end;
    end;
  end;
  result:=TRUE;
end;


{ There should only be one "file" on the volume that has the volume attribute set, and that file must be in the root directory.
  This name of this file is actually the label for the volume. DIR_FstClusHI and DIR_FstClusLO must always be 0 for the volume label
  (no data clusters are allocated to the volume label file). }
// sec is only used if clus = 0 (for FAT12/16)
function TFATAnalyser.IsRootC(drv: TfatDrive; clus, sec: longword): boolean;
var
  no: integer;
  recno: word;
  currclus: longword;
  entry: TDirEntry;
  volfound: boolean;
  recsec: longword;
  res: boolean;
begin
  currclus:=clus; recsec:=sec;
  recno:=0;
  volfound:=FALSE;
  for no:=1 to (drv.BootSec.BPB_SecPerClus * 16) do
  begin
    res:=drv.GetNextRecord(currclus, recsec, recno, @entry);
    if (NOT res) then break;
    with TLnDirEntry(entry) do
      if NOT (DIR_Attr = $0f) AND (DIR_First = 0) AND (DIR_sig AND $40=$40) then  // no long file name...
    begin
      if (entry.DIR_attr AND attrVolume <> 0) AND (entry.DIR_FstClusHI = 0) AND (entry.DIR_FstClusLO = 0) then
      begin
        if volfound then    // more than one volume label => no root!
        begin
          result:=FALSE;
          exit;
        end else volfound:=TRUE;
      end;
    end;
  end;
  result:=volfound;
end;


function TFATAnalyser.IsDirC(drv: TfatDrive; clus, sec: longword): boolean;
var
  currclus: longword;
  recno: word;
  entry: TDirEntry;
  recsec: longword;
  res: boolean;
begin
  recno:=0; currclus:=clus; recsec:=sec;
  res:=drv.GetNextRecord(currclus, recsec, recno, @entry);
  if (res) AND (entry.DIR_attr AND attrSubDir <>0) AND (CnvName(@entry.DIR_name) = '.') then
  begin
    res:=drv.GetNextRecord(currclus, recsec, recno, @entry);
    if (res) then
    begin
      if (entry.DIR_attr AND attrSubDir <>0) AND (CnvName(@entry.DIR_name) = '..') then
      begin
        result:=TRUE;
        exit;
      end;
    end;
  end;
  result:=FALSE;
end;

{ checks if directory is empty }
function TFATAnalyser.IsDirEmptyC(drv: TfatDrive; clus, sec: longword): boolean;
var
  lfname: string;
  currclus, recsec: longword;
  res: boolean;
  recno: word;
  entry: TDirEntry;
begin
  result:=TRUE;
  recsec:=sec; currclus:=clus;
  recno:=2;
  res:=drv.FindNextEntry(currclus, recsec, recno, entry, lfname, FALSE);
  if res then result:=FALSE;
end;

function TFATAnalyser.IsFileC(drv: TfatDrive; scanclus: longword; var fileext: string; var datatype: byte): boolean;
var
  buf: pointer;
  res: boolean;
begin
  getmem(buf, drv.bootsec.BPB_SecPerClus*512);
  try
    res:=drv.ReadSec(drv.Cluster2Sec(scanclus), drv.bootsec.BPB_SecPerClus, buf, TRUE);
    if res then res:=filedet.IsFile(buf, drv.bootsec.BPB_SecPerClus*512, fileext, datatype)

  finally
    freemem(buf, drv.bootsec.BPB_SecPerClus*512);
  end;
  result:=res;
end;


function TFATAnalyser.GetNextRecordB(buf: pointer; bufsize: longword; var recno: word; xferbuf: pointer): boolean;
var
  recsec: longword;
  res: boolean;
  secentry: word;
begin
  if (recno+1) * 32 <= bufsize then
  begin
    move(pointer(longword(buf)+recno*32)^, xferbuf^, 32);
    inc(recno);
    result:=TRUE;
  end else result:=FALSE;
end;

function TFATAnalyser.IsDirB(buf: pointer; bufsize: longword; var cluster: longword): boolean;
var
  recno: word;
  entry: TDirEntry;
  res: boolean;
  clus: longword;
begin
  recno:=0;
  res:=GetNextRecordB(buf, bufsize, recno, @entry);
  if (res) AND (entry.DIR_attr AND attrSubDir <>0) AND (CnvName(@entry.DIR_name) = '.') then
  begin
    clus:=entry.DIR_FstClusLO OR (longint(entry.DIR_FstClusHI) SHL 16);
    res:=GetNextRecordB(buf, bufsize, recno, @entry);
    if (res) then
    begin
      if (entry.DIR_attr AND attrSubDir <>0) AND (CnvName(@entry.DIR_name) = '..') then
      begin
        cluster:=clus;
        result:=TRUE;
        exit;
      end;
    end;
  end;
  result:=FALSE;
end;

{ There should only be one "file" on the volume that has the volume attribute set, and that file must be in the root directory.
  This name of this file is actually the label for the volume. DIR_FstClusHI and DIR_FstClusLO must always be 0 for the volume label
  (no data clusters are allocated to the volume label file). }
// sec is only used if clus = 0 (for FAT12/16)
function TFATAnalyser.IsRootB(buf: pointer; bufsize: longword): boolean;
var
  no: integer;
  recno: word;
  entry: TDirEntry;
  volfound: boolean;
  res: boolean;
begin
  recno:=0;
  volfound:=FALSE;
  for no:=1 to bufsize DIV 32 do
  begin
    res:=GetNextRecordB(buf, bufsize, recno, @entry);
    if (NOT res) then break;
    with TLnDirEntry(entry) do
      if NOT (DIR_Attr = $0f) AND (DIR_First = 0) AND (DIR_sig AND $40=$40) then  // no long file name...
    begin
      if (entry.DIR_attr AND attrVolume <> 0) AND (entry.DIR_FstClusHI = 0) AND (entry.DIR_FstClusLO = 0) then
      begin
        if volfound then    // more than one volume label => no root!
        begin
          result:=FALSE;
          exit;
        end else volfound:=TRUE;
      end;
    end;
  end;
  result:=volfound;
end;


// checks for FAT beginning
function TFATAnalyser.IsFATbeginB(buf: pointer; bufsize: longword; FATtype: byte): boolean;
const
  media: array[1..8] of byte = ($F0, $F8, $F9, $FA, $FB, $FC, $FD, $FE);
   // $FF is also allowed but used for EOF mark by MS disk driver...
var
  i: integer;
  res: boolean;
begin
  result:=FALSE;
  // check BPB_Media byte value...
  for i:=1 to 8 do
  begin
    case FATtype of
      FAT12: res:= (pwordarray(buf)^[0]=$0F00 OR media[i]);
      FAT16: res:= (pwordarray(buf)^[0]=$FF00 OR media[i]);
      FAT32: res:= (plongarray(buf)^[0]=$0FFFFF00 OR media[i]);
    end;
    if res then
    begin
      result:=TRUE;
      exit;
    end;
  end;
end;

// get FAT entry number N (for specified buf and FAT type)
function TFATAnalyser.GetFATB(buf: pointer; bufsize: longword; FATtype: byte; N: longword): longword;
var
  offset: longword;
  value: longword;
begin
  if Fattype = FAT16 then
    offset:=N*2
  else if FATtype = FAT32 then
    offset:=N*4
  else if FATtype = FAT12 then
    offset:=N+(N div 2);

  if FATtype = FAT16 then
    value:=word((@pbytearray(buf)^[offset])^)
  else if FATtype = FAT32 then
    value:=longword((@pbytearray(buf)^[offset])^) and $0fffffff
  else if FATtype = FAT12 then
  begin
    value:=word((@pbytearray(buf)^[offset])^);
    if (N AND 1 <> 0) then
      value:=value SHR 4
    else value:=value and $0fff;
  end;
  result:=value;
end;

// checks if part of FAT found (and if FAT begins)
function TFATAnalyser.IsFATB(buf: pointer; bufsize: longword; var IsFATbegin: boolean;
  var FATtype: byte; var maxclus: longword): boolean;
var
  i, j: integer;
  clus, lastclus: longword;
  ftype: byte;
  max, succclus, cluscount: longword;
  samecount: longword;
  fbegin: boolean;
  startclus: longword;
begin
  result:=FALSE;
  for ftype := FAT32 downto FAT12 do
  begin
    fbegin:=IsFATbeginB(buf, bufsize,  ftype);
    case ftype of
      FAT12: cluscount:=(2*bufsize + bufsize) DIV 4;     // DIV 1.5 without float
      FAT16: cluscount:=bufsize DIV 2;
      FAT32: cluscount:=bufsize DIV 4;
    end;
    lastclus:=0; succclus:=0; max:=0;
    if NOT fbegin then startclus:=0
      else startclus:=2;
    for i:=startclus to cluscount-1 do
    begin
      clus:=GetFATB(buf, bufsize, ftype, i);
      if ClusterType(ftype, clus) = clusEOF then
      begin
        lastclus:=0;
      end else if ClusterType(ftype, clus) = clusUsed then
      begin
        // check if there are multiple equal cluster numbers...
        samecount:=0;
        for j:=startclus to cluscount-1 do
          if (j <> i) AND (clus = GetFATB(buf, bufsize, ftype, j)) then inc(samecount);
        if samecount > 4 then
        begin
          succclus:=0; break;
        end;

        if clus > max then max:=clus;
        if clus = lastclus + 1 then inc(succclus);
        lastclus:=clus;
      end;
    end;
    if succclus > 16 then
    begin
      result:=TRUE; FATtype:=ftype; maxclus:=max; IsFATbegin:=fbegin;
      exit;
    end;
  end;
end;


//-------- Find lost drives analyser -----------------

// starts FAT sector analyser
procedure TFATAnalyser.AnalyseSecStart(dev: TDevice);
begin
  AnalyseDev:=dev;
  getmem(FirstFATsecbuf, AnalyseDev.BytesPerSec);
  AnalyseRestart;
end;

procedure TFATAnalyser.AnalyseRestart;
begin
  FATAreaFound:=FALSE;
  MaxFATclus:=0;
  LastFATsec:=0;

  FATstartsec:=0;
  FATendsec:=0;
  NumberOfFATsfound:=0;
  OneFATsecsize:=0;

  FirstDataSec:=0;
  FirstDirClus:=0; FirstDirSec:=0;
  DataAreaFound:=FALSE;
  ClusterCount:=0;
  SecPerClus:=0;
  RootAreaFound:=FALSE;
  RootClus:=0;
  RootEntCnt:=0;
  RootSec:=0;
  succSecValidEntries:=0;
  succSecValidEntriesRoot:=0;
end;

// stops FAT sector analyser
procedure TFATAnalyser.AnalyseSecStop;
begin
  freemem(FirstFATsecbuf, AnalyseDev.BytesPerSec);
end;


procedure TFATAnalyser.RebuildBootSec(dev: TDevice; pbs: pbootsec32; FATtype, SecPerclus: word;
  RootEntCnt: word; RootClus: longword;
  OneFATsecSize: longword; NumFATs: byte;
  ClusterCount: longword);
var
  drv: TFATdrive;
  totalsec: longword;
  RootDirSectors: longword;
  DataSectors: longword;
begin
  fillchar(pbs^, sizeof(tbootsec32), 0);
  pbs^.BPB_BytesPSec:=512;
  pbs^.BPB_SecPerClus:=SecPerClus;
  pbs^.BPB_RsvdSecCnt:=1;            // let sector 0 begin one sector before the FAT
  pbs^.BPB_RootEntCnt:=RootEntCnt;

  RootDirSectors:=((pbs^.BPB_RootEntCnt * 32) + (pbs^.BPB_BytesPSec - 1)) div pbs^.BPB_BytesPSec;
  DataSectors:=ClusterCount * pbs^.BPB_SecPerClus;

  if fattype = FAT32 then
  begin
    pbs^.BPB_FATsz16:=0;
    pbs^.BPB_FATSz32:=OneFATSecSize;
  end else
  begin
    pbs^.BPB_FATsz16:=OneFATSecSize;
  end;
  pbs^.BPB_NumFATs:=NumFATs;

  totalsec:=pbs^.BPB_RsvdSecCnt + (pbs^.BPB_NumFATs * OneFATSecSize) + RootDirSectors + DataSectors;

  if totalsec > $FFFF then
  begin
    pbs^.BPB_TotSec16:=0;
    pbs^.BPB_TotSec32:=totalsec;
  end else
  begin
    pbs^.BPB_TotSec16:=totalsec;
  end;

  if (dev.attr AND DEVICE_ATTR_REMOVABLE <> 0) then
    pbs^.BPB_Media:=$F0
  else
    pbs^.BPB_Media:=$F8;

  if fattype = FAT32 then pbs^.BPB_RootClus:=RootClus;
end;


{  |----|-------|-------|-------|------------|----------------------------|
   | BS | FAT 1 |  ...  | FAT n | root 12/16 |  data area                 |
   |----|-------|-------|-------|------------|----------------------------| }

// process new sector by FAT sector analyser
function TFATAnalyser.AnalyseSec(dev: TDevice; physsec: longword; buf: pointer; bufsize: longword): boolean;
var
  maxclus: longword;
  dummy: integer;
  IsFATbegin: boolean;
  fatfound: boolean;
  diff: word;
  clus: longword;
  validentries, dirfound: boolean;
  newdrv: tfatDrive;
begin
  if physsec = 95 then   // part1: FAT1: sec 95-9225     FAT2: sec 9226-18356  clus 2: sec 18357  (clus count 1168442)
  begin
    //hexdump(pbytearray(buf)^, bufsize);
  end;

  if NOT DataAreaFound then
  begin
    fatfound:=IsFATB(buf, bufsize, IsFATbegin, fattype, maxclus);

    if fatfound then
    begin
      // FAT sector found...
      FATareafound:=TRUE;
      if maxclus > MaxFATclus then MaxFATclus:=maxclus;
      if IsFATbegin then
      begin
        // First FAT sector...
        if (FATstartsec=0) then
        begin
          // first time FAT found...
          FATstartsec:=physsec;
          inc(NumberOfFATsFound);
          move(buf^, FirstFATsecbuf^, 512); // remember first FAT sector
        end else
        begin
          // this may be a copy of the first FAT...
          diff:=BytesEqual(buf, FirstFATsecbuf, 512);
          if diff < 16 then
          begin
            // this is a copy of the first one...
            inc(NumberOfFATsFound);
            FATendsec:=physsec-1;
            OneFATsecsize:=FATendsec-FATstartsec+1;
          end;
        end;
      end;

      LastFATSec:=physsec;
    end;
  end;

  if FATareafound then
  begin
    if Clustercount = 0 then
    begin
      case FATtype of
        FAT12: ClusterCount:=OneFATSecsize * 512 DIV 2-2;
        FAT16: ClusterCount:=(2*OneFATSecsize * 512 + OneFATSecsize * 512) DIV 4-2;   // x 1.5 without float...
        FAT32: ClusterCount:=OneFATSecsize * 512 DIV 4-2;
      end;
    end;
    dirfound:=IsDirB(buf, bufsize, clus);
    validentries:=DirEntriesValidB(buf, bufsize, clustercount, clustercount*128);
    if validentries then         // determine successive sector with valid directory entries...
      inc(succSecValidEntries)
    else
      succSecValidEntries:=0;

    if (NOT dirfound) AND (validentries) AND (IsRootB(buf, bufsize)) then
    begin
      // this "may be" the root...
      RootAreaFound:=TRUE;
      RootSec:=physsec;
      succSecValidEntriesRoot:=succSecValidEntries;
    end;

    if dirfound then
    begin
      // data area and directory found...
      DataAreaFound:=TRUE;

      if FirstDirClus = 0 then
      begin
        // first directory found...
        FirstDirClus:=clus;
        FirstDirSec:=physsec;
      end else
      begin
        // second directory found ?...
        if SecPerClus = 0 then
        begin
          SecPerClus:=(physsec-FirstDirSec) DIV (clus-FirstDirClus);
          firstDataSec:=physsec - (clus-2) * SecPerClus;
          if FATendsec=0 then
          begin
            // no FAT copy found before...
            FATendsec:=firstDataSec-1;
            OneFATsecsize:=FATendsec-FATstartsec+1;
          end;

          // --- now FAT and data area are found... ----------
          debug(Format('FAT startsec: %d, endsec: %d', [FATstartsec, FATendsec]), debugHIGH);
          debug(Format('data startsec: %d', [firstDataSec]), debugHIGH);
        end;
      end;
    end;

    if (RootAreaFound) AND (SecPerClus <> 0) then
    begin
      // sector per clusters is known...
      // check if all sectors of the root clusters are successive valid entries...
      RootAreaFound:=(succSecValidEntriesRoot >= (RootSec-firstDataSec) MOD SecPerClus);
    end;

    if (RootAreaFound) AND (SecPerClus <> 0) then
    begin
      // --- now all FAT, data area and root are found... ----------
      RootClus:=(RootSec-firstDataSec) DIV SecPerClus +2;
      debug(Format('root clus: %d'#13, [rootclus]), debugHIGH);

      newdrv:=tfatdrive.create;
      MainForm.DrvList.drives.add(newdrv);        
      RebuildBootSec(dev, @newdrv.bootsec, fattype, secperclus, rootentcnt, rootclus,
        oneFATsecSize, NumberOfFATsFound, ClusterCount);
      newdrv.dev:=dev;
      newdrv.condition := drv_cond_virtual + drv_cond_bootsecRebuild;
      // let the drive begin one sector before the FAT... 
      newdrv.PosBootSec:=FATstartsec-1;
      newdrv.PartOfs:=FATstartsec-1;

      newdrv.CalcDriveInfo(@newdrv.bootsec, TRUE);

      newdrv.PartSectors:=newdrv.TotSec;
      newdrv.name:=newdrv.name + 'DRIVE (without BootSec)';
      newdrv.CreateStartItems;

      AnalyseRestart;          // restart the FAT analyser...
      result:=TRUE;
      exit;
    end;
  end;
  result:=FALSE;
end;


// ---------------------------------------------------------------------------
//  TfatDrive
// ---------------------------------------------------------------------------

constructor TFATdrive.Create;
begin
  FFATAnalyser := TFATAnalyser.create;
end;


destructor TFATdrive.Destroy;
var
   i : Integer;
begin
  if assigned(RootDir) then RootDir.free;
  if assigned(RootDirDeleted) then RootDirDeleted.free;
  if assigned(RootDirLost) then RootDirLost.Free;
  if assigned(RootDirSearched) then RootDirSearched.free;
  FFATAnalyser.free;
end;


{ Read partition Sector (FDD uses only LBA:word) to address buf }
function TFATdrive.ReadSec(LBA: longint; blocks: word; buf: pointer; ErrorDlg: boolean): boolean;
var
  res: boolean;
begin
  res:=dev.ReadSec(PartOfs + LBA, blocks, buf, ErrorDlg);
  result:=res;
end;

(*{ Write partition Sector (FDD uses only LBA:word) to address buf }
function TFATdrive.WritePartSec(drvno: byte; LBA: longint; buf: pointer): boolean;
var
  res: boolean;
begin
  {res:=ExtendedWrite($80+currDrive, Dev[devno].DI_PartOfs + LBA, 1, buf);}
  WritePartSec:=res;
end;*)

{ Converts cluster to a logical sector number }
function TFATdrive.Cluster2Sec(cluster: longword): longword;
begin
  Cluster2Sec:=FirstDataSector + (cluster-2) * BootSec.BPB_SecPerClus;
end;

function TFATdrive.MountDrive(quiet: boolean): boolean;
var
  BS: array[0..511] of byte;
  res: boolean;
begin
  res:=false;
  if ReadSec(PosBootSec-PartOfs, 1, @BS, true) then
  begin
    Hexdump(BS[0], sizeof(BS));
    move(BS, BootSec, sizeof(TBootSec32));
    res:=CalcDriveInfo(@bootsec, quiet);

    CreateStartItems;
  end;
  result:=res;
end;

procedure TFATdrive.CreateStartItems;
var
  item: TFATdirectory;
begin
  // create TFatDirectory objects for 'root' and 'root deleted'
  // add root dir...
  item:=TFATDirectory.create;
  item.cluster:=RootClus;
  item.sector:=FirstRootDirSecNum;
  item.name:='Root';
  item.expanded:=false;
  item.flags:=0;
  item.drive:=self;
  RootDir:=item;

  // add deleted dirs...
  item:=TFATDirectory.create;
  item.cluster:=RootClus;
  item.sector:=FirstRootDirSecNum;
  item.name:='Deleted';
  item.expanded:=false;
  item.flags:=0;
  item.drive:=self;
  RootDirDeleted:=item;

  // lost dir and searched dir have both children already assigned (so there will not be executed 'AddChildren'!)
  // add lost dir...
  item:=TFATDirectory.create;
  item.cluster:=0;
  item.sector:=0;
  item.name:='Lost';
  item.expanded:=false;
  item.flags:=0;
  item.drive:=self;
  item.children:=TList.create;
  RootDirLost:=item;

  // add searched dir...
  item:=TFATDirectory.create;
  item.cluster:=0;
  item.sector:=0;
  item.name:='Searched';
  item.expanded:=false;
  item.flags:=0;
  item.drive:=self;
  item.children:=TList.create;
  RootDirSearched:=item;
end;


{ Calculate Logical Drive Information from Boot Sector }
function TFATdrive.CalcDriveInfo(pbs: PBootSec; quiet: boolean): boolean;
var
  res: boolean;
  i: longword;
  maxclus: longword;
  value: longword;
  quick_formatted: boolean;
begin
  res:=true;
  try
  with pbootsec32(pbs)^ do
  begin
    { Determine count of sectors occupied by the root directory... }
    RootDirSectors:=((BPB_RootEntCnt * 32) + (BPB_BytesPSec - 1)) div BPB_BytesPSec;
    if (BPB_FATsz16 <> 0) then
      FATsz:=BPB_FATsz16
    else
      FATsz:=BPB_FATsz32;

    { Determine count of sectors occupied by all FATs... }
    FATSectors:=FATsz * BPB_NumFATs;

    { Compute start of data region, the first sector of cluster 2... }
    FirstDataSector := BPB_RsvdSecCnt + (BPB_NumFATs * FATsz) + RootDirSectors;
    if (BPB_TotSec16 <> 0) then
      TotSec:=BPB_TotSec16
    else
      TotSec:=BPB_TotSec32;

    { Determine count of data sectors }
    TotDataSec:=TotSec - FirstDataSector;
    { Determine count of clusters ... }
    CountofClusters:=TotDataSec div BPB_SecPerClus;

    { FAT type detection... }
    if CountofClusters < 4085 then
      FATtype:=FAT12
    else if CountofClusters < 65525 then
      FATtype:=FAT16
    else FATtype:=FAT32;

    { Calculate Bytes per Cluster... }
    BytePerClus:=BPB_BytesPSec * BPB_SecPerClus;
    if (NOT quiet) AND (BytePerClus > 32768) then
      messagebox(0, pchar('bad parameter in boot sector: bytes per sector ('+Inttostr(BytePerClus)+') > 32768 !'),
       'Information', mb_iconinformation + mb_ok);

    if FATtype = FAT32 then
    begin
      if (NOT quiet) AND (BPB_RootClus > countofclusters) then
        messagebox(0, pchar('bad parameter in boot sector: root directory cluster ('+Inttostr(BPB_RootClus)+') > count of clusters !'),
          'Information', mb_iconinformation + mb_ok)
        else if (NOT quiet) AND (BPB_RootClus < 2)  then
          messagebox(0, pchar('bad parameter in boot sector: root directory cluster ('+Inttostr(BPB_RootClus)+') < 2 !'),
            'Information', mb_iconinformation + mb_ok);
      FirstRootDirSecNum:=FirstDataSector + longword(BPB_RootClus-2) * longword(BPB_SecPerClus); // -> sometimes Integer Overflow!!
    end
    else
      FirstRootDirSecNum:=BPB_RsvdSecCnt + (BPB_NumFATs * FATsz);

    if FATType = FAT32 then RootClus:=BPB_RootClus
      else RootClus:=0;

    FFATsec:=0;
    FWorkSec:=0;
    FDataSec:=0;
    useFAT:=1;

    if FATtype = fat32 then name:=bs32_vollab
      else name:=pbootsec(@BootSec)^.bs_vollab;

    // determine the drive's condition (if it is quick-formatted, etc.) ...
    quick_formatted:=TRUE;
    maxclus:=16;
    if maxclus > CountOfClusters+1 then maxclus:=CountOfClusters+1;
    for i:=2 to maxclus do
    begin
      value:=GetFAT(i, 0);
      if value <> RootClus then
      begin
        if (FfatAnalyser.ClusterType(FATtype, value) = clusUsed) then
        begin
          quick_formatted:=FALSE;
          break;
        end;
      end;
    end;
    if quick_formatted then
    begin
      condition:=condition OR drv_cond_quickformatted;
      useFAT:=0;
    end;

  end;
   except
   else
     if NOT quiet then
       messagebox(0, 'Bad parameters for calculating logical drive information from Boot Sector', 'Error', mb_iconinformation + mb_ok);
     res:=false;
  end;
  result:=res;
end;




{-----------------------------------------------------------------------------}
{ ------ FAT manipulaton functions -------------------------------------------}
{-----------------------------------------------------------------------------}


{ Obtain next link number from FAT (12/16/32), FATno: 0=first FAT, 1=2nd FAT, ...}
{ Note: the CountofClusters value is exactly the count of data clusters starting at cluster 2.
  The maximum valid cluster number for the volume is CountofClusters + 1, and the "count of clusters including the two reserved
  clusters" is CountofClusters + 2. }
function TFATdrive.GetFAT(N: longword; FATno: byte): longword;
var
  nextentry, sector, byteoffset: word;
  i: word;
  value: longword;
  FATOffset: longword;
  nFATSec: longword;
  EntOffset: word;
begin
 try
  with bootsec do
  begin
    if (N < 2) OR (N > countofclusters+1) then
    begin
      Debug(format('GetFAT error, wrong parameter: N %d (valid range is 2-%d)',
        [N, Countofclusters+1]), debugHigh);
      result:=0;
      exit;
    end;

    if Fattype = FAT16 then
      FATOffset:=N*2
    else if FATtype = FAT32 then
      FATOffset:=N*4
    else if FATtype = FAT12 then
      FATOffset:=N+(N div 2);

    nFATSec:=BPB_RsvdSecCnt + (FATno * FATsz) + (FATOffset div BPB_BytesPSec);
    if FFATsec <> nFatSec then
    begin
      ReadSec(nFATsec, 1, @FFATSecBuf, true);
      FFATsec:=nFATsec;
    end;

    EntOffset:=FATOffset MOD BPB_BytesPSec;
    if (FATtype = FAT12) AND (EntOffset = BPB_BytesPSec - 1) then
    begin
      { avoid sector boundary }
      ReadSec(nFATsec+1, 1, @FFATSecBuf[512], true);
    end;

    {  FAT sector has been read now... }
    if FATtype = FAT16 then
      value:=word(addr(FFATSecBuf[EntOffset])^)
    else if FATtype = FAT32 then
      value:=longword(addr(FFATSecBuf[EntOffset])^) and $0fffffff
    else if FATtype = FAT12 then
    begin
      value:=word(addr(FFatSecBuf[EntOffset])^);
      if (N AND 1 <> 0) then
        value:=value SHR 4
      else value:=value and $0fff;
    end;
    GetFAT:=value;
  end;
 except
   else GetFAT:=0;
 end;
end;


{ Get cluster mask for special cluster type (bad, end of file or used) }
function TFATdrive.GetClusterMask(clustype: byte): longword;
var
  mask: longword;
begin
  if clustype = clusFree then mask:=0;

  begin
    case FATtype of
      FAT12: begin
               if clustype = clusEOF then mask:=$0fff
                 else if clustype = clusBad then mask:=$0ff7;
             end;
      FAT16: begin
               if clustype = clusEOF then mask:=$ffff
                 else if clustype = clusBad then mask:=$fff7;
             end;
      FAT32: begin
               if clustype = clusEOF then mask:=$0fffffff
                 else if clustype = clusBad then mask := $0ffffff7;
             end;
    end;
  end;
  GetClusterMask:=mask;
end;


{ checks a cluster value for EOC (End Of Clusterchain) }
function TFATdrive.IsEOF(value: longword): boolean;
var
  res: boolean;
begin
  res:=false;
    if (FATtype=FAT12) AND (value >= $0ff8) then res:=true
      else if (FATtype=FAT16) AND (value >=$fff8) then res:=true
        else if (FATtype=FAT32) AND (value >= $0ffffff8) then res:=true;
  IsEOF:=res;
end;

function TFATdrive.Sec2Cluster(sec: longword): longword;
begin
  with BootSec do
    Sec2Cluster:=(sec-FirstDataSector) div BPB_SecPerClus+2;
end;

{ sector paramter is only used if cluster=0 (for FAT12/16) }
function TFATdrive.ReadDataSec(var cluster: longword; var sector: longword; var clussec: word; xferbuf: pointer): boolean;
var
  recsec: longword;
  res: boolean;
  secentry: word;
begin
 try
  res:=true;
  if (cluster > countofclusters+1) then
  begin
    Debug(format('ReadDataSec error, wrong parameter: cluster %d (valid range is 0-%d)',
      [cluster, countofclusters+1]), debugHigh);
    res:=false;
  end;

  if (res) then
  begin
    with BootSec do
    begin
      if cluster = 0 then { use sector paramter ? }
      begin
        if clussec > 0 then { read next sector ? }
        begin
          clussec:=0;
          inc(sector);
        end;
        recsec:=sector;
      end
      else begin { none root }
        if clussec >= BPB_SecPerClus then  { nchsten Cluster laden? }
        begin
          if useFAT > 0 then cluster:=GetFAT(cluster, useFAT-1)
            else inc(cluster);
          if IsEOF(cluster) then res:=false     { EndOfDir! }
            else begin
              clussec:=0;
            end;
        end;
        if res then recsec:=cluster2sec(cluster) + clussec;
      end;

      if res then
      begin
        if FWorkSec <> recsec then
        begin
          ReadSec(recsec, 1, @FworkSecBuf, true);
          FWorkSec:=recsec;
        end;
        move(Fworksecbuf[0], xferbuf^, BPB_BytesPSec);
        inc(clussec);
      end;
    end;
  end;
  result:=res;
 except
   else result:=false;
 end;
end;


{ sector paramter is only used if cluster=0 (for FAT12/16) }
function TFATdrive.GetNextRecord(var cluster: longword; var sector: longword; var recno: word; xferbuf: pointer): boolean;
var
  recsec: longword;
  res: boolean;
  secentry: word;
begin
 try
  res:=true;
  if (cluster > countofclusters+1) then
  begin
    Debug(format('GetNextRecord error, wrong parameter: cluster %d (valid range is 0-%d)',
      [cluster, countofclusters+1]), debugHigh);
    res:=false;
  end;

  if (res) then
  begin
    with BootSec do
    begin
      if cluster = 0 then { use sector paramter ? }
      begin
        if recno >= (BPB_BytesPSec div 32) then { read next sector ? }
        begin
          recno:=0; inc(sector);
        end;
        recsec:=sector + (recno*32 div BPB_BytesPSec);
      end
      else begin { none root }
        {writeln('recno: ',recno, ' sec: ',recsec);}
        if recno >= (BytePerClus div 32) then  { nchsten Cluster laden? }
        begin
          {write('getnxtrec cluster:',cluster);}
          if useFAT > 0 then cluster:=GetFAT(cluster, useFAT-1)
            else inc(cluster);
          {writeln('getnxtrec getfatcluster:',cluster);
          if iseof(cluster) then writeln('iseof!');
          readkey;}
          if IsEOF(cluster) then res:=false     { EndOfDir! }
            else begin
              {write('curr cluster is: ',cluster, ' next cluster is: ');
              writeln(cluster);
              readkey;}
              recno:=0;
            end;
        end;
        if res then recsec:=cluster2sec(cluster)+(recno*32 div BPB_BytesPSec);
      end;

      if res then
      begin
        if FWorkSec <> recsec then
        begin
          {writeln('neue sec:',recsec);}
          ReadSec(recsec, 1, @FworkSecBuf, true);
          FWorkSec:=recsec;
        end;
        secentry:=recno mod (BPB_BytesPSec div 32);
        move(Fworksecbuf[secentry*32], xferbuf^, 32);
        inc(recno);
      end;
    end;
  end;
  GetNextRecord:=res;
 except
   else GetNextRecord:=false;
 end;
end;


{ ermittelt langen Dateinamen an aktueller Position und gibt zugehrigen kurzen Dateieintrag
  (DirEntry) zurck }
function TFATdrive.GetLongName(var cluster: longword; var recsec: longword; var recno: word;
  entry: PDirEntry; var longname: string): boolean;
var
  oldcluster: longword;
  oldrecno: longword;
  chksum, sum: byte;
  res: boolean;
  slotno: byte;
  N: byte;
  uname: array[0..255] of word;
  len: byte;
  p: byte;
  lnentry: Tdirentry;
  i: integer;
  multibytebuf: array[0..510] of byte;
begin
  res:=true;

  with PLnDirEntry(entry)^ do
  begin
    if (DIR_Attr = $0f) AND (DIR_First = 0) AND (DIR_sig AND $40=$40) then
    begin
      { ---- long name dir entry found ------- }
      //N:=DIR_Sig AND (NOT $40);
      N:=18; // maximum is slotno=18
      slotno:=N; p:=N*13; len:=0;
      chksum:=DIR_chksum;
      { ---- check for valid long name entry... ------- }
      while (DIR_attr = $0f) AND (DIR_first = 0) {AND (DIR_Sig AND (NOT $40)=slotno)}
        AND (slotno > 0) AND (DIR_chksum = chksum) do
      begin
        if p >= 13 then dec(p,13);
        inc(len,13);
        move(DIR_LName1, uname[p+0], 10);
        move(DIR_LName2, uname[p+5], 12);
        move(DIR_LName3, uname[p+11], 4);

        dec(slotno);
        { ----- get next directory record .... ------- }
        if slotno > 0 then res:=GetNextRecord(cluster, recsec, recno, entry);
      end;

      { ------- long filename is build - get following short -------- }
      //if slotno > 0 then MessageBox(0, 'slot error', pchar('diskfs error (cluster:'+inttostr(cluster)
      //  +', recno:'+inttostr(recno)+')'), mb_ok);

      if 1=1 {slotno = 0} then
      begin
        //res:=GetNextRecord(cluster, recno, entry);
        if res then
        begin
          { generate checksum... }
          sum:=0;
          for i:=0 to 10 do
            sum:=((sum AND 1) SHL 7) OR ((sum AND $fe) SHR 1) + entry^.DIR_Name[i];

          { convert Unicode to Ansi/Multibyte... }
          if 1=1 {sum = chksum} then
          begin
            //WideCharToMultiByte(CP_ACP, 0, @uname[p], len, @multibytebuf[0],
            //  sizeof(multibytebuf), nil, nil);
            longname:=Unicode2ASCII(@uname[p], len);
            //longname:=StrPas(@multibytebuf[0]);
          end else
          begin { checksum error }
            //MessageBox(0, 'chksum error', pchar('diskfs error (cluster:'+inttostr(cluster)
            //  +', recno:'+inttostr(recno)+')'), mb_ok);
            res:=false
          end;
        end;
      end else res:=false; { invalid slot }
    end else res:=false; { invalid slot }
  end;
  if not res then
  begin
    {cluster:=oldcluster;
    recno:=oldrecno;}
  end;
  GetLongName:=res;
end;


{ sucht nchsten Dateieintrag (incl. lange Dateinamen) }
function TFATdrive.FindNextEntry(var cluster: longword; var recsec: longword; var recno: word;
  var entry: Tdirentry; var name: string; findDeleted: boolean): boolean;
var
  res: boolean;
begin
  with entry do
  begin
    repeat
      res:=GetNextRecord(cluster, recsec, recno, @entry);
      if DIR_Name[0]=0 then  { keine weiteren Eintrge mehr }
        res:=false
    until (NOT res) OR ((DIR_NAME[0] <> $e5) AND (NOT findDeleted)) OR (findDeleted);

    if res then
    begin
      { Eintrag gefunden... }
      if (DIR_ATTR=$0f) then  { langer Dateieintrag? }
      begin
        res:=GetLongName(cluster, recsec, recno, @entry, name);
        if NOT res then // illegal long filename... so take short one...
        begin
          res:=true;
          name:=FFATanalyser.CnvName(@DIR_Name);
        end;
      end else name:=FFATanalyser.CnvName(@DIR_Name);
    end;
  end;
  FindNextEntry:=res;
end;



{ ------------------------------------------------------------------------ }
{ --------------- file accessing routines  ------------------------------- }
{ ------------------------------------------------------------------------ }

function TFATdrive.resetByCluster(cluster, size: longword; var handle: TFATFileInfo): boolean;
var
  res: boolean;
  h: byte;
begin
  res:=true;
  try
    with handle do
    begin
      FI_Cluster:=cluster;
      FI_FirstClus:=FI_Cluster;
      FI_recpos:=0;
      FI_size:=size;
    end;
  except
    else res:=false;
  end;
  result:=res;
end;




// read bytes from file (from current position)
// input: file handle <handle>, number of bytes to read <count>, FAT number <FATno>: 0=noFAT, 1=first FAT, 2=second FAT,...  
// output: buffer data <buf>, bytes read <bytesread> (TRUE if success)
function TFATdrive.blockread(var handle: TFATFileInfo; count: longword; buf: pointer;  FATno: byte;
  var bytesread: longword): boolean;
var
  startsec, sec: longword;
  secofs: word;
  rest: longword;
  blockrest: word;
  res: boolean;
  csec: byte;
  transferlen: word;
  nextclus: longword;
begin
 try
  res:=false;
  with handle do
  begin
    if (FI_recpos < FI_size) then
    begin
      rest:=count; bytesread:=0;
      if FI_recpos + rest > FI_size then { ggf. zu lesende Bytes durch Dateigre korrigieren }
        rest:=FI_size - FI_recpos;

      startsec:=FI_recpos div 512;
      secofs:=FI_recpos MOD 512;
      csec:=startsec MOD bootsec.BPB_SecPerClus;        { Startsektor im Cluster ermitteln... }
      blockrest:=512-secofs;                    { restliche Bytes in Sektor berechnen... }

      repeat
        transferlen:=blockrest;
        if rest < blockrest then transferlen:=rest; { ggf. zu lesende Bytes nach unten korrigieren }

        sec:=Cluster2Sec(FI_Cluster) + csec;
        if sec <> FDataSec then
        begin
          FDataSec:=sec;
          res:=ReadSec(FDataSec, 1, @FDataSecBuf[0], true);
        end;

        move(FDataSecBuf[secofs], buf^, transferlen);
        secofs:=0;                             { copy from start next time }
        inc(FI_recpos, transferlen);
        inc(bytesread, transferlen);
        inc(longword(buf), transferlen);        { Zielpointer erhhen }
        if (transferlen=blockrest) then        { last byte of sector? (nchster Sektor ?) }
        begin
          inc(csec);
          if csec = bootsec.BPB_SecPerClus then        { Ende des Sektors => nchster Cluster }
          begin
            csec:=0;
            if FATno > 0 then
            begin
              // use FAT...
              if not IsEOF(FI_cluster) then      { im nchsten Cluster weiterlesen... }
              begin
                nextclus:=GetFAT(FI_cluster, FATno-1);
                if (nextclus > 2) AND (nextclus <=CountOfClusters)
                  then FI_cluster:=nextclus;
              end;
            end else
            begin
              // do not use FAT - read clusters sequentially...
              if (FI_cluster <=CountOfClusters) then inc(FI_cluster);
            end;
          end;
        end;
        blockrest:=512;                        { full block next time }
        dec(rest, transferlen);
      until (rest=0) OR (NOT res);

      res:=true;
    end else
    begin
      //messagebox(0, pchar('FI_size:'+inttostr(FI_size)), 'info', mb_OK);
      if (FI_recpos = FI_size) then  { Spezialfall fr Dateilnge 0 und Zeiger auf Dateiende }
        res:=true;
      bytesread:=0;
    end;
  end;
  result:=res;
 except
   else result:=false;
 end;
end;


// returns cluster of a given record position of a file by seeking from
// the start to the specified record position (or until end of file)
// input: start cluster <N>, record position <recpos>, FAT number <FATno>: 0=no FAT, 1=fst FAT, 2=2nd FAT, ...
// ouput: cluster of record position <N> (<FALSE> if invalid)
function TFATdrive.GetRecPosCluster(var N: longword; recpos: longword; FATno: byte): boolean;
var
  clus, count: longword;
  res: boolean;
begin
  res:=false;
  clus:=N;
  count:=recpos div BytePerClus; // amount of clusters
  while (count > 0) AND (FfatAnalyser.ClusterType(FATtype, clus) <> clusEOF) do
  begin
    if FATno = 0 then clus:=clus+1
      else clus:=GetFAT(clus, FATno-1);
    dec(count);
  end;
  if (count = 0) then // not EOF => success?
  begin
    N:=clus;
    res:=true;
  end;
  result:=res;
end;


// file seek
// input: absolute seek position <N> (zero based), FAT number <FATno>: 0=no FAT, 1=fst FAT, 2=2nd FAT, ...
// ouput: <TRUE> if success
function TFATdrive.seek(var handle: TFATFileInfo; N: longword; FATno: byte): boolean;
var
  r: boolean;
  clus: longword;
begin
  r:=false;
  with handle do
  begin
    if FI_FirstClus <> 0 then
    begin
      if n < FI_size then
      begin
        clus:=FI_FirstClus;
        if GetRecPosCluster(clus, N, FATno) then
        begin
          FI_cluster:=clus;
          FI_recpos:=n;
          r:=true;
        end;
      end;
    end;
  end;
  result:=r;
end;

function TFATdrive.filepos(handle: TFATFileInfo): longword;
begin
  with handle do
    filepos:=FI_firstclus
end;

function TFATdrive.filesize(handle: TFATFileInfo): longword;
begin
  with handle do
    filesize:=FI_size;
end;

procedure TFATdrive.close(var handle: TFATFileInfo);
begin
  //dispose(handle);
end;

{ ----------------------------------------------------------------------------}

function clusmapGetBit(p: pointer; bitpos: longint): byte;
var
  byteno: longint;
  pmap: ^byte;
begin
  byteno:=bitpos SHR 3;
  pmap:=p;
  inc(longint(pmap), byteno);
  clusmapGetBit:=(pmap^ SHR (bitpos AND 7)) AND 1;
end;


procedure clusmapSetBit(p: pointer; bitpos:longint);
var
  byteno: longint;
  pmap: ^byte;
begin
  byteno:=bitpos SHR 3;
  pmap:=p;
  inc(longint(pmap), byteno);
  pmap^:=pmap^ OR (1 SHL (bitpos AND 7));
end;


procedure TFATdrive.FindLostData(dlg: TStatusDialog);
var
  code: integer;
  startclus, endclus: longword;
  scanclus, dirclus, fileclus, clus, filecluscount: longword;
  scansec: longword;
  pclusmap: ^pointer;
  cluscount: longword;
  res: boolean;
  dirfound, rootfound: boolean;
  ditem: TFATDirectory;
  fitem: TFATfile;
  sext, slast_ext: string;
  lastfileclus: longword;
  i: integer;
  datatype: byte;
  newfile: boolean;
  lostdircount, lostfilecount: integer;


  procedure CheckLostList(clus, sec: longword);
  var
    j: integer;
    ditem: TFATDirectory;
  begin
    for j:=0 to RootDirLost.children.count-1 do
    begin
      // if "clus" is in the lost list, throw it out of the lost list (because it will be found recursively)
      if assigned(RootDirLost.children.items[j]) then
      begin
         ditem:=RootDirLost.children.items[j];
         if ((ditem.cluster > 0) AND (ditem.cluster = clus)) OR ((ditem.cluster=0) AND (ditem.sector=sec)) then
         begin
          ditem.free;
          RootDirLost.children[j]:=NIL;
         end;
      end;
    end;
    //move all non-nil items to the front of the Items array and reduce the Count property to the number of items actually used.
    RootDirLost.children.Pack;
    // To free up the memory for the unused entries
    RootDirLost.children.Capacity:=RootDirLost.children.Count;
  end;


  procedure scandir(clus, sec: longword; checklist: boolean);
  var
    recno: word;
    fclus, oldclus, fsec: longword;
    entry: Tdirentry;
    lfname: string;
    res: boolean;
  begin
    fclus:=clus; fsec:=sec;
    if dlg.UserCancel then exit;
    if dlg.TimeForUserUpdate then
    begin
      if fclus <> 0 then dlg.UpdateStatus('', Format('Scanning cluster %D (sub-directory)',[fclus]), '', '', '')
        else dlg.UpdateStatus('', Format('Scanning sector %D (sub-directory)',[fsec]), '', '', '');
      MainForm.ProcessMessages;
    end;

    // first mark it as scanned...
    clusmapSetBit(pclusmap, fclus);
    recno:=0; oldclus:=fclus;

    // if this actually scanned one is in the lost list, throw it out of the lost list (because it will be found recursively)
    if checklist then CheckLostList(fclus, fsec);

    repeat
      res:=FindNextEntry(fclus, fsec, recno, entry, lfname, true);

      if fclus <> 0 then // do not use cluster bitmap if using sector paramter!
      begin
        if (fclus < startclus) OR (fclus > endclus) then break; // valid user cluster range specified?
        if oldclus <> fclus then  // cluster changed?
        begin
          // mark new cluster as scanned...
          clusmapSetBit(pclusmap, fclus);
          dlg.ProgressStepIt;
          oldclus:=fclus;
        end;
      end;

      if (res) then
      begin
        if (entry.DIR_attr AND attrSubDir <>0) then
        begin
          if (lfname <> '.') AND (lfname <> '..') then // directory found?
          begin
            // scan directory recursively...
            dirclus:=entry.DIR_FstClusLO OR (longint(entry.DIR_FstClusHI) SHL 16);
            if (dirclus >= startclus) AND (dirclus <=endclus)
              AND (FfatAnalyser.IsDirC(self, dirclus, 0))
              AND (clusmapGetBit(pclusmap, dirclus) = 0) then  // valid user cluster range specified?
              begin
                scandir(dirclus, 0, TRUE);
                dlg.ProgressStepIt;
              end;
          end;
        end else if (entry.DIR_attr AND (attrSubDir+attrVolume) =0) then
        begin
          // file found... mask file clusters as scanned...
          fileclus:=entry.DIR_FstClusLO OR (longint(entry.DIR_FstClusHI) SHL 16);
          CheckLostList(fileclus, 0);
          if useFAT > 0 then
          begin
            // FAT valid...
            repeat
              // mark file cluster as scanned...
              if (fileclus >= startclus) AND (fileclus <= endclus)
                AND (clusmapGetBit(pclusmap, fileclus) = 0) then  // valid user cluster range specified?
              begin
                clusmapSetBit(pclusmap, fileclus);
                dlg.ProgressStepIt;
              end;
              fileclus:=GetFAT(fileclus, useFAT-1);
            until (FfatAnalyser.ClusterType(FATtype, fileclus) IN [clusEOF, clusFree]);
          end else
          begin
            if (fileclus >= startclus) AND (fileclus <= endclus)
              AND (clusmapGetBit(pclusmap, fileclus) = 0) then  // valid user cluster range specified?
            begin
              clusmapSetBit(pclusmap, fileclus);
              dlg.ProgressStepIt;
            end;
          end;
        end;
      end;
    until not res;
  end;


begin

  ClusterDialog.totalclus:=CountOfClusters;
  ClusterDialog.startclus:=2;
  ClusterDialog.endclus:=CountOfClusters+1;
  ClusterDialog.bytesPerClus:=BytePerClus;

  if ClusterDialog.execute then
  begin
    RootDirLost.DeleteChildren;
    RootDirLost.expanded:=FALSE;
    startclus:=clusterDialog.startclus;
    endclus:=clusterDialog.endclus;

    MainForm.enabled:=false;
    dlg.ProgressMax:=endclus-startclus+1;
    dlg.ProgressStep:=1;
    dlg.ProgressUpdateInterval:=500;
    dlg.SetStatus('Find lost data - Please wait...', '', '', '', '', true, true);
    dlg.Show;

    //messagebox(0, 'info', pchar(inttostr(startclus)+'-'+inttostr(endclus)), mb_ok);
    // allocate cluster map...
    cluscount:=endclus+1; { 0 (for the root) to endclus }
    try
      getmem(pclusmap, cluscount div 8+1);
      fillchar(pclusmap^, cluscount div 8+1, 0); // set all clusters as not scanned

      // first scan all directories that can be accessed by the root
      // to mark them as being examined (so that they cannot be found as lost)!
      dev.useCache(FALSE);
      scandir(RootClus, FirstRootDirSecNum, FALSE);  // scan root directory recursively...
      dev.useCache(TRUE);

      if NOT dlg.UserCancel then
      begin
        { first try to find root directory on FAT12/16... }
        if FATtype <> FAT32 then
        begin
          // scan region between last FAT and data area (there will be the root!)
          for scansec:=(bootsec.BPB_RsvdSecCnt + FATsectors) to FirstDataSector do
          begin
            if (NOT FfatAnalyser.IsDirC(self, 0, scansec) AND FfatAnalyser.DirEntriesValidC(self, 0, scansec)) then
            begin
              // now we can assume a lost FAT12/16 root!
              ditem:=TFATDirectory.create;
              ditem.cluster:=0;
              ditem.sector:=scansec;
              ditem.name:='sector '+Inttostr(scansec)+' (root)';
              ditem.expanded:=false;
              ditem.flags:=item_lost;
              ditem.drive:=self;
              ditem.parent:=RootDirLost;
              ditem.Children:=NIL;
              RootDirLost.Children.add(ditem);
              // scan directory recursively...
              dev.UseCache(FALSE);
              scandir(0, scansec, FALSE);
              dev.UseCache(TRUE);
            end;
            if dlg.TimeForUserUpdate then
            begin
              dlg.UpdateStatus('', Format('Scanning sector %D',[scansec]),
                Format('Lost directories found: %D',[rootdirlost.children.count]), '', '');
              MainForm.ProcessMessages;
              if dlg.userCancel then break;
            end;
          end;
        end;

        if NOT dlg.UserCancel then
        begin
          slast_ext:=''; lastfileclus:=0;
          // scan clusters and find files/directories...
          for scanclus:=startclus to endclus do
          begin
            // if cluster isn't scanned yet...
            if clusmapGetBit(pclusmap, scanclus) = 0 then
            begin
              // first mark it as scanned...
              clusmapSetBit(pclusmap, scanclus);

              {if clustertype(GetFAT(clus, 0)) IN [clusUsed, clusEOF] then}
              if NOT ((SkipBadMarkedClus) AND (GetFAT(scanclus, useFAT) = clusBad)) then
              begin
                dirfound:=FfatAnalyser.IsDirC(self, scanclus, 0) AND (NOT FfatAnalyser.IsDirEmptyC(self, scanclus, 0));
                rootfound:=(NOT dirfound AND FfatAnalyser.DirEntriesValidC(self, scanclus, 0) AND FfatAnalyser.IsRootC(self, scanclus, 0));
                if (dirfound) OR (rootfound) then
                begin
                  // now we can assume a lost directory!
                  ditem:=TFATDirectory.create;
                  ditem.cluster:=scanclus;
                  ditem.name:='cluster '+Inttostr(scanclus);
                  if rootfound then ditem.name:=ditem.name + ' (root)';
                  ditem.expanded:=false;
                  ditem.flags:=item_lost;
                  ditem.drive:=self;
                  ditem.parent:=RootDirLost;
                  ditem.Children:=NIL;
                  RootDirLost.Children.add(ditem);
                  // scan directory recursively...
                  dev.UseCache(FALSE);
                  scandir(scanclus, 0, FALSE);
                  dev.UseCache(TRUE);
                end else if (MainForm.Options.FindLostFiles) AND (FfatAnalyser.IsFileC(self, scanclus, sext, datatype)) then
                begin
                  // now we can assume a lost file!
                  newfile:=TRUE;
                  if (lastfileclus = scanclus-1) AND (slast_ext = sext) then
                  begin
                    // the file has same extensions as last found file and is one cluster behind...
                    if datatype = data_stream then newfile:=FALSE;  // is it the same file data type as the cluster before?
                  end;
                  if newfile then
                  begin
                    // now we can assume a NEW lost file!
                    fitem:=TfatFile.create;
                    fitem.cluster:=scanclus;
                    fitem.name:='cluster '+Inttostr(scanclus)+'.'+sext;
                    fitem.size:=MainForm.options.LostDefaultSize;
                    fitem.attr:=attrArchive;
                    fitem.flags:=item_lost;
                    fitem.drive:=self;
                    fitem.parent:=RootDirLost;
                    RootDirLost.Children.add(fitem);
                    // mark new file clusters as scanned...
                    (*fileclus:=fitem.cluster; filecluscount:=(MainForm.options.LostDefaultSize + BytePerClus-1) div BytePerClus;
                    repeat
                       if (fileclus >= startclus) AND (fileclus <= endclus) then  // valid user cluster range specified?
                         clusmapSetBit(pclusmap, fileclus);
                      if useFAT > 0 then
                      begin
                        // FAT valid...
                        clus:=GetFAT(fileclus, useFAT-1);
                        if clus = 0 then inc(fileclus)
                          else fileclus:=clus;
                      end else inc(fileclus);
                      dec(filecluscount);
                      if IsFile(fileclus, s2ext) then
                        if s2ext <> sext then filecluscount:=0;
                    until (ClusterType(fileclus) IN [clusEOF, clusFree]) OR (filecluscount=0);*)
                  end;
                  slast_ext:=sext;
                  lastfileclus:=scanclus;
                end;
              end;
              dlg.ProgressStepIt;
            end;

            if dlg.TimeForUserUpdate then
            begin
              // determine count of lost directories/files...
              lostdircount:=0;
              lostfilecount:=0;
              for i:=0 to rootdirlost.children.Count-1 do
              begin
                if TObject(rootdirlost.Children.Items[i]) is TfatDirectory then inc(lostdircount)
                  else if TObject(rootdirlost.Children.Items[i]) is TfatFile then inc(lostfilecount);
              end;
              dlg.UpdateStatus('', Format('Scanning cluster %D',[scanclus]),
                Format('Lost directories found: %D',[lostdircount]),
                Format('Lost files found: %D',[lostfilecount]), '');
              MainForm.ProcessMessages;
               if dlg.userCancel then break;
            end;
          end;
        end;
      end;

    finally
      dev.UseCache(FALSE);
      freemem(pclusmap, cluscount div 8+1);
      MainForm.enabled:=true;
      dlg.Hide;
    end;
  end;
end;

//: shows ListView columns
procedure TfatDrive.AddListViewColumns(ListView: TListView);
var
  NewColumn: TListColumn;
begin
  NewColumn := ListView.columns.add;
  NewColumn.Caption := 'Name';
  NewColumn.width:=150;

  NewColumn := ListView.columns.add;
  NewColumn.Caption := 'Size';
  NewColumn.width:=60;

  NewColumn := ListView.columns.add;
  NewColumn.Caption := 'Date modified';
  NewColumn.width:=100;

  NewColumn := ListView.columns.add;
  NewColumn.Caption := 'Cluster';
  NewColumn.width:=60;

  NewColumn := ListView.columns.add;
  NewColumn.Caption := 'Condition';
  NewColumn.width:=60;

  NewColumn := ListView.columns.add;
  NewColumn.Caption := 'Type';
  NewColumn.width:=100;
end;

//: Generiert die Startknoten fr den File Browser
procedure TFATDrive.AddDriveToTree(TreeView: TTreeView);
var
  node: TTreeNode;
  ditem: TFATDirectory;
  fitem: TFATfile;
  j: integer;
begin
  with MainForm do
  begin
    // add root directory...
    RNode := TreeView.Items.AddChildObject(
       nil, 'Root', RootDir);
    RNode.ImageIndex := idxFolderClosed;
    RNode.SelectedIndex := idxFolderClosed;
    RNode.HasChildren:=TFATDirectory(RootDir).SubDirFound(FALSE);
    RNode.Expanded:=FALSE;

    // add deleted dirs...
    DNode := TreeView.Items.AddChildObject(
       nil, 'Deleted', RootDirDeleted);
    DNode.ImageIndex := idxRecycle;
    DNode.SelectedIndex := idxRecycle;
    DNode.HasChildren:=TFATDirectory(RootDir).SubDirFound(TRUE);
    DNode.Expanded:=FALSE;

    // add lost dirs...
    LNode := TreeView.Items.AddChildObject(
         nil, 'Lost', RootDirLost);
    LNode.ImageIndex := idxFolderFound;
    LNode.SelectedIndex := idxFolderFound;
    LNode.HasChildren:=RootDirLost.ChildIsSubDir;
    LNode.Expanded:=FALSE;

    // add searched files/dirs...
    SNode := TreeView.Items.AddChildObject(
         nil, 'Searched', RootDirSearched);
    SNode.ImageIndex := idxFileFound;
    SNode.SelectedIndex := idxFileFound;
    SNode.HasChildren:=RootDirSearched.ChildIsSubDir;
    SNode.Expanded:=FALSE;
  end;
end;


function TFATDrive.FindFiles: boolean;


  // userParams = 0 means: match only deleted directories
  // userParams = 1 means: match all directories
  function processItem(item: TObject; userParams: integer): boolean; stdcall;
  var
    ditem, ditem2: TfatDirectory;
    fitem, fitem2: TfatFile;
  begin
    if (TObject(item) is TCustomFile) then
    begin
      // new file...
      fitem:=TfatFile(item);
      FindDialog.LabelScanning.caption:=AbbreviatePath(fitem.GetPath(NIL), 30);
      //messagebox(0, pchar(fitem.GetPath(NIL)), pchar(fitem.name), mb_ok);
      if MatchWildCard(ReplaceDeletedChar(fitem.name), FindDialog.filename, '?', '*', FALSE) then
      begin
        // file matched...
        FindDialog.AddItem(ReplaceDeletedChar(fitem.name));
        fitem2:=TfatFile.create;
        fitem.duplicate(fitem2);
        fitem2.parent:=MainForm.currDrv.RootDirSearched;
        MainForm.currDrv.RootDirSearched.children.Add(fitem2);
        MainForm.ProcessMessages;
      end;
    end else if (TObject(item) is TCustomDirectory) then
    begin
      // new directory...
      ditem:=TfatDirectory(item);
      FindDialog.LabelScanning.caption:=AbbreviatePath(ditem.GetPath(NIL), 30);
      if (UserParams=0) OR (ditem.flags AND item_deleted <> 0) then
      begin
        //messagebox(0, pchar(ditem.GetPath(NIL)), pchar(ditem.name), mb_ok);
        if MatchWildCard(ReplaceDeletedChar(ditem.name), FindDialog.filename, '?', '*', FALSE) then
        begin
          // directory matched...
          FindDialog.AddItem(ReplaceDeletedChar(ditem.name));
          ditem2:=TfatDirectory.create;
          ditem.duplicate(ditem2);
          ditem2.parent:=MainForm.currDrv.RootDirSearched;
          ditem2.Children:=NIL;
          MainForm.currDrv.RootDirSearched.children.Add(ditem2);
        end;
      end;
      MainForm.ProcessMessages;
    end;
    result:=(NOT FindDialog.UserCancel);
  end;

begin
  if FindDialog.execute then
  begin
    RootDirSearched.DeleteChildren;
    RootDirSearched.expanded:=FALSE;
    (RootDirDeleted as TFATDirectory).ForEachChild(@processItem, true, true, true, 1);
    (RootDirLost as TFATDirectory).ForEachChild(@processItem, false, false, true, 0);
    (RootDir as TFATDirectory).ForEachChild(@processItem, false, false, true, 0);
    FindDialog.hide;
    result:=TRUE;
  end else result:=FALSE;
end;

function TfatDrive.SaveFile(afile: TfatFile; dest: string; FATno: byte;
  dlg: TStatusDialog; var overwrite: boolean): integer;
var
  dlgRes: word;
  sdrv: string; 
begin
  result:=1;
  if (NOT overwrite) AND (FileExists(dest+ReplaceDeletedChar(afile.name))) then
  begin
    dlgRes:=MessageDlg(format('This folder already contains a file called ''%s''.',[dest+ReplaceDeletedChar(afile.name)])+
      #13#13'Would you like to replace the existing file?',
      mtConfirmation, [mbyes, mbno, mball], 0);
    if assigned(dlg) then dlg.ProgressStopTime;
    if dlgRes = mrNo then exit;
    if dlgRes = mrAll then overwrite:=TRUE;
  end;
  sdrv:=GetRootPath(dest);
  if (NOT GetDriveType(pchar(sdrv))=DRIVE_REMOTE) AND (NewDiskFree(sdrv) < afile.size) then
  begin
    MessageDlg(format('Cannot save file %s. There is not enough free space on drive %s.',
      [ReplaceDeletedChar(afile.name), sdrv]), mtError, [mbOK], 0); 
    result:=0;
  end else
    afile.SaveTo(dest, FATno, dlg);
end;


function TfatDrive.SaveDirectory(dir: TfatDirectory; dest: string;
  deleted, exclusive, recursive: boolean; FATno: byte; dlg: TStatusDialog;
  var overwrite: boolean; var fileoverwrite: boolean): integer;
var
  res: boolean;
  discard: boolean;
  i: integer;
  dlgRes: word;
  ditem: TfatDirectory;
begin
  if assigned(dlg) then if dlg.UserCancel then exit;
  if (DirectoryExists(dest+ReplaceDeletedChar(dir.name))) then
  begin
    if (NOT overwrite) then
    begin
      dlgRes:=MessageDlg(format('This folder already contains a folder called ''%s''', [ReplaceDeletedChar(dir.name)])
        +#13#13'If the files in the existing folder have the same name as files you are saving, '
        +'do you want to replace the existing files?',
        mtConfirmation, [mbyes, mbno, mball], 0);
      if assigned(dlg) then dlg.ProgressStopTime;
      if dlgRes = mrNo then
      begin
        fileoverwrite:=FALSE;
        exit;
      end
      else if dlgRes = mrYes then
      begin
        fileoverwrite:=TRUE
      end
      else if dlgRes = mrAll then
      begin
        overwrite:=TRUE; fileoverwrite:=TRUE;
      end;
    end;
  end else
  begin
    {$I-}
    MkDir(dest+ReplaceDeletedChar(dir.name));
    {$I+}
    if IOResult <> 0 then
      MessageDlg('Could not create directory: '+dest+ReplaceDeletedChar(dir.name)+'!', mtWarning, [mbOk], 0);
  end;

  discard:=false;
  if NOT assigned(dir.children) then
  begin
    // children are not created so far, so create them...
    dir.AddChildren(deleted, exclusive);
    discard:=true;
  end;
  if assigned(dlg) then dlg.ProgressStopTime;

  result:=1;
  if assigned(dir.children) then
  begin
    // children are already created
    for i:=0 to dir.children.count-1 do
    begin
     if (TObject(dir.children.items[i]) is TfatFile) then
     begin
       result:=SaveFile(TfatFile(dir.children.items[i]), dest + ReplaceDeletedChar(dir.name) + '\', FATno, dlg, fileoverwrite);
     end else if (recursive) AND (TObject(dir.children.items[i]) is TfatDirectory) then
     begin
       result:=SaveDirectory(TfatDirectory(dir.children.items[i]), dest + ReplaceDeletedChar(dir.name) + '\', deleted,
         exclusive, recursive, FATno, dlg, overwrite, fileoverwrite);
     end;
     if (dlg.UserCancel) OR (result=0) then break;
    end;
    // discard new created children...
    if discard then
    begin
     dir.deleteChildren;
     dir.children.free;
     dir.children:=NIL;
    end;
  end;
end;


procedure TFATDrive.SaveListViewItems(ListView: TListView);
var
  destdir: shortstring;
  i: integer;
  fitem: TfatFile;
  ditem: TfatDirectory;
  FATno: byte;
  finddeleted: boolean;
  listitem: tlistitem;
  fileoverwrite: boolean;
  overwrite, exists: boolean;
  overallsize: longword;
  res: integer;

  function processItem(item: TObject; userParams: integer): boolean; stdcall;
  var
    overallsize: ^longword;
    ditem, ditem2: TfatDirectory;
    fitem, fitem2: TfatFile;
  begin
    overallsize:=pointer(userParams);     // get overallsize reference, otherwise access impossible?!
    if (TObject(item) is TCustomFile) then
    begin
      // new file...
      fitem:=TfatFile(item);
      inc(overallsize^, fitem.size);
    end else if (TObject(item) is TCustomDirectory) then
    begin
      // new directory...
      // ditem:=TfatDirectory(item);
      MainForm.ProcessMessages;
    end;
    result:=(NOT FindDialog.UserCancel);
  end;

begin
  with DirSelectDialog do
  begin
    ComboBoxFAT.clear;
    ComboBoxFAT.items.add('no FAT (consecutive)');
    for i:=1 to BootSec.BPB_NumFATs do
      ComboBoxFAT.items.add('FAT '+inttostr(i));
    if (MainForm.treeview.selected=MainForm.DNode)
      OR (MainForm.treeview.selected.HasAsParent(MainForm.DNode)) then
    begin
      finddeleted:=true;
      ComboBoxFAT.itemindex:=0; // no FAT for undeleted files!!
    end
    else begin
      findDeleted:=false;
      ComboBoxFAT.itemindex:=useFAT;
    end;
  end;

  if DirSelectDialog.Execute then
  begin
    FATno:=DirSelectDialog.FATno;
    destdir:=DirSelectDialog.Directory;
    if destdir[length(destdir)] <> '\' then destdir:=destdir+'\';

    MainForm.enabled:=false;

    // first determine overall size of all files to be saved...
    StatusDialog.SetStatus('Saving files...', 'Preparing for saving...',  '', '', '', true, true);
    StatusDialog.Show;
    overallsize:=0;
    listitem:=ListView.selected;
    for i:=0 to ListView.selcount-1 do
    begin
      if TObject(listitem.data) is TfatFile then
      begin
        fitem:=TfatFile(listitem.data);
        inc(overallsize, fitem.size);
      end else if TObject(listitem.data) is TfatDirectory then
      begin
        ditem:=TfatDirectory(listitem.data);
        ditem.ForEachChild(@processItem, finddeleted, finddeleted, true, integer(@overallsize));
      end;
      listitem:=ListView.GetNextItem(listitem, sdAll, [isSelected]);
      if StatusDialog.userCancel then break;
    end;
    StatusDialog.ProgressMax:=overallsize;
    StatusDialog.ProgressUpdateInterval:=0;

    if NOT StatusDialog.userCancel then
    begin
      // now save all selected files/directories...
      listitem:=ListView.selected;
      overwrite:=FALSE; fileoverwrite:=FALSE;
      for i:=0 to ListView.selcount-1 do
      begin
        if TObject(listitem.data) is TfatFile then
        begin
          fitem:=TfatFile(listitem.data);
          res:=SaveFile(fitem, destdir, FATno, statusdialog, fileoverwrite);
          if res = 0 then break;             // error (e.g. due to missing disk space)
        end else if TObject(listitem.data) is TfatDirectory then
        begin
          ditem:=TfatDirectory(listitem.data);
          res:=SaveDirectory(ditem, destdir, finddeleted, finddeleted, TRUE, FATno, statusdialog,
            overwrite, fileoverwrite);
          if res = 0 then break;             // error (e.g. due to missing disk space)
        end;
        listitem:=ListView.GetNextItem(listitem, sdAll, [isSelected]);
        if StatusDialog.userCancel then break;
      end;
    end;

    MainForm.enabled:=true;
    StatusDialog.Hide;
  end;
end;


// ---------------------------------------------------------------------------
//  TfatDirectory
// ---------------------------------------------------------------------------

procedure TfatDirectory.duplicate(dest: TCustomDirectory);
begin
  inherited duplicate(dest);
  (dest as TfatDirectory).date:=date;
  (dest as TfatDirectory).time:=time;
  (dest as TfatDirectory).sector:=sector;
  (dest as TfatDirectory).cluster:=cluster;
end;

function TFATDirectory.SubDirFound(finddeleted: boolean): boolean;
var
  res, found: boolean;
  lfname: string;
  recno: word;
  entry: Tdirentry;
  clus, recsec: longword;
begin
  found:=false; clus:=cluster; recsec:=sector; recno:=0;
  repeat
    res:=(drive as TFATDrive).FindNextEntry(clus, recsec, recno, entry, lfname, finddeleted);
    if (res) AND (entry.DIR_attr AND attrSubDir <>0) then
    begin
      if (lfname <> '.') AND (lfname <> '..') then
      begin
        found:=true;
      end;
    end;
  until (not res) OR (found);
  SubDirFound:=found;
end;



//:creates the children directory list (children: TList) and adds children (List of TfatDirectory/TfatFile)
procedure TFATDirectory.AddChildren(deleted, exclusive: boolean);
var
  drv: TFATdrive;
  ditem: TFATDirectory;
  fitem: TFATfile;
  res: boolean;
  clus, recsec: longword;
  recno: word;
  entry: Tdirentry;
  lfname: string;
  selfisdir: boolean;
  valid: boolean;
begin
  try
    drv:=(drive as TFATDrive);

    children:=TList.create;
    recno:=0; clus:=cluster; recsec:=sector;
    // if we are a deleted directory check if it is valid (otherwise skip this directory because it's damaged)...
    (*if (flags AND item_deleted <> 0) then
    begin
      if (NOT IsValid) then exit;
    end;*)

    repeat
      try
        res:=drv.FindNextEntry(clus, recsec, recno, entry, lfname, deleted);
        if (res) then
        begin
          if (entry.DIR_attr AND attrSubDir <>0) then
          begin
            if (CompareStr(lfname, '..') <> 0) AND (CompareStr(lfname, '.') <> 0) then
            begin
              valid:=TRUE;
              if (entry.DIR_name[0]=$e5) then
              begin
                valid:=drv.FfatAnalyser.IsDirC(drv, entry.DIR_FstClusLO OR (longint(entry.DIR_FstClusHI) SHL 16), 0);
              end;
              if valid then
              begin
                // directory found...
                ditem:=TFATDirectory.create;
                ditem.drive:=drv;
                ditem.cluster:=entry.DIR_FstClusLO OR (longint(entry.DIR_FstClusHI) SHL 16);
                ditem.sector:=0;
                ditem.name:=lfname;
                ditem.expanded:=false;
                ditem.flags:=0;
                ditem.parent:=self;
                if (entry.DIR_name[0]=$e5) then
                begin
                  ditem.flags:=ditem.flags OR item_deleted;
                  ditem.condition:=rec_cond_poor;
                  if (ditem.cluster >= 2) AND (drv.FfatAnalyser.ClusterType(drv.FATtype, drv.getFAT(ditem.cluster, 0)) = clusFree)
                    then ditem.condition:=rec_cond_good;
                end;
                children.add(ditem);
              end;
            end;
          end else if (res) AND (entry.DIR_attr AND (attrSubDir+attrVolume) =0)
            AND (    ( (NOT deleted)                                         )
                  OR ( (deleted) AND (entry.DIR_name[0]=$e5) AND (exclusive) )
                  OR ( (deleted) AND (not exclusive)                       )    ) then
          begin
            // file found...
            fitem:=TFATfile.create;
            fitem.drive:=drv;
            fitem.cluster:=entry.DIR_FstClusLO OR (longint(entry.DIR_FstClusHI) SHL 16);
            fitem.attr:=entry.DIR_attr;
            fitem.size:=longword(entry.DIR_filesize);
            fitem.time:=entry.DIR_WrtTime;
            fitem.date:=entry.DIR_WrtDate;
            fitem.name:=lfname;
            fitem.parent:=self;
            fitem.flags:=0;
            if (entry.DIR_name[0]=$e5) then
            begin
              fitem.flags:=fitem.flags OR item_deleted;
              fitem.condition:=rec_cond_poor;
              if (fitem.cluster >= 2) AND (drv.FFATAnalyser.ClusterType(drv.FATtype, drv.getFAT(fitem.cluster, 0)) = clusFree)
                then fitem.condition:=rec_cond_good;
            end;
            children.add(fitem);
          end;
        end;
      except
        on E: Exception do;
      end;
    until (not res);

  except
    on E: Exception do if assigned(children) then
    begin
      deleteChildren;
      children.free;
      children:=NIL;
    end;
  end;
end;

{: calls function 'proc' for each child of the directory (recursive if declared)
   if 'proc' returns FALSE the ForEachChild stops immediately }
procedure TFATDirectory.ForEachChild(proc: TProcessDirProc; deleted, exclusive,
  recursive: boolean; UserParams: integer);


  function foreach(dir: TfatDirectory; proc: TProcessDirProc; deleted, exclusive,
    recursive: boolean; UserParams: integer): boolean;
  var
    i: integer;
    cont: boolean;
    discard: boolean;
  begin
    discard:=false;
    if NOT assigned(dir.children) then
    begin
      // children are not created so far, so create them...
      dir.AddChildren(deleted, exclusive);
      discard:=true;
    end;

    if assigned(dir.children) then
    begin
      // children are already created, so call proc for each item
      for i:=0 to dir.children.count-1 do
      begin
        cont:=proc(dir.children.items[i], UserParams);
        if NOT cont then break;
        if (recursive) AND (TObject(dir.children.items[i]) is TCustomDirectory) then
        begin
          cont:=foreach(dir.children.items[i], proc, deleted,
            exclusive, recursive, UserParams);
          if NOT cont then break;
        end;
      end;
      // discard new created children...
      if discard then
      begin
        dir.deleteChildren;
        dir.children.free;
        dir.children:=NIL;
      end;
    end else cont:=TRUE;

    result:=cont;
  end;

begin
  foreach(self, proc, deleted, exclusive, recursive, UserParams);
end;

procedure TFATDirectory.AddDirToTree(TreeView: TTreeView; node: TTreeNode; deleted: boolean);
var
  i, j: Integer;
  CNode: TTreeNode;
  item: TFATDirectory;
  res: boolean;
begin
  if not assigned(node.data) then exit;
  if (expanded) then exit;

  TreeView.items.BeginUpdate;
  try
    try
      if not assigned(children) then AddChildren(deleted, true);

      if assigned(children) then
      begin
        // Add directories to TreeView control... (needs much CPU time?!)
        for i:=0 to children.count-1 do
        begin
          if TObject(children.items[i]) is TFATDirectory then
          begin
            item:=children[i];
            if item is TFATDirectory then
              if (item.flags AND item_deleted <> 0) then
              begin
                CNode := TreeView.Items.AddChildObject(Node,  ReplaceDeletedChar(item.name), item);
                CNode.ImageIndex := idxFolderClosedDel;
                CNode.SelectedIndex := idxFolderClosedDel;
                if {(item.IsValid) AND} (item.SubDirFound(deleted)) then
                  CNode.HasChildren:=true;         // TreeView an dieser Stelle aufklappbar gestalten
              end
              else begin
                CNode := TreeView.Items.AddChildObject( Node, item.name, item);
                CNode.ImageIndex := idxFolderClosed;
                CNode.SelectedIndex := idxFolderClosed;
                if (item.SubDirFound(deleted)) then
                  CNode.HasChildren:=true;         // TreeView an dieser Stelle aufklappbar gestalten
              end;
          end;
        end;
        node.AlphaSort(true);
        expanded:=true;
      end;

    except
      on E: Exception do
      begin
        // if ShowTreeDir fails set directory expanded flag to FALSE
        MessageDlg('Error: '+E.Message, mtError, [mbOK], 0);
        expanded:=false;
      end;

    end;

  finally
    TreeView.Items.EndUpdate;
  end;
end;


{: Compares one pair of children (TListView.Listitem item1 and item2) after index 'useIdx'
 Returns: -1 if item1 < item2,
           1 if item1 > item2
           0 if item1 = item2 }
function TFATDirectory.CompareChildren(item1, item2: TListItem; useIdx: integer): integer;
var
  sr1,sr2 : string;
  number1, number2, code: integer;
  date1, date2: tdatetime;
begin
  if (TObject(item1.data) is TCustomDirectory) AND
    (TObject(item2.data) is TCustomFile) then result:= -1
  else if (TObject(item1.data) is TCustomFile) AND
    (TObject(item2.data) is TCustomDirectory) then result:= 1
  else
  begin
    if useidx = 0 then
    begin
      sr1 := item1.caption;
      sr2 := item2.caption
    end else
    begin
      sr1:= item1.subitems[useidx-1];
      sr2:= item2.subitems[useidx-1];
    end;

    case useidx of
      0, 4, 5: result := lstrcmp(pchar(sr1), pchar(sr2)); { Text vergleichen }
      1,3 : begin { Zahlen vergleichen }
              if sr1 = '' then sr1:='0';
              if sr2 = '' then sr2:='0';
              val(sr1, number1, code);
              val(sr2, number2, code);
              if number1 < number2 then result := -1
                else if number1 > number2 then result := 1
                  else result:=0;
             end;
      2: begin { Datum vergleichen }
           try
             date1:=StrToDate(copy(sr1,1,8));
             date2:=StrToDate(copy(sr2,1,8));
             if date1 < date2 then result := -1
               else if date1 > date2 then result :=1
                 else result :=0;
           except
             on E: Exception do result:=0;
           end;
         end;
    end;
  end;
end;


procedure TfatDirectory.ChangeListViewItem(listitem: TListItem);
begin
  listitem.caption:=ReplaceDeletedChar(name);
  listitem.SubItems[0]:='';
  listitem.SubItems[2]:=inttostr(cluster);  
end;

procedure TFATDirectory.AddChildrenToListView(listview: TListView; deleted: boolean);
var
  i: integer;
  res: boolean;
  ditem: TFATdirectory;
  fitem: TFATfile;
  listitem: tlistitem;
  sFileType: string;
  sysidx: integer;
begin
  listview.hide;
  listview.items.clear;
  listview.show;
  if not assigned(children) then AddChildren(deleted, true);

  if assigned (children) then
  begin
   { Add directories/files to ListView control }
    with listview do
    begin
      MainForm.ListViewBeginUpdate;
      listview.items.BeginUpdate;
      try
        for i:=0 to children.count-1 do
        begin
          if TObject(children.items[i]) is TFATDirectory then
          begin
            // add directory...
            ditem:=children.items[i];
            listitem:=items.add;
            if MainForm.Options.UseSystemIcons then
            begin
              sysidx:=ShellGetFileInfo(fitem.name, FILE_ATTRIBUTE_DIRECTORY, sFileType);
              ListItem.ImageIndex:=sysidx;
            end else
            begin
              if (ditem.flags AND item_deleted <> 0) then // deleted directory ?
                ListItem.ImageIndex:=idxFolderClosedDel
              else
                ListItem.ImageIndex:=idxFolderClosed;
            end;
            listitem.caption:=ReplaceDeletedChar(ditem.name);
            listitem.subitems.add('');
            listitem.subitems.add(fsTimeDate2Str(ditem.time, ditem.date));
            listitem.subitems.add(inttostr(ditem.cluster));
            if (ditem.flags AND item_deleted <> 0) then
            begin
              if ditem.condition = rec_cond_poor then listitem.subitems.add('poor')
                else listitem.subitems.add('good');
            end else listitem.subitems.add('');
            ShellGetFileType(ditem.name, FILE_ATTRIBUTE_DIRECTORY, sFileType);
            listitem.subitems.add(sFileType);
            listitem.data:=ditem;
          end
          else if TObject(children.items[i]) is TFATfile then
          begin
            // add file...
            fitem:=children.items[i];
            listitem:=items.add;
            //GetFileType(fitem.name, sfiletype);
            if (fitem.flags AND item_deleted <> 0) then // deleted file ?
              ListItem.ImageIndex:=idxFileDel
            else begin
              if MainForm.Options.UseSystemIcons then
              begin
                sysidx:=ShellGetFileInfo(fitem.name, FILE_ATTRIBUTE_NORMAL, sFileType);
                ListItem.ImageIndex:=sysidx;
              end
              else
                ListItem.ImageIndex:=idxFile;
            end;
            listitem.caption:=ReplaceDeletedChar(fitem.name);
            listitem.subitems.add(longwordToStr(fitem.size));
            listitem.subitems.add(fsTimeDate2Str(fitem.time, fitem.date));
            listitem.subitems.add(inttostr(fitem.cluster));
            if (fitem.flags AND item_deleted <>0) then
            begin
              if fitem.condition = rec_cond_poor then listitem.subitems.add('poor')
                else listitem.subitems.add('good');
            end else listitem.subitems.add('');
            ShellGetFileType(fitem.name, FILE_ATTRIBUTE_NORMAL, sFileType);
            listitem.subitems.add(sFileType);
            listitem.data:=fitem;
          end;
        end;
        alphaSort;
      finally
        items.EndUpdate;
        MainForm.ListViewEndUpdate;
      end;
    end;
  end;
end;

function TFATDirectory.Rename(aname: string): boolean;
begin
  name:=aname;
  result:=true;
end;


{: Get path string relative to a given directory
  if RelativeToDir is NIL get complete path }
function TFatDirectory.GetPath(RelativeToDir: TCustomDirectory): string;
var
  s: string;
  item: TCustomDirectory;
begin
  s:='';
  item:=self;
  if (item <> RelativeToDir) then s:=ReplaceDeletedChar(name);
  while (assigned(item.parent) AND (item.parent <> RelativeToDir)) do
  begin
    item:=item.parent;
    s:=ReplaceDeletedChar(item.name) + '\' + s
  end;
  result:=s;
end;

{: Is any one of the children a sub-directory? }
function TfatDirectory.ChildIsSubDir: boolean;
var
  i: integer;
begin
  result:=FALSE;
  if NOT assigned(children) then exit;
  for i:=0 to children.count-1 do
    if TObject(children.Items[i]) is TCustomDirectory then
    begin
      result:=TRUE; exit;
    end;
end;


// ---------------------------------------------------------------------------
//  TfatFile
// ---------------------------------------------------------------------------

procedure TfatFile.duplicate(dest: TCustomFile);
begin
  inherited duplicate(dest);

  (dest as TfatFile).attr:=attr;
  (dest as TfatFile).cluster:=cluster;
  (dest as TfatFile).time:=time;
  (dest as TfatFile).date:=date;
  (dest as TfatFile).size:=size;
end;


function TFATfile.Rename(aname: string): boolean;
begin
  name:=aname;
  result:=true;
end;


procedure TfatFile.ChangeListViewItem(listitem: TListItem);
begin
  listitem.caption:=ReplaceDeletedChar(name);
  listitem.SubItems[0]:=longwordToStr(size);
  listitem.SubItems[2]:=inttostr(cluster);
end;

{: Get path string relative to a given directory
  if RelativeToDir is NIL get complete path }
function TFatfile.GetPath(RelativeToDir: TCustomDirectory): string;
var
  s: string;
  fitem: TCustomFile;
  ditem: TCustomDirectory;
begin
  s:='';
  fitem:=(self as TCustomFile);
  if assigned(fitem.parent) AND (fitem.parent <> RelativeToDir) then
  begin
    ditem:=fitem.parent;
    s:=ReplaceDeletedChar(ditem.name);
    while (assigned(ditem.parent) AND (ditem.parent <> RelativeToDir)) do
    begin
      ditem:=ditem.parent;
      s:=ReplaceDeletedChar(ditem.name) + '\' + s;
    end;
  end;
  result:=s;
end;


procedure TfatFile.SaveTo(dest: string; FATno: byte; dlg: TStatusDialog);
const
  bufsize = 65536;
var
 ToF: file;
 res: boolean;
 fshandle: tfatfileinfo;
 bytesread: longword;
 buf: pointer;
 numwritten: integer;
 bytescopied: longword;
 drv: TfatDrive;
begin
  drv:=(drive as TfatDrive);
  if drv.resetByCluster(cluster, size, fshandle) then
  begin
    if assigned(dlg) then
      StatusDialog.UpdateStatus('Saving files...', 'File: '+AbbreviatePath(dest+ReplaceDeletedChar(name), 30), '',  '', '');
    Mainform.processMessages;
    try
      getmem(buf, bufsize);
      assignfile(ToF, dest+ReplaceDeletedChar(name));
      {$I-}
      rewrite(ToF,1);
      {FileSetAttr(destdir+name, attr);
      FileSetDate(ToF, age);}

      bytescopied:=0;
      if size > 0 then
      begin
       repeat
         res:=drv.Blockread(fshandle, bufsize, buf, FATno, bytesread);
         inc(bytescopied, bytesread);
         if assigned(dlg) then
         begin
           dlg.ProgressStep:=bytesread;
           dlg.ProgressStepIt; //(bytescopied/size*100));
         end;
         if res then
         begin
           blockwrite(ToF, buf^, bytesread, numwritten);
         end else begin
           MessageDlg('Could not read source file! (dev:'+drv.dev.name+', cluster:'+inttostr(cluster)
             +', size: '+inttostr(size)+')', mtWarning, [mbOk], 0);
           if assigned(dlg) then dlg.ProgressStopTime;
         end;
       until (numwritten =0) OR (not res) OR (bytesread < sizeof(buf))
         OR (assigned(dlg) AND (dlg.UserCancel));
      end;
      closefile(ToF);
      if (not res) OR (dlg.UserCancel) then erase(ToF);        // delete partially written file
      {$I+}
      if IOResult <> 0 then
      begin
        erase(ToF);                                           // delete partially written file
        MessageDlg('Could not write to destination file: '+dest+ReplaceDeletedChar(name)+'!', mtWarning, [mbOk], 0);
        if assigned(dlg) then dlg.ProgressStopTime;
      end;
      drv.close(fshandle);

    finally
      freemem(buf, bufsize);
    end;
  end else begin
    MessageDlg('Could not open source file! (dev:'+drv.dev.name+', cluster:'+inttostr(cluster)
      +', size: '+inttostr(size)+')', mtWarning, [mbOk], 0);
    if assigned(dlg) then dlg.ProgressStopTime;
  end;
end;


//---------------------------------------------------------------------------
//                TfatStream
//---------------------------------------------------------------------------


constructor TfatStream.create(drv: TfatDrive; clus, size: longint; FATno: byte);
begin
  Fsize:=size;
  Fcluster:=clus;
  Fposition:=0;
  Fdrv:=drv;
  FFATno:=FATno;
  if NOT drv.resetByCluster(clus, size, Fhandle) then
  begin
    // reset failed...
    Fsize:=0;
  end;
end;

destructor TfatStream.destroy;
begin
  Fdrv.close(Fhandle);
end;

function TfatStream.read(var buffer; count: longint): longint;
var
  bytesread: longword;
  res: boolean;
begin
  res:=FDrv.Blockread(Fhandle, count, @buffer, FFATno, bytesread);
  if res then
  begin
    result:=bytesread;
  end else result:=0;
end;

function TfatStream.seek(offset: longint; origin: word): longint;
var
  res: boolean;
begin
  case Origin of
    soFromBeginning : FPosition:=Offset;
    soFromCurrent   : FPosition:=FPosition+Offset;
    soFromEnd       : FPosition:=FSize+Offset;
  end;
  res:=Fdrv.seek(Fhandle, FPosition, FFATno);
  Result:=FPosition;
end;

procedure TfatStream.SetPosition(offset: longint);
begin
  Fdrv.seek(Fhandle, offset, FFATno);
end;


//---------------------------------------------------------------------------
//
//---------------------------------------------------------------------------


end.

