unit QSpiderGraph;


    { ****************************************************************** }
    { v 1.0                                                              }
    {           Delphi (6) unit  --  SpiderGraph with several features...}
    {                                                                    }
    {           Copyright  2004 by Olivier Touzot "QnnO"                }
    {    (http://mapage.noos.fr/qnno/delphi_en.htm  -  qnno@noos.fr)     }
    {                                                                    }
    {              ----------------------------------                    }
    {                                                                    }
    { History :                                                          }
    { v 1.0 : 2004-07-04    First release ;                              }
    { ****************************************************************** }


    //  This component is freeware, but under copyrights that remain mine for
    //  my parts of the code, and original writters for their parts of the code.
    //  This is mainly the case with the computation of a polygon area (which
    //  can be found anywhere on the web).

    //  This unit can be freely used in any application, freeware, shareware
    //  or commercial. However, I would apreciate your sending me an email if
    //  you decide to use it. Of course, you use it under your own and single
    //  responsability. Neither me, nor contributors, could be held responsible
    //  for any problem resulting from the use of this unit.  ;-)

    //  It can also be freely distributed, provided this licence and the copyright 
    //  notice remains within it unchanged, and its html howTo file is distributed
    //  with it too.

interface

uses
  Windows, Messages, SysUtils, Classes, Controls,
  StdCtrls, // TStaticText : the boxes showing lines values near th pointer.
  graphics, // TColors, TBitmap, aso.
  Types,    // TIntegerDynArray. (TIntegerDynArray = Array of Integer;) You could define your own.
  math,     // Min(), Max(); You could define your own min() and max(); too, if you don't use Math elsewhere in your code;
  Forms;    // Only for : Application.ProcessMessages; when flashing a line, but if your app displays a graph, it has at least a form so ...

type

  TQSummitArray   =  Array of TPoint;
  TQSingleArray   =  Array of Single;
  TQBackGround    = (bgTransparent,bgColored,bgTopBottom,bgBottomTop,bgLeftToRight,bgRightToLeft);
  TQTitlePosition = (qtpTop,qtpBottom);
  TQBorderStyle   = (bsNone,bsFlat,bs3D);
  TQHighlightMode = (hmShowPoints, hmWidened, hmColorLine, hmFlashLine);

  TRIVERTEX = packed record
    X, Y : DWORD;
    Red, Green, Blue, Alpha : Word;
  end;

  TQLinesInfoBox  = Record
    box : TStaticText;
    defined : Boolean;
  End;

  TQMouseLineEvt = procedure(Sender: TObject; const lineIndex: integer) of object;

  TQSpiderGraph  = class;                                   // Forward class declaration, needed by both TQSpiderAxe & TQSGLine.

  TQSpiderAxe = Class                                       // Axes definition
  private
    angle,                                                  // angle of this axe compared to zero
    fMin,                                                   // axes min ...
    fMax         : Single;                                  // ... and max values;
    fIndex       : Integer;                                 // index of this axe in the SpiderGraph instance axes array.
    fAxeOwner    : TQSpiderGraph;                           // The graph upon which this axe will show.
    fAutoSizeMin,                                           // shall min and max's values be computed ...
    fAutoSizeMax : Boolean;                                 // ... internally or will they be provided ?
    procedure   SetAutoSizeMin (value: Boolean);
    procedure   SetAutoSizeMax (value: Boolean);
    procedure   SetAutoSzBoth  (value: Boolean);
    function    GetAutoSzBoth  : Boolean ;
    procedure   SetMin         (value: Single);
    procedure   SetMax         (value: Single);
    Constructor CreateOwned    (anOwner: TQSpiderGraph);
  public
    caption : String;
    Destructor  Destroy; override;
    property    autoSizeMin : Boolean   read fAutoSizeMin   write SetAutoSizeMin;
    property    autoSizeMax : Boolean   read fAutoSizeMax   write SetAutoSizeMax;
    property    autoSized: Boolean      read GetAutoSzBoth  write SetAutoSzBoth;
    property    minValue : Single       read fMin           write SetMin;
    property    maxValue : Single       read fMax           write SetMax;
  End;      


  TQSGLine = Class                                          // Lines descriptor
  Private
    fLnOwner    : TQSpiderGraph;                            // The graph upon which this line will show.
    fValues     : TQSingleArray;                            // Line values {Array of Single}
    fPoints     : Array of TPoint;                          // Coordinates of each value
    fHasValues  : Boolean;                                  // False until values are sent
    fPenWidthMem,                                           // Storage needed to reset the boolean below (after highlighting)
    fPenWidth   : Word;                                     // Default pen width for this line.
    fShow       : Boolean;                                  // If false, this line won't be drawn.
    fShowPointsMem,                                         // Storage needed to reset the boolean below (after highlighting)
    fShowPoints : Boolean;                                  // In normal state, or when highlighted
    fColorMem,                                              // Like above. The line color, when not highlighted
    fColor      : TColor;                                   // The line color to be drawn.
    Constructor CreateOwned   (anOwner: TQSpiderGraph);
    procedure   SetPenWidth   (value: Word);
    procedure   SetShowPoints (value: Boolean);
    procedure   SetValues     (Const vals: TQSingleArray);
    procedure   SetColor      (value: TColor);
  Public
    caption     : String;                                   // Line caption
    property    color      : TColor         read fColor        write SetColor;
    property    penWidth   : Word           read fPenWidth     write SetPenWidth;
    property    values     : TQSingleArray  read fValues       write SetValues;
    property    showPoints : Boolean        read fShowPoints   write SetShowPoints;
    property    visible    : Boolean        read fShow         write fShow;
    procedure   Hide;
    procedure   Show;
    Destructor  Destroy; override;
  End;


  TQSpiderGraph = class(TCustomControl)
  private        { Dclarations prives }
    fBackGround    : TQBackGround;                          // transparent, colored, gradient, ...
    fBackgroundColor,                                       // background color, if single color
    fBackGStartColor,                                       // gradient, if background shows a gradient
    fBackGFinalColor : TColor;                              // gradient, if background shows a gradient
    fUsfHeight,                                             // width and height of the component, without the border
    fUsfWidth      : Integer;                               // 
    fBorderStyle   : TQBorderStyle;                         // component's border : none, flat or 3DLight
    fBorderSize    : Integer;                               // internal mem of its size (0<size<2), depending on above
    fGraphRect     : TRect;                                 // the rectangle where the graph itself will be drawn
    fGraphCenter   : TPoint;                                // the graph center's coordinates

    fAxes          : Array of TQSpiderAxe;                  // Axes of the graph ;
    fAxesCount     : Integer;                               // Number of axes; At least 3
    fAxesLen       : Integer;                               // physical length of axes, in bytes
    fAxesMMdefined : Boolean ;                              // Each axe max and min, to be recomputed or not
    fAxesColor     : TColor;                                // Axes and their polygon border color;
    fAxesAutoSized : Boolean;                               // See setter (SetAxesSizeState();) for more...
    fSummits       : TQSummitArray;                         // The axes summits(+1), for polyline() ;
    fSummitsPolygon: TQSummitArray;                         // The axes summits, for polygone() ;
    fAngle         : Single;                                // Angle between two axes;
    fAxesCapFramed : Boolean;                               // Angle between top and an axe ; Depends upon the number of axes...
    fPolygonFill   : Boolean;                               // Fill axes polygon area ?
    fPolygonColor  : TColor;                                // If yes, with what color ?

    fTitleCaption  : String;                                // Title to dispaly
    fTitlePosition : TQTitlePosition ;                      // At top or bottom
    fTitleRect     : TRect;                                 // Needed to place the graph rect

    fLines         : Array of TQSGLine;                     // Lines values collection
    fLinesCount    : Integer;                               // Logically, at least 1;
    fMultiLines    : Boolean;                               // := fLineCount > 1;
    fDfltPenWidth  : Word;                                  // Lines width;
    fShowLnPoints  : Boolean;                               // Lines points. USed to setup new lines.
    fHighlightColor: TColor;                                // Color to be used to highlight line(s)
    fLinesCpnBmp   : TBitmap;                               // The box where lines captions and their colors will be drawn
    fShowLinesCpn  : Boolean;                               // Is this box to be drawn ?
    fLinesCpnBmpdef: Boolean;                               // Forces re-drawing of the bitmap above each time something in the layout changes
    fLinesCpnBmpTrp: Boolean;                               // Lines captions box background transparent or not
    fLinesCpnBmpClr: TColor;                                // Lines captions box backgroud color, if not transparent

    fLinesInfoBoxes: Array of TQLinesInfoBox;               // The box showing lines values, if tracking mouseMoves
    fMBoxParent    : TWinControl;                           // Will be used to draw the mouse info boxes on the graph or on the whole form
    fShowMouseBox  : Boolean;                               // Shall the popup box (above) be shown ?
    fMBoxBackColor : TColor;                                // Back color is shared by all these info boxes;
    fMBoxForColor  : TColor;                                // Possibly common to all these info boxes ... or ignored (see below);
    fMBoxUsesLnClr : Boolean;                               // If True, fMBoxForColor above will be = to it's line's color;

    fTrackMsMoves  : Boolean;                               // Mouse moves global interruptor
    fMouseOnLine   : Boolean;                               // Memory witness to know if things have changed between two moves
    fMouseLineIndex: Integer;                               // The index of the line the mouse is over
    fMouseEnterLine: TQMouseLineEvt;                        // Event to be raised
    fMouseExitLine : TQMouseLineEvt;                        // Event to be raised

  protected      { Dclarations protges }
    procedure   Paint;  Override;
    procedure   Resize; Override;
    procedure   InitSummits;
    procedure   DefineDrawSpace;
    function    GetLinePoint      (iMin,iMax,position,iAngle: Single): TPoint;
    function    ResizeRect        (aRect: Trect; hm: Integer):Trect;
    procedure   SetBorderStyle    (value: TQBorderStyle);
    procedure   SetBackGround     (value: TQBackGround);
    procedure   SetBackGroundColor(value: TColor);
    procedure   SetBackGStartColor(value: TColor);
    procedure   SetBackGFinalColor(value: TColor);
    procedure   SetAxesColor      (value: TColor);
    procedure   SetAxesCount      (value: Integer);
    procedure   SetAxesFramed     (value: Boolean);
    procedure   SetAxesSizeState  (value: Boolean);
    procedure   SetPolygonFill    (value: Boolean);
    procedure   SetPolygonColor   (value: TColor);
    function    GetMinValue       (axeIx: integer; var minVal: single; cnd: Boolean = False): TIntegerDynArray;
    function    GetMaxValue       (axeIx: integer; var maxVal: single; cnd: Boolean = False): TIntegerDynArray;
    procedure   SetTitleCaption   (Value: String);
    procedure   SetTitlePosition  (value: TQTitlePosition);
    function    GetAxesByIndex    (ix: Integer) : TQSpiderAxe;
    function    GetCaptionRect    (aString: String; aCanvas: TCanvas; anAngle: Single;
                                   destinationRect: TRect; inflateBy,Xbase,Ybase: Integer): TRect;
    procedure   DefineLinesInfoBox(lineIx: Integer);
    function    GetLinesCaptionHeight : integer;
    procedure   SetLinesCount     (value: Integer);
    function    GetLinesByIndex   (ix: Integer) : TQSGLine;
    procedure   FlashLine         (targets:TIntegerDynArray);
    procedure   SetDfltPenWidth   (value: Word);
    procedure   SetShowMouseBox   (value: Boolean);
    procedure   SetMBoxForColor   (value: TColor);
    procedure   SetMBoxBackColor  (value: TColor);
    procedure   SetMBoxUsesLnClr  (value: Boolean);
    procedure   SetShowLnPoints   (value: Boolean);
    procedure   SetTrackMsMoves   (value: Boolean);
    procedure   SetShowLinesCpn   (value: Boolean);
    procedure   SetLinesCpnBmpTrp (value: Boolean);
    procedure   SetLinesCpnBmpClr (value: TColor);
    procedure   ResetLinesAppearance;
    procedure   MouseTrack        (Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure   SetMBoxParent     (value : TWinControl);

  public         { Dclarations publiques }
    axesFont                   : TFont;
    titleFont                  : TFont;
    linesCpnFont               : TFont;
    highlightMode              : Set of TQHighlightMode;

    property axes[ix:Integer]  : TQSpiderAxe       read GetAxesByIndex;
    property linesCount        : Integer           read fLinesCount       write SetLinesCount;
    property line              : TQSGLine  Index 0 read GetLinesByIndex;
    property lines[ix:Integer] : TQSGLine          read GetLinesByIndex;
    property mBoxParent        : TWinControl       read fMBoxParent       write SetMBoxParent;

    function    GetBestLineByArea    (maxWanted: Boolean = True) : TIntegerDynArray;
    function    GetBestLineByAxe     (axeIx: Integer; maxWanted: Boolean = True) : TIntegerDynArray;
    procedure   HighlightLineByCrit  (criteria: Integer; maxWanted: Boolean = True);
    procedure   HighlightLineByIndex (indexArray: TIntegerDynArray); overload;
    procedure   HighlightLineByIndex (index: Integer); overload;
    function    AddLine              (Const vals: TQSingleArray) : Integer;
    function    RemoveLine           (lineIndex: Integer) : Boolean;
    constructor Create               (aOwner: TComponent); override;
    destructor  Destroy; override;

  published      { Dclarations publies }
    property Anchors;
    property polygonFill     : Boolean             read fPolygonFill      write SetPolygonFill;
    property polygonColor    : TColor              read fPolygonColor     write SetPolygonColor;
    property axesColor       : TColor              read fAxesColor        write SetAxesColor;
    property axesCount       : Integer             read fAxesCount        write SetAxesCount default 7;
    property axesCaptionsFramed : Boolean          read fAxesCapFramed    write SetAxesFramed;
    property axesAutoSized   : Boolean             read fAxesAutoSized    write SetAxesSizeState;
    property borderStyle     : TQBorderStyle       read fBorderStyle      write SetBorderStyle;
    property backGround      : TQBackGround        read fBackGround       write SetBackGround;
    property backGroundColor : TColor              read fBackgroundColor  write SetBackgroundColor;
    property backGStartColor : TColor              read fBackGStartColor  write SetBackGStartColor;
    property backGFinalColor : TColor              read fBackGFinalColor  write SetBackGFinalColor;
    property titleCaption    : String              read fTitleCaption     write SetTitleCaption;
    property titlePosition   : TQTitlePosition     read fTitlePosition    write SetTitlePosition;
    property defaultPenWidth : Word                read fDfltPenWidth     write SetDfltPenWidth;
    property showMouseBox    : Boolean             read fShowMouseBox     write SetShowMouseBox;
    property mBoxForColor    : TColor              read fMBoxForColor     write SetMBoxForColor;
    property mBoxBackColor   : TColor              read fMBoxBackColor    write SetMBoxBackColor;
    property mBoxUsesLnColor : Boolean             read fMBoxUsesLnClr    write SetMBoxUsesLnClr;
    property showLinesPoints : Boolean                                    write SetShowLnPoints;
    property showLinesCaption: Boolean             read fShowLinesCpn     write SetShowLinesCpn;
    property linesBoxTransparent : Boolean         read fLinesCpnBmpTrp   write SetLinesCpnBmpTrp;
    property linesBoxColor   : TColor              read fLinesCpnBmpClr   write SetLinesCpnBmpClr;
    property highlightColor  : TColor              read fHighlightColor   write fHighlightColor;
    property trackMouseMoves : Boolean             read fTrackMsMoves     write SetTrackMsMoves;
    property OnMouseEnterLine: TQMouseLineEvt      read fMouseEnterLine   write fMouseEnterLine;
    property OnMouseExitLine : TQMouseLineEvt      read fMouseExitLine    write fMouseExitLine;
  end; {TQSpiderGraph definition}

  function GradientFill(DC : hDC; pVertex : Pointer;
           dwNumVertex : DWORD;  pMesh : Pointer;
           dwNumMesh, dwMode: DWORD) : DWord; stdcall; external 'msimg32.dll';

Const
   HC_NONE = -1;
   HC_AREA = -2;
   // HC_CENTROID could be a future enhancement...

procedure Register;


implementation


                      // ---------------------- //
                      //        Register        //
                      // ---------------------- //


procedure Register;
begin
  RegisterComponents('Samples', [TQSpiderGraph]);
end;



                      // ---------------------- //
                      //         AXES           //
                      // ---------------------- //


Constructor TQSpiderAxe.CreateOwned(anOwner: TQSpiderGraph);
// Axes need to know their owner, in order to apply the autosize feature
Begin
  Inherited Create;
  fAxeOwner    := anOwner;
  fAutoSizeMin := anOwner.fAxesAutoSized;
  fAutoSizeMax := anOwner.fAxesAutoSized;
  fIndex       := -1;
  fMax := 100;
  fMin := 0.0;
End;

Destructor TQSpiderAxe.Destroy;
Begin
  fAxeOwner := Nil;
  Inherited;
End;

// Axes autoSize min and max general behaviour :
//
// If set to False : Nothing is touched. This axe's min wil stay what it
//   is at that moment (a value you did set, the default one, or the last one
//   computed. However, this value may change if you send another line later,
//   with values below the actual axe's min (or obviously, if fMin is
//   explicitely set Through "axes[i].minValue := ").
// If set to True : fMin will be computed now, and each time a new line is
//   added to the graph.
//   min & max are then set to 1/twentiest above and bellow the actual range of this axe.

// note : Most of these properties rely on the graph's .GetMinValue(); function
//   to retrieve the actual min or max value of an axe.
//   GetMinValue(); return the min if at least one line has values, or the actual
//   fMin property of this axe otherwise.


procedure TQSpiderAxe.SetAutoSizeMin(value: Boolean);
Var highestVal,
    lowestVal : Single;
Begin
  self.fAutoSizeMin := Value;
  If value Then
  Begin
    fAxeOwner.GetMinValue(self.fIndex,lowestVal,False);
    fAxeOwner.GetMaxValue(self.fIndex,highestVal,False);
    If (lowestVal = 0.0) And (highestVal = 0.0)
       Then self.fMin := -1
       Else self.fMin := lowestVal - ( ( Abs(highestVal) + Abs(lowestVal) ) / 20 );
  End;
End;

procedure TQSpiderAxe.SetAutoSizeMax(value: Boolean);
Var highestVal,
    lowestVal : Single;
Begin
  self.fAutoSizeMax := Value;
  If value Then
  Begin
    fAxeOwner.GetMinValue(self.fIndex,lowestVal,False);
    fAxeOwner.GetMaxValue(self.fIndex,highestVal,False);
    If (lowestVal = 0.0) and (highestVal = 0.0)
       Then self.fMax := 1
       Else self.fMax := highestVal + ( ( Abs(highestVal) + Abs(lowestVal) ) / 20 );
  End;
End;

function TQSpiderAxe.GetAutoSzBoth: Boolean;
// Care ! Result is True if at least one of the two possibilities is True,
// in opposite to the "both" setter. To explicitely check wether BOTH are
// True, both values have to be read directly :
// (If (myGraph.Axe[i].autoSizeMin) AND (myGraph.Axe[i].autoSizeMax) Then ...)
Begin
  result := (fAutoSizeMin Or fAutoSizeMax);
End;

procedure TQSpiderAxe.SetAutoSzBoth(value: Boolean);
Begin
  self.SetAutoSizeMin(Value);
  self.SetAutoSizeMax(Value);
End;

// The two following properties are "requests", not "demands" : If you try to set
// a "min" above the actual lowest lines value for this axe, or siblingly try to set
// a "max" bellow the actual highest lines value for this axe, the request will be
// ignored.

procedure TQSpiderAxe.SetMin(value: Single);
var lowVal:single;
Begin
  lowVal := value;
  fAutoSizeMin := False;	
  fAxeOwner.GetMinValue(self.fIndex,lowVal,True);
  If value <= lowVal Then self.fMin := value;
End;

procedure TQSpiderAxe.SetMax(value: Single);
var highVal:Single;
Begin
  highVal := value;
  fAutoSizeMax := False;
  fAxeOwner.GetMaxValue(self.fIndex,highVal,True);
  If value >= highVal Then self.fMax := value;
End;



                      // ---------------------- //
                      //         LINES          //
                      // ---------------------- //

Constructor TQSGLine.CreateOwned(anOwner:TQSpiderGraph);
// Sizes the two internal arrays. anOwner will allow accesses
// to some needed owner's values, like axesCount, linesCount, aso;
Begin
  Inherited Create;
  fLnOwner := anOwner;
  SetLength(fValues, fLnOwner.fAxesCount);
  SetLength(fPoints, fLnOwner.fAxesCount + 1);                    // Prepares polyline (ie. Polygon not filled)
  fHasValues  := False;
  caption     := ' ';
  Color       := clRed;                                           // Red is the default color for lines ;
  fPenWidth   := flnOwner.defaultPenWidth;
  fPenWidthMem:= flnOwner.defaultPenWidth;
  fShowPoints := flnOwner.fShowLnPoints;
  fShow       := True;
  fShowPointsMem := flnOwner.fShowLnPoints;
  fLnOwner.fLinesCpnBmpdef := False;                              // tells owner that the box showing lines caption and ...
End;                                                              // ... the corresponding colors has to be copmputed again.

Destructor TQSGLine.Destroy;
Begin
  If Assigned(fLnOwner) Then fLnOwner.fLinesCpnBmpdef := False;
  SetLength(fValues, 0);
  SetLength(fPoints, 0);
  fLnOwner := Nil;
  caption := '';
  Inherited;
End;

procedure  TQSGLine.SetPenWidth(value : Word);
Begin
  If (value < 1) Then value := 1;
  self.fPenWidth    := value;
  self.fPenWidthMem := value;
End;

procedure  TQSGLine.SetValues(Const vals:TQSingleArray);
// Accepts less values than axes       -> the lasts ones will default to Axe[i].min ;
// Don't accept  more values than axes -> the lasts ones provided (exeeding axeCount) will be ignored ;
// Moreover, if values sent are bellow the corresponding axe current value,
// this axe's min will be reseted to this new min; Siblingly for Max.
// If an axe has its autosize property to true, a boolean will be set,
// to force the recomputing before next Paint;
var i, loopTmp : Integer;
    minVal,
    maxVal :Single;
Begin
  // Which is the smallest : [vals]'s length or fAxesCount ?
  loopTmp := Min(fLnOwner.fAxesCount,Length(vals));

  // values storage, and control of max and min.
  For i := 0 To loopTmp -1 Do
  Begin
    self.fValues[i] := vals[i];
    minVal := vals[i];
    maxVal := vals[i];
    fLnOwner.GetMinValue(i, minVal,fLnOwner.fAxes[i].autoSizeMin) ;                  // Get the min
    fLnOwner.GetMaxValue(i, maxVal,fLnOwner.fAxes[i].autoSizeMax) ;                  // and the max

    If ( vals[i] <= minVal ) Then fLnOwner.fAxes[i].fMin := vals[i];
    If ( vals[i] >= maxVal ) Then fLnOwner.fAxes[i].fMax := vals[i];
    // If needed, forces recomputing of max and min values;
    If fLnOwner.fAxes[i].autoSized Then fLnOwner.fAxesMMdefined := False;
  End;

  // Finalization : If some axes values haven't been provided, set them to the
  // lowest value of the corresponding axes...
  For i := loopTmp To (fLnOwner.fAxesCount-1) Do
    self.fValues[i] := fLnOwner.fAxes[i].fMin;
  self.fHasValues := True;
End;

procedure TQSGLine.SetShowPoints(value: Boolean);
Begin
  self.fShowPoints    := Value;
  self.fShowPointsMem := Value;
End;

procedure TQSGLine.SetColor(value: TColor);
// Filters received color to ensure that the color finally used by the system
// is the same as the one stored by the component : In some cases the system
// may use a color "close to" the one requested, instead of the exact one
// requested. But the graph uses the color below the pointer to detect mouse
// moves over a line, comparaing the one read with the ones stored, so that we
// need to make sure they correspond.
var r,g,b : Byte;

Begin
  // don't know if the conversion to R,G,B and back is usefull or not ...
  r := GetRValue(value);
  g := GetGValue(value);
  b := GetBValue(value);
  Self.fColor    := GetNearestColor(self.fLnOwner.Canvas.Handle, RGB(r,g,b));
  self.fColorMem := Self.fColor
End;

procedure TQSGLine.Hide;
Begin  self.fShow := False; self.fLnOwner.Invalidate; End;

procedure TQSGLine.Show;
Begin  self.fShow := True; self.fLnOwner.Invalidate; End;



                      // ---------------------- //
                      //         Mouse          //
                      // ---------------------- //



Procedure TQSpiderGraph.MouseTrack(Sender:TObject; Shift:TShiftState; X,Y:Integer);
var i, lnIx : integer;
    xP,yP   : Integer;      // X and Y of the parent
    aString : String;
begin
  // Ignores mouse moves outside the graphRect area (title, line captions box)
  // or if the graph shall not track mouse moves
  If Not(fTrackMsMoves)
     Or (Y > self.fGraphRect.Bottom -2)
     Or (Y < self.fGraphRect.top +2)  Then Exit;

  // If the mouse info boxe has to be drawn outside the bounds of the
  // graph itself, X and Y have to be related to the whole form.
  // ( fMBoxParent can be Self or the graph's parent form)
  If Self.fMBoxParent = Self Then
  Begin
    xP := 0;
    yP := 0;
  End Else
  Begin
    xP := self.Left;
    yP := self.Top;
  End;

  lnIx := -1;
  aString := '';
  For i := 0 To fLinesCount -1 Do
      If ( (Canvas.Pixels[X,Y]   = fLines[i].color) Or
           (Canvas.Pixels[X+1,Y] = fLines[i].color) Or
           (Canvas.Pixels[X-1,Y] = fLines[i].color) Or
           (Canvas.Pixels[X,Y+1] = fLines[i].color) Or
           (Canvas.Pixels[X,Y-1] = fLines[i].color) )
      AND fLines[i].fHasValues
      Then lnIx := i;

  // Here above, the checking of "fLines[i].fHasValues" is needed because the loop returns the last "i" whose
  // line's color equals the color below the mouse. But if the line count has been set to more lines
  // than needed, the exceeding ones have been created with clRed as color (default one). and if the line
  // actually bellow the mouse is red too, then the loop will return the last "i" instead of the first one.
  // Checking fHasValues avoids that (a "break" could have do the trick too)

  // -1- If mouse isn't on a line, was it above before ? If yes, we've just "exited" one line.
  If lnIx < 0 Then
  Begin
    If (fMouseOnLine) Then                                  // exiting a line
    Begin
      fMouseOnLine := False;
      fLinesInfoBoxes[fMouseLineIndex].box.Hide;
      Invalidate;
      If Assigned( fMouseExitLine ) Then
         fMouseExitLine(Self, fMouseLineIndex);
      fMouseLineIndex := lnIx;
    End
  // -2- Otherwise, mouse is actually upon a line. Question becomes :
  //     Was it on a line before ? If no, we've just entered a line ;
  //     If yes, was it upon another line (implying to first raise a mouseExit before the new mouseEnter) ?
  End Else If (fMouseOnLine) And (lnIx <> fMouseLineIndex) Then
      Begin
        fLinesInfoBoxes[fMouseLineIndex].box.Hide;
        Invalidate;
        If Assigned( fMouseExitLine ) Then fMouseExitLine(Self, fMouseLineIndex);
        If (fShowMouseBox) Then
        Begin
          If Not(fLinesInfoBoxes[lnIx].defined) Then DefineLinesInfoBox(lnIx);
          fLinesInfoBoxes[lnIx].box.Top  := Y + yP;
          fLinesInfoBoxes[lnIx].box.Left := X + xP +15;
          fLinesInfoBoxes[lnIx].box.Show;
        End;
        fMouseLineIndex := lnIx;
        If Assigned( fMouseEnterLine ) Then fMouseEnterLine(Self, fMouseLineIndex);
     End Else
     Begin
       fLinesInfoBoxes[lnIx].box.Top  := Y + yP;
       fLinesInfoBoxes[lnIx].box.Left := X + xP +15;
       If not(fMouseOnLine) Then
       Begin
         If (fShowMouseBox) Then
         Begin
           If Not(fLinesInfoBoxes[lnIx].defined) Then DefineLinesInfoBox(lnIx);
           fLinesInfoBoxes[lnIx].box.Show;
         End;
         fMouseOnLine := True;
         fMouseLineIndex := lnIx;
         If Assigned( fMouseEnterLine ) Then
         fMouseEnterLine(Self, fMouseLineIndex);
       End;
     End;
End;


                      // ---------------------- //
                      //   Graph's internals    //
                      // ---------------------- //


function TQSpiderGraph.ResizeRect(aRect:Trect;hm:Integer):Trect;
// aRect : The rect to be resized; hm : "How much ?".
// returns aRect decreased by hm in any direction (or inflated, if hm < 0)
Begin
  result.Left  := aRect.Left   + hm;
  result.Right := aRect.Right  - hm;
  result.Top   := aRect.Top    + hm;
  result.Bottom:= aRect.Bottom - hm;
End;


function TQSpiderGraph.GetLinesCaptionHeight : integer;
// Builds fLinesCpnBmp, the TBitmap showing lines captions + color
// and returns its height;
var maxLineLen, maxWidth, iTmp,
    i, j, txtWidth,
    lineCnt,lineHgt,
    lineLen   : Integer;
    xClr, yClr,
    xCap, yCap,
    linePtr : Integer ;
    arCpnDescr : Array of integer;                          // will store the number of captions to draw, line by line

Begin
  result := 0;
  If (fLinesCount = 0) or (Not (fLines[0].fHasValues) ) Then Exit;
  TRY
    If assigned(fLinesCpnBmp) And (fLinesCpnBmpdef)
    Then Begin result := fLinesCpnBmp.height; Exit; End
    Else Begin
           If Assigned(fLinesCpnBmp) Then fLinesCpnBmp.Free;
           fLinesCpnBmp := TBitmap.Create;
           SetLength( arCpnDescr,1 );

           fLinesCpnBmp.Canvas.Font := linesCpnFont;
           fLinesCpnBmp.Transparent := fLinesCpnBmpTrp;
           fLinesCpnBmp.TransparentColor := fLinesCpnBmpClr;
           fLinesCpnBmp.TransparentMode  := tmFixed;
           fLinesCpnBmp.Canvas.Brush.Color := clGray;
           fLinesCpnBmp.canvas.FrameRect( Rect(1, 1, fLinesCpnBmp.Width-2, fLinesCpnBmp.Height-2));
           fLinesCpnBmp.Canvas.Brush.Color := fLinesCpnBmpClr;
                      
           maxLineLen := 0;
           maxWidth := Self.width - fBorderSize - 4 ;       // -4 preserves 2 pixels on both sides
           lineCnt  := 1;
           arCpnDescr[0] := 0;
           lineHgt  := fLinesCpnBmp.Canvas.TextHeight('Ip') + 2 ;
           lineLen  := 0;

           // -1- Counting lines
           For i := 0 To self.fLinesCount -1 Do
           Begin
             txtWidth := fLinesCpnBmp.canvas.TextWidth(fLines[i].caption) + 28; // the colored rectangle, some space and the caption
             iTmp := lineLen + txtWidth;
             If iTmp <= maxWidth Then
             Begin
             	 lineLen := iTmp;
             	 arCpnDescr[lineCnt-1] := arCpnDescr[lineCnt-1] +1;
             	 If lineLen > maxLineLen Then maxLineLen := lineLen;
             End Else
             Begin
               Inc(lineCnt);
               SetLength(arCpnDescr, lineCnt);
               lineLen := txtWidth;
               arCpnDescr[lineCnt-1] := 1;                  // new line to display, and so, at least one caption (the current one)
             End;
           End;
           
           // -2- Drawing
           fLinesCpnBmp.Height := lineHgt * lineCnt +3;
           fLinesCpnBmp.Width  := maxLineLen;

           xClr := 12;
           yClr := ((lineHgt -5) Div 2)+1;
           xCap := 25;
           yCap :=  2;
           linePtr := 0;

           For i:= 0 To Length(arCpnDescr) -1 Do
           Begin
             For j := 0 To arCpnDescr[i] -1 Do
             Begin
               With fLinesCpnBmp Do
               Begin
                 Canvas.pen.Color := clGray;
                 Canvas.Brush.Color := fLines[linePtr].Color;
                 Canvas.Rectangle(xClr,yClr,xClr+10,yClr+5);
                 Canvas.Brush.Color := fLinesCpnBmpClr;
                 Canvas.Pen.Color := clBlack;
                 Canvas.TextOut(xCap,yCap,fLines[linePtr].caption);
               End;
               xClr := xClr + fLinesCpnBmp.canvas.TextWidth(fLines[linePtr].caption) + 28;
               xCap := xCap + fLinesCpnBmp.canvas.TextWidth(fLines[linePtr].caption) + 28;
               Inc(linePtr);
             End;
             xClr := 12;
             yClr := yClr + lineHgt;
             xCap := 25;
             yCap := yCap + lineHgt;
           End;
           fLinesCpnBmp.Canvas.Brush.Color := clGray;
           fLinesCpnBmp.canvas.FrameRect( Rect(0, 0, fLinesCpnBmp.Width, fLinesCpnBmp.Height));
           fLinesCpnBmpdef := True;
         End;
  result := fLinesCpnBmp.height;
  EXCEPT; result := 0; END;
End;


procedure TQSpiderGraph.DefineDrawSpace;
var oldFont : TFont;
    capHgt  : Integer;
Begin
  // width & height
  Case self.fBorderStyle of
    bsNone :
         Begin
           fBorderSize := 0;
           fUsfWidth   := width;
           fUsfHeight  := height;
           fGraphRect  := ClientRect;
         End;
    bsFlat :
         Begin
           fBorderSize := 1;
           fUsfWidth   := width  -2;
           fUsfHeight  := height -2;
           fGraphRect  := ResizeRect(ClientRect, 1) ;
         End;
    Else Begin         //bs3D
           fBorderSize := 2;
           fUsfWidth   := width  -4;
           fUsfHeight  := height -4;
           fGraphRect  := ResizeRect(ClientRect, 2) ;
         End;
  End; {Case}

  // Graph's title
  If self.titleCaption <> '' Then
  Begin
    oldFont := canvas.Font;
    canvas.Font := TitleFont;
    Case fTitlePosition of
      qtpTop: Begin
                fTitleRect.Top    := self.fBorderSize + 4 ;
                fTitleRect.Left   := (self.fUsfWidth  - canvas.TextWidth(titleCaption)) div 2;
                fTitleRect.Right  := fTitleRect.Left  + canvas.TextWidth(titleCaption);
                fTitleRect.Bottom := fTitleRect.Top   + canvas.TextHeight(titleCaption);
              End;
      Else    Begin   // qtpBottom
                fTitleRect.Left   := (self.fUsfWidth  - canvas.TextWidth(titleCaption)) div 2;
                fTitleRect.Right  := fTitleRect.Left  + canvas.TextWidth(titleCaption);
                fTitleRect.Bottom := self.Height - fBorderSize - 4;
                fTitleRect.Top    := fTitleRect.Bottom - canvas.TextHeight(titleCaption) ;
              End;
    End;{Case}
    canvas.Font := oldFont;
  End;

  // Graph's drawSpace
  If titlecaption = '' Then
  Begin
    fGraphCenter.X    := fBorderSize + (fUsfWidth  Div 2 );
    fGraphCenter.Y    := fBorderSize + (fUsfHeight Div 2 );
    fGraphRect.Left   := fBorderSize;
    fGraphRect.Top    := fBorderSize;
    fGraphRect.Right  := self.Width  - (fBorderSize shl 1);
    fGraphRect.Bottom := self.Height - (fBorderSize shl 1);
  End Else
  Begin
    Case titlePosition of
      qtpTop: Begin
                fGraphRect.Left   := fBorderSize;
                fGraphRect.Top    := fTitleRect.Top + fTitleRect.Bottom ;
                fGraphRect.Right  := self.Width  - (fBorderSize shl 1);
                fGraphRect.Bottom := self.Height - (fBorderSize shl 1);
                fGraphCenter.X    := (fGraphRect.Right  - fGraphRect.Left)  Div 2 ;
                fGraphCenter.Y    := fGraphRect.Top + ((fGraphRect.Bottom - fGraphRect.Top)  Div 2) ;
              End;
      Else    Begin               // ie. case qtpBottom
                                  //             - (fBorderSize shl 1) : self.height still contains both boders.
                fGraphRect.Bottom := self.Height - (fBorderSize shl 1)
                                   - (fTitleRect.Bottom - fTitleRect.Top);
                fGraphRect.Top    := fBorderSize;
                fGraphRect.Left   := fBorderSize;
                fGraphRect.Right  := self.Width  - (fBorderSize shl 1);
                fGraphCenter.X    := (fGraphRect.Right  - fGraphRect.Left) SHR 1 ;
                fGraphCenter.Y    := (fGraphRect.Bottom - fGraphRect.Top) SHR 1;
              End;
    End;{case}
  End;

  // Lines captions : drawn at opposite side of title.
  // Kept unchanged : fGraphRect.Left; fGraphRect.Right; fGraphCenter.X;
  If fShowLinesCpn Then
  Begin
    capHgt := GetLinesCaptionHeight;                        // Computes the caption rect and returns its height
    Case titlePosition of
      qtpTop: Begin
                fGraphRect.Bottom := fGraphRect.Bottom - capHgt;
                fGraphCenter.Y := fGraphRect.Top + ((fGraphRect.Bottom - fGraphRect.Top)  Div 2) ;
              End;
      Else    Begin
                fGraphRect.Top := fGraphRect.Top + capHgt;
                fGraphCenter.Y := fGraphRect.Top + ((fGraphRect.Bottom - fGraphRect.Top)  Div 2) ;
              End;
    End;{case}
  End;
End;


procedure TQSpiderGraph.InitSummits;
var i : Integer;                               
    baseAngle : Single;
    gwd,ght : integer;
Begin
  If Length(self.fAxes) < self.fAxesCount Then SetLength(fAxes,fAxesCount);
  SetLength(self.fSummits,self.fAxesCount+1);                 // Trick, to allow polyline instead of polygone, see help.
  Self.fAngle := 360 / self.fAxesCount;
  baseAngle   := 0;

  gwd := fGraphRect.Right - fGraphRect.Left;
  ght := fGraphRect.Bottom- fGraphRect.Top;
  i   := Min(gwd,ght);
  fAxesLen    := (i div 20) * 9 ;                             // 8/10 du rayon

  For i := 0 To self.fAxesCount -1 Do
  Begin
    Self.fSummits[i].X := trunc(fGraphCenter.x + fAxesLen * cos((baseAngle-90)*pi/180));
    Self.fSummits[i].Y := trunc(fGraphCenter.y + fAxesLen * sin((baseAngle-90)*pi/180));
    // Corrects most visible approximations ...
    If       (baseAngle =  45) Then fSummits[i].Y := fSummits[i].Y +1
    Else If  (baseAngle = 225) Then fSummits[i].Y := fSummits[i].Y +1
    Else If ((baseAngle =  90) Or (baseAngle = 270)) Then fSummits[i].Y := fGraphCenter.Y
    Else If  (baseAngle = 180) Then fSummits[i].X := fGraphCenter.X;
    fAxes[i].Angle := baseAngle;                            // axes
    baseAngle := baseAngle + fAngle;
  End;
  Self.fSummits[self.fAxesCount].X := Self.fSummits[0].X;
  Self.fSummits[self.fAxesCount].Y := Self.fSummits[0].Y;
  SetLength(fSummitsPolygon,fAxesCount);
  fSummitsPolygon := fSummits;
End;


function TQSPiderGraph.GetCaptionRect(aString:String;aCanvas:TCanvas;anAngle:Single;
                       destinationRect:TRect;inflateBy,Xbase,Ybase:Integer): TRect;
// Returns the rectangle that will hold this axe's caption(optionally including it's shape)
// Base for work = fsummit[i].X,fsummit[i].Y  (Xbase,Ybase)
// Captions for angles 0, 90, 180, 270 are considered special cases ;
// Others depend of the quandrant they're into.
// Basically, and to make things not too complicated, we turn like the niddles of a watch,
// and consider in turn each axe's summit to be the reference angle to draw the caption.
// Moreover, we constantly control that the resulting rectangle doesn't excess the graph limits.

var strWidth,
    strHeight: Integer;

Begin
  result.Top  := inflateBy;       
  result.Left := inflateBy;       
  strWidth  := aCanvas.TextWidth(aString);
  strHeight := aCanvas.TextHeight(aString);
  result.Right  := result.Left + strWidth;
  result.Bottom := result.Top  + strHeight;
  ResizeRect(result,inflateBy);                       
  TRY
    strWidth  := strWidth  + (inflateBy SHL 1);
    strHeight := strHeight + (inflateBy SHL 1);

    If anAngle = 0 Then                               // central-top caption : center horizontaly and check .Top ;
    Begin                                            
      // top  = summit.Y + rectResized + 2, but can't exces graphrect.top
      // left = summit.X (-1/2 * caption.Width) but can't exces graphrect.left;
      result.Top  := ( Max( destinationRect.Top,  Ybase - strHeight - 2 ));
      result.Left := ( Max( destinationRect.Left, Xbase - (strWidth SHR 1 ) ));
    End Else
    If anAngle < 90 Then                              //1st quadrant : reference = bottom-left corner of caption
    Begin
      // top = summit.Y + rectResized.Height - 2 ou .top de graphrect
      // left = summit.X -1/2 de caption ou .left de graphrect;
      result.Top := Ybase - 2 - strHeight;
      result.Left:= Xbase + 2 ;
      If (result.Left + strWidth) > destinationRect.Right
         Then result.left := destinationRect.Right - strWidth;
    End Else
    If anAngle = 90 Then                              // central-left caption : centered vertically ;
    Begin                                             // and check the right margin.
      result.Top  := Ybase - (strHeight SHR 1) ;
      result.Left := Xbase + 2;
      If (result.Left + strWidth) > destinationRect.Right
         Then result.Left := destinationRect.Right - strWidth;
    End Else
    If anAngle < 180 Then                             //2nd quadrant : reference = Top-left corner of caption
    Begin
    result.Top  := Ybase + 2;
    result.Left := Xbase + 2 ;
      If (result.Left + strWidth) > destinationRect.Right
          Then result.Left := destinationRect.Right - strWidth;
    End Else
    If anAngle = 180 Then                             // central-bottom caption : centered horizontally ;
    Begin                                             // and check the bottom margin.
      result.Top  := Ybase + 2;
      result.Left := Xbase - (strWidth SHR 1) ;
      If (result.Top + strHeight) > destinationRect.Bottom
         Then result.Top := destinationRect.Bottom - strHeight;
    End Else
    If anAngle < 270 Then                             // 3rd quadrant : reference = Top-right corner of caption
    Begin
      result.Top  := Ybase + 2;
      result.Left := Xbase - 2 - strWidth;
    If result.Left < destinationRect.left Then result.Left := destinationRect.left;
    End Else
    If anAngle = 270 Then                             // central-left caption
    Begin
      result.Top  := Ybase - (strHeight SHR 1) ;
      result.Left := Xbase - 2 - strWidth;
      If result.Left < destinationRect.left Then result.Left := destinationRect.left;
    End Else                                          // Last quadrant;  270 < anAngle < 359
    Begin
      result.Top := Ybase - 2 - strHeight;
      result.Left:= Xbase - 2 - strWidth;
      If result.Left < destinationRect.left Then result.Left := destinationRect.left;
    End;
  result.Right  := result.Left + strWidth;
  result.Bottom := result.Top  + strHeight;
  EXCEPT;END;
End;

function TQSpiderGraph.GetLinePoint(iMin,iMax,position,iAngle:Single):TPoint;
// returns the TPoint coordinates of lines points.
var posNorm,
    posPct,
    posDist :  Single;
    axeRange: Extended; 
Begin
  TRY
    axeRange := (iMax-iMin);
    If axeRange = 0 Then
    Begin
      result.X := fGraphCenter.x;
      result.Y := fGraphCenter.y;
      exit;
    End;
    posNorm := position - iMin;
    posPct  := posNorm / axeRange;
    posDist := fAxesLen * posPct;
    result.X := trunc(fGraphCenter.x + posDist * cos((iAngle -90 )*pi/180));  // NIH
    result.Y := trunc(fGraphCenter.y + posDist * sin((iAngle -90 )*pi/180));
  EXCEPT;END;
End;


                      // ---------------------- //
                      //         GRAPH          //
                      // ---------------------- //

Constructor TQSpiderGraph.Create(aOwner : TComponent);
var i : Integer;
    anAxe:TQSpiderAxe;

Begin
  Inherited Create(aOwner) ;
  // explicitely setting the parent Avoids the  &/#!@-!  message :
  // "control '' has no parent window !"
  // (http://gethelp.devx.com/techtips/delphi_pro/10min/10min0601/10min0601-3.asp)
  Parent := TWinControl(aOwner);

  fAxesCount := 7;
  fAxesColor := clNavy;
  axesFont   := TFont.Create;
  fAxesAutoSized := False;

  SetLength(self.fAxes,self.fAxesCount);
  For i := 0 To fAxesCount -1 Do
  Begin
    anAxe := TQSpiderAxe.CreateOwned(Self);
    anAxe.fMin    := 0;
    anAxe.fMax    := 100;
    anAxe.Angle   := 0;
    anAxe.fIndex  := i;
    anAxe.caption := IntToStr(i);
    fAxes[i] := anAxe;
  End;

  fGraphRect     := Self.ClientRect;
  fGraphCenter.X := Self.Width Div 2;
  fGraphCenter.Y := Self.Height Div 2;
  DoubleBuffered := True;

  fBorderStyle:= bsNone;
  fBorderSize := 0;
  fBackGround := bgTransparent;
  fBackgroundColor := clWhite;
  fBackGStartColor := RGB(230,230,230);
  fBackGFinalColor := clWhite;

  fTitleCaption   := '';
  fTitlePosition  := qtpTop;
  titlefont       := TFont.Create;
  titlefont.Color := clBlue;
  titleFont.Style := [fsBold];
  titleFont.Size  := 12;
  self.fPolygonFill := False;
  self.fPolygonColor:= RGB(230,255,255);

  fDfltPenWidth := 2;
  fShowLnPoints := False;
  fLinesCount   := 1;                                       // A graph has allways at least one line (even if this line is "empty")
  fMultilines   := False;
  SetLength(fLines,1);
  fLines[0] := TQSGLine.CreateOwned(Self);

  linesCpnFont   := TFont.Create;
  fLinesCpnBmpClr:= clWhite;
  fLinesCpnBmpDef:= False;
  fShowLinesCpn  := False;
  // lines information window (Mouse track result)
  fShowMouseBox  := True;
  fMBoxBackColor := clInfoBk;
  fMBoxForColor  := clBlack;
  fMBoxUsesLnClr := False;
  SetLength(fLinesInfoBoxes,1);
  fMBoxParent := Self;                                       // Prepares the future Boxes parent
  fLinesInfoBoxes[0].Box := TStaticText.Create(Self);
  fLinesInfoBoxes[0].Box.Parent := Self;
  With fLinesInfoBoxes[0].Box Do
  Begin
    Hide;
    Parent := Self;
    AutoSize := False;
    BorderStyle :=  sbsSingle;
    Caption := '';
    Color := mBoxBackColor;
  End;
  fLinesInfoBoxes[0].defined := False;

  fTrackMsMoves  := True;
  fMouseOnLine   := False;
  HighlightMode  := [hmShowPoints, hmFlashLine];
  fHighlightColor:= clYellow;
  fMouseLineIndex  := -1;
  Self.OnMouseMove := MouseTrack;

  Self.DefineDrawSpace;
  Self.InitSummits;

  parent := Nil;                                            // see above ...
End;


Destructor TQSpiderGraph.Destroy;
var i : integer;
Begin
  For i := 0 To linesCount-1 Do fLines[i].free;
  If Assigned(fLinesCpnBmp) Then fLinesCpnBmp.Free;
  Inherited;
End;


Procedure TQSpiderGraph.Paint;
Var i,ix0,ix1 : Integer;
    oldBkMode : integer;
    oldFont   : TFont;
    //gradientFill API : Source msft : http://msdn.microsoft.com/library/default.asp?url=/library/en-us/gdi/bitmaps_7nj9.asp
    vert : array[0..1] of TRIVERTEX;
    gRect: GRADIENT_RECT;
    aRect: TRect;
    gDir : integer;
    aClr : TColor;
    aPoint  : TPoint;
    lc,pw : integer;

begin
  // -1- Initializing margins and summits
  DefineDrawSpace;
  InitSummits;

  // -2- (if needed) max and min
  If Not(fAxesMMdefined) Then
  Begin
    For i := 0 To Self.fAxesCount -1 Do
    Begin
      fAxes[i].SetAutoSizeMin(fAxes[i].fAutoSizeMin);
      fAxes[i].SetAutoSizeMax(fAxes[i].fAutoSizeMax);
    End;
  fAxesMMdefined := True;
  End;

  // -3- Component's border
  Case self.fBorderStyle of
    bsFlat :
         Begin
           aClr := self.Canvas.Brush.Color;
           self.Canvas.Brush.Color := clGray;
           self.Canvas.FrameRect(Self.ClientRect);
           self.Canvas.Brush.Color := aClr;
         End;
    bs3D :
         Begin
           aClr  := self.Canvas.Brush.Color;
           self.Canvas.Brush.Color := clWhite;
           aRect := Self.ClientRect;
           aRect.Left := Self.ClientRect.Left +1;
           aRect.Top  := Self.ClientRect.Top  +1;
           self.Canvas.FrameRect(aRect);
           
           self.Canvas.Brush.Color := RGB(172,168,153);
           aRect := Self.ClientRect;
           aRect.Right  := Self.ClientRect.Right-1;
           aRect.Bottom := Self.ClientRect.Bottom-1;
           self.Canvas.FrameRect(aRect);
           self.Canvas.Brush.Color := aClr;
         End;
  End;{Case}
  // -4- Drawing the background
  If fBackGround > bgcolored Then
  Begin
    Case fBackGround of
      bgTopBottom : Begin
                      gDir := GRADIENT_FILL_RECT_V;
                      ix0 := 0;
                      ix1 := 1;
                    End;
      bgBottomTop : Begin
                      gDir := GRADIENT_FILL_RECT_V;
                      ix0 := 1;
                      ix1 := 0;
                    End;
      bgLeftToRight : Begin
                      gDir := GRADIENT_FILL_RECT_H;
                      ix0 := 0;
                      ix1 := 1;
                    End;
      Else Begin
             gDir := GRADIENT_FILL_RECT_H;
             ix0 := 1;
             ix1 := 0;
           End;
    End;{case}
    vert [0] .x       := self.fBorderSize;
    vert [0] .y       := self.fBorderSize;
    vert [ix0] .Red   := GetRValue(self.fBackGStartColor) shl 8;
    vert [ix0] .Green := GetGValue(self.fBackGStartColor) shl 8;
    vert [ix0] .Blue  := GetBValue(self.fBackGStartColor) shl 8;
    vert [ix0] .Alpha := $0000;
    vert [1] .x       := self.Width - self.fBorderSize;
    vert [1] .y       := self.Height - self.fBorderSize;
    vert [ix1] .Red   := GetRValue(self.fBackGFinalColor) shl 8;
    vert [ix1] .Green := GetGValue(self.fBackGFinalColor) shl 8;
    vert [ix1] .Blue  := GetBValue(self.fBackGFinalColor) shl 8;
    vert [ix1] .Alpha := $0000;
    gRect.UpperLeft   := 0;
    gRect.LowerRight  := 1;
    GradientFill(Self.Canvas.Handle, @vert,2,@gRect,1,gDir);
  End Else
      If fBackGround = bgColored Then
      Begin
      	aClr  := self.Canvas.Brush.Color;
        canvas.Brush.Color := self.fBackgroundColor;
        canvas.FillRect(ResizeRect(Self.ClientRect,fBorderSize));
        self.Canvas.Brush.Color := aClr;
      End;

  // -5- Axes
  // Linking the axes summits can be done two ways :
  // -> using Canvas.Polygon() will fill the polygon with the brush's color ;
  // -> Otherwise, D6's help suggests using Canvas.Polyline() to avoid 
  //    this filling, passing the first point two times (as first and last point);
  With canvas Do
  Begin
    Pen.Width := 1 ;
    Pen.Color := self.fAxesColor;
    // summits
    If Self.fPolygonFill Then
    Begin
      aClr  := self.Canvas.Brush.Color;
      Canvas.Brush.Color := fPolygonColor;
      Polygon( Self.fSummitsPolygon );
      self.Canvas.Brush.Color := aClr;
    End Else Polyline(Self.fSummits);
    // axes
    For i := 0 To Length(self.fSummits)-1 Do
    Begin
      moveto(fGraphCenter.x,fGraphCenter.y);
      lineTo(Self.fSummits[i].X, Self.fSummits[i].Y);
    End;
  End; 

  // -6- Title
  If (self.titleCaption <> '') Then
  Begin
    oldFont := canvas.Font;
    canvas.Font := TitleFont;
    OldBkMode := SetBkMode(Canvas.Handle,TRANSPARENT);
    Canvas.TextOut(fTitleRect.Left, fTitleRect.Top, titleCaption);
    SetBkMode(Canvas.Handle,OldBkMode);
    canvas.Font := oldFont;
  End;

  // -7- Axes' captions
  oldBkMode := SetBkMode(Canvas.Handle,TRANSPARENT);
  oldFont := canvas.Font;
  canvas.Font := axesFont;
  For i := 0 To fAxesCount - 1 Do
  Begin
    If (self.fAxes[i].caption <> '') Then
    Begin
      aRect := GetCaptionRect(fAxes[i].caption,self.Canvas,fAxes[i].angle,
                              fGraphRect,2, fSummits[i].X, fSummits[i].Y );
      SetBkMode(Canvas.Handle,TRANSPARENT);
      Canvas.TextOut(aRect.Left+2, aRect.Top+2, fAxes[i].caption);
      If fAxesCapFramed Then
      Begin
        aClr := Canvas.Brush.Color;
        Canvas.Brush.Color := canvas.Font.Color;
        Canvas.FrameRect(aRect);
        Canvas.Brush.Color := aClr;
      End;
     End
  End;
  SetBkMode(Canvas.Handle,OldBkMode);
  canvas.Font := oldFont;

  // -8- Lines
  For lc := 0 To self.fLinesCount-1 Do
  Begin
    If ( Not( fLines[lc].fHasValues ) ) or ( Not( fLines[lc].fshow )) Then Continue;
    For i := 0 To fAxesCount-1 Do
    Begin
      aPoint := GetLinePoint(fAxes[i].fMin,fAxes[i].fmax,
                             fLines[lc].fValues[i],fAxes[i].angle);
      fLines[lc].fPoints[i].X := aPoint.X;
      fLines[lc].fPoints[i].Y := aPoint.Y;
    end;
    // polyline finalization : linking last and first values;
    // (The "fPoints" array has soon been sized in the lines constructor to fAxesCount +1)
    fLines[lc].fPoints[fAxesCount].X := fLines[lc].fPoints[0].X;
    fLines[lc].fPoints[fAxesCount].Y := fLines[lc].fPoints[0].Y;
    // HighLight management : the current line's pen width, showPoints, aso, has been set earlier :
    // either by the TQLine instance constructor, or by the programmer at run-time.
    canvas.Pen.Width := fLines[lc].fPenWidth;
    canvas.Pen.Color := fLines[lc].color;
    canvas.Polyline(fLines[lc].fPoints);
    canvas.Pen.Width := 2;
    If fLines[lc].fShowPoints Then
    Begin
      If fLines[lc].penWidth < 4 Then pw := 4 Else
         pw := fLines[lc].penWidth;
      For i := 0 To fAxesCount-1 Do
        Canvas.Ellipse(fLines[lc].fPoints[i].X - pw,
                       fLines[lc].fPoints[i].Y - pw,
                       fLines[lc].fPoints[i].X + pw,
                       fLines[lc].fPoints[i].Y + pw );
    End;
  End;
  canvas.Pen.Width := 1;

  // -9- lines captions with their colors :
  If fShowLinesCpn And (GetLinesCaptionHeight > 0) Then
  Begin
    Case titlePosition of
      qtpTop : Canvas.Draw( (self.Width - fLinesCpnBmp.Width) Div 2,
                            self.Height - fLinesCpnBmp.Height - fBorderSize -2,
                            fLinesCpnBmp );
      Else    Canvas.Draw( (self.Width - fLinesCpnBmp.Width) Div 2,
                            fBorderSize + 2,
                            fLinesCpnBmp );
    End;{case}
  End;
End;


Procedure TQSpiderGraph.Resize;
Begin
  Inherited;
  self.fLinesCpnBmpdef := False;
  Invalidate;
End;

Procedure TQSpiderGraph.SetBorderStyle(value:TQBorderStyle);
Begin
  Self.fBorderStyle := Value;
  defineDrawSpace; 
  Invalidate;
End;

Procedure TQSpiderGraph.SetBackGround(value:TQBackGround);
Begin
  fBackGround := Value;
  Invalidate;
End;

Procedure TQSpiderGraph.SetBackGroundColor(value:TColor);
Begin
  fBackgroundColor := value;
  invalidate;
End;

Procedure TQSpiderGraph.SetAxesColor(value:TColor);
Begin
  fAxesColor := value;
  invalidate;
End;

Procedure TQSpiderGraph.SetBackGStartColor(value:TColor);
Begin
  Self.fBackGStartColor := value;
  invalidate;
End;

Procedure TQSpiderGraph.SetBackGFinalColor(value:TColor);
Begin
  Self.fBackGFinalColor := value;
  invalidate;
End;

Procedure TQSpiderGraph.SetAxesCount(value:Integer);
// Care : Changing axesCount erases stored lines...
// May seam boring, but look : The line values setter stores only
// as many values as there is axes. 
// values in excess have been rejected, missing ones have been set to a 
// "Min" which is no longer good.
var i : Integer;
    anAxe:TQSpiderAxe; //PQSpiderAxe;

Begin
  For i := 0 To fAxesCount -1 Do
  Begin
    anAxe := fAxes[i];
    anAxe.Free;
  End;
  SetLength(self.fAxes,0);

  If value < 3
     Then Self.fAxesCount := 3
     Else Self.fAxesCount := value;

  SetLength(self.fAxes,fAxesCount);

  For i := 0 To fAxesCount -1 Do
  Begin
    anAxe := TQSpiderAxe.CreateOwned(Self);
    anAxe.fMin    := 0;
    anAxe.fMax    := 100;
    anAxe.Angle   := 0;
    anAxe.fIndex  := i;
    anAxe.caption := IntToStr(i);
    fAxes[i] := anAxe;
  End;
 
  // reset lines collection
  For i := 0 To fLinesCount -1 Do
      If Assigned( fLines[i] ) Then fLines[i].free;
  SetLength(fLines,0);
  self.fMultilines := False;
  self.fLinesCount := 1;
  SetLength(fLines,1);
  fLines[0] := TQSGLine.CreateOwned(Self);  

  // and their corresponding infoBoxes
  For i := 0 To Length(fLinesInfoBoxes) -1 Do
      If (Assigned(fLinesInfoBoxes[i].box) ) Then
         Begin
           fLinesInfoBoxes[i].box.Free;
           fLinesInfoBoxes[i].box := Nil;
         End;

  SetLength(fLinesInfoBoxes,1);
  fLinesInfoBoxes[0].Box := TStaticText.Create(Self);
  With fLinesInfoBoxes[0].Box Do
  Begin
    Hide;
    Parent := Self.fMBoxParent;
    AutoSize := False;
    BorderStyle :=  sbsSingle;
    Caption := '';
    Color := mBoxBackColor;
  End;
  fLinesInfoBoxes[0].defined := False;
  fLinesCpnBmpdef := False;
  
  Self.InitSummits;
  invalidate;
End;

procedure TQSpiderGraph.SetAxesSizeState(value:Boolean);
// The computing of the new min and max are done in the axes autosize
// setter. If lines are added to the graph later, the lines values
// setter will request a new computed at he next paint.
var i:Integer;
Begin
  self.fAxesAutoSized := value;
  For i := 0 To self.fAxesCount -1 Do
      fAxes[i].autoSized := value;
End;

Procedure TQSPiderGraph.SetTitleCaption(Value:String);
Begin
  self.fTitleCaption := value;
  DefineDrawSpace;
  invalidate;
End;

procedure TQSpiderGraph.SetTitlePosition(value:TQTitlePosition);
Begin
  self.fTitlePosition := value;
  If self.titleCaption <> '' Then invalidate;
End;

function  TQSpiderGraph.GetAxesByIndex(ix:Integer):TQSpiderAxe;//PQSpiderAxe;
Begin
  result := self.faxes[ix];
End;

procedure TQSPiderGraph.SetAxesFramed(value:Boolean);
Begin
  self.fAxesCapFramed := value;
  invalidate;
End;

procedure TQSpiderGraph.SetPolygonFill(value: Boolean);
Begin
  Self.fPolygonFill := value;
  Invalidate;
End;

procedure TQSpiderGraph.SetPolygonColor(value: TColor);
Begin
  Self.fPolygonColor := value;
  Invalidate;
End;


Function  TQSPiderGraph.GetMinValue(axeIx:integer; var minVal:single;
                                    cnd:Boolean = False) : TIntegerDynArray;
// axeIx  : index of the axe to be searched ;
// value  : the lowest value ;
// cnd    : "candidate" : minVal will be returned if no line has values
// result : array of  indexe(s) of the line(s) with the lowest value for this axe.

// If no line has yet been defined, GetMinValue will return either the actual fMin,
// (cnd=False) or minVal (cnd=True);

// Works in two passes, because the min value is not known until we've searched the
// entire lines collection. (or could have work in one pass, resetting the result
// array at each new lowest value.)

var i : Integer;
    mVal : Single;
Begin
  setLength(result,0);
  mVal := Infinity;                                         // <=> High(single);
  TRY
    // -1- Searches the lowest value :
    For i := 0 To self.fLinesCount -1 Do
        If fLines[i].fHasValues
           Then If {(IsInfinite(mVal)) or}  fLines[i].fValues[axeIx] < mVal
                Then mVal := fLines[i].fValues[axeIx];

    // -2- If no line has values, returns either this axes'min, or the value received,
    //     else searches the index of the corresponding line(s) :
    If IsInfinite(mVal)
       Then Begin If Not(cnd) Then minVal := self.fAxes[axeIx].fMin; End
       Else Begin
              If cnd Then minVal := mVal
              Else minVal := self.fAxes[axeIx].fMin;
              For i := 0 To self.fLinesCount -1 Do
              If fLines[i].fValues[axeIx] = mVal Then
              Begin
                setLength(result,Length(result)+1);
                result[length(result)-1] := i;
              End;
            End;
  EXCEPT;END;
End;

Function  TQSPiderGraph.GetMaxValue(axeIx:integer; var maxVal:single;
                                    cnd:Boolean = False)  : TIntegerDynArray;
// axeIx  : index of the axe to be searched ;
// value  : the highest value ;
// cnd    : "candidate" : minVal will be returned if no line has values
// result : array of  indexe(s) of the line(s) with the highest value for this axe.

// If no line has yet been defined, GetMaxValue will return either the actual fMax,
// (cnd=False) or maxVal (cnd=True);

var i : Integer;
    mVal : Single;
Begin
  setLength(result,0);
  mVal := NegInfinity;                                      // <=> Low(single);
  TRY
    // -1- Searches the highest value :
    For i := 0 To self.fLinesCount -1 Do
        If fLines[i].fHasValues
           Then If fLines[i].fValues[axeIx] > mVal
                Then mVal := fLines[i].fValues[axeIx];
                
    // -2- If no line has values, returns either this axes'max, or the value received,
    //     else searches the index of the corresponding line(s) :
    If IsInfinite(mVal) Then
    Begin
      If Not(cnd) Then maxVal := self.fAxes[axeIx].fMax;
    End
    Else Begin
           If cnd Then maxVal := mVal
           Else maxVal := self.fAxes[axeIx].fMax;
              For i := 0 To self.fLinesCount -1 Do
              If fLines[i].fValues[axeIx] = mVal Then
              Begin
                setLength(result,Length(result)+1);
                result[length(result)-1] := i;
              End;
            End;
  EXCEPT;END;
End;



                      // ---------------------- //
                      //     Lines (generic)    //
                      // ---------------------- //


procedure TQSpiderGraph.DefineLinesInfoBox(lineIx:Integer);
// Lines info boxes are the ones which appear next to the mouse pointer, and
// display the lines values, when mouseTracking is enabled.
// They are build once only, to avoid CPU time consuming at each mouse move
// (and obviously rebuild if any setting changes).

            { ------------------------  Local  ------------------------ }
              function ResizeCaption(aString:String;len:Integer):String;
              var i : Integer;
              Begin
                result := aString;
                For i := 0 To (len - Length(aString)) Do result := result + ' ';
              End;

              function CenterLine(aString:String;len:Integer):String;
              var i,target : Integer;
              Begin
                result := aString;
                target := len - Length(aString);
                For i := 0 To (target Div 2)   Do result := ' ' + result + ' ';
                For i := 0 To (target MOD 2)-1 Do result := result + ' ';
              End;
            { ------------------------ \Local\ ------------------------ }


var oldFont : TFont;
    tslAxesCap,
    tslLnsVals : TStringList;
    i,axMax,lineLen ,
    lnMax : Integer;
    bxTitle : String;
    
Begin
  TRY
    fLinesInfoBoxes[lineIx].box.Font.Name := 'Courier New';
    fLinesInfoBoxes[lineIx].box.Font.Size := 8;
  EXCEPT;
  END;
  fLinesInfoBoxes[lineIx].box.color := mBoxBackColor;
  If mBoxUsesLnColor 
     Then fLinesInfoBoxes[lineIx].box.Font.Color := fLines[lineIx].Color
     Else fLinesInfoBoxes[lineIx].box.Font.Color := mBoxForColor;

  axMax := 0; lnMax := 0;
  tslAxesCap := TStringList.Create;
  tslLnsVals := TStringList.Create;
  tslAxesCap.Capacity := self.fAxesCount +1;
  tslLnsVals.Capacity := self.fAxesCount +1;

  For i := 0 To self.fAxesCount -1 Do
  Begin                                                    
    tslAxesCap.Add( fAxes[i].caption );
    tslLnsVals.Add( FloatToStrF( fLines[lineIx].values[i], ffGeneral, 7, 2) );
    axMax := Max( axMax, Length( fAxes[i].caption ) );
    lnMax := Max( lnMax, Length( tslLnsVals.Strings[i] ) );
  End;

  // resizes lines
  For i := 0 To Self.fAxesCount -1 Do
    tslAxesCap.Strings[i] := ' ' + ResizeCaption( tslAxesCap.Strings[i], axMax )
                           + ': ' + tslLnsVals.Strings[i] ;
  If fLines[lineIx].caption = ''
     Then bxTitle := 'Line N '+ intToStr(lineIx)
     Else bxTitle := fLines[lineIx].caption;
  lineLen := Max( axMax + lnMax + 5, Length( bxTitle )+2 ) ;                    //"1 + axMax + 3 + lnMax + 1";

  // Builds title
  bxTitle := CenterLine(bxTitle,lineLen);

  // Fill this boxe's caption
  fLinesInfoBoxes[lineIx].box.Caption := bxTitle;
  For i := 0 To self.fAxesCount -1 Do
      fLinesInfoBoxes[lineIx].box.Caption := fLinesInfoBoxes[lineIx].box.Caption + #13#10 + tslAxesCap.Strings[i];

  oldFont := canvas.Font;
  canvas.Font := fLinesInfoBoxes[lineIx].box.Font;
  bxTitle := '';                                            // re-used (no longer needed for the title);
  fLinesInfoBoxes[lineIx].box.Height := (Canvas.TextHeight('Ip')+2)*(fAxesCount+2);
  fLinesInfoBoxes[lineIx].box.Width  :=  Canvas.TextWidth( ResizeCaption( bxTitle, lineLen ));
  canvas.Font := oldFont ;
  tslAxesCap.Free;
  tslLnsVals.Free;

  // decides which is the parent : The graph or a form ?
  If Assigned(self.fMBoxParent) Then
     fLinesInfoBoxes[lineIx].box.Parent := self.fMBoxParent;

  fLinesInfoBoxes[lineIx].defined := True;
End;


procedure TQSpiderGraph.SetLinesCount(value:Integer);
var i : Integer;
Begin
  // Free previous lines
  For i := 0 To self.fLinesCount -1 Do fLines[i].free;
  SetLength(fLines,0);

  // corresponding infoBoxes
  For i := 0 To Length(fLinesInfoBoxes) -1 Do
      If Assigned(fLinesInfoBoxes[i].box ) Then fLinesInfoBoxes[i].box.free;

  // Define new ones
  If value < 2 Then
  Begin
    self.fMultilines := False;
    self.fLinesCount := 1;
    SetLength(fLines,1);
    fLines[0] := TQSGLine.CreateOwned(Self);
  End Else
  Begin
    self.fMultilines := True;
    self.fLinesCount := value;
    SetLength(fLines, value);
    For i := 0 To value -1 Do fLines[i] := TQSGLine.CreateOwned(Self)
  End;
  // corresponding boxes
  SetLength(fLinesInfoBoxes,self.fLinesCount);
  For i := 0 To self.fLinesCount -1 Do
  Begin
    fLinesInfoBoxes[i].Box := TStaticText.Create(Self);
    With fLinesInfoBoxes[i].Box Do
    Begin
      Hide;
      Parent := Self.fMBoxParent;
      AutoSize := False;
      BorderStyle :=  sbsSingle;
      Caption := '';
      Color := mBoxBackColor;                        
    End ;
    fLinesInfoBoxes[i].defined := False;
  End;
  fLinesCpnBmpdef := False;
End;

function  TQSpiderGraph.GetLinesByIndex(ix:Integer):TQSGLine;
// Returns a pointer to the specified line object
// CARE : If accessed before the lineCount has been set, or with an index above
// lineCount -1, will return a NIL pointer, and your app will show the classical
// eAccessViolation message ...
Begin
  If (ix >= fLinesCount) or (ix < 0)
     Then result := Nil
     Else result := fLines[ix];                    // D6 help : dynmic assignment of arrays alloweded.
End;

function TQSpiderGraph.RemoveLine(lineIndex :Integer):Boolean;
var i : integer;
Begin
  Result := True;
  TRY
    If lineIndex > fLinesCount -1 Then exit;

    // Free fLines[lineIndex]
    If Assigned(fLines[lineIndex]) Then fLines[lineIndex].free;
    For i := lineIndex To fLinesCount -2 Do
        fLines[i] := fLines[i+1];
    SetLength(fLines, Length(fLines)-1);
  
    // Free corresponding infoBox
    If Assigned(fLinesInfoBoxes[lineIndex].box ) Then fLinesInfoBoxes[lineIndex].box.free;
    For i := lineIndex To fLinesCount -2 Do
        fLinesInfoBoxes[i] := fLinesInfoBoxes[i+1];
    SetLength(fLinesInfoBoxes, Length(fLinesInfoBoxes)-1);

    Dec(fLinesCount);
    fLinesCpnBmpdef := False;
    Invalidate;
  EXCEPT; result := False;
  END;
End;

function TQSpiderGraph.AddLine(Const vals:TQSingleArray):Integer;
Begin
  TRY
    // add one line to the collection
    If (fLinesCount = 1) And (fLines[0].fHasValues = False) Then
    Begin
      If Assigned(fLines[0]) Then fLines[0].Free;
      SetLength(fLines, 0);
      fLinesCount := 0;
      If Assigned(fLinesInfoBoxes[0].box ) Then fLinesInfoBoxes[0].box.free;
    End;
    SetLength(fLines, Length(fLines)+1);
    fLinesCount := Length(fLines);
    fMultilines := fLinesCount > 1;
    result := fLinesCount -1;
    fLines[result] := TQSGLine.CreateOwned(Self);
    fLines[result].values := vals;
    // prepares the corresponding infoBox
    SetLength(fLinesInfoBoxes,fLinesCount);  
    fLinesInfoBoxes[result].Box := TStaticText.Create(Self);
    With fLinesInfoBoxes[result].Box Do
    Begin
      Hide;
      Parent     := Self.fMBoxParent;
      AutoSize   := False;
      BorderStyle:= sbsSingle;
      Caption    := '';
      Color      := mBoxBackColor;
    End ;
    fLinesInfoBoxes[result].defined := False;
    fLinesCpnBmpdef := False;
    invalidate;
  EXCEPT; result := -1 ;
  END;
End;


procedure TQSpiderGraph.SetDfltPenWidth(value:Word);
var i:integer;
Begin
  If (value < 1) or (value > 255) Then value := 1;
  self.fDfltPenWidth := value; 
  For i:= 0 To Self.fLinesCount -1 Do
  Begin
    self.fLines[i].penWidth := value;
  End;
  invalidate;
End;

procedure TQSpiderGraph.SetShowLnPoints(value: Boolean);
var i : integer;
Begin
  Self.fShowLnPoints := Value;
  For i:= 0 To self.fLinesCount -1 Do
    self.fLines[i].showPoints  := Value;
  invalidate;
End;

procedure TQSPiderGraph.SetShowLinesCpn(value: Boolean);
Begin
  self.fShowLinesCpn := Value;
  Invalidate;
End;

procedure TQSpiderGraph.SetLinesCpnBmpTrp(value: Boolean);
Begin
  self.fLinesCpnBmpTrp := Value;
  self.fLinesCpnBmpdef := False;
  invalidate;
End;

procedure TQSpiderGraph.SetLinesCpnBmpClr(value: TColor);
Begin
  self.fLinesCpnBmpClr := Value;
  self.fLinesCpnBmpdef := False;
  Invalidate;
End;


                      // ---------------------- //
                      //     Mouse (generic)    //
                      // ---------------------- //


procedure TQSpiderGraph.SetMBoxParent(value : TWinControl);
// Allows the mouse info boxes to be displayed on the whole form area,
// instead of the graph only;
var i : Integer;
Begin
  If (value = Nil) or Not(value is TForm) then value := Self;
  Self.fMBoxParent := value;
  For i := 0 To Self.fLinesCount -1 Do
      If Assigned(fLinesInfoBoxes[i].box) Then
         fLinesInfoBoxes[i].box.Parent := value;
End;

procedure TQSpiderGraph.SetTrackMsMoves(value: Boolean);
// If set to False, ensure that there is no box actually visible
var i : Integer;
Begin
  Self.ftrackMsMoves:=Value;
  If not(Value) Then
     For i := 0 To Self.fLinesCount -1 Do
         If (fLinesInfoBoxes[i].box.Visible) Then
         Begin
           fLinesInfoBoxes[i].box.Hide;
           Invalidate;
         End;
end;

procedure TQSpiderGraph.SetShowMouseBox(value: Boolean);
// If set to False, ensure that there is no box actually visible
var i : Integer;
Begin
  Self.fShowMouseBox := Value;
  If not(Value) Then
     For i := 0 To Self.fLinesCount -1 Do
         If (fLinesInfoBoxes[i].box.Visible) Then
         Begin
           fLinesInfoBoxes[i].box.Hide;
           Invalidate;
         End;
End;

procedure TQSpiderGraph.SetMBoxForColor(value:TColor);
var i : Integer;
Begin
  self.fMBoxForColor := value;
  For i := 0 To Length(fLinesInfoBoxes) -1 Do
      fLinesInfoBoxes[i].defined := False;                  // set .defined to false to have the box computed again
End;

procedure TQSpiderGraph.SetMBoxBackColor(value:TColor);
var i : Integer;
Begin
  self.fMBoxBackColor := value;
  For i := 0 To Length(fLinesInfoBoxes) -1 Do
      fLinesInfoBoxes[i].defined := False;                  // set .defined to false to have the box computed again
End;

procedure TQSpiderGraph.SetMBoxUsesLnClr(value:Boolean);
var i : Integer;
Begin
  self.fMBoxUsesLnClr := value;
  For i := 0 To Length(fLinesInfoBoxes) -1 Do
      fLinesInfoBoxes[i].defined := False;                  // set .defined to false to have the box computed again
End;



                      // ---------------------- //
                      //        HIGHLIGHT       //
                      // ---------------------- //


function TQSpiderGraph.GetBestLineByArea(maxWanted: Boolean = True): TIntegerDynArray;
// NIH : Polygon area computing can be find anywhere on the web...
// Returns an array of integers containing the indexes of the line(s)
// with the widest area. return[0] is worth -1 if a probleme occured
// or no line has values.
// length(return) = fLinesCount if all areas are strictly equal...

var i,lc,iTmp : Integer;
    arBestArea : Array of Real;
    bestArea : Real;
    valuesRead : Boolean;
Begin
  SetLength(result,1);
  result[0] := -1;
  SetLength(arBestArea,fLinesCount);
  valuesRead := False;

  If maxWanted
     Then bestArea := 0
     Else bestArea := Infinity;

  TRY
  // searches best area
  For lc := 0 To fLinesCount-1 Do
  Begin
    iTmp := 0;
    For i := 0 To fAxesCount -1 Do
    Begin
      If fLines[lc].fHasValues Then valuesRead := True;
      iTmp := iTmp + fLines[lc].fPoints[i].X   * fLines[lc].fPoints[i+1].Y
                   - fLines[lc].fPoints[i+1].X * fLines[lc].fPoints[i].Y;
       arBestArea[lc] := abs(iTmp / 2);
    End;
    If maxWanted Then
    Begin
      If arBestArea[lc] > bestArea Then bestArea := arBestArea[lc]
    End Else
      If arBestArea[lc] < bestArea Then bestArea := arBestArea[lc];
  End;

  // populates result
  If (bestArea <=  0) or Not(valuesRead) Then Exit;
  SetLength(result,0);                                      // if still there, resize the result, to ease next step.
  For lc := 0 To fLinesCount-1 Do
  Begin
    If arBestArea[lc] = bestArea Then
    Begin
      SetLength(result,Length(result)+1);
      result[length(result)-1]:= lc;
    End;
  End;
  If Length(result) = 0 Then                                // useless (?)
  Begin
    SetLength(result,1);
    result[0] := -1;
  End;
  EXCEPT;END;
End;

function TQSpiderGraph.GetBestLineByAxe(axeIx:Integer;maxWanted: Boolean = True) : TIntegerDynArray;
var lc : Integer;
    bestLine : Real;
    valuesRead : Boolean;
Begin
  SetLength(result,1);
  result[0] := -1;

  If (axeIx > (self.fAxesCount -1)) or (axeIx < 0) Then Exit;
  TRY
    If maxWanted Then bestLine := fAxes[axeIx].fMin
                 Else bestLine := fAxes[axeIx].fMax;

    valuesRead := False;                                    // There can be lines (due to the lineCount setter) ...
    // searches the best(s) line(s)                         // ... but these may not have yet values. So, check it.
    For lc := 0 To fLinesCount-1 Do
      If fLines[lc].fHasValues Then
      Begin
        valuesRead := True;
        If maxWanted Then
        Begin
          If fLines[lc].values[axeIx] > bestLine Then bestLine := fLines[lc].values[axeIx];
        End Else
          If fLines[lc].values[axeIx] < bestLine Then bestLine := fLines[lc].values[axeIx];
      End;

    // populates result
    If Not(valuesRead) Then Exit;                           // No line has values, get "outta here"
    SetLength(result,0);                                    // If still there, resize the result, to ease next step.
    For lc := 0 To fLinesCount-1 Do
    Begin
      If fLines[lc].values[axeIx] = bestLine Then
      Begin
        SetLength(result,Length(result)+1);
        result[length(result)-1]:= lc;
      End;
    End;

    If Length(result) = 0 Then                              // useless (?)
    Begin
      SetLength(result,1);
      result[0] := -1;
    End;
  EXCEPT;END;
End;


procedure TQSpiderGraph.ResetLinesAppearance;
var i : integer;
Begin
  For i := 0 To self.fLinesCount-1 Do
  Begin
    fLines[i].fPenWidth   := fLines[i].fPenWidthMem;
    fLines[i].fShowPoints := fLines[i].fShowPointsMem;
    fLines[i].fColor      := fLines[i].fColorMem;
    fLines[i].fShow       := True;                          // Anycase ...
  End;
End;


procedure TQSpiderGraph.FlashLine(targets:TIntegerDynArray);
// "Flashes" the line(s). At end, they are in their original state.
var i, lc, tgtCount:Integer;
    gtc : Cardinal;
Begin
  If Length( targets ) < 1 then exit;
  tgtCount := Length( targets )-1;
  For i:= 0 To 7 Do
  Begin
    For lc := 0 To tgtCount Do
        fLines[targets[lc]].fColor := highlightColor;
    gtc := GetTickCount + 50;
    Invalidate;
    Repeat
      Application.ProcessMessages;
    Until
      GetTickCount > gtc ;

    For lc := 0 To tgtCount Do
        fLines[targets[lc]].fColor := fLines[targets[lc]].fColorMem;
    gtc := GetTickCount + 50;
    Invalidate;
    Repeat
      Application.ProcessMessages;
    Until
      GetTickCount > gtc ;
  End;
End;


procedure TQSpiderGraph.HighlightLineByCrit(criteria:Integer;maxWanted:Boolean=True);
// rq :  Const HC_NONE=-1; HC_AREA=-2;
// => :  -2  <  criteria  <  self.fAxesCount-1 ;
var arBests:TIntegerDynArray;
Begin
  If (criteria < HC_AREA) or (criteria > fAxesCount-1)      // axes indexes are ZERO-BASED.
     Then criteria := HC_NONE;
  SetLength(arBests,0);
  ResetLinesAppearance;
  Case criteria of
   HC_NONE : Begin Invalidate; Exit; End;
   HC_AREA : arBests := GetBestLineByArea(maxWanted);
   Else      arBests := GetBestLineByAxe(criteria, maxWanted);
  End; {case}

  If arBests[0] <> -1 Then
     Self.HighlightLineByIndex(arBests);
End;


procedure TQSpiderGraph.HighlightLineByIndex(index:Integer);
var wArr : TIntegerDynArray;
Begin
  SetLength(wArr,1);
  wArr[0] := index;
  self.HighlightLineByIndex(wArr);
End;

procedure TQSpiderGraph.HighlightLineByIndex(indexArray:TIntegerDynArray);
var i,wCnt : integer;
    wArr   : TIntegerDynArray;
Begin
  TRY
    ResetLinesAppearance();
    If ( Length(indexArray) = 0) or ( Length(indexArray) > fLinesCount) Then exit;
    // Rejects invalid indexes. As long as ResetLinesAppearance(); has been called,
    // an eventual reset request (throught (index=-1) ) has soon been applied.
    // We can then discard any index outside the 0..flinesCount-1 range
    wCnt := 0;
    SetLength(wArr, length(indexArray));
    For i := 0 To length(indexArray) -1 Do
        If (indexArray[i] >= 0)  And  ((indexArray[i] < fLinesCount))
        Then Begin
                wArr[wCnt] := indexArray[i];
                inc(wCnt);
             End;
    If wCnt = 0 Then exit Else SetLength(wArr,wCnt);
    Dec(wCnt);                                                // will then be used bellow as an <=> for "length(wArr)-1"
    If hmShowPoints in self.HighlightMode Then
    Begin
      For i := 0 To fLinesCount -1  Do fLines[i].fShowPoints := False;
      For i := 0 To wCnt  Do fLines[wArr[i]].fShowPoints := True;
    End;
    If hmWidened in self.HighlightMode Then
    Begin
      For i := 0 To fLinesCount -1     Do fLines[i].fPenWidth := self.fDfltPenWidth;
      For i := 0 To wCnt  Do fLines[wArr[i]].fPenWidth   := 2* self.fDfltPenWidth;
    End;

    // Then, the eventual flashing, but not in the loop;
    If hmFlashLine in self.HighlightMode Then FlashLine(indexArray);
    // Then the color, if requested;
    For i := 0 To wCnt Do
        If hmColorLine in self.HighlightMode Then fLines[wArr[i]].fColor  := highlightColor;
  // finaly, repaint the whole thing.
  FINALLY
    Invalidate;
  END;
End;

                      // ---------------------- //
                      //        THE END...      //
                      // ---------------------- //

end.
