unit FormatLabel;

{
  Format Label
  ============
  Date: Sep 2003
  Author: Rosi (http://sweb.cz/rosisoft)

  Description:
  FormatLabel is enhanced TLabel component which provides functionality
  for displaying caption with more font styles and colors according
  HTML tags in caption string.

  Caption can contain parameters (between <VAR> and </VAR> tags), for each
  parameter will be called OnGetParameter events.
  FormatLabel can also display links (anchor) and can handle click on it.

  FormatLabel can display caption with ellipsis, with 3D effect and beveled.
  For very long caption can be used new memo property - CaptionStrings.

  Supported HTML tags:
  <B>, </B>, <I>, </I>, <U>, </U>, <BIG>, </BIG>, <SMALL>, </SMALL>,
  <A HREF=target>  for automatic launching of URL target (http, mailto, file etc.),
  <A NAME=param> for calling of user defined events,
  </A>,
  <COLOR=Red>,
  <BR>, <TAB>
  <VAR>, </VAR> - defines parameter

  Note:
  Full functional demo
{}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, StdCtrls, Forms,
  Dialogs, ShellApi;

type
  TEllipsisType=(etNone, etPath, etEnd);

  TAnchorInfo=class
    LinkType: string;
    LinkName: string;
    Rect: TRect;
  end;

  TAnchorClickEvent=procedure(Sender: TObject; LinkName: string) of object;
  TGetParameter=procedure(Sender: TObject; var ParamStr: string) of object;

  TFormatLabel = class(TLabel)
  private
    CaptionAsHint: boolean;
    ColorEnabled: boolean;
    BeforeAnchorFont: TFontRecall;
    actAnchorInfo: TAnchorInfo;
    actAnchorItem: integer;
    AnchorInfoList: TList;
    IsAnchor: boolean;
    FCaptionStrings: TStrings;
    FEllipsisType: TEllipsisType;
    FShowText3D: boolean;
    FShowBeveled: boolean;
    FOnAnchorClick: TAnchorClickEvent;
    FOnGetParameter: TGetParameter;
    procedure FreeAnchorInfo;
    function FindFirstFrm(s: string): integer;
    procedure ProcFrm(var s: string; Calc: boolean);
    procedure SetEllipsisType(Value: TEllipsisType);
    procedure SetCaptionStrings(Value: TStrings);
    procedure SetShowText3D(Value: boolean);
    procedure SetShowBeveled(Value: boolean);
    procedure CMMouseLeave(var AMsg: TMessage); message CM_MOUSELEAVE;
  protected
    procedure Loaded; override;
    function GetLabelText: string; override;
    procedure SetAutoSize(Value: Boolean); override;
    procedure DoDrawText(var Rect: TRect; Flags: Longint); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure Click; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property CaptionStrings: TStrings
      read FCaptionStrings write SetCaptionStrings;
      {CaptionStrings are used as Caption}
    property EllipsisType: TEllipsisType
      read FEllipsisType write SetEllipsisType;
      {Defines type of ellipsis - None, End, Path}
    property ShowText3D: boolean
      read FShowText3D write SetShowText3D;
    property ShowBeveled: boolean
      read FShowBeveled write SetShowBeveled;
    property OnAnchorClick: TAnchorClickEvent
      read FOnAnchorClick write FOnAnchorClick;
    property OnGetParameter: TGetParameter
      read FOnGetParameter write FOnGetParameter;
  end;

procedure Register;

implementation

uses rtool;

{$R *.res}

const
  FrmStrCount=13;
  FrmStr: array [1..FrmStrCount] of string =
    ('<B>','</B>','<I>','</I>','<U>','</U>',
     '<BIG>','</BIG>','<SMALL>','</SMALL>',
     '<A','</A>',
     '<COLOR=');

  FrmAnchorStartNum=11;
  FrmAnchorStopNum=12;
  FrmColorNum=13;
  FrmTAB='<TAB>';
  FrmCR='<BR>';
  FrmStop='>';
  FrmParamStart='<VAR>';
  FrmParamStop='</VAR>';

procedure Register;
begin
  RegisterComponents('Rosi', [TFormatLabel]);
end;

constructor TFormatLabel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCaptionStrings:=TStringList.Create;
  AnchorInfoList:=TList.Create;
  actAnchorItem:=-1;
end;

destructor TFormatLabel.Destroy;
begin
  FCaptionStrings.Free;
  FreeAnchorInfo;
  AnchorInfoList.Free;
  inherited Destroy;
end;

procedure TFormatLabel.Loaded;
begin
  inherited Loaded;
  CaptionAsHint:=Hint='';
end;

procedure TFormatLabel.FreeAnchorInfo;
var i: integer;
begin
  for i:=0 to AnchorInfoList.Count-1 do
  begin
    actAnchorInfo:=AnchorInfoList.Items[i];
    actAnchorInfo.Free;
  end;
  AnchorInfoList.Clear;
  IsAnchor:=false;
end;

function TFormatLabel.FindFirstFrm(s: string): integer;
var a, p, m: integer;
begin
  p:=Length(s)+1;
  for a:=1 to FrmStrCount do
  begin
    m:=AnsiPosEx(FrmStr[a],s,1,false);
    if (m>0) and (m<p) then p:=m;
  end;
  if p=Length(s)+1 then Result:=0
                   else Result:=p;
end;

procedure TFormatLabel.ProcFrm(var s: string; Calc: boolean);
var a: integer;

  function ExtractFrmParam(FrmStr: string): string;
  var i,l: integer;
  begin
    Result:='';
    i:=AnsiPosEx(FrmStop,s,1,false);
    if i=0 then Exit;
    l:=Length(FrmStr);
    Result:=copy(s,l+1,i-l-1);
    s:=FrmStr+copy(s,i+1,Length(s));
  end;

  procedure ProcAnchorStart;
  var temp: string;
  begin
    //Find anchor info
    temp:=ExtractFrmParam(FrmStr[FrmAnchorStartNum]);
    // Set anchor font and color
    BeforeAnchorFont:=TFontRecall.Create(Canvas.Font);
    Canvas.Font.Style:=Canvas.Font.Style+[fsUnderline];
    if ColorEnabled then
    begin
      if AnchorInfoList.Count=actAnchorItem then Canvas.Font.Color:=clHotLight
                                            else Canvas.Font.Color:=clHighLight;
    end;
    // Save anchor info
    if not Calc then
    begin
      IsAnchor:=true;
      actAnchorInfo:=TAnchorInfo.Create;
      actAnchorInfo.LinkType:=AnsiUpperCase(Trim(FindStrPart(temp,'=',1,false)));
      actAnchorInfo.LinkName:=Trim(FindStrPart(temp,'=',2,false));
    end;
  end;

  procedure ProcAnchorStop;
  begin
    try
      BeforeAnchorFont.Free;
    except
    end;
    if IsAnchor and not Calc then
    begin
      IsAnchor:=false;
      AnchorInfoList.Add(actAnchorInfo);
    end;
  end;

  procedure ProcColor;
  var temp: string;
      Color: LongInt;
  begin
    temp:=ExtractFrmParam(FrmStr[FrmColorNum]);
    if ColorEnabled and IdentToColor('cl'+temp,Color) then
      Canvas.Font.Color:=Color;
  end;

begin
  for a:=1 to FrmStrCount do
  begin
    if AnsiPosEx(FrmStr[a],s,1,false)=1 then
    with Canvas.Font do
    begin
      case a of
        1: Style:=Style+[fsBold];
        2: Style:=Style-[fsBold];
        3: Style:=Style+[fsItalic];
        4: Style:=Style-[fsItalic];
        5: Style:=Style+[fsUnderline];
        6: Style:=Style-[fsUnderline];
        7: Size:=Size+2;
        8: Size:=Font.Size;
        9: Size:=Size-2;
        10: Size:=Font.Size;
        FrmAnchorStartNum: ProcAnchorStart;
        FrmAnchorStopNum: ProcAnchorStop;
        FrmColorNum: ProcColor;
      end;
      s:=copy(s,Length(FrmStr[a])+1,Length(s));
    end;
  end;
end;

procedure TFormatLabel.SetEllipsisType(Value: TEllipsisType);
begin
  FEllipsisType:=Value;
  if Value<>etNone then AutoSize:=false;
  Invalidate;
end;

procedure TFormatLabel.SetCaptionStrings(Value: TStrings);
begin
  FCaptionStrings.Assign(Value);
  Caption:='';
  Invalidate;
  AdjustBounds;
end;

procedure TFormatLabel.SetShowText3D(Value: boolean);
begin
  FShowText3D:=Value;
  Invalidate;
end;

procedure TFormatLabel.SetShowBeveled(Value: boolean);
begin
  FShowBeveled:=Value;
  Invalidate;
end;

procedure TFormatLabel.SetAutoSize(Value: Boolean);
begin
  if Value then EllipsisType:=etNone;
  inherited SetAutoSize(Value);
end;

function TFormatLabel.GetLabelText: string;
var s,param: string;
begin
  s:=Caption;
  if s='' then s:=TrimRight(FCaptionStrings.Text);
  // proceed parameters
  repeat
    param:=FindStrFromTo(FrmParamStart,FrmParamStop,s,false);
    if param='' then Break;
    param:=copy(param,Length(FrmParamStart)+1,Length(param));
    if (csDesigning in ComponentState) then param:='<'+param+'>'
    else if Assigned(FOnGetParameter) then FOnGetParameter(Self,param)
                                      else param:='';
    param:=ClearStr(FrmParamStart,param,false);
    param:=ClearStr(FrmParamStop,param,false);
    s:=ReplaceStrFromTo(FrmParamStart,FrmParamStop,param,s,true,false,false);
  until false;
  // replace spec chars
  s:=ReplaceStr(FrmTAB,#9,s,true,false);
  s:=ReplaceStr(FrmCR,#13,s,true,false);
  s:=ReplaceStr(#10,'',s,true,false);
  // replace empty lines with space
  if copy(s,1,1)=#13 then s:=' '+s;
  while Pos(#13#13,s)>0 do s:=ReplaceStr(#13#13,#13+' '+#13,s,true,false);
  if not (csDesigning in ComponentState) and CaptionAsHint then Hint:=s;
  Result:=s;
end;

procedure TFormatLabel.DoDrawText(var Rect: TRect; Flags: Longint);
var Text: string;
    LineCount, X, W, H, MaxW: integer;
    s: string;
    Rc, Rd: TRect;
    oldFont: TFontRecall;
    oldPen: TPenRecall;
    CalcOnly: boolean;

  procedure DrawTextPart(var R: TRect; F: LongInt; s: string);
  begin
    if s='' then s:=' ';
    // Save anchor start point
    if IsAnchor then actAnchorInfo.Rect:=R;
    DrawText(Canvas.Handle,PChar(s),Length(s),R,F);
    // Save anchor stop point
    if IsAnchor then
    begin
      actAnchorInfo.Rect.Right:=actAnchorInfo.Rect.Left+Canvas.TextWidth(s);
      actAnchorInfo.Rect.Bottom:=actAnchorInfo.Rect.Top+Canvas.TextHeight(s);
    end;
    Inc(W,R.Right-R.Left);
    R.Left:=R.Left+Canvas.TextWidth(s);
  end;

  procedure ProcLine(var R: TRect; F: LongInt; Calc: boolean; s: string);
  var p: integer;
  begin
    if Calc then F:=F or DT_CALCRECT;
    while true do
    begin
      p:=FindFirstFrm(s);
      if p=0 then Break;
      if p>1 then DrawTextPart(R,F,copy(s,1,p-1));
      s:=copy(s,p,Length(s));
      ProcFrm(s,Calc);
    end;
    if s<>'' then DrawTextPart(R,F,s);
    if IsAnchor then
    begin
      // End of line without </A>; add it
      s:=FrmStr[FrmAnchorStopNum];
      ProcFrm(s,Calc);
    end;
  end;

begin
  CalcOnly:=(Flags and DT_CALCRECT)<>0;
  Text := GetLabelText;
  ColorEnabled:=true;
  // set ellipsis Flags
  if FEllipsisType=etEnd then Flags:=Flags or DT_END_ELLIPSIS;
  if FEllipsisType=etPath then Flags:=Flags or DT_PATH_ELLIPSIS;

  if FShowBeveled and (not CalcOnly) then
  begin
    // Draw bevel
    oldPen:=TPenRecall.Create(Canvas.Pen);
    Canvas.Pen.Color:=clBtnShadow;
    Canvas.PolyLine([Point(0, Height-1),
                     Point(0, 0),
                     Point(Width-1, 0)]);
    Canvas.Pen.Color:=clBtnHighlight;
    Canvas.PolyLine([Point(Width-1, 0),
                     Point(Width-1, Height-1),
                     Point(0, Height-1)]);
    oldPen.Free;
  end;

  if (not FShowText3D) and (FindFirstFrm(Text)=0) then
  begin
    // Without formating => standard DoDrawText
    inherited DoDrawText(Rect,Flags);
    Exit;
  end;

  // Exteded formating
  // Cancel WordWrap and Ellipsis for formated text
  // Alignment is done not by Flags
  Flags:=Flags and
    (not (DT_WORDBREAK+DT_RIGHT+DT_CENTER+DT_END_ELLIPSIS+DT_PATH_ELLIPSIS));
  Flags:=Flags or DT_NOPREFIX;
  Flags := DrawTextBiDiModeFlags(Flags);

  Canvas.Font := Font;
  Rd:=Rect;
  LineCount:=1;
  maxW:=0;
  H:=0;
  FreeAnchorInfo;
  while true do
  begin
    // Find and proc one line
    s:=FindStrPart(Text,#13,LineCount,false);
    if s='' then Break;

    // calc width of the line
    Rc:=Rect;
    W:=0;
    if CalcOnly then ProcLine(Rc,Flags,true,s)
    else
    begin
      // Reset font back after proc line for Drawing
      oldFont:=TFontRecall.Create(Canvas.Font);
      ProcLine(Rc,Flags,true,s);
      oldFont.Free;
    end;
    // store maxWidth and Height
    if W>maxW then maxW:=W;
    Inc(H, Rc.Bottom-Rc.Top);

    if not CalcOnly then
    begin
      // draw the line
      case Alignment of
        taCenter: X:=(Rect.Right-Rect.Left-W) div 2 + Rect.Left;
        taRightJustify: X:=Rect.Right-W-1;
        else X:=Rect.Left+1;
      end;
      Rd.Left:=X;

      if (not Enabled) or FShowText3D then
      begin
        // Draw shadow
        ColorEnabled:=false;
        oldFont:=TFontRecall.Create(Canvas.Font);
        OffsetRect(Rd, 1, 1);
        if not Enabled then Canvas.Font.Color:=clBtnHighlight
                       else Canvas.Font.Color:=clBtnShadow;
        ProcLine(Rd,Flags,false,s);
        oldFont.Free;
        OffsetRect(Rd, -1, -1);
        Rd.Left:=X;
        if not Enabled then Canvas.Font.Color := clBtnShadow;
      end;
      ColorEnabled:=Enabled;
      ProcLine(Rd,Flags,false,s);

      // Set Rect for next line
      Rd.Top:=Rect.Top+H;
    end;
    Inc(LineCount);
  end;
  // Set calculated bounds
  Rect.Right:=Rect.Left+maxW+2;
  Rect.Bottom:=Rect.Top+H+2;
end;

procedure TFormatLabel.MouseMove(Shift: TShiftState; X, Y: Integer);
var oldAct, i: integer;
begin
  oldAct:=actAnchorItem;
  actAnchorItem:=-1;
  // Find active anchor
  for i:=0 to AnchorInfoList.Count-1 do
  begin
    actAnchorInfo:=AnchorInfoList.Items[i];
    if (X>actAnchorInfo.Rect.Left) and (X<actAnchorInfo.Rect.Right)
       and (Y>actAnchorInfo.Rect.Top) and (Y<actAnchorInfo.Rect.Bottom) then
    begin
      actAnchorItem:=i;
      Break;
    end;
  end;
  if actAnchorItem<>-1 then Screen.Cursor:=crHandPoint
                       else Screen.Cursor:=Cursor;
  if oldAct<>actAnchorItem then Repaint;
  inherited;
end;

procedure TFormatLabel.CMMouseLeave(var AMsg: TMessage);
begin
  Screen.Cursor:=Cursor;
  if actAnchorItem<>-1 then
  begin
    actAnchorItem:=-1;
    Repaint;
  end;
end;

procedure TFormatLabel.Click;
begin
  if actAnchorItem<>-1 then
  begin
    Screen.Cursor:=Cursor;
    actAnchorInfo:=AnchorInfoList.Items[actAnchorItem];
    if actAnchorInfo.LinkType='HREF' then
      ShellExecute(Application.Handle,'open',PChar(actAnchorInfo.LinkName),
                   nil,nil,SW_NORMAL)
    else
      if Assigned(FOnAnchorClick) then FOnAnchorClick(Self,actAnchorInfo.LinkName);
  end;
  inherited;
end;

end.
