{ Test Application for UnitList }
unit UnitMain;

{ ContinuIT Free Library License (CFLL)

  This source code is owned and only owned by ContinuIT BV.
  (C) 2002 ContinuIT BV, All rights reserved.

  Permission is granted to anyone to use this software for any kind of application, and to
  alter it and redistribute it freely, subject to the following restrictions:
  - The origin of this software must not be misrepresented, you must not claim that you wrote
    the original software.
  - Altered source versions must be plainly marked as such, and must not be misrepresented as
    being the original software.
  - You may not create a library that uses this library as a main part and sell that library.
  - You must have a visible line in your programs about box and/or documentation stating that this
    ContinuIT library is used, with a link where this library can be found.
  - This notice may not be removed or altered from any source distribution.
  - This software is provided 'as-is', without any expressed or implied warranty. In no event will
    the author(s) or ContinuIT BV be held liable for any damages arising from the use of this
    software.

  This license is subject to Dutch Law. }

{ ContinuIT Library License (CLL):

  (C) 2002 ContinuIT BV, All rights reserved.
  This source code is owned and only owned by ContinuIT BV.

  Usage of this source code is granted if, and only if, all requirements stated below are met:
  - You have all rights of at least one application initially developed by ContinuIT BV.
  - This source code is passed to you by ContinuIT BV. during the delivery of that application.
  - You use this source code for compilation of that application only.
  - This notice may not be removed or altered from any source distribution.
  - You are NOT ALLOWED to sell, publicize, make available to third parties, use it in other than
    the original delivered application, or change this source code without prior written approval
    from ContinuIT BV.

  This license is subject to Dutch Law. }

{ History:
  2002-11-20  ritsaert@continuit.nl       Initial version.
  2003-01-07  ritsaert@continuit.nl       Latest version.
  2004-01-08  ritsaert@continuit.nl       Moved to CFLL.
  2004-02-10  ritsaert@continuit.nl       Added some examples. }

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, UnitList, ImgList, ActnList, StdStyleActnCtrls, ActnMan,
  StdCtrls, ToolWin, ActnCtrls, ActnMenus;

type
  TfrmMain = class(TForm)
    amActions: TActionManager;
    ilImages: TImageList;
    MainMenu: TActionMainMenuBar;
    mmResult: TMemo;
    acUnitHashIDTest: TAction;
    acUnitHashIDSpeedTest: TAction;
    asUnitHashNameTest: TAction;
    acExit: TAction;
    acUnitHashNameSpeedTest: TAction;
    acUnitHashNameIDSpeedTest: TAction;
    acUnitHashNameIDTest: TAction;
    acSortingTest: TAction;
    acDerivedExample: TAction;
    procedure acUnitHashIDTestExecute(Sender: TObject);
    procedure acUnitHashIDSpeedTestExecute(Sender: TObject);
    procedure asUnitHashNameTestExecute(Sender: TObject);
    procedure acExitExecute(Sender: TObject);
    procedure acUnitHashNameSpeedTestExecute(Sender: TObject);
    procedure acUnitHashNameIDSpeedTestExecute(Sender: TObject);
    procedure acUnitHashNameIDTestExecute(Sender: TObject);
    procedure acSortingTestExecute(Sender: TObject);
    procedure acDerivedExampleExecute(Sender: TObject);
  private
  public
  end;

var
  frmMain: TfrmMain;

implementation

{$R *.dfm}

// From UnitTiming
function CPUTimestamp: Int64;
asm
  db $0f, $31; // rdtsc; as hex so it will work with Ddlphi 6
end;

procedure TfrmMain.acUnitHashIDTestExecute(Sender: TObject);
var
  ListID: TIDList;
  ItemID: TIDItem;
  i,j,k: Integer;
  S: TStringList;
  X: string;
begin
  mmResult.Lines.Add('UnitHash.TIDList testing...');
  S := TStringList.Create;
  ListID := TIDList.Create;
  // Perform 100 test rounds
  for i := 0 to 99 do begin
    // Restart test
    ListID.Clear;
    S.Clear;
    // Build up a list with approx. 100 random items
    // Also build a mapping stringlist of these numbers to 0
    for j := 0 to 99 do begin
      k := random(10000);
      X := IntToStr(k);
      if S.Values[X] <> '' then Continue;
      S.Values[X] := '0';
      ItemID := TIDItem.Create;
      ItemID.ID := k;
      ListID.Add(ItemID);
    end;
    // Now walk all items in the list and change the =0 setting into =1.
    ItemID := TIDItem(ListID.First);
    while Assigned(ItemID) do begin
      X := IntToStr(ItemID.ID);
      Assert(S.Values[X] = '0');
      S.Values[X] := '1';
      ItemID := TIDItem(ItemID.Next);
    end;
    // And assert that all =0 are canged to =1
    for j := 0 to S.Count - 1 do begin
      Assert(S.Values[S.Names[j]] = '1');
    end;
    // Now walk all items in the list in reverse order and change the =1 setting into =2.
    ItemID := TIDItem(ListID.Last);
    while Assigned(ItemID) do begin
      X := IntToStr(ItemID.ID);
      Assert(S.Values[X] = '1');
      S.Values[X] := '2';
      ItemID := TIDItem(ItemID.Previous);
    end;
    // And assert that all =0 are canged to =1
    for j := 0 to S.Count - 1 do begin
      Assert(S.Values[S.Names[j]] = '2');
    end;
    // We now know that the list's double linked list is OK!

    // Walk the stringlist and check that we find the items we're know are there
    for j := 0 to S.Count - 1 do begin
      X := S.Names[j];
      ItemID := ListID.ByID[StrToInt(X)];
      Assert(Assigned(ItemID) and (ItemID.ID = StrToInt(X)));
    end;
    // Now do it again, and keep deleting the items...
    for j := 0 to S.Count - 1 do begin
      X := S.Names[j];
      ItemID := ListID.ByID[StrToInt(X)];
      Assert(Assigned(ItemID) and (ItemID.ID = StrToInt(X)));
      ListID.Remove(ItemID);
      ItemID.Free;
    end;
    // We're done testing
    Assert(ListID.Count = 0);
  end;
  ListID.Free;
  S.Free;
  mmResult.Lines.Add('Test finished succesfully');
end;

procedure TfrmMain.acUnitHashIDSpeedTestExecute(Sender: TObject);
var
  Item: TIDItem;
  List: TIDList;
  X: Int64;
  i,j: Integer;
begin
  SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_TIME_CRITICAL);
  try
    mmResult.Lines.Add('UnitHash.TIDList speed testing...');
    List := TIDList.Create;
    try
      List.Options := [ loReplaceOnCollision ];
      List.MinimumHashTableSize := 1 shl 21;
      X := CPUTimestamp;
      for i := 0 to 999999 do begin
        Item := TIDItem.Create;
        Item.ID := i; //Random(1000000000);
        List.Add(Item);
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/Add',[X * 1e-6]));

      X := CPUTimestamp;
      for j := 0 to 9 do begin
        for i := 0 to 999999 do begin
          List.ByID[i];
        end;
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/ByID',[X * 1e-7]));

      X := CPUTimestamp;
      for i := 0 to 999999 do begin
        Item := List.ByID[i];
        List.Remove(Item);
        Item.Free;
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/ByID+Remove+Free',[X * 1e-6]));
    finally
      List.Free;
    end;
  finally
    SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_NORMAL);
  end;
end;

procedure TfrmMain.asUnitHashNameTestExecute(Sender: TObject);
var
  ListName: TNameList;
  ItemName: TNameItem;
  i,j,k: Integer;
  S: TStringList;
  X: string;
begin
  mmResult.Lines.Add('UnitHash.TNameList testing...');
  S := TStringList.Create;
  ListName := TNameList.Create;
  // Perform 100 test rounds
  for i := 0 to 99 do begin
    // Restart test
    ListName.Clear;
    S.Clear;
    // Build up a list with approx. 100 random items
    // Also build a mapping stringlist of these numbers to 0
    for j := 0 to 99 do begin
      k := random(10000);
      X := IntToStr(k);
      if S.Values[X] <> '' then Continue;
      S.Values[X] := '0';
      ItemName := TNameItem.Create;
      ItemName.Name := X;
      ListName.Add(ItemName);
    end;
    // Now walk all items in the list and change the =0 setting into =1.
    ItemName := TNameItem(ListName.First);
    while Assigned(ItemName) do begin
      X := ItemName.Name;
      Assert(S.Values[X] = '0');
      S.Values[X] := '1';
      ItemName := TNameItem(ItemName.Next);
    end;
    // And assert that all =0 are canged to =1
    for j := 0 to S.Count - 1 do begin
      Assert(S.Values[S.Names[j]] = '1');
    end;
    // Now walk all items in the list in reverse order and change the =1 setting into =2.
    ItemName := TNameItem(ListName.Last);
    while Assigned(ItemName) do begin
      X := ItemName.Name;
      Assert(S.Values[X] = '1');
      S.Values[X] := '2';
      ItemName := TNameItem(ItemName.Previous);
    end;
    // And assert that all =0 are canged to =1
    for j := 0 to S.Count - 1 do begin
      Assert(S.Values[S.Names[j]] = '2');
    end;
    // We now know that the list's double linked list is OK!

    // Walk the stringlist and check that we find the items we're know are there
    for j := 0 to S.Count - 1 do begin
      X := S.Names[j];
      ItemName := ListName.ByName[X];
      Assert(Assigned(ItemName) and (ItemName.Name = X));
    end;
    // Now do it again, and keep deleting the items...
    for j := 0 to S.Count - 1 do begin
      X := S.Names[j];
      ItemName := ListName.ByName[X];
      Assert(Assigned(ItemName) and (ItemName.Name = X));
      ListName.Remove(ItemName);
      ItemName.Free;
    end;
    // We're done testing
    Assert(ListName.Count = 0);
  end;
  ListName.Free;
  S.Free;
  mmResult.Lines.Add('Test finished succesfully');
end;

procedure TfrmMain.acExitExecute(Sender: TObject);
begin
  Close;
end;

procedure TfrmMain.acUnitHashNameSpeedTestExecute(Sender: TObject);
var
  Names: array of string;
  Item: TNameItem;
  List: TNameList;
  X: Int64;
  i,j: Integer;
begin
  SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_TIME_CRITICAL);
  try
    mmResult.Lines.Add('UnitHash.TNameList speed testing...');
    // prebuild a strings array
    SetLength(Names,1000000);
    for i := 0 to High(Names) do Names[i] := IntToStr(i);
    List := TNameList.Create;
    try
      List.Options := [ loReplaceOnCollision ];
      List.MinimumHashTableSize := 1 shl 21;

      X := CPUTimestamp;
      for i := 0 to 999999 do begin
        Item := TNameItem.Create;
        Item.Name := Names[i];
        List.Add(Item);
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/Add',[X * 1e-6]));

      X := CPUTimestamp;
      for j := 0 to 9 do begin
        for i := 0 to 999999 do begin
          List.ByName[Names[i]];
        end;
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/ByName',[X * 1e-7]));

      X := CPUTimestamp;
      for i := 0 to 999999 do begin
        Item := List.ByName[Names[i]];
        List.Remove(Item);
        Item.Free;
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/ByName+Remove+Free',[X * 1e-6]));
    finally
      List.Free;
    end;
  finally
    SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_NORMAL);
  end;
end;

procedure TfrmMain.acUnitHashNameIDSpeedTestExecute(Sender: TObject);
var
  Names: array of string;
  Item: TNameIDItem;
  List: TNameIDList;
  X: Int64;
  i,j: Integer;
begin
  SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_TIME_CRITICAL);
  try
    mmResult.Lines.Add('UnitHash.TNameIDList speed testing...');
    // prebuild a strings array
    SetLength(Names,1000000);
    for i := 0 to High(Names) do Names[i] := IntToStr(i);
    List := TNameIDList.Create;
    try
      List.Options := [ loReplaceOnCollision ];
      List.MinimumHashTableSize := 1 shl 21;

      X := CPUTimestamp;
      for i := 0 to 999999 do begin
        Item := TNameIDItem.Create;
        Item.Name := Names[i];
        Item.ID := i;
        List.Add(Item);
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/Add',[X * 1e-6]));

      X := CPUTimestamp;
      for j := 0 to 9 do begin
        for i := 0 to 999999 do begin
          List.ByName[Names[i]];
        end;
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/ByName',[X * 1e-7]));

      X := CPUTimestamp;
      for j := 0 to 9 do begin
        for i := 0 to 999999 do begin
          List.ByID[i];
        end;
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/ByID',[X * 1e-7]));

      X := CPUTimestamp;
      for i := 0 to 999999 do begin
        Item := List.ByID[i];
        List.Remove(Item);
        Item.Free;
      end;
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%F cycles/ByID+Remove+Free',[X * 1e-6]));
    finally
      List.Free;
    end;
  finally
    SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_NORMAL);
  end;
end;

procedure TfrmMain.acUnitHashNameIDTestExecute(Sender: TObject);
var
  ListNameID: TNameIDList;
  ItemNameID: TNameIDItem;
  i,j,k: Integer;
  S: TStringList;
  X: string;
begin
  mmResult.Lines.Add('UnitHash.TNameList testing...');
  S := TStringList.Create;
  ListNameID := TNameIDList.Create;
  // Perform 100 test rounds
  for i := 0 to 99 do begin
    // Restart test
    ListNameID.Clear;
    S.Clear;
    // Build up a list with approx. 100 random items
    // Also build a mapping stringlist of these numbers to 0
    for j := 0 to 99 do begin
      k := random(10000);
      X := IntToStr(k);
      if S.Values[X] <> '' then Continue;
      S.Values[X] := '0';
      ItemNameID := TNameIDItem.Create;
      ItemNameID.Name := X;
      ItemNameID.ID := k;
      ListNameID.Add(ItemNameID);
    end;
    // Now walk all items in the list and change the =0 setting into =1.
    ItemNameID := TNameIDItem(ListNameID.First);
    while Assigned(ItemNameID) do begin
      X := ItemNameID.Name;
      Assert(S.Values[X] = '0');
      S.Values[X] := '1';
      ItemNameID := TNameIDItem(ItemNameID.Next);
    end;
    // And assert that all =0 are canged to =1
    for j := 0 to S.Count - 1 do begin
      Assert(S.Values[S.Names[j]] = '1');
    end;
    // Now walk all items in the list in reverse order and change the =1 setting into =2.
    ItemNameID := TNameIDItem(ListNameID.Last);
    while Assigned(ItemNameID) do begin
      X := ItemNameID.Name;
      Assert(S.Values[X] = '1');
      S.Values[X] := '2';
      ItemNameID := TNameIDItem(ItemNameID.Previous);
    end;
    // And assert that all =0 are canged to =1
    for j := 0 to S.Count - 1 do begin
      Assert(S.Values[S.Names[j]] = '2');
    end;
    // We now know that the list's double linked list is OK!

    // Walk the stringlist and check that we find the items we're know are there
    for j := 0 to S.Count - 1 do begin
      X := S.Names[j];
      ItemNameID := ListNameID.ByName[X];
      Assert(Assigned(ItemNameID) and (ItemNameID.Name = X));
      ItemNameID := ListNameID.ByID[StrToInt(X)];
      Assert(Assigned(ItemNameID) and (ItemNameID.ID = StrToInt(X)));
    end;
    // Now do it again, and keep deleting the items...
    for j := 0 to S.Count - 1 do begin
      X := S.Names[j];
      ItemNameID := ListNameID.ByName[X];
      Assert(Assigned(ItemNameID) and (ItemNameID.Name = X));
      ListNameID.Remove(ItemNameID);
      ItemNameID.Free;
    end;
    // We're done testing
    Assert(ListNameID.Count = 0);
  end;
  ListNameID.Free;
  S.Free;
  mmResult.Lines.Add('Test finished succesfully');
end;

type
  TID = class
    ID: Integer;
  end;

  TMyID = class(TBaseItem)
    ID: Integer;
  end;

function TIDCompare(Item1, Item2: Pointer): Integer;
begin
  Result := TID(Item2).ID - TID(Item1).ID;
end;

function LessOrEqualIDItem(ItemA, ItemB: TBaseItem): Boolean;
begin
  Result := TMyID(ItemA).ID <= TMyID(ItemB).ID;
end;


procedure TfrmMain.acSortingTestExecute(Sender: TObject);
const
  Rep = 5;
  Cnt = 10000; // Use some representative count here for your own app.               
var
  Item, Last: TMyID;
  List: TBaseList;
  ID: TID;
  L: TList;
  X: Int64;
  i,j: Integer;

  procedure Fill;
  var
    i: Integer;
  begin
    List.Clear;
    for i := 1 to Cnt do begin
      Item := TMyID.Create;
      Item.ID := Random(1000000);
      List.Add(Item);
    end;
  end;


  procedure Test;
  begin
    // test...
    i := 0;
    Item := TMyID(List.First);
    Last := Item;
    while Assigned(Item) do begin
      Assert(LessOrEqualIDItem(Last,Item));
      Inc(i);
      Last := Item;
      Item := TMyID(Item.Next);
    end;
    Assert(i = List.Count);
    Item := TMyID(List.Last);
    Last := Item;
    while Assigned(Item) do begin
      Assert(LessOrEqualIDItem(Item,Last));
      Last := Item;
      Item := TMyID(Item.Previous);
    end;
  end;
begin
  Randomize;
  mmResult.Lines.Add('UnitHash.TIDList speed testing...');
  List := TIDList.Create;
  try
    SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_TIME_CRITICAL);
    List.Options := [ loReplaceOnCollision ];
    mmResult.Lines.Add('Natural order');
    for j := 1 to Rep do begin
      Fill;
      X := CPUTimestamp;
      List.Sort(LessOrEqualIDItem,True);
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%2d: %F cycles/item sorting (%f sec)',[j, X / List.Count, X / 1.2e9]));
      Test;
    end;
    mmResult.Lines.Add('No natural order');
    for j := 1 to Rep do begin
      Fill;
      X := CPUTimestamp;
      List.Sort(LessOrEqualIDItem,False);
      X := CPUTimestamp - X;
      mmResult.Lines.Add(Format('%2d: %F cycles/item sorting (%f sec)',[j, X / List.Count, X / 1.2e9]));
      Test;
    end;
    mmResult.Lines.Add('TList sorting QuickSort');
    L := TList.Create;
    try
      for j := 1 to Rep do begin
        L.Clear;
        for i := 1 to Cnt do begin
          ID := TID.Create;
          ID.ID := Random(1000000);
          L.Add(ID);
        end;
        X := CPUTimestamp;
        L.Sort(TIDCompare);
        X := CPUTimestamp - X;
        for i := 0 to L.Count - 1 do TObject(L[i]).Free;
        mmResult.Lines.Add(Format('%2d: %F cycles/item sorting (%f sec)',[j, X / L.Count, X / 1.2e9]));
      end;
    finally
      L.Free;
    end;
  finally
    List.Free;
    SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_NORMAL);
  end;
end;

// This class is used in the action execute below to demonstate how you can use the
// associative accessor ByName with your own classes.
type
  TVariable = class(TNameItem)
  public
    Value: Double;
  end;

procedure TfrmMain.acDerivedExampleExecute(Sender: TObject);
var
  Vars: TNameList;
  V: TVariable;
begin
  Vars := TNameList.Create;
  try
    // Add two test vars
    V := TVariable.Create;
    V.Name := 'A';
    V.Value := 10;
    Vars.Add(V);

    V := TVariable.Create;
    V.Name := 'B';
    V.Value := 20;
    Vars.Add(V);

    // Now test associativity...
    ShowMessage('10 + 20 = ' + FloatToStr(TVariable(Vars.ByName['A']).Value +
                                          TVariable(Vars.ByName['B']).Value));
  finally
    Vars.Free;
  end;
end;

end.
