unit AudioTest ;

// TAudioTester class
// (c) 2002 Frederic DA VITORIA

interface

uses Classes, SysUtils, MpegFrameHdr, AudioInfo, Globals ;

type
  TProcessMessages = procedure of object ;
  TProgressStepBy = procedure (Delta : integer) of object ;

  TAudioTester = class (TAudioInfo)
    private
      fFileSize : integer ; // to avoid repetitively reading the size property
      fLastOkHeaderFound : longint ; // position of the last correct frame
        // found. The last value of this field will be returned in the
        // corresponding TestMp3File parameter
      fMpegHeader : TMpegFrameHdr ;
      fBravaTagPos : integer ; // position of the Brava Software tag if found.
      fMaxSynchroShift : integer ; // maximum synchro error shift. The last
        // value of this field will be returned in the corresponding
        // TestMp3File parameter
      fRefSynchro : array [0..3] of byte ; // first header found
      fStream : TFileStream;
      fStreamPos : integer ;
      fSynchroErrors : integer ; // number of errors found. the last value of
        // this field will be returned in the corresponding TestMp3File
        // parameter
      fSynchroErrorList : string ; // list of positions of the successive
        // synchro errors. The last value of this field will be returned in the
        // corresponding TestMp3File parameter
      fTiming : double ;                        // current position in milliseconds
      function CheckBravaTag : boolean ;
      procedure ReSynchronize ;
    public
      procedure TestMp3File (
        const FileName : string ;
        var LastFrameEndPos : longint ; // end of the last correct frame found.
        var LastFrameEndTime : longint ;
        var LastFrameIsOK : boolean ;
        var BravaTag : boolean ;
        var SynchroErrors, // number of errors found.
        MaxSynchroShift : integer ; // maximum synchro error shift.
        var SynchroErrorList : string ; // list of positions of the successive
          // synchro errors.
        ProcessMessages : TProcessMessages) ; // testing a file may be long,
          // but this parameter may be left to NIL
  end;

implementation

const
  // see a description of the Mp3 frame header if you wish to understand the
  // following masks.  
  // no MASK_0 or MASK_1, since these bytes should remain constant
  MASK_2 = $0E ; // bitrate and private bit may vary, sampling rate should
    // remain constant, and I can' see why padding could vary
  MASK_3 = $0C ; // channel mode, mode extension and maybe emphasis may vary,
    // but not copyright or original

function TAudioTester.CheckBravaTag : boolean ;
// test for a Brava Software trailer I have observed empirically
  var
    buff  : array [0..1023] of char ;
    i     : integer ;
  begin                                                 { CheckBravaTag }
    fStream.Position := fLastOkHeaderFound + MpegFrameHdrSz ;
    result := FALSE ;
    if fStream.Position + SizeOf (buff) > fFileSize
      then Exit ;
    fStream.ReadBuffer (buff, SizeOf (buff)) ;
    result := FALSE ;
    for i := 0 to High (buff) do                // check for signature
      if     (buff [i+0] = '1')
         and (buff [i+1] = '8')
         and (buff [i+2] = '2')
         and (buff [i+3] = '7')
         and (buff [i+4] = '3')
         and (buff [i+5] = '6')
         and (buff [i+6] = '4')
         and (buff [i+7] = '5')
        then begin
          result := TRUE ;
          fBravaTagPos := fLastOkHeaderFound + MpegFrameHdrSz + i ;
          Exit
        end {then}
  end ;                                                 { CheckBravaTag }

procedure TAudioTester.ReSynchronize ;
// seek a mp3 frame header when it is not where it was expected
  const SYNCHRO_BUFF_SIZE = 2048 ;
  var
    high_buff     : integer ;
    shift         : integer ; // from position of last header
    synchro_buff  : array [-1..SYNCHRO_BUFF_SIZE-1] of char ; // test buffer
  procedure LoadBuffer ;
    begin                                               { LoadBuffer }
      if fStream.Position + SYNCHRO_BUFF_SIZE > fFileSize
        then high_buff := fFileSize - fStream.Position
        else high_buff := SYNCHRO_BUFF_SIZE ;
      Inc (shift, SYNCHRO_BUFF_SIZE) ;
      fStream.ReadBuffer (synchro_buff [0], high_buff)
    end ;                                               { LoadBuffer }
  var i : integer ; // position in test buffer
  procedure SetSynchro ;
  // record the new position found
    var new_pos : integer ;
    begin                                               { SetSynchro }
      Inc (fSynchroErrors) ;
      new_pos := fLastOkHeaderFound + 1 + shift - SYNCHRO_BUFF_SIZE + i ;
      Insert (  '#'
              + TimeToStr (fTiming/(1000*60*60*24)) + '/'
                                 (* ms *s *m *h *)
              + IntToHex (fLastOkHeaderFound, 6) + '/'
              + IntToHex (fStreamPos, 6) + '/'
              + IntToHex (new_pos, 6),
              fSynchroErrorList, 32000) ;       { If the list ever gets longer,
                                                  you have a BIG problem ! }
      if Abs (new_pos - fStreamPos) > fMaxSynchroShift
        then fMaxSynchroShift := Abs (new_pos - fStreamPos) ;
      fStreamPos := new_pos
    end ;                                               { SetSynchro }
  begin                                                 { ReSynchronize }
    fStream.Position := fLastOkHeaderFound + 1 ;
    synchro_buff [-1] := char (fRefSynchro [0]) ;
    shift := 0 ;
    repeat
      LoadBuffer ;
      i := 0 ;
      repeat
        if synchro_buff [i] = char (fRefSynchro [0])
          then begin
            if i >= high_buff - 2
              then if fStream.Position + SYNCHRO_BUFF_SIZE <= fFileSize
                then begin                      { load next buffer }
                  LoadBuffer ;
                  i := -1                       { point on "cheat" position }
                end {then}
                else Exit ;                     { end of file, finished ! }
            if     (synchro_buff [i+1] = char (fRefSynchro [1]))
               and (byte (synchro_buff [i+2]) and MASK_2 = fRefSynchro [2])
               and (byte (synchro_buff [i+3]) and MASK_3 = fRefSynchro [3])
              then begin                        { found ! }
                fMpegHeader.Load (synchro_buff [i],  MpegFrameHdrSz) ;
                SetSynchro ;
                Exit
              end {then}
          end {then} ;
        Inc (i)
      until i >= high_buff
    until fStream.Position >= fFileSize
  end ;                                                 { ReSynchronize }

procedure TAudioTester.TestMp3File (const FileName : string ;
                                    var LastFrameEndPos : longint ;
                                    var LastFrameEndTime : longint ;
				    var LastFrameIsOK : boolean ;
                                    var BravaTag : boolean ;
				    var SynchroErrors, MaxSynchroShift : integer ;
                                    var SynchroErrorList : string ;
                                    ProcessMessages : TProcessMessages) ;
  var mpeg_header_buffer : TMPegHeaderBuffer ;
  begin                                                 { TestMp3File }
    LastFrameEndPos := 0 ;                      // initializations
    LastFrameEndTime := 0 ;
    LastFrameIsOK := FALSE ;
    BravaTag := FALSE ;
    fBravaTagPos := 0 ;
    fSynchroErrors := 0 ;
    fSynchroErrorList := '' ;
    fMaxSynchroShift := 0 ;
    fLastOkHeaderFound := 0 ;
    fTiming := 0 ;
    LoadFromFile (FileName) ;                   // Get mp3 characteristics
    fStream := TFileStream.Create (FileName, fmOpenRead or fmShareDenyNone) ;
    try
      if     (FirstMpegFramePos >= 0)
	 and (FirstMpegFramePos + MpegFrameHdrSz < fStream.Size)
	then begin                              // file is mp3 and large enough
	  fStreamPos := FirstMpegFramePos ;
          fStream.Position := fStreamPos ;
	  fStream.ReadBuffer (mpeg_header_buffer, MpegFrameHdrSz) ;
	  fMpegHeader := TMpegFrameHdr.Create ;
          fMpegHeader.Load (mpeg_header_buffer, MpegFrameHdrSz) ;
          fRefSynchro [0] := mpeg_header_buffer [0] ; // get reference header
          fRefSynchro [1] := mpeg_header_buffer [1] ;
          fRefSynchro [2] := mpeg_header_buffer [2] and MASK_2 ;
          fRefSynchro [3] := mpeg_header_buffer [3] and MASK_3 ;
	  fFileSize := fStream.Size ;
          if ID3v1.Ok
            then Dec (fFileSize, 128) ;         { then, last 128 bytes are not
                                                  part of the sound stream }
          if XingVBR.Ok
            then begin
              if FirstMpegFramePos + MpegBytesNum < fFileSize
                then fFileSize := FirstMpegFramePos + MpegBytesNum
            end {then} ;
	  try
	    repeat                              // mp3 tag loop
              if QuitRequested                  // quit application
                then Abort ;
              if Assigned (ProcessMessages)     // stop treatment :
                then ProcessMessages ;          //
	      if fStreamPos + MpegFrameHdrSz >= fFileSize
		then begin
                  fLastOkHeaderFound := fStreamPos ; // not enough data left to
                    // check for a new frame header, I assume data ends here
                  Break
                end {then} ;
	      fStream.Position := fStreamPos ;
	      fStream.ReadBuffer (mpeg_header_buffer, MpegFrameHdrSz) ;
	      fMpegHeader.Load (mpeg_header_buffer, MpegFrameHdrSz) ;
              if not fMpegHeader.ok
                then if (fFileSize - fStreamPos <= $2188) and CheckBravaTag // 2188 is empirical
                  then begin { stream is interrupted by a brava software tag }
                    BravaTag := TRUE ;
                    Break
                  end {then}
                  else Resynchronize ;    { try to recover from synchro error }
              if fMpegHeader.ok
                then begin
                  fLastOkHeaderFound := fStreamPos ;
                  if BitRate <> 0
                    then fTiming := fTiming + fMpegHeader.FrameLen * 8000 / BitRate ;
		  Inc (fStreamPos, fMpegHeader.FrameLen)
                end {then}
	    until not fMpegHeader.ok or BravaTag or (fStreamPos >= fFileSize) ;
            if fMpegHeader.ok
              then begin
                if BravaTag
                  then LastFrameEndPos := fBravaTagPos
                  else if fLastOkHeaderFound + fMpegHeader.FrameLen > fFileSize
                    then LastFrameEndPos := fLastOkHeaderFound
                    else LastFrameEndPos := fFileSize ;
                LastFrameIsOK := (fLastOkHeaderFound + fMpegHeader.FrameLen = LastFrameEndPos)
              end
              else begin
                LastFrameEndPos := fLastOkHeaderFound ;
	        LastFrameIsOK := FALSE
              end {else}
	  finally
            LastFrameEndTime := Round (fTiming) ;
            if fSynchroErrorList > ''
              then Delete (fSynchroErrorList, 1, 1) ; { deletes initial '#' } ;
            SynchroErrorList := fSynchroErrorList ;
            MaxSynchroShift := fMaxSynchroShift ;
            fMpegHeader.Free
          end {finally}
	end {then}
    finally
      fStream.Free ;
      SynchroErrors := fSynchroErrors
    end
  end ;                                                 { TestMp3File }

end.

