{-----------------------------------------------------------------------------
 -
 Unit Name : rgnpkg
 Purpose   : helper for TmpRegionImage, store normal, hilite and selected
             bitmaps and regions list in one compressed file
 Author    : markus stephany
 Copyright : (C) 1997-2002 markus stephany
 -

 This source code is freeware. You may use, change, and distribute without
 charge the source code as you like. This unit can be used freely in any
 commercial applications. However, it may not be sold as a standalone product
 and the source code may not be included in a commercial product. This unit
 is provided as is with no warrent or support. Make sure to read relevant
 information and documentation from Microsoft before using this unit.

 -
 Version   : 0.1
 Date      : 26.01.2002 16:08:33
 -
 -----------------------------------------------------------------------------}

unit rgnpkg;

interface

uses
  Windows, Graphics, Classes, SysUtils, rgnimg;

{-----------------------------------------------------------------------------
  Arguments: sFile: TFileName; fNBmp, fHBmp, fSBmp: TBitmap; fList: TStringList
  Result:    None
  Purpose:   load three bitmaps and a stringlist from a compressed file
-----------------------------------------------------------------------------}
procedure ExtractPackage(sFile: TFileName; fNBmp, fHBmp, fSBmp: TBitmap;
  fList: TStringList);

{-----------------------------------------------------------------------------
  Arguments: sFile: TFileName; fNBmp, fHBmp, fSBmp: TBitmap; fList: TStringList
  Result:    None
  Purpose:   store three bitmaps and a stringlist in a compressed file
-----------------------------------------------------------------------------}
procedure CollectPackage(sFile: TFileName; fNBmp, fHBmp, fSBmp: TBitmap;
  fList: TStringList);

{-----------------------------------------------------------------------------
  Arguments: fObj: TmpRegionImage; sFile: TFileName
  Result:    None
  Purpose:   load a TRegionImage's bitmaps and region list from a compressed
             file
-----------------------------------------------------------------------------}
procedure LoadRegionImage(fObj: TmpRegionImage; sFile: TFileName);

{-----------------------------------------------------------------------------
  Arguments: fObj: TmpRegionImage; sFile: TFileName
  Result:    None
  Purpose:   store a TRegionImage's bitmaps and region list in a compressed
             file
-----------------------------------------------------------------------------}
procedure SaveRegionImage(fObj: TmpRegionImage; sFile: TFileName);

  (* for TOpenPictureDialog *)
type
  TPKGBitmap = class(TBitmap)
  public
    procedure LoadFromFile(const Filename: string); override;
  end;

implementation

uses
  zLib;

const
  _PkgHeader = 'TMapImage';  // historical
  _PkgVersion = 2;

procedure ExtractPackage(sFile: TFileName; fNBmp, fHBmp, fSBmp: TBitmap;
  fList: TStringList);
var
  fs: TFileStream;
  cs: TDecompressionStream;
  ms: TMemoryStream;
  cVer: cardinal;
  s: string;

  procedure LoadStr(fObj: TObject);
  begin
    cs.Read(cVer, sizeof(cVer));
    ms.Size := 0;
    if cVer > 0 then
      ms.CopyFrom(cs, cVer);
    ms.Position := 0;
    if fObj is TBitmap then
      TBitmap(fObj).LoadFromStream(ms)
    else if fObj is TStringList then
      TStringList(fObj).LoadFromStream(ms);
  end;
begin
  fs := TFileStream.Create(sFile, fmOpenRead);
  cs := TDecompressionStream.Create(fs);
  ms := TMemoryStream.Create;
  try
    SetLength(s, Length(_PkgHeader));
    cs.Read(s[1], Length(_PkgHeader));
    if s <> _PkgHeader then
      raise Exception.Create('Invalid package file');
    cs.Read(cVer, sizeof(cVer));
    if cVer <> _PkgVersion then
      raise Exception.Create('Unknown package file version');

    LoadStr(fNBmp);
    LoadStr(fHBmp);
    LoadStr(fSBmp);
    LoadStr(fList);
  finally
    cs.Free;
    fs.Free;
    ms.Free;
  end;
end;

procedure CollectPackage(sFile: TFileName; fNBmp, fHBmp, fSBmp: TBitmap;
  fList: TStringList);
var
  fs: TFileStream;
  cs: TCompressionStream;
  ms: TMemoryStream;
  cVer: cardinal;

  procedure SaveStr(fObj: TObject);
  begin
    ms.Size := 0;
    if fObj is TBitmap then
      TBitmap(fObj).SaveToStream(ms)
    else if fObj is TStringList then
      TStringList(fObj).SaveToStream(ms);
    cVer := ms.Size;
    cs.Write(cVer, sizeof(cVer));
    ms.SaveToStream(cs);
  end;
begin
  fs := TFileStream.Create(ChangeFileExt(sFile, '.pkg'), fmCreate);
  cs := TCompressionStream.Create(clMax, fs);
  ms := TMemoryStream.Create;
  try
    cs.Write(_PkgHeader[1], Length(_PkgHeader));
    cVer := _PkgVersion;
    cs.Write(cVer, sizeof(cVer));

    SaveStr(fNBmp);
    SaveStr(fHBmp);
    SaveStr(fSBmp);
    SaveStr(fList);
  finally
    cs.Free;
    fs.Free;
    ms.Free;
  end;
end;

procedure LoadRegionImage(fObj: TmpRegionImage; sFile: TFileName);
begin
  ExtractPackage(sFile, fObj.NormalBitmap, fObj.HiliteBitmap,
    fObj.SelectedBitmap, fObj.Regions);
end;

procedure SaveRegionImage(fObj: TmpRegionImage; sFile: TFileName);
begin
  CollectPackage(sFile, fObj.NormalBitmap, fObj.HiliteBitmap,
    fObj.SelectedBitmap, fObj.Regions);
end;

{ TPKGBitmap }

procedure TPKGBitmap.LoadFromFile(const Filename: string);
begin
  ExtractPackage(FileName, self, nil, nil, nil);
end;

initialization
  TPicture.RegisterFileFormat('PKG', 'TRegionImage Package', TPKGBitmap);
end.
