{-----------------------------------------------------------------------------
 -
 Unit Name : rgnimg
 Purpose   : TMPRegionImage, a Bitmap display component with clickeable
             regions
 Author    : markus stephany
 Copyright : (C) 1997-2002 markus stephany
 -

 This source code is freeware. You may use, change, and distribute without
 charge the source code as you like. This unit can be used freely in any
 commercial applications. However, it may not be sold as a standalone product
 and the source code may not be included in a commercial product. This unit
 is provided as is with no warrent or support. Make sure to read relevant
 information and documentation from Microsoft before using this unit.

 -
 Version   : 3.0
 Date      : 09/21/2002
 History   : see rgnimg.html/history
 -
 -----------------------------------------------------------------------------}

unit rgnimg;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls;


type
  (* forward *)
  TRegionList = class;

  // dynamic array of integers
  TIntArray = array of integer;

  // canvas with additional region handling methods
  TMPRegionImageCanvas = class(TCanvas)
  public
    procedure PaintRegion(hR: HRGN);
    procedure FillRegion(hR: HRGN; Brush: TBrush);
    procedure InvertRegion(hR: HRGN);
  end;

  // event handler (OnEnterRegion, OnLeaveRegion)
  TRegionEvent = procedure(Sender: TObject; const AIndex: integer) of object;

  // custom TMPRegionImage base class, derived from TGraphicControl
  TCustomMPRegionImage = class(TGraphicControl)
  private
    FBitmap,
    FHiBitmap,
    FSelBitmap,
    FWorkBitmap: TBitmap;
    FRegions: TRegionList;
    FRegionsList: TStringList;
    FRegionsCursor,
    FCursor: TCursor;
    FCurrentRegion: integer;
    FMoving: boolean;
    FTransparent: boolean;
    FRegionsChanging: boolean;
    FDrawing: boolean;
    FHiliteCurrent: boolean;
    FLastHilited: integer;
    FOnEnterRegion: TRegionEvent;
    FOnLeaveRegion: TRegionEvent;
    FMouseDown: boolean;
    FMouseInPos: boolean;
    FAutoSelect: boolean;
    FSettingCursor: boolean;
    FOnSelectionChange: TNotifyEvent;
    FDoChangeList: boolean;
    FMultiSelect: Boolean;
    function GetCanvas: TCanvas;
    procedure SetBitmap(Index: integer; Value: TBitmap);
    function GetBitmap(Index: integer): TBitmap;
    procedure SetTransparent(Value: boolean);
    procedure SetCursor(Index: integer; Value: TCursor);
    procedure BitmapChanged(Sender: TObject);
    procedure RegionListChanged(Sender: TObject);
    procedure SetHiliteCurrent(const Value: boolean);
    procedure ClearLastRgn;
    procedure CheckHilite(x, y: integer; bSelChanged: boolean); overload;
    procedure CheckHilite(bSelChanged: boolean); overload;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure StyleChanged(Sender: TObject);
    procedure SetRegionList(const Value: TStringList);
    function GetCombinedRegion(Index: integer): HRGN;
    function GetSingleRegion(Index: integer): HRGN;
    procedure SetAutoSelect(const Value: boolean);
    procedure SetItemIndex(const Value: integer);
    function CopyRegion(hr: HRGN): HRGN;
    function GetStrTag(Index: integer): string;
    procedure SetStrTag(Index: integer; Value: string);
    procedure SplitRegionlistLine(iLine: integer; s: string; var cCmd: char;
      var sDesc: string; var rCoords: TIntArray);
    procedure CreateHiAndSelBitmap;
    procedure ShowCursor(ACursor: TCursor);
    function GetSelected(Index: Integer): Boolean;
    procedure SetSelected(Index: Integer; const Value: Boolean);
    function GetItemIndex: integer;
    function GetSelCount: integer;
    procedure SetMultiSelect(const Value: Boolean);
    function GetVersion: string;
    procedure SetVersion(const Value: string);
  protected
    procedure Paint; override;
    function GetSelectedCombiRegion: HRGN;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: integer);
      override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: integer); override;
    procedure ImportRegion(Value: TStrings); virtual;
    procedure Resize; override;
    procedure Loaded; override;

    property Align;
    property Anchors;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property ParentShowHint;
    // this is the standard bitmap to display
    property NormalBitmap: TBitmap index 0 read GetBitmap write SetBitmap;
    // a highlighted region is displayed by drawing this bitmap
    property HiliteBitmap: TBitmap index 1 read GetBitmap write SetBitmap;
    // currently selected region is displayed by drawing this bitmap
    property SelectedBitmap: TBitmap index 2 read GetBitmap write SetBitmap;
    property PopupMenu;
    property ShowHint;
    property Transparent: boolean read FTransparent write SetTransparent default False;
    property Visible;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
    // this cursor is shown when the mouse is over a region
    property RegionCursor: TCursor index 0 read FRegionsCursor write SetCursor
      default crHandPoint;
    property Cursor: TCursor index 1 read FCursor write SetCursor default crDefault;
    // textual representation of the regions list
    property Regions: TStringList read FRegionsList write SetRegionList;
    // if set to true, the selection changes on clicking a region
    property AutoSelect: boolean read FAutoSelect write SetAutoSelect default False;
    // if set to true, the region at the current cursor position is highlighted
    property HiliteCurrent: boolean read FHiliteCurrent write SetHiliteCurrent;
    // called when cursor moves over a region
    property OnEnterRegion: TRegionEvent read FOnEnterRegion write FOnEnterRegion;
    // called when cursor leaves a region
    property OnLeaveRegion: TRegionEvent read FOnLeaveRegion write FOnLeaveRegion;
    // called when the selected region changes
    property OnSelectionChange: TNotifyEvent
      read FOnSelectionChange write FOnSelectionChange;
    // multi-selection
    property MultiSelect: Boolean read FMultiSelect write SetMultiSelect default False;
    property Version: string read GetVersion write SetVersion;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    property Canvas: TCanvas read GetCanvas;
    // query region index at the given cursor position (-1: none)
    function RegionAtPos(X, Y: integer): integer;
    // move a region's coordinates
    procedure OffsetRegion(Index, dX, dY: integer);
    // query region index at current cursor position (-1: none)
    function RegionAtCursorPos: integer;
    // paint a region to the canvas
    procedure PaintHRegion(h: HRGN);
    // query windows HRGN object (don't DeleteObject !)
    property SingleRegion[Index: integer]: HRGN read GetSingleRegion;
    // query a combined HRGN of the Index'ed region an all overlapping regions
    property CombinedRegion[Index: integer]: HRGN read GetCombinedRegion;
    // query/set current selection
    property ItemIndex: integer read GetItemIndex write SetItemIndex;
    // textual "Tag" property of each region
    property StrTag[Index: integer]: string read GetStrTag write SetStrTag;
    // query region currently hilited
    property Hilited: integer read FLastHilited;
    // find a StrTag by name
    function IndexOfStrTag(sName: string): integer;
    // query number of regions
    function RegionsCount: integer;
    // get the actually displayed bitmap
    property Bitmap: TBitmap read FWorkBitmap;
    // multi-selection
    property Selected[Index: Integer]: Boolean read GetSelected write SetSelected;
    property SelCount: integer read GetSelCount;
  end;

  (* published component *)
  TMPRegionImage = class(TCustomMPRegionImage)
  published
    property Align;
    property Anchors;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property ParentShowHint;
    property NormalBitmap;
    property HiliteBitmap;
    property SelectedBitmap;
    property PopupMenu;
    property ShowHint;
    property Transparent;
    property Visible;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
    property RegionCursor;
    property Cursor;
    property Regions;
    property HiliteCurrent;
    property OnEnterRegion;
    property OnLeaveRegion;
    property AutoSelect;
    property OnSelectionChange;
    property MultiSelect;
    property Version;
  end;

  // exception class
  EmpRegionImage = class(Exception);

  // one region entry in a TRegionList
  PRegionEntry = ^TRegionEntry;
  TRegionEntry = record
    StrTag: string;
    Region: HRGN;
    // associated with assoc's entry in regions string list
    Assoc: integer;
    Selected: Boolean;
  end;

  // regions container and manager
  TRegionList = class(TList)
  private
    function _Get(Index: integer): PRegionEntry;
    procedure _Put(Index: integer; const Value: PRegionEntry);
  protected
    procedure Notify(Ptr: Pointer; Action: TListNotification); override;
  public
    procedure DeSelectAll;
    property Items[Index: integer]: PRegionEntry read _Get write _Put; default;
  end;

  // region to 'X' type region string
function XRegion(h: HRGN): string;

procedure Register;

implementation
const
  _ABOUT = 'TMPRegionImage 3.0, (C) 1997-2002 Markus Stephany';


procedure Register;
begin
  RegisterComponents('mirkes.de', [TMPRegionImage]);
end;

{ TMPRegionImageCanvas }

{-----------------------------------------------------------------------------
  Arguments: hR: HRGN; Brush: TBrush
  Result:    None
  Purpose:   fill a region
-----------------------------------------------------------------------------}
procedure TMPRegionImageCanvas.FillRegion(hR: HRGN; Brush: TBrush);
begin
  Changing;
  FillRgn(Handle, hR, Brush.Handle);
  Changed;
end;

{-----------------------------------------------------------------------------
  Arguments: hR: HRGN
  Result:    None
  Purpose:   invert a region
-----------------------------------------------------------------------------}
procedure TMPRegionImageCanvas.InvertRegion(hR: HRGN);
begin
  Changing;
  InvertRgn(Handle, hR);
  Changed;
end;

{-----------------------------------------------------------------------------
  Arguments: hR: HRGN
  Result:    None
  Purpose:   paint a region
-----------------------------------------------------------------------------}
procedure TMPRegionImageCanvas.PaintRegion(hR: HRGN);
begin
  Changing;
  PaintRgn(Handle, hR);
  Changed;
end;

{ TCustomMPRegionImage }

{-----------------------------------------------------------------------------
  Arguments: AOwner: TComponent
  Result:    None
  Purpose:   constructor
-----------------------------------------------------------------------------}
constructor TCustomMPRegionImage.Create(AOwner: TComponent);
begin
  inherited;
  ControlStyle := ControlStyle + [csReplicatable, csOpaque];
  FDoChangeList := True;
  FMultiSelect := False;
  FSettingCursor := False;
  FMouseDown := False;
  FMouseInPos := False;
  FTransparent := False;
  FRegions := TRegionList.Create;
  FAutoSelect := False;
  FRegionsCursor := crHandPoint;
  FCursor := crDefault;
  AutoSize := True;
  FMoving := False;
  FBitmap := TBitmap.Create;
  Height := 32;
  Width := 32;
  FBitmap.Height := 32;
  FBitmap.Width := 32;
  FHiBitmap := TBitmap.Create;
  FHiBitmap.Width := 0;
  FHiBitmap.Height := 0;

  FSelBitmap := TBitmap.Create;
  FSelBitmap.Width := 0;
  FSelBitmap.Height := 0;

  FDrawing := False;
  FWorkBitmap := TBitmap.Create;
  FWorkBitmap.Transparent := False;
  FRegionsChanging := False;
  FBitmap.OnChange := BitmapChanged;
  FHiBitmap.OnChange := StyleChanged;
  FSelBitmap.OnChange := StyleChanged;
  FRegionsList := TStringList.Create;
  FRegionsList.OnChange := RegionListChanged;
  FHiliteCurrent := False;
  FLastHilited := -1;
end;

{-----------------------------------------------------------------------------
  Arguments: None
  Result:    None
  Purpose:   destructor
-----------------------------------------------------------------------------}
destructor TCustomMPRegionImage.Destroy;
begin
  ClearLastRgn;
  FRegions.Free;
  FBitmap.Free;
  FHiBitmap.Free;
  FSelBitmap.Free;
  FWorkBitmap.Free;
  FRegionsList.Free;
  inherited;
end;

{-----------------------------------------------------------------------------
  Arguments: var rInt: TIntArray; const i: integer
  Result:    None
  Purpose:   add an integer to an array of integers
-----------------------------------------------------------------------------}
procedure AddInt(var rInt: TIntArray; const i: integer);
begin
  SetLength(rInt, Succ(Length(rInt)));
  rInt[Pred(Length(rInt))] := i;
end;

{-----------------------------------------------------------------------------
  Arguments: s: string; iLine: integer
  Result:    TIntArray
  Purpose:   split integers in string format ("x,y,z...") into an
             integer array
-----------------------------------------------------------------------------}
function SplitCoords(s: string; iLine: integer): TIntArray;
var
  i, j: integer;
  s1: string;
begin
  Result := nil;
  repeat
    i := Pos(',', s);
    if i = 0 then
      i := MaxInt;
    s1 := Copy(s, 1, i - 1);
    Delete(s, 1,i);

    if s1 = '' then
      Break;

    if not TryStrToInt(s1, j) then
      raise EmpRegionImage.CreateFmt('Invalid coordinate (line %d)', [iLine + 1]);

    AddInt(Result, j);
  until s = '';
end;

{-----------------------------------------------------------------------------
  Arguments: Value: TStrings
  Result:    None
  Purpose:   create regions list from strings
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.ImportRegion(Value: TStrings);
var
  i, j: integer;
  pEntry: PRegionEntry;
  rCoords: TIntArray;
  h: HRGN;
  sDesc: string;
  cType: char;
begin
  FRegions.Clear;

  if csDesigning in ComponentState then
    Exit;

  if Value.Count > 0 then
    for i := 0 to Pred(Value.Count) do
    begin
      if Trim(Value[i]) = '' then
        Continue;
      SplitRegionlistLine(i, Value[i], cType, sDesc, rCoords);
      case UpCase(cType) of
        'X': // extended region
          begin
            if (Length(rCoords) mod 4) <> 0 then
              raise EmpRegionImage.CreateFmt('Invalid X coordinates count (%d, line %d)',
                [Length(rCoords), i + 1]);

            // header + region data (hope, they are four byte aligned)
            Assert(rCoords[0] = sizeof(TRGNDATAHEADER));
            h := ExtCreateRegion(nil, sizeof(cardinal) * Length(rCoords),
              PRGNDATA(@rCoords[0])^);
          end;
        '+': // combination of two or more previous lines
          begin
            if (not (Length(rCoords) in [1,2])) or (rCoords[0] < 2) then
              raise EmpRegionImage.CreateFmt('Invalid combination flag count (line %d)',
                [i + 1]);

            if Length(rCoords) = 1 then
              AddInt(rCoords, RGN_OR);

            h := FRegions[FRegions.Count - rCoords[0]].Region;
            FRegions[FRegions.Count - rCoords[0]].Region := 0;

            for j := 1 to Pred(rCoords[0]) do
            begin
              CombineRgn(h, h, FRegions[FRegions.Count - j].Region, rCoords[1]);
            end;

            for j := 1 to rCoords[0] do
              FRegions.Delete(Pred(FRegions.Count));
          end;
        'C', 'E': // ellipse
          begin
            if not (Length(rCoords) in [3,4]) then
              raise EmpRegionImage.CreateFmt('Invalid coordinates count (line %d)',
                [i + 1]);

            if Length(rCoords) = 3 then
              AddInt(rCoords, (rCoords[1] - rCoords[0]) + rCoords[2]); // circle
            h := CreateEllipticRgn(rCoords[0], rCoords[1], rCoords[2], rCoords[3]);
          end;
        'S', 'R': //rectangle
          begin
            if not (Length(rCoords) in [3,4]) then
              raise EmpRegionImage.CreateFmt('Invalid coordinates count (line %d)',
                [i + 1]);

            if Length(rCoords) = 3 then
              AddInt(rCoords, (rCoords[1] - rCoords[0]) + rCoords[2]);  // square
            h := CreateRectRgn(rCoords[0], rCoords[1], rCoords[2], rCoords[3]);
          end;
        'P': //polygon
          begin
            if (Length(rCoords) mod 2) <> 0 then
              raise EmpRegionImage.CreateFmt('Invalid coordinates count (line %d)',
                [i + 1]);
            h := CreatePolygonRgn(rCoords[0], Length(rCoords) div 2, WINDING);
          end;
        else
          raise EmpRegionImage.CreateFmt('Invalid region type (line %d)', [i + 1]);
      end;

      // add
      New(pEntry);
      with pEntry^ do
      begin
        StrTag := sDesc;
        Region := h;
        Assoc := i;
        Selected := False;
      end;
      FRegions.Add(pEntry)
    end;

  ClearLastRgn;
  CheckHilite(False);
end;

{-----------------------------------------------------------------------------
  Arguments: Shift: TShiftState; X, Y: integer
  Result:    None
  Purpose:   mouse mevement handler
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.MouseMove(Shift: TShiftState; X, Y: integer);
begin
  if FSettingCursor then
    Exit;
  FMoving := False;
  CheckHilite(X, Y, False);
  FMoving := True;
  inherited;
end;

{-----------------------------------------------------------------------------
  Arguments: x, y: integer
  Result:    integer
  Purpose:   what region is under the given cursor pos
-----------------------------------------------------------------------------}
function TCustomMPRegionImage.RegionAtPos(x, y: integer): integer;
var
  i: integer;
begin
  Result := -1;
  if FMoving then
    Result := FCurrentRegion
  else if FRegions.Count > 0 then
    for i := Pred(FRegions.Count) downto 0 do
    begin
      if PtInRegion(FRegions[i].Region, X, Y) then
      begin
        Result := i;
        Break;
      end;
    end;
  FCurrentRegion := Result;
  FMoving := False;
end;

{-----------------------------------------------------------------------------
  Arguments: Index: integer
  Result:    HRGN
  Purpose:   create one region based on two ore more overlapping regions
-----------------------------------------------------------------------------}
function TCustomMPRegionImage.GetCombinedRegion(Index: integer): HRGN;
var
  h: HRGN;
  i: integer;
begin
  Result := 0;
  if Index < FRegions.Count then
  begin
    h := CopyRegion(FRegions[Index].Region);
    if Index < Pred(FRegions.Count) then
    begin
      for i := (Index +1) to Pred(FRegions.Count) do
      begin
        CombineRgn(h, h, FRegions[i].Region, RGN_DIFF);
      end;
    end;
    Result := h;
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: Index: integer; Value: TCursor
  Result:    None
  Purpose:   set normal/over region cursor
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.SetCursor(Index: integer; Value: TCursor);
var
  pt, pt1: TPoint;
begin
  case Index of
    0: FRegionsCursor := Value;
    else
      FCursor := Value;
  end;
  GetCursorPos(pt);
  pt1 := ScreenToClient(pt);

  CheckHilite(pt1.X, pt1.Y, False);

  // redraw cursor
  SetCursorPos(pt.x, pt.y + 1);
  SetCursorPos(pt.x, pt.y);
end;

{-----------------------------------------------------------------------------
  Arguments: p: Pointer; isize: integer
  Result:    string
  Purpose:   create a string from region data
-----------------------------------------------------------------------------}
function XRegion2Str(p: Pointer; isize: integer): string;
var
  pCoord: PDWORD;
begin
  Result := '';
  if (isize > 0) and ((iSize mod 4) = 0) then
  begin
    with PRGNDATAHEADER(p)^ do
      if (iType = RDH_RECTANGLES) and (nCount > 0) then
      begin
        pCoord := p;
        for iSize := 1 to (iSize div 4) do
        begin
          Result := Result + IntToStr(pCoord^) + ',';
          Inc(pCoord);
        end;
      end;
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: h: HRGN
  Result:    string
  Purpose:   string from region data ("X,..." format)
-----------------------------------------------------------------------------}
function XRegion(h: HRGN): string;
var
  ms: TMemoryStream;
begin
  ms := TMemoryStream.Create;
  try
    ms.Size := GetRegionData(h, 0,nil);
    GetRegionData(h, ms.Size, ms.Memory);
    Result := 'X,' + XRegion2Str(ms.memory, ms.size);
  finally
    ms.Free;
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: Index, dX, dY: integer
  Result:    None
  Purpose:   move a region and its textual representation
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.OffsetRegion(Index, dX, dY: integer);

  procedure OffsetRegionListEntry(const Index, dx, dy: integer);
  var
    cCmd: char;
    sDesc: string;
    rCoords: TIntArray;
    j: integer;
    h: HRGN;
  begin
    SplitRegionlistLine(Index, FRegionsList[Index], cCmd, sDesc, rCoords);
    case UpCase(cCmd) of
      'X': // extended region
        begin
          if (Length(rCoords) mod 4) <> 0 then
            raise EmpRegionImage.CreateFmt('Invalid X coordinates count (%d, line %d)',
              [Length(rCoords), Index + 1]);

          // header + region data (hope, they are four byte aligned)
          Assert(rCoords[0] = sizeof(TRGNDATAHEADER));
          h := ExtCreateRegion(nil, sizeof(cardinal) * Length(rCoords),
            PRGNDATA(@rCoords[0])^);
          try
            OffsetRgn(h, dx, dy);
            FRegionsList[Index] := sDesc + ';' + XRegion(h);
          finally
            DeleteObject(h);
          end;
        end;
      '+': // combine
        begin
          if (not (Length(rCoords) in [1,2])) or (rCoords[0] < 2) then
            raise EmpRegionImage.CreateFmt('Invalid combination flag count (line %d)',
              [Index +1]);

          for j := Index -1 to Index -rCoords[0] do
            OffsetRegionListEntry(j, dX, dY);
        end;
      'C', 'E', 'R', 'S':
        begin
          if not (Length(rCoords) in [3,4]) then
            raise EmpRegionImage.CreateFmt('Invalid coordinates count (line %d)',
              [Index +1]);

          if Length(rCoords) = 3 then
            AddInt(rCoords, (rCoords[1] - rCoords[0]) + rCoords[2]);

          OffsetRect(PRect(@rCoords[0])^, dX, dy);
          FRegionsList[Index] := Format('%s;%s,%d,%d,%d,%d,%s',
            [sDesc, UpCase(cCmd), rCoords[0], rCoords[1], rCoords[2], rCoords[3]]);
        end;
      'P': //polygon
        begin
          if (Length(rCoords) mod 2) <> 0 then
            raise EmpRegionImage.CreateFmt('Invalid coordinates count (line %d)',
              [Index +1]);


          FRegionsList[Index] := sDesc + ';' + UpCase(cCmd) + ',';
          j := 0;
          while j < Length(rCoords) do
          begin
            Inc(rCoords[j], dX);
            Inc(rCoords[j + 1], dY);
            FRegionsList[Index] := FRegionsList[Index] + Format('%d,%d,',
              [rCoords[j], rCoords[j + 1]]);
            Inc(j, 2);
          end;
        end;
      else
        raise EmpRegionImage.CreateFmt('Invalid region type (line %d)', [Index +1]);
    end;
  end;
begin
  if Index >= FRegions.Count then
    raise EmpRegionImage.Create('Invalid region index');
  OffsetRgn(FRegions[Index].Region, dX, dY);

  FDoChangeList := False;
  try
    OffsetRegionListEntry(FRegions[Index].Assoc, dX, dY);
  finally
    FDoChangeList := True;
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: None
  Result:    integer
  Purpose:   get region at current cursor position
-----------------------------------------------------------------------------}
function TCustomMPRegionImage.RegionAtCursorPos: integer;
var
  pt: TPoint;
begin
  GetCursorPos(pt);
  pt := ScreenToClient(pt);
  Result := RegionAtPos(pt.x, pt.y);
end;

{-----------------------------------------------------------------------------
  Arguments: Index: integer
  Result:    string
  Purpose:   get textual tag of a region entry
-----------------------------------------------------------------------------}
function TCustomMPRegionImage.GetStrTag(Index: integer): string;
begin
  if Index >= FRegions.Count then
    raise EmpRegionImage.CreateFmt('Invalid region index [%d]', [Index]);

  Result := FRegions[Index].StrTag;
end;

{-----------------------------------------------------------------------------
  Arguments: Index: integer; Value: string
  Result:    None
  Purpose:   set textual tag
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.SetStrTag(Index: integer; Value: string);
var
  cCmd: char;
  sDesc: string;
  rCoords: TIntArray;
  i: integer;
begin
  if Index >= FRegions.Count then
    raise EmpRegionImage.Create('Invalid region index');

  if Pos(';', Value) > 0 then
    raise EmpRegionImage.Create('No ";" is allowed in StrTag');

  FRegions[Index].StrTag := Value;
  // zeilenbeschreibung setzen
  SplitRegionListLine(FRegions[Index].Assoc, FRegionsList[FRegions[Index].Assoc],
    cCmd, sDesc, rCoords);
  sDesc := Value + ';' + cCmd + ',';
  for i := 0 to Pred(Length(rCoords)) do
    sDesc := sDesc + IntToStr(rCoords[i]) + ',';
  FDoChangeList := False;
  try
    FRegionsList[FRegions[Index].Assoc] := sDesc;
  finally
    FDoChangeList := True;
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: None
  Result:    None
  Purpose:   paint handler
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.Paint;
var
  SaveDrawing: boolean;
begin
  if (csDestroying in ComponentState) or (csLoading in ComponentState) then
    Exit;

  if csDesigning in ComponentState then
    with inherited Canvas do
    begin
      Pen.Style := psDash;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
    end;

  SaveDrawing := FDrawing;
  FDrawing := True;
  with inherited Canvas do
    try
      Draw(0,0, FWorkBitmap);
    finally
      FDrawing := SaveDrawing;
    end;
end;

{-----------------------------------------------------------------------------
  Arguments: Sender: TObject
  Result:    None
  Purpose:   handle bitmap change
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.BitmapChanged(Sender: TObject);
var
  SaveDrawing: boolean;
begin
  SaveDrawing := FDrawing;
  FDrawing := True;
  try
    FWorkBitmap.Transparent := False;
    FWorkBitmap.Assign(FBitmap);
    FWorkBitmap.Transparent := FTransparent;
    if (Width <> FBitmap.Width) or (Height <> FBitmap.Height) then
    begin
      if (FBitmap.Width > 0) or (not (csDesigning in ComponentState)) then
        Width := FBitmap.Width;
      if (FBitmap.Height > 0) or (not (csDesigning in ComponentState)) then
        Height := FBitmap.Height;
    end;
    if not FTransparent then
      ControlStyle := ControlStyle + [csOpaque]
    else
      ControlStyle := ControlStyle - [csOpaque]
    finally
      FDrawing := SaveDrawing;
  end;
  if not FDrawing then
    Invalidate;
end;

{-----------------------------------------------------------------------------
  Arguments: None
  Result:    None
  Purpose:   get the components canvas
-----------------------------------------------------------------------------}
function TCustomMPRegionImage.GetCanvas: TCanvas;
begin
  Result := FBitmap.Canvas;
end;

{-----------------------------------------------------------------------------
  Arguments: Index: integer; Value: TBitmap
  Result:    None
  Purpose:   set one of the internal bitmaps
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.SetBitmap(Index: integer; Value: TBitmap);
begin
  case Index of
    1: FHiBitmap.Assign(Value);
    2: FSelBitmap.Assign(Value);
    else
      FHiBitmap.Height := 0;
      FHiBitmap.Width := 0;
      FSelBitmap.Height := 0;
      FSelBitmap.Width := 0;
      FBitmap.Width := 0;
      FBitmap.Height := 0;
      FBitmap.Assign(Value);
      CreateHiAndSelBitmap;
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: Value: boolean
  Result:    None
  Purpose:   set transparency
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.SetTransparent(Value: boolean);
begin
  if Value <> FTransparent then
  begin
    FTransparent := Value;
    FWorkBitmap.Transparent := Value;
    BitmapChanged(Self);
  end;
end;

procedure TCustomMPRegionImage.PaintHRegion(h: HRGN);
begin
  with TMPRegionImageCanvas(Canvas) do
    PaintRegion(h);
end;

{-----------------------------------------------------------------------------
  Arguments: Sender: TObject
  Result:    None
  Purpose:   handle regions list change
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.RegionListChanged(Sender: TObject);
begin
  if FDoChangeList then
    ImportRegion(FRegionsList);
end;

procedure TCustomMPRegionImage.SetHiliteCurrent(const Value: boolean);
begin
  if Value <> FHiliteCurrent then
  begin
    ClearLastRgn;
    FHiliteCurrent := Value;
    Invalidate;
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: None
  Result:    None
  Purpose:   always autosize
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.Resize;
begin
  if (Width < FBitmap.Width) or (Height < FBitmap.Height) and
    (not (csDesigning in ComponentState)) then
    SetBounds(Left, Top, FBitmap.Width, FBitmap.Height);

  inherited;
end;

procedure TCustomMPRegionImage.ClearLastRgn;
begin
  if FLastHilited <> -1 then
    FLastHilited := -1;
  FWorkBitmap.Transparent := False;
  FWorkBitmap.Assign(FBitmap);
  FWorkBitmap.Transparent := FTransparent;
end;

procedure TCustomMPRegionImage.CheckHilite(bSelChanged: boolean);
var
  pt: TPoint;
begin
  GetCursorPos(pt);
  pt := ScreenToClient(pt);
  CheckHilite(pt.X, pt.Y, bSelChanged);
end;

{-----------------------------------------------------------------------------
  Arguments: x, y: integer; bSelChanged: boolean
  Result:    None
  Purpose:   check current hilighted/selected region
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.CheckHilite(x, y: integer; bSelChanged: boolean);
var
  i: integer;
  FReg, FSave: HRGN;
  bPaint: boolean;
begin
  i := RegionAtPos(x, y);
  bPaint := False;

  if bSelChanged then
  begin
    ClearLastRgn;
    bPaint := True;
  end;

  if (i > -1) and (not FMouseDown) and FMouseInPos then
  begin
    ShowCursor(FRegionsCursor);
    if FHiliteCurrent and (FLastHilited <> i) then
    begin
      if not bPaint then
        ClearLastRgn;
      FLastHilited := i;
      if Assigned(FOnEnterRegion) then
        FOnEnterRegion(self, FLastHilited);
      FReg := CombinedRegion[i];
      with FWorkBitmap.Canvas do
        try
          // draw hilite +++++++++++
          FSave := SelectObject(Handle, FReg);
          try
            Draw(0,0,FHiBitmap);
          finally
            SelectObject(Handle, fSave);
          end;
        finally
          DeleteObject(FReg);
        end;
      bPaint := True;
    end;
  end
  else
  begin
    ShowCursor(FCursor);
    if FHiliteCurrent and (FlastHilited <> -1) then
    begin
      if Assigned(FOnLeaveRegion) then
        FOnLeaveRegion(self, FLastHilited);
      if not bPaint then
        ClearLastRgn;
      bPaint := True;
    end;
  end;

  if FAutoSelect and (ItemIndex > -1) then
  begin
    if (not FMultiSelect) or (SelCount < 2)
    then
      FReg := CombinedRegion[ItemIndex]
    else
      FReg := GetSelectedCombiRegion;
    try
      // draw selected +++++++++++++++++
      with FWorkBitmap.Canvas do
        try
          FSave := SelectObject(Handle, FReg);
          try
            Draw(0,0,FSelBitmap);
          finally
            SelectObject(Handle, fSave);
          end;
        finally
          DeleteObject(FReg);
        end;
    finally
      DeleteObject(FReg);
    end;
  end;

  if bPaint then
    Paint;
end;

procedure TCustomMPRegionImage.CMMouseLeave(var Message: TMessage);
var
  b: boolean;
begin
  inherited;
  b := FMouseInPos;
  FMouseInPos := False;
  if b then
    CheckHilite(False);
end;

procedure TCustomMPRegionImage.StyleChanged(Sender: TObject);
begin
  CheckHilite(True);
  Invalidate;
end;

procedure TCustomMPRegionImage.MouseDown(Button: TMouseButton;
  Shift: TShiftState; X, Y: integer);
var
  b: boolean;
  i: integer;
  bSel: boolean;
begin
  b := FMouseDown;
  bSel := False;
  FMouseDown := True;

  if FAutoSelect and (Shift = [ssLeft]) then
  begin
    i := RegionAtPos(x, y);
    if (i <> ItemIndex) or (SelCount <> 1) then
    begin
      ItemIndex := i;
      if Assigned(FOnSelectionChange) then
        FOnSelectionChange(self);
    end;
    b := False; // force redrawing
    bSel := True;
  end
  else
    if FAutoSelect and FMultiSelect and (Shift = [ssLeft, ssShift]) then
    begin
      i := RegionAtPos(x, y);
      if i > -1 then
      begin
        FRegions[i].Selected := not FRegions[i].Selected;
        if Assigned(FOnSelectionChange) then
          FOnSelectionChange(self);
        b := False; // force redrawing
        bSel := True;
      end;
    end;

  if not b then
    CheckHilite(x, y, bSel);
  inherited;
end;

procedure TCustomMPRegionImage.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: integer);
var
  b: boolean;
begin
  inherited;
  b := FMouseDown;
  FMouseDown := False;
  if b then
    CheckHilite(False);
end;


procedure TCustomMPRegionImage.CMMouseEnter(var Message: TMessage);
var
  b: boolean;
begin
  inherited;
  b := FMouseInPos;
  FMouseInPos := True;
  if not b then
    CheckHilite(False);
end;

procedure TCustomMPRegionImage.SetRegionList(const Value: TStringList);
var
  b: boolean;
begin
  b := FRegionsChanging;
  try
    FRegionsChanging := True;
    FRegionsList.Assign(Value);
  finally
    FRegionsChanging := b;
  end;
  if not b then
    FRegionsList.OnChange(FRegionsList);
end;

function TCustomMPRegionImage.GetSingleRegion(Index: integer): HRGN;
begin
  Result := FRegions[Index].Region;
end;

procedure TCustomMPRegionImage.Assign(Source: TPersistent);
begin
  if Source is TCustomMPRegionImage then
  begin
    FRegionsChanging := True;
    FDrawing := True;
    with TCustomMPRegionImage(Source) do
    begin
      self.FBitmap.Assign(FBitmap);
      self.FHiBitmap.Assign(FHiBitmap);
      self.FSelBitmap.Assign(FSelBitmap);
    end;
    FRegionsChanging := False;
    FDrawing := False;
    CheckHilite(True);
  end;
  inherited;
end;

procedure TCustomMPRegionImage.SetAutoSelect(const Value: boolean);
begin
  if FAutoSelect <> Value then
  begin
    FAutoSelect := Value;
    if ItemIndex <> -1 then
      CheckHilite(True);
  end;
end;

procedure TCustomMPRegionImage.SetItemIndex(const Value: integer);
begin
  if (ItemIndex <> Value) or (SelCount <> 1) then
  begin
    FRegions.DeSelectAll;
    if Value > -1
    then
      FRegions.Items[Value].Selected := True;

    if Assigned(FOnSelectionChange) then
      FOnSelectionChange(self);

    if FAutoSelect then
      CheckHilite(True);
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: hr: HRGN
  Result:    HRGN
  Purpose:   create a copy of a region
-----------------------------------------------------------------------------}
function TCustomMPRegionImage.CopyRegion(hr: HRGN): HRGN;
var
  Buf: Pointer;
  isz: cardinal;
begin
  Result := 0;
  isz := GetRegionData(hr, 0, nil);
  if isz > 0 then
  begin
    GetMem(Buf, isz);
    try
      if GetRegionData(hr, isz, Buf) = isz then
      begin
        Result := ExtCreateRegion(nil, isz, TRgnData(Buf^));
      end;
    finally
      FreeMem(Buf);
    end;
  end;
end;

{-----------------------------------------------------------------------------
  Arguments: iLine: integer; s: string; var cCmd: char; var sDesc: string;
             var rCoords: TIntArray
  Result:    None
  Purpose:   split a regionlist line into its components
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.SplitRegionlistLine(iLine: integer; s: string;
  var cCmd: char; var sDesc: string; var rCoords: TIntArray);
var
  j: integer;
begin
  j := Pos(';', s);
  if j > 0 then
  begin
    sDesc := Copy(s, 1,j - 1);
    Delete(s, 1,j);
  end
  else
    sDesc := '';

  j := Pos(',', s);
  if j = 2 then
  begin
    cCmd := s[1];

    // skip type and first comma
    Delete(s, 1,2);

    // get coords
    rCoords := SplitCoords(s, iLine);
  end
  else
    raise EmpRegionImage.CreateFmt('Invalid region type in line %d', [iLine + 1]);
end;

{-----------------------------------------------------------------------------
  Arguments: None
  Result:    None
  Purpose:   copy normal into hilite/selected bitmap
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.CreateHiAndSelBitmap;
begin
  with FHiBitmap do
    try
      OnChange := nil;
      if (Width = 0) and (Height = 0) then
        Assign(FBitmap);
    finally
      OnChange := StyleChanged;
    end;

  with FSelBitmap do
    try
      OnChange := nil;
      if (Width = 0) and (Height = 0) then
        Assign(FBitmap);
    finally
      OnChange := StyleChanged;
    end;
end;

function TCustomMPRegionImage.GetBitmap(Index: integer): TBitmap;
begin
  case Index of
    1: Result := FHiBitmap;
    2: Result := FSelBitmap;
    else
      Result := FBitmap;
  end;
end;

procedure TCustomMPRegionImage.Loaded;
begin
  inherited;
  CreateHiAndSelBitmap;
end;

function TCustomMPRegionImage.RegionsCount: integer;
begin
  Result := FRegions.Count;
end;

{-----------------------------------------------------------------------------
  Arguments: sName: string
  Result:    integer
  Purpose:   find a region whose strtag is sName
-----------------------------------------------------------------------------}
function TCustomMPRegionImage.IndexOfStrTag(sName: string): integer;
var
  i: integer;
begin
  Result := -1;
  for i := 0 to Pred(FRegions.Count) do
    if AnsiCompareText(sName, FRegions[i].StrTag) = 0 then
    begin
      Result := i;
      Break;
    end;
end;

{-----------------------------------------------------------------------------
  Arguments: ACursor: TCursor
  Result:    None
  Purpose:   show cursor (normal/hilite)
-----------------------------------------------------------------------------}
procedure TCustomMPRegionImage.ShowCursor(ACursor: TCursor);
var
  pt: TPoint;
  b: boolean;
begin
  if inherited Cursor = ACursor then
    Exit;
  b := FSettingCursor;
  FSettingCursor := True;
  try
    inherited Cursor := ACursor;
    GetCursorPos(pt);

    // move cursor to immediately show changed cursor shape
    SetCursorPos(pt.X, pt.Y + 1);
    SetCursorPos(pt.X, pt.Y);
  finally
    FSettingCursor := b;
  end;
end;

function TCustomMPRegionImage.GetSelected(Index: Integer): Boolean;
begin
  if not FMultiSelect
  then
    Result := ItemIndex = Index
  else
    Result := FRegions.Items[Index].Selected;
end;

procedure TCustomMPRegionImage.SetSelected(Index: Integer;
  const Value: Boolean);
begin
  if not FMultiSelect
  then
    FRegions.DeSelectAll;
  FRegions.Items[Index].Selected := Value;
  CheckHilite(True);
end;

function TCustomMPRegionImage.GetItemIndex: integer;
var
  LIntLoop: integer;
begin
  Result := -1;
  with FRegions
  do
    if Bool(Count)
    then
      for LIntLoop := 0 to Pred(Count)
      do
        if Items[LIntLoop].Selected then
        begin
          Result := LIntLoop;
          Break;
        end;
end;

function TCustomMPRegionImage.GetSelectedCombiRegion: HRGN;
var
  h,h1: HRGN;
  LIntLoop: integer;
begin
  Result := 0;
  if SelCount > 0 then
  begin
    h := 0;
    for LIntLoop := 0 to Pred(RegionsCount)
    do
      if Selected[LIntLoop] then
      begin
        if h = 0
        then
          h := CombinedRegion[LIntLoop]
        else
        begin
          h1 := CombinedRegion[LIntLoop];
          CombineRgn(h, h, h1, RGN_OR);
          DeleteObject(h1);
        end;
      end;
    Result := h;
  end;
end;

function TCustomMPRegionImage.GetSelCount: integer;
var
  LIntLoop: Integer;
begin
  if not FMultiSelect then
  begin
    if ItemIndex > -1
    then
      Result := 1
    else
      Result := 0;
  end
  else
  begin
    Result := 0;
    with FRegions
    do
      if Bool(Count)
      then
        for LIntLoop := 0 to Pred(Count)
        do
          if Items[LIntLoop].Selected
          then
            Inc(Result);
  end;
end;

procedure TCustomMPRegionImage.SetMultiSelect(const Value: Boolean);
begin
  if FMultiSelect <> Value then
  begin
    FMultiSelect := Value;
    FRegions.DeSelectAll;
    CheckHilite(True);
  end;
end;

function TCustomMPRegionImage.GetVersion: string;
begin
  Result := _ABOUT;
end;

procedure TCustomMPRegionImage.SetVersion(const Value: string);
begin
  //
end;


{ TRegionList }

function TRegionList._Get(Index: integer): PRegionEntry;
begin
  Result := inherited Get(Index);
end;

procedure TRegionList._Put(Index: integer; const Value: PRegionEntry);
begin
  inherited Put(Index, Value);
end;

procedure TRegionList.Notify(Ptr: Pointer; Action: TListNotification);
begin
  if (Action in [lnDeleted, lnExtracted]) and Assigned(Ptr) then
    with PRegionEntry(Ptr)^ do
    begin
      StrTag := '';
      DeleteObject(Region);
      Dispose(Ptr);
    end;
  inherited;
end;

procedure TRegionList.DeSelectAll;
var
  LIntLoop: Integer;
begin
  if Bool(Count)
  then
    for LIntLoop := 0 to Pred(Count)
    do
      Items[LIntLoop].Selected := False;
end;

end.
