unit RRFResize;
  (* A RRFResizer control facilitates maintenance and utilization of a database
   of screen-size-specific bounds for controls on forms.  Any number of forms
   can share a single dataset of such information, and the control will create
   the dataset if it does not exist.  Each such dataset has the following fields

          ScreenHeight   integer   key
          ScreenWidth    integer   key
          FormName       A 60      key
          ControlName    A 60      key
          Height         integer
          Left           integer
          Top            integer
          Width          integer

   Paradox datasets are assumed, but this restriction could easily be relaxed.

     A RRFResizer control appears on its parent control as a button with the
   caption 'Record hints' (if there are no data for the owner) or 'Delete hints'
   (if there are such data).  Pressing the button in its 'Record hints' mode
   causes entries for the owner (a form) and each of its controls to be added
   to the table for the current screen size.  The 'Delete hints' button deletes
   all of the data related to the current screen size for the owning form and
   each of its controls.

     The active side of the RRFResizer control is invoked with the

          procedure   UseDisplayHints(Form: tForm; ShowButton: boolean);

   call, which, if the table contains data for this form at the current screen
   size, moves and resizes the form and each of its owned controls.  In the
   contemplated use, data might be recorded and revised for a while during
   development, but thereafter the button would no longer be used, and it could
   be made invisible.  A typical call in a form's OnShow event handler might be

          Resizer.UseDisplayHints(self, DevelopmentMode)

   where DevelopmentMode was a boolean maintained by the designer.

     The database in which the hint dataset will be stored must be specified
   by setting

          property DatabaseName: string;

   The dataset itself may be specified by setting

          property TableName: string;

   but the dataset will be created with a default name ('DisplayHints.DB') if
   TableName is not set.

      *)
      (* eject *)
  interface
    uses
      Buttons       (* tBitBtn *),
      Classes       (* tComponent *),
      DBTables      (* tDatabase, tTable, tTableType *),
      Forms         (* tForm, Screen *);

    type
      tRRFResizer = class(tBitBtn)
        private
          HintsExist: boolean;
          fDataBaseName: string;
          fTableName: string;
          function  Operable: boolean;
          procedure SetDataBaseName(const Value: string);
        protected
          Table (* not private to allow access by property editor *): tTable;
          procedure ButtonClicked(Sender: tObject);
        public
          constructor Create(AnOwner: tComponent); override;
          destructor  Destroy;                     override;
          procedure   Loaded;                      override;
          procedure   UseDisplayHints(Form: tForm; ShowButton: boolean);
        published
          property DatabaseName: string read fDatabaseName write SetDataBaseName;
          property TableName:    string read fTableName    write fTableName;
        end (* tRRFResizer *);

  implementation
    uses
      Controls   (* tControl *),
      DB         (* tDataSet, tField, tFieldType, tIndexDef, tIndexOptions *),
      Graphics   (* tColor *),
      {MyConsts   (* nullstring *),}
      Variants   (* VarArrayOf *),
      Windows    (* LoadBitmap *);

    const
      nullstring = '';

    resourcestring
      DefaultTableName = 'DisplayHints';
      DeleteHints = 'Delete hints';
      RecordHints = 'Record hints';
      (* eject *)
{ tRRFResizer }

    procedure tRRFResizer.ButtonClicked(Sender: tObject);
      (* Collect or delete the display hints *)

      var
        Form: tForm;
        SH, SW: integer;

      procedure DeleteLayout;
        (* Delete from the DisplayHints DB all of the records regarding this
        form at this screen size *)

        begin (* DeleteLayout *)
          while Table.Locate('ScreenHeight;ScreenWidth;FormName',
                             VarArrayOf([SH, SW, Form.Name]),
                             []) do
            Table.Delete
        end   (* DeleteLayout *);

      procedure RecordLayout;
        (* Add the current configuration of this form to the DisplayHints DB *)

        var
          I: integer;

        procedure CollectDisplayBounds(Control: tControl);
          (* Put this control's info into the database *)

          begin (* CollectDisplayBounds *)
            with Table do
              begin (* with *)
                if Locate('ScreenHeight;ScreenWidth;FormName;ControlName',
                          VarArrayOf([SH, SW, Form.Name, Control.Name]),
                          []) then
                  Edit
                else
                  begin (* new record *)
                    Insert;
                    FieldByName('ScreenHeight').AsInteger := SH;
                    FieldByName( 'ScreenWidth').AsInteger := SW;
                    FieldByName(  'FormName'  ).AsString  := Form.Name;
                    FieldByName( 'ControlName').AsString  := Control.Name
                  end   (* new record *);

                FieldByName('Height').AsInteger := Control.Height;
                FieldByName( 'Left' ).AsInteger := Control.Left;
                FieldByName(  'Top' ).AsInteger := Control.Top;
                FieldByName( 'Width').AsInteger := Control.Width;

                Post
              end   (* with *)
          end   (* CollectDisplayBounds *);
          (* eject *)
        begin (* RecordLayout *)
          with Form do
            begin (* with *)
              CollectDisplayBounds(Form);
              for I := 1 to ComponentCount do
                if Components[I-1] is tControl then
                  CollectDisplayBounds(tControl(Components[I-1]))
            end   (* with *)
        end   (* RecordLayout *);

      begin (* tRRFResizer.ButtonClicked *)
        if Operable then
          if self.Owner is tForm then
            begin (* nondegenerate *)
              SH := Screen.Height;
              SW := Screen.Width;
              Form := tForm(self.Owner);
              if HintsExist then
                DeleteLayout
              else
                RecordLayout
            end   (* nondegenerate *)
      end   (* tRRFResizer.ButtonClicked *);

    constructor tRRFResizer.Create(AnOwner: tComponent);
      (* allocate the subsidiary components *)

      begin (* tRRFResizer.Create *)
        inherited Create(AnOwner);
        Height := 49;
        Width := 113;
        Glyph.Handle := LoadBitmap(HInstance, 'TRRFRESIZER');
        Font.Color := clRed;
        Font.Style := [fsBold];
        Caption := RecordHints;

        Table := tTable.Create(nil);
        Table.TableType := ttParadox;

        OnClick := ButtonClicked
      end   (* tRRFResizer.Create *);

    destructor tRRFResizer.Destroy;
      (* reclaim storage *)

      begin (* tRRFResizer.Destroy *)
        Table.Close;
        Table.Free;
        inherited Destroy
      end   (* tRRFResizer.Destroy *);
      (* eject *)
    procedure tRRFResizer.Loaded;
      (* All properties in, so pass them on to the Table *)

      begin (* tRRFResizer.Loaded *)
        inherited Loaded;

        if fDatabaseName <> nullstring then
          Table.DatabaseName := fDatabaseName;
        if fTableName <> nullstring then
          Table.TableName := fTableName
      end   (* tRRFResizer.Loaded *);

    function tRRFResizer.Operable: boolean;
      (* If the necessary table can't be found (or created) and opened, return
      false *)

      procedure AddAField(const FieldName: string; ADataType: tFieldType);
        (* Create a field of the specified kind *)

        const
          StringSize = 60;

        begin (* AddAField *)
          with Table.FieldDefs.AddFieldDef do
            begin (* specify the field *)
              Name := FieldName;
              DataType := ADataType;
              Required := true;
              if DataType = ftString then
                Size := StringSize
            end   (* specify the field *)
        end   (* AddAField *);

      begin (* tRRFResizer.Operable *)
        with Table do
          begin (* with fTable *)
            if not Exists then
              if fDatabaseName <> nullstring then
                begin (* try to create it *)
                  Close;
                  Table.DatabaseName := fDatabaseName;
                  if fTableName = nullstring then
                    TableName := DefaultTableName
                  else
                    TableName := fTableName;
                  FieldDefs.Clear;
                  AddAField('ScreenHeight', ftInteger);
                  AddAField( 'ScreenWidth', ftInteger);
                  AddAField(  'FormName'  , ftString );
                  AddAField( 'ControlName', ftString );
                  AddAField(   'Height'   , ftInteger);
                  AddAField(    'Left'    , ftInteger);
                  AddAField(     'Top'    , ftInteger);
                  AddAField(    'Width'   , ftInteger);

                  IndexDefs.Clear;
                  with IndexDefs.AddIndexDef do
                    begin (* create primary index *)
                      Name := nullstring;
                      Fields := 'ScreenHeight;ScreenWidth;FormName;ControlName';
                      Options := [ixPrimary, ixUnique]
                    end   (* create primary index *);

                  CreateTable
                end   (* try to create it *);

            if Exists then
              try    (* might not be openable *)
                if not Active then
                  Open;
                Result := true
              except (* might not be openable *)
                Result := false
              end    (* might not be openable *)
            else
              Result := false
          end   (* with fTable *)
      end   (* tRRFResizer.Operable *);
      (* eject *)
    procedure tRRFResizer.SetDataBaseName(const Value: string);
      (* For object inspector *)

      begin (* tRRFResizer.SetDataBaseName *)
        fDatabaseName := Value
      end   (* tRRFResizer.SetDataBaseName *);

    procedure tRRFResizer.UseDisplayHints(Form: tForm; ShowButton: boolean);
      (* This is meant to be called from the OnShow event of the
       Form.  For each control on the form (including the form itself),
       try to find an entry in the hints DB for the current screen size.
       If such an entry is found, use it to reset the bounds of the control.
       Finally, set the button to be Visible or not as described by
       ShowButton, and set its caption to show whether hints are available
       ('Delete hints') or not ('Record hints').
       *)

      var
        I, OldFormHeight, OldFormWidth, SH, SW: integer;

      function UsedCoordinatesOf(Control: tControl): boolean;
        (* Return true iff successful in locating Control *)

        begin (* UsedCoordinatesOf *)
          with Table do
            begin (* with *)
              Result := Locate('ScreenHeight;ScreenWidth;FormName;ControlName',
                               VarArrayOf([SH, SW, Form.Name, Control.Name]),
                               []);
              if Result then
                Control.SetBounds(FieldByName( 'Left' ).AsInteger,
                                  FieldByName(  'Top' ).AsInteger,
                                  FieldByName( 'Width').AsInteger,
                                  FieldByName('Height').AsInteger)
            end   (* with *)
        end   (* UsedCoordinatesOf *);
        (* eject *)
      begin (* tRRFResizer.UseDisplayHints *)
        OldFormHeight := Form.Height;
        OldFormWidth  := Form.Width;
        SH := Screen.Height;
        SW := Screen.Width;
        HintsExist := false;

        if Operable then
          with Form do
            if UsedCoordinatesOf(Form) then
              begin (* have old data *)
                HintsExist := true;
                if (Form.Height <> OldFormHeight) or
                   (Form.Width  <> OldFormWidth) then
                  for I := 1 to ComponentCount do
                    if Components[I-1] is tControl then
                      if not UsedCoordinatesOf(tControl(Components[I-1])) then
                        (* ignore it *)
              end   (* have old data *);

        if HintsExist then
          Caption := DeleteHints
        else
          Caption := RecordHints;
        Visible := ShowButton
      end   (* tRRFResizer.UseDisplayHints *);

end.
 