unit IndexTab;
  (* The IndexTab component provides hot buttons for jumping through a dataset
  to specific values of a stringfield or integerfield.  In the stringfield case,
  the buttons are initially single letters, but after jumping to (say) the first
  record whose key field begins with 'M', the 'in' button turns the buttons
  into 'Ma', 'Mb', etc., and so on to arbitrary depth; the 'out' button has
  the opposite effect.

    The integer case is managed similarly, except that the 'in' button causes
  the new buttons to range between those around the current selection.  When
  there are many more records than distinct integer values (for example, when
  integers are calendar years, and there are many more event records than
  distinct years), one may -- sometimes greatly -- accelerate the work of the
  IndexTab by supplying guidance
          property HighInteger: integer default 0;
          property LowInteger:  integer default 0;
  as to the upper and lower bounds of the field.  Conversely, when there are no
  more records than there are distinct integer values (for example, when the
  values are unique record IDs), it may be best to leave HighInteger and
  LowInteger at their default (inactive) values.  In a real-life application and
  a dataset with 5505 records and a complex OnFilterRecord event, the use of
  (LowInteger, HighInteger) sped the initialization of IndexTab from 91 seconds
  to 2 seconds.

    Buttons are not generated unless corresponding records exist in the dataset.
  Also, when the IndexTab is too small to contain all of the buttons that the
  dataset might support, buttons are generated selectively.  The buttons can be
  regenerated by clicking the button whose caption is an asterisk.

    The IndexTab will be set into integer or string mode as the
       property IndexType: tixType default ixString;
  property is ixstring [default] or ixInteger.

    The dataset and the index field are identified by setting the
       property DataSource: tDataSource;
       property DataField:  string;
  properties, respectively (the DataField property is somewhat misnamed, since
  it is, as noted, a field name (a string), not a field).

    Because an IndexTab is often placed just to the side of a DBGrid or Panel,
  the
       property HeightStd: tControl;
  property is provided.  If this is assigned to another control C, then whenever
  the buttons of the IndexTab are regenerated, the IndexTab is resized to be the
  same height as C.

    In the stringfield case, the 'in' option will not be offered if the current
  button captions are not shorter than
       property MaxDepth: integer default 2

    IndexTab components are not self-initializing.  Before the first use of
  an IndexTab, the
       procedure Initialize
  must be called, typically from the OnActivate method of the form on which the
  IndexTab has been placed.  If the dataset is large, it may not be desirable
  to initialize the IndexTab until the application has arranged that the dataset
  is using the appropriate index.  The
       procedure Arm
  does not do any initialization beyond arranging that the IndexTab will self-
  initialize when any tab is clicked. *)
  (* eject *)
  interface
    uses
      Classes  (* RegisterComponents, tStringList *),
      ComCtrls (* tCustomTabControl, tTabStyle *),
      Controls (* tAnchorKind, tControl, tCreateParams, crHourGlass, tCursor *),
      DB       (* tDataSource, tIntegerField, tLocateOption, tSmallIntField,
                    tStringField *),
      DBCtrls  (* tFieldDataLink *),
      Windows;

    type
      tixType = (ixInteger, ixString);

    type
      tRRFIndexTab = class(tCustomTabControl)
        private
          Initialized  (* don't do it twice              *): boolean;
          fHeightStd   (* optional; usually a DBgrid     *): tControl;
          fDataLink    (* help choose field              *): tFieldDataLink;
          FirstInt     (* for in/out with integer field  *),
          fHighInteger (* scan if High > Low             *),
          LastInt      (* for in/out with integer field  *),
          fLowInteger  (* scan if High > Low             *),
          fMaxDepth    (* string: max length of captions *),
          MyTabHeight  (* take account of font size      *): integer;
          fIndexType   (* integer or string for now      *): tixType;
          NextTab      (* nullstring or next integer     *),
          Prefix       (* length strictly < MaxDepth     *),
          PriorTab     (* will be new prefix if In       *): string;
          function  GetDataSource: tDataSource;
          function  GetFieldName: string;
          procedure SetDataSource(const Value: tDataSource);
          procedure SetFieldName (const Value: string);
          procedure SetIndexType (const Value: tixType);
          procedure SetUpTabs;
          function  Usable: boolean;
        protected
          procedure CreateWindowHandle(const Params: tCreateParams); override;
          procedure TabClicked(Sender: tObject);
        public
          procedure   Arm;
          constructor Create(AnOwner: tComponent); override;
          destructor  Destroy;                     override;
          procedure   Initialize;
        published
          property DataField: string       read GetFieldName  write SetFieldName
            (* 'DataField' is only name that will automatically invoke field
             editing in Object Inspector *);
          property DataSource:  tDataSource read GetDataSource write SetDataSource;
          property HeightStd:   tControl    read fHeightStd    write fHeightStd;
          property HighInteger: integer     read fHighInteger  write fHighInteger
                     default 0;
          property IndexType:   tixType     read fIndexType    write SetIndexType
                     default ixString;
          property LowInteger:  integer     read fLowInteger   write fLowInteger
                     default 0;
          property MaxDepth:    integer     read fMaxDepth     write fMaxDepth
                     default 2;
          property Anchors;
          property Visible;
        end (* tRRFIndexTab *);

    procedure Register;
    (* eject *)
  implementation
    uses
      DateUtils  (* YearOf *),
      DBTables   (* tDBDataSet, tQuery *),
      DesignIntf (* RegisterPropertyInCategory *),
      Forms      (* Screen *),
      Graphics   (* tFontStyle *),
      {MyConsts   (* nullstring, STAR *),}
      SysUtils   (* FreeAndNil, IntToStr *);

    const
      nullstring = '';
      STAR       = '*';

    procedure Register;
      (* Install into palette *)

      begin (* Register *)
        RegisterComponents('RRF', [tRRFIndexTab]);
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'DataField');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'DataSource');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'HeightStd');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'HighInteger');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'IndexType');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'LowInteger');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'MaxDepth');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'Name')
      end   (* Register *);

{ tRRFIndexTab }

    procedure tRRFIndexTab.Arm;
      (* Set the click-catcher, but don't otherwise initialize.  This is
      provided so that an application can organize dataset indices before
      trying to go through the searching of SetUpTabs *)

      begin (* tRRFIndexTab.Arm *)
        OnChange := TabClicked
      end   (* tRRFIndexTab.Arm *);

    constructor tRRFIndexTab.Create(AnOwner: tComponent);
      (* Initialize simple properties *)

      begin (* tRRFIndexTab.Create *)
        inherited Create(AnOwner);
        fDataLink := tFieldDataLink.Create;
        fDataLink.Control := self;
        fHighInteger := 0;
        fIndexType := ixString;
        fLowInteger := 0;
        fMaxDepth := 2;
        Initialized := false
      end   (* tRRFIndexTab.Create *);
      (* eject *)
    procedure tRRFIndexTab.CreateWindowHandle(const Params: tCreateParams);
      (* Only now does this component have a parent, without which cannot
      touch subproperties *)

      begin (* tRRFIndexTab.CreateWindowHandle *)
        inherited CreateWindowHandle(Params);
        Constraints.MinHeight := 80;
        Height := 185;
        Width := 49;
        MultiLine := true;
        with Tabs do
          begin (* just for design *)
            Clear;
            Add(STAR);
            Add('in');
            Add('out');
            TabIndex := -1
          end   (* just for design *);
        Anchors := [akBottom, akRight, akTop];
        Style := tsButtons;
        RaggedRight := false;
        ScrollOpposite := false;
        TabHeight := 20
      end   (* tRRFIndexTab.CreateWindowHandle *);

    destructor tRRFIndexTab.Destroy;
      (* reclaim storage *)

      begin (* tRRFIndexTab.Destroy *)
        fDataLink.Free;
        fDataLink := nil;
        inherited Destroy
      end   (* tRRFIndexTab.Destroy *);

    function tRRFIndexTab.GetDataSource: tDataSource;
      (* utilize the DataLink *)

      begin (* tRRFIndexTab.GetDataSource: tDataSource *)
        Result := fDataLink.DataSource
      end   (* tRRFIndexTab.GetDataSource: tDataSource *);

    function tRRFIndexTab.GetFieldName: string;
      (* utilize the DataLink *)

      begin (* tRRFIndexTab.GetFieldName *)
        Result := fDataLink.FieldName
      end   (* tRRFIndexTab.GetFieldName *);
      (* eject *)
    procedure tRRFIndexTab.Initialize;
      (* original setup *)

      const
        AscenderDescender = 'Wg';
        TabVMargin = 10;

      begin (* tRRFIndexTab.Initialize *)
        FirstInt := Low(integer);
        LastInt := maxint;
        NextTab := nullstring;
        PriorTab := nullstring;
        Prefix := nullstring;
        ParentFont := false;
        Font.Style := Font.Style + [fsBold];
        MyTabHeight := Canvas.TextHeight(AscenderDescender) + TabVMargin;
        Initialized := true;
        SetUpTabs
      end   (* tRRFIndexTab.Initialize *);

    procedure tRRFIndexTab.SetDataSource(const Value: tDataSource);
      (* for Object Inspector *)

      begin (* tRRFIndexTab.SetDataSource *)
        fDataLink.DataSource := Value
      end   (* tRRFIndexTab.SetDataSource *);

    procedure tRRFIndexTab.SetFieldName(const Value: string);
      (* for Object Inspector *)

      begin (* tRRFIndexTab.SetFieldName *)
        if Assigned(fDataLink) then
          with fDataLink do
            begin (* with fDataLink *)
              FieldName := Value;
              if Assigned(fDataLink.Field) then
                if fDataLink.Field is tIntegerField then
                  SetIndexType(ixInteger)
                else if fDataLink.Field is tSmallIntField then
                  SetIndexType(ixInteger)
                else if fDataLink.Field is tStringField then
                  SetIndexType(ixString)
            end   (* with fDataLink *)
      end   (* tRRFIndexTab.SetFieldName *);

    procedure tRRFIndexTab.SetIndexType(const Value: tixType);
      (* for Object Inspector *)

      begin (* tRRFIndexTab.SetIndexType *)
        fIndexType := Value;
        case fIndexType of
          ixInteger: MaxDepth := maxint;
           ixString: MaxDepth := 2
          end (* case on IndexType *)
      end   (* tRRFIndexTab.SetIndexType *);
      (* eject *)
    procedure tRRFIndexTab.SetUpTabs;
      (* tab captions are dataset-dependent *)

      var
        WantIn, WantOut: boolean;
        OldCursor: tCursor;
        TabsComing: tStringList;

      procedure BeSelective;
        (* Use only as many tabs as will fit *)

        var
          I, KeptTabs, MaxTabs, SeenTabs: integer;
          KeepFraction: real;

        begin (* BeSelective *)
          MaxTabs := (Height div MyTabHeight) - 1 (* for star *);
          if WantIn then
            dec(MaxTabs);
          if WantOut then
            dec(MaxTabs);
          if MaxTabs < 1 then
            MaxTabs := 1;
          KeepFraction := MaxTabs / TabsComing.Count;
          KeptTabs := 1;
          SeenTabs := 1;
          with TabsComing do
            for I := Count downto 2 do
              begin (* keep this tab? *)
                if KeptTabs/SeenTabs > KeepFraction then
                  Delete(I-1)
                else
                  inc(KeptTabs);
                inc(SeenTabs)
              end   (* keep this tab? *)
        end   (* BeSelective *);
        (* eject *)
      procedure Gather;
        (* Collect list of possible tabs *)

        var
          SavedPlace: tBookmark;

        procedure LocateIntegers;
          (* use a for-loop technique like that of LocateStrings *)

          var
            I, LeftBracket, RightBracket: integer;

          begin (* LocateIntegers *)
            if FirstInt > fLowInteger then
              LeftBracket := FirstInt
            else
              LeftBracket := fLowInteger;

            if LastInt < fHighInteger then
              RightBracket := LastInt
            else
              RightBracket := fHighInteger;

            with fDataLink.DataSet do
              for I := LeftBracket to RightBracket do
                if Locate(GetFieldName, I, []) then
                    TabsComing.Add(IntToStr(I))
          end   (* LocateIntegers *);

        procedure LocateStrings;
          (* look for records beginning with Prefix + ch *)

          var
            Ch: char;

          begin (* LocateStrings *)
            with fDataLink.DataSet do
              for Ch := chr(33) to chr(126) do
                if Ch <> STAR then
                  if Locate(GetFieldName, Prefix + Ch, [loPartialKey]) then
                    TabsComing.Add(Prefix + Ch)
          end   (* LocateStrings *);
          (* eject *)
        procedure ScanForIntegers;
          (* Nicer to do with query, but that would work with only some types of
           dataset *)

          begin (* ScanForIntegers *)
            with fDataLink.DataSet, TabsComing do
              if RecordCount > 0 then
                begin (* must look at every record *)
                  First;
                  while not EOF do
                    begin (* process one record *)
                      with fDataLink.Field do
                        if not IsNull then
                          if AsInteger >= FirstInt then
                            if AsInteger <= LastInt then
                              if IndexOf(AsString) < 0 then
                                Add(AsString);
                      Next
                    end   (* process one record *)
                end   (* must look at every record *)
          end   (* ScanForIntegers *);

        begin (* Gather *)
          with fDataLink.DataSet do
            begin (* with table *)
              SavedPlace := GetBookmark;
              try     (* use bookmark *)
                DisableControls;
                try     (* with controls disabled *)
                  case fIndexType of
                    ixInteger: if fHighInteger > fLowInteger then
                                 LocateIntegers
                               else
                                 ScanForIntegers;
                     ixString: LocateStrings
                    end (* case on fIndexType *)
                finally (* with controls disabled *)
                  EnableControls
                end     (* with controls disabled *);

                if BookmarkValid(SavedPlace) then
                  GotoBookmark(SavedPlace)
              finally (* use bookmark *)
                FreeBookmark(SavedPlace)
              end     (* use bookmark *)
            end   (* with table *)
        end   (* Gather *);
        (* eject *)
      procedure MoveThemIn;
        (* Move the (pruned) list from TabsComing to the actual tabs *)

        var
          I: integer;

        begin (* MoveThemIn *)
          with Tabs do
            begin (* with *)
              Clear;
              Add(STAR);
              if WantIn then
                Add('in');
              if WantOut then
                Add('out');
              for I := 1 to TabsComing.Count do
                Add(TabsComing[I-1])
            end   (* with *)
        end   (* MoveThemIn *);

      begin (* tRRFIndexTab.SetUpTabs *)
        if Usable then
          begin (* nondegenerate *)
            OnChange := nil;
            OldCursor := Screen.Cursor;
            Screen.Cursor := crHourGlass;
            TabsComing := tStringList.Create;
            try     (* must restore cursor, free TabsComing *)
              TabsComing.Sorted := false (* otherwise get '1000' before '2' *);
              WantIn := (length(Prefix) < MaxDepth-1);
              WantOut := (length(Prefix) > 0);
              if Assigned(fHeightStd) then
                Height := fHeightStd.Height;
              Gather;
              BeSelective;
              MoveThemIn;
              TabIndex := -1
            finally (* must restore cursor, free TabsComing *)
              FreeAndNil(TabsComing);
              Screen.Cursor := OldCursor
            end     (* must restore cursor, free TabsComing *);

            Arm
          end   (* nondegenerate *)
      end   (* tRRFIndexTab.SetUpTabs *);
      (* eject *)
    procedure tRRFIndexTab.TabClicked(Sender: tObject);
      (* dispatch on identity of clicked tab *)

      var
        CaptionClicked: string;
        SeekThis: variant;

      procedure GoDeeper;
        (* Shorten TabClicked *)

        begin (* GoDeeper *)
          case fIndexType of
            ixInteger: begin (* set brackets *)
                         Prefix := 'x';
                         FirstInt := StrToInt(PriorTab);
                         if NextTab <> nullstring then
                           LastInt := StrToInt(NextTab) - 1
                         else
                           LastInt := maxint
                       end   (* set brackets *);

             ixString: Prefix := PriorTab
            end (* case on IndexType *);

          SetUpTabs
        end   (* GoDeeper *);
        (* eject *)
      begin (* tRRFIndexTab.TabClicked *)
        if not Initialized then
          Initialize;
        if TabIndex >= 0 then
          begin (* nondegenerate *)
            CaptionClicked := Tabs[TabIndex];

            if Usable then
              if CaptionClicked = STAR then
                SetUpTabs

              else if CaptionClicked = 'in' then
                GoDeeper

              else if CaptionClicked = 'out' then
                begin (* decrease depth *)
                  case fIndexType of
                    ixInteger: begin (* relax brackets *)
                                 FirstInt := Low(integer);
                                 LastInt := maxint;
                                 Prefix := nullstring
                               end   (* relax brackets *);

                     ixString: Prefix := copy(Prefix, 1, length(Prefix)-1)
                    end (* case on IndexType *);

                  SetUpTabs
                end   (* decrease depth *)

              else
                begin (* general tab *)
                  PriorTab := CaptionClicked;
                  with fDataLink.DataSet do
                    begin (* with dataset *)
                      case fIndexType of
                        ixInteger: begin (* in means between this integer and next *)
                                     if TabIndex < Tabs.Count - 1 then
                                       NextTab := Tabs[TabIndex+1]
                                     else
                                       NextTab := nullstring;
                                     SeekThis := StrToInt(CaptionClicked)
                                   end   (* in means between this integer and next *);

                         ixString: SeekThis := CaptionClicked
                        end (* case on IndexType *);

                      if not Locate(GetFieldName, SeekThis, [loPartialKey]) then
                                     (* ignore it *)
                    end   (* with dataset *)
                end   (* general tab *)
          end   (* nondegenerate *)
      end   (* tRRFIndexTab.TabClicked *);
      (* eject *)
    function  tRRFIndexTab.Usable: boolean;
      (* Don't bother if links not in place *)

      begin (* tRRFIndexTab.Usable *)
        Result := Assigned(fDataLink)         and
                  Assigned(fDataLink.DataSet) and
                  fDataLink.DataSet.Active;

        case fIndexType of
          ixInteger: Result := Result and ((fDataLink.Field is tIntegerField) or
                                           (fDataLink.Field is tSmallIntField));
           ixString: Result := Result and (fDataLink.Field is tStringField)
          end (* case on IndexType *);
      end   (* tRRFIndexTab.Usable *);

end.
