unit IndexTab;
  (* The IndexTab component provides hot buttons for jumping through a dataset
  whose primary index is, or begins with, a stringfield.  The buttons are
  initially single letters, but after jumping to (say) the first record whose
  key field begins with 'M', the 'in' button turns the buttons into 'Ma', 'Mb',
  etc., and so on to arbitrary depth; the 'out' button has the opposite effect.
  Buttons are not generated unless corresponding records exist in the dataset.
  Also, when the IndexTab is too small to contain all of the buttons that the
  dataset might support, buttons are generated selectively.  The buttons can be
  regenerated by clicking the button whose caption is an asterisk.

    A Datasource leading to the dataset and the stringfield are identified by
  setting the
       property DataSource: tDataSource
       property DataField: string;
  properties, respectively (the DataField property is somewhat misnamed, since
  it is, as noted, a field name (a string), not a field).

    Because an IndexTab is often placed just to the side of a DBGrid or Panel,
  the
       property HeightStd: tControl;
  property is provided.  If this is assigned to another control C, then whenever
  the buttons of the IndexTab are regenerated, the IndexTab is resized to be the
  same height as C.

    The 'in' option will not be offered if the current button captions are not
  shorter than
       property MaxDepth: integer

    IndexTab components are not self-initializing.  Before the first use of
  an IndexTab, the
       procedure Initialize
  must be called, typically from the OnActivate method of the form on which the
  IndexTab has been placed. *)

  interface
    uses
      Classes  (* RegisterComponents *),
      ComCtrls (* tCustomTabControl, tTabStyle *),
      Controls (* tAnchorKind, tControl, tCreateParams *),
      DB       (* tDataSource, tLocateOption *),
      DBCtrls  (* tFieldDataLink *),
      Windows;

    type
      tRRFIndexTab = class(tCustomTabControl)
        private
          fHeightStd  (* optional; usually a DBgrid     *): tControl;
          fDataLink   (* help choose field              *): tFieldDataLink;
          fMaxDepth   (* maximum length of tab captions *),
          MyTabHeight (* take account of font size      *): integer;
          Prefix      (* length strictly < MaxDepth     *),
          PriorTab    (* will be new prefix if In       *): string;
          function  GetDataSource: tDataSource;
          function  GetFieldName: string;
          procedure SetDataSource(const Value: tDataSource);
          procedure SetFieldName (const Value: string);
          procedure SetUpTabs;
          function  Usable: boolean;
        protected
          procedure CreateWindowHandle(const Params: tCreateParams); override;
          procedure TabClicked(Sender: tObject);
        public
          constructor Create(AnOwner: tComponent); override;
          destructor  Destroy;                     override;
          procedure   Initialize;
        published
          property DataField: string       read GetFieldName  write SetFieldName
            (* 'DataField' is only name that will automatically invoke field
             editing in Object Inspector *);
          property DataSource: tDataSource read GetDataSource write SetDataSource;
          property HeightStd: tControl     read fHeightStd    write fHeightStd;
          property MaxDepth: integer       read fMaxDepth     write fMaxDepth;
          property Anchors;
        end (* tRRFIndexTab *);

    procedure Register;
    (* eject *)
  implementation
    uses
      DesignIntf (* RegisterPropertyInCategory *),
      Graphics   (* tFontStyle *);

    const
      nullstring = '';
      STAR       = '*';
      DefaultMaxDepth = 2;

    procedure Register;
      (* Install into palette *)

      begin (* Register *)
        RegisterComponents('RRF', [tRRFIndexTab]);
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'DataField');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'DataSource');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'HeightStd');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'MaxDepth');
        RegisterPropertyInCategory('RRF', tRRFIndexTab, 'Name')
      end   (* Register *);

{ tRRFIndexTab }

    constructor tRRFIndexTab.Create(AnOwner: tComponent);
      (* Initialize properties *)

      begin (* tRRFIndexTab.Create *)
        inherited Create(AnOwner);
        fMaxDepth := DefaultMaxDepth;
        fDataLink := tFieldDataLink.Create;
        fDataLink.Control := self
      end   (* tRRFIndexTab.Create *);

    procedure tRRFIndexTab.CreateWindowHandle(const Params: tCreateParams);
      (* Only now does this component have a parent, without which cannot
      touch subproperties *)

      begin (* tRRFIndexTab.CreateWindowHandle *)
        inherited CreateWindowHandle(Params);
        Constraints.MinHeight := 80;
        Height := 185;
        Width := 49;
        MultiLine := true;
        with Tabs do
          begin (* just for design *)
            Add(STAR);
            Add('in');
            Add('out');
            Add('b');
            Add('o');
            Add('g');
            Add('u');
            Add('s')
          end   (* just for design *);
        Anchors := [akBottom, akRight, akTop];
        Style := tsButtons;
        RaggedRight := false;
        ScrollOpposite := false;
        TabHeight := 20
      end   (* tRRFIndexTab.CreateWindowHandle *);
      (* eject *)
    destructor tRRFIndexTab.Destroy;
      (* reclaim storage *)

      begin (* tRRFIndexTab.Destroy *)
        fDataLink.Free;
        fDataLink := nil;
        inherited Destroy
      end   (* tRRFIndexTab.Destroy *);

    function tRRFIndexTab.GetDataSource: tDataSource;
      (* utilize the DataLink *)

      begin (* tRRFIndexTab.GetDataSource: tDataSource *)
        Result := fDataLink.DataSource
      end   (* tRRFIndexTab.GetDataSource: tDataSource *);

    function tRRFIndexTab.GetFieldName: string;
      (* utilize the DataLink *)

      begin (* tRRFIndexTab.GetFieldName *)
        Result := fDataLink.FieldName
      end   (* tRRFIndexTab.GetFieldName *);

    procedure tRRFIndexTab.Initialize;
      (* original setup *)

      const
        AscenderDescender = 'Wg';
        TabVMargin = 10;

      begin (* tRRFIndexTab.Initialize *)
        PriorTab := nullstring;
        Prefix := nullstring;
        ParentFont := false;
        Font.Style := Font.Style + [fsBold];
        MyTabHeight := Canvas.TextHeight(AscenderDescender) + TabVMargin;
        SetUpTabs
      end   (* tRRFIndexTab.Initialize *);

    procedure tRRFIndexTab.SetDataSource(const Value: tDataSource);
      (* for Object Inspector *)

      begin (* tRRFIndexTab.SetDataSource *)
        fDataLink.DataSource := Value
      end   (* tRRFIndexTab.SetDataSource *);

    procedure tRRFIndexTab.SetFieldName(const Value: string);
      (* for Object Inspector *)

      begin (* tRRFIndexTab.SetFieldName *)
        fDataLink.FieldName := Value
      end   (* tRRFIndexTab.SetFieldName *);
      (* eject *)
    procedure tRRFIndexTab.SetUpTabs;
      (* tab captions are dataset-dependent *)

      var
        WantIn, WantOut: boolean;
        TabLetters: string;

      procedure BeSelective;
        (* Use only as many tabs as will fit *)

        var
          I, KeptLetters, MaxLetterTabs, SeenLetters: integer;
          KeepFraction: real;

        begin (* BeSelective *)
          MaxLetterTabs := (Height div MyTabHeight) - 1 (* for star *);
          if WantIn then
            dec(MaxLetterTabs);
          if WantOut then
            dec(MaxLetterTabs);
          if MaxLetterTabs < 1 then
            MaxLetterTabs := 1;
          KeepFraction := MaxLetterTabs / length(TabLetters);
          KeptLetters := 1;
          SeenLetters := 1;
          for I := length(TabLetters) downto 2 do
            begin (* keep this letter? *)
              if KeptLetters/SeenLetters > KeepFraction then
                delete(Tabletters, I, 1)
              else
                inc(KeptLetters);
              inc(SeenLetters)
            end   (* keep this letter? *)
        end   (* BeSelective *);
        (* eject *)
      procedure Gather;
        (* See what letters are in use *)

        var
          SavedPlace: tBookmark;
          Ch: char;

        begin (* Gather *)
          with fDataLink.DataSet do
            begin (* with table *)
              SavedPlace := GetBookmark;
              try     (* use bookmark *)
                DisableControls;
                try     (* with controls disabled *)
                  TabLetters := nullstring;
                  for Ch := chr(33) to chr(126) do
                    if Ch <> STAR then
                      if Locate(GetFieldName, Prefix + Ch, [loPartialKey]) then
                        TabLetters := TabLetters + Ch
                finally (* with controls disabled *)
                  EnableControls
                end     (* with controls disabled *);
                if BookmarkValid(SavedPlace) then
                  GotoBookmark(SavedPlace)
              finally (* use bookmark *)
                FreeBookmark(SavedPlace)
              end     (* use bookmark *)
            end   (* with table *)
        end   (* Gather *);

      procedure SetTheTabs;
        (* set the captions of the tabs.  The last tab is duplicated because
        the TabControl sometimes ignores the last tab *)

        var
          I: integer;

        begin (* SetTheTabs *)
          with Tabs do
            begin (* with *)
              Clear;
              Add(STAR);
              if WantIn then
                Add('in');
              if WantOut then
                Add('out');
              for I := 1 to length(TabLetters) do
                Add(Prefix + TabLetters[I])
            end   (* with *)
        end   (* SetTheTabs *);
        (* eject *)
      begin (* tRRFIndexTab.SetUpTabs *)
        if Usable then
          begin (* nondegenerate *)
            OnChange := nil;
            WantIn := (length(Prefix) < MaxDepth-1);
            WantOut := (length(Prefix) > 0);
            if Assigned(fHeightStd) then
              Height := fHeightStd.Height;
            Gather;
            BeSelective;
            SetTheTabs;
            TabIndex := {Tabs.Count} - 1;
            OnChange := TabClicked
          end   (* nondegenerate *)
      end   (* tRRFIndexTab.SetUpTabs *);

    procedure tRRFIndexTab.TabClicked(Sender: tObject);
      (* dispatch on identity of clicked tab *)

      var
        CaptionClicked: string;

      begin (* tRRFIndexTab.TabClicked *)
        CaptionClicked := Tabs[TabIndex];

        if Usable then
          if CaptionClicked = STAR then
            SetUpTabs

          else if CaptionClicked = 'in' then
            begin (* increase depth *)
              Prefix := PriorTab;
              SetUpTabs
            end   (* increase depth *)

          else if CaptionClicked = 'out' then
            begin (* decrease depth *)
              Prefix := copy(Prefix, 1, length(Prefix)-1);
              SetUpTabs
            end   (* decrease depth *)

          else
            begin (* general tab *)
              PriorTab := CaptionClicked;
              with fDataLink.DataSet do
                if not Locate(GetFieldName, CaptionClicked, [loPartialKey]) then
                  (* ignore it *)
            end   (* general tab *)
      end   (* tRRFIndexTab.TabClicked *);

    function  tRRFIndexTab.Usable: boolean;
      (* Don't bother if links not in place *)

      begin (* tRRFIndexTab.Usable *)
        Result := Assigned(fDataLink)         and
                  Assigned(fDataLink.DataSet) and
                  fDataLink.DataSet.Active    and
                  Assigned(fDataLink.Field)   and
                  (fDataLink.Field is tStringField)
      end   (* tRRFIndexTab.Usable *);

end.
