{|
  Unit   : uEstSearchDialogTypes
  Datum  : 6-12-2003
  Auteur : Erik Stok
  Doel   : Collection en CollectionItem classes voor de zoekvelddefinities          
|}

unit uEstSearchDialogTypes;

interface

uses
  Classes, Controls, StdCtrls, DB, SysUtils, uEstSearchDialogConst;

type

  TEstSearchDialogField = class(TCollectionItem)
  private
    FFieldName: String;
    FWhereSyntax: String;
    FDisplayLabel: String;
    FDisplayWidth: Integer;
    FFieldType: TFieldType;
    FSearch: Boolean;
    FEmptyOperation: TSearchEmptyOperation;
    FDisplayFormat: String;
    FDisplayColumnWidth: Integer;
    FDefaultComparison: TSearchComparison;
    FSearchCase: TSearchCase;
  protected
    procedure AssignTo(Dest: TPersistent); override;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
  published
    property FieldName: String read FFieldName write FFieldName;
    property WhereSyntax: String read FWhereSyntax write FWhereSyntax;
    property DisplayLabel: String read FDisplayLabel write FDisplayLabel;
    property DisplayFormat: String read FDisplayFormat write FDisplayFormat;
    property DisplayWidth: Integer read FDisplayWidth write FDisplayWidth;
    property DisplayColumnWidth: Integer read FDisplayColumnWidth write FDisplayColumnWidth;
    property FieldType: TFieldType read FFieldType write FFieldType;
    property Search: Boolean read FSearch write FSearch;
    property EmptyOperation: TSearchEmptyOperation read FEmptyOperation write FEmptyOperation;
    property DefaultComparison: TSearchComparison read FDefaultComparison write FDefaultComparison;
    property SearchCase: TSearchCase read FSearchCase write FSearchCase;
  end;

  TEstSearchDialogFieldList = class(TCollection)
  private
    FOwner: TComponent;
  protected
    function GetOwner: TPersistent; override;
  public
    constructor Create(AOwner: TComponent);

    procedure Add(FieldName: String;
                  WhereSyntax: String;
                  DisplayLabel: String;
                  DisplayWidth: Integer;
                  DisplayFormat: String;
                  DisplayColumnWidth: Integer;
                  FieldType: TFieldType;
                  Search: Boolean;
                  DefaultComparison: TSearchComparison;
                  SearchCase: TSearchCase); virtual;

    procedure Delete(FieldName: String); overload; virtual;

    function IndexOf(FieldName: String): Integer; virtual;

    function ItemByFieldName(FieldName: String): TEstSearchDialogField;
  end;

  TEstSearchDialogStore = (dsNone, dsFields, dsFieldsAndValues);

  TEstSearchDialogSetSqlEvent = procedure(Sender: TObject; SQL: String) of object;

  TEstSearchDialogBeforeExecuteQueryEvent = procedure(Sender: TObject; SQL: String) of object;
  TEstSearchDialogAfterExecuteQueryEvent = procedure(Sender: TObject; SQL: String; DataSet: TDataSet) of object;

  TEstSearchDialogSetupEvent = procedure(Sender: TObject) of object;

  TEstSearchDialogInitControlsEvent = procedure(Sender: TObject;
                                                Field: TEstSearchDialogField;
                                                FieldControl: TComboBox;
                                                Comparison: TSearchComparison;
                                                ComparisonControl: TComboBox;
                                                ValueControl: TEdit;
                                                UsingOperator: Boolean;
                                                OperatorControl: TComboBox) of object;

  EEstSearchDialogException = class(Exception);

implementation

{|
  Procedure : SameText
  Auteur    : Erik Stok
  Doel      : Hoofdletterongevoelige vergelijking van 2 strings, overbodig voor
              Delphi 7
|}
function SameText(t1, t2: String): Boolean;
begin
 Result := AnsiUpperCase(t1) = AnsiUpperCase(t2);
end;

{ TEstSearchDialogField }

constructor TEstSearchDialogField.Create(Collection: TCollection);
begin
  inherited Create(Collection);

  // Initialiseer privates
  FFieldName := '';
  FWhereSyntax := '';
  FDisplayLabel := '';
  FDisplayFormat := '';
  FDisplayWidth := 0;
  FFieldType := ftUnknown;
  FDefaultComparison := scEqual;
  FDisplayColumnWidth := 0;
  FSearch := True;
  FEmptyOperation := eoNull;
  FSearchCase := scMixed;
end;

destructor TEstSearchDialogField.Destroy;
begin
  inherited;
end;

{|
  Procedure : TEstSearchDialogField.AssignTo
  Auteur    : Erik Stok
  Doel      : Assign deze TEstSearchDialogField aan een ander
|}
procedure TEstSearchDialogField.AssignTo(Dest: TPersistent);
var
  DestAsEstSearchDialogField: TEstSearchDialogField;
begin
  if Dest is TEstSearchDialogField then
  begin
    DestAsEstSearchDialogField := (Dest as TEstSearchDialogField);

    // Kopieer alle properties naar het dest object
    DestAsEstSearchDialogField.FieldName := FFieldName;
    DestAsEstSearchDialogField.WhereSyntax := FWhereSyntax;
    DestAsEstSearchDialogField.DisplayLabel := FDisplayLabel;
    DestAsEstSearchDialogField.DisplayWidth := FDisplayWidth;
    DestAsEstSearchDialogField.DisplayFormat := FDisplayFormat;
    DestAsEstSearchDialogField.FieldType := FFieldType;
    DestAsEstSearchDialogField.DefaultComparison := FDefaultComparison;
    DestAsEstSearchDialogField.DisplayColumnWidth := FDisplayColumnWidth;
    DestAsEstSearchDialogField.Search := FSearch;
    DestAsEstSearchDialogField.EmptyOperation := FEmptyOperation;
    DestAsEstSearchDialogField.SearchCase := FSearchCase;
  end
  else
  begin
    inherited AssignTo(Dest);
  end;
end;

{ TEstSearchDialogFieldList }

constructor TEstSearchDialogFieldList.Create(AOwner: TComponent);
begin
  inherited Create(TEstSearchDialogField);

  // Onthoud owner
  FOwner := AOwner;
end;

{|
  Procedure : TEstSearchDialogFieldList.GetOwner
  Auteur    : Erik Stok
  Doel      : Owner wordt anders teruggegeven vanwege alternatieve constructor
|}
function TEstSearchDialogFieldList.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

{|
  Procedure : TEstSearchDialogFieldList.Add
  Auteur    : Erik Stok
  Doel      : Voeg een nieuw veld toe aan de lijst volgens opgegeven parameters
|}
procedure TEstSearchDialogFieldList.Add(FieldName: String;
                                        WhereSyntax: String;
                                        DisplayLabel: String;
                                        DisplayWidth: Integer;
                                        DisplayFormat: String;
                                        DisplayColumnWidth: Integer;
                                        FieldType: TFieldType;
                                        Search: Boolean;
                                        DefaultComparison: TSearchComparison;
                                        SearchCase: TSearchCase);
var
  f : TEstSearchDialogField;
begin
  // Controleer of het betreffende veld niet reeds in de lijst aanwezig is
  if IndexOf(FieldName) <> -1 then
    raise Exception.Create('Duplicate searchdialog field');

  // Maak een nieuw searchdialog field aan
  f := (inherited Add) as TEstSearchDialogField;

  // Vul alle properties volgens opgegevens parameters
  f.FieldName := FieldName;
  f.WhereSyntax := WhereSyntax;
  f.DisplayLabel := DisplayLabel;
  f.DisplayWidth := DisplayWidth;
  f.DisplayFormat := DisplayFormat;
  f.DisplayColumnWidth := DisplayColumnWidth;
  f.FieldType := FieldType;
  f.Search := Search;
  f.DefaultComparison := DefaultComparison;
  f.SearchCase := SearchCase;
end;

{|
  Procedure : TEstSearchDialogFieldList.Delete
  Auteur    : Erik Stok
  Doel      : Gooi een bepaald veld uit de lijst
|}
procedure TEstSearchDialogFieldList.Delete(FieldName: String);
var
  o : TObject;
begin
  o := ItemByFieldName(FieldName);
  if Assigned(o) then
    o.Free
  else
    raise Exception.CreateFmt('Item with fieldname %s not found', [FieldName]);
end;

{|
  Procedure : TEstSearchDialogFieldList.IndexOf
  Auteur    : Erik Stok
  Doel      : Bepaal de index van een zeker veld in de lijst
|}
function TEstSearchDialogFieldList.IndexOf(FieldName: String): Integer;
var
  i : Integer;
begin
  // Standaard resultaat
  Result := -1;

  // Loop door de lijst
  for i := 0 to Count - 1 do
  begin

    // Als de veldnaam overeenkomt, geef dan de huidige index terug
    if SameText(TEstSearchDialogField(Items[i]).FieldName, FieldName) then
    begin
      Result := i;
      Break;
    end;

  end;
end;

{|
  Procedure : TEstSearchDialogFieldList.ItemByFieldName
  Auteur    : Erik Stok
  Doel      : Geef een specifiek item uit de lijst terug op basis van veldnaam
|}
function TEstSearchDialogFieldList.ItemByFieldName(FieldName: String): TEstSearchDialogField;
var
  i : Integer;
begin
  // Zoek item
  i := IndexOf(FieldName);

  // Indien gevonden, geef dan het item terug. Anders nil.
  if i <> -1 then
    Result := TEstSearchDialogField(Items[i])
  else
    Result := nil;
end;

end.

