{|
  Unit   : uEstSearchDialog
  Datum  : 26-12-2003
  Auteur : Erik Stok
  Doel   : Search dialog component
|}
unit uEstSearchDialog;

interface

uses
  Classes, SysUtils, Forms, StdCtrls, Controls, DB, TypInfo, uEstSearchDialogConst,
  uEstSearchDialogTypes, fEstSearchDialog, uEstSearchDialogIntl;

const
  REG_LOCATION = 'Software\ErikStok';

type
  TEstSearchDialog = class(TComponent)
  private
    FFrmSearchDialog: TFrmEstSearchDialog;

    FSearchSql: String;
    FCountSql: String;

    FMaxCriteria: Integer;
    FCriteriaCount: Integer;
    FMax: Integer;
    FSearchDialogFieldList: TEstSearchDialogFieldList;
    FSearchQuery: TStrings;
    FCountQuery: TStrings;
    FResultList: TStringList;
    FResultQuery: String;
    FRegistryPath: String;
    FDialogWidth: Integer;
    FDialogHeight: Integer;
    FDialogCaption: String;
    FStore: TEstSearchDialogStore;
    FQuoteChar: Char;
    FDecimalChar: Char;
    FCompareFormatDate: String;
    FCompareFormatDateTime: String;
    FCompareFormatTime: String;
    FQuotedDateTime: Boolean;
    FTrueExpression: String;
    FAdditionalWhere: String;
    FOnFilterRecord: TFilterRecordEvent;
    FOnBeforeExecuteCountQuery: TEstSearchDialogBeforeExecuteQueryEvent;
    FOnAfterExecuteCountQuery: TEstSearchDialogAfterExecuteQueryEvent;
    FOnBeforeExecuteSearchQuery: TEstSearchDialogBeforeExecuteQueryEvent;
    FOnAfterExecuteSearchQuery: TEstSearchDialogAfterExecuteQueryEvent;
    FOnInitControls: TEstSearchDialogInitControlsEvent;
    FOnSelect: TNotifyEvent;
    FStartOpen: Boolean;
    FSearchStyle: TSearchStyle;
    FOnClose: TNotifyEvent;
  protected
    procedure DoDialogSetup(Sender: TObject); virtual;

    procedure DoBeforeSearchQueryOpen(DataSet: TDataSet); virtual;
    procedure DoAfterSearchQueryOpen(DataSet: TDataSet); virtual;

    procedure DoBeforeCountQueryOpen(DataSet: TDataSet); virtual;
    procedure DoAfterCountQueryOpen(DataSet: TDataSet); virtual;

    procedure SetCriteriaCount(const Value: Integer); virtual;
    procedure SetMaxCriteria(const Value: Integer); virtual;
    procedure SetSearchDialogFieldList(const Value: TEstSearchDialogFieldList); virtual;

    function SearchDataSet: TDataSet; virtual; abstract;
    function CountDataSet: TDataSet; virtual; abstract;

    procedure DoSetSearchQuery(Sender: TObject; SQL: String); virtual;
    procedure DoSetCountQuery(Sender: TObject; SQL: String); virtual;

    function ValidConnection: Boolean; virtual; abstract;

    procedure DoOK(Sender: TObject); virtual;
    procedure DoCancel(Sender: TObject); virtual;

    procedure DoClose; virtual;
    procedure DoSelect; virtual;

    procedure DoInitControls(Sender: TObject;
                             Field: TEstSearchDialogField;
                             FieldControl: TComboBox;
                             Comparison: TSearchComparison;
                             ComparisonControl: TComboBox;
                             ValueControl: TEdit;
                             UsingOperator: Boolean;
                             OperatorControl: TComboBox); virtual;

  public

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure AssignTo(Dest: TPersistent); override;

    function Execute: Boolean; virtual;
    procedure Close; virtual;

    function ResultFieldAsString(FieldName: String): String;
    function ResultFieldAsInteger(FieldName: String): Integer;
    function ResultFieldAsFloat(FieldName: String): Double;
    function ResultFieldAsDateTime(FieldName: String): TDateTime;

    function ResultFieldExists(FieldName: String): Boolean;

    function FullRegistryPath: string; virtual;

    property ResultQuery: String read FResultQuery;

  published

    property SearchStyle: TSearchStyle read FSearchStyle write FSearchStyle default ssModal;

    property CriteriaCount: Integer read FCriteriaCount write SetCriteriaCount default 1;
    property MaxCriteria: Integer read FMaxCriteria write SetMaxCriteria default 1;

    property SearchQuery: TStrings read FSearchQuery write FSearchQuery;
    property CountQuery: TStrings read FCountQuery write FCountQuery;
    property Max: Integer read FMax write FMax default 0;

    property SearchDialogFieldList: TEstSearchDialogFieldList read FSearchDialogFieldList
                                                              write FSearchDialogFieldList;

    property RegistryPath: string read FRegistryPath write FRegistryPath;

    property DialogCaption: String read FDialogCaption write FDialogCaption;
    property DialogWidth: Integer read FDialogWidth write FDialogWidth;
    property DialogHeight: Integer read FDialogHeight write FDialogHeight;

    property Store: TestSearchDialogStore read FStore write FStore;

    property CompareFormatDate: String read FCompareFormatDate write FCompareFormatDate;
    property CompareFormatTime: String read FCompareFormatTime write FCompareFormatTime;
    property CompareFormatDateTime: String read FCompareFormatDateTime write FCompareFormatDateTime;
    property QuotedDateTime: Boolean read FQuotedDateTime write FQuotedDateTime default True;

    property QuoteChar: Char read FQuoteChar write FQuoteChar default '''';
    property DecimalChar: Char read FDecimalChar write FDecimalChar default '.';

    property TrueExpression: String read FTrueExpression write FTrueExpression;
    property AdditionalWhere: String read FAdditionalWhere write FAdditionalWhere;
    property StartOpen: Boolean read FStartOpen write FStartOpen default False;

    property OnFilterRecord: TFilterRecordEvent read FOnFilterRecord write FOnFilterRecord;

    property OnBeforeExecuteSearchQuery: TEstSearchDialogBeforeExecuteQueryEvent
                                           read FOnBeforeExecuteSearchQuery
                                           write FOnBeforeExecuteSearchQuery;
    property OnAfterExecuteSearchQuery: TEstSearchDialogAfterExecuteQueryEvent
                                          read FOnAfterExecuteSearchQuery
                                          write FOnAfterExecuteSearchQuery;

    property OnBeforeExecuteCountQuery: TEstSearchDialogBeforeExecuteQueryEvent
                                           read FOnBeforeExecuteCountQuery
                                           write FOnBeforeExecuteCountQuery;
    property OnAfterExecuteCountQuery: TEstSearchDialogAfterExecuteQueryEvent
                                          read FOnAfterExecuteCountQuery
                                          write FOnAfterExecuteCountQuery;

    property OnInitControls: TEstSearchDialogInitControlsEvent read FOnInitControls
                                                               write FOnInitControls;

    property OnSelect: TNotifyEvent read FOnSelect write FOnSelect;
    property OnClose: TNotifyEvent read FOnClose write FOnClose;
  end;

  procedure RegisterSearchDialogFormClass(FrmEstSearchDialogClass: TFrmEstSearchDialogClass);

implementation

var
  InternalFrmEstSearchDialogClass: TFrmEstSearchDialogClass;

{|
  Procedure : RegisterSearchDialogFormClass
  Auteur    : Erik Stok
  Doel      : Geef een eigen TFrmEstSearchDialog afgeleide aan als te gebruiken
              form voor het search dialog
|}
procedure RegisterSearchDialogFormClass(FrmEstSearchDialogClass: TFrmEstSearchDialogClass);
begin
  InternalFrmEstSearchDialogClass := FrmEstSearchDialogClass;
end;

{ TEstSearchDialog }

constructor TEstSearchDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  // Init privates
  FFrmSearchDialog := nil;
  
  FSearchSql := '';
  FCountSql := '';

  FMaxCriteria := 1;
  FCriteriaCount := 1;
  FMax := 0;
  FResultQuery := '';

  FSearchDialogFieldList := TEstSearchDialogFieldList.Create(Self);

  FSearchQuery := TStringList.Create;
  FCountQuery := TStringList.Create;
  FResultList := TStringList.Create;

  FStore := dsFields;

  FRegistryPath := '';

  FDialogCaption := EstSearchDialogIntl.CaptionDialog;
  FDialogWidth := 0;
  FDialogHeight := 0;

  FCompareFormatDate := 'YYYYMMDD';
  FCompareFormatTime := 'HHNNSS';
  FCompareFormatDateTime := 'YYYYMMDD';
  FQuotedDateTime := True;

  FQuoteChar := '''';
  FDecimalChar := '.';

  FTrueExpression := '1=1';
  FAdditionalWhere := '';
  FStartOpen := False;
  FSearchStyle := ssModal;

  FOnFilterRecord := nil;
  FOnBeforeExecuteCountQuery := nil;
  FOnAfterExecuteCountQuery := nil;
  FOnAfterExecuteSearchQuery := nil;
  FOnBeforeExecuteSearchQuery := nil;
  FOnInitControls := nil;
  FOnSelect := nil;
  FOnClose := nil;

end;

destructor TEstSearchDialog.Destroy;
begin
  // Ruim privates op
  FSearchDialogFieldList.Free;

  FSearchQuery.Free;
  FCountQuery.Free;
  FResultList.Free;

  inherited Destroy;
end;

procedure TEstSearchDialog.SetSearchDialogFieldList(const Value: TEstSearchDialogFieldList);
begin
  FSearchDialogFieldList := Value;
end;

{|
  Procedure : TEstSearchDialog.DoDialogSetup
  Auteur    : Erik Stok
  Doel      : Stel het dialog in volgens de properties van het component
|}
procedure TEstSearchDialog.DoDialogSetup(Sender: TObject);
var
  Dialog : TfrmEstSearchDialog;
begin
  // Cast naar een TfrmEstSearchDialog
  Dialog := (Sender as TfrmEstSearchDialog);

  // Stel het form in volgens de properties van het component
  Dialog.SearchStyle := FSearchStyle;

  Dialog.Caption := FDialogCaption;
  if FDialogWidth <> 0 then
    Dialog.Width := FDialogWidth;
  if FDialogHeight <> 0 then
    Dialog.Height := FDialogHeight;

  Dialog.SearchDataSet := SearchDataSet;
  Dialog.CountDataSet := CountDataSet;

  Dialog.SearchQuery := FSearchQuery.Text;
  Dialog.CountQuery := FCountQuery.Text;

  Dialog.MaxCriteria := FMaxCriteria;
  Dialog.CriteriaCount := FCriteriaCount;
  Dialog.Max := FMax;

  Dialog.RegistryPath := FullRegistryPath;
  Dialog.Store := FStore;

  Dialog.CompareFormatDate := FCompareFormatDate;
  Dialog.CompareFormatTime := FCompareFormatTime;
  Dialog.CompareFormatDateTime := FCompareFormatDateTime;
  Dialog.QuotedDateTime := FQuotedDateTime;

  Dialog.QuoteChar := FQuoteChar;
  Dialog.DecimalChar := FDecimalChar;

  Dialog.SearchDialogFieldList := FSearchDialogFieldList;

  Dialog.AdditionalWhere := FAdditionalWhere;
  Dialog.StartOpen := FStartOpen;

  Dialog.OnSetSearchQuery := DoSetSearchQuery;
  Dialog.OnSetCountQuery := DoSetCountQuery;
end;

{|
  Procedure : TEstSearchDialog.Execute
  Auteur    : Erik Stok
  Doel      : Het daadwerkelijk uitvoeren van de zoekopdracht. Dus het aanmaken
              van het dialog volgens de instellingen van het component en het
              afhandelen van de selectie van de gebruiker in dat dialog.
|}
function TEstSearchDialog.Execute: Boolean;
var
  i : Integer;
  SearchFieldAvailable : Boolean;
begin
  if not ValidConnection then
    raise EEstSearchDialogException.Create('Invalid connection');

  // Als er al een dialog operationeel is, activeer dat dan. Stel anders een
  // heel nieuw dialog op.
  if Assigned(FFrmSearchDialog) then
  begin

    // Breng bestaand dialog naar voren
    FFrmSearchDialog.BringToFront;

    // Er is geen resultaat bepaald
    Result := False;

  end
  else
  begin

    // Controleer of er velden zijn waarop gezocht kan worden
    SearchFieldAvailable := False;
    for i := 0 to FSearchDialogFieldList.Count - 1 do
    begin

      if TEstSearchDialogField(FSearchDialogFieldList.Items[i]).Search then
      begin
        SearchFieldAvailable := True;
        Break;
      end;

    end;

    if not SearchFieldAvailable then
      raise EEstSearchDialogException.Create('No search fields are defined for the search dialog');

    // Controleer of er een zoek query is
    if Trim(FSearchQuery.Text) = '' then
      raise EEstSearchDialogException.Create('No search query is defined for the search dialog');

    // Controleer of er een count query is indien nodig
    if (Trim(FSearchQuery.Text) = '') and (FMax > 0) then
      raise EEstSearchDialogException.Create('No count query is defined for the search dialog');

    if not (csDesigning in ComponentState) then
    begin
      // Koppel filterrecord event aan de searchquery
      SearchDataSet.OnFilterRecord := FOnFilterRecord;
      SearchDataSet.Filter := '';
      SearchDataSet.Filtered := True;
    end;

    // Koppel events aan datasets
    SearchDataSet.BeforeOpen := DoBeforeSearchQueryOpen;
    SearchDataSet.AfterOpen := DoAfterSearchQueryOpen;

    CountDataSet.BeforeOpen := DoBeforeCountQueryOpen;
    CountDataSet.AfterOpen := DoAfterCountQueryOpen;

    // Wis eventueel vorige resultaat
    FResultList.Clear;
    FResultQuery := '';

    // Toon scherm
    if (FSearchStyle <> ssMDIChild) or (csDesigning in ComponentState) then
      FFrmSearchDialog := InternalFrmEstSearchDialogClass.Create(Self, False, DoDialogSetup)
    else
      FFrmSearchDialog := InternalFrmEstSearchDialogClass.Create(Self, True, DoDialogSetup);

    // Koppel initcontrol event aan het dialog
    FFrmSearchDialog.OnInitControls := DoInitControls;

    // Als het form modaal is, handel het dan als dusdanig af
    if (FSearchStyle = ssModal) or (csDesigning in ComponentState) then
    begin

      try
        // Aleen als er OK gegeven is wordt het resultaat intern opgeslagen
        if FFrmSearchDialog.ShowModal = mrOK then
        begin
          FResultList.Text := FFrmSearchDialog.GetResultList;
          FResultQuery := FFrmSearchDialog.GetResultQuery;
          DoClose;
          Result := True;
        end
        else
        begin
          DoClose;
          Result := False;
        end;

      finally
        FFrmSearchDialog.Free;
        FFrmSearchDialog := nil;
      end;

    end
    else
    begin
      // Koppel extra events
      FFrmSearchDialog.OnOK := DoOK;
      FFrmSearchDialog.OnCancel := DoCancel;

      // Toon dialog
      FFrmSearchDialog.Show;

      // Geef altijd False terug voor non-modal executes, er is immers (nog) niets
      // geselecteerd
      Result := False;
    end;

  end;

end;

{|
  Procedure : TEstSearchDialog.Assign
  Auteur    : Erik Stok
  Doel      : Assign een ander TEstSearchDialog aan deze
|}
procedure TEstSearchDialog.AssignTo(Dest: TPersistent);
var
  DestSearchDialog : TEstSearchDialog;
begin
  DestSearchDialog := Dest as TEstSearchDialog;

  DestSearchDialog.CriteriaCount := FCriteriaCount;
  DestSearchDialog.MaxCriteria := FMaxCriteria;

  DestSearchDialog.SearchQuery.Text := FSearchQuery.Text;
  DestSearchDialog.CountQuery.Text := FCountQuery.Text;
  DestSearchDialog.Max := FMax;

  DestSearchDialog.RegistryPath := FRegistryPath;

  DestSearchDialog.DialogCaption := FDialogCaption;
  DestSearchDialog.DialogWidth := FDialogWidth;
  DestSearchDialog.DialogHeight := FDialogHeight;
  DestSearchDialog.Store := FStore;

  DestSearchDialog.CompareFormatDate := FCompareFormatDate;
  DestSearchDialog.CompareFormatTime := FCompareFormatTime;
  DestSearchDialog.CompareFormatDateTime := FCompareFormatDateTime;
  DestSearchDialog.QuotedDateTime := FQuotedDateTime;

  DestSearchDialog.QuoteChar := FQuoteChar;
  DestSearchDialog.DecimalChar := FDecimalChar;

  DestSearchDialog.SearchStyle := FSearchStyle;

  DestSearchDialog.TrueExpression := FTrueExpression;
  DestSearchDialog.AdditionalWhere := FAdditionalWhere;
  DestSearchDialog.StartOpen := FStartOpen;

  DestSearchDialog.SearchDialogFieldList.Assign(FSearchDialogFieldList);

  DestSearchDialog.OnFilterRecord := FOnFilterRecord;
  DestSearchDialog.OnBeforeExecuteSearchQuery := FOnBeforeExecuteSearchQuery;
  DestSearchDialog.OnAfterExecuteSearchQuery := FOnAfterExecuteSearchQuery;

  DestSearchDialog.OnBeforeExecuteCountQuery := FOnBeforeExecuteCountQuery;
  DestSearchDialog.OnAfterExecuteCountQuery := FOnAfterExecuteCountQuery;

  DestSearchDialog.OnInitControls := FOnInitControls;

  DestSearchDialog.OnSelect := FOnSelect;
  DestSearchDialog.OnClose := FOnClose;

end;

{|
  Procedure : TEstSearchDialog.SetCriteriaCount
  Auteur    : Erik Stok
  Doel      : Setter van CriteriaCount property voor extra validatie
|}
procedure TEstSearchDialog.SetCriteriaCount(const Value: Integer);
begin
  // Waarde mag niet lager dan 1 zijn
  if Value < 1 then
    FCriteriaCount := 1
  else
    FCriteriaCount := Value;

  // Maximum aantal criteria wordt automatisch aangepast indien dit niet voldoet
  if FMaxCriteria < FCriteriaCount then
    MaxCriteria := FCriteriaCount;
end;

{|
  Procedure : TEstSearchDialog.SetMaxCriteria
  Auteur    : Erik Stok
  Doel      : Setter van MaxCriteria property voor extra validatie
|}
procedure TEstSearchDialog.SetMaxCriteria(const Value: Integer);
begin
  // Waarde mag niet lager dan 1 zijn
  if Value < 1 then
    FMaxCriteria := 1
  else
    FMaxCriteria := Value;


  // Aantal criteria wordt automatisch aangepast indien dit niet voldoet
  if FCriteriaCount > FMaxCriteria then
    CriteriaCount := FMaxCriteria;
end;

{|
  Procedure : TEstSearchDialog.ResultFieldAsString
  Auteur    : Erik Stok
  Doel      : Resultaatveldbenadering
|}
function TEstSearchDialog.ResultFieldAsString(FieldName: String): String;
begin
  if FResultList.IndexOfName(FieldName) <> -1 then
    Result := FResultList.Values[FieldName]
  else
    raise EEstSearchDialogException.CreateFmt('Field %s is not a searchdialog result field', [FieldName]);
end;

{|
  Procedure : TEstSearchDialog.ResultFieldAsInteger
  Auteur    : Erik Stok
  Doel      : Resultaatveldbenadering
|}
function TEstSearchDialog.ResultFieldAsInteger(FieldName: String): Integer;
begin
  Result := StrToInt(ResultFieldAsString(FieldName));
end;

{|
  Procedure : TEstSearchDialog.ResultFieldAsFloat
  Auteur    : Erik Stok
  Doel      : Resultaatveldbenadering
|}
function TEstSearchDialog.ResultFieldAsFloat(FieldName: String): Double;
begin
  Result := StrToFloat(ResultFieldAsString(FieldName));
end;

{|
  Procedure : TEstSearchDialog.ResultFieldAsDateTime
  Auteur    : Erik Stok
  Doel      : Resultaatveldbenadering
|}
function TEstSearchDialog.ResultFieldAsDateTime(FieldName: String): TDateTime;
begin
  Result := StrToDateTime(ResultFieldAsString(FieldName));
end;

{|
  Procedure : TEstSearchDialog.ResultFieldExists
  Auteur    : Erik Stok
  Doel      : Controleer of een bepaald veld in het zoekresultaat aanwezig is
|}

function TEstSearchDialog.ResultFieldExists(FieldName: String): Boolean;
begin
  Result := FResultList.IndexOfName(FieldName) > -1; {Door: Frank van Boven}
end;

{|
  Procedure : TEstSearchDialog.FullRegistryPath
  Auteur    : Erik Stok
  Doel      : Bepaal het volldige registry pad waar instellingen worden
              opgeslagen. Dit pad worden bepaald door de instelling van de
              RegistryPath property.
|}
function TEstSearchDialog.FullRegistryPath: string;
var
  OwnerName : String;
begin
  // Bepaal de naam van de owner. Als die er niet is, sla dan op onder application
  if Assigned(Owner) then
    OwnerName := Owner.ClassName
  else
    OwnerName := Application.ClassName;

  // Als er geen specifiek registry path gegevens is, neem dan het standaard pad
  if FRegistryPath <> '' then
    Result := Format('%s\%s\%s', [FRegistryPath, OwnerName, Name])
  else
    Result := Format('%s\%s\LastSearch\%s\%s',
      [REG_LOCATION, ChangeFileExt(ExtractFileName(Application.ExeName), ''),
        OwnerName, Name]);
end;

{|
  Procedure : TEstSearchDialog.DoBeforeSearchQueryOpen
  Auteur    : Erik Stok
  Doel      : Roep OnBeforeExecuteSearchQuery event handler aan
|}
procedure TEstSearchDialog.DoBeforeSearchQueryOpen(DataSet: TDataSet);
begin
  if Assigned(FOnBeforeExecuteSearchQuery) and (not (csDesigning in ComponentState)) then
    FOnBeforeExecuteSearchQuery(Self, FSearchSql);
end;

{|
  Procedure : TEstSearchDialog.DoAfterSearchQueryOpen
  Auteur    : Erik Stok
  Doel      : Roep OnAfterExecuteSearchQuery event handler aan
|}
procedure TEstSearchDialog.DoAfterSearchQueryOpen(DataSet: TDataSet);
begin
  if Assigned(FOnAfterExecuteSearchQuery) and (not (csDesigning in ComponentState)) then
    FOnAfterExecuteSearchQuery(Self, FSearchSql, SearchDataSet);
end;

{|
  Procedure : TEstSearchDialog.DoBeforeCountQueryOpen
  Auteur    : Erik Stok
  Doel      : Roep OnBeforeExecuteCountQuery event handler aan
|}
procedure TEstSearchDialog.DoBeforeCountQueryOpen(DataSet: TDataSet);
begin
  if Assigned(FOnBeforeExecuteCountQuery) and (not (csDesigning in ComponentState)) then
    FOnBeforeExecuteCountQuery(Self, FCountSql);
end;

{|
  Procedure : TEstSearchDialog.DoAfterCountQueryOpen
  Auteur    : Erik Stok
  Doel      : Roep OnAfterExecuteCountQuery event handler aan
|}
procedure TEstSearchDialog.DoAfterCountQueryOpen(DataSet: TDataSet);
begin
  if Assigned(FOnAfterExecuteCountQuery) and (not (csDesigning in ComponentState)) then
    FOnAfterExecuteCountQuery(Self, FCountSql, CountDataSet);
end;

{|
  Procedure : TEstSearchDialog.DoSetCountQuery
  Auteur    : Erik Stok
  Doel      : Stel de zoek query in. Afgeleiden moeten een override maken van
              deze method om de database specifieke componenten in te stellen.
|}
procedure TEstSearchDialog.DoSetCountQuery(Sender: TObject; SQL: String);
begin
  FCountSql := SQL;
end;

{|
  Procedure : TEstSearchDialog.DoSetCountQuery
  Auteur    : Erik Stok
  Doel      : Stel de zoek query in. Afgeleiden moeten een override maken van
              deze method om de database specifieke componenten in te stellen.
|}
procedure TEstSearchDialog.DoSetSearchQuery(Sender: TObject; SQL: String);
begin
  FSearchSql := SQL;
end;

{|
  Procedure : TEstSearchDialog.DoOK
  Auteur    : Erik Stok
  Doel      : Handel OK af van Normal of MDIChild search dialog
|}
procedure TEstSearchDialog.DoOK(Sender: TObject);
begin
  if not (csDesigning in ComponentState) then
  begin

    // Stel huidig resultaat in
    FResultList.Text := (Sender as TFrmEstSearchDialog).GetResultList;
    FResultQuery := (Sender as TFrmEstSearchDialog).GetResultQuery;

    // Execute eventueel event
    DoSelect;

  end;
end;

{|
  Procedure : TEstSearchDialog.DoCancel
  Auteur    : Erik Stok
  Doel      : Handel Cancel van dialog af
|}
procedure TEstSearchDialog.DoCancel(Sender: TObject);
begin
  if not (csDesigning in ComponentState) then
    DoClose;
end;

{|
  Procedure : TEstSearchDialog.DoClose
  Auteur    : Erik Stok
  Doel      : Handel OnClose event af
|}
procedure TEstSearchDialog.DoClose;
begin
  if not (csDesigning in ComponentState) then
  begin

    if Assigned(FOnClose) then
      FOnClose(Self);

    // Wis form referentie voor niet modale dialogs
    if FSearchStyle <> ssModal then
      FFrmSearchDialog := nil;

  end;

  // Sluit datasets
  SearchDataSet.Close;
  CountDataSet.Close;
end;

{|
  Procedure : TEstSearchDialog.DoSelect
  Auteur    : Erik Stok
  Doel      : Handel OnSelect event af
|}
procedure TEstSearchDialog.DoSelect;
begin
  if not (csDesigning in ComponentState) then
  begin

    // Roep event handler aan indien gekoppeld
    if Assigned(FOnSelect) then
      FOnSelect(Self);

  end;
end;

{|
  Procedure : TEstSearchDialog.Close
  Auteur    : Erik Stok
  Doel      : Sluit eventuele niet modale dialogs
|}
procedure TEstSearchDialog.Close;
begin
  if Assigned(FFrmSearchDialog) then
    FFrmSearchDialog.Close;
end;

{
  Procedure : TEstSearchDialog.DoInitControls
  Auteur    : Erik Stok
  Doel      : Initlialiseer controls indien er een event handler
              gekoppeld is
}
procedure TEstSearchDialog.DoInitControls(Sender: TObject;
                                          Field: TEstSearchDialogField;
                                          FieldControl: TComboBox;
                                          Comparison: TSearchComparison;
                                          ComparisonControl: TComboBox;
                                          ValueControl: TEdit;
                                          UsingOperator: Boolean;
                                          OperatorControl: TComboBox);
begin
  // Als de init controls event handler is gekoppeld, roep die dan aan
  // met de betreffende controls
  if Assigned(FOnInitControls) then
    FOnInitControls(Self, Field, FieldControl, Comparison, ComparisonControl,
      ValueControl, UsingOperator, OperatorControl);
end;

initialization
  // Stel form classes op de standaard forms in
  InternalFrmEstSearchDialogClass := TFrmEstSearchDialog;

end.
