{|
  Unit    : fEstSearchDialogEditor
  Datum   : 26-12-2003
  Auteur  : Erik Stok
  Doel    : ErikStok search dialog designtime editor
|}
unit fEstSearchDialogEditor;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, ActnList, Db, Grids, DBGrids, ComCtrls, TypInfo,
  uEstSearchDialog, uEstSearchDialogTypes, fEstSearchDialogEditorFieldGeneration,
  uEstSearchDialogConst, fEstSearchDialog;

type
  TEstSearchDialogClass = class of TEstSearchDialog;

  TFrmEstSearchDialogEditor = class(TForm)
    aclSearch: TActionList;
    actOK: TAction;
    actCancel: TAction;
    pnlMain: TPanel;
    pnlButtons: TPanel;
    btnOK: TButton;
    btnCancel: TButton;
    pnlSettings: TPanel;
    pgtSettings: TPageControl;
    tbsGeneral: TTabSheet;
    tbsCount: TTabSheet;
    pnlQuery: TPanel;
    pnlFields: TPanel;
    pnlFieldSettings: TPanel;
    mmoSearchQuery: TMemo;
    dtsSearch: TDataSource;
    grdSearch: TDBGrid;
    pnlQuerySeperator: TPanel;
    btnQuery: TButton;
    actQuery: TAction;
    lbxFields: TListBox;
    lblQUery: TLabel;
    lblResult: TLabel;
    lblFields: TLabel;
    lblFieldSettings: TLabel;
    pnlFieldButtons: TPanel;
    btnDelete: TButton;
    btnAdd: TButton;
    lblFieldName: TLabel;
    edtFieldname: TEdit;
    lblWhereSyntax: TLabel;
    edtWhereSyntax: TEdit;
    lblDisplayLabel: TLabel;
    edtDisplayLabel: TEdit;
    lblDisplayWidth: TLabel;
    edtDisplayWidth: TEdit;
    lblFieldType: TLabel;
    cbxFieldType: TComboBox;
    btnGenerate: TButton;
    actGenerate: TAction;
    actAdd: TAction;
    actDelete: TAction;
    btnTest: TButton;
    actTest: TAction;
    Panel1: TPanel;
    lblCountQuery: TLabel;
    lblCountResult: TLabel;
    mmoCountQuery: TMemo;
    grdCount: TDBGrid;
    Panel2: TPanel;
    btnCountQuery: TButton;
    dtsCount: TDataSource;
    actCountQuery: TAction;
    actPriorField: TAction;
    actNextField: TAction;
    btnMoveUp: TButton;
    btnMoveDown: TButton;
    actMoveUp: TAction;
    actMoveDown: TAction;
    lblWhenEmpty: TLabel;
    cbxEmptyOperation: TComboBox;
    lblDisplayFormat: TLabel;
    edtDisplayFormat: TEdit;
    lblDefaultComparison: TLabel;
    cbxDefaultComparison: TComboBox;
    lblSearch: TLabel;
    cbxSearch: TComboBox;
    lblColumnWidth: TLabel;
    edtColumnWidth: TEdit;
    lblSearchCase: TLabel;
    cbxSearchCase: TComboBox;
    procedure FormDestroy(Sender: TObject);
    procedure lbxFieldsClick(Sender: TObject);
    procedure actQueryExecute(Sender: TObject);
    procedure actGenerateExecute(Sender: TObject);
    procedure actAddExecute(Sender: TObject);
    procedure actDeleteExecute(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure edtFieldnameExit(Sender: TObject);
    procedure edtFieldnameKeyPress(Sender: TObject; var Key: Char);
    procedure edtWhereSyntaxExit(Sender: TObject);
    procedure edtWhereSyntaxKeyPress(Sender: TObject; var Key: Char);
    procedure edtDisplayLabelExit(Sender: TObject);
    procedure edtDisplayLabelKeyPress(Sender: TObject; var Key: Char);
    procedure edtDisplayWidthExit(Sender: TObject);
    procedure edtDisplayWidthKeyPress(Sender: TObject; var Key: Char);
    procedure edtColumnWidthExit(Sender: TObject);
    procedure edtColumnWidthKeyPress(Sender: TObject; var Key: Char);
    procedure cbxFieldTypeChange(Sender: TObject);
    procedure actOKExecute(Sender: TObject);
    procedure actCancelExecute(Sender: TObject);
    procedure actTestExecute(Sender: TObject);
    procedure actCountQueryExecute(Sender: TObject);
    procedure actPriorFieldExecute(Sender: TObject);
    procedure actNextFieldExecute(Sender: TObject);
    procedure actMoveUpExecute(Sender: TObject);
    procedure actMoveDownExecute(Sender: TObject);
    procedure cbxEmptyOperationChange(Sender: TObject);
    procedure edtDisplayFormatExit(Sender: TObject);
    procedure edtDisplayFormatKeyPress(Sender: TObject; var Key: Char);
    procedure cbxDefaultComparisonChange(Sender: TObject);
    procedure cbxSearchCaseChange(Sender: TObject);
    procedure cbxSearchChange(Sender: TObject);
  private
    FSearchDialog : TEstSearchDialog;
    FUpdatingControls: Boolean;
    FSearchDialogClass: TEstSearchDialogClass;

    procedure ClearFieldListBox;
    function AddField(FieldName: String;
                      DisplayLabel: String;
                      WhereSyntax: String;
                      DisplayWidth: Integer;
                      DisplayFormat: String;
                      DisplayColumnWidth: Integer;
                      FieldType: TFieldType;
                      EmptyOperation: TSearchEmptyOperation;
                      Search: Boolean;
                      DefaultComparison: TSearchComparison;
                      SearchCase: TSearchCase): Integer;
    procedure SetFieldName;
    procedure SetWhereSyntax;
    procedure SetDisplayLabel;
    procedure SetDisplayWidth;
    procedure SetColumnWidth;
    procedure SetDisplayFormat;

    procedure ValidateSearchQuery;
    procedure ValidateCountQuery;
    procedure ValidateDefaultSearch;
  protected
    procedure UpdateSearchDialog(const ASearchDialog: TEstSearchDialog); virtual;

    procedure SetSearchDialog(const Value: TEstSearchDialog); virtual;

    procedure SelectField; virtual;

    function SearchDataSet: TDataSet; virtual; abstract;
    function CountDataSet: TDataSet; virtual; abstract;
    procedure SetSearchSQL(SQL: String); virtual; abstract;
    procedure SetCountSQL(SQL: String); virtual; abstract;
  public
    property SearchDialog : TEstSearchDialog read FSearchDialog write SetSearchDialog;
    property SearchDialogClass : TEstSearchDialogClass read FSearchDialogClass write FSearchDialogClass;
  end;

implementation

{$R *.DFM}

{ TFrmCblSearchDialogEditor }

procedure TFrmEstSearchDialogEditor.SetSearchDialog(const Value: TEstSearchDialog);
var
  i : Integer;
begin
  // Onthoud origineel component (voor terugzetten properties bij OK)
  FSearchDialog := Value;

  // Zet properties in editor scherm
  mmoSearchQuery.Lines.Text := FSearchDialog.SearchQuery.Text;
  mmoCountQuery.Lines.Text := FSearchDialog.CountQuery.Text;

  for i := 0 to FSearchDialog.SearchDialogFieldList.Count - 1 do
  begin
    with FSearchDialog.SearchDialogFieldList.Items[i] as TEstSearchDialogField do
      AddField(FieldName, DisplayLabel, WhereSyntax, DisplayWidth, DisplayFormat,
                 DisplayColumnWidth, FieldType, EmptyOperation, Search,
                 DefaultComparison, SearchCase);
  end;

  if lbxFields.Items.Count > 0 then
    lbxFields.ItemIndex := 0;

  SelectField;

end;

{|
  Procedure : TFrmCblSearchDialogEditor.ClearFieldListBox
  Auteur    : Erik Stok
  Doel      : Ruim fields listbox op (en alle eventueel aan items gekoppelde objecten)
|}
procedure TFrmEstSearchDialogEditor.ClearFieldListBox;
begin
  // Disconnect alle events die problemen kunnen veroorzaken
  lbxFields.OnClick := nil;

  edtFieldname.OnExit := nil;
  edtWhereSyntax.OnExit := nil;
  edtDisplayLabel.OnExit := nil;
  edtDisplayWidth.OnExit := nil;
  edtDisplayFormat.OnExit := nil;
  cbxFieldType.OnChange := nil;
  cbxEmptyOperation.OnChange := nil;
  cbxDefaultComparison.OnChange := nil;
  cbxSearch.OnChange := nil; 

  // Zolang er op te ruimen items zijn, ruim op
  while lbxFields.Items.Count > 0 do
  begin
    // Als er een gekoppeld object is, ruim dat dan op
    if Assigned(lbxFields.Items.Objects[0]) then
      lbxFields.Items.Objects[0].Free;

    lbxFields.Items.Delete(0);
  end;
end;

{|
  Procedure : TFrmCblSearchDialogEditor.AddField
  Auteur    : Erik Stok
  Doel      : Voeg veld toe aan listbox
|}
function TFrmEstSearchDialogEditor.AddField(FieldName: String;
                                            DisplayLabel: String;
                                            WhereSyntax: String;
                                            DisplayWidth: Integer;
                                            DisplayFormat: String;
                                            DisplayColumnWidth: Integer;
                                            FieldType: TFieldType;
                                            EmptyOperation: TSearchEmptyOperation;
                                            Search: Boolean;
                                            DefaultComparison: TSearchComparison;
                                            SearchCase: TSearchCase): Integer;
var
  f : TEstSearchDialogField;
begin
  // Maak item aan
  f := TEstSearchDialogField.Create(nil);

  f.FieldName := FieldName;
  f.DisplayLabel := DisplayLabel;
  f.WhereSyntax := WhereSyntax;
  f.DisplayWidth := DisplayWidth;
  f.DisplayFormat := DisplayFormat;
  f.DisplayColumnWidth := DisplayColumnWidth;
  f.FieldType := FieldType;
  f.EmptyOperation := EmptyOperation;
  f.Search := Search;
  f.DefaultComparison := DefaultComparison;
  f.SearchCase := SearchCase;

  // Voeg item toe
  Result := lbxFields.Items.AddObject(f.FieldName, f);
end;

procedure TFrmEstSearchDialogEditor.FormCreate(Sender: TObject);
var
  t : TFieldType;
  s : TSearchComparison;
  c : TSearchCase;
begin
  // Koppel de datasources aan de datasets
  dtsSearch.DataSet := SearchDataSet;
  dtsCount.DataSet := CountDataSet;

  // Initieel worden er geen controls bijgewerkt
  FUpdatingControls := False;

  // De searchdialogclass is een standaard search dialog
  FSearchDialogClass := TEstSearchDialog;

  // Vul fieldtype combobox met alle bekende fieldtypes
  cbxFieldType.Items.Clear;
  for t := Low(TFieldType) to High(TFieldType) do
    cbxFieldType.Items.AddObject(GetEnumName(TypeInfo(TFieldType), Ord(t)), TObject(t));

  // Vul default comparison combobox met alle bekende vergelijkingen
  cbxDefaultComparison.Items.Clear;
  for s := Low(TSearchComparison) to High(TSearchComparison) do
    cbxDefaultComparison.Items.AddObject(GetEnumName(TypeInfo(TSearchComparison), Ord(s)), TObject(s));

  // Vul search case combobox met alle bekende cases
  cbxSearchCase.Items.Clear;
  for c := Low(TSearchCase) to High(TSearchCase) do
    cbxSearchCase.Items.AddObject(GetEnumName(TypeInfo(TSearchCase), Ord(c)), TObject(c));
end;

procedure TFrmEstSearchDialogEditor.FormDestroy(Sender: TObject);
begin
  // Maak fields listbox schoon
  ClearFieldListBox;

  // Sluit eventuele openstaande queries
  SearchDataSet.Close;
  CountDataSet.Close;
end;

procedure TFrmEstSearchDialogEditor.lbxFieldsClick(Sender: TObject);
begin
  SelectField;
end;

procedure TFrmEstSearchDialogEditor.SelectField;
var
  f : TEstSearchDialogField;
begin
  // Geef aan dat er controls worden bijgewerkt
  FUpdatingControls := True;

  try

    // Als er een selectie is, maak die dan. Wis anders de selectie
    if lbxFields.ItemIndex <> -1 then
    begin
      f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

      edtFieldname.Text := f.Fieldname;
      edtWhereSyntax.Text := f.WhereSyntax;
      edtDisplayLabel.Text := f.DisplayLabel;
      edtDisplayWidth.Text := IntToStr(f.DisplayWidth);
      edtColumnWidth.Text := IntToStr(f.DisplayColumnWidth);
      edtDisplayFormat.Text := f.DisplayFormat;
      cbxFieldType.ItemIndex := Ord(f.FieldType);
      cbxEmptyOperation.ItemIndex := Ord(f.EmptyOperation);
      cbxSearch.ItemIndex := Ord(f.Search);
      cbxSearchCase.ItemIndex := Ord(f.SearchCase);
      cbxDefaultComparison.ItemIndex := cbxDefaultComparison.Items.IndexOfObject(TObject(Ord(f.DefaultComparison)));
    end
    else
    begin
      edtFieldname.Text := '';
      edtWhereSyntax.Text := '';
      edtDisplayLabel.Text := '';
      edtDisplayWidth.Text := '0';
      edtDisplayFormat.Text := '';
      edtColumnWidth.Text := '0';
      cbxFieldType.ItemIndex := 0;
      cbxEmptyOperation.ItemIndex := 0;
      cbxSearch.ItemIndex := Ord(True);
      cbxSearchCase.ItemIndex := Ord(scMixed);
      cbxDefaultComparison.ItemIndex := 0;
    end;

  finally
    FUpdatingControls := False;
  end;
end;

procedure TFrmEstSearchDialogEditor.actQueryExecute(Sender: TObject);
begin
  // Valideer query
  ValidateSearchQuery;

  // Sluit een eventueel vorig resultaat
  SearchDataSet.Close;

  // Stel query in
  SetSearchSQL(StringReplace(mmoSearchQuery.Lines.Text,
                             WHERE_TAG,
                             SearchDialog.TrueExpression,
                             [rfIgnoreCase]));

  // Probeer nieuwe query te openen
  try
    SearchDataSet.Open;
  except
    on e: Exception do
      MessageDlg(Format('Unable to execute query (%s)', [e.Message]), mtError, [mbOK], 0);
  end;
end;

procedure TFrmEstSearchDialogEditor.actGenerateExecute(Sender: TObject);
var
  i : Integer;
  GenerationStyle : Integer;
  f : TFrmEstSearchDialogEditorFieldGeneration;
begin
  // Bepaal stijl van aanmaken velden
  f := TFrmEstSearchDialogEditorFieldGeneration.Create(nil);

  try
    if f.ShowModal = mrOK then
      GenerationStyle := f.rgpWhereSyntaxStyle.ItemIndex
    else
      GenerationStyle := -1;
  finally
    f.Free;
  end;

  if GenerationStyle <> -1 then
  begin

    // Sluit een eventueel vorig resultaat
    SearchDataSet.Close;

    // Stel query in
    SetSearchSQL(StringReplace(mmoSearchQuery.Lines.Text,
                               WHERE_TAG,
                               SearchDialog.TrueExpression,
                               [rfIgnoreCase]));

    // Probeer nieuwe query te openen
    try
      SearchDataSet.Open;
    except
      on e: Exception do
      begin
        MessageDlg(Format('Unable to execute query (%s)', [e.Message]), mtError, [mbOK], 0);
        Exit;
      end;
    end;

    // Loop door de fieldlist van de query
    for i := 0 to SearchDataSet.Fields.Count - 1 do
    begin

      // Kijk of er niet reeds een SearchField is. Zo ja, doe dan niets, zo nee
      // maak dan een nieuwe
      if lbxFields.Items.IndexOf(SearchDataSet.Fields[i].FieldName) = -1 then
      begin
        // Maak item aan
        with SearchDataSet.Fields[i] do
        begin
          case GenerationStyle of
            0 : AddField(FieldName, FieldName, FieldName, DisplayWidth, '', 0,
                           DataType, eoNull, True, FieldTypeDefaultComparison(DataType), scMixed);
            1 : AddField(FieldName, FieldName, Format('''%s''', [FieldName]), DisplayWidth, '', 0,
                           DataType, eoNull, True, FieldTypeDefaultComparison(DataType), scMixed);
            2 : AddField(FieldName, FieldName, Format('"%s"', [FieldName]), DisplayWidth, '', 0,
                           DataType, eoNull, True, FieldTypeDefaultComparison(DataType), scMixed);
            3 : AddField(FieldName, FieldName, Format('[%s]', [FieldName]), DisplayWidth, '', 0,
                           DataType, eoNull, True, FieldTypeDefaultComparison(DataType), scMixed);
          end;
        end;
        
      end;

    end;

    // Als er niets geselecteerd was, en er valt nu wel iets te selecteren,
    // selecteer dan het eerste item
    if (lbxFields.ItemIndex = -1) and (lbxFields.Items.Count > 0) then
      lbxFields.ItemIndex := 0;

    // Voer selectie uit
    SelectField;

  end;
end;

procedure TFrmEstSearchDialogEditor.actAddExecute(Sender: TObject);
var
  i : Integer;
  n : Integer;
  f : String;
begin

  i := 1;

  repeat
    f := Format('NewField%d', [i]);
    Inc(i);
  until (lbxFields.Items.IndexOf(f) = -1);

  // Voeg veld toe
  n := AddField(f, f, f, 0, '', 0, ftInteger, eoNull, True, scEqual, scMixed);

  // Selecteer nieuw field
  lbxFields.ItemIndex := n;
  SelectField;
end;

procedure TFrmEstSearchDialogEditor.actDeleteExecute(Sender: TObject);
var
  i : Integer;
begin
  // Controleer voor alle zekerheid of er een item geselecteerd is
  i := lbxFields.ItemIndex;
  if i = -1 then
    Exit;

  // Verwijder geselecteerd item
  if Assigned(lbxFields.Items.Objects[i]) then
    lbxFields.Items.Objects[i].Free;
  lbxFields.Items.Delete(i);

  // Kijk of volgend item geselecteerd kan worden. Zo ja, selecteer dat dan
  if i < lbxFields.Items.Count then
  begin
    lbxFields.ItemIndex := i;
  end
  else
  begin
    // Zo nee, probeer vorig item
    if i > 0 then
      lbxFields.ItemIndex := i - 1;
  end;

  // Voer selectie uit
  SelectField;
end;

procedure TFrmEstSearchDialogEditor.edtFieldnameExit(Sender: TObject);
begin
  SetFieldName;
end;

procedure TFrmEstSearchDialogEditor.edtFieldnameKeyPress(Sender: TObject;
  var Key: Char);
begin
  // Reageer op enter
  if Key = #13 then
  begin
    SetFieldName;
    Key := #0;
  end;
end;

procedure TFrmEstSearchDialogEditor.edtWhereSyntaxExit(Sender: TObject);
begin
  SetWhereSyntax;
end;

procedure TFrmEstSearchDialogEditor.edtWhereSyntaxKeyPress(Sender: TObject;
  var Key: Char);
begin
  // Reageer op enter
  if Key = #13 then
  begin
    SetWhereSyntax;
    Key := #0;
  end;
end;

procedure TFrmEstSearchDialogEditor.edtDisplayLabelExit(Sender: TObject);
begin
  SetDisplayLabel;
end;

procedure TFrmEstSearchDialogEditor.edtDisplayLabelKeyPress(
  Sender: TObject; var Key: Char);
begin
  // Reageer op enter
  if Key = #13 then
  begin
    SetDisplayLabel;
    Key := #0;
  end;
end;

procedure TFrmEstSearchDialogEditor.edtDisplayWidthExit(Sender: TObject);
begin
  SetDisplayWidth;
end;

procedure TFrmEstSearchDialogEditor.edtDisplayWidthKeyPress(Sender: TObject; var Key: Char);
begin
  // Reageer op enter
  if Key = #13 then
  begin
    SetDisplayWidth;
    Key := #0;
  end;
end;

procedure TFrmEstSearchDialogEditor.edtColumnWidthExit(Sender: TObject);
begin
  SetColumnWidth;
end;

procedure TFrmEstSearchDialogEditor.edtColumnWidthKeyPress(Sender: TObject; var Key: Char);
begin
  // Reageer op enter
  if Key = #13 then
  begin
    SetColumnWidth;
    Key := #0;
  end;
end;

procedure TFrmEstSearchDialogEditor.edtDisplayFormatExit(Sender: TObject);
begin
  SetDisplayFormat;
end;

procedure TFrmEstSearchDialogEditor.edtDisplayFormatKeyPress(Sender: TObject; var Key: Char);
begin
  // Reageer op enter
  if Key = #13 then
  begin
    SetDisplayFormat;
    Key := #0;
  end;
end;

procedure TFrmEstSearchDialogEditor.SetFieldName;
var
  f : TEstSearchDialogField;
begin
  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Controleer of de veldnaam niet reeds gebruikt wordt. Zo niet, werk dan het
  // object bij
  if (lbxFields.Items.IndexOf(edtFieldName.Text) = -1) or
     (lbxFields.Items.IndexOf(edtFieldName.Text) = lbxFields.ItemIndex) then
  begin
    // Zet fieldname
    f.FieldName := edtFieldName.Text;

    // Werk lijst bij
    lbxFields.Items[lbxFields.ItemIndex] := f.FieldName;
  end
  else
  begin
    // Toon foutmelding
    MessageDlg(Format('Duplicate FieldName %s', [edtFieldName.Text]), mtError, [mbOK], 0);

    // Sta niet toe het control te verlaten
    edtFieldName.SetFocus;
    Abort;
  end;
end;

procedure TFrmEstSearchDialogEditor.SetWhereSyntax;
var
  f : TEstSearchDialogField;
begin
  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Zet wheresyntax
  f.WhereSyntax := edtWhereSyntax.Text;
end;

procedure TFrmEstSearchDialogEditor.SetDisplayLabel;
var
  f : TEstSearchDialogField;
begin
  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Zet displaylabel
  f.DisplayLabel := edtDisplayLabel.Text;
end;

procedure TFrmEstSearchDialogEditor.SetDisplayWidth;
var
  f : TEstSearchDialogField;
  i : Integer;
begin
  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Controleer of er een geldige waarde is ingevuld. Zo ja, werk dan het
  // object bij
  i := StrToIntDef(edtDisplayWidth.Text, -MAXINT);
  if i >= 0 then
  begin
    // Zet displaywidth
    f.DisplayWidth := i;
  end
  else
  begin
    // Toon foutmelding
    MessageDlg(Format('Invalid displaywidth %s', [edtDisplayWidth.Text]), mtError, [mbOK], 0);

    // Sta niet toe het control te verlaten
    edtDisplayWidth.SetFocus;
    Abort;
  end;
end;

procedure TFrmEstSearchDialogEditor.SetColumnWidth;
var
  f : TEstSearchDialogField;
  i : Integer;
begin
  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Controleer of er een geldige waarde is ingevuld. Zo ja, werk dan het
  // object bij
  i := StrToIntDef(edtColumnWidth.Text, -MAXINT);
  if i >= 0 then
  begin
    // Zet displaywidth
    f.DisplayColumnWidth := i;
  end
  else
  begin
    // Toon foutmelding
    MessageDlg(Format('Invalid columnwidth %s', [edtColumnWidth.Text]), mtError, [mbOK], 0);

    // Sta niet toe het control te verlaten
    edtColumnWidth.SetFocus;
    Abort;
  end;
end;

procedure TFrmEstSearchDialogEditor.SetDisplayFormat;
var
  f : TEstSearchDialogField;
begin
  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Zet displayformat
  f.DisplayFormat := edtDisplayFormat.Text;
end;

procedure TFrmEstSearchDialogEditor.cbxFieldTypeChange(Sender: TObject);
var
  f : TEstSearchDialogField;
begin
  // Als de controls worden bijgewerkt vanuit de lijst, doe dan niets
  if FUpdatingControls then
    Exit;

  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Zet fieldtype als dat geselecteerd is
  if cbxFieldType.ItemIndex <> -1 then
    f.FieldType := TFieldType(cbxFieldType.Items.Objects[cbxFieldType.ItemIndex]);
end;

procedure TFrmEstSearchDialogEditor.cbxEmptyOperationChange(Sender: TObject);
var
  f : TEstSearchDialogField;
begin
  // Als de controls worden bijgewerkt vanuit de lijst, doe dan niets
  if FUpdatingControls then
    Exit;

  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Zet empty operation als dat geselecteerd is
  if cbxEmptyOperation.ItemIndex <> -1 then
    f.EmptyOperation := TSearchEmptyOperation(cbxEmptyOperation.ItemIndex);
end;

procedure TFrmEstSearchDialogEditor.cbxDefaultComparisonChange(Sender: TObject);
var
  f : TEstSearchDialogField;
begin
  // Als de controls worden bijgewerkt vanuit de lijst, doe dan niets
  if FUpdatingControls then
    Exit;

  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Zet default comparison
  if cbxDefaultComparison.ItemIndex <> -1 then
    f.DefaultComparison := TSearchComparison(cbxDefaultComparison.Items.Objects[cbxDefaultComparison.ItemIndex]);
end;

procedure TFrmEstSearchDialogEditor.cbxSearchCaseChange(Sender: TObject);
var
  f : TEstSearchDialogField;
begin
  // Als de controls worden bijgewerkt vanuit de lijst, doe dan niets
  if FUpdatingControls then
    Exit;

  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Zet default comparison
  if cbxSearchCase.ItemIndex <> -1 then
    f.SearchCase := TSearchCase(cbxSearchCase.Items.Objects[cbxSearchCase.ItemIndex]);
end;

procedure TFrmEstSearchDialogEditor.cbxSearchChange(Sender: TObject);
var
  f : TEstSearchDialogField;
begin
  // Als de controls worden bijgewerkt vanuit de lijst, doe dan niets
  if FUpdatingControls then
    Exit;

  // Als er geen veld geselecteerd is, doe dan niets
  if lbxFields.ItemIndex = -1 then
    Exit;

  // Zoek object bij selected
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Zet search
  f.Search := (cbxSearch.ItemIndex = Ord(True));
end;

{|
  Procedure : TFrmCblSearchDialogEditor.UpdateSearchDialog
  Auteur    : Erik Stok
  Doel      : Werk gegevens search dialog bij aan de hand van de instellingen
              op het scherm
|}
procedure TFrmEstSearchDialogEditor.UpdateSearchDialog(const ASearchDialog: TEstSearchDialog);
var
  i : Integer;
begin

  // Zet search query
  ASearchDialog.SearchQuery.Text := mmoSearchQuery.Lines.Text;
  ASearchDialog.CountQuery.Text := mmoCountQuery.Lines.Text;

  // Zet fieldlist
  ASearchDialog.SearchDialogFieldList.Clear;

  for i := 0 to lbxFields.Items.Count - 1 do
  begin
    with TEstSearchDialogField(lbxFields.Items.Objects[i]) do
      ASearchDialog.SearchDialogFieldList.Add(FieldName, WhereSyntax, DisplayLabel,
                                                DisplayWidth, DisplayFormat,
                                                DisplayColumnWidth, FieldType,
                                                Search, DefaultComparison, SearchCase);
  end;

end;

procedure TFrmEstSearchDialogEditor.actOKExecute(Sender: TObject);
begin
  // Check queries
  ValidateSearchQuery;
  ValidateCountQuery;
  ValidateDefaultSearch;

  // Werk component bij
  UpdateSearchDialog(SearchDialog);

  // Sluit scherm
  modalResult := mrOK;
end;

procedure TFrmEstSearchDialogEditor.actCancelExecute(Sender: TObject);
begin
  // Sluit scherm zonder iets bij te werken
  ModalResult := mrCancel;
end;

procedure TFrmEstSearchDialogEditor.actTestExecute(Sender: TObject);
var
  s : TEstSearchDialog;
begin
  // Maak tijdelijk search dialog component aan van het juiste type
  s := FSearchDialogClass.Create(nil);

  try
    // Kopieer origineel searchdialog
    s.Assign(SearchDialog);

    // Werk bij aan de hand van het scherm
    UpdateSearchDialog(s);

    // Designtime search dialog is altijd modal
    s.SearchStyle := ssModal;

    // Test dialog
    try
      s.Execute;
    except
      on e:Exception do
        MessageDlg(Format('Cannot execute search dialog (%s)', [e.Message]), mtError, [mbOK], 0);
    end;

  finally
    s.Free;
  end;
end;

procedure TFrmEstSearchDialogEditor.actCountQueryExecute(Sender: TObject);
begin
  // Valideer query
  ValidateCountQuery;

  // Sluit een eventueel vorig resultaat
  CountDataSet.Close;

  // Stel query in
  SetCountSQL(StringReplace(mmoCountQuery.Lines.Text,
                            WHERE_TAG,
                            SearchDialog.TrueExpression,
                            [rfIgnoreCase]));

  // Probeer nieuwe query te openen
  try
    CountDataSet.Open;
  except
    on e: Exception do
      MessageDlg(Format('Unable to execute query (%s)', [e.Message]), mtError, [mbOK], 0);
  end;
end;

procedure TFrmEstSearchDialogEditor.actPriorFieldExecute(Sender: TObject);
begin
  // Selecteer vorig veld indien mogelijk
  if (lbxFields.Items.Count > 0) and
     (lbxFields.ItemIndex > 0) then
  begin
    lbxFields.ItemIndex := lbxFields.ItemIndex - 1;
    SelectField;
  end;
end;

procedure TFrmEstSearchDialogEditor.actNextFieldExecute(Sender: TObject);
begin
  // Selecteer volgend veld indien mogelijk
  if (lbxFields.Items.Count > 0) and
     (lbxFields.ItemIndex < (lbxFields.Items.Count - 1)) then
  begin
    lbxFields.ItemIndex := lbxFields.ItemIndex + 1;
    SelectField;
  end;
end;

procedure TFrmEstSearchDialogEditor.actMoveUpExecute(Sender: TObject);
var
  i : Integer;
  f : TEstSearchDialogField;
begin
  // Controleer voor alle zekerheid of er een geldig item geselecteerd is
  i := lbxFields.ItemIndex;
  if i < 1 then
    Exit;

  // Verkrijg referentie naar field object
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Verplaats geselecteerd item
  lbxFields.Items.Move(i, i-1);
  f.Index := i-1;

  // Selecteer item
  lbxFields.ItemIndex := i-1;
end;

procedure TFrmEstSearchDialogEditor.actMoveDownExecute(Sender: TObject);
var
  i : Integer;
  f : TEstSearchDialogField;
begin
  // Controleer voor alle zekerheid of er een geldig item geselecteerd is
  i := lbxFields.ItemIndex;
  if (i = -1) or (i = lbxFields.Items.Count - 1) then
    Exit;

  // Verkrijg referentie naar field object
  f := TEstSearchDialogField(lbxFields.Items.Objects[lbxFields.ItemIndex]);

  // Verplaats geselecteerd item
  lbxFields.Items.Move(i, i+1);
  f.Index := i+1;

  // Selecteer item
  lbxFields.ItemIndex := i+1;
end;

{|
  Procedure : TFrmEstSearchDialogEditor.ValidateCountQuery
  Auteur    : Erik Stok
  Doel      : Controleer of de count query de where tag bevat
|}
procedure TFrmEstSearchDialogEditor.ValidateCountQuery;
begin
  if (Trim(mmoCountQuery.Lines.Text) <> '') and
     (Pos(WHERE_TAG, UpperCase(mmoCountQuery.Lines.Text)) = 0) then
  begin
    raise Exception.CreateFmt('Count query does not contain required %s tag', [WHERE_TAG]);
  end;
end;

{|
  Procedure : TFrmEstSearchDialogEditor.ValidateSearchQuery
  Auteur    : Erik Stok
  Doel      : Controleer of de search query de where tag bevat
|}
procedure TFrmEstSearchDialogEditor.ValidateSearchQuery;
begin
  if (Trim(mmoSearchQuery.Lines.Text) <> '') and
     (Pos(WHERE_TAG, UpperCase(mmoSearchQuery.Lines.Text)) = 0) then
  begin
    raise Exception.CreateFmt('Search query does not contain required %s tag', [WHERE_TAG]);
  end;
end;

{|
  Procedure : TFrmEstSearchDialogEditor.ValidateDefaultSearch
  Auteur    : Erik Stok
  Doel      : Controleer of er geldige instellingen gedaan zijn voor de default
              vergelijkingen
|}
procedure TFrmEstSearchDialogEditor.ValidateDefaultSearch;
var
  i      : Integer;
  Answer : TModalResult;
begin
  // Loop door de field objecten die zijn ingesteld
  for i := 0 to lbxFields.Items.Count - 1 do
  begin

    with TEstSearchDialogField(lbxFields.Items.Objects[i]) do
    begin

      // Kijk of de default comparison geldig is
      if not (DefaultComparison in FieldTypeComparisons(FieldType)) then
      begin
        // Vraag de gebruiker wat te doen
        Answer := MessageDlg(Format('The default comparison of field ''%s'' is not ' +
                                    'valid for the indicated fieldtype. Correct ' +
                                    'it to the default value?', [FieldName]),
                             mtConfirmation, mbYesNoCancel, 0);

        // Fix?
        if Answer = mrYes then
        begin
          DefaultComparison := FieldTypeDefaultComparison(FieldType);
        end
        else
        begin
          // Stop?
          if Answer = mrCancel then
            Abort;
        end;

      end;

    end;

  end;

end;

end.
