{|
  Unit   : fEstSearchDialog
  Datum  : 6-12-2003
  Auteur : Erik Stok
  Doel   : Scherm voor zoekdialoog component
|}

unit fEstSearchDialog;


interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Grids, DBGrids, Db, ExtCtrls, ActnList, ComCtrls,  Registry,
  uEstSearchDialogTypes, uEstSearchDialogConst, uEstSearchDialogDateUtils,
  uEstSearchDialogProperties, uEstSearchDialogIntl, uEstSearchDialogFunction;

type
  TFrmEstSearchDialog = class(TForm)
    pnlMain: TPanel;
    pnlCriteria: TPanel;
    pnlGrid: TPanel;
    pnlButtons: TPanel;
    dtsSearch: TDataSource;
    grdResult: TDBGrid;
    btnOK: TButton;
    btnCancel: TButton;
    pnlInput: TPanel;
    pnlActions: TPanel;
    btnMore: TButton;
    btnLess: TButton;
    btnSearch: TButton;
    pnlCriteriaLine01: TPanel;
    pnlCriterium01: TPanel;
    pnlOperator01: TPanel;
    cbxField01: TComboBox;
    cbxComparison01: TComboBox;
    edtValue01: TEdit;
    cbxOperator01: TComboBox;
    aclSearch: TActionList;
    actMore: TAction;
    actLess: TAction;
    actSearch: TAction;
    actOK: TAction;
    actCancel: TAction;
    stbInfo: TStatusBar;
    pnlLabels: TPanel;
    lblField: TLabel;
    lblComparison: TLabel;
    lblValue: TLabel;
    lblOperator: TLabel;
    procedure actSearchExecute(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure cbxFieldChange(Sender: TObject);
    procedure actOKExecute(Sender: TObject);
    procedure actCancelExecute(Sender: TObject);
    procedure edtValueChange(Sender: TObject);
    procedure edtValueKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure actMoreExecute(Sender: TObject);
    procedure actLessExecute(Sender: TObject);
    procedure cbxComparisonChange(Sender: TObject);
    procedure grdResultDblClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
  private
    FOnSetup: TEstSearchDialogSetupEvent;
    FMdi: Boolean;

    FMaxMessageSet: Boolean;
    FInitializing: Boolean;

    FMaxCriteria: Integer;
    FCriteriaCount: Integer;

    FSearchQuery: String;
    FCountQuery: String;
    FMax: Integer;
    FSearchDialogFieldList: TEstSearchDialogFieldList;

    FFieldIndex : Integer;
    FRegistryPath: string;
    FStore: TEstSearchDialogStore;

    FSearchDataSet: TDataSet;
    FCountDataSet: TDataSet;

    FOnSetSearchQuery: TEstSearchDialogSetSqlEvent;
    FOnSetCountQuery: TEstSearchDialogSetSqlEvent;

    FResultQuery: String;

    FCompareFormatDate: String;
    FCompareFormatDateTime: String;
    FCompareFormatTime: String;

    FQuoteChar: Char;
    FQuotedDateTime: Boolean;
    FAdditionalWhere: String;
    FStartOpen: Boolean;
    FTrueExpression: String;
    FDecimalChar: Char;

    FSearchStyle : TSearchStyle;
    FOnOK: TNotifyEvent;
    FOnCancel: TNotifyEvent;

    FOnInitControls: TEstSearchDialogInitControlsEvent;

    function LineNr(Sender: TObject): String;

    function FieldComboBox(Line: String): TComboBox;
    function ComparisonComboBox(Line: String): TComboBox;
    function ValueEdit(Line: String): TEdit;
    function OperatorComboBox(Line: String): TComboBox;

    procedure ShowOperators;

    procedure InitSearchAgain;
    procedure InitFound;

  protected
    procedure DoCreate; override;

    procedure ColorValue(Line: String); virtual;

    procedure ToggleActionsAndControls; virtual;
    procedure SetFieldSettings; virtual;

    function DateTimeValue(FormatString: String; Value: TDateTime): String;
    function ValueToDate(Value: String): TDateTime;

    function  BuildQuery(InitialQuery: String): String; virtual;
    function  BuildWhereClause: String; virtual;

    procedure InitControls(Line: String); virtual;
    procedure DoInitControls(Sender: TObject;
                             Field: TEstSearchDialogField;
                             FieldControl: TComboBox;
                             Comparison: TSearchComparison;
                             ComparisonControl: TComboBox;
                             ValueControl: TEdit;
                             UsingOperator: Boolean;
                             OperatorControl: TComboBox); virtual;

    procedure AddCriterium(FieldIndex: Integer = -1); virtual;
    procedure DeleteCriterium; virtual;

    procedure Search; virtual;

    procedure Initialise; virtual;

    procedure SetSearchDialogFieldList(Value: TEstSearchDialogFieldList); virtual;

    procedure SetSearchQuery(SQL: String); virtual;
    procedure SetCountQuery(SQL: String); virtual;

    procedure LoadSearchFields(IncludeValues: Boolean); virtual;
    procedure SaveSearchFields(IncludeValues: Boolean); virtual;

    procedure PerformOK; virtual;
    procedure PerformCancel; virtual;
  public
    constructor Create(AOwner: TComponent; Mdi: Boolean; OnSetup: TEstSearchDialogSetupEvent); reintroduce;

    property SearchStyle: TSearchStyle read FSearchStyle write FSearchStyle;

    function GetResultList: String;
    function GetResultQuery: String;

    property CriteriaCount: Integer read FCriteriaCount write FCriteriaCount;
    property MaxCriteria: Integer read FMaxCriteria write FMaxCriteria;

    property SearchQuery: String read FSearchQuery write FSearchQuery;
    property CountQuery: String read FCountQuery write FCountQuery;
    property Max: Integer read FMax write FMax;

    property RegistryPath: string read FRegistryPath write FRegistryPath;

    property SearchDialogFieldList: TEstSearchDialogFieldList read FSearchDialogFieldList
                                                              write SetSearchDialogFieldList;

    property Store: TEstSearchDialogStore read FStore write FStore;

    property SearchDataSet: TDataSet read FSearchDataSet write FSearchDataSet;
    property CountDataSet: TDataSet read FCountDataSet write FCountDataSet;

    property OnSetSearchQuery: TEstSearchDialogSetSqlEvent read FOnSetSearchQuery
                                                           write FOnSetSearchQuery;

    property OnSetCountQuery: TEstSearchDialogSetSqlEvent read FOnSetCountQuery
                                                          write FOnSetCountQuery;

    property OnInitControls: TEstSearchDialogInitControlsEvent read FOnInitControls
                                                               write FOnInitControls;

    property OnOK: TNotifyEvent read FOnOK write FOnOK;
    property OnCancel: TNotifyEvent read FOnCancel write FOnCancel;

    property CompareFormatDate: String read FCompareFormatDate write FCompareFormatDate;
    property CompareFormatTime: String read FCompareFormatTime write FCompareFormatTime;
    property CompareFormatDateTime: String read FCompareFormatDateTime write FCompareFormatDateTime;
    property QuotedDateTime: Boolean read FQuotedDateTime write FQuotedDateTime default True;

    property QuoteChar: Char read FQuoteChar write FQuoteChar;
    property DecimalChar: Char read FDecimalChar write FDecimalChar;

    property AdditionalWhere: String read FAdditionalWhere write FAdditionalWhere;
    property StartOpen: Boolean read FStartOpen write FStartOpen;
    property TrueExpression: String read FTrueExpression write FTrueExpression;
  end;

  TFrmEstSearchDialogClass = class of TFrmEstSearchDialog;

function FieldTypeComparisons(FieldType: TFieldType): TSearchComparisons;
function FieldTypeDefaultComparison(FieldType: TFieldType): TSearchComparison;

implementation

{$R *.DFM}

// Class om bij VisibleRowCount property van DBGrid te kunnen komen
type
  TDBGridAccessClass = class(TDBGrid)
  public
    property VisibleRowCount;
  end;

{|
  Procedure : FieldTypeComparisons
  Auteur    : Erik Stok
  Doel      : Vind voor een specifiek veldtype de geldige vergelijkingen
|}
function FieldTypeComparisons(FieldType: TFieldType): TSearchComparisons;
var
  i : Integer;
begin
  // Standaard resultaat
  Result := VALIDCOMPARISON_NONE;

  // Zoek in de lijst van geldige vergelijking voor velden naar het veldtype
  for i := Low(VALIDCOMPARISONS) to High(VALIDCOMPARISONS) do
  begin

    // Als dit gevonden is, geef dan de bijbehorende vergelijkingen
    if VALIDCOMPARISONS[i].FieldType = FieldType then
    begin
      Result := VALIDCOMPARISONS[i].Comparisons;
      Break;
    end;

  end;

end;

{|
  Procedure : FieldTypeDefaultComparison
  Auteur    : Erik Stok
  Doel      : Vind voor een specifiek veldtype de standaard vergelijking
|}
function FieldTypeDefaultComparison(FieldType: TFieldType): TSearchComparison;
var
  i : Integer;
begin
  // Standaard resultaat
  Result := scEqual;

  // Zoek in de lijst van standaard vergelijkingen voor velden naar het veldtype
  for i := Low(VALIDCOMPARISONS) to High(VALIDCOMPARISONS) do
  begin

    // Als dit gevonden is, geef dan de bijbehorende vergelijkingen
    if VALIDCOMPARISONS[i].FieldType = FieldType then
    begin
      Result := VALIDCOMPARISONS[i].DefaultComparison;
      Break;
    end;

  end;

end;

{|
  Procedure : TFrmEstSearchDialog.DateTimeValue
  Auteur    : Erik Stok
  Doel      : Formatteer datum/tijd naar zoekwaarde die in where clausule
              gebruikt kan worden
|}
function TFrmEstSearchDialog.DateTimeValue(FormatString: String; Value: TDateTime): String;
begin
  // Bepaal datum tijd volgens gegeven format, met of zonder quotes
  if FQuotedDateTime then
    Result := AnsiQuotedStr(FormatDateTime(FormatString, Value), FQuoteChar)
  else
    Result := FormatDateTime(FormatString, Value);

end;

{|
  Procedure : TFrmEstSearchDialog.ValueToDate
  Author    : Erik Stok
  Purpose   : Converteer string naar datum/tijd rekening houdend met de speciale
              waarden die een gebruiker mag invoeren
|}
function TFrmEstSearchDialog.ValueToDate(Value: String): TDateTime;
var
  ValueWithoutSpaces : String;
  ValueOffset : Integer;

  function DetermineDate(SpecialValue: String;
                         ValueToCheck: String;
                         var Offset: Integer): Boolean;
  begin
    // Standaard resultaat
    Result := False;

    // Controleer of het begin overeenkomt
    if Copy(ValueToCheck, 1, Length(SpecialValue)) = SpecialValue then
    begin
      // Zo ja, bepaal dan de offset
      if Length(ValueToCheck) <> Length(SpecialValue) then
        Offset := StrToInt(Copy(ValueToCheck, Length(SpecialValue) + 1, Length(ValueToCheck)))
      else
        Offset := 0;

      // En geef terug dat er een match is geweest
      Result := True;
    end;

  end;

begin
  // Controleer of de waarde zonder spaties begint met een speciale waarde
  ValueWithoutSpaces := UpperCase(StringReplace(Value, ' ', '', [rfReplaceAll]));

  try

    if DetermineDate(EstSearchDialogIntl.DateTimeValueSecond, ValueWithoutSpaces, ValueOffset) then
      Result := OffsetForSecond(ValueOffset)
    else if DetermineDate(EstSearchDialogIntl.DateTimeValueMinute, ValueWithoutSpaces, ValueOffset) then
      Result := OffsetForMinute(ValueOffset)
    else if DetermineDate(EstSearchDialogIntl.DateTimeValueHour, ValueWithoutSpaces, ValueOffset) then
      Result := OffsetForHour(ValueOffset)
    else if DetermineDate(EstSearchDialogIntl.DateTimeValueDay, ValueWithoutSpaces, ValueOffset) then
      Result := OffsetForDay(ValueOffset)
    else if DetermineDate(EstSearchDialogIntl.DateTimeValueWeek, ValueWithoutSpaces, ValueOffset) then
      Result := OffsetForWeek(ValueOffset)
    else if DetermineDate(EstSearchDialogIntl.DateTimeValueMonth, ValueWithoutSpaces, ValueOffset) then
      Result := OffsetForMonth(ValueOffset)
    else if DetermineDate(EstSearchDialogIntl.DateTimeValueQuarter, ValueWithoutSpaces, ValueOffset) then
      Result := OffsetForQuarter(ValueOffset)
    else if DetermineDate(EstSearchDialogIntl.DateTimeValueYear, ValueWithoutSpaces, ValueOffset) then
      Result := OffsetForYear(ValueOffset)
    else if EstSearchDialogIntl.DateTimeValueNow = ValueWithoutSpaces then
      Result := Now
    else
      Result := StrToDate(Value);

  except
    raise EEstSearchDialogException.Create(EstSearchDialogIntl.ErrorInvalidDateTimeFormula);
  end;

end;

{|
  Procedure : TfrmSearchDialog.BuildQuery
  Auteur    : Erik Stok
  Doel      : Bouw SQL statement op basis van algemeen SQL statement en de
              opgegeven zoekcriteria
|}
function TFrmEstSearchDialog.BuildQuery(InitialQuery: String): String;
var
  WhereClause : String;
begin
  // Maak basis query
  Result := InitialQuery;

  // Bouw de WHERE clausule en voeg deze toe aan de query. Tijdens initialisatie
  // wordt de where clausule slechts bepaald door de additionalwhere en de
  // eventueel de trueexpression
  if not FInitializing then
    WhereClause := BuildWhereClause
  else
    WhereClause := FAdditionalWhere;

  if WhereClause = '' then
    WhereClause := FTrueExpression;

  // Vervang de WHERE tag door de clausule
  Result := StringReplace(Result, WHERE_TAG, WhereClause, [rfIgnoreCase]);
end;

{|
  Procedure : TfrmSearchDialog.BuildWhereClause
  Auteur    : Erik Stok
  Doel      : Bouw een WHERE clausule op basis van de opgegeven zoekcriteria
|}
function TFrmEstSearchDialog.BuildWhereClause: String;
var
  i               : Integer;
  sc              : TSearchComparison;
  Line            : String;
  Field           : String;
  FieldObject     : TEstSearchDialogField;
  Comparison      : String;
  Value           : String;
  WhereComparison : String;
  OperatorIndex   : Integer;
  Operator        : TSearchOperator;
  BracketsToClose : Integer;
  FloatValue      : Extended;
  IntegerValue    : Int64;
  OldSeparator    : Char;
begin
  // Default resultaat
  Result := '';

  // Tot dusverre geen haakjes te sluiten
  BracketsToClose := 0;

  // Doorloop alle criteria
  for i := 1 to FCriteriaCount do
  begin

    // Bepaal regelnummer
    Line := Format('%.2d', [i]);

    // Bepaal veld
    with FieldComboBox(Line) do
    begin

      if ItemIndex <> -1 then
      begin
        FieldObject := TEstSearchDialogField(Items.Objects[ItemIndex]);
        Field := FieldObject.WhereSyntax
      end
      else
      begin
        FieldObject := nil;
        Field := '';
      end;

    end;

    if Field <> '' then
    begin

      // Bepaal vergelijking
      with ComparisonComboBox(Line) do
      begin
        if ItemIndex <> -1 then
        begin
          sc := TSearchComparison(Items.Objects[ItemIndex]);

          // Als er geen empty vergelijking gedaan wordt, gebruik dan de
          // vergelijking volgens constante lijst. Anders, bouw de empty
          // comparison op volgens opgegeven veldinstelling
          if not (sc in [scEmpty, scNotEmpty]) then
          begin
            Comparison := StringReplace(SEARCH_COMPARISON[sc], '"', FQuoteChar, [rfReplaceAll])
          end
          else
          begin

            if sc = scEmpty then
            begin

              case FieldObject.EmptyOperation of
                eoEmptyString : Comparison := StringReplace(EMPTYCOMPARISON[FieldObject.EmptyOperation], '"', FQuoteChar, [rfReplaceAll]);
                eoBoth        : Comparison := StringReplace(EMPTYCOMPARISON[FieldObject.EmptyOperation], '"', FQuoteChar, [rfReplaceAll]);
              else
                Comparison := StringReplace(EMPTYCOMPARISON[FieldObject.EmptyOperation], '"', FQuoteChar, [rfReplaceAll]);
              end;

            end
            else
            begin

              case FieldObject.EmptyOperation of
                eoEmptyString : Comparison := StringReplace(NOTEMPTYCOMPARISON[FieldObject.EmptyOperation], '"', FQuoteChar, [rfReplaceAll]);
                eoBoth        : Comparison := StringReplace(NOTEMPTYCOMPARISON[FieldObject.EmptyOperation], '"', FQuoteChar, [rfReplaceAll]);
              else
                Comparison := StringReplace(NOTEMPTYCOMPARISON[FieldObject.EmptyOperation], '"', FQuoteChar, [rfReplaceAll]);
              end;

            end;

          end;

        end
        else
        begin
          sc := scEmpty;
          Comparison := '';
        end;
      end;

      if Comparison <> '' then
      begin

        // Bepaal waarde indien nodig
        with ValueEdit(Line) do
        begin

          if not (sc in [scEmpty, scNotEmpty]) then
          begin
            // Voor niet datetime velden wordt geen conversie gedaan
            if FieldObject.FieldType in [ftDate, ftTime, ftDateTime] then
            begin

              try

                case FieldObject.FieldType of
                  ftDate: Value := DateTimeValue(FCompareFormatDate, ValueToDate(Text));
                  ftTime: Value := DateTimeValue(FCompareFormatTime, ValueToDate(Text));
                  ftDateTime: Value := DateTimeValue(FCompareFormatDateTime, ValueToDate(Text));
                end;

              except

                // Zet focus op de edit box met de ongeldige waarde
                SetFocus;

                // Geef een foutmelding
                case FieldObject.FieldType of
                  ftDate: raise EEstSearchDialogException.Create(EstSearchDialogIntl.ErrorInvalidDate);
                  ftTime: raise EEstSearchDialogException.Create(EstSearchDialogIntl.ErrorInvalidTime);
                  ftDateTime: raise EEstSearchDialogException.Create(EstSearchDialogIntl.ErrorInvalidDateTime);
                end;

              end;

            end
            else
            begin

              if sc < scBeginsWith then
              begin

                try
                  case FieldObject.FieldType of
                    ftSmallint, ftInteger, ftWord, ftAutoInc, ftLargeint:
                      begin
                        IntegerValue := StrToInt(Text);
                        Value := IntToStr(IntegerValue);
                      end;
                    ftFloat, ftCurrency, ftBCD:
                      begin
                        FloatValue := StrToFloat(Text);
                        OldSeparator := DecimalSeparator;
                        try
                          DecimalSeparator := FDecimalChar;
                          Value := FloatToStr(FloatValue);
                        finally
                          DecimalSeparator := OldSeparator;
                        end;
                      end;
                  else
                    begin
                      case FieldObject.SearchCase of
                        scMixed: Value := Text;
                        scUpper: Value := AnsiUpperCase(Text);
                        scLower: Value := AnsiLowerCase(Text);
                      else
                        raise EEstSearchDialogException.CreateFmt(
                                'Invalid case value for search field %s',
                                [FieldObject.FieldName]);
                      end;
                    end;
                  end;
                except
                  SetFocus;
                  raise EEstSearchDialogException.Create(EstSearchDialogIntl.ErrorInvalidNumber);
                end;

              end
              else
              begin
                case FieldObject.SearchCase of
                  scMixed: Value := StringReplace(Text, FQuoteChar, FQuoteChar+FQuoteChar, [rfReplaceAll]);
                  scUpper: Value := StringReplace(AnsiUpperCase(Text), FQuoteChar, FQuoteChar+FQuoteChar, [rfReplaceAll]);
                  scLower: Value := StringReplace(AnsiLowerCase(Text), FQuoteChar, FQuoteChar+FQuoteChar, [rfReplaceAll]);
                else
                  raise EEstSearchDialogException.CreateFmt(
                          'Invalid case value for search field %s',
                          [FieldObject.FieldName]);
                end;
              end;

            end;

          end
          else
          begin
            Value := '';
          end;


        end;

        // Bouw de vergelijking zoals deze in de where clausule tussen de
        // haken geplaatst zal worden. Voor empty wordt een aparte format
        // gebruikt, aangezien het veld daar meerdere keren in voor kan komen en
        // de value geheel ontbreekt.
        if not (sc in [scEmpty, scNotEmpty]) then
        begin

          WhereComparison := Field + ' ' + Format(Comparison, [Value]);

        end
        else
        begin

          case FieldObject.EmptyOperation of
            eoEmptyString : WhereComparison := Format(Comparison, [Field]);
            eoBoth        : WhereComparison := Format(Comparison, [Field, Field]);
          else
            WhereComparison := Format(Comparison, [Field]);
          end;

        end;

        // Als er een vorige regel is, bepaal dan de operator daarvan
        if i > 1 then
        begin

          OperatorIndex := OperatorComboBox(Format('%.2d', [i - 1])).ItemIndex;

          if OperatorIndex <> -1 then
            Operator := TSearchOperator(OperatorComboBox(Format('%.2d', [i - 1])).Items.Objects[OperatorIndex])
          else
            Operator := soAnd;


          // Als er vorige operator was en het betrof een ...All, vorm dan
          // haakjes om de nog komende vergelijkingen.
          // Anders zijn haakjes om de vergelijking voldoende.
          if (OperatorIndex <> -1) and
             (Operator in [soAndAll, soOrAll]) then
          begin
            Result := Result + OPERATOR_SQL[Operator]  +
                               ' ('#13#10'(' + WhereComparison + ') '
                               + #13#10;
            BracketsToClose := BracketsToClose + 1;
          end
          else
          begin
            Result := Result + OPERATOR_SQL[Operator]  +
                               ' (' + WhereComparison + ') '
                               + #13#10;
          end;

        end
        else
        begin
          // Er was geen operator van de vorige regel, dus haakjes zijn voldoende
          Result := Result + '(' + WhereComparison + ') ' + #13#10;
        end;

      end;

    end;

  end;

  Result := Result + StringOfChar(')', BracketsToClose);

  // Als er een resultaat is en een additionele where clausule, pas die dan ook
  // nog toe
  if (Result <> '') and (FAdditionalWhere <> '') then
    Result := Format('(%s) AND (%s)', [Result, FAdditionalWhere]);
end;

{|
  Procedure : TfrmSearchDialog.LineNr
  Auteur    : Erik Stok
  Doel      : Geef het nummer van een control. Dit wordt gebruikt om controls
              uit dezelfde criteriumregels te kunnen selecteren.
|}
function TFrmEstSearchDialog.LineNr(Sender: TObject): String;
var
  s : String;
begin
  s := (Sender as TComponent).Name;

  Result := Copy(s, Length(s) - 1, 2);
end;

{|
  Procedure : TfrmSearchDialog.FieldComboBox
  Auteur    : Erik Stok
  Doel      : Bepaald voor een specifieke criteriumregel de veldcombobox
|}
function TFrmEstSearchDialog.FieldComboBox(Line: String): TComboBox;
var
  c : TComponent;
begin
  c := FindComponent('cbxField' + Line);

  if Assigned(c) then
    Result := TComboBox(c)
  else
    raise EEstSearchDialogException.CreateFmt('Cannot find field combobox for line %s', [Line]);
end;

{|
  Procedure : TfrmSearchDialog.ComparisonComboBox
  Auteur    : Erik Stok
  Doel      : Bepaald voor een specifieke criteriumregel de vergelijkingcombobox
|}
function TFrmEstSearchDialog.ComparisonComboBox(Line: String): TComboBox;
var
  c : TComponent;
begin
  c := FindComponent('cbxComparison' + Line);

  if Assigned(c) then
    Result := TComboBox(c)
  else
    raise EEstSearchDialogException.CreateFmt('Cannot find comparison combobox for line %s', [Line]);
end;

{|
  Procedure : TfrmSearchDialog.ValueEdit
  Auteur    : Erik Stok
  Doel      : Bepaald voor een specifieke criteriumregel de waardeedit
|}
function TFrmEstSearchDialog.ValueEdit(Line: String): TEdit;
var
  c : TComponent;
begin
  c := FindComponent('edtValue' + Line);

  if Assigned(c) then
    Result := TEdit(c)
  else
    raise EEstSearchDialogException.CreateFmt('Cannot find value edit for line %s', [Line]);
end;

{|
  Procedure : TfrmSearchDialog.OperatorComboBox
  Auteur    : Erik Stok
  Doel      : Bepaald voor een specifieke criteriumregel de operatorcombobox
|}
function TFrmEstSearchDialog.OperatorComboBox(Line: String): TComboBox;
var
  c : TComponent;
begin
  c := FindComponent('cbxOperator' + Line);

  if Assigned(c) then
    Result := TComboBox(c)
  else
    raise EEstSearchDialogException.CreateFmt('Cannot find operator combobox for line %s', [Line]);
end;

{|
  Procedure : TfrmSearchDialog.ToggleActionsAndControls
  Auteur    : Erik Stok
  Doel      : Schakel de actions aan en uit volgens de status van het form
|}
procedure TFrmEstSearchDialog.ToggleActionsAndControls;
begin
  actLess.Enabled := FCriteriaCount > 1;
  actMore.Enabled := FCriteriaCount < FMaxCriteria;
end;

{|
  Procedure : TfrmSearchDialog.DeleteCriterium
  Auteur    : Erik Stok
  Doel      : Verwijder zoekcriterium uit lijst van criteria
|}
procedure TFrmEstSearchDialog.DeleteCriterium;
var
  c : TComponent;
begin
  // Voorkom verwijderen regel 1
  if FCriteriaCount = 1 then
    Exit;

  // Verlaag criteriapanel met 1 criteriumhoogte
  pnlCriteria.Height := pnlCriteria.Height - pnlCriteriaLine01.Height;

  // Ruim hoogst genummerd criteriumpanel op
  c := FindComponent(Format('pnlCriteriaLine%.2d', [FCriteriaCount]));

  if Assigned(c) then
    c.Free
  else
    raise EEstSearchDialogException.Create('Cannot decrease nr of criteria');

  // Verlaag het aantal aanwezige criteria
  FCriteriaCount := FCriteriaCount - 1;

  // Werk de operators bij
  ShowOperators;

  // Stel knoppen in
  ToggleActionsAndControls;
end;

{|
  Procedure : TfrmSearchDialog.AddCriterium
  Auteur    : Erik Stok
  Doel      : Voeg criterium toe aan lijst van criteria
|}
procedure TFrmEstSearchDialog.AddCriterium(FieldIndex: Integer);
var
  PriorLine       : String;
  pnlCriteriaLine : TPanel;
  pnlCriterium    : TPanel;
  cbxField        : TComboBox;
  cbxComparison   : TComboBox;
  edtValue        : TEdit;
  pnlOperator     : TPanel;
  cbxOperator     : TComboBox;
begin
  // Voorkom overschrijden maximum
  if FCriteriaCount = FMaxCriteria then
    Exit;

  // Lock window om knipperen te voorkomen
  LockWindowUpdate(Handle);

  try
    // Bepaal vorige regel index
    PriorLine := Format('%.2d', [FCriteriaCount]);

    // Verhoog criteriapanel met 1 criteriumhoogte
    pnlCriteria.Height := pnlCriteria.Height + pnlCriteriaLine01.Height;

    // Maak nieuw criteriumpanel aan met alle componenten erin
    FCriteriaCount := FCriteriaCount + 1;

    pnlCriteriaLine := TPanel.Create(Self);
    with pnlCriteriaLine do
    begin
      Name := Format('pnlCriteriaLine%.2d', [FCriteriaCount]);
      Parent := pnlInput;
      Top := FCriteriaCount * pnlCriteriaLine01.Height;
      Height := pnlCriteriaLine01.Height;
      Align := pnlCriteriaLine01.Align;
      BevelOuter := pnlCriteriaLine01.BevelOuter;
      Caption := pnlCriteriaLine01.Caption;
    end;

    pnlCriterium := TPanel.Create(Self);
    with pnlCriterium do
    begin
      Name := Format('pnlCriterium%.2d', [FCriteriaCount]);
      Parent := pnlCriteriaLine;
      Align := pnlCriterium01.Align;
      BevelOuter := pnlCriterium01.BevelOuter;
      Caption := pnlCriterium01.Caption;
    end;

    pnlOperator := TPanel.Create(Self);
    with pnlOperator do
    begin
      Name := Format('pnlOperator%.2d', [FCriteriaCount]);
      Parent := pnlCriteriaLine;
      Width := pnlOperator01.Width;
      Height := pnlOperator01.Height;
      Align := pnlOperator01.Align;
      BevelOuter := pnlOperator01.BevelOuter;
      Caption := pnlOperator01.Caption;
    end;

    cbxField := TComboBox.Create(Self);
    with cbxField do
    begin
      Name := Format('cbxField%.2d', [FCriteriaCount]);
      Parent := pnlCriterium;
      Left := cbxField01.Left;
      Top := cbxField01.Top;
      Width := cbxField01.Width;
      Height := cbxField01.Height;
      Style := csDropDownList;
      Items := cbxField01.Items;

      // Als er geen veld is opgegeven of het veld met de betreffende index
      // bestaat niet, neem dan het veld van het vorige criterium. Neem anders
      // het veld met de betreffende index.
      if (FieldIndex = -1) or (FieldIndex >= FSearchDialogFieldList.Count) then
        ItemIndex := FieldComboBox(PriorLine).ItemIndex
      else
        ItemIndex := Items.IndexOfObject(FSearchDialogFieldList.Items[FieldIndex]);

      OnChange := cbxField01.OnChange;

      // Kopieer horizontale focuslocatie
      if Assigned(ActiveControl) and
         (Copy(ActiveControl.Name, 1, Length('cbxField')) = 'cbxField') then
        SetFocus;

    end;

    cbxComparison := TComboBox.Create(Self);
    with cbxComparison do
    begin
      Name := Format('cbxComparison%.2d', [FCriteriaCount]);
      Parent := pnlCriterium;
      Left := cbxComparison01.Left;
      Top := cbxComparison01.Top;
      Width := cbxComparison01.Width;
      Height := cbxComparison01.Height;
      Style := csDropDownList;
      Items := cbxComparison01.Items;
      ItemIndex := ComparisonComboBox(PriorLine).ItemIndex;
      OnChange := cbxComparison01.OnChange;

      // Kopieer horizontale focuslocatie
      if Assigned(ActiveControl) and
         (Copy(ActiveControl.Name, 1, Length('cbxComparison')) = 'cbxComparison') then
        SetFocus;

    end;

    edtValue := TEdit.Create(Self);
    with edtValue do
    begin
      Name := Format('edtValue%.2d', [FCriteriaCount]);
      Parent := pnlCriterium;
      Left := edtValue01.Left;
      Top := edtValue01.Top;
      Width := edtValue01.Width;
      Height := edtValue01.Height;
      Anchors := edtValue01.Anchors;
      Text := '';
      OnChange := edtValue01.OnChange;
      OnKeyDown := edtValue01.OnKeyDown;

      // Kopieer horizontale focuslocatie
      if Assigned(ActiveControl) and
         (Copy(ActiveControl.Name, 1, Length('edtValue')) = 'edtValue') then
        SetFocus;
    end;

    cbxOperator := TComboBox.Create(Self);
    with cbxOperator do
    begin
      Name := Format('cbxOperator%.2d', [FCriteriaCount]);
      Parent := pnlOperator;
      Left := cbxOperator01.Left;
      Top := cbxOperator01.Top;
      Width := cbxOperator01.Width;
      Height := cbxOperator01.Height;
      Style := csDropDownList;
      Items := cbxOperator01.Items;
      ItemIndex := OperatorComboBox(PriorLine).ItemIndex;

      // Kopieer horizontale focuslocatie
      if Assigned(ActiveControl) and
         (Copy(ActiveControl.Name, 1, Length('cbxOperator')) = 'cbxOperator') then
        SetFocus;

      // Stel operators in
      ShowOperators;

      // Stel knoppen in
      ToggleActionsAndControls;
    end;

    // Initialiseer controls
    InitControls(LineNr(cbxField));

  finally
    LockWindowUpdate(0);
  end;
end;

{|
  Procedure : TfrmSearchDialog.Search
  Auteur    : Erik Stok
  Doel      : Voer de zoekopdracht uit volgens opgegeven input waarden
|}
procedure TFrmEstSearchDialog.Search;
var
  Query : String;
begin
  // Wis eventuele vorige boodschap
  if FMaxMessageSet then
    stbInfo.SimpleText := '';

  // Controleer of max aantal is ingesteld. Indien dit het geval is, controleer
  // dan eerst het aantal.
  if FMax > 0 then
  begin
    // Stel statement samen
    Query := BuildQuery(FCountQuery);
    SetCountQuery(Query);

    try

      try
        // Open de query
        FCountDataSet.Open;
      except
        // Als dit misgaat, communiceer dan duidelijke melding naar de
        // gebruiker
        on e:Exception do
          raise EEstSearchDialogException.CreateFmt('Cannot determine count, ' +
                                                    'invalid count query (%s)'#13#10'%s',
                                                    [e.Message, Query]);
      end;

      // Veronderstel dat het eerste veld het aantal bevat. Als er geen
      // eerste veld is, koppel dat dan terug
      if FCountDataSet.Fields.Count > 0 then
      begin

        // Bepaal aantal
        try

          // Als aantal het maximum overschrijdt, toon dan een melding
          if FCountDataSet.Fields[0].AsInteger > FMax then
          begin
            stbInfo.SimpleText := Format(EstSearchDialogIntl.MessageTooManyRecords,
                                         [FCountDataSet.Fields[0].AsInteger,
                                          FMax]);

            FMaxMessageSet := True;

            // Sluit aantal query
            FCountDataSet.Close;

            // Stop verdere zoekactie
            Exit;
          end;

        except
          // Koppel onjuist aantal veldtype terug
          on e:Exception do
            raise EEstSearchDialogException.CreateFmt('Cannot determine count, invalid ' +
                                                      'count fieldtype (%s)', [e.Message]);
        end;

      end
      else
      begin
        // Koppel fout terug dat aantal veld mist
        raise EEstSearchDialogException.Create('Cannot find count field, search count ' +
                                               'query has an invalid format');
      end;

    finally
      // Sluit aantal query
      FCountDataSet.Close;
    end;

  end;

  // Bevries controls
  FSearchDataSet.DisableControls;

  try

    // Sluit eventueel vorig openstaand resultaat
    FSearchDataSet.Close;

    // Stel statement samen
    Query := BuildQuery(FSearchQuery);
    SetSearchQuery(Query);

    try

      // Open de query
      FSearchDataSet.Open;

      // Stel de velden in volgens specificatie
      SetFieldSettings;

      // Onthoud deze query als resultaat query
      FResultQuery := Query;

    except
      // Als dit misgaat, communiceer dan duidelijke melding naar de
      // gebruiker
      on e:Exception do
      begin

        raise EEstSearchDialogException.CreateFmt('Cannot determine result, invalid ' +
                                                  'search query (%s)'#13#10'%s',
                                                  [e.Message, Query]);

      end;

    end;

  finally
    // Ontdooi controls
    FSearchDataSet.EnableControls;
  end;

  // Stel knoppen in voor gevonden
  if not FSearchDataSet.IsEmpty then
    InitFound
  else
    InitSearchAgain;
end;

procedure TFrmEstSearchDialog.actSearchExecute(Sender: TObject);
var
  OldCursor : TCursor;
begin
  // Onthoud oude cursor
  OldCursor := Screen.Cursor;

  // Stel zandloper in
  Screen.Cursor := crHourGlass;

  try

    // Voer zoekopdracht uit
    Search;

  finally
    // Herstel oude cursor
    Screen.Cursor := OldCursor;
  end;

end;

procedure TFrmEstSearchDialog.FormCreate(Sender: TObject);
begin
  // Initialiseer privates
  FSearchStyle := ssModal;

  FMaxCriteria := 1;
  FCriteriaCount := 1;

  FSearchQuery := '';
  FCountQuery := '';
  FMax := 0;

  FregistryPath := '';

  FStore := dsNone;

  FFieldIndex := -1;

  FSearchDataSet := nil;
  FCountDataSet := nil;

  FOnSetSearchQuery := nil;
  FOnSetCountQuery := nil;

  FOnInitControls := nil;

  FOnOK := nil;
  FOnCancel := nil;

  FInitializing := False;
  FMaxMessageSet := False;

  FRegistryPath := '';

  FResultQuery := '';

  FCompareFormatDate := '';
  FCompareFormatDateTime := '';
  FCompareFormatTime := '';

  FQuoteChar := '''';
  FDecimalChar := '.';
  FQuotedDateTime := True;
  FAdditionalWhere := '';
  FStartOpen := False;
  FTrueExpression := '1=1';

  // Creeer privates
  FSearchDialogFieldList := TEstSearchDialogFieldList.Create(Self);

  // Stel dialog properties in
  if Assigned(FOnSetup) then
    FOnSetup(Self);

  // Stel captions in
  if FSearchStyle = ssModal then
  begin
    actOK.Caption := EstSearchDialogIntl.ButtonOK;
    actCancel.Caption := EstSearchDialogIntl.ButtonCancel;
  end
  else
  begin
    actOK.Caption := EstSearchDialogIntl.ButtonSelect;
    actCancel.Caption := EstSearchDialogIntl.ButtonClose;
  end;

  actMore.Caption := EstSearchDialogIntl.ButtonMore;
  actLess.Caption := EstSearchDialogIntl.ButtonLess;
  actSearch.Caption := EstSearchDialogIntl.ButtonSearch;

  lblField.Caption := EstSearchDialogIntl.LabelField;
  lblComparison.Caption := EstSearchDialogIntl.LabelComparison;
  lblValue.Caption := EstSearchDialogIntl.LabelValue;
  lblOperator.Caption := EstSearchDialogIntl.LabelOperator;

end;

procedure TFrmEstSearchDialog.FormDestroy(Sender: TObject);
begin
  // Ruim privates op
  FSearchDialogFieldList.Free;
end;

procedure TFrmEstSearchDialog.FormShow(Sender: TObject);
begin
  // Zodra het scherm getoond word kan het worden geinitialiseerd
  Initialise;
end;

{|
  Procedure : TfrmSearchDialog.SetFieldSettings
  Auteur    : Erik Stok
  Doel      : Stel velden in volgens opgegeven SearchDialogFieldList
|}
procedure TFrmEstSearchDialog.SetFieldSettings;
var
  i : Integer;
  f : TEstSearchDialogField;
  d : TField;
begin

  // Loop door alle velden
  for i := 0 to FSearchDataSet.FieldCount - 1 do
  begin

    // Controleer of er voor het veld instellingen zijn gedaan. Zo niet,
    // verberg het dan.
    f := FSearchDialogFieldList.ItemByFieldName(FSearchDataSet.Fields[i].FieldName);

    if Assigned(f) then
    begin

      // Stel veld in volgens specificaties
      FSearchDataSet.Fields[i].DisplayLabel := f.DisplayLabel;

      if f.DisplayWidth <> 0 then
        FSearchDataSet.Fields[i].DisplayWidth := f.DisplayWidth;

      SetStringPropertyByName(FSearchDataSet.Fields[i], 'DisplayFormat', f.DisplayFormat);

      FSearchDataSet.Fields[i].Visible := True;

    end
    else
    begin

      // Verberg veld
      FSearchDataSet.Fields[i].Visible := False;

    end;

  end;

  // Stel de veldvolgorde in
  for i := 0 to FSearchDialogFieldList.Count - 1 do
  begin
    f := TEstSearchDialogField(FSearchDialogFieldList.Items[i]);
    d := FSearchDataSet.FindField(f.FieldName);
    if Assigned(d) then
      d.Index := i;
  end;
end;

{|
  Procedure : TfrmSearchDialog.SetSearchDialogFieldList
  Auteur    : Erik Stok
  Doel      : Neem opgegeven zoekveldinstellingen over
|}
procedure TFrmEstSearchDialog.SetSearchDialogFieldList(Value: TEstSearchDialogFieldList);
begin
  FSearchDialogFieldList.Assign(Value);
end;

{|
  Procedure : TfrmSearchDialog.Initialise
  Auteur    : Erik Stok
  Doel      : Initialiseer alle scherminstellingen op basis van de properties
|}
procedure TFrmEstSearchDialog.Initialise;
var
  i : Integer;
  f : TEstSearchDialogField;
  FirstSearchIndex : Integer;
  so : TSearchOperator;
  tm : TTextMetric;
begin
  // Geef aan dat initalisatie bezig is
  FInitializing := True;

  try
    // Controleer instellingen van dataset
    if not Assigned(FSearchDataSet) then
      raise EEstSearchDialogException.Create('Search dataset not set');

    if not Assigned(FCountDataSet) then
      raise EEstSearchDialogException.Create('Count dataset not set');

    if not Assigned(FOnSetSearchQuery) then
      raise EEstSearchDialogException.Create('Search query set event not set');

    if not Assigned(FOnSetCountQuery) then
      raise EEstSearchDialogException.Create('Count query set event not set');

    // Stel datasource van het grid in
    dtsSearch.DataSet := FSearchDataSet;

    // Tot dusverre is er nog geem maximum boodschap getoond
    FMaxMessageSet := False;

    // Vul de combobox met velden waarop gezocht kan worden
    FirstSearchIndex := -1;
    for i := 0 to FSearchDialogFieldList.Count - 1 do
    begin
      f := TEstSearchDialogField(FSearchDialogFieldList.Items[i]);
      if f.Search then
      begin
        cbxField01.Items.AddObject(f.DisplayLabel, f);
        FirstSearchIndex := i;
      end;

      // Voeg kolom toe aan het grid
      with grdResult.Columns.Add do
      begin
         FieldName := f.FieldName;
         Title.Caption := f.DisplayLabel;

         // Als er een ColumnDisplayWidth gezet is, gebruik deze dan. Anders,
         // gebruik de DisplayWidth van het veld (door: Marc Geldon)
         if (f.DisplayColumnWidth  > 0) then
           Width := f.DisplayColumnWidth
         else
         begin
           GetTextMetrics(grdResult.Canvas.Handle, tm);
           Width := f.DisplayWidth * (grdResult.Canvas.TextWidth('0') - tm.tmOverhang) + tm.tmOverhang + 4;
         end;
      end;

    end;

    // Stel als standaard het eerste veld in
    if cbxField01.Items.Count > 0 then
      cbxField01.ItemIndex := cbxField01.Items.IndexOfObject(FSearchDialogFieldList.Items[FirstSearchIndex]);

    // Forceer change
    cbxFieldChange(cbxField01);

    // Wis zoekcriterium
    edtValue01.Text := '';

    // Vul operator combobox
    for so := Low(TSearchOperator) to High(TSearchOperator) do
      cbxOperator01.Items.AddObject(OperatorText(so), Pointer(so));

    // En selecteer 'and' standaard
    cbxOperator01.ItemIndex := 0;

    // Als er een initieel aantal criteria is, stel dat dan in
    if FCriteriaCount <> 1 then
    begin
      i := FCriteriaCount;
      FCriteriaCount := 1;
      while FCriteriaCount < i do
        AddCriterium(FCriteriaCount);
    end;

    // Toon operator alleen als er meerdere criteria mogelijk zijn. Idem voor
    // de meer en minder criteria actions.
    if FMaxCriteria > 1 then
    begin
      lblOperator.Visible := True;
      ShowOperators;
    end
    else
    begin
      lblOperator.Visible := False;
      pnlOperator01.Visible := False;
    end;

    actMore.Visible := FMaxCriteria > 1;
    actMore.Enabled := FMaxCriteria > 1;
    actLess.Visible := FMaxCriteria > 1;
    actLess.Enabled := FMaxCriteria > 1;

    // Stel knoppen in
    actOK.Enabled := False;
    ToggleActionsAndControls;

    // Herstel laatste zoekopdracht indien aangegeven
    if FStore <> dsNone then
      LoadSearchFields(FStore = dsFieldsAndValues);

    // Zet focus op waarde veld
    edtValue01.SetFocus;

    // Initialiseer basis controls
    InitControls(LineNr(cbxField01));

    // Start open indien aangegeven
    if FStartOpen then
      Search;

  finally
    // Geef aan dat initialisatie voorbij is
    FInitializing := False;
  end;
end;

procedure TFrmEstSearchDialog.cbxFieldChange(Sender: TObject);
var
  Line        : String;
  FieldType   : TFieldType;
  FieldObject : TEstSearchDialogField;
  n           : Integer;
  Comparison  : TSearchComparison;
  cbx         : TComboBox;
begin
  // Bepaal welke criteriaregel gebruik wordt
  cbx  := Sender as TComboBox;
  Line := LineNr(Sender);

  // Bepaal veld
  FieldObject := TEstSearchDialogField(cbx.Items.Objects[cbx.ItemIndex]);
  FieldType   := ftUnknown;

  with ComparisonComboBox(Line) do
  begin
    // Wis de combobox met vergelijkingen
    Items.Clear;

    // Bepaal veldtype van huidige selectie
    if cbx.ItemIndex <> -1 then
    begin
      FieldType := FieldObject.FieldType;

      // Vul met de voor dit veldtype geldige vergelijkingen
      for Comparison := Low(TSearchComparison) to High(TSearchComparison) do
      begin

        // Kijk of vergelijking geldig is voor dit veldtype. Zo ja, voeg deze dan
        // toe aan de combobox
        if Comparison in FieldTypeComparisons(FieldType) then
        begin
          // Voeg toe
          Items.AddObject(SearchComparisionText(Comparison), Pointer(Comparison));
        end;

      end;

    end;

    // Kies de standaard selectie. Als die niet geldig is probeer dan de
    // algemene regel
    if FieldObject.DefaultComparison in FieldTypeComparisons(FieldType) then
    begin
      n := Items.IndexOfObject(Pointer(FieldObject.DefaultComparison));
      ItemIndex := n;
    end
    else
    begin
      n := Items.IndexOfObject(Pointer(FieldTypeDefaultComparison(FieldType)));
      if n <> -1 then
        ItemIndex := n;
    end;

  end;

  // Wis waarde en stel kleur in
  ValueEdit(Line).Text := '';
  ColorValue(Line);

  // Initialiseer controls
  InitControls(Line);
end;

procedure TFrmEstSearchDialog.actOKExecute(Sender: TObject);
begin
  PerformOK;
end;

procedure TFrmEstSearchDialog.actCancelExecute(Sender: TObject);
begin
  PerformCancel;
end;

{|
  Procedure : TfrmSearchDialog.InitOpnieuwZoeken
  Auteur    : Erik Stok
  Doel      : Initialiseer knoppen voor opnieuw zoeken
|}
procedure TFrmEstSearchDialog.InitSearchAgain;
begin
  // Zet zoek knop als default
  btnSearch.Default := True;
  btnOK.Default := False;

  // Toggle OK aan de hand van het resultaat
  actOK.Enabled := (FSearchDataSet.Active) and (not FSearchDataSet.IsEmpty);
end;

{|
  Procedure : TfrmSearchDialog.InitFound
  Auteur    : Erik Stok
  Doel      : Initialiseer knoppen voor gevonden en selecteren
|}
procedure TFrmEstSearchDialog.InitFound;
begin
  // Toggle OK aan de hand van het resultaat
  actOK.Enabled := (FSearchDataSet.Active) and (not FSearchDataSet.IsEmpty);

  // Zet OK knop als default en focussed als er data is
  if actOK.Enabled then
  begin
    btnSearch.Default := False;
    btnOK.Default := True;

    if btnSearch.Focused then
      btnOK.Focused;
  end;
end;

procedure TFrmEstSearchDialog.edtValueChange(Sender: TObject);
begin
  InitSearchAgain;
end;

procedure TFrmEstSearchDialog.edtValueKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  // Als er met pijltjestoetsen of page up/down genavigeerd wordt in de
  // waarde editbox, navigeer dan in het grid.

  case Key of
    VK_UP:
      begin
        Key := 0;
        if not FSearchDataSet.Bof then
          FSearchDataSet.Prior;
      end;

    VK_DOWN:
      begin
        Key := 0;
        if not FSearchDataSet.Eof then
          FSearchDataSet.Next;
      end;

    VK_PRIOR:
      begin
        Key := 0;
        if not FSearchDataSet.Bof then
          FSearchDataSet.MoveBy(-1 * TDBGridAccessClass(grdResult).VisibleRowCount);
      end;

    VK_NEXT:
      begin
        Key := 0;
        if not FSearchDataSet.Eof then
          FSearchDataSet.MoveBy(TDBGridAccessClass(grdResult).VisibleRowCount);
      end;

  end;
end;

procedure TFrmEstSearchDialog.actMoreExecute(Sender: TObject);
begin
  AddCriterium;
end;

procedure TFrmEstSearchDialog.actLessExecute(Sender: TObject);
begin
  DeleteCriterium;
end;

{|
  Procedure : TfrmEstSearchDialog.ShowOperators
  Auteur    : Erik Stok
  Doel      : Toon de operator comboboxen overeenkomstig het aantal criterium
              regels, waarbij op de laatste regel natuurlijk geen operator
              gekozen moet kunnen worden.
|}
procedure TFrmEstSearchDialog.ShowOperators;
var
  i : Integer;
begin
  // Doe niets als er maar 1 criterium getoond mag worden
  if FMaxCriteria <= 1 then
    Exit;

  // Verberg label als er maar 1 criterium is (door: Marc Geldon)
  if (FCriteriaCount = 1) then
     lblOperator.Hide
  else
     lblOperator.Show;

  // Loop anders door de operators en toon deze, behalve de laatste
  for i := 1 to FCriteriaCount - 1 do
    OperatorComboBox(Format('%.2d', [i])).Visible := True;

  OperatorComboBox(Format('%.2d', [FCriteriaCount])).Visible := False;

end;

{|
  Procedure : TfrmEstSearchDialog.GetResultList
  Auteur    : Erik Stok
  Doel      : Geef resultaatrecord in stringlist
|}
function TFrmEstSearchDialog.GetResultList: String;
var
  i : Integer;
  l : TStringList;
begin
  // Alleen als er resultaat is wordt dat teruggegeven
  if (FSearchDataSet.Active) and (not FSearchDataSet.IsEmpty) then
  begin

    // Maak lijst om resultaat in op te bouwen
    l := TStringList.Create;

    try

      // Loop door fields en vis alle stringwaarden er uit
      for i := 0 to FSearchDataSet.Fields.Count - 1 do
        l.Add(Format('%s=%s', [FSearchDataSet.Fields[i].FieldName,
                               FSearchDataSet.Fields[i].AsString]));

      // En geef dat terug
      Result := l.Text;

    finally
      // Ruim de lijst op
      l.Free;
    end;

  end
  else
  begin
    // Geef niet terug bij geen resultaat
    Result := '';
  end;
end;

{|
  Procedure : TfrmEstSearchDialog.GetResultQuery
  Auteur    : Erik Stok
  Doel      : Geef query waarmee het resultaat bepaald is
|}
function TFrmEstSearchDialog.GetResultQuery: String;
begin
  Result := FResultQuery;
end;

{|
  Procedure : TfrmEstSearchDialog.LoadSearchFields
  Auteur    : Erik Stok
  Doel      : Laad instellingen laatste zoekactie
|}
procedure TFrmEstSearchDialog.LoadSearchFields(IncludeValues: Boolean);
var
  r : TRegistry;
  i : Integer;
  Line : String;
  Count : Integer;

  function ReadItemIndexFromRegistry(ValueName: String; MaxValue: Integer): Integer;
  begin
    // Lees index
    if r.ValueExists(ValueName) then
      Result := r.ReadInteger(ValueName)
    else
      Result := -1;

    // Als index meer dan de maximaal toegstane waarde is, geef dan ook -1
    if Result > MaxValue then
      Result := -1;
  end;

begin

  // Maak referentie naar registry aan
  r := TRegistry.Create;

  try

    // Lees uit current user
    r.RootKey := HKEY_CURRENT_USER;
    r.Access := KEY_READ;

    // Alleen als sleutel bestaat hoeft gelezen te worden
    if r.OpenKeyReadOnly(FRegistryPath) then
    begin

      // Lees aantal criteria
      if r.ValueExists(REG_SDL_COUNT) then
        Count := r.ReadInteger(REG_SDL_COUNT)
      else
        Exit;

      // Ga niet over het maximum
      if Count > FMaxCriteria then
        Count := FMaxCriteria;

      // Maak regels aan indien deze niet reeds bestaan
      while not Assigned(FindComponent(Format('pnlCriteriaLine%.2d', [Count]))) do
        AddCriterium;

      // Ruim een eventueel teveel aan regels op
      while Assigned(FindComponent(Format('pnlCriteriaLine%.2d', [Count + 1]))) do
        DeleteCriterium;

      // Lees de criteria instellingen
      for i := 1 to FCriteriaCount do
      begin

        // Bepaal regelnummer
        Line := Format('%.2d', [i]);

        // Lees veld
        with FieldComboBox(Line) do
        begin
          ItemIndex := ReadItemIndexFromRegistry(REG_SDL_FIELD + Line, Items.Count - 1);

          // Forceer change
          if ItemIndex <> -1 then
            cbxFieldChange(FieldComboBox(Line));
        end;

        // Lees comparison
        with ComparisonComboBox(Line) do
        begin
          ItemIndex := ReadItemIndexFromRegistry(REG_SDL_COMPARISON + Line, Items.Count - 1);

          // Forceer change
          if ItemIndex <> -1 then
            cbxComparisonChange(ComparisonComboBox(Line));
        end;

        // Lees value indien aangegeven
        if IncludeValues then
          ValueEdit(Line).Text := r.ReadString(REG_SDL_VALUE + Line);

        // Lees operator als het niet de laatste regel is
        if i < FCriteriaCount then
        begin
          with OperatorComboBox(Line) do
            ItemIndex := ReadItemIndexFromRegistry(REG_SDL_OPERATOR + Line, Items.Count - 1);
        end;

      end;

    end;

  finally
    r.Free;
  end;
end;

{|
  Procedure : TfrmEstSearchDialog.SaveSearchFields
  Auteur    : Erik Stok
  Doel      : Sla deze zoekactie op als laatste zoekactie
|}
procedure TFrmEstSearchDialog.SaveSearchFields(IncludeValues: Boolean);
var
  r : TRegistry;
  i : Integer;
  Line : String;
  l : TStringList;
begin
  // Maak referentie naar registry aan
  r := TRegistry.Create;

  try

    // Lees uit current user
    r.RootKey := HKEY_CURRENT_USER;
    r.Access := KEY_ALL_ACCESS;

    // Alleen als sleutel aangemaakt kan worden wordt er weggeschreven
    if r.OpenKey(FRegistryPath, True) then
    begin

      // Gooi alle values uit de key
      l := TStringList.Create;
      try
        r.GetValueNames(l);
        while l.Count > 0 do
        begin
          r.DeleteValue(l[0]);
          l.Delete(0);
        end;
      finally
        l.Free;
      end;

      // Schrijf aantal criteria
      r.WriteInteger(REG_SDL_COUNT, FCriteriaCount);

      // Schrijf de criteria instellingen
      for i := 1 to FCriteriaCount do
      begin

        // Bepaal regelnummer
        Line := Format('%.2d', [i]);

        // Schrijf veld
        with FieldComboBox(Line) do
          r.WriteInteger(REG_SDL_FIELD + Line, ItemIndex);

        // Schrijf comparison
        with ComparisonComboBox(Line) do
          r.WriteInteger(REG_SDL_COMPARISON + Line, ItemIndex);

        // Schrijf value indien aangegeven
        if IncludeValues then
          r.writeString(REG_SDL_VALUE + Line, ValueEdit(Line).Text);

        // Schrijf operator als het niet de laatste regel is
        if i < FCriteriaCount then
        begin
          with OperatorComboBox(Line) do
            r.WriteInteger(REG_SDL_OPERATOR + Line, ItemIndex);
        end;

      end;

    end;

  finally
    r.Free;
  end;
end;

{|
  Procedure : TFrmEstSearchDialog.SetSearchQuery
  Auteur    : Erik Stok
  Doel      : Stel search query in op aanroepen component
|}
procedure TFrmEstSearchDialog.SetSearchQuery(SQL: String);
begin
  if Assigned(FOnSetSearchQuery) then
    FOnSetSearchQuery(Self, SQL);
end;

{|
  Procedure : TFrmEstSearchDialog.SetCountQuery
  Auteur    : Erik Stok
  Doel      : Stel count query in op aanroepen component
|}
procedure TFrmEstSearchDialog.SetCountQuery(SQL: String);
begin
  if Assigned(FOnSetCountQuery) then
    FOnSetCountQuery(Self, SQL);
end;

procedure TFrmEstSearchDialog.cbxComparisonChange(Sender: TObject);
begin
  // Als de operation wordt gewijzigd, controleer dan of dit naar empty
  // is. Zo ja, geef dan door de tekst in de edit box te dimmen aan dat
  // de waarde genegeerd wordt
  ColorValue(LineNr(Sender));

  // Initialiseer controls
  InitControls(LineNr(Sender));
end;

procedure TFrmEstSearchDialog.grdResultDblClick(Sender: TObject);
var
  p : TPoint;
  g : TGridCoord;
begin
  // Bepaal muiscoordinaat waar dubbelklik plaatsvond
  p := grdResult.ScreenToClient(Mouse.CursorPos);

  // Bepaal cel onder muis
  g := grdResult.MouseCoord(p.x, p.y);

  // Dubbelklik was geldig als er een rij staat en OK is enabled
  if (g.Y <> -1) and (actOK.Enabled) then
    PerformOK;

end;

{|
  Procedure : TFrmEstSearchDialog.PerformOK
  Auteur    : Erik Stok
  Doel      : Voer OK uit (OK knop of dubbelklik)
|}
procedure TFrmEstSearchDialog.PerformOK;
begin
  // Geef OK als resultaat voor modal form. Anders voer OnOK uit.
  if FSearchStyle = ssModal then
  begin
    ModalResult := mrOK;
  end
  else
  begin
    if Assigned(FOnOK) then
      FOnOK(Self);
  end;
end;

{|
  Procedure : TFrmEstSearchDialog.PerformCancel
  Auteur    : Erik Stok
  Doel      : Voer Cancel uit (alleen knop)
|}
procedure TFrmEstSearchDialog.PerformCancel;
begin
  // Geef cancel als resultaat voor modal form. Sluit anders het form.
  if FSearchStyle = ssModal then
    ModalResult := mrCancel
  else
    Close;
end;

constructor TFrmEstSearchDialog.Create(AOwner: TComponent; Mdi: Boolean; OnSetup: TEstSearchDialogSetupEvent);
begin
  FOnSetup := OnSetup;
  FMdi := Mdi;

  inherited Create(AOwner);
end;

procedure TFrmEstSearchDialog.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  // Als het een niet modaal scherm betreft en er is een oncancel event
  // gekoppeld, trigger dan de oncancel
  if (FSearchStyle <> ssModal) and Assigned(FOnCancel) then
    FOnCancel(Self);

  if (FSearchStyle = ssModal) and (not (ModalResult in [mrOK, mrCancel])) then
    ModalResult := mrCancel;

  // Sla zoekopdracht op indien opgegeven en niet designtime
  if (FStore <> dsNone) and (not (csDesigning	in ComponentState)) then
    SaveSearchFields(FStore = dsFieldsAndValues);

  // Ruim non modal form zelf op
  if FSearchStyle <> ssModal then
    Action := caFree;

end;

{|
  Procedure : TFrmEstSearchDialog.ColorValue
  Auteur    : Erik Stok
  Doel      : Stel de tekstkleur van de value editbox in aan de hand van de
              vergelijking voor de huidige regel
|}
procedure TFrmEstSearchDialog.ColorValue(Line: String);
begin
  with ComparisonComboBox(Line) do
  begin
    if TSearchComparison(Items.Objects[ItemIndex]) in [scEmpty, scNotempty] then
      ValueEdit(Line).Font.Color := clGrayText
    else
      ValueEdit(Line).Font.Color := clWindowText;
  end;
end;

{
  Procedure : TFrmEstSearchDialog.InitControls
  Auteur    : Erik Stok
  Doel      : Initialiseer de controls gebaseerd op de huidige selectie
}

procedure TFrmEstSearchDialog.InitControls(Line: String);
var
  cbxField        : TComboBox;
  cbxComparison   : TComboBox;
  edtValue        : TEdit;
  cbxOperator     : TComboBox;
begin
  cbxField := FieldComboBox(Line);
  cbxComparison := ComparisonComboBox(Line);
  edtValue := ValueEdit(Line);
  cbxOperator := OperatorComboBox(Line);

  DoInitControls(Self, TEstSearchDialogField(cbxField.Items.Objects[cbxField.ItemIndex]),
    cbxField, TSearchComparison(cbxComparison.Items.Objects[cbxComparison.ItemIndex]),
    cbxComparison, edtValue, cbxOperator.Visible, cbxOperator);

end;

{
  Procedure : TFrmEstSearchDialog.DoInitControls
  Auteur    : Erik Stok
  Doel      : Roep de InitControls event handler aan
}

procedure TFrmEstSearchDialog.DoInitControls(Sender: TObject;
  Field: TEstSearchDialogField; FieldControl: TComboBox;
  Comparison: TSearchComparison; ComparisonControl: TComboBox;
  ValueControl: TEdit; UsingOperator: Boolean; OperatorControl: TComboBox);
begin
  // Als de init controls event handler is gekoppeld, roep die dan aan
  // met de betreffende controls en waarden
  if Assigned(FOnInitControls) then
  begin
    FOnInitControls(Self, Field, FieldControl, Comparison, ComparisonControl,
      ValueControl, UsingOperator, OperatorControl);
  end;
end;

procedure TFrmEstSearchDialog.DoCreate;
begin
  inherited;

  // Zet formstyle op mdi indien van toepassing
  if FMdi then
    FormStyle := fsMDIChild;
end;

end.
