Unit StringTokenizer;

{*****************************************************************************} 
{                                                                              
{ Helper Class for Panoramo Component for CLX
{
{ Copyright (c) 2002 MindLink Software GmbH
{
{ This file may be distributed and/or modified under the terms of the GNU
{ General Public License version 2 as published by the Free Software
{ Foundation and appearing at http://www.gnu.org/copyleft/gpl.html.
{
{ http://www.mind-link.com
{*****************************************************************************}

  Interface

Uses sysutils, classes;


(* Pascal port of
 * @(#)StringTokenizer.java	
 *)



(**
 * The string tokenizer class allows an application to break a 
 * string into tokens. The tokenization method is much simpler than 
 * the one used by the <code>StreamTokenizer</code> class. The 
 * <code>StringTokenizer</code> methods do not distinguish among 
 * identifiers, numbers, and quoted strings, nor do they recognize 
 * and skip comments.
 * <p>
 * The set of delimiters (the characters that separate tokens) may 
 * be specified either at creation time or on a per-token basis. 
 * <p>
 * An instance of <code>StringTokenizer</code> behaves in one of two 
 * ways, depending on whether it was created with the 
 * <code>returnTokens</code> flag having the value <code>true</code> 
 * or <code>false</code>: 
 * <ul>
 * <li>If the flag is <code>false</code>, delimiter characters serve to 
 *     separate tokens. A token is a maximal sequence of consecutive 
 *     characters that are not delimiters. 
 * <li>If the flag is <code>true</code>, delimiter characters are themselves 
 *     considered to be tokens. A token is thus either one delimiter 
 *     character, or a maximal sequence of consecutive characters that are 
 *     not delimiters.
 * </ul><p>
 * A <tt>StringTokenizer</tt> object internally maintains a current
 * position within the string to be tokenized. Some operations advance this 
 * current position past the characters processed.<p>
 * A token is returned by taking a substring of the string that was used to 
 * create the <tt>StringTokenizer</tt> object.
 * <p>
 * The following is one example of the use of the tokenizer. The code:
 * <blockquote><pre>
 *     StringTokenizer st = new StringTokenizer("this is a test");
 *     while (st.hasMoreTokens()) {
 *         println(st.nextToken());
 *     }
 * </pre></blockquote>
 * <p>
 * prints the following output:
 * <blockquote><pre>
 *     this
 *     is
 *     a
 *     test
 * </pre></blockquote>
 *
 * @author  unascribed
 * @version 1.19, 03/18/98
 * @see     java.io.StreamTokenizer
 * @since   JDK1.0
 *)
Type
  TStringTokenizer = Class
  Private
    currentPosition: Integer;
    maxPosition: Integer;
    str: String;
    delimiters: String;
    retTokens: Boolean;
    Procedure skipDelimiters;

  Public
    Constructor Create(str: String; delim: String; returnTokens: Boolean); Overload;
    Constructor Create(str: String; delim: String); Overload;
    Constructor Create(str: String); Overload;
    Function hasMoreTokens: Boolean;
    Function nextToken: String; Overload;
    Function nextToken(delim: String): String; Overload;
    Function hasMoreElements: Boolean;
    Function nextElement: String;
    Function countTokens: Integer;
  End;

Implementation
    (**
     * Constructs a string tokenizer for the specified string. All  
     * characters in the <code>delim</code> argument are the delimiters 
     * for separating tokens. 
     * <p>
     * If the <code>returnTokens</code> flag is <code>true</code>, then 
     * the delimiter characters are also returned as tokens. Each 
     * delimiter is returned as a string of length one. If the flag is 
     * <code>false</code>, the delimiter characters are skipped and only
     * serve as separators between tokens.
     *
     * @param   str            a string to be parsed.
     * @param   delim          the delimiters.
     * @param   returnTokens   flag indicating whether to return the delimiters
     *                         as tokens.
     *)
Constructor TStringTokenizer.Create(str: String; delim: String; returnTokens: Boolean);
Begin
  currentPosition := 1;
  self.str := str;
  maxPosition := length(str) + 1;
  delimiters := delim;
  retTokens := returnTokens;
End;

    (**
     * Constructs a string tokenizer for the specified string. The
     * characters in the <code>delim</code> argument are the delimiters
     * for separating tokens. Delimiter characters themselves will not
     * be treated as tokens.
     *
     * @param   str     a string to be parsed.
     * @param   delim   the delimiters.
     *)
Constructor TStringTokenizer.Create(str: String; delim: String);
Begin
  Create(str, delim, False);
End;

    (**
     * Constructs a string tokenizer for the specified string. The
     * tokenizer uses the default delimiter set, which is
     * <code>"&#92;t&#92;n&#92;r&#92;f"</code>: the space character, the tab
     * character, the newline character, the carriage-return character,
     * and the form-feed character. Delimiter characters themselves will
     * not be treated as tokens.
     *
     * @param   str   a string to be parsed.
     *)
Constructor TStringTokenizer.Create(str: String);
Begin
  Create(str, #10 + #13 { \t\n\r\f'}, False);
End;

    (**
     * Skips delimiters.
     *)
Procedure TStringTokenizer.skipDelimiters;
Begin
  While (Not retTokens And
    (currentPosition < maxPosition) And
    (pos(str[currentPosition], delimiters) > 0)) Do
   Begin
    inc(currentPosition);
   End;
End;

    (**
     * Tests if there are more tokens available from this tokenizer's string. 
     * If this method returns <tt>true</tt>, then a subsequent call to 
     * <tt>nextToken</tt> with no argument will successfully return a token.
     *
     * @return  <code>true</code> if and only if there is at least one token 
     *          in the string after the current position; <code>false</code> 
     *          otherwise.
     *)
Function TStringTokenizer.hasMoreTokens: Boolean;
Begin
  skipDelimiters();
  Result := (currentPosition < maxPosition);
End;

    (**
     * Returns the next token from this string tokenizer.
     *
     * @return     the next token from this string tokenizer.
     * @exception  NoSuchElementException  if there are no more tokens in this
     *               tokenizer's string.
     *)
Function TStringTokenizer.nextToken: String;
Var
  start: Integer;
Begin
  skipDelimiters();

  If (currentPosition >= maxPosition) Then
   Begin
    Exception.Create('NoSuchElementException()');;
   End;

  start := currentPosition;
  While ((currentPosition < maxPosition) And
    (pos(str[currentPosition], delimiters) < 1)) Do
   Begin
    inc(currentPosition);
   End;
  If (retTokens And (start = currentPosition) And
    (pos(str[currentPosition], delimiters) > 0)) Then
   Begin
    inc(currentPosition);
   End;
  Result := copy(str, start, currentPosition - start);
End;

    (**
     * Returns the next token in this string tokenizer's string. First,
     * the set of characters considered to be delimiters by this
     * <tt>StringTokenizer</tt> object is changed to be the characters in
     * the string <tt>delim</tt>. Then the next token in the string
     * after the current position is returned. The current position is
     * advanced beyond the recognized token.  The new delimiter set
     * remains the default after this call.
     *
     * @param      delim   the new delimiters.
     * @return     the next token, after switching to the new delimiter set.
     * @exception  NoSuchElementException  if there are no more tokens in this
     *               tokenizer's string.
     *)
Function TStringTokenizer.nextToken(delim: String): String;
Begin
  delimiters := delim;
  Result := nextToken();
End;

    (**
     * Returns the same value as the <code>hasMoreTokens</code>
     * method. It exists so that this class can implement the
     * <code>Enumeration</code> interface.
     *
     * @return  <code>true</code> if there are more tokens;
     *          <code>false</code> otherwise.
     * @see     java.util.Enumeration
     * @see     java.util.StringTokenizer#hasMoreTokens()
     *)
Function TStringTokenizer.hasMoreElements: Boolean;
Begin
  Result := hasMoreTokens();
End;

    (**
     * Returns the same value as the <code>nextToken</code> method,
     * except that its declared return value is <code>Object</code> rather than
     * <code>String</code>. It exists so that this class can implement the
     * <code>Enumeration</code> interface.
     *
     * @return     the next token in the string.
     * @exception  NoSuchElementException  if there are no more tokens in this
     *               tokenizer's string.
     * @see        java.util.Enumeration
     * @see        java.util.StringTokenizer#nextToken()
     *)
Function TStringTokenizer.nextElement: String;
Begin
  Result := nextToken();
End;

    (**
     * Calculates the number of times that this tokenizer's
     * <code>nextToken</code> method can be called before it generates an
     * exception. The current position is not advanced.
     *
     * @return  the number of tokens remaining in the string using the current
     *          delimiter set.
     * @see     java.util.StringTokenizer#nextToken()
     *)
Function TStringTokenizer.countTokens: Integer;
Var
  Count: Integer;
  currpos: Integer;
  start: Integer;
Begin
  Count := 0;
  currpos := currentPosition;

  While (currpos < maxPosition) Do
   Begin
      (*
	     * This is just skipDelimiters(); but it does not affect
	     * currentPosition.
	     *)
    While (Not retTokens And
      (currpos < maxPosition) And
      (pos(str[currpos], delimiters) > 0)) Do
     Begin
      inc(currpos);
     End;

    If (currpos >= maxPosition) Then
     Begin
      break;
     End;

    start := currpos;
    While ((currpos < maxPosition) And
      (pos(str[currpos], delimiters) <= 0)) Do
     Begin
      inc(currpos);
     End;
    If (retTokens And (start = currpos) And
      (pos(str[currpos], delimiters) >= 1)) Then
     Begin
      inc(currpos);
     End;
    inc(Count);
   End;
  Result := Count;
End;

End.
