Unit CLXPano;

{*****************************************************************************}
{                                                                              
{ Panorama Component for CLX Delphi 6 + Kylix 2
{ V 1.0 February 2002
{
{ Copyright (c) 2002 MindLink Software GmbH
{ http://www.mind-link.com
{
{ The contents of this file are used with permission, subject to   }
{ the Mozilla Public License Version 1.1 (the "License"); you may  }
{ not use this file except in compliance with the License. You may }
{ obtain a copy of the License at                                  }
{ http://www.mozilla.org/MPL/MPL-1.1.html                          }
{                                                                  }
{ Software distributed under the License is distributed on an      }
{ "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or   }
{ implied. See the License for the specific language governing     }
{ rights and limitations under the License.
{
{ Main code is a port of Janorama.java from Sven Meier
{ http://www.meiers.org/sven/
{*****************************************************************************}


Interface

Uses
 {$IFDEF LINUX}
  Libc,
{$ELSE}
  Windows,
{$ENDIF}

  SysUtils, Types, Classes, QForms,QGraphics, QControls, QDialogs,QT,
  Math, StringTokenizer;


Const
  MaxPixelCount = maxint Div 4;

Type

  TRGBPixelArray = Array[0..MaxPixelCount - 1] Of Integer;
  pRGBPixelArray = ^TRGBPixelArray;

  TSpotClickEvent=procedure (sender:TObject;spotnr:integer) of object;



  TSpot = Record
    Stype: (stpoly);
    Vertices: Array Of Array Of Integer;
  End;

  TMLVRThread = Class;


  TCLXPano = Class(TCustomControl)

  Private

    { Private-Deklarationen }

    imageWidth: Integer;

    imageHeight: Integer;

    fhorizon: Integer;

    imageRadius: Double;

    imageXPos: Integer;

    imageYPos: Integer;



    maxZoom: Double;

    imageZoom: Double;

    imageXs: Array Of Integer;

    imageYs: Array Of Integer;

    renderWidth: Integer;

    renderHeight: Integer;

    renderWidthHalf: Integer;

    renderHeightHalf: Integer;

    rerender: Boolean;

    factive: Boolean;

    prepared: Boolean;

    reprepare: Boolean;

    prepareString: String;

    errorString: String;

    mouseXPos: Integer;

    mouseYPos: Integer;

    mouseXStart: Integer;

    mouseYStart: Integer;

    dragging: Boolean;

    actualSpot: Integer;

    showSpots: Boolean;

    spots: TList;

    Fonpaint: TNotifyEvent;

    moveZoom: Double;

    atime: Integer;

    thread: TMLVRThread;

    Fspotdata: TStrings;

    fspotcolor:Tcolor;

    fonspotclick:TSpotClickEvent;

    fspotcursor:TCursor;
    normcursor:TCursor;

    FImage: TFilename;

    imagePixels: TBitmap;

    Procedure prepare;

    Procedure prepareImage;
    Procedure prepareRender;
    Procedure prepareSpots;
    Procedure paintSpots;

    Procedure checkMove;
    Procedure checkCursor;
    Procedure checkSpots;

    Procedure paintImage;
    Procedure DoSmooth;

    Procedure Setonpaint(Const Value: TNotifyEvent);


    Procedure Sethorizon(Const Value: Integer);
    Procedure Setspotdata(Const Value: TStrings);
    Procedure SetImage(Const Value: TFilename);

    Procedure SetSpotColor(Const Value: TColor);
    Procedure SetOnSpotClick(Const Value:TSpotClickEvent);

    Procedure SetSpotCursor(Const Value:TCursor);

    Procedure SetActive(Const Value: Boolean);

    procedure clearSpots;

  Protected

    Procedure paint; Override;

    function WidgetFlags: Integer; override;

    Procedure KeyPress(Var Key: Char); Override;
    Procedure KeyDown(Var Key: Word; Shift: TShiftState); Override;
    Procedure KeyUP(Var Key: Word; Shift: TShiftState); Override;
    Procedure MouseMove(Shift: TShiftState; X, Y: Integer); Override;
    Procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
      Override;
    Procedure MouseUP(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); Override;

    Procedure Loaded; Override;
    Procedure init;
    Procedure start;
    Procedure stop;

    { Protected-Deklarationen }


  Public

    { Public-Deklarationen }

    moveX: Integer;
    moveY: Integer;
    bufferImage: TBitmap;

    Destructor Destroy; Override;

    Constructor Create(AOwner: TComponent); Override;


  Published
    { Published-Deklarationen }

    Property OnPaint: TNotifyEvent Read Fonpaint Write Setonpaint;
    Property Image: TFilename Read FImage Write SetImage;
    Property Horizon: Integer Read Fhorizon Write Sethorizon;
    Property SpotData: TStrings Read Fspotdata Write Setspotdata;
    Property SpotColor:TColor Read fspotcolor write SetSpotColor;
    Property OnSpotClick:TSpotClickEvent Read fonspotclick write SetOnSpotClick;
    Property SpotCursor:TCursor Read fspotcursor write SetSpotCursor;
    Property Active: Boolean Read Factive Write SetActive;

  End;

  TMLVRThread = Class(TThread)

  Private
    aowner: TCLXPano;

    { Private-Deklarationen }
  Protected
    Procedure Execute; Override;
    Procedure dopaint;
  Public
    Constructor Create(suspended: Boolean; owner: TCLXPano);
  End;



Procedure Register;
    {$R CLXPano.dcr}

Implementation


Function Rotate90DegreesCounterClockwise(Const Bitmap: Tbitmap): TBitmap;
Var
  i: Integer;
  j: Integer;
  rowIn: pRGBPixelArray;
Begin
  Result := TBitmap.Create;
  Result.Width := Bitmap.Height;
  Result.Height := Bitmap.Width;
  Result.PixelFormat := Bitmap.PixelFormat;    // only pf32bit for now


  For  j := 0 To Bitmap.Height - 1 Do
   Begin
    rowIn := Bitmap.ScanLine[j];
    For i := 0 To Bitmap.Width - 1 Do
      pRGBPixelArray(Result.ScanLine[Bitmap.Width - i - 1])[j] := rowIn[i]
   End
End {Rotate90DegreesCounterClockwise};



Function GetTickCount: Cardinal;
Begin
  {$IFDEF LINUX}
  Result := clock Div (CLOCKS_PER_SEC Div 1000);
  {$ELSE}
  Result := Windows.GetTickCount;
  {$ENDIF}
End;



(***

 * Class for viewing panoramic pictures.

 *

 * Terminology:

 *  xxx_ means a variable with 16 bits right from the decimal point

 *  _xxx means a temporarily variable

 *)

Constructor TCLXPano.Create(AOwner: TComponent);
Begin
  controlstyle := controlstyle + [csopaque];

  Inherited Create(aowner);

  brush.color := clblack;

  fspotdata := TStringList.Create;

  fspotcolor:=clred;

  fspotcursor:=crHandPoint;

  Thread := TMLVrThread.Create(True, self);
End;


Destructor TCLXPano.Destroy;
Begin
  factive := False;

  reprepare := False;

  ClearSpots;

  imagepixels.Free;
  bufferImage.Free;

  Fspotdata.Free;
  Thread.Free;
  spots.free;
  Inherited Destroy;
End;


function TCLXPano.WidgetFlags: Integer;
begin
  // To reduce flickering on LINUX
  Result := Inherited WidgetFlags
  or Integer(WidgetFlags_WRepaintNoErase)
   or Integer(WidgetFlags_WResizeNoErase);
end;


Procedure TCLXPano.init;
Begin    // prepare
  prepare;
End;


Procedure TCLXPano.start;
Begin
  Thread.Resume;
End;



Procedure TCLXPano.stop;
Begin
  Thread.Suspend;
End;


Procedure TCLXPano.prepareImage;
Var
  image: Tbitmap;
  pimagePixels:pRGBPixelArray;
  x,y,value,pixel,_pixel,__pixel:integer;
Begin
  If fimage = '' Then
    exit;

  If imagepixels <> Nil Then
    imagepixels.Free;

  prepareString := 'Loading image.';

  invalidate;

  image := TBitmap.Create;

  image.loadfromfile(fimage);
  
  // set parameters of image

  image.PixelFormat := pf32bit;

  imageWidth := image.Width;
  imageHeight := image.Height;

  If ((imageWidth < 1) Or (imageHeight < 1)) Then
    Exception.Create('Could not load image '+fimage);

 if not (csloading in componentstate) then  fhorizon := imageHeight Div 2;

  imageRadius := imageWidth / (2 * PI);

  imageXPos := 0;

  imageYPos := imageHeight Div 2;

  imageZoom := 1.0;

  prepareString := 'Preparing image.';

  //imagepixels := Rotate90DegreesCounterClockwise(image);

  pimagePixels:=image.ScanLine[0];

     // swap image x and y
     for pixel:=imageWidth * imageHeight - 1 downto 0 do
     begin
    	pimagePixels[pixel] := pimagePixels[pixel] and $00ffffff;
   end;

    for pixel:=imageWidth * imageHeight - 1 downto 0 do
    begin
       value := pimagePixels[pixel];
        if ((value and $ff000000) = 0) then
        begin
        _pixel := pixel;
        repeat
          x := _pixel div imageHeight;
          y := _pixel mod imageHeight;

          __pixel := x + y * imageWidth;

          if (__pixel = pixel)
           then
            begin
             pimagePixels[_pixel] := value or $ff000000;
            break;
            end;

          pimagePixels[_pixel] := pimagePixels[__pixel] or $ff000000;
          _pixel := __pixel;
          until (0<0)
        end;
      end;

  imagepixels:=image;  //image.Free;
End;


(**

 * Prepare Render.

 *)

Procedure TCLXPano.prepareRender;
Var
  screenmax, screenx, imageX, imageY_: Integer;
  alpha, screenDist: Double;
Begin
  prepareString := 'Preparing render engine.';

  invalidate;

  // set parameters of screen
  renderWidth := Width;
  renderHeight := Height;

  If (renderHeight > imageHeight) Then
   Begin
    renderHeight := imageHeight;
   End;
  renderWidthHalf := renderWidth Div 2;
  renderHeightHalf := renderHeight Div 2;

  maxZoom := imageHeight / (renderHeight + 1);
  imageZoom := maxZoom;

  // get pixels for the screen and create image of them

  // calculate imageXs & imageYs
  screenMax := trunc(renderWidthHalf * maxZoom);

  setlength(imageXs, screenMax + 1);

  setlength(imageYs, screenMax + 1);


  For screenX := 0 To screenMax Do
   Begin
    // Calculate angle alpha (in radian) from the center to screenX.

    alpha := arctan((screenX + 0.5) / imageRadius);



    // Calculate offset imageX, seen in angle alpha from the center.

    imageX := trunc(alpha * imageRadius);



    // Calculate distance screenDist to screenX.

    screenDist := imageRadius / cos(alpha);



    // Calculate size of pixel in image, seen at screenX.

    imageY_ := trunc((imageRadius * 1) / screenDist * 65536.0);



    imageXs[screenX] := imageX;

    imageYs[screenX] := imageY_;
   End;
End;



(** * Prepare. *)

Procedure TCLXPano.prepare;
Begin
   normcursor:=cursor;

  If (assigned(bufferImage)) Then
    bufferImage.Free;
  If (assigned(spots)) Then 
    spots.Free;

  // set variables
  bufferImage := Tbitmap.Create;
  bufferImage.PixelFormat := pf32bit;
  bufferImage.Width := Width;
  bufferImage.Height := Height;

    
  prepared := False;

  reprepare := False;

  prepareString := ' ';

  errorString := ' ';

  actualSpot := -1;

  showSpots := false;

  spots := TList.Create;

  Try
    prepareImage();

    prepareRender();

    prepareSpots();

    prepared := True;

    rerender := True;

  Except
    invalidate;
   End;
End;


procedure TCLXPano.ClearSpots;
var
vertices: ^tSpot;
spot:integer;
begin
 // all spots

    For spot := 0 To spots.Count - 1 Do
     Begin
      vertices := spots[spot];
      setlength(vertices.Vertices,0);
      freemem(vertices);
     end;

   spots.clear;

end;


(**

 * Prepare spots.

 *)

Procedure TCLXPano.prepareSpots;
Var
  spotcount: Integer;
  spotparam: String;
  tokenizer: TStringTokenizer;
  vertex, anz: Integer;
  vertices: ^tSpot;
Begin
  ClearSpots;

  tokenizer := Nil;
  prepareString := 'Preparing spots.';

  invalidate;

  // set parameters of spots

  spotCount := 0;

  if spotdata.Count=0 then exit;

  spotParam := spotdata.Strings[spotCount];

  While (spotParam <> '') Do
   Begin

    If assigned(tokenizer) Then
      tokenizer.Free;
    tokenizer := TStringTokenizer.Create(spotParam, ';');
    anz := tokenizer.countTokens();
    new(vertices);
    vertices.Stype := stpoly;

    setlength(vertices.vertices, anz + 1,2);

    // calculate vertices

    Try

      For vertex := 0 To high(vertices.vertices) - 1 Do
       Begin
        // xPos

        vertices.vertices[vertex][0] := StrToInt(tokenizer.nextToken(';,'));

        // yPos

        vertices.vertices[vertex][1] := StrToInt(tokenizer.nextToken(';,'));
       End;

      vertices.vertices[high(vertices.vertices)][0] := vertices.vertices[0][0];

      vertices.vertices[high(vertices.vertices)][1] := vertices.vertices[0][1];

      spots.Add(vertices);

    Except
      //catch (Exception e){

      //spots.addElement(null);
     End;

    inc(spotCount);

    if spotCount<spotdata.Count then
     spotParam :=  spotdata.Strings[spotCount]
     else
     spotParam:='';

   End;
End;





Procedure TCLXPano.paint;
Begin
  brush.style := bsclear;

  If (bufferImage <> Nil) Then
   Begin
    If (Not prepared) Then
     Begin
      // paint prepare
     End
    Else
     Begin
      // paint image

      paintImage;
      If (movex = 0) And (movey = 0) And (moveZoom = 0) Then 
        DoSmooth;


      // paint spots

      paintSpots;

      // paint comment

      // paintComments(bufferGraphics);
     End;


    //bufferimage.Canvas.textout(0,0,inttostr(atime));
    If assigned(onpaint) Then 
      onpaint(self);
    canvas.draw(0,0,bufferImage);
   End;
End;


Procedure TCLXPano.paintImage;
Var
  imageRPos, horizon_, renderX, renderLPixel, renderRPixel, imageLPos: Integer;
  imageLColumn, imageRColumn, imageY_, zoomedImageY_, imageRow, imageYPos_,
  renderYPos: Integer;
  pimagepixels: pRGBPixelArray;
  prenderpixels: pRGBPixelArray;
Begin
  // render

  If (rerender) Then
   Begin
    atime := gettickcount;

    pimagepixels := imagepixels.scanline[0];
    prenderpixels := bufferImage.scanline[0];

    rerender := False;

    horizon_ := fhorizon * 65536;

    For renderX := renderWidthHalf - 1 Downto 0 Do
     Begin
      renderLPixel := renderWidthHalf - (renderX + 1);

      renderRPixel := renderWidthHalf + renderX;

      imageLPos := (imageXPos - imageXs[trunc(((renderX + 1) * imageZoom))]);

      If (imageLPos < 0) Then
       Begin
        imageLPos := imageLPos + imageWidth;
       End;


      imageRPos := (imageXPos + imageXs[trunc((renderX * imageZoom))]);

      If (imageRPos >= imageWidth) Then
       Begin
        imageRPos := imageRPos - imageWidth;
       End;



      imageLColumn := imageLPos * imageHeight;

      imageRColumn := imageRPos * imageHeight;



      imageY_ := imageYs[trunc(renderX * imageZoom)];

      zoomedImageY_ := trunc(imageY_ * imageZoom);

      imageYPos_ := horizon_ +

        (imageYPos - fhorizon) * imageY_ -

        ((renderHeightHalf) * zoomedImageY_);



      For renderYPos := renderHeight - 1 Downto 0 Do
       Begin
        imageRow := imageYPos_ Shr 16;

        prenderPixels[renderLPixel] := pimagePixels[imageLColumn + imageRow];

        prenderPixels[renderRPixel] := pimagePixels[imageRColumn + imageRow];



        renderLPixel := renderLPixel + renderWidth;

        renderRPixel := renderRPixel + renderWidth;



        imageYPos_ := imageYPos_ + zoomedImageY_;
       End
     End;




    atime := gettickcount - atime;
   End;

End;



Procedure TCLXPano.paintSpots;
Var
  spot, vertex: Integer;
  aspot: ^TSpot;
  _renderXPos, _renderYPos: Integer;
  vertexXPos, vertexYPos: Integer;
  imageX, renderx: Integer;
  alpha, renderDist, renderY: Double;
  renderxpos, renderypos: Integer;

  isvisible: Boolean;
Begin
  bufferImage.canvas.pen.Color := fspotcolor;

  // all spots

  For spot := 0 To spots.Count - 1 Do
   Begin
    aspot := spots[spot];
    isvisible := False;


    If ((aspot <> Nil) And (actualSpot = spot)) or showSpots  Then
     Begin
      _renderXPos := 0;

      _renderYPos := 0;


      // all vertices

      For vertex := 0 To high(aspot.vertices) Do
       Begin
        vertexXPos := aspot.vertices[vertex][0];

        vertexYPos := aspot.vertices[vertex][1];


        imageX := (vertexXPos - self.imageXPos) Mod imageWidth;

        If (imageX < 0) Then
         Begin
          imageX := imageX + imageWidth;
         End;



        // Calculate angle alpha (in radian) from the center to imageX.

        alpha := imageX / imageRadius;

        // Calculate offset screenX

        renderX := trunc((tan(alpha) * imageRadius / imageZoom));

        // Calculate distance screenDist to screenX.

        renderDist := imageRadius / cos(alpha);

        // Calculate offset screenY

        renderY := (vertexYPos - fhorizon) * renderDist / imageRadius;


        renderXPos := renderWidthHalf + renderX;

        renderYPos := renderHeightHalf +

          trunc((renderY + fhorizon - imageYPos) / imageZoom);

        // vertex in front ?

        If (((alpha >= (2 * PI) * 1 / 4) And (alpha <= (2 * PI) * 3 / 4)) Or

          (renderX >= 2048) Or (renderX <= -2048)) Then
         Begin
          If aspot.Stype = stpoly Then
            break;
         End
        Else
          isvisible := True;




        If (vertex > 0) And (aspot.Stype = stpoly) Then
         Begin
          bufferImage.Canvas.MoveTo(_renderXPos,
            _renderYPos);
          bufferImage.Canvas.lineto(renderXPos, renderYPos);
         End;



        _renderXPos := renderXPos;

        _renderYPos := renderYPos;
       End;
     End;
   End; //For Spot
End;



(**

 * Check move.

 *)

Procedure TCLXPano.checkMove;
Begin
  If ((moveX <> 0) Or (moveY <> 0) Or (moveZoom <> 0)) Then
   Begin
    imageZoom := imageZoom + moveZoom;

    If (imageZoom < 0.2) Then
     Begin
      imageZoom := 0.2;
     End;

    If (imageZoom > maxZoom) Then
     Begin
      imageZoom := maxZoom;
     End;


    imageXPos := imageXPos + moveX;
    imageXPos := imageXPos Mod imageWidth;
    If (imageXPos < 0) Then
     Begin
      imageXPos := imageXPos + imageWidth;
     End;



    imageYPos := imageYPos + moveY;
    If (imageYPos > imageHeight - trunc(renderHeightHalf * imageZoom) - 1) Then
     Begin
      imageYPos := imageHeight - trunc(renderHeightHalf * imageZoom) - 1;
     End;
    If (imageYPos < trunc(renderHeightHalf * imageZoom) + 1) Then
     Begin
      imageYPos := trunc(renderHeightHalf * imageZoom) + 1;
     End;

    rerender := True;
   End;
End;



(**

 * Check spots.

 *)

Procedure TCLXPano.checkSpots;
Var
  newSpot: Integer;
  vertices: ^TSpot;
  renderX, renderY, imageXPos, imageYPos: Integer;
  zoomedImageY, imageY: Integer;
  spot: Integer;
  crossings, _vertexXPos, _vertexYPos: Integer;
  vertexXPos, vertexYPos, vertex: Integer;
Begin
  newSpot := -1;



  // translate mousePos to imagePos

  renderX := mouseXPos - renderWidthHalf;

  renderY := renderHeightHalf - mouseYPos;

  If ((renderX > -renderWidthHalf) And (renderX < renderWidthHalf)) Then
   Begin
    // imageXPos

    If (renderX < 0) Then
     Begin
      renderX := -renderX;

      imageXPos := (self.imageXPos - imageXs[trunc(renderX * imageZoom)]) Mod imageWidth;
     End
    Else
     Begin
      imageXPos := (self.imageXPos + imageXs[trunc(renderX * imageZoom)]) Mod imageWidth;
     End;

    If (imageXPos < 0) Then
     Begin
      imageXPos := imageXPos + imageWidth;
     End;



    // imageYPos

    imageY := imageYs[trunc(renderX * imageZoom)];

    zoomedImageY := trunc(imageY * imageZoom);

    imageYPos := (fhorizon * 65536 +

      (self.imageYPos - fhorizon) * imageY -

      (renderY * zoomedImageY)) Shr 16;



    // all spots

    For spot := 0 To spots.Count - 1 Do
     Begin
      vertices := spots[spot];

      If (vertices <> Nil) Then
       Begin
        // count the crossings of a infinit line from the image position

        // of the mouse to the right and all edges of the spot

        crossings := 0;



        _vertexXPos := 0;

        _vertexYPos := 0;

        For vertex := 0 To high(vertices.vertices) Do
         Begin
          vertexXPos := vertices.vertices[vertex][0];

          vertexYPos := vertices.vertices[vertex][1];



          If (vertex > 0) Then
           Begin
            // One vertex has to be over and one under the horizonal

            // line. The imageXPos of mouse has to be to the left of

            // the edge between the two vertices.

            // Beware: divisor can't be 0 because of the first condition.

            If (((vertexYPos >= imageYPos) And (_vertexYPos < imageYPos)) Or

              ((vertexYPos < imageYPos) And (_vertexYPos >= imageYPos))) And

              (imageXPos < _vertexXPos + (imageYPos - _vertexYPos) *

              (vertexXPos - _vertexXPos) /

              (vertexYPos - _vertexYPos)) Then
             Begin
              crossings := crossings + 1;
             End;
           End;



          _vertexXPos := vertexXPos;

          _vertexYPos := vertexYPos;
         End;



        // spot is hit if # of crossings is not even

        If ((crossings And 1) = 1) Then
         Begin
          newSpot := spot;
         End;
       End;
     End;
   End;



  // spot changed?

  If (newSpot <> actualSpot) Then
   Begin
    actualSpot := newSpot;



    If (actualSpot >= 0) Then
     Begin

     End;



    //showStatus(actualLink);



    rerender := True;
   End;
End;



(**

 * Check cursor.

 *)

Procedure TCLXPano.checkCursor;
var
acursor:Tcursor;
Begin


 acursor:=normcursor;

      if (actualSpot <> -1) then

        acursor:=spotCursor;


      if (dragging) then

        acursor:=crSizeAll;

        if acursor<>cursor then
        begin
        cursor:=acursor;
        end;
      //QWidget_setCursor(Handle, Screen.Cursors[acursor]);



End;

Procedure TCLXPano.Sethorizon(Const Value: Integer);
Begin
  Fhorizon := Value;
  rerender := True;
  invalidate;
End;


Procedure TCLXPano.Setspotdata(Const Value: TStrings);
Begin
  Fspotdata.Assign(Value);
End;


Procedure TCLXPano.SetImage(Const Value: TFilename);
Begin
  FImage := Value;
  If Not (csloading In componentstate) Then
   Begin
    prepareImage;
    rerender := True;
    invalidate;
   End;
End;


Procedure TCLXPano.SetSpotColor(Const Value: TColor);
Begin
 fspotcolor:=value;
end;

Procedure TCLXPano.SetOnSpotClick(Const Value:TSpotClickEvent);
Begin
 fonspotclick:=value;
end;


Procedure TCLXPano.SetSpotCursor(Const Value: TCursor);
Begin
 fspotcursor:=value;
end;

Procedure TCLXPano.SetActive(Const Value: Boolean);
Begin
  Factive := Value;

  if  (csdesigning in componentstate) Then
   begin
    exit;
   end;

  If active Then
    start
  Else
    stop;
End;

Procedure TCLXPano.KeyPress(Var Key: Char);
Begin
  Inherited;
End;


Procedure TCLXPano.mousedown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
Begin
  Inherited;

  mouseXStart := x;

  mouseYStart := y;

  dragging := True;
End;

Procedure TCLXPano.MouseMove(Shift: TShiftState; X, Y: Integer);
Begin
  Inherited;
  If Not dragging Then
   Begin
    mouseXPos := x;
    mouseYPos := y;
   End
  Else
   Begin
    mouseXPos := x;

    mouseYPos := y;

    moveX := (x - mouseXStart) Div 4;

    moveY := (y - mouseYStart) Div 4;

   End;
End;

Procedure TCLXPano.MouseUP(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
Begin
  Inherited;

  moveX := 0;
  moveY := 0;

  if assigned(fonspotclick) then
    begin
     if actualSpot>-1 then fonspotclick(self,actualSpot);
    end;

  dragging := False;
  checkCursor;
End;

Procedure TCLXPano.KeyDown(Var Key: Word; Shift: TShiftState);
Begin
  Inherited;
  Case key Of
    43:
      moveZoom := -0.05;
    45:
      moveZoom := +0.05;
   End;
End;

Procedure TCLXPano.KeyUP(Var Key: Word; Shift: TShiftState);
Begin
  Inherited;
  moveZoom := 0;
End;

Procedure TCLXPano.DoSmooth;
Begin
  //not implemented
End;



Procedure TCLXPano.Setonpaint(Const Value: TNotifyEvent);
Begin
  Fonpaint := Value;
End;


Procedure TCLXPano.Loaded;
var
state:Tcomponentstate;
Begin
 state:=componentstate;

  If (fimage <> '') Then
   Begin
    init;
    invalidate;
   End;
   Inherited Loaded;   
End;



{ TMLVRThread }

Constructor TMLVRThread.Create(suspended: Boolean; owner: TCLXPano);
Begin
  aowner := owner;
  Inherited Create(suspended);
End;

Procedure TMLVRThread.dopaint;
Begin
  if  (csdestroying In aowner.componentstate) then exit;
  aowner.paint;
End;



Procedure TMLVRThread.Execute;
Var
  time, wait: Integer;
Begin
  { Thread-Code hier plazieren }


  While (Not terminated) Do
   Begin
    If (aowner.prepared) Then
     Begin
      While (aowner.active) Do
       Begin
        Try
          aowner.checkMove();
          aowner.checkSpots();
          aowner.checkCursor();
          If (aowner.rerender) Then
           Begin
            time := gettickcount;

            synchronize(dopaint);

            wait := 40 - (gettickcount - time);
            If (wait > 0) Then
              sleep(wait)
            Else
              //Thread.yield();
           End
          Else
           Begin
            sleep(50);
           End;
        Except
         End;
       End;

     End;
   End; // while (reprepare);
End;

Procedure Register;
Begin
  RegisterComponents('MindLink', [TCLXPano]);
End;

End.
