{******************************************************************************}
{                                                                              }
{                           Data Navigator Library                             }
{                                                                              }
{                   Copyright (c) 2002 - 2003 IMG Software                     }
{                           http://www.imgsoft.com                             }
{                         e-mail: support@imgsoft.com                          }
{                              All rights reserved.                            }
{                                                                              }
{******************************************************************************}
{$I igOptions.inc}
unit igModelConverter;

interface
{$IFNDEF V6_MORE}
  type TVarType = Word;
{$ENDIF}
  function TryVarToVar(const Source: Variant; out Dest: Variant; VarType: TVarType): Boolean;
  function VarTypeIsStr(const AVarType: TVarType): Boolean;
  function AddChar(const c: Char; Count: Integer): String;
  function AddWideChar(const c: WideChar; Count: Integer): WideString;
{$IFNDEF V6_MORE}
  function TryStrToInt(const S: string; out Value: Integer): Boolean;
  function TryStrToInt64(const S: string; out Value: Int64): Boolean;
  function TryStrToFloat(const S: string; out Value: Extended): Boolean; overload;
  function TryStrToFloat(const S: string; out Value: Double): Boolean; overload;
  function TryStrToFloat(const S: string; out Value: Single): Boolean; overload;
  function TryStrToCurr(const S: string; out Value: Currency): Boolean;
  function TryStrToDate(const S: string; out Value: TDateTime): Boolean;
  function TryStrToTime(const S: string; out Value: TDateTime): Boolean;
  function TryStrToDateTime(const S: string; out Value: TDateTime): Boolean;

var
  TrueBoolStrs: array of String;
  FalseBoolStrs: array of String;

const
  DefaultTrueBoolStr = 'True';   // DO NOT LOCALIZE
  DefaultFalseBoolStr = 'False'; // DO NOT LOCALIZE

  function TryStrToBool(const S: string; out Value: Boolean): Boolean;

{$ENDIF}

implementation

uses  SysUtils {$IFDEF V6_MORE}, SqlTimSt, FMTBcd , Variants {$ELSE}, Windows {$ENDIF};


function AddChar(const c: Char; Count: Integer): String;
var
  i: Integer;
begin
  result := '';
  for i := 0 to Count - 1 do result := result + c;
end;

function AddWideChar(const c: WideChar; Count: Integer): WideString;
var
  i: Integer;
begin
  result := '';
  for i := 0 to Count - 1 do result := result + c;
end;

{$IFNDEF V5_MORE}
function AnsiSameText(const S1, S2: string): Boolean;
begin
  Result := AnsiCompareText(S1, S2) = 0;
end;
{$ENDIF}

function TryVarToVar(const Source: Variant; out Dest: Variant; VarType: TVarType): Boolean;

  function ToInt(const S: String; out IsSuccessful: Boolean): Integer;
  begin
    IsSuccessful := TryStrToInt(S, result);
  end;

  function ToInt64(const S: String; out IsSuccessful: Boolean): Int64;
  begin
    IsSuccessful := TryStrToInt64(S, result);
  end;

  function ToSingle(const S: String; out IsSuccessful: Boolean): Single;
  begin
    IsSuccessful := TryStrToFloat(S, result);
  end;

  function ToDouble(const S: String; out IsSuccessful: Boolean): Double;
  begin
    IsSuccessful := TryStrToFloat(S, result);
  end;

  function ToCurr(const S: String; out IsSuccessful: Boolean): Currency;
  begin
    IsSuccessful := TryStrToCurr(S, result);
  end;

  function ToDate(const S: String; out IsSuccessful: Boolean): TDateTime;
  begin
    IsSuccessful := TryStrToDateTime(S, result);
  end;

  function ToBool(const S: String; out IsSuccessful: Boolean): Boolean;
  begin
    IsSuccessful := TryStrToBool(S, result);
  end;

  {$IFDEF V6_MORE}
  function ToSQLTimeStamp(const S: String; out IsSuccessful: Boolean): TSQLTimeStamp;
  begin
    IsSuccessful := TryStrToSQLTimeStamp(S, result);
  end;

  function ToBcd(const S: String; out IsSuccessful: Boolean): TBcd;
  begin
    IsSuccessful := TryStrToBcd(S, result);
  end;
  {$ENDIF}


begin
  result := true;
  try
    case (TVarData(Source).VType and varTypeMask) of
      varEmpty, varDispatch, varError, varUnknown: result := false;
      varNull:
        case varType of
          varOleStr, varString{$IFDEF V5_MORE}, varStrArg{$ENDIF}:
              Dest := VarToStr(Source);
          else
            Dest := null;
        end;
      varOleStr, varString{$IFDEF V6_MORE}, varStrArg{$ENDIF}:
        if (Source = '') and not VarTypeIsStr(VarType) then Dest := null else
        case VarType of
          varEmpty: Result := false;
          varSmallInt:
            Dest := ToInt(Source, result);
          varInteger:
            Dest := ToInt(Source, result);
          varSingle:
            Dest := ToSingle(Source, result);
          varDouble:
            Dest := ToDouble(Source, result);
          varCurrency:
            Dest := ToCurr(Source, result);
          varDate:
            Dest := ToDate(Source, result);
          varDispatch:
            result := false;
          varError:
            result := false;
          varBoolean:
            Dest := ToBool(Source, result);
          varUnknown:
            result := false;
          {$IFDEF V6_MORE}
          varShortInt:
            Dest := ToInt(Source, result);
          {$ENDIF}
          varByte:
            Dest := ToInt(Source, result);
          {$IFDEF V6_MORE}
          varWord:
            Dest := ToInt(Source, result);
          varLongWord:
            Dest := ToInt(Source, result);
          varInt64:
            Dest := ToInt64(Source, result);
          {$ENDIF}
          else
          begin
          {$IFDEF V6_MORE}
            if VarSQLTimeStamp = VarType then
              Dest := VarSQLTimeStampCreate(ToSQLTimeStamp(Source, result))
            else if VarFMTBcd = VarType then
              Dest := VarFMTBcdCreate(ToBcd(Source, result))
            else
          {$ENDIF}
              Dest := VarAsType(Source, VarType);
          end;
        end;
      else Dest := VarAsType(Source, VarType);
    end;
  except
    result := false;
  end;
end;

function VarTypeIsStr(const AVarType: TVarType): Boolean;
begin
  Result := ((AVarType and varTypeMask) = varOleStr) or
            ((AVarType and varTypeMask) = varString);
end;

{$IFNDEF V6_MORE}

function TryStrToInt(const S: string; out Value: Integer): Boolean;
var
  E: Integer;
begin
  Val(S, Value, E);
  Result := E = 0;
end;

function TryStrToInt64(const S: string; out Value: Int64): Boolean;
var
  E: Integer;
begin
  Val(S, Value, E);
  Result := E = 0;
end;

function TryStrToFloat(const S: string; out Value: Extended): Boolean;
begin
  Result := TextToFloat(PChar(S), Value, fvExtended);
end;

function TryStrToFloat(const S: string; out Value: Double): Boolean;
var
  LValue: Extended;
begin
  Result := TextToFloat(PChar(S), LValue, fvExtended);
  if Result then
    Value := LValue;
end;

function TryStrToFloat(const S: string; out Value: Single): Boolean;
var
  LValue: Extended;
begin
  Result := TextToFloat(PChar(S), LValue, fvExtended);
  if Result then
    Value := LValue;
end;

function TryStrToCurr(const S: string; out Value: Currency): Boolean;
begin
  Result := TextToFloat(PChar(S), Value, fvCurrency);
end;

{ ------------------ }
type
  TDateOrder = (doMDY, doDMY, doYMD);


function CurrentYear: Word;
var
  SystemTime: TSystemTime;
begin
  GetLocalTime(SystemTime);
  Result := SystemTime.wYear;
end;

procedure ScanBlanks(const S: string; var Pos: Integer);
var
  I: Integer;
begin
  I := Pos;
  while (I <= Length(S)) and (S[I] = ' ') do Inc(I);
  Pos := I;
end;

function ScanNumber(const S: string; var Pos: Integer;
  var Number: Word; var CharCount: Byte): Boolean;
var
  I: Integer;
  N: Word;
begin
  Result := False;
  CharCount := 0;
  ScanBlanks(S, Pos);
  I := Pos;
  N := 0;
  while (I <= Length(S)) and (S[I] in ['0'..'9']) and (N < 1000) do
  begin
    N := N * 10 + (Ord(S[I]) - Ord('0'));
    Inc(I);
  end;
  if I > Pos then
  begin
    CharCount := I - Pos;
    Pos := I;
    Number := N;
    Result := True;
  end;
end;

function ScanString(const S: string; var Pos: Integer;
  const Symbol: string): Boolean;
begin
  Result := False;
  if Symbol <> '' then
  begin
    ScanBlanks(S, Pos);
    if AnsiCompareText(Symbol, Copy(S, Pos, Length(Symbol))) = 0 then
    begin
      Inc(Pos, Length(Symbol));
      Result := True;
    end;
  end;
end;

function ScanChar(const S: string; var Pos: Integer; Ch: Char): Boolean;
begin
  Result := False;
  ScanBlanks(S, Pos);
  if (Pos <= Length(S)) and (S[Pos] = Ch) then
  begin
    Inc(Pos);
    Result := True;
  end;
end;

function GetDateOrder(const DateFormat: string): TDateOrder;
var
  I: Integer;
begin
  Result := doMDY;
  I := 1;
  while I <= Length(DateFormat) do
  begin
    case Chr(Ord(DateFormat[I]) and $DF) of
      'E': Result := doYMD;
      'Y': Result := doYMD;
      'M': Result := doMDY;
      'D': Result := doDMY;
    else
      Inc(I);
      Continue;
    end;
    Exit;
  end;
  Result := doMDY;
end;

procedure ScanToNumber(const S: string; var Pos: Integer);
begin
  while (Pos <= Length(S)) and not (S[Pos] in ['0'..'9']) do
  begin
    if S[Pos] in LeadBytes then Inc(Pos);
    Inc(Pos);
  end;
end;

{$IFDEF V4_MORE}
function GetEraYearOffset(const Name: string): Integer;
var
  I: Integer;
begin
  Result := 0;
  for I := Low(EraNames) to High(EraNames) do
  begin
    if EraNames[I] = '' then Break;
    if AnsiStrPos(PChar(EraNames[I]), PChar(Name)) <> nil then
    begin
      Result := EraYearOffsets[I];
      Exit;
    end;
  end;
end;
{$ENDIF}

function DoEncodeDate(Year, Month, Day: Word; var Date: TDateTime): Boolean;
var
  I: Integer;
  DayTable: PDayTable;
begin
  Result := False;
  DayTable := @MonthDays[IsLeapYear(Year)];
  if (Year >= 1) and (Year <= 9999) and (Month >= 1) and (Month <= 12) and
    (Day >= 1) and (Day <= DayTable^[Month]) then
  begin
    for I := 1 to Month - 1 do Inc(Day, DayTable^[I]);
    I := Year - 1;
    Date := I * 365 + I div 4 - I div 100 + I div 400 + Day - DateDelta;
    Result := True;
  end;
end;

function ScanDate(const S: string; var Pos: Integer;
  var Date: TDateTime): Boolean;
var
  DateOrder: TDateOrder;
  N1, N2, N3, Y, M, D: Word;
  L1, L2, L3{$IFDEF V4_MORE}, YearLen{$ENDIF}: Byte;
{$IFDEF V4_MORE}
  EraName : string;
  EraYearOffset: Integer;
  CenturyBase: Integer;

  function EraToYear(Year: Integer): Integer;
  begin
    if SysLocale.PriLangID = LANG_KOREAN then
    begin
      if Year <= 99 then
        Inc(Year, (CurrentYear + Abs(EraYearOffset)) div 100 * 100);
      if EraYearOffset > 0 then
        EraYearOffset := -EraYearOffset;
    end
    else
      Dec(EraYearOffset);
    Result := Year + EraYearOffset;
  end;
{$ENDIF}

begin
  Y := 0;
  M := 0;
  D := 0;
  {$IFDEF V4_MORE}
  YearLen := 0;
  {$ENDIF}
  Result := False;
  DateOrder := GetDateOrder(ShortDateFormat);
  {$IFDEF V4_MORE}
  EraYearOffset := 0;
  {$ENDIF}
  if ShortDateFormat[1] = 'g' then  // skip over prefix text
  begin
    ScanToNumber(S, Pos);
    {$IFDEF V4_MORE}
    EraName := Trim(Copy(S, 1, Pos-1));
    EraYearOffset := GetEraYearOffset(EraName);
    {$ENDIF}
  end
  {$IFDEF V4_MORE}
  else
    if AnsiPos('e', ShortDateFormat) > 0 then
      EraYearOffset := EraYearOffsets[1]{$ENDIF};

  if not (ScanNumber(S, Pos, N1, L1) and ScanChar(S, Pos, DateSeparator) and
    ScanNumber(S, Pos, N2, L2)) then Exit;
  if ScanChar(S, Pos, DateSeparator) then
  begin
    if not ScanNumber(S, Pos, N3, L3) then Exit;
    case DateOrder of
      doMDY: begin Y := N3; {$IFDEF V4_MORE}YearLen := L3;{$ENDIF} M := N1; D := N2; end;
      doDMY: begin Y := N3; {$IFDEF V4_MORE}YearLen := L3;{$ENDIF} M := N2; D := N1; end;
      doYMD: begin Y := N1; {$IFDEF V4_MORE}YearLen := L1;{$ENDIF} M := N2; D := N3; end;
    end;
    {$IFDEF V4_MORE}
    if EraYearOffset > 0 then
      Y := EraToYear(Y)
    else if (YearLen <= 2) then
    begin
      CenturyBase := CurrentYear - TwoDigitYearCenturyWindow;
      Inc(Y, CenturyBase div 100 * 100);
      if (TwoDigitYearCenturyWindow > 0) and (Y < CenturyBase) then
        Inc(Y, 100);
    end;
    {$ELSE}
    if Y <= 99 then Inc(Y, CurrentYear div 100 * 100);
    {$ENDIF}
  end else
  begin
    Y := CurrentYear;
    if DateOrder = doDMY then
    begin
      D := N1; M := N2;
    end else
    begin
      M := N1; D := N2;
    end;
  end;
  ScanChar(S, Pos, DateSeparator);
  ScanBlanks(S, Pos);
  if SysLocale.FarEast and (System.Pos('ddd', ShortDateFormat) <> 0) then
  begin     // ignore trailing text
    if ShortTimeFormat[1] in ['0'..'9'] then  // stop at time digit
      ScanToNumber(S, Pos)
    else  // stop at time prefix
      repeat
        while (Pos <= Length(S)) and (S[Pos] <> ' ') do Inc(Pos);
        ScanBlanks(S, Pos);
      until (Pos > Length(S)) or
        (AnsiCompareText(TimeAMString, Copy(S, Pos, Length(TimeAMString))) = 0) or
        (AnsiCompareText(TimePMString, Copy(S, Pos, Length(TimePMString))) = 0);
  end;
  Result := DoEncodeDate(Y, M, D, Date);
end;

function DoEncodeTime(Hour, Min, Sec, MSec: Word; var Time: TDateTime): Boolean;
begin
  Result := False;
  if (Hour < 24) and (Min < 60) and (Sec < 60) and (MSec < 1000) then
  begin
    Time := (Hour * 3600000 + Min * 60000 + Sec * 1000 + MSec) / MSecsPerDay;
    Result := True;
  end;
end;

function ScanTime(const S: string; var Pos: Integer;
  var Time: TDateTime): Boolean;
var
  BaseHour: Integer;
  Hour, Min, Sec, MSec: Word;
  Junk: Byte;
begin
  Result := False;
  BaseHour := -1;
  if ScanString(S, Pos, TimeAMString) or ScanString(S, Pos, 'AM') then
    BaseHour := 0
  else if ScanString(S, Pos, TimePMString) or ScanString(S, Pos, 'PM') then
    BaseHour := 12;
  if BaseHour >= 0 then ScanBlanks(S, Pos);
  if not ScanNumber(S, Pos, Hour, Junk) then Exit;
  Min := 0;
  if ScanChar(S, Pos, TimeSeparator) then
    if not ScanNumber(S, Pos, Min, Junk) then Exit;
  Sec := 0;
  if ScanChar(S, Pos, TimeSeparator) then
    if not ScanNumber(S, Pos, Sec, Junk) then Exit;
  MSec := 0;
  if ScanChar(S, Pos, DecimalSeparator) then
    if not ScanNumber(S, Pos, MSec, Junk) then Exit;
  if BaseHour < 0 then
    if ScanString(S, Pos, TimeAMString) or ScanString(S, Pos, 'AM') then
      BaseHour := 0
    else
      if ScanString(S, Pos, TimePMString) or ScanString(S, Pos, 'PM') then
        BaseHour := 12;
  if BaseHour >= 0 then
  begin
    if (Hour = 0) or (Hour > 12) then Exit;
    if Hour = 12 then Hour := 0;
    Inc(Hour, BaseHour);
  end;
  ScanBlanks(S, Pos);
  Result := DoEncodeTime(Hour, Min, Sec, MSec, Time);
end;
{ ------------------- }


function TryStrToDate(const S: string; out Value: TDateTime): Boolean;
var
  Pos: Integer;
begin
  Pos := 1;
  Result := ScanDate(S, Pos, Value) and (Pos > Length(S));
end;

function TryStrToTime(const S: string; out Value: TDateTime): Boolean;
var
  Pos: Integer;
begin
  Pos := 1;
  Result := ScanTime(S, Pos, Value) and (Pos > Length(S));
end;

function TryStrToDateTime(const S: string; out Value: TDateTime): Boolean;
var
  Pos: Integer;
  Date, Time: TDateTime;
begin
  Result := True;
  Pos := 1;
  Time := 0;
  if not ScanDate(S, Pos, Date) or
     not ((Pos > Length(S)) or ScanTime(S, Pos, Time)) then

    // Try time only
    Result := TryStrToTime(S, Value)
  else
    if Date >= 0 then
      Value := Date + Time
    else
      Value := Date - Time;
end;

procedure VerifyBoolStrArray;
begin
  if Length(TrueBoolStrs) = 0 then
  begin
    SetLength(TrueBoolStrs, 1);
    TrueBoolStrs[0] := DefaultTrueBoolStr;
  end;
  if Length(FalseBoolStrs) = 0 then
  begin
    SetLength(FalseBoolStrs, 1);
    FalseBoolStrs[0] := DefaultFalseBoolStr;
  end;
end;

function TryStrToBool(const S: string; out Value: Boolean): Boolean;
  function CompareWith(const aArray: array of string): Boolean;
  var
    I: Integer;
  begin
    Result := False;
    for I := Low(aArray) to High(aArray) do
      if AnsiSameText(S, aArray[I]) then
      begin
        Result := True;
        Break;
      end;
  end;
var
  LResult: Extended;
begin
  Result := TryStrToFloat(S, LResult);
  if Result then
    Value := LResult <> 0
  else
  begin
    VerifyBoolStrArray;
    if CompareWith(TrueBoolStrs) then
      Value := True
    else if CompareWith(FalseBoolStrs) then
      Value := False
    else
      Result := False;
  end;
end;
{$ENDIF}

end.
