{******************************************************************************}
{                                                                              }
{                           Data Navigator Library                             }
{                                                                              }
{                   Copyright (c) 2002 - 2003 IMG Software                     }
{                           http://www.imgsoft.com                             }
{                         e-mail: support@imgsoft.com                          }
{                                                                              }
{******************************************************************************}
{$I IgOptions.inc}
unit igKeyDBModel;

interface

uses Classes, DB, UdbDatabase, igBaseModel, igDBBaseModel
  {$IFDEF V6_MORE}, Variants {$ENDIF};

type
  TigKeyDBDataSetModel = class(TigExtDataSetModel)
  public
    function GetIndexDefs: TIndexDefs; override;
    function GetIndexFieldNames: String; override;
    function GetIndexName: String; override;
    function FindRecord(const KeyValues: Variant; const Exact: Boolean): Boolean; override;
    function GetQuerySQL: String; override;
    procedure SetQuerySQL(Value: string); override;
    procedure SetRange(StartValues, EndValues: Variant); override;
    function IsSupported(const Value: TDataSetSupportedOption): Boolean; override;
    procedure SetParam(const Name: String; const Value: Variant); override;
  end;

  {::TigKeyDBFindModel is component for the data searching in KeyDB dataset components (TUdbTable, TUdbQuery and so on).
  Component uses specific properties and methods of TUdbDataSet object for quick searching with a current index.
  @component
  }
  TigKeyDBFindModel = class(TigCustomDBFindModel)
  protected
    procedure CreateExtDataSetModel(out ExtDataSetModel: TigExtDataSetModel); override;
    procedure CheckDataSet; override;
  end;

  {::TigKeyDBFilterModel is component for the data filtering in KeyDB dataset components (TUdbTable, TUdbQuery and so on).
  @component
  }
  TigKeyDBFilterModel = class(TigCustomDBFilterModel)
  private
    FReFiltering: Boolean;
    procedure ReFiltered;
  protected
    procedure CreateExtDataSetModel(out ExtDataSetModel: TigExtDataSetModel); override;
    procedure CheckDataSet; override;
    function InternalApply: Boolean; override;
    procedure LayoutChanged; override;
    procedure ActiveChanged; override;
    procedure DataSetChanged; override;
  end;

  procedure Register;

implementation

uses igTypeInfo, igModelConsts;

const
  DataSetName: String = 'TUdbDataSet';

procedure Register;
begin
  RegisterComponents('Navigator Access', [TigKeyDBFilterModel]);
  RegisterComponents('Navigator Access', [TigKeyDBFindModel]);
end;

{ TigKeyDBIndexModel }

function TigKeyDBDataSetModel.GetIndexDefs: TIndexDefs;
begin
  if DataSet is TUdbTable  then
    result := TUdbTable(DataSet).IndexDefs
  else
    result := nil;
end;

function TigKeyDBDataSetModel.GetIndexFieldNames: String;
begin
  if DataSet is TUdbTable  then
    result := TUdbTable(DataSet).IndexFieldNames
  else
    result := '';
end;

function TigKeyDBDataSetModel.GetIndexName: String;
begin
  if DataSet is TUdbTable then
    result := TUdbTable(DataSet).IndexName
  else
    result := '';
end;

function TigKeyDBDataSetModel.FindRecord(const KeyValues: Variant; const Exact: Boolean): Boolean;
var
  i, HighBound: Integer;
begin
  if DataSet is TUdbTable then
  begin
    HighBound := VarArrayHighBound(KeyValues, 1);
    with TUdbTable(DataSet) do
    begin
      SetKey;
      for i := 0 to HighBound do
        Items[i].Field.Value := KeyValues[i];
      for i := HighBound + 1 to Count - 1 do
        Items[i].Field.Value := null;
      if Exact then result := ApplyKey(true)
      else
      begin
        ApplyKey(false);
        result := true;
      end;
    end;
  end else
    Result := inherited FindRecord(KeyValues, Exact);
end;

function TigKeyDBDataSetModel.GetQuerySQL: String;
var
  Sql: TObject;
begin
  Sql := GetPropertyClass(DataSet, 'SQL');
  if Sql is TStrings then
    Result := TStrings(Sql).Text
  else
    Result := '';
end;

procedure TigKeyDBDataSetModel.SetQuerySQL(Value: string);
var
  Sql: TObject;
begin
  Sql := GetPropertyClass(DataSet, 'SQL');
  if Sql is TStrings then
    TStrings(Sql).Text := Value
end;

function TigKeyDBDataSetModel.IsSupported(const Value: TDataSetSupportedOption): Boolean;
begin
  case Value of
    dsDefaultIndex: result := (DataSet is TUdbTable);
    dsMultiCaseInsIndexSearch, dsIndexRange: result := DataSet is TUdbTable;
    dsSQL: result := (DataSet is TUdbDataSet) and (GetPropertyClass(DataSet, 'SQL') is TStrings);
    else result := inherited IsSupported(Value);
  end;
end;

procedure TigKeyDBDataSetModel.SetRange(StartValues, EndValues: Variant);
var
  HighBound, i: Integer;
begin
  if IsSupported(dsIndexRange) then
  with TUdbTable(DataSet) do
  begin
    if VarIsArray(StartValues) then
    begin
      HighBound := VarArrayHighBound(StartValues, 1);
      SetRangeStart;
      for i := 0 to HighBound do
      begin
        if not VarIsEmpty(StartValues[i]) then
          Items[i].Field.Value := StartValues[i];
      end;
      SetRangeEnd;
      for i := 0 to HighBound do
      begin
        if not VarIsEmpty(EndValues[i]) then
          Items[i].Field.Value := EndValues[i];
      end;
      ApplyRange;
    end else
    begin
      CancelRange;
    end;
  end;
end;

procedure TigKeyDBDataSetModel.SetParam(const Name: String; const Value: Variant);
var
  Params: TObject;
  Param: TParam;
begin
  Params := GetPropertyClass(DataSet, 'Params');
  if Params is TParams then
  begin
    Param := TParams(Params).FindParam(Name);
    if Param <> nil then Param.Value := Value;
  end;
end;

{ TigKeyDBFindModel }

procedure TigKeyDBFindModel.CreateExtDataSetModel(out ExtDataSetModel: TigExtDataSetModel);
begin
  ExtDataSetModel := TigKeyDBDataSetModel.Create(Self);
end;

procedure TigKeyDBFindModel.CheckDataSet;
begin
  if not (DataLink.DataSet is TUdbDataSet) then
    igBaseModelErrorFmt(SIsNotDataSet, [DataLink.DataSet.Name, DataSetName], Self);
end;

{ TigKeyDBFilterModel }

procedure TigKeyDBFilterModel.CheckDataSet;
begin
  if not (DataLink.DataSet is TUdbDataSet) then
    igBaseModelErrorFmt(SIsNotDataSet, [DataLink.DataSet.Name, DataSetName], Self);
end;

procedure TigKeyDBFilterModel.CreateExtDataSetModel(out ExtDataSetModel: TigExtDataSetModel);
begin
  ExtDataSetModel := TigKeyDBDataSetModel.Create(Self);
end;

function TigKeyDBFilterModel.InternalApply: Boolean;
begin
  result := true;
  if DataLink.Active then
  begin
    SetDataSetOnFilterRecord(FilterRecord);
    if GetDataSetFiltered <> Filtered then
    begin
      SetDataSetFiltered(Filtered);
    end
  end;
end;

procedure TigKeyDBFilterModel.LayoutChanged;
begin
  inherited;
  ReFiltered;
end;

procedure TigKeyDBFilterModel.ActiveChanged;
begin
  inherited;
  ReFiltered;
end;

procedure TigKeyDBFilterModel.ReFiltered;
begin
  if DataLink.Active and DataLink.DataSet.Filtered then
  begin
    FReFiltering := True;
    try
      DataLink.DataSet.Filtered := False;
      SetDataSetOnFilterRecord(FilterRecord);
      DataLink.DataSet.Filtered := True;
    finally
      FReFiltering := False;
    end;
  end;
end;

procedure TigKeyDBFilterModel.DataSetChanged;
begin
  if not FReFiltering then inherited;
end;

end.
