Unit UltimDBFooter;

{
-----------------------------------------------------------------------------

                              UltimDBFooter 1.0
                        
                        VCL for Delphi and C++Builder
                       (C) Frdric Leneuf-Magaud 2001
                      Portions (C) Borland/Inprise 2001

-----------------------------------------------------------------------------

 Please read ReadMe.txt for conditions of use

 This component is provided 'as-is', without any express or implied warranty.
 In no event shall the author be held liable for any damages arising from the
 use of this component.

 Veuillez lire le fichier LisezMoi.txt pour les conditions d'utilisation

 Ce composant est fourni tel quel, sans aucune garantie.
 En aucun cas, l'auteur ne pourra tre tenu pour responsable des dommages
 rsultant de l'utilisation de ce composant.

-----------------------------------------------------------------------------
}

Interface

{$I OPTIONS.INC}

Uses
   Windows, Messages, SysUtils, Forms, Classes, Controls, Graphics, CommCtrl, ComCtrls,
   DB, DBGrids, Grids, UltimDBGrid;

Const
   FTR_ACTIVE = 1;
   FTR_COLWIDTHS = 2;
   FTR_SCROLL = 3;
   FTR_ERROR = '#ERR!';
   FTR_DEFAULTWIDTH = 64;

Type
   TUltimDBFooter = Class;

   TUltimDataLink = Class(TDataLink)
   Private
      FFooterCtrl: TUltimDBFooter;
      FRecCount: Integer;
      FUpdateNeeded: Boolean;
   Protected
      Procedure DataSetChanged; Override;
      Procedure EditingChanged; Override;
   Public
      Constructor Create(Footer: TUltimDBFooter);
   End;

   TFooterPanelStyle = (psText, psOwnerDraw);
   TFooterPanelBevel = (pbNone, pbLowered, pbRaised);

   TFooterPanel = Class(TCollectionItem)
   Private
      FText: String;
      FFieldName: String;
      FWidth: Integer;
      FAlignment: TAlignment;
      FBevel: TFooterPanelBevel;
      FBiDiMode: TBiDiMode;
      FParentBiDiMode: Boolean;
      FStyle: TFooterPanelStyle;
      FUpdateNeeded: Boolean;
      Procedure SetAlignment(Value: TAlignment);
      Procedure SetBevel(Value: TFooterPanelBevel);
      Procedure SetBiDiMode(Value: TBiDiMode);
      Procedure SetParentBiDiMode(Value: Boolean);
      Procedure SetStyle(Value: TFooterPanelStyle);
      Procedure SetText(Const Value: String);
      Procedure SetWidth(Value: Integer);
      Function IsBiDiModeStored: Boolean;
   Protected
      Function GetDisplayName: String; Override;
   Public
      Constructor Create(Collection: TCollection); Override;
      Procedure Assign(Source: TPersistent); Override;
      Procedure ParentBiDiModeChanged;
      Function UseRightToLeftAlignment: Boolean;
      Function UseRightToLeftReading: Boolean;
      Property Width: Integer Read FWidth Write SetWidth;
   Published
      Property Alignment: TAlignment Read FAlignment Write SetAlignment Default taCenter;
      Property Bevel: TFooterPanelBevel Read FBevel Write SetBevel Default pbLowered;
      Property BiDiMode: TBiDiMode Read FBiDiMode Write SetBiDiMode Stored IsBiDiModeStored;
      Property FieldName: String Read FFieldName Write FFieldName;
      Property ParentBiDiMode: Boolean Read FParentBiDiMode Write SetParentBiDiMode Default True;
      Property Style: TFooterPanelStyle Read FStyle Write SetStyle Default psText;
      Property Text: String Read FText Write SetText;
   End;

   TFooterPanels = Class(TCollection)
   Private
      FFooterBar: TUltimDBFooter;
      Function GetItem(Index: Integer): TFooterPanel;
      Procedure SetItem(Index: Integer; Value: TFooterPanel);
   Protected
      Function GetOwner: TPersistent; Override;
      Procedure Update(Item: TCollectionItem); Override;
   Public
      Constructor Create(FooterBar: TUltimDBFooter);
      Function Add: TFooterPanel;
      Property Items[Index: Integer]: TFooterPanel Read GetItem Write SetItem; Default;
   End;

   TFooterColumnOperation = (coNone, coSum, coCount, coAverage, coMax, coMin);
   TNumType = (ntInteger, ntFixed, ntGeneral, ntScientific, ntCurrency);

   TFooterColumn = Class(TCollectionItem)
   Private
      FAlignment: TAlignment;
      FBevel: TFooterPanelBevel;
      FFieldName: String;
      FNumPrecision: Byte;
      FNumType: TNumType;
      FOperation: TFooterColumnOperation;
      FTextBefore: String;
      FTextAfter: String;
      FWidthIfNoResize: Integer;
      FFieldNo: Integer;
      FCalcValue: Variant;
      FError: Boolean;
      Procedure SetAlignment(Value: TAlignment);
      Procedure SetBevel(Value: TFooterPanelBevel);
      Procedure SetFieldName(Value: String);
      Procedure SetNumPrecision(Value: Byte);
      Procedure SetNumType(Value: TNumType);
      Procedure SetOperation(Value: TFooterColumnOperation);
      Procedure SetTextBefore(Value: String);
      Procedure SetTextAfter(Value: String);
      Procedure SetWidthIfNoResize(Value: Integer);
   Protected
      Function GetDisplayName: String; Override;
   Public
      Constructor Create(Collection: TCollection); Override;
      Procedure Assign(Source: TPersistent); Override;
   Published
      Property Alignment: TAlignment Read FAlignment Write SetAlignment Default taCenter;
      Property Bevel: TFooterPanelBevel Read FBevel Write SetBevel Default pbLowered;
      Property FieldName: String Read FFieldName Write SetFieldName;
      Property NumPrecision: Byte Read FNumPrecision Write SetNumPrecision;
      Property NumType: TNumType Read FNumType Write SetNumType Default ntFixed;
      Property Operation: TFooterColumnOperation Read FOperation Write SetOperation Default coNone;
      Property TextBefore: String Read FTextBefore Write SetTextBefore;
      Property TextAfter: String Read FTextAfter Write SetTextAfter;
      Property WidthIfNoResize: Integer Read FWidthIfNoResize Write SetWidthIfNoResize
         Default FTR_DEFAULTWIDTH;
   End;

   TFooterColumns = Class(TCollection)
   Private
      FFooterBar: TUltimDBFooter;
      Function GetItem(Index: Integer): TFooterColumn;
      Procedure SetItem(Index: Integer; Value: TFooterColumn);
   Protected
      Function GetOwner: TPersistent; Override;
      Procedure Update(Item: TCollectionItem); Override;
   Public
      Constructor Create(FooterBar: TUltimDBFooter);
      Function Add: TFooterColumn;
      Property Items[Index: Integer]: TFooterColumn Read GetItem Write SetItem; Default;
   End;

   TDrawPanelEvent = Procedure(FooterBar: TUltimDBFooter; Panel: TFooterPanel; Const Rect: TRect)
      Of Object;
   TBeforeDisplayTextEvent = Procedure(FooterBar: TUltimDBFooter; Column: TFooterColumn;
      Const Value: Variant; Var Text: String; Var Style: TFooterPanelStyle) Of Object;
   TBeforeReadFieldsEvent = Procedure(FooterBar: TUltimDBFooter) Of Object;
   TReadFieldValueEvent = Procedure(FooterBar: TUltimDBFooter; Column: TFooterColumn;
      Const Value: Variant) Of Object;

   TUltimDBFooter = Class(TWinControl)
   Private
      FPanels: TFooterPanels;
      FColumns: TFooterColumns;
      FCanvas: TCanvas;
      FSizeGrip: Boolean;
      FLeftPanel: Integer;
      FUseSystemFont: Boolean;
      FDataLink: TUltimDataLink;
      FDBGrid: TUltimDBGrid;
      FIgnoreResizing: Boolean;
      FIgnoreScrolling: Boolean;
      FOnDrawPanel: TDrawPanelEvent;
      FOnBeforeDisplayText: TBeforeDisplayTextEvent;
      FOnBeforeReadFields: TBeforeReadFieldsEvent;
      FOnReadFieldValue: TReadFieldValueEvent;
      Procedure DoRightToLeftAlignment(Var Str: String; AAlignment: TAlignment;
         ARTLAlignment: Boolean);
      Function IsFontStored: Boolean;
      Procedure SetPanels(Value: TFooterPanels);
      Procedure SetColumns(Value: TFooterColumns);
      Procedure SetSizeGrip(Value: Boolean);
      Procedure SyncToSystemFont;
      Procedure UpdatePanel(Index: Integer; Repaint: Boolean);
      Procedure UpdatePanels(UpdateRects, UpdateText: Boolean);
      Procedure CMBiDiModeChanged(Var Message: TMessage); Message CM_BIDIMODECHANGED;
      Procedure CMColorChanged(Var Message: TMessage); Message CM_COLORCHANGED;
      Procedure CMParentFontChanged(Var Message: TMessage); Message CM_PARENTFONTCHANGED;
      Procedure CMSysColorChange(Var Message: TMessage); Message CM_SYSCOLORCHANGE;
      Procedure CMWinIniChange(Var Message: TMessage); Message CM_WININICHANGE;
      Procedure CMSysFontChanged(Var Message: TMessage); Message CM_SYSFONTCHANGED;
      Procedure CNDrawItem(Var Message: TWMDrawItem); Message CN_DRAWITEM;
      Procedure WMPaint(Var Message: TWMPaint); Message WM_PAINT;
      Procedure WMSize(Var Message: TWMSize); Message WM_SIZE;
      Procedure SetUseSystemFont(Const Value: Boolean);
      Procedure SetDBGrid(Value: TUltimDBGrid);
   Protected
      Procedure ChangeScale(M, D: Integer); Override;
      Procedure CreateParams(Var Params: TCreateParams); Override;
      Procedure CreateWnd; Override;
      Procedure DrawPanel(Panel: TFooterPanel; Const Rect: TRect); Dynamic;
      Procedure Notification(AComponent: TComponent; Operation: TOperation); Override;
      Property Panels: TFooterPanels Read FPanels Write SetPanels;
      Procedure CalcFields; Virtual;
   Public
      Constructor Create(AOwner: TComponent); Override;
      Destructor Destroy; Override;
      Function ExecuteAction(Action: TBasicAction): Boolean; Override;
      Procedure FlipChildren(AllLevels: Boolean); Override;
      Property Canvas: TCanvas Read FCanvas;
      Procedure FooterMessage(Msg: Integer); Virtual;
      Procedure UpdateDataPanels(ReCalc: Boolean); Virtual;
   Published
      Property Action;
      Property Align Default alNone;
      Property Anchors;
      Property BiDiMode;
      Property BorderWidth;
      Property Color Default clBtnFace;
      Property DBGrid: TUltimDBGrid Read FDBGrid Write SetDBGrid Default Nil;
      Property DragCursor;
      Property DragKind;
      Property DragMode;
      Property Enabled;
      Property Font Stored IsFontStored;
      Property IgnoreResizing: Boolean Read FIgnoreResizing Write FIgnoreResizing Default False;
      Property IgnoreScrolling: Boolean Read FIgnoreScrolling Write FIgnoreScrolling Default False;
      Property Columns: TFooterColumns Read FColumns Write SetColumns;
      Property Constraints;
      Property ParentBiDiMode;
      Property ParentColor Default False;
      Property ParentFont Default False;
      Property ParentShowHint;
      Property PopupMenu;
      Property ShowHint;
      Property SizeGrip: Boolean Read FSizeGrip Write SetSizeGrip Default False;
      Property UseSystemFont: Boolean Read FUseSystemFont Write SetUseSystemFont Default True;
      Property Visible;
      Property OnClick;
      Property OnContextPopup;
      Property OnDblClick;
      Property OnDragDrop;
      Property OnDragOver;
      Property OnEndDock;
      Property OnEndDrag;
      Property OnMouseDown;
      Property OnMouseMove;
      Property OnMouseUp;
      Property OnDrawPanel: TDrawPanelEvent Read FOnDrawPanel Write FOnDrawPanel;
      Property OnResize;
      Property OnStartDock;
      Property OnStartDrag;
      Property OnBeforeDisplayText: TBeforeDisplayTextEvent Read FOnBeforeDisplayText
         Write FOnBeforeDisplayText;
      Property OnBeforeReadFields: TBeforeReadFieldsEvent Read FOnBeforeReadFields
         Write FOnBeforeReadFields;
      Property OnReadFieldValue: TReadFieldValueEvent Read FOnReadFieldValue
         Write FOnReadFieldValue;
   End;

Implementation

{ TUltimDataLink }

Constructor TUltimDataLink.Create(Footer: TUltimDBFooter);
Begin
   Inherited Create;
   FFooterCtrl := Footer;
   FRecCount := 0;
   FUpdateNeeded := False;
   VisualControl := True;
End;

Procedure TUltimDataLink.DataSetChanged;
Begin
   If FUpdateNeeded Or (FRecCount <> DataSet.RecordCount) Then
   Begin
      FUpdateNeeded := False;
      FRecCount := DataSet.RecordCount;
      If FFooterCtrl <> Nil Then
         FFooterCtrl.UpdateDataPanels(True);
   End;
End;

Procedure TUltimDataLink.EditingChanged;
Begin
   If DataSet.State = dsBrowse Then
      FUpdateNeeded := True;
End;

{ TFooterPanel }

Constructor TFooterPanel.Create(Collection: TCollection);
Begin
   FWidth := FTR_DEFAULTWIDTH;
   FBevel := pbLowered;
   FParentBiDiMode := True;
   Inherited Create(Collection);
   ParentBiDiModeChanged;
End;

Procedure TFooterPanel.Assign(Source: TPersistent);
Begin
   If Source Is TFooterPanel Then
   Begin
      Alignment := TFooterPanel(Source).Alignment;
      Bevel := TFooterPanel(Source).Bevel;
      Style := TFooterPanel(Source).Style;
      Text := TFooterPanel(Source).Text;
      Width := TFooterPanel(Source).Width;
   End
   Else
      Inherited Assign(Source);
End;

Procedure TFooterPanel.SetBiDiMode(Value: TBiDiMode);
Begin
   If Value <> FBiDiMode Then
   Begin
      FBiDiMode := Value;
      FParentBiDiMode := False;
      Changed(False);
   End;
End;

Function TFooterPanel.IsBiDiModeStored: Boolean;
Begin
   Result := Not FParentBiDiMode;
End;

Procedure TFooterPanel.SetParentBiDiMode(Value: Boolean);
Begin
   If FParentBiDiMode <> Value Then
   Begin
      FParentBiDiMode := Value;
      ParentBiDiModeChanged;
   End;
End;

Procedure TFooterPanel.ParentBiDiModeChanged;
Begin
   If FParentBiDiMode Then
   Begin
      If GetOwner <> Nil Then
      Begin
         BiDiMode := TFooterPanels(GetOwner).FFooterBar.BiDiMode;
         FParentBiDiMode := True;
      End;
   End;
End;

Function TFooterPanel.UseRightToLeftReading: Boolean;
Begin
   Result := SysLocale.MiddleEast And (BiDiMode <> bdLeftToRight);
End;

Function TFooterPanel.UseRightToLeftAlignment: Boolean;
Begin
   Result := SysLocale.MiddleEast And (BiDiMode = bdRightToLeft);
End;

Function TFooterPanel.GetDisplayName: String;
Begin
   Result := Text;
   If Result = '' Then
      Result := Inherited GetDisplayName;
End;

Procedure TFooterPanel.SetAlignment(Value: TAlignment);
Begin
   If FAlignment <> Value Then
   Begin
      FAlignment := Value;
      Changed(False);
   End;
End;

Procedure TFooterPanel.SetBevel(Value: TFooterPanelBevel);
Begin
   If FBevel <> Value Then
   Begin
      FBevel := Value;
      Changed(False);
   End;
End;

Procedure TFooterPanel.SetStyle(Value: TFooterPanelStyle);
Begin
   If FStyle <> Value Then
   Begin
      FStyle := Value;
      Changed(False);
   End;
End;

Procedure TFooterPanel.SetText(Const Value: String);
Begin
   If FText <> Value Then
   Begin
      FText := Value;
      Changed(False);
   End;
End;

Procedure TFooterPanel.SetWidth(Value: Integer);
Begin
   If FWidth <> Value Then
   Begin
      FWidth := Value;
      Changed(True);
   End;
End;

{ TFooterPanels }

Constructor TFooterPanels.Create(FooterBar: TUltimDBFooter);
Begin
   Inherited Create(TFooterPanel);
   FFooterBar := FooterBar;
End;

Function TFooterPanels.Add: TFooterPanel;
Begin
   Result := TFooterPanel(Inherited Add);
End;

Function TFooterPanels.GetItem(Index: Integer): TFooterPanel;
Begin
   Result := TFooterPanel(Inherited GetItem(Index));
End;

Function TFooterPanels.GetOwner: TPersistent;
Begin
   Result := FFooterBar;
End;

Procedure TFooterPanels.SetItem(Index: Integer; Value: TFooterPanel);
Begin
   Inherited SetItem(Index, Value);
End;

Procedure TFooterPanels.Update(Item: TCollectionItem);
Begin
   If Item <> Nil Then
      FFooterBar.UpdatePanel(Item.Index, False)
   Else
      FFooterBar.UpdatePanels(True, False);
End;

{ TFooterColumn }

Constructor TFooterColumn.Create(Collection: TCollection);
Begin
   Inherited Create(Collection);
   FAlignment := taCenter;
   FBevel := pbLowered;
   FFieldName := '';
   FNumPrecision := 2;
   FNumType := ntFixed;
   FOperation := coNone;
   FTextBefore := '';
   FTextAfter := '';
   FWidthIfNoResize := FTR_DEFAULTWIDTH;
End;

Procedure TFooterColumn.Assign(Source: TPersistent);
Begin
   If Source Is TFooterColumn Then
   Begin
      Alignment := TFooterColumn(Source).Alignment;
      Bevel := TFooterColumn(Source).Bevel;
      FieldName := TFooterColumn(Source).FieldName;
      NumPrecision := TFooterColumn(Source).NumPrecision;
      NumType := TFooterColumn(Source).NumType;
      Operation := TFooterColumn(Source).Operation;
   End
   Else
      Inherited Assign(Source);
End;

Function TFooterColumn.GetDisplayName: String;
Begin
   Result := FieldName;
   If Result = '' Then
      Result := Inherited GetDisplayName;
End;

Procedure TFooterColumn.SetAlignment(Value: TAlignment);
Begin
   If FAlignment <> Value Then
   Begin
      FAlignment := Value;
      Changed(False);
   End;
End;

Procedure TFooterColumn.SetBevel(Value: TFooterPanelBevel);
Begin
   If FBevel <> Value Then
   Begin
      FBevel := Value;
      Changed(False);
   End;
End;

Procedure TFooterColumn.SetFieldName(Value: String);
Begin
   If FFieldName <> Value Then
   Begin
      FFieldName := Value;
      Changed(False);
   End;
End;

Procedure TFooterColumn.SetNumPrecision(Value: Byte);
Begin
   If FNumType = ntInteger Then 
      Value := 0;
   If FNumPrecision <> Value Then
   Begin
      FNumPrecision := Value;
      Changed(False);
   End;
End;

Procedure TFooterColumn.SetNumType(Value: TNumType);
Begin
   If FOperation = coCount Then
      Value := ntInteger;
   If FNumType <> Value Then
   Begin
      FNumType := Value;
      If FNumType = ntInteger Then 
         SetNumPrecision(0);
      Changed(False);
   End;
End;

Procedure TFooterColumn.SetOperation(Value: TFooterColumnOperation);
Begin
   If FOperation <> Value Then
   Begin
      FOperation := Value;
      If FOperation = coCount Then
         SetNumType(ntInteger);
      Changed(False);
   End;
End;

Procedure TFooterColumn.SetTextBefore(Value: String);
Begin
   If FTextBefore <> Value Then
   Begin
      FTextBefore := Value;
      Changed(False);
   End;
End;

Procedure TFooterColumn.SetTextAfter(Value: String);
Begin
   If FTextAfter <> Value Then
   Begin
      FTextAfter := Value;
      Changed(False);
   End;
End;

Procedure TFooterColumn.SetWidthIfNoResize(Value: Integer);
Begin
   If FWidthIfNoResize <> Value Then
   Begin
      FWidthIfNoResize := Value;
      Changed(True);
   End;
End;

{ TFooterColumns }

Constructor TFooterColumns.Create(FooterBar: TUltimDBFooter);
Begin
   Inherited Create(TFooterColumn);
   FFooterBar := FooterBar;
End;

Function TFooterColumns.Add: TFooterColumn;
Begin
   Result := TFooterColumn(Inherited Add);
End;

Function TFooterColumns.GetOwner: TPersistent;
Begin
   Result := FFooterBar;
End;

Procedure TFooterColumns.Update(Item: TCollectionItem);
Begin
   FFooterBar.UpdateDataPanels(True);
End;

Function TFooterColumns.GetItem(Index: Integer): TFooterColumn;
Begin
   Result := TFooterColumn(Inherited GetItem(Index));
End;

Procedure TFooterColumns.SetItem(Index: Integer; Value: TFooterColumn);
Begin
   Inherited SetItem(Index, Value);
End;

{ TUltimDBFooter }

Constructor TUltimDBFooter.Create(AOwner: TComponent);
Begin
   Inherited Create(AOwner);
   ControlStyle := [csCaptureMouse, csClickEvents, csDoubleClicks, csOpaque];
   Color := clBtnFace;
   Height := 19;
   Width := 200;
   Align := alNone;
   FPanels := TFooterPanels.Create(Self);
   FColumns := TFooterColumns.Create(Self);
   FCanvas := TControlCanvas.Create;
   TControlCanvas(FCanvas).Control := Self;
   FSizeGrip := False;
   FLeftPanel := -1;
   ParentFont := False;
   FUseSystemFont := True;
   SyncToSystemFont;
   FDataLink := TUltimDataLink.Create(Self);
   FDataLink.DataSource := Nil;
   FDBGrid := Nil;
   IgnoreResizing := False;
   IgnoreScrolling := False;
End;

Destructor TUltimDBFooter.Destroy;
Begin
   If FDBGrid <> Nil Then
   Begin
      FDBGrid.UnregisterFooter(Self);
      FDBGrid := Nil;
   End;
   FDataLink.FFooterCtrl := Nil;
   FDataLink.Free;
   FDataLink := Nil;
   FCanvas.Free;
   FColumns.Free;
   FPanels.Free;
   Inherited Destroy;
End;

Procedure TUltimDBFooter.CreateParams(Var Params: TCreateParams);
Const
   GripStyles: Array[Boolean] Of DWORD = (CCS_TOP, SBARS_SIZEGRIP);
Begin
   InitCommonControl(ICC_BAR_CLASSES);
   Inherited CreateParams(Params);
   CreateSubClass(Params, STATUSCLASSNAME);
   With Params Do
   Begin
      Style := Style Or GripStyles[FSizeGrip And (Parent Is TCustomForm) And
         (TCustomForm(Parent).BorderStyle In [bsSizeable, bsSizeToolWin])];
      WindowClass.style := WindowClass.style And Not CS_HREDRAW;
   End;
End;

Procedure TUltimDBFooter.CreateWnd;
Begin
   Inherited CreateWnd;
   SendMessage(Handle, SB_SETBKCOLOR, 0, ColorToRGB(Color));
   UpdatePanels(True, False);
End;

Procedure TUltimDBFooter.DrawPanel(Panel: TFooterPanel; Const Rect: TRect);
Begin
   If Assigned(FOnDrawPanel) Then
      FOnDrawPanel(Self, Panel, Rect)
   Else
      FCanvas.FillRect(Rect);
End;

Procedure TUltimDBFooter.SetPanels(Value: TFooterPanels);
Begin
   FPanels.Assign(Value);
End;

Procedure TUltimDBFooter.DoRightToLeftAlignment(Var Str: String;
   AAlignment: TAlignment; ARTLAlignment: Boolean);
Begin
   If ARTLAlignment Then
      ChangeBiDiModeAlignment(AAlignment);

   Case AAlignment Of
      taCenter: Insert(#9, Str, 1);
      taRightJustify: Insert(#9#9, Str, 1);
   End;
End;

Procedure TUltimDBFooter.CMBiDiModeChanged(Var Message: TMessage);
Var
   Loop: Integer;
Begin
   Inherited;
   If HandleAllocated Then
   Begin
      For Loop := 0 To Panels.Count - 1 Do
         If Panels[Loop].ParentBiDiMode Then
            Panels[Loop].ParentBiDiModeChanged;
      UpdatePanels(True, True);
   End;
End;

Procedure TUltimDBFooter.FlipChildren(AllLevels: Boolean);
Var
   Loop, FirstWidth, LastWidth: Integer;
   APanels: TFooterPanels;
Begin
   If HandleAllocated And (Panels.Count > 0) Then
   Begin
      { Get the true width of the last panel }
      LastWidth := ClientWidth;
      FirstWidth := Panels[0].Width;
      For Loop := 0 To Panels.Count - 2 Do
         Dec(LastWidth, Panels[Loop].Width);
      { Flip 'em }
      APanels := TFooterPanels.Create(Self);
      Try
         For Loop := 0 To Panels.Count - 1 Do
            With APanels.Add Do
               Assign(Self.Panels[Loop]);
         For Loop := 0 To Panels.Count - 1 Do
            Panels[Loop].Assign(APanels[Panels.Count - Loop - 1]);
      Finally
         APanels.Free;
      End;
      { Set the width of the last panel }
      If Panels.Count > 1 Then
      Begin
         Panels[Panels.Count - 1].Width := FirstWidth;
         Panels[0].Width := LastWidth;
      End;
      UpdatePanels(True, True);
   End;
End;

Procedure TUltimDBFooter.SetSizeGrip(Value: Boolean);
Begin
   If FSizeGrip <> Value Then
   Begin
      FSizeGrip := Value;
      RecreateWnd;
   End;
End;

Procedure TUltimDBFooter.SyncToSystemFont;
Begin
   If FUseSystemFont Then
      Font := Screen.HintFont;
End;

Procedure TUltimDBFooter.UpdatePanel(Index: Integer; Repaint: Boolean);
Var
   Flags: Integer;
   S: String;
   PanelRect: TRect;
Begin
   If HandleAllocated Then
      With Panels[Index] Do
      Begin
         If Not Repaint Then
         Begin
            FUpdateNeeded := True;
            SendMessage(Handle, SB_GETRECT, Index, Integer(@PanelRect));
            InvalidateRect(Handle, @PanelRect, True);
            Exit;
         End
         Else If Not FUpdateNeeded Then
            Exit;
         FUpdateNeeded := False;
         Flags := 0;
         Case Bevel Of
            pbNone: Flags := SBT_NOBORDERS;
            pbRaised: Flags := SBT_POPOUT;
         End;
         If UseRightToLeftReading Then
            Flags := Flags Or SBT_RTLREADING;
         If Style = psOwnerDraw Then
            Flags := Flags Or SBT_OWNERDRAW;
         S := Text;
         If UseRightToLeftAlignment Then
            DoRightToLeftAlignment(S, Alignment, UseRightToLeftAlignment)
         Else
            Case Alignment Of
               taCenter: Insert(#9, S, 1);
               taRightJustify: Insert(#9#9, S, 1);
            End;
         SendMessage(Handle, SB_SETTEXT, Index Or Flags, Integer(PChar(S)));
      End;
End;

Procedure TUltimDBFooter.UpdatePanels(UpdateRects, UpdateText: Boolean);
Const
   MaxPanelCount = 128;
Var
   I, Count, PanelPos: Integer;
   PanelEdges: Array[0..MaxPanelCount - 1] Of Integer;
Begin
   If HandleAllocated Then
   Begin
      Count := Panels.Count;
      If UpdateRects Then
      Begin
         If Count > MaxPanelCount Then
            Count := MaxPanelCount;
         If Count = 0 Then
         Begin
            PanelEdges[0] := -1;
            SendMessage(Handle, SB_SETPARTS, 1, Integer(@PanelEdges));
            SendMessage(Handle, SB_SETTEXT, 0, Integer(PChar('')));
         End
         Else
         Begin
            PanelPos := 0;
            For I := 0 To Count - 2 Do
            Begin
               Inc(PanelPos, Panels[I].Width);
               PanelEdges[I] := PanelPos;
            End;
            PanelEdges[Count - 1] := -1;
            SendMessage(Handle, SB_SETPARTS, Count, Integer(@PanelEdges));
         End;
      End;
      For I := 0 To Count - 1 Do
         UpdatePanel(I, UpdateText);
   End;
End;

Procedure TUltimDBFooter.CMWinIniChange(Var Message: TMessage);
Begin
   Inherited;
   If (Message.WParam = 0) Or (Message.WParam = SPI_SETNONCLIENTMETRICS) Then
      SyncToSystemFont;
End;

Procedure TUltimDBFooter.CNDrawItem(Var Message: TWMDrawItem);
Var
   SaveIndex: Integer;
Begin
   With Message.DrawItemStruct^ Do
   Begin
      SaveIndex := SaveDC(hDC);
      FCanvas.Lock;
      Try
         FCanvas.Handle := hDC;
         FCanvas.Font := Font;
         FCanvas.Brush.Color := clBtnFace;
         FCanvas.Brush.Style := bsSolid;
         DrawPanel(Panels[itemID], rcItem);
      Finally
         FCanvas.Handle := 0;
         FCanvas.Unlock;
         RestoreDC(hDC, SaveIndex);
      End;
   End;
   Message.Result := 1;
End;

Procedure TUltimDBFooter.WMPaint(Var Message: TWMPaint);
Begin
   UpdatePanels(False, True);
   Inherited;
End;

Procedure TUltimDBFooter.WMSize(Var Message: TWMSize);
Begin
   { Eat WM_SIZE message to prevent control from doing alignment }
   If Not (csLoading In ComponentState) Then
      Resize;
   Repaint;
End;

Function TUltimDBFooter.IsFontStored: Boolean;
Begin
   Result := Not FUseSystemFont And Not ParentFont And Not DesktopFont;
End;

Procedure TUltimDBFooter.SetUseSystemFont(Const Value: Boolean);
Begin
   If FUseSystemFont <> Value Then
   Begin
      FUseSystemFont := Value;
      If Value Then
      Begin
         If ParentFont Then
            ParentFont := False;
         SyncToSystemFont;
      End;
   End;
End;

Procedure TUltimDBFooter.CMColorChanged(Var Message: TMessage);
Begin
   Inherited;
   RecreateWnd;
End;

Procedure TUltimDBFooter.CMParentFontChanged(Var Message: TMessage);
Begin
   Inherited;
   If FUseSystemFont And ParentFont Then
      FUseSystemFont := False;
End;

Function TUltimDBFooter.ExecuteAction(Action: TBasicAction): Boolean;
Begin
   Result := Inherited ExecuteAction(Action);
End;

Procedure TUltimDBFooter.CMSysColorChange(Var Message: TMessage);
Begin
   Inherited;
   RecreateWnd;
End;

Procedure TUltimDBFooter.CMSysFontChanged(Var Message: TMessage);
Begin
   Inherited;
   SyncToSystemFont;
End;

Procedure TUltimDBFooter.ChangeScale(M, D: Integer);
Begin
   If UseSystemFont Then
      { Footer bar size based on system font size }
      ScalingFlags := [sfTop];
   Inherited;
End;

Procedure TUltimDBFooter.SetColumns(Value: TFooterColumns);
Begin
   FColumns.Assign(Value);
End;

Procedure TUltimDBFooter.Notification(AComponent: TComponent; Operation: TOperation);
Begin
   Inherited Notification(AComponent, Operation);
   If (Operation = opRemove) And (AComponent = FDBGrid) Then
   Begin
      FDBGrid := Nil;
      FPanels.Clear;
   End;
End;

Procedure TUltimDBFooter.SetDBGrid(Value: TUltimDBGrid);
Begin
   If FDBGrid <> Value Then
   Begin
      If Assigned(FDBGrid) Then
         FDBGrid.UnregisterFooter(Self);
      FDBGrid := Value;
      FPanels.Clear;
      If Assigned(FDBGrid) Then
         FDBGrid.RegisterFooter(Self);
   End;
   If Value <> Nil Then
      Value.FreeNotification(Self);
End;

Procedure TUltimDBFooter.FooterMessage(Msg: Integer);
Begin
   Case Msg Of
      FTR_ACTIVE:
      Begin
         FDataLink.DataSource := FDBGrid.DataSource;
         If FDataLink.Active Then
            FDataLink.FRecCount := FDataLink.DataSet.RecordCount
         Else
            FDataLink.FRecCount := 0;
         UpdateDataPanels(True);
      End;
      FTR_COLWIDTHS:
      Begin
         UpdateDataPanels(False);
      End;
      FTR_SCROLL:
      Begin
         If Not IgnoreScrolling And (FLeftPanel <> FDBGrid.LeftCol) Then
            UpdateDataPanels(False);
      End;
   End;
End;

Procedure TUltimDBFooter.CalcFields;
Var
   RecCount: Integer;
   FtrCol: Integer;
   Field: TField;
   FldValue: Variant;
Begin
   { Initialisations }
   RecCount := 0;
   For FtrCol := 0 To FColumns.Count - 1 Do
   Begin
      Field := FDataLink.DataSet.FindField(FColumns[FtrCol].FieldName);
      If Field = Nil Then
         FColumns[FtrCol].FFieldNo := -1
      Else
         FColumns[FtrCol].FFieldNo := Field.Index;
      FColumns[FtrCol].FCalcValue := Null;
      FColumns[FtrCol].FError := False;
   End;
   If Assigned(FOnBeforeReadFields) Then
      FOnBeforeReadFields(Self);
   { Mmorisation de l'enregistrement en cours }
   FDataLink.DataSet.DisableControls;
   FDBGrid.SaveGridPosition;
   { Parcours des enregistrements et calcul }
   Try
      FDataLink.DataSet.First;
      While Not FDataLink.DataSet.EOF Do
      Begin
         Inc(RecCount);
         For FtrCol := 0 To FColumns.Count - 1 Do
            With FColumns[FtrCol] Do
            Begin
               If FFieldNo <> -1 Then
               Begin
                  Try
                     FldValue := FDataLink.DataSet.Fields[FFieldNo].AsVariant;
                     If Assigned(FOnReadFieldValue) Then
                        FOnReadFieldValue(Self, FColumns[FtrCol], FldValue);
                  Except
                     FError := True;
                     If Assigned(FOnReadFieldValue) Then
                        FOnReadFieldValue(Self, FColumns[FtrCol], FTR_ERROR);
                  End;
                  Try
                     If Not FError And (FldValue <> Null) Then
                        Case FOperation Of
                           coSum, coAverage:
                              If FCalcValue = Null Then
                                 FCalcValue := FldValue
                              Else
                                 FCalcValue := FCalcValue + FldValue;
                           coMax:
                              If (FCalcValue = Null) Or (FldValue > FCalcValue) Then
                                 FCalcValue := FldValue;
                           coMin:
                              If (FCalcValue = Null) Or (FldValue < FCalcValue) Then
                                 FCalcValue := FldValue;
                        End;
                  Except
                     FError := True;
                  End; 
               End;
            End;
         FDataLink.DataSet.Next;
      End;
      { Calculs complmentaires }
      For FtrCol := 0 To FColumns.Count - 1 Do
         With FColumns[FtrCol] Do
         Begin
            If Not FError Then
               Try
                  Case FOperation Of
                     coAverage: FCalcValue := FCalcValue / RecCount;
                     coCount: FCalcValue := RecCount;
                  End;
               Except
                  FError := True;
               End;
         End;
   Finally
      { Retour  l'enregistrement mmoris }
      FDBGrid.RestoreGridPosition;
      FDataLink.DataSet.EnableControls;
   End;
End;

Procedure TUltimDBFooter.UpdateDataPanels(ReCalc: Boolean);

   Function FormatVariant(VarColumn: TFooterColumn): String;
   Var
      Precision: String;
   Begin
      Try
         If VarColumn.FCalcValue = Null Then
            { Pas de valeur }
            Result := ''
         Else If VarType(VarColumn.FCalcValue) In [varSmallint..varCurrency] Then
         Begin
            { Valeur numrique }
            Precision := IntToStr(VarColumn.FNumPrecision);
            Case VarColumn.FNumType Of
               ntInteger:
                  Result := Format('%d', [Integer(VarColumn.FCalcValue)]);
               ntFixed:
                  Result := Format('%.' + Precision + 'n', [Double(VarColumn.FCalcValue)]);
               ntGeneral:
                  Result := Format('%.' + Precision + 'g', [Double(VarColumn.FCalcValue)]);
               ntScientific:
                  Result := Format('%.' + Precision + 'e', [Double(VarColumn.FCalcValue)]);
               ntCurrency:
                  Result := Format('%.' + Precision + 'm', [Currency(VarColumn.FCalcValue)]);
               Else
                  Result := String(VarColumn.FCalcValue);
            End;
         End
         Else
            { Valeur non numrique }
            Result := String(VarColumn.FCalcValue);
         { Ajout des textes d'encadrement }
         Result := VarColumn.FTextBefore + Result + VarColumn.FTextAfter;
      Except
         Result := FTR_ERROR;
      End;
   End;

Var
   GridCol: Integer;

   Procedure CreatePanels;
   Var  
      FtrCol: Integer;
      Trouve: Boolean;
      DisplayText: String;
      DrawStyle: TFooterPanelStyle;
   Begin
      If FDBGrid.Columns[GridCol].Visible Then
         With FPanels.Add Do
         Begin
            Trouve := False;
            For FtrCol := 0 To FColumns.Count - 1 Do
               If CompareText(FDBGrid.Columns[GridCol].FieldName,
                  FColumns[FtrCol].FieldName) = 0 Then
               Begin
                  Alignment := FColumns[FtrCol].FAlignment;
                  Bevel := FColumns[FtrCol].FBevel;
                  FieldName := FColumns[FtrCol].FFieldName;
                  If FColumns[FtrCol].FError Then
                     DisplayText := FTR_ERROR
                  Else
                     DisplayText := FormatVariant(FColumns[FtrCol]);
                  If Assigned(FOnBeforeDisplayText) Then
                  Begin
                     DrawStyle := Style;
                     FOnBeforeDisplayText(Self, FColumns[FtrCol], FColumns[FtrCol].FCalcValue,
                        DisplayText, DrawStyle);
                     Style := DrawStyle;
                  End;
                  Text := DisplayText;
                  If IgnoreResizing Then
                     Width := FColumns[FtrCol].FWidthIfNoResize
                  Else
                     Width := FDBGrid.Columns[GridCol].Width;
                  Trouve := True;
                  Break;
               End;
            If Not Trouve Then
            Begin
               Bevel := pbNone;
               FieldName := '';
               Text := '';
               Width := FDBGrid.Columns[GridCol].Width;
            End;
            If dgColLines In FDBGrid.Options Then
               Width := Width + TDrawGrid(FDBGrid).GridLineWidth;
         End;
   End;

Var
   MaxCol: Integer;

Begin
   FPanels.BeginUpdate;
   Try
      FPanels.Clear;
      If (FColumns.Count > 0) And (FDBGrid <> Nil) And FDataLink.Active Then
      Begin
         { Calcul des valeurs  afficher ? }
         If ReCalc Then
            CalcFields;
         { Cration du panel de l'indicateur }
         If dgIndicator In FDBGrid.Options Then
            With FPanels.Add Do
            Begin
               Bevel := pbNone;
               FieldName := '';
               Text := '';
               Width := IndicatorWidth;
               If dgColLines In FDBGrid.Options Then
                  Width := Width + TDrawGrid(FDBGrid).GridLineWidth;
            End;
         { Cration des panels des colonnes figes }
         For GridCol := 0 To FDBGrid.FixedCols - 1 Do
            CreatePanels;
         { Cration des panels des colonnes mobiles }
         If IgnoreScrolling Then
         Begin
            FLeftPanel := FDBGrid.FixedCols + 1;
            MaxCol := FDBGrid.Columns.Count - 1;
         End
         Else
         Begin
            FLeftPanel := FDBGrid.LeftCol;
            MaxCol := FLeftPanel + FDBGrid.VisibleColCount - 1;
            If MaxCol >= FDBGrid.Columns.Count Then
               MaxCol := FDBGrid.Columns.Count - 1;
         End;
         For GridCol := FLeftPanel - 1 To MaxCol Do
            CreatePanels;
         { Cration du panel de fin occupant l'espace restant }
         With FPanels.Add Do
         Begin
            Bevel := pbNone;
            FieldName := '';
            Text := '';
            Width := 1;
         End;
      End;
   Finally
      FPanels.EndUpdate;
   End;
End;

End.