Unit DBImageEditor;

{
-----------------------------------------------------------------------------

                              DBImageEditor 1.0
                        
                        VCL for Delphi and C++Builder
                       (C) Frdric Leneuf-Magaud 2001

-----------------------------------------------------------------------------

 Please read ReadMe.txt for conditions of use

 This component is provided 'as-is', without any express or implied warranty.
 In no event shall the author be held liable for any damages arising from the
 use of this component.

 Veuillez lire le fichier LisezMoi.txt pour les conditions d'utilisation

 Ce composant est fourni tel quel, sans aucune garantie.
 En aucun cas, l'auteur ne pourra tre tenu pour responsable des dommages
 rsultant de l'utilisation de ce composant.

-----------------------------------------------------------------------------
}

Interface

{$I OPTIONS.INC}

Uses
   Windows, Messages, SysUtils, Classes, Controls, Forms, Dialogs, ExtDlgs, Buttons, ComCtrls,
   ToolWin, DB, DBCtrls, FileUtil;

{$I LANG_STR.INC}

Type
   TImageEditButtons = (ibImport, ibEdit, ibEditWith, ibExport);
   TIEButtonsSet = Set Of TImageEditButtons;

   TDBImageEditor = Class(TFrame)
      DBImage: TDBImage;
      ToolBar: TToolBar;
      SB_Import: TSpeedButton;
      SB_Edit: TSpeedButton;
      SB_EditWith: TSpeedButton;
      SB_Export: TSpeedButton;
      Procedure DBImageClick(Sender: TObject);
      Procedure ToolBarClick(Sender: TObject);
      Procedure SB_ImportClick(Sender: TObject);
      Procedure SB_EditClick(Sender: TObject);
      Procedure SB_EditWithClick(Sender: TObject);
      Procedure SB_ExportClick(Sender: TObject);
   Private
      FImageLink: TFieldDataLink;
      FButtons: TIEButtonsSet;
      FPaintProgram: String;
      FWaitBeforeUpdate: Boolean;
      FCanRevert: Boolean;
      FPicSaved: Boolean;
      FOriginalPic: String;
      Function GetDataSource: TDataSource;
      Procedure SetDataSource(Value: TDataSource);
      Function GetDataField: String;
      Procedure SetDataField(Value: String);
      Procedure SetButtons(Value: TIEButtonsSet);
      Procedure ActiveChange(Sender: TObject);
      Procedure SetPaintProgram(Value: String);
      Procedure SetWaitBeforeUpdate(Value: Boolean);
      Procedure SetCanRevert(Value: Boolean);
      Procedure DataChange(Sender: TObject);
      Procedure SaveOriginalPic;
      Procedure WMChar(Var Msg: TWMChar); Message WM_CHAR;
   Protected
   Public
      Constructor Create(AOwner: TComponent); Override;
      Destructor Destroy; Override;
      Function Execute(Const Command, Path: String): Boolean;
      Function GetUniqueFileName(Const Prefix, Extension: String): String;
   Published
      Property DataSource: TDataSource Read GetDataSource Write SetDataSource;
      Property DataField: String Read GetDataField Write SetDataField;
      { Buttons: dfinit les boutons affichs sur la barre d'outils }
      Property Buttons: TIEButtonsSet Read FButtons Write SetButtons;
      { PaintProgram: dfinit le programme de dessin ou de retouche utilis pour l'dition }
      Property PaintProgram: String Read FPaintProgram Write SetPaintProgram;
      { WaitBeforeUpdate: attend la confirmation de la fin d'dition avant de mettre  jour.
        Normalement, le composant est capable de dtecter la fermeture du programme d'dition mais
        certains programmes (plutt rares) ont un chargement en cascade qui leurre le composant.
        Pour tre sr que la mise  jour se fasse bien aprs l'dition et non pendant, on peut
        demander une confirmation de la fin d'dition en mettant WaitBeforeUpdate  True.
        PaintBrush, par exemple, ncessite WaitBeforeUpdate  True. }
      Property WaitBeforeUpdate: Boolean Read FWaitBeforeUpdate Write SetWaitBeforeUpdate
         Default True;
      { CanRevert: permet de revenir  l'tat de l'image avant sa modification avec Echap/Esc }
      Property CanRevert: Boolean Read FCanRevert Write SetCanRevert Default True;
   End;

{ Nota bene: l'change entre le composant et l'diteur d'image se fait  l'aide d'un fichier
  temporaire au nom unique dtruit en fin d'dition. }

Implementation

{$R *.DFM}

Constructor TDBImageEditor.Create(AOwner: TComponent);
Begin
   Inherited;
   FImageLink := TFieldDataLink.Create;
   FImageLink.Control := Self;
   FImageLink.OnActiveChange := ActiveChange;
   FImageLink.OnDataChange := DataChange;
   FButtons := [ibImport, ibEdit, ibEditWith, ibExport];
   FPaintProgram := '';
   FWaitBeforeUpdate := True;
   FCanRevert := True;
   FPicSaved := False;
   FOriginalPic := '';
End;

Destructor TDBImageEditor.Destroy;
Begin
   FImageLink.Control := Nil;
   FImageLink.Free;
   FImageLink := Nil;
   Inherited;
End;

Function TDBImageEditor.GetDataSource: TDataSource;
Begin
   If DBImage <> Nil Then
      Result := DBImage.DataSource
   Else
      Result := Nil;
End;

Procedure TDBImageEditor.SetDataSource(Value: TDataSource);
Begin
   If DBImage <> Nil Then   
      DBImage.DataSource := Value;
   If FImageLink <> Nil Then  
      FImageLink.DataSource := Value;
End;

Function TDBImageEditor.GetDataField: String;
Begin
   If DBImage <> Nil Then
      Result := DBImage.DataField
   Else
      Result := '';
End;

Procedure TDBImageEditor.SetDataField(Value: String);
Begin
   If DBImage <> Nil Then   
      DBImage.DataField := Value;
   If FImageLink <> Nil Then
   Begin
      FImageLink.FieldName := Value;
      ActiveChange(Self);
   End;
End;

Procedure TDBImageEditor.SetButtons(Value: TIEButtonsSet);
Begin
   FButtons := Value;
   { On ne peut pas avoir EditWith sans Edit }
   If Not SB_Edit.Visible And (ibEditWith In FButtons) Then
      FButtons := FButtons + [ibEdit];
   If Not (ibEdit In FButtons) Then
      FButtons := FButtons - [ibEditWith];
   { Modification de la visibilit des boutons }
   SB_Import.Visible := (ibImport In FButtons);
   SB_Edit.Visible := (ibEdit In FButtons);
   SB_EditWith.Visible := (ibEditWith In FButtons);
   SB_Export.Visible := (ibExport In FButtons);
End;

Procedure TDBImageEditor.ActiveChange(Sender: TObject);
Begin
   If FImageLink.Active Then
   Begin
      SB_Import.Enabled := FImageLink.CanModify;
      SB_Edit.Enabled := FImageLink.CanModify;
      SB_EditWith.Enabled := SB_Edit.Enabled;
      SB_Export.Enabled := FImageLink.Field <> Nil;
   End
   Else
   Begin
      SB_Import.Enabled := False;
      SB_Edit.Enabled := False;
      SB_EditWith.Enabled := False;
      SB_Export.Enabled := False;
   End;
End;

Procedure TDBImageEditor.SetPaintProgram(Value: String);
Begin
   If FPaintProgram <> Value Then
      FPaintProgram := Value;
End;

Procedure TDBImageEditor.SetWaitBeforeUpdate(Value: Boolean);
Begin
   If FWaitBeforeUpdate <> Value Then
      FWaitBeforeUpdate := Value;
End;

Procedure TDBImageEditor.SetCanRevert(Value: Boolean);
Begin
   If FCanRevert <> Value Then
      FCanRevert := Value;
End;

Procedure TDBImageEditor.DataChange(Sender: TObject);
Begin
   If Assigned(DBImage.DataSource) And (DBImage.DataSource.State = dsBrowse) Then
      FPicSaved := False;
End;

Procedure TDBImageEditor.SaveOriginalPic;
Begin
   If FCanRevert Then
   Begin
      { Stockage de l'image originale (avant modification) dans FOriginalPic }
      If (DBImage.Field <> Nil) And (DBImage.Field Is TBlobField) Then
         FOriginalPic := DBImage.Field.AsString
      Else
         FOriginalPic := '';
      FPicSaved := True;
   End;
End;

Procedure TDBImageEditor.WMChar(Var Msg: TWMChar);
Begin
   If (Char(Msg.CharCode) = #27) And FCanRevert And FPicSaved Then
   Begin
      { Annulation: on renvoie la valeur de champ stocke dans FOriginalPic }
      If (DBImage.Field <> Nil) And (DBImage.Field Is TBlobField) Then
      Begin
         DBImage.Field.AsString := FOriginalPic;
         FPicSaved := False;
      End;
   End;
   Inherited;
End;

Procedure TDBImageEditor.DBImageClick(Sender: TObject);
Begin
   { Passage du focus au TFrame pre }
   If Visible Then
      SetFocus;
End;

Procedure TDBImageEditor.ToolBarClick(Sender: TObject);
Begin
   { Passage du focus au TFrame pre }
   If Visible Then
      SetFocus;
End;

Procedure TDBImageEditor.SB_ImportClick(Sender: TObject);
Var
   FileOpen: TOpenPictureDialog;
Begin
   { Passage du focus au TFrame pre }
   If Visible Then
      SetFocus;
   { Dataset ouvert ? }
   If (DBImage.DataSource.DataSet = Nil) Or Not DBImage.DataSource.DataSet.Active Then
      Exit;
   { Ouverture de l'cran de slection d'image }
   FileOpen := TOpenPictureDialog.Create(Self);
   Try
      With FileOpen Do
      Begin
         Title := lsChooseImage;
         Filename := '';
         InitialDir := '';
         Filter := lsBMPFilter + '|' + lsDefaultFilter;
         Options := Options + [ofHideReadOnly, ofPathMustExist, ofFileMustExist];
      End;
      If FileOpen.Execute Then
      Begin
         { Mmorisation de l'image  remplacer }
         SaveOriginalPic;
         { Chargement de l'image slectionne }
         DBImage.DataSource.Edit;
         DBImage.Picture.LoadFromFile(FileOpen.FileName);
         DBImage.Field.Assign(DBImage.Picture);
      End;
   Finally
      FileOpen.Free;
   End;
End;

Function TDBImageEditor.Execute(Const Command, Path: String): Boolean;
Var
   StartupInfo: TStartupInfo;
   ProcessInfo: TProcessInformation;
Begin
   { Initialisations }
   Result := False;
   FillChar(StartupInfo, SizeOf(TStartupInfo), 0);
   With StartupInfo Do
   Begin
      cb := SizeOf(TStartupInfo);
      dwFlags := STARTF_USESHOWWINDOW Or STARTF_FORCEONFEEDBACK;
      wShowWindow := SW_SHOWNORMAL;
   End;
   { Lancement du programme }
   If Not CreateProcess(Nil, PChar(Command), Nil, Nil, False, NORMAL_PRIORITY_CLASS, Nil,
      PChar(Path), StartupInfo, ProcessInfo) Then
      Exit;
   { Attente de la fin du programme }
   Application.Minimize;
   With ProcessInfo Do
   Begin
      WaitForInputIdle(hProcess, INFINITE);
      WaitForSingleObject(hProcess, INFINITE);
      CloseHandle(hThread);
      CloseHandle(hProcess);
   End;
   Application.Restore;
   Result := True;
End;

Function TDBImageEditor.GetUniqueFileName(Const Prefix, Extension: String): String;
Var
   CalcSuffix: String;
Begin
   { Format: prfixe + nombre en hexa de millisecondes coules depuis le dmarrage + extension }
   Repeat
      CalcSuffix := IntToHex(GetTickCount, 4);
      Result := Prefix + CalcSuffix + Extension;
   Until Not FileExists(Result);
End;

Procedure TDBImageEditor.SB_EditClick(Sender: TObject);
Var
   TempImg: String;
Begin
   { Passage du focus au TFrame pre }
   If Visible Then
      SetFocus;
   { Dataset ouvert ? }
   If (DBImage.DataSource.DataSet = Nil) Or Not DBImage.DataSource.DataSet.Active Then
      Exit;
   { Programme d'dition dfini ? }
   FPaintProgram := Trim(FPaintProgram);
   If FPaintProgram = '' Then
   Begin
      Application.MessageBox(lsChooseBefore, lsError, MB_ICONWARNING);
      Exit;
   End;
   { Cration d'un fichier temporaire unique }
   TempImg := GetUniqueFileName('~T', '.BMP'); // .BMP obligatoire pour Picture.LoadFromFile
   TempImg := GetTempDir + TempImg;
   { Remplissage et dition du fichier temporaire }
   Screen.Cursor := crHourGlass;
   Try
      DBImage.Picture.SaveToFile(TempImg);
      If Execute('"' + FPaintProgram + '" "' + TempImg + '"', ExtractFilePath(FPaintProgram)) Then
      Begin
         { Message d'attente ? }
         If FWaitBeforeUpdate Then
            Application.MessageBox(lsWaiting, PChar(ExtractFileName(TempImg)), MB_OK);
         { Mmorisation de l'image  remplacer }
         SaveOriginalPic;
         { Chargement de l'image modifie }
         DBImage.DataSource.Edit;
         DBImage.Picture.LoadFromFile(TempImg);
         DBImage.Field.Assign(DBImage.Picture);
      End;
   Finally
      { Effacement du fichier temporaire }
      DeleteFile(TempImg);
      Screen.Cursor := crDefault;
   End;
   { Retour au composant }
   Application.BringToFront;
   If Visible Then
      SetFocus;
End;

Procedure TDBImageEditor.SB_EditWithClick(Sender: TObject);
Var
   FileOpen: TOpenDialog;
Begin
   { Passage du focus au TFrame pre }
   If Visible Then
      SetFocus;
   { Ouverture de l'cran de slection de programme }
   FileOpen := TOpenDialog.Create(Self);
   Try
      With FileOpen Do
      Begin
         Title := lsChoosePaintProg;
         Filename := FPaintProgram;
         InitialDir := ExtractFilePath(FPaintProgram);
         Filter := lsEXEFilter + '|' + lsDefaultFilter;
         Options := Options + [ofHideReadOnly, ofPathMustExist, ofFileMustExist];
      End;
      If FileOpen.Execute Then
      Begin
         { Mmorisation du programme slectionn }
         SetPaintProgram(FileOpen.FileName);
         SetWaitBeforeUpdate(Application.MessageBox(lsWaitBeforeUpdate,
            PChar(ExtractFileName(FileOpen.FileName)), MB_ICONQUESTION + MB_YESNO) = ID_YES);
      End;
   Finally
      FileOpen.Free;
   End;
End;

Procedure TDBImageEditor.SB_ExportClick(Sender: TObject);
Var
   FileSave: TSavePictureDialog;
Begin
   { Passage du focus au TFrame pre }
   If Visible Then
      SetFocus;
   { Dataset ouvert ? }
   If (DBImage.DataSource.DataSet = Nil) Or Not DBImage.DataSource.DataSet.Active Then
      Exit;
   { Ouverture de l'cran de sauvegarde de l'image }
   FileSave := TSavePictureDialog.Create(Self);
   Try
      With FileSave Do
      Begin
         Title := '';
         Filename := '';
         InitialDir := '';
         DefaultExt := 'BMP';
         Filter := lsBMPFilter + '|' + lsDefaultFilter;
         Options := Options + [ofHideReadOnly, ofNoReadOnlyReturn, ofOverwritePrompt];
      End;
      If FileSave.Execute Then
      Begin
         { Sauvegarde de l'image dans le fichier slectionn }
         DBImage.Picture.SaveToFile(FileSave.FileName);
      End;
   Finally
      FileSave.Free;
   End;
End;

End.