{$INCLUDE ..\cDefines.inc}
unit cRTFEncode;

{                                                                              }
{ Revision history:                                                            }
{  05/10/2002  0.01  Initial version                                           }
{                                                                              }

interface

uses
  { Delphi }
  Graphics;



{                                                                              }
{ TRTFText                                                                     }
{                                                                              }
type
  TRTFText = class
  protected
    FTitle     : String;
    FGenerator : String;
    FRTFText   : String;
    FColors    : Array of TColor;

    procedure AddTag(const Tag: String);
    function  GetColorIndex(const Color: TColor): Integer;
    function  GetAsString: String;

  public
    property  Title: String read FTitle write FTitle;
    property  Generator: String read FGenerator write FGenerator;

    procedure Clear;
    procedure AddText(const S: String);
    procedure AddLineBreak;
    procedure SetForeColor(const Color: TColor);
    procedure SetBoldStart;
    procedure SetBoldEnd;
    procedure SetItalicStart;
    procedure SetItalicEnd;
    procedure SetUnderlineStart;
    procedure SetUnderlineEnd;

    property  AsString: String read GetAsString;
  end;

implementation

uses
  { Delphi }
  Windows,
  SysUtils,

  { Fundamentals }
  cStrings;



{                                                                              }
{ RTF functions                                                                }
{                                                                              }
function EncodeRTFColor(const Color: TColor): String;
var C : LongInt;
  begin
    C := ColorToRGB(Color);
    Result := '\red' + IntToStr(GetRValue(C)) +
              '\green' + IntToStr(GetGValue(C)) +
              '\blue' + IntToStr(GetBValue(C)) + ';';
  end;

function rtfSafeText(const S: String): String;
begin
  Result := S;
  Result := StrReplaceChar(['\', '{', '}'], '_', Result);
  Result := StrRemoveChar(Result, [#0..#31]);
end;



{                                                                              }
{ TRTFText                                                                     }
{                                                                              }
procedure TRTFText.Clear;
  begin
    FRTFText := '';
  end;

procedure TRTFText.AddText(const S: String);
var T : String;
  begin
    T := StrReplace('\', '\\', S);
    T := StrReplace('{', '\{', T);
    T := StrReplace('}', '\}', T);
    FRTFText := FRTFText + T;
  end;

procedure TRTFText.AddTag(const Tag: String);
var S : String;
  begin
    S := '\' + Tag + ' ';
    FRTFText := FRTFText + S;
  end;

procedure TRTFText.AddLineBreak;
  begin
    FRTFText := FRTFText + CRLF;
    AddTag('par');
  end;

procedure TRTFText.SetForeColor(const Color: TColor);
  begin
    AddTag('cb' + IntToStr(GetColorIndex(Color)));
  end;

procedure TRTFText.SetBoldStart;
  begin
    AddTag('b');
  end;

procedure TRTFText.SetBoldend;
  begin
    AddTag('b0');
  end;

procedure TRTFText.SetItalicStart;
  begin
    AddTag('i');
  end;

procedure TRTFText.SetItalicend;
  begin
    AddTag('i0');
  end;

procedure TRTFText.SetUnderlineStart;
  begin
    AddTag('ul');
  end;

procedure TRTFText.SetUnderlineend;
  begin
    AddTag('ul0');
  end;

function TRTFText.GetColorIndex(const Color: TColor): Integer;
var I, L : Integer;
  begin
    L := Length(FColors);
    For I := 0 to L - 1 do
      if FColors [I] = Color then
        begin
          Result := I;
          exit;
        end;
    Result := L;
    SetLength(FColors, L + 1);
    FColors [L] := Color;
  end;

function TRTFText.GetAsString: String;
var I : Integer;
  begin
    Result := '{\rtf1\ansi\deff0\deftab720' + CRLF;
    Result := Result + '{\colortbl';
    For I := 0 to Length(FColors) - 1 do
      Result := Result + EncodeRTFColor(FColors [I]);
    Result := Result + '}' + CRLF;
    if FGenerator <> '' then
      Result := Result + '{\info{\comment Generated by ' + rtfSafeText(FGenerator) + '}' + CRLF;
    if FTitle <> '' then
      Result := Result + '{\title ' + rtfSafeText(FTitle) + '}}' + CRLF;
    Result := Result + FRTFText + '}';
  end;



end.

