{  Created by Jan Verhoeven - 10 June 2000
   jan1.verhoeven@wxs.nl
   http://jansfreeware.com

   This code may be freely used in any freeware application
   provided you keep this text in the source code.
   When you want to use this code in a commercial application
   you must obtain approval from the author.}

unit mtrXML;

interface

uses
  Windows,  SysUtils, Classes,  Dialogs, mtrGlobal;

type
  TXMLValueType=(xvtString,xvtCDATA);
  TXMLFilterOperator=(xfoNOP,xfoEQ,xfoIEQ,xfoNE,xfoINE,xfoGE,xfoIGE,xfoLE,xfoILE,xfoGT,xfoIGT,xfoLT,xfoILT);

  TXMLTree = class;

  TXMLFilterAtom=class(TObject)
  private
    FValue: string;
    FName: string;
    FOperator: TXMLFilterOperator;
    FAttributeFilter: boolean;
    procedure SetName(const Value: string);
    procedure SetOperator(const Value: TXMLFilterOperator);
    procedure SetValue(const Value: string);
    procedure SetAttributeFilter(const Value: boolean);
  public
    property Name:string read FName write SetName;
    property Operator:TXMLFilterOperator read FOperator write SetOperator;
    property Value:string read FValue write SetValue;
    property AttributeFilter:boolean read FAttributeFilter write SetAttributeFilter;
  end;


  TXMLFilter=class(TObject)
  private
    FName: string;
    FFilters: TList;
    procedure SetName(const Value: string);
    procedure SetFilters(const Value: TList);
  public
    constructor Create(FilterStr:string);
    destructor  Destroy; override;
    property    Name:string read FName write SetName;
    property    Filters:TList read FFilters write SetFilters;
  end;

  TXMLAttribute=class(TObject)
  private
    FName: string;
    FValue: variant;
    procedure   SetName(const Value: string);
    procedure   SetValue(const Value: variant);
  public
    constructor create(aName:string;aValue:variant);
    function    document:string;
    property    Name:string read FName write SetName;
    property    Value:variant read FValue write SetValue;
  end;

  TXMLNode= class(TObject)
  private
    FName: string;
    FValue: variant;
    FNodes: TList;
    FAttributes: TList;
    FParentNode: TXMLNode;
    FValueType: TXMLValueType;
    procedure SetName(const Value: string);
    procedure SetValue(const Value: variant);
    function GetValue: variant;
    procedure SetNodes(const Value: TList);
    procedure SetAttributes(const Value: TList);
    procedure SetParentNode(const Value: TXMLNode);
    procedure SetValueType(const Value: TXMLValueType);
  public
    constructor Create(aName:string;aValue:variant;aParent:TXMLNode);
    destructor  Destroy;override;
    function    AddNode(aName:string;aValue:variant):TXMLNode;
    function    AddNodeEx(aName:string;aValue:variant):TXMLNode;
    procedure   DeleteNode(index:integer);
    procedure   ClearNodes;
    function    AddAttribute(aName:string;aValue:variant):TXMLAttribute;
    procedure   DeleteAttribute(index:integer);
    procedure   ClearAttributes;
    function    Document(aLevel:integer):string;
    function    GetNodePath:string;
    function    GetNamedNode(aName:string):TXMLNode;
    function    SelectSingleNode(pattern:string):TXMLNode;
    procedure   SelectNodes(pattern:string;aList:TList);
    function    TransformNode(stylesheet:TXMLNode):string;
    function    Process(aLevel:integer;node:TXMLNode):string;
    function    FindNamedNode(aName:string):TXMLNode;
    procedure   FindNamedNodes(aName:string;aList:TList);
    procedure   GetAllNodes(aList:TList);
    function    GetNamedAttribute(aName:string):TXMLAttribute;
    procedure   FindNamedAttributes(aName:string;aList:TList);
    function    MatchFilter(objFilter:TXMLFilter):boolean;
    procedure   MatchPattern(aPattern:string;aList:TList);
    procedure   GetNodeNames(aList:TStringList);
    procedure   GetAttributeNames(aList:TStringList);
    function    GetNameSpace:string;
    function    HasChildNodes:boolean;
    function    CloneNode:TXMLNode;
    function    FirstChild:TXMLNode;
    function    LastChild:TXMLNode;
    function    PreviousSibling:TXMLNode;
    function    NextSibling:TXMLNode;
    function    MoveAddNode(Dest:TXMLNode):TXMLNode;
    function    MoveInsertNode(Dest:TXMLNode):TXMLNode;
    function    RemoveChildNode(aNode:TXMLNode):TXMLNode;
    property    Name:string read FName write SetName;
    property    Value:variant read GetValue write SetValue;
    property    ValueType:TXMLValueType read FValueType write SetValueType;
    property    Nodes:TList read FNodes write SetNodes;
    property    ParentNode:TXMLNode read FParentNode write SetParentNode;
    property    Attributes:TList read FAttributes write SetAttributes;
  end;


  TXMLTree = class(TXMLNode)
  private
    FLines: TStringlist;
    FNodeCount:integer;
    procedure   SetLines(const Value: TStringlist);
    function getText: string;
    procedure setText(const Value: string);
    { Private declarations }
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(aName:string;aValue:variant;aParent:TXMLNode);
    destructor  Destroy;override;
    procedure   ParseXML;
    procedure   LoadFromFile(fn:string);
    procedure   LoadFromStream(Stream:TStream);
    procedure   SaveToFile(aFile:string);
    procedure   SaveToStream(Stream:TStream);
    function    asText:string;
    property    Lines:TStringlist read FLines write SetLines;
    property    NodeCount:integer read FNodeCount;
    property    Text:string read getText write setText;
  published
    { Published declarations }
  end;


  procedure PreProcessXML(aList:Tstringlist);

implementation

const
  cr = chr(13)+chr(10);
  tab = chr(9);

function ExpandCDATA(aValue:string):string;
begin
  result:=stringreplace(aValue,'\n ',cr,[rfreplaceall]);
  result:=stringreplace(result,'\t ',tab,[rfreplaceall]);
end;

procedure PreProcessXML(aList:Tstringlist);
const
  crlf  = chr(13)+chr(10);
  tab = chr(9);
var oList:TStringlist;
    s,xTag,xText,xData:string;
    p1,p2,c:integer;
    aLevel:integer;

    function clean(aText:string):string;
    begin
      result:=stringreplace(aText,crlf,' ',[rfreplaceall]);
      result:=stringreplace(result,tab,' ',[rfreplaceall]);
      result:=trim(result);
    end;

    function cleanCDATA(aText:string):string;
    begin
      result:=stringreplace(aText,crlf,'\n ',[rfreplaceall]);
      result:=stringreplace(result,tab,'\t ',[rfreplaceall]);
    end;

    function spc:string;
    begin
      if alevel<1 then
        result:=''
      else
        result:=stringofchar(' ',2*aLevel);
    end;
begin
  oList:=TStringlist.create;
  s:=aList.text;
  xText:='';
  xTag:='';
  p1:=1;
  c:=length(s);
  aLevel:=0;
  repeat
    p2:=Q_posstr('<',s,p1);
    if p2>0 then begin
      xText:=trim(copy(s,p1,p2-p1));
      if xText<>'' then begin
        oList.Append('TX:'+clean(xText));
      end;
      p1:=p2;
      // check for CDATA
      if uppercase(copy(s,p1,9))='<![CDATA[' then begin
        p2:=Q_posstr(']]>',s,p1);
        xData:=copy(s,p1+9,p2-p1-9);
        oList.Append('CD:'+cleanCDATA(xData));
        p1:=p2+2;
      end
      else begin
        p2:=Q_posstr('>',s,p1);
        if p2>0 then begin
          xTag:=copy(s,p1+1,p2-p1-1);
          p1:=p2;
          if xTag[1]='/' then begin
            delete(xTag,1,1);
            oList.Append('CT:'+clean(xTag));
            dec(aLevel);
          end
          else if xtag[length(xTag)]='/' then begin
            oList.Append('ET:'+clean(xTag));
          end
          else begin
            inc(aLevel);
            oList.Append('OT:'+clean(xTag));
          end
        end
      end
    end
    else begin
      xText:=trim(copy(s,p1,length(s)));
      if xText<>'' then begin
        oList.Append('TX:'+clean(xText));
      end;
      p1:=c;
    end;
    inc(p1);
  until p1>c;
  alist.assign(oList);
  oList.free;
end;

(*procedure SaveString(aFile, aText:string);
begin
  with TFileStream.Create(aFile, fmCreate) do try
    writeBuffer(aText[1],length(aText));
    finally free; end;
end;*)


{ TXMLNode }

function TXMLNode.AddAttribute(aName: string;
  aValue: variant): TXMLAttribute;
var n:TXMLAttribute;
begin
  n:=TXMLAttribute.create(aName,aValue);
  Attributes.Add(n);
  result:=n;
end;

function TXMLNode.AddNode(aName: string; aValue: variant): TXMLNode;
var n:TXMLNode;
begin
  n:=TXMLNode.create(aName,aValue,self);
  self.Nodes.Add(n);
  result:=n
end;

// adds node and parses any attributes;
function TXMLNode.AddNodeEx(aName: string; aValue: variant): TXMLNode;
var n:TXMLNode;
    s,sn,sv:string;
    c,p1,p2:integer;
begin
  n:=TXMLNode.create(aName,aValue,self);
  self.Nodes.Add(n);
  result:=n;
  c:=length(aName);
  //first parse name
  p1:=Q_posstr(' ',aName,1);
  if p1=0 then exit;
  s:=copy(aName,1,p1-1);
  n.Name:=s;
  repeat
  // find '='
    p2:=Q_posstr('=',aName,p1);
    if p2=0 then exit;
    sn:=trim(copy(aName,p1,p2-p1));
    p1:=p2;
  // find begin of value
    p1:=Q_posstr('"',aName,p1);
    if p1=0 then exit;
    p2:=Q_posstr('"',aName,p1+1);
    if p2=0 then exit;
    sv:=copy(aName,p1+1,p2-p1-1);
    n.AddAttribute(sn,sv);
    p1:=p2+1;
  until p1>c;
end;

function TXMLNode.getNamedAttribute(aName: string): TXMLAttribute;
var i:integer;
    n:TXMLAttribute;
begin
  result:=nil;
  if Attributes.Count=0 then exit;
  for i:=0 to Attributes.count-1 do
  begin
    n:=TXMLAttribute(Attributes[i]);
    if AnsiUpperCase(n.name) = AnsiUpperCase(aName) then begin //Konst-24.10.00, added AnsiUpperCase 
      result:=n;
      exit;
    end;
  end;
end;

procedure TXMLNode.ClearAttributes;
var i:integer;
begin
  if Attributes.count<>0 then begin
    for i:=0 to Attributes.count-1 do
      TXMLAttribute(Attributes[i]).free;
    Attributes.clear;
  end;
end;

procedure TXMLNode.ClearNodes;
var i:integer;
begin
  i:=nodes.count;
  if i<>0 then begin
    for i:=0 to nodes.count-1 do
      TXMLNode(Nodes[i]).free;
    nodes.clear;
  end;
end;

constructor TXMLNode.create(aName:string;aValue:variant;aParent:TXMLNode);
begin
  FNodes:=TList.Create;
  FName:=aName;
  FValue:=aValue;
  FValueType:=xvtString;
  FParentNode:=aParent;
  FAttributes:=TList.Create;
end;

procedure TXMLNode.DeleteAttribute(index: integer);
begin
  TXMLAttribute(Attributes[index]).free;
end;

procedure TXMLNode.DeleteNode(index: integer);
begin
  TXMLNode(Nodes[index]).free;
end;

destructor TXMLNode.destroy;
begin
  ClearNodes;
  FNodes.free;
  ClearAttributes;
  FAttributes.Free;
  inherited;
end;


function TXMLNode.document(aLevel:integer):string;
var
    i:integer;
    spc:string;
begin
  if aLevel>0 then
    spc:=StringOfChar(' ',aLevel*2)
  else
    spc:='';
  result:=spc+'<'+Name;
  if Attributes.Count>0 then
  for i:=0 to Attributes.count-1 do
    result:=result+TXMLAttribute(Attributes[i]).document;
  if (nodes.count=0) and (value='') then
  begin
    result:=result+' />'+cr;
    exit;
  end
  else
    result:=result+'>'+cr;
  if Value<>'' then
  begin
    if ValueType=xvtString then
      result:=result+spc+'  '+Value+cr
    else if ValueType=xvtCDATA then begin
      result:=result+spc+'  '+'<![CDATA['+ExpandCDATA(value)+']]>'+cr;
    end
  end;
  if nodes.count<>0 then
    for i:=0 to nodes.count-1 do
      result:=result+TXMLNode(nodes[i]).document(aLevel+1);
  result:=result+spc+'</'+Name+'>'+cr;
end;

// duplicates a node recursively
function TXMLNode.cloneNode: TXMLNode;
var i:integer;
    n:TXMLNode;
begin
  result:=TXMLNode.create(name,value,nil);
  result.name:=name;
  result.value:=value;
  if Attributes.count>0 then begin
    for i:=0 to Attributes.count-1 do begin
      result.AddAttribute(TXMLAttribute(Attributes[i]).name,TXMLAttribute(Attributes[i]).value);
    end;
  end;
  if nodes.count>0 then begin
    for i:=0 to nodes.count-1 do begin
      n:=TXMLNode(nodes[i]).cloneNode;
      result.Nodes.Add(n);
    end;
  end;
end;

function TXMLNode.getNamedNode(aName:string): TXMLNode;
var i:integer;
    n:TXMLNode;
begin
  result:=nil;
  if Nodes.Count=0 then exit;
  for i:=0 to Nodes.count-1 do
  begin
    n:=TXMLNode(nodes[i]);
    if AnsiUpperCase(n.name)=AnsiUpperCase(aName) then begin //Konst-24.10.00, added AnsiUpperCase
      result:=n;
      exit;
    end;
  end;
end;

procedure TXMLNode.SetAttributes(const Value: TList);
begin
  FAttributes := Value;
end;

procedure TXMLNode.SetName(const Value: string);
begin
  FName := Value;
end;

procedure TXMLNode.SetNodes(const Value: TList);
begin
  FNodes := Value;
end;

procedure TXMLNode.SetParentNode(const Value: TXMLNode);
begin
  FParentNode := Value;
end;

procedure TXMLNode.SetValue(const Value: variant);
var
  s: string;
begin
  if ValueType = xvtCDATA then begin
    s := StringReplace(Value,cr,'\n ',[rfreplaceall]);
    s := StringReplace(s,tab,'\t ',[rfreplaceall]);
    FValue := s;
  end
  else
    FValue := Value;
end;

function TXMLNode.firstChild: TXMLNode;
begin
 if Nodes.Count>0 then
   result:=TXMLNode(nodes[0])
 else
   result:=nil;  
end;

function TXMLNode.lastChild: TXMLNode;
begin
  if nodes.count>0 then
    result:=TXMLNode(nodes[nodes.count-1])
  else
    result:=nil;
end;

function TXMLNode.nextSibling: TXMLNode;
var index:integer;
begin
  result:=nil;
  if ParentNode=nil then exit;
  index:= ParentNode.Nodes.IndexOf(self);
  if index=-1 then exit;
  if index<ParentNode.nodes.Count-1 then
    result:=TXMLNode(ParentNode.nodes[index+1]);
end;

function TXMLNode.previousSibling: TXMLNode;
var index:integer;
begin
  result:=nil;
  if ParentNode=nil then exit;
  index:= ParentNode.Nodes.IndexOf(self);
  if index=-1 then exit;
  if index>0 then
    result:=TXMLNode(ParentNode.nodes[index-1]);
end;
// moves a node to a new location
function TXMLNode.moveInsertNode(Dest: TXMLNode): TXMLNode;
var index1,index2:integer;
begin
  result:=nil;
  if Dest.parentNode=nil then exit;  // can not move to root
  index1:=self.parentNode.Nodes.IndexOf(self);
  if index1=-1 then exit;
  index2:=dest.parentNode.Nodes.IndexOf(dest);
  if index2=-1 then exit;
  dest.parentNode.Nodes.Insert(index2,self);
  self.parentNode.nodes.Delete(index1);
  self.parentNode:=dest.parentnode;
  result:=self;
end;

function TXMLNode.moveAddNode(Dest: TXMLNode): TXMLNode;
var index:integer;
begin
  result:=nil;
  if Dest=nil then exit;  // can not move to root
  index:=self.parentNode.Nodes.IndexOf(self);
  if index=-1 then exit;
  dest.Nodes.Add(self);
  self.parentNode.nodes.Delete(index);
  self.parentNode:=dest;
  result:=self;
end;


// removes and frees the childnode recursively.
// returns self when done, or nil in case of error
function TXMLNode.removeChildNode(aNode: TXMLNode): TXMLNode;
var index:integer;
begin
  result:=nil;
  index:=nodes.IndexOf(aNode);
  if index=-1 then exit;
  nodes.Delete(index);
  aNode.free;
  result:=self;
end;

function TXMLNode.hasChildNodes: boolean;
begin
  result:=nodes.count>0;
end;

procedure TXMLNode.getAttributeNames(aList: TStringList);
var i,c:integer;
begin
  aList.Clear;
  c:=Attributes.count;
  if c=0 then exit;
  for i:=0 to c-1 do
    aList.append(TXMLAttribute(Attributes[i]).name);
end;

procedure TXMLNode.getNodeNames(aList: TStringList);
var i,c:integer;
begin
  aList.Clear;
  c:=Nodes.count;
  if c=0 then exit;
  for i:=0 to c-1 do
    aList.append(TXMLNode(Nodes[i]).name);
end;

function TXMLNode.getNodePath: string;
var n:TXMLNode;
begin
  n:=self;
  result:=name;
  while n.parentNode<>nil do
  begin
    n:=n.parentNode;
    result:=n.name+'/'+result;
  end;
end;


// search recursively for a named node
function TXMLNode.findNamedNode(aName: string): TXMLNode;
var i:integer;
    n:TXMLNode;
begin
  result:=nil;
  if Nodes.Count=0 then exit;
  for i:=0 to Nodes.count-1 do
  begin
    n:=TXMLNode(nodes[i]);
    if n.name=aName then begin
      result:=n;
      exit;
    end
    else begin  // recurse
      result:=n.findNamedNode(aName);
      if result<>nil then exit;
    end;
  end;
end;

// add all found named nodes to aList
procedure TXMLNode.findNamedNodes(aName: string; aList: TList);
var i:integer;
    n:TXMLNode;
begin
  if Nodes.Count=0 then exit;
  for i:=0 to Nodes.count-1 do
  begin
    n:=TXMLNode(nodes[i]);
    if AnsiUpperCase(n.name)= AnsiUpperCase(aName) then
      alist.Add(n);
    // recurse
    n.findNamedNodes(aName,aList);
  end;
end;

// add recursively all nodes to aList
// the list only contains pointers to the nodes
// typecast to use, e.g. n:=TXMLNode(aList[0]);
procedure TXMLNode.getAllNodes(aList: TList);
var i:integer;
    n:TXMLNode;
begin
  if Nodes.Count=0 then exit;
  for i:=0 to Nodes.count-1 do
  begin
    n:=TXMLNode(nodes[i]);
    alist.Add(n);
    // recurse
    n.getAllNodes(aList);
  end;
end;

// add recursively all nodes with matching named attribute to aList
// the list only contains pointers to the nodes
// typecast to use, e.g. n:=TXMLNode(aList[0]);
procedure TXMLNode.findNamedAttributes(aName: string; aList: TList);
var i,c:integer;
    n:TXMLNode;
begin
  c:= Attributes.count;
  if c>0 then
    for i:=0 to c-1 do begin
      if TXMLAttribute(Attributes[i]).name=aName then
      begin
        aList.Add(self);
        break;
      end;
    end;
  if Nodes.Count=0 then exit;
  for i:=0 to Nodes.count-1 do
  begin
    n:=TXMLNode(nodes[i]);
    n.findNamedAttributes(aName,aList);
  end;
end;

{
this procedure adds the node to aList when it matches the pattern
this will be the key procedure for XSL implementation
only basic matching is provided in the first release
path operators
 /  child path
 // recursive descent
 .  curren context or node
 @  attribute
 *  wildcar
some examples
 /  the root node only
 book/author  <author> elements that are children of <book> elements
 // the root node and all nodes below
 //*  all element nodes below the root node
 book//author  <author> elements that are descendants of <book> elements
 .//author  <author elements that are descendants of the current element
 *  non-root elements, irrespective of the element name
 book/*  elements that are children of <book> elements
 book//* elements that are descendants of <book> elements
 book/*/author  <author> elements that are grandchildren of <book> elements
 book/@print_date print_date attributes that are attached to <book> elements
 */@print_date print_date atrtributes that are attached to any elements

index can be used to specify a particular node within a matching set
 /booklist/book[0]  First <book> node in root <booklist> element
 /booklist/book[2]  Third <book> node in root <booklist> element
 /booklist/book[end()] Last <book> node in root <booklist> element
}
procedure TXMLNode.matchPattern(aPattern: string; aList:TList);
begin
// to be implemented
end;

procedure TXMLNode.SetValueType(const Value: TXMLValueType);
begin
  FValueType := Value;
end;

{select a node based on path info
 e.g. booklist/book/category will find the first
 <category> that is a child of <book> that is a child of <booklist>
 }
function TXMLNode.SelectSingleNode(pattern: string): TXMLNode;
var npattern,aFilter:string;
    p,i,c:integer;
    n:TXMLNode;
    objFilter:TXMLFilter;
begin
  result:=nil;
  c:=nodes.count;
  if c=0 then exit;
  p:=pos('/',pattern);
  if p=0 then begin
    objFilter:=TXMLFilter.create(pattern);
    for i:=0 to c-1 do begin
      n:=TXMLNode(nodes[i]);
      if n.matchFilter(objFilter) then begin
        result:=n;
        objFilter.free;
        exit;
      end;
    end;
    objFilter.free;
    exit; // not found;
  end
  else begin
    aFilter:=copy(pattern,1,p-1);
    nPattern:=copy(pattern,p+1,length(pattern));
    objFilter:=TXMLFilter.create(aFilter);
    for i:=0 to c-1 do begin
      n:=TXMLNode(nodes[i]);
      if n.matchFilter(objFilter) then begin
        result:=n.SelectSingleNode(npattern);
        if result<>nil then begin objFilter.free; exit end;
      end;
    end;
    objFilter.free;
  end;
end;

// filter contains name + any filters between []
function TXMLNode.matchFilter(objFilter: TXMLFilter): boolean;
var
  i,j:integer;
  attName:string;
  a:TXMLAttribute;
  n:TXMLNode;
  atom:TXMLFilterAtom;
  attResult:boolean;

  function evalAtom(aValue:string):boolean;
  begin
    result:=false;
    case atom.Operator of
      xfoNOP: result:=true;
      xfoEQ : result:=aValue=atom.Value;
      xfoIEQ: result:=comparetext(aValue,atom.value)=0;
      xfoNE : result:=avalue<>atom.value;
      xfoINE: result:=comparetext(aValue,atom.value)<>0;
      xfoGT :
        try
          result:= Strtofloat(avalue)>strtofloat(atom.value);
          except
          end;
      xfoIGT: result:=comparetext(aValue,atom.value)>0;
      xfoLT :
        try
          result:= Strtofloat(avalue)<strtofloat(atom.value);
          except
          end;
      xfoILT: result:=comparetext(aValue,atom.value)<0;
      xfoGE :
        try
          result:= Strtofloat(avalue)>=strtofloat(atom.value);
          except
          end;
      xfoIGE: result:=comparetext(aValue,atom.value)>=0;
      xfoLE :
        try
          result:= Strtofloat(avalue)<=strtofloat(atom.value);
          except
          end;
      xfoILE: result:=comparetext(aValue,atom.value)<=0;
    end;

  end;
begin
  result:=false;
  attResult := false;
  if objFilter.Filters.Count=0 then
  begin  // just filter on name
    result:=AnsiUpperCase(objFilter.Name)=AnsiUpperCase(name); //Konst-13.10.00 added AnsiUpperCase
    exit;
  end;
  for i:=0 to objFilter.Filters.count-1 do
  begin
    atom:=TXMLFilterAtom(objFilter.Filters[i]);
    if atom.AttributeFilter then begin
      attName:=atom.Name;
      if attName='*' then begin  // match any attribute
        if Attributes.Count=0 then exit;
        for j:=0 to Attributes.count-1 do
        begin
          a:=TXMLAttribute(Attributes[j]);
          attResult:=evalAtom(a.value);
          if AttResult then break;
        end;
        if not AttResult then exit;
      end
      else begin
        a:=GetNamedAttribute(attName);
        if a=nil then exit;
        if not evalAtom(a.value) then exit;
      end;
    end
    else begin
      attName:=atom.Name;
      n:=GetNamedNode(attName);
      if n=nil then exit;
      if not evalAtom(n.value) then exit;
    end;
  end;
  result:=true;
end;

procedure TXMLNode.SelectNodes(pattern: string; aList: TList);
var npattern:string;
    p,i,c:integer;
    n:TXMLNode;
    aFilter:string;
    objFilter:TXMLFilter;
    recurse:boolean;
begin
  c:=nodes.count;
  if c=0 then exit;
  if copy(pattern,1,2)='//' then begin  //recursive
    delete(pattern,1,2);
    recurse:=true;
  end
  else recurse:=false;
  p:=pos('/',pattern);
  if p=0 then begin
    aFilter:=pattern;
    objFilter:=TXMLFilter.Create(aFilter);
    for i:=0 to c-1 do begin
      n:=TXMLNode(nodes[i]);
      if n.matchFilter(objFilter) then
        aList.Add(n)
      else begin
        if recurse then
          n.SelectNodes('//'+pattern,aList);
      end;
    end;
    objFilter.free;
  end
  else begin
    aFilter:=copy(pattern,1,p-1);
    if copy(pattern,p,2)='//' then
      npattern:=copy(pattern,p,length(pattern))
    else
      npattern:=copy(pattern,p+1,length(pattern));
    objFilter:=TXMLFilter.Create(aFilter);
    for i:=0 to c-1 do begin
      n:=TXMLNode(nodes[i]);
      if n.matchFilter(objFilter) then
        n.SelectNodes(npattern,aList)
      else begin
        if recurse then
          n.selectNodes('//'+pattern,aList);
      end;
    end;
    objFilter.free;
  end;
end;

// the XSL implementation
// although this function returns a string, the string itself can be parsed to create a DOM
function TXMLNode.transformNode(stylesheet: TXMLNode): string;
begin
  // to be implemented;
  result:=stylesheet.process(0,self);
end;

// used in conjunction with the transformNode function.
// basically works like the document function except for nodes with processing instructions
function TXMLNode.process(aLevel:integer;node: TXMLNode): string;
var
    i:integer;
    spc:string;
begin
  if parentNode=nil then begin
    if nodes.count<>0 then
    for i:=0 to nodes.count-1 do
      result:=result+TXMLNode(nodes[i]).process(aLevel+1,node);
    exit;
  end;
  if aLevel>0 then
    spc:=StringOfChar(' ',aLevel*2)
  else
    spc:='';
  result:=spc+'<'+Name;
  if Attributes.Count>0 then
  for i:=0 to Attributes.count-1 do
    result:=result+TXMLAttribute(Attributes[i]).document;
  if (nodes.count=0) and (value='') then
  begin
    result:=result+' />'+cr;
    exit;
  end
  else
    result:=result+'>'+cr;
  if Value<>'' then
  begin
    if ValueType=xvtString then
      result:=result+spc+'  '+Value+cr
    else if ValueType=xvtCDATA then begin
      result:=result+spc+'  '+'<![CDATA['+ExpandCDATA(value)+']]>'+cr;
    end
  end;
  if nodes.count<>0 then
    for i:=0 to nodes.count-1 do
      result:=result+TXMLNode(nodes[i]).process(aLevel+1,node);
  result:=result+spc+'</'+Name+'>'+cr;
end;


function TXMLNode.getNameSpace: string;
var p:integer;
begin
  p:=pos(':',FName);
  if p>0 then
    result:=copy(FName,1,p-1)
  else
    result:='';
end;

function TXMLNode.GetValue: variant;
begin
  if FValueType = xvtCDATA then
    result := ExpandCDATA(FValue)
  else
    result := FValue;
end;

{ TXMLTree }



constructor TXMLTree.Create(aName:string;aValue:variant;aParent:TXMLNode);
begin
  inherited Create(aName,aValue,aParent);
  FLines:=TStringList.create;
end;

destructor TXMLTree.Destroy;
begin
  FLines.free;
  inherited destroy;
end;

function TXMLTree.asText: string;
var
  i,c:integer;
begin
  c:=Nodes.Count;
  if c=0 then exit;
  result:='<'+Name;
  if Attributes.Count>0 then
  for i:=0 to Attributes.count-1 do
    result:=result+TXMLAttribute(Attributes[i]).document;
  result:=result+'>'+cr;
  for i:=0 to c-1 do
   result:=result+TXMLNode(nodes[i]).document(1);
  result:=result+'</'+Name+'>'+cr;
end;

procedure TXMLTree.SaveToFile(aFile: string);
begin
  Lines.text:=Text;
  Lines.SaveToFile(aFile)
end;


procedure TXMLTree.SetLines(const Value: TStringlist);
begin
  FLines.assign(Value);
end;






procedure TXMLTree.LoadFromStream(Stream:TStream);
begin
  ClearNodes;
  ClearAttributes;
  Lines.LoadFromStream(Stream);
  PreProcessXML(FLines);
  ParseXML;
end;

procedure TXMLTree.SaveToStream(Stream: TStream);
begin
  Lines.text:=asText;
  Lines.SaveToStream(Stream);
end;

function TXMLTree.getText: string;
var
  i,c:integer;
begin
  c:=Nodes.Count;
  if c=0 then exit;
//  result:='<'+Name;
//  if Attributes.Count>0 then
//  for i:=0 to Attributes.count-1 do
//    result:=result+TXMLAttribute(Attributes[i]).document;
//  result:=result+'>'+cr;
  result:='';
  for i:=0 to c-1 do
   result:=result+TXMLNode(nodes[i]).document(0);
//  result:=result+'</'+Name+'>'+cr;
end;

procedure TXMLTree.setText(const Value: string);
begin
  ClearNodes;
  ClearAttributes;
  Lines.text:=Value;
  PreProcessXML(FLines);
  ParseXML;
end;

{ TXMLAttribute }

constructor TXMLAttribute.create(aName: string; aValue: variant);
begin
  FName:=aName;
  FValue:=aValue;
end;

function TXMLAttribute.document:string;
var s:string;
begin
  s:=Value;
  result:=' '+Name+'="'+s+'"';
end;

procedure TXMLAttribute.SetName(const Value: string);
begin
  FName := Value;
end;

procedure TXMLAttribute.SetValue(const Value: variant);
begin
  FValue := Value;
end;




{ TXMLTree }

procedure TXMLTree.ParseXML;
var i,c:integer;
    s,token,aName:string;
    n:TXMLNode;
begin
  i:=0;
  FNodeCount:=0;
  ClearNodes;
  ClearAttributes;
  Name:='root';
  n:=self;
  c:=Lines.Count-1;
  if c < 0 then Exit; //Konst-05.10.00 
  repeat
    s:=Lines[i];
    token:=copy(s,1,3);
    aName:=copy(s,4,length(s));
    if token='OT:' then begin
      n:=n.AddNodeEx(aName,'');
      inc(FNodeCount);
    end
    else if token='CT:' then begin
      n:=n.ParentNode;
    end
    else if token='ET:' then begin
      n.AddNodeEx(aName,'');
    end
    else if token='TX:' then begin
      n.Value:=aName;
      n.ValueType:=xvtString;
    end
    else if token='CD:' then begin
      n.value:=aName;
      n.ValueType:=xvtCDATA;
    end;
    inc(i);
  until i>c;
end;




procedure TXMLTree.LoadFromFile(fn: string);
begin
  ClearNodes;
  ClearAttributes;
  Lines.LoadFromFile(fn);
  PreProcessXML(FLines);
  ParseXML;
end;


{ TXMLFilter }

constructor TXMLFilter.Create(FilterStr: string);
var theFilter:string;
    p1,p2:integer;
    attName,attValue:string;
    attOperator:TXMLFilterOperator;
    atom:TXMLFilterAtom;

    function trimquotes(s:string):string;
    var cc:integer;
    begin
      result:=trim(s);
      if s='' then exit;
      if (s[1]='"') or (s[1]='''') then delete(result,1,1);
      if s='' then exit;
      cc:=length(result);
      if (result[cc]='"') or (result[cc]='''') then delete(result,cc,1);
    end;

    function splitNameValue(s:string):boolean;
    var pp:integer;
    begin
      //result:=false;
      pp:=q_posstr(' $ne$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoNE;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+6,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $ine$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoINE;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+7,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $ge$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoGE;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+6,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $ige$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoIGE;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+7,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $gt$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoGT;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+6,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $igt$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoIGT;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+7,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $le$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoLE;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+6,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $ile$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoILE;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+7,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $lt$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoLT;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+6,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $ilt$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoILT;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+7,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $eq$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoEQ;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+6,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' $ieq$ ',s,1);
      if pp>0 then begin
        attOperator:=xfoIEQ;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+7,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      pp:=q_posstr(' = ',s,1);
      if pp>0 then begin
        attOperator:=xfoEQ;
        attName:=trim(copy(s,1,pp-1));
        attvalue:=trimquotes(copy(s,pp+3,length(s)));
        result:=(attName<>'') and (attValue<>'');
        exit;
      end;
      attOperator:=xfoNOP;
      attName:=s;
      attValue:='';
      result:=true;
      exit;
    end;

begin
  Filters:=TList.Create;
  p1:=q_posstr('[',FilterStr,1);
  if p1=0 then begin  // just a name filter on name
    name:=FilterStr;
    exit;
  end
  else begin
    Name:=copy(FilterStr,1,p1-1);
    delete(FilterStr,1,p1-1);
  end;
  repeat
    FilterStr:=trim(FilterStr);
    p1:=q_posstr('[',FilterStr,1);
    if p1=0 then exit;
    p2:=q_posstr(']',FilterStr,p1+1);
    if p2=0 then exit;
    theFilter:=copy(FilterStr,p1+1,p2-p1-1);
    delete(FilterStr,1,p2);
    if theFilter='' then exit;
    // check for attribute filter
    if theFilter[1]='@' then begin
      if not splitNameValue(copy(theFilter,2,length(theFilter))) then exit;
      atom:=TXMLFilterAtom.create;
      atom.Name:=attName;
      atom.Operator:=attOperator;
      atom.Value:=attValue;
      atom.AttributeFilter:=true;
      Filters.Add(atom);
    end
    else begin // childfilter
        if not splitNameValue(theFilter) then exit;
      atom:=TXMLFilterAtom.create;
      atom.Name:=attName;
      atom.Operator:=attOperator;
      atom.Value:=attValue;
      atom.AttributeFilter:=false;
      Filters.Add(atom);
    end;
  until FilterStr='';
end;




destructor TXMLFilter.Destroy;
var i:integer;
begin
  if Filters.Count>0 then
    for i:=0 to Filters.count-1 do
      TXMLFilterAtom(Filters[i]).free;
  filters.free;    
  inherited Destroy;
end;

procedure TXMLFilter.SetFilters(const Value: TList);
begin
  FFilters := Value;
end;

procedure TXMLFilter.SetName(const Value: string);
begin
  FName := Value;
end;

{ TXMLFilterAtom }

procedure TXMLFilterAtom.SetAttributeFilter(const Value: boolean);
begin
  FAttributeFilter := Value;
end;

procedure TXMLFilterAtom.SetName(const Value: string);
begin
  FName := Value;
end;

procedure TXMLFilterAtom.SetOperator(
  const Value: TXMLFilterOperator);
begin
  FOperator := Value;
end;

procedure TXMLFilterAtom.SetValue(const Value: string);
begin
  FValue := Value;
end;

end.
