{*******************************************************} 
{                                                       }
{  MTReport Components Suite 1.0                        }
{                                                       }
{  Wrappers for database access components              }
{                                                       }
{  Copyright (c) 2001 MT Tools.                         }
{                                                       }
{*******************************************************}

unit mtrDBAccess;

{$I MTR.INC}

interface

uses Classes, SysUtils;

type
  TDBMSType = (dbmsBDE, dbmsInterbase, dbmsOracle);

  TmtrQueryField = class;
  TmtrQueryFields = class;
  TmtrQueryParams = class;

  TmtrQuery = class
  private
    FDBServerType: TDBMSType;
    FQuery: TComponent;
    FField: TmtrQueryField;
    FFields: TmtrQueryFields;
    FParams: TmtrQueryParams;
    function GetSQL: TStrings;
    procedure SetSQL(const Value: TStrings);
    function GetSQLParams: TmtrQueryParams;
    function GetBOF: Boolean;
    function GetEOF: Boolean;
    function GetFieldIndex(FieldName: String): Integer;
    function GetFields(const Idx: Integer): TmtrQueryField;
    function GetOpen: Boolean;
    function GetPlan: String;
    function GetPrepared: boolean;
    function GetSelectSQL: TStrings;
    procedure SetSelectSQL(const Value: TStrings);
    function GetRecordCount: Integer;
    function GetDatabase: TComponent;
    function GetTransaction: TComponent;
    procedure SetDatabase(const Value: TComponent);
    procedure SetTransaction(const Value: TComponent);
    function GetFieldCount: integer;
    function GetParamCount: integer;
  public
    constructor Create(ADBMSType: TDBMSType);
    destructor Destroy; override;
    procedure ExecQuery;
    procedure Close;
    procedure Next;
    function FieldByName(FieldName: string): TmtrQueryField;
    function ParamNames: string;
    procedure SetParamValue(ParamName: string; Value: variant);
    procedure Prepare;
    property Bof: Boolean read GetBOF;
    property Eof: Boolean read GetEOF;
    property Fields[const Idx: Integer]: TmtrQueryField read GetFields;
    property FieldIndex[FieldName: String]: Integer read GetFieldIndex;
    property FieldCount: integer read GetFieldCount;
    property Open: Boolean read GetOpen;
    property Plan: String read GetPlan;
    property Params: TmtrQueryParams read GetSQLParams;
    property ParamCount: integer read GetParamCount;
    property Query: TComponent read FQuery;
    property Prepared: boolean read GetPrepared;
    property RecordCount: Integer read GetRecordCount;
    property DBMSType: TDBMSType read FDBServerType;
    property SelectSQL: TStrings read GetSelectSQL write SetSelectSQL;
    property SQL: TStrings read GetSQL write SetSQL;
    property Transaction: TComponent read GetTransaction write SetTransaction;
    property Database: TComponent read GetDatabase write SetDatabase;
  end;

  TmtrQueryField = class
  private
    FField: TObject;
    function GetAsCurrency: Currency;
    function GetAsDateTime: TDateTime;
    function GetAsDouble: Double;
    function GetAsInt64: Int64;
    function GetAsLong: LongInt;
    function GetAsPointer: Pointer;
    function GetAsShort: SmallInt;
    function GetAsString: String;
    function GetAsVariant: Variant;
    function GetIsNull: Boolean;
    function GetIsNullable: Boolean;
    function GetModified: Boolean;
    function GetName: String;
    function GetSize: Integer;
    function GetSQLType: Integer;
    procedure SetAsCurrency(const Value: Currency);
    procedure SetAsDate(const Value: TDateTime);
    procedure SetAsDateTime(const Value: TDateTime);
    procedure SetAsDouble(const Value: Double);
    procedure SetAsInt64(const Value: Int64);
    procedure SetAsLong(const Value: LongInt);
    procedure SetAsPointer(const Value: Pointer);
    procedure SetAsShort(const Value: SmallInt);
    procedure SetAsString(const Value: String);
    procedure SetAsTime(const Value: TDateTime);
    procedure SetAsVariant(const Value: Variant);
    procedure SetIsNull(const Value: Boolean);
    procedure SetIsNullable(const Value: Boolean);
    procedure SetSQLType(const Value: Integer);
  public
    property Field: TObject read FField write FField;
    property AsDate: TDateTime read GetAsDateTime write SetAsDate;
    property AsTime: TDateTime read GetAsDateTime write SetAsTime;
    property AsDateTime: TDateTime read GetAsDateTime write SetAsDateTime;
    property AsDouble: Double read GetAsDouble write SetAsDouble;
    property AsCurrency: Currency read GetAsCurrency write SetAsCurrency;
    property AsInt64: Int64 read GetAsInt64 write SetAsInt64;
    property AsInteger: Integer read GetAsLong write SetAsLong;
    property AsLong: LongInt read GetAsLong write SetAsLong;
    property AsPointer: Pointer read GetAsPointer write SetAsPointer;
    property AsShort: SmallInt read GetAsShort write SetAsShort;
    property AsString: String read GetAsString write SetAsString;
    property AsVariant: Variant read GetAsVariant write SetAsVariant;
    property IsNull: Boolean read GetIsNull write SetIsNull;
    property IsNullable: Boolean read GetIsNullable write SetIsNullable;
    property Modified: Boolean read GetModified;
    property Name: String read GetName;
    property Size: Integer read GetSize;
    property SQLType: Integer read GetSQLType write SetSQLType;
    property Value: Variant read GetAsVariant write SetAsVariant;
  end;

  TmtrQueryFields = class
  private
    FDBServerType: TDBMSType;
    FQuery: TObject;
    FField: TmtrQueryField;
    function GetFields(Idx: integer): TmtrQueryField;
  public
    constructor Create(AObject: TObject; ADBMSType: TDBMSType);
    destructor Destroy; override;
    function ByName(Idx: String): TmtrQueryField;
    property Fields[Idx: integer]: TmtrQueryField read GetFields; default;
  end;

  TmtrQueryParams = class
  private
    FDBServerType: TDBMSType;
    FQuery: TObject;
    FParam: TmtrQueryField;
    function GetParams(Idx: integer): TmtrQueryField;
  public
    constructor Create(AObject: TObject; ADBMSType: TDBMSType);
    destructor Destroy; override;
    function ByName(Idx: String): TmtrQueryField;
    property Params[Idx: integer]: TmtrQueryField read GetParams; default;
  end;

const
  CRLF = #13#10;

implementation

uses
  {$IFDEF IBX}IBHeader, IBDatabase, IBSQL,{$ENDIF}
  {$IFDEF BDE}Db, DbTables,{$ENDIF}
  mtrGlobal, mtrString;

{ TmtrQuery }

procedure TmtrQuery.Close;
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: TQuery(FQuery).Close;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: TIBSQL(FQuery).Close;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

constructor TmtrQuery.Create(ADBMSType: TDBMSType);
begin
  FDBServerType := ADBMSType;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: FQuery := TQuery.Create(nil);
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: FQuery := TIBSQL.Create(nil);
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
  FFields := TmtrQueryFields.Create(Self, FDBServerType);
  FField := TmtrQueryField.Create;
  FParams := TmtrQueryParams.Create(Self, FDBServerType);
end;

destructor TmtrQuery.Destroy;
begin
  FQuery.Free; FQuery := nil;
  FFields.Free; FFields := nil;
  FField.Free; FField := nil;
  FParams.Free; FParams := nil;
  inherited;
end;

procedure TmtrQuery.ExecQuery;
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: TQuery(FQuery).Open;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: TIBSQL(FQuery).ExecQuery;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.FieldByName(FieldName: String): TmtrQueryField;
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: begin
      result := FField;
      FField.Field := TQuery(FQuery).FieldByName(FieldName);
    end;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: begin
      result := FField;
      FField.Field := TIBSQL(FQuery).FieldByName(FieldName);
    end;
{$ENDIF}
    dbmsOracle: begin result := nil; Assert(false, SNotImplemented); end;
    else result := nil;
  end;
end;

function TmtrQuery.GetBOF: Boolean;
begin
  result := false;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).Bof;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Bof;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetDatabase: TComponent;
begin
  result := nil;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).Database;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Database;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetEOF: Boolean;
begin
  result := false;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).Eof;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Eof;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetFieldCount: integer;
begin
  result := 0;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).Fields.Count;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Current.Count;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetFieldIndex(FieldName: String): Integer;
begin
  result := 0;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).FieldByName(FieldName).Index;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).FieldIndex[FieldName];
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetFields(const Idx: Integer): TmtrQueryField;
begin
  result := FField;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: FField.Field := TQuery(FQuery).Fields[Idx];
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: FField.Field := TIBSQL(FQuery).Fields[Idx];
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetOpen: Boolean;
begin
  result := false;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).Active;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Open;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetParamCount: integer;
begin
  result := 0;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).ParamCount;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Params.Count;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetPlan: String;
begin
  result := '';
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: Assert(false, SNotImplemented);
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Plan;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetPrepared: boolean;
begin
  result := false;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).Prepared;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Prepared;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.GetRecordCount: Integer;
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).RecordCount;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).RecordCount;
{$ENDIF}
    dbmsOracle: begin result := 0; Assert(false, SNotImplemented); end;
    else result := 0;
  end;
end;

function TmtrQuery.GetSelectSQL: TStrings;
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).SQL;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).SQL;
{$ENDIF}
    dbmsOracle: begin result := nil; Assert(false, SNotImplemented); end;
    else result := nil;
  end;
end;

function TmtrQuery.GetSQL: TStrings;
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: result := TQuery(FQuery).SQL;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).SQL;
{$ENDIF}
    dbmsOracle: begin result := nil; Assert(false, SNotImplemented); end;
    else result := nil;
  end;
end;

function TmtrQuery.GetSQLParams: TmtrQueryParams;
begin
  result := FParams;
end;

function TmtrQuery.GetTransaction: TComponent;
begin
  result := nil;
  case FDBServerType of
    dbmsBDE: Assert(false, SNotImplemented);
{$IFDEF IBX}
    dbmsInterbase: result := TIBSQL(FQuery).Transaction;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

procedure TmtrQuery.Next;
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: TQuery(FQuery).Next;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: TIBSQL(FQuery).Next;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

function TmtrQuery.ParamNames: string;
var
  i: integer;
begin
  result := '';
  for i:=0 to ParamCount-1 do
    result := result + Params[i].Name + CRLF;
end;

procedure TmtrQuery.Prepare;
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: TQuery(FQuery).Prepare;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: TIBSQL(FQuery).Prepare;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

procedure TmtrQuery.SetDatabase(const Value: TComponent);
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: TQuery(FQuery).DatabaseName := TDatabase(Value).DatabaseName;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: TIBSQL(FQuery).Database := TIBDatabase(Value);
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

procedure TmtrQuery.SetParamValue(ParamName: string; Value: variant);
var
  par: TmtrQueryField;
begin
  par := Params.ByName(ParamName);
  case FDBServerType of
    dbmsBDE, dbmsInterbase: begin
      try
        case par.SQLType of
          SQL_VARYING, SQL_TEXT:
            par.AsVariant := Value;
          SQL_DOUBLE, SQL_FLOAT, SQL_D_FLOAT, SQL_INT64:
            if ((VarType(Value) = varOleStr) or (VarType(Value) = varString)) and
               (Value = '') then
              par.AsDouble := 0
            else
              par.AsDouble := Value;
          SQL_LONG, SQL_SHORT:
            if ((VarType(Value) = varOleStr) or (VarType(Value) = varString)) and
               (Value = '') then
              par.AsInteger := 0
            else
              par.AsInteger := Value;
          SQL_TIMESTAMP, SQL_TYPE_TIME, SQL_TYPE_DATE:
            par.AsVariant := Value;
        end;
      except
        raise Exception.Create(SCantSetParamValue + ParamName);
      end;
    end;
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

procedure TmtrQuery.SetSelectSQL(const Value: TStrings);
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: TQuery(FQuery).SQL.Assign(Value);
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: TIBSQL(FQuery).SQL.Assign(Value);
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

procedure TmtrQuery.SetSQL(const Value: TStrings);
begin
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: TQuery(FQuery).SQL.Assign(Value);
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: TIBSQL(FQuery).SQL.Assign(Value);
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

procedure TmtrQuery.SetTransaction(const Value: TComponent);
begin
  case FDBServerType of
{$IFDEF IBX}
    dbmsInterbase: TIBSQL(FQuery).Transaction := TIBTransaction(Value);
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

{ TmtrQueryField }

function TmtrQueryField.GetAsCurrency: Currency;
begin
  result := 0;
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).AsCurrency;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsCurrency;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetAsDateTime: TDateTime;
begin
  result := 0;
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).AsDateTime;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsDateTime;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetAsDouble: Double;
begin
  result := 0;
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).AsFloat;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsDouble;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetAsInt64: Int64;
begin
  result := 0;
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).AsInteger;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsInt64;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetAsLong: LongInt;
begin
  result := 0;
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).AsInteger;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsLong;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetAsPointer: Pointer;
begin
  result := nil;
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsPointer;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetAsShort: SmallInt;
begin
  result := 0;
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).AsInteger;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsShort;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetAsString: String;
begin
  result := '';
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).AsString;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsString;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetAsVariant: Variant;
begin
  result := Null;
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).AsVariant;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).AsVariant;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetIsNull: Boolean;
begin
  result := false;
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).IsNull;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetIsNullable: Boolean;
begin
  result := false;
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).IsNullable;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetModified: Boolean;
begin
  result := false;
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).Modified;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetName: String;
begin
  result := '';
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).FieldName;
    Exit;
  end
  else if FField is TParam then begin
    result := TParam(FField).Name;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).Name;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetSize: Integer;
begin
  result := 0;
{$IFDEF BDE}
  if FField is TField then begin
    result := TField(FField).Size;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).Size;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

function TmtrQueryField.GetSQLType: Integer;
begin
  result := 0;
{$IFDEF BDE}
  if FField is TField then begin
    case TField(FField).DataType of
      ftString: result := SQL_TEXT;
      ftSmallint: result := SQL_SHORT;
      ftInteger: result := SQL_LONG;
      ftBoolean: result := SQL_SHORT;
      ftFloat: result := SQL_DOUBLE;
      ftCurrency: result := SQL_INT64;
      ftDate: result := SQL_TYPE_DATE;
      ftTime: result := SQL_TYPE_TIME;
      ftDateTime: result := SQL_TIMESTAMP;
      ftAutoInc: result := SQL_LONG;
      ftBlob: result := SQL_BLOB;
      ftMemo: result := SQL_TEXT;
      ftGraphic: result := SQL_BLOB;
      ftFmtMemo: result := SQL_TEXT;
      ftWideString: result := SQL_TEXT;
      ftLargeInt: result := SQL_LONG;
      else
        Assert(false, SNotImplemented);
    end; //case
    Exit;
  end
  else if FField is TParam then begin
    case TParam(FField).DataType of
      ftString: result := SQL_TEXT;
      ftSmallint: result := SQL_SHORT;
      ftInteger: result := SQL_LONG;
      ftBoolean: result := SQL_SHORT;
      ftFloat: result := SQL_DOUBLE;
      ftCurrency: result := SQL_INT64;
      ftDate: result := SQL_TYPE_DATE;
      ftTime: result := SQL_TYPE_TIME;
      ftDateTime: result := SQL_TIMESTAMP;
      ftAutoInc: result := SQL_LONG;
      ftBlob: result := SQL_BLOB;
      ftMemo: result := SQL_TEXT;
      ftGraphic: result := SQL_BLOB;
      ftFmtMemo: result := SQL_TEXT;
      ftWideString: result := SQL_TEXT;
      ftLargeInt: result := SQL_LONG;
      else
        Assert(false, SNotImplemented);
    end; //case
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    result := TIBXSQLVAR(FField).SQLType;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsCurrency(const Value: Currency);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsCurrency := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsCurrency := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsCurrency := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsDate(const Value: TDateTime);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsDateTime := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsDate := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsDate := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsDateTime(const Value: TDateTime);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsDateTime := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsDateTime := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsDateTime := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsDouble(const Value: Double);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsFloat := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsFloat := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsDouble := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsInt64(const Value: Int64);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsInteger := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsInteger := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsInt64 := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsLong(const Value: LongInt);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsInteger := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsInteger := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsLong := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsPointer(const Value: Pointer);
begin
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsPointer := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsShort(const Value: SmallInt);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsInteger := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsSmallint := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsShort := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsString(const Value: String);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsString := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsString := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsString := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsTime(const Value: TDateTime);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsDateTime := Value;
    Exit;
  end;
  if FField is TParam then begin
    TParam(FField).AsTime := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsTime := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetAsVariant(const Value: Variant);
begin
{$IFDEF BDE}
  if FField is TField then begin
    TField(FField).AsVariant := Value;
    Exit;
  end;
  if FField is TParam then begin
    case TParam(FField).DataType of
      ftString: TParam(FField).AsString := Value;
      ftSmallint, ftInteger: TParam(FField).AsInteger := Value;
      ftBoolean: TParam(FField).AsBoolean := Value;
      ftFloat: TParam(FField).AsFloat := Value;
      ftCurrency: TParam(FField).AsCurrency := Value;
      ftDate: TParam(FField).AsDate := Value;
      ftTime: TParam(FField).AsTime := Value;
      ftDateTime: TParam(FField).AsDateTime := Value;
      ftBlob, ftMemo, ftGraphic, ftFmtMemo: TParam(FField).AsBlob := Value;
      ftWideString: TParam(FField).AsString := Value;
      ftLargeInt: TParam(FField).AsInteger := Value;
    end; //case
    Exit;
  end;
{$ENDIF}
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).AsVariant := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetIsNull(const Value: Boolean);
begin
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).IsNull := Value;
    Exit;
  end;
{$ENDIF}
{$IFDEF BDE}
  if FField is TParam then begin
    TParam(FField).Clear;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetIsNullable(const Value: Boolean);
begin
{$IFDEF IBX}
  if FField is TIBXSQLVAR then begin
    TIBXSQLVAR(FField).IsNullable := Value;
    Exit;
  end;
{$ENDIF}
  Assert(false, SNotImplemented);
end;

procedure TmtrQueryField.SetSQLType(const Value: Integer);
begin
{$IFDEF BDE}
  if FField is TParam then begin
    case Value of
      SQL_VARYING, SQL_TEXT: TParam(FField).DataType := ftString;
      SQL_DOUBLE, SQL_FLOAT: TParam(FField).DataType := ftFloat;
      SQL_LONG, SQL_SHORT: TParam(FField).DataType := ftInteger;
      SQL_TIMESTAMP: TParam(FField).DataType := ftDateTime;
      SQL_BLOB: TParam(FField).DataType := ftBlob;
      SQL_TYPE_TIME: TParam(FField).DataType := ftTime;
      SQL_TYPE_DATE: TParam(FField).DataType := ftDate;
      SQL_INT64: TParam(FField).DataType := ftCurrency;
    end; //case
  end;
{$ENDIF}
end;

{ TmtrQueryFields }

function TmtrQueryFields.ByName(Idx: String): TmtrQueryField;
begin
  result := nil;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: begin
      result := FField;
      FField.Field := TQuery(TmtrQuery(FQuery).Query).FieldByName(Idx);
    end;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: begin
      result := FField;
      FField.Field := TIBSQL(TmtrQuery(FQuery).Query).FieldByName(Idx);
    end;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

constructor TmtrQueryFields.Create(AObject: TObject; ADBMSType: TDBMSType);
begin
  FDBServerType := ADBMSType;
  FQuery := AObject;
  FField := TmtrQueryField.Create;
end;

destructor TmtrQueryFields.Destroy;
begin
  FField.Free; FField := nil;
  inherited;
end;

function TmtrQueryFields.GetFields(Idx: integer): TmtrQueryField;
begin
  result := nil;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: begin
      result := FField;
      FField.Field := TQuery(TmtrQuery(FQuery).Query).Fields[Idx];
    end;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: begin
      result := FField;
      FField.Field := TIBSQL(TmtrQuery(FQuery).Query).Fields[Idx];
    end;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

{ TmtrQueryParams }

function TmtrQueryParams.ByName(Idx: String): TmtrQueryField;
begin
  result := nil;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: begin
      result := FParam;
      FParam.Field := TQuery(TmtrQuery(FQuery).Query).ParamByName(Idx);
    end;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: begin
      result := FParam;
      FParam.Field := TIBSQL(TmtrQuery(FQuery).Query).Params.ByName(Idx);
    end;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

constructor TmtrQueryParams.Create(AObject: TObject; ADBMSType: TDBMSType);
begin
  FDBServerType := ADBMSType;
  FQuery := AObject;
  FParam := TmtrQueryField.Create;
end;

destructor TmtrQueryParams.Destroy;
begin
  FParam.Free; FParam := nil;
  inherited;
end;

function TmtrQueryParams.GetParams(Idx: integer): TmtrQueryField;
begin
  result := nil;
  case FDBServerType of
{$IFDEF BDE}
    dbmsBDE: begin
      result := FParam;
      FParam.Field := TQuery(TmtrQuery(FQuery).Query).Params[Idx];
    end;
{$ENDIF}
{$IFDEF IBX}
    dbmsInterbase: begin
      result := FParam;
      FParam.Field := TIBSQL(TmtrQuery(FQuery).Query).Params[Idx];
    end;
{$ENDIF}
    dbmsOracle: Assert(false, SNotImplemented);
  end;
end;

end.




